/*=============================================================================
 *     Copyright Texas Instruments 2002. All Rights Reserved.
 */


/**
 * This contains a hardware register field class.  A register-field is a
 * logical parameter within a register.  A register usually contains one
 * or more fields.
 * 
 * @param name         the string name of this register field
 * @param bitOffset    the offset to the first bit in this field
 * @param bitWidth     the size of this field in bits
 * @param writable     <code>true</code> if this field is writable
 */
public function RegisterField( name, bitOffset, bitWidth, writable )
{
  var fieldMask = ( ( 0xFFFFFFFF >> ( 32 - (bitWidth) ) ) << (bitOffset) );
  
  /**
   * Get the name of this register-field.
   * 
   * @return a string
   */
  public function getName()
  {
    return name;
  }
  
  /**
   * Get the bit-offset of the beginning of this field.  A bit-offset of
   * zero indicates that this field starts at the LSb of the register
   * containing this field.
   * 
   * @return the bit-offset (integer)
   */
  public function getBitOffset()
  {
    return bitOffset;
  }
  
  /**
   * Get the size of this field in bits.
   * 
   * @return the bit-size (integer)
   */
  public function getBitWidth()
  {
    return bitWidth;
  }
  
  /**
   * Is this register writable?
   * 
   * @return boolean
   */
  public function isWritable()
  {
    return writable;
  }
  
  /**
   * Get a bit-mask that can be used to isolate this field within
   * the register containing this field.
   * 
   * @return an integer
   */
  public function getFieldMask()
  {
    return fieldMask;
  }
  
  /**
   * For debugging.
   * 
   * @return a string
   */
  public function castToString()
  {
    return "[register field: " + getName() + "]";
  }
}


/*
 *   Local Variables:
 *   tab-width: 2
 *   indent-tabs-mode: nil
 *   mode: java
 *   c-indentation-style: java
 *   c-basic-offset: 2
 *   eval: (c-set-offset 'statement-cont '0)
 *   eval: (c-set-offset 'substatement-open '0)
 *   eval: (c-set-offset 'case-label '+)
 *   eval: (c-set-offset 'inclass '+)
 *   eval: (c-set-offset 'inline-open '0)
 *   End:
 */
