/*=============================================================================
 *     Copyright Texas Instruments 2002. All Rights Reserved.
 */


/**
 * This contains a hardware register Block class.  A register block contains
 * one or more registers.
 * 
 * @param name         the name of this register block
 * @param startAddr    the base address of this register block
 * @param regOffsetMult ???
 * @param registers    an array of <code>Register</code>s contained in this block
 */
public function RegisterBlock( name, startAddr, regOffsetMult, registers )
{
  /**
   * For now the registers must be ordered from lowest addess to highest address
   */
  var endAddr = ( registers[ (registers.length() - 1) ].getAddr() + regOffsetMult );
  
  var readListenerList  = new java.util.LinkedList();
  var writeListenerList = new java.util.LinkedList();
  
  
  for( var i=0; i<registers.length(); i++ )
    registers[i].setParent(this);
  
  
  /**
   * Get the name of this register block.
   * 
   * @return a string
   */
  public function getName()
  {
    return( name );
  }
  
  /**
   * Get the array of <code>Register</code>s in this block.
   * 
   * @return an array of <code>Register</code>
   */
  public function getRegisters()
  {
    return( registers );
  }
  
  /**
   * Get the base address of this block.
   * 
   * @return start address, an integer
   */
  public function getStartAddr()
  {
    return( startAddr );
  }
  
  /**
   * Get the end address of this block.
   * 
   * @return end address, an integer.
   */
  public function getEndAddr()
  {
    return( endAddr );
  }
  
  /**
   * This method returns the register information for a particular register
   * 
   * @param regName      the name of the register
   */
  public function getRegister( regName )
  {
    var register = null;
    
    for( var i = 0;
         ( (i < registers.length()) &&
           (register == null) );
         i++ )
    {
      if( regName == registers[i].getName() )
      {
        register = registers[i];
      }
    }
    
    return( register );
  }
  
  /**
   * Add a read listener.  A listener is a function that will be called when
   * the register is read from the software under test.  The register whose
   * value is read is passed to the callback function.  For example:
   * <pre>
   *   regBlock.addReadListener( function(reg) {
   *     
   *     pkg.output.writeln("the " + reg + " register has been read");
   *     
   *   } );
   * </pre>
   * The read-listener is called before the value of the register is read,
   * so it has the opportunity to modify the value of the register before
   * it is read.
   * 
   * @param fxn          the callback function
   * @see #removeReadListener
   * @see #addWriteListener
   */
  public function addReadListener(fxn)
  {
    readListenerList.add(fxn);
  }
  
  /**
   * Remove a read listener.
   * 
   * @param fxn          the callback function
   * @see #addReadListener
   */
  public function removeReadListener(fxn)
  {
    readListenerList.remove(fxn);
  }
  
  function dispatchRead(reg)
  {
    var arr = readListenerList.toArray();
    for( var i=0; i<arr.length(); i++ )
      arr[i](reg);
  }
  
  
  /**
   * Add a write listener.  A listener is a function that will be called when
   * the register is write from the software under test.  The register whose
   * value is write is passed to the callback function, as well as the new
   * value that is written to the register.  For example:
   * <pre>
   *   regBlock.addWriteListener( function( reg, val ) {
   *     
   *     pkg.output.writeln("the " + reg + " register has been written");
   *     
   *   } );
   * </pre>
   * The write-listener is called before the memHandler has written the new
   * value to the register, so it is possible to act on the value about to
   * be written before it is actually written.
   * 
   * @param fxn          the callback function
   * @see #removeWriteListener
   * @see #addWriteListener
   */
  public function addWriteListener(fxn)
  {
    writeListenerList.add(fxn);
  }
  
  /**
   * Remove a write listener.
   * 
   * @param fxn          the callback function
   * @see #addWriteListener
   */
  public function removeWriteListener(fxn)
  {
    writeListenerList.remove(fxn);
  }
  
  function dispatchWrite( reg, val )
  {
    var arr = writeListenerList.toArray();
    for( var i=0; i<arr.length(); i++ )
      arr[i]( reg, val );
  }
  
  
  /**
   * This is the method called from the MemHandler when a memory read operation
   * is requested from the VirtualMemoryHandler for a region of memory that this
   * object has claimed responsibility for. This method must check the
   * address to make sure the MemHandler didn't pass in a weird address.
   * 
   * @param baseAddr     the base address of the memory being read
   * @param unit         the unit type 8-bit, 16-bit, or 32-bit 
   * @param numUnits     the number of units requested to read.
   * 
   * @return This method returns the read object. An exception is thrown if the
   *         address was invalid
   */
  public function handleRead( baseAddr, unit, numUnits )
  {
    if( ( baseAddr >= getStartAddr() ) &&
        ( baseAddr <= getEndAddr() ) )
    {
      /*
       * Call the handler function
       */
      return read( baseAddr, unit, numUnits, dispatchRead );
    }
    else
    {
      throw new Exception( "Address: " + baseAddr + " not Handled by " + regBlock.getName() + " register Block" );
    }
  }
  
  /**
   * This is the method called from the MemHandler when a memory writer operation
   * is requested from the VirtualMemoryHandler to a region of memory that this
   * object has claimed responsibility for. This method must check the
   * address to make sure the MemHandler didn't pass in a weird address. Then
   * call the local writeReg function if the address is ok.
   * 
   * @param baseAddr     the base address of the memory being written to
   * @param unit         the unit type 8-bit, 16-bit, or 32-bit 
   * @param numUnits     the number of units requested to write.
   * @param buf          the data to write to the requested location.
   * 
   * @return This method returns the read object. An exception is thrown if the
   *         address was invalid
   */
  public function handleWrite( baseAddr, unit, numUnits, buf )
  {
    if( ( baseAddr >= getStartAddr() ) &&
        ( baseAddr <= getEndAddr() ) )
    {
      /*
       * Call the handler function
       */
      return write( baseAddr, unit, numUnits, buf, dispatchWrite );
    }
    else
    {
      throw new Exception( "Address: " + baseAddr + " not Handled by " + regBlock.getName() + " register Block" );
    }
  }
  
  /**
   * This method reads a single unit from the register block. The
   * unit may be any size. If the unit is greater than a single
   * register then multiple registers will be read into a single
   * return element. If the unit is less than a single register
   * unit then the most significant bits will be returned as a
   * single value. If the address is not valid then 0 is returned.
   * <p>
   * The callback function is called if the address is a valid
   * register.
   * <p>
   * Note: Current implementation only allows reads of the complete
   *       register. It also allows reads of invalid addresses.
   * 
   * @param addr         the address of the register to read
   * @param unit         ???
   * @param readCallback ???
   * @return A single value of the proper unit size read from the 
   * requested address.
   */
  function readUnit( addr, unit, readCallback )
  {
    var regAddr = 0;
    var regOctetSize = 0;
    var octetNum = 0;
    
    /*
     * Find the baseAddress in the list of registers
     */
    for( var i = 0;
         i < registers.length();
         i++ )
    {
      regAddr = registers[i].getAddr();
      regOctetSize = registers[i].getOctetWidth();
      
      /*
       * See if the requested address is in the register
       */
      if( ( addr >= regAddr ) &&
          ( addr <= ( regAddr + regOctetSize - 1 ) ) )
      {
        /*
         * The only case handled so far is where the address equals
         * the register address. No off boundary reads are supported yet.
         */
        if( ( addr == regAddr ) &&
            ( unit == registers[i].getBitWidth() ) )
        {
          var returnVal = registers[i].getValue();
          
          if( readCallback != null )
          {
            readCallback( registers[i] );
          }
          return( returnVal );
        }
        else
        {
          pkg.output.writeln( "" + unit + "-bit read of Address: " + addr + " is not valid" );  // XXX if this is fatal, throw exception!
          return( 0 );
        }
      }
    }
    
    /*
     * If we made it this far then a register containing the requested address
     * couldn't be found so return 0
     */ 
    return( 0 );
  }
  
  /**
   * This function handles the read within a logical block of registers
   * The block of registers doesn't need to be contiguous, and the requested
   * address doesn't need to start on any particular boundary. address
   * locations that have no registers will read back as 0.
   * <p>
   * Note: The current implementation restricts accesses to be on unit
   *       boundaries. Also The unit must match the units of the registers.
   * 
   * @param baseAddr     the address to start reading at
   * @param unit         ??? 8/16/32 ???
   * @param numUnits     number of units to read
   * @param readCallback ???
   * @return An array of data. The size of each array element will be the
   *   same as the unit size.
   */
  public /*XXX?*/ function read( baseAddr, unit, numUnits, readCallback )
  {
    var readData = [];
    
    /*
     * Need to figure out how many units are being read.
     */
    var beginOctets = ( baseAddr % ( unit / 8 ) );
    var endOctets   = ( unit / 8 ) - beginOctets;
    var inBetweenUnits = numUnits;
    
    /*
     * Do a quick check to see if begin address is on a unit
     * boundary. Off boundary reads are not supported yet.
     */
    if( beginOctets != 0 )
    {
      pkg.output.writeln( "Warning: Read Address " + baseAddr + " is not on " + unit + "-bit  boundary" );
      inBetweenUnits -= 1;
    }
      
    for( var i = 0;
         i < inBetweenUnits;
         i ++ )
    {
      readData[i] = readUnit( ( baseAddr + i ), unit, readCallback );
    }
    
    return( readData );
  }
  
  /**
   * This method ???
   * 
   * @param addr         the address of the register to write
   * @param unit         ???
   * @param data         ???
   * @param writeCallback ???
   * @return ???
   */
  function writeUnit( addr, unit, data, writeCallback )
  {
    var regAddr = 0;
    var regOctetSize = 0;
    var octetNum = 0;
    
    /*
     * Find the baseAddress in the list of registers
     */
    for( var i = 0;
         i < registers.length();
         i++ )
    {
      regAddr = registers[i].getAddr();
      regOctetSize = registers[i].getOctetWidth();
      
      /*
       * See if the requested address is in the register
       */
      if( ( addr >= regAddr ) &&
          ( addr <= ( regAddr + regOctetSize - 1 ) ) )
      {
        /*
         * The only case handled so far is where the address equals
         * the register address. No off boundary writes are supported yet.
         */
        if( ( addr == regAddr ) &&
            ( unit == registers[i].getBitWidth() ) )
        {
          if( writeCallback != null )
          {
            writeCallback( registers[i], data );
          }
          return( registers[i].setValue( data ) );
        }
        else
        {
          pkg.output.writeln( "" + unit + "-bit read of Address: " + addr + " is not valid" );
          return( false );
        }
      }
    }
  }
    
  /**
   * This method ???
   * 
   * @param baseAddr     the address to start writeing at
   * @param unit         ??? 8/16/32 ???
   * @param numUnits     number of units to write
   * @param buf          ???
   * @param writeCallback ???
   * @return ???
   */
  public /*XXX?*/ function write( baseAddr, unit, numUnits, buf, writeCallback )
  {
    /*
     * Need to figure out how many units are being read.
     */
    var beginOctets = ( baseAddr % ( unit / 8 ) );
    var endOctets   = ( unit / 8 ) - beginOctets;
    var inBetweenUnits = numUnits;
    
    /*
     * Do a quick check to see if begin address is on a unit
     * boundary. Off boundary reads are not supported yet.
     */
    if( beginOctets != 0 )
    {
      pkg.output.writeln( "Warning: Write Address " + baseAddr + " is not on " + unit + "-bit  boundary" );
      inBetweenUnits -= 1;
    }
      
    for( var i = 0;
         (i < inBetweenUnits);
         i ++ )
    {
      writeUnit( ( baseAddr + i ), unit, buf[i], writeCallback );
    }
   
    return( numUnits );
  }
  
  /**
   * For debugging.
   * 
   * @return a string
   */
  public function castToString()
  {
    return "[register block: " + getName() + "]";
  }
}


/*
 *   Local Variables:
 *   tab-width: 2
 *   indent-tabs-mode: nil
 *   mode: java
 *   c-indentation-style: java
 *   c-basic-offset: 2
 *   eval: (c-set-offset 'statement-cont '0)
 *   eval: (c-set-offset 'substatement-open '0)
 *   eval: (c-set-offset 'case-label '+)
 *   eval: (c-set-offset 'inclass '+)
 *   eval: (c-set-offset 'inline-open '0)
 *   End:
 */
