/*=============================================================================
 *     Copyright Texas Instruments 2002. All Rights Reserved.
 */


/**
 * Wrapper for the DBM event handler.
 * 
 * @param nodeId
 *            The id of the hostbuild to send/recv to/from
 * 
 * @param taskController
 *             The task controller object used to register for receiving
 *             notification of certain primitives.
 */
public function Wrapper( nodeId, taskController )
  extends pkg.util.Subsystem( nodeId, "DbmWrapper" )
{
  var prim = null;
  var transactionId = 0x12345678;
  var esf = services["esf routing"];
  var rctx = esf.getRoutingContext();
  
  /**
   * This method sends the Read Request to the DBM and waits for the read confirm, checks
   * its parameters to make sure the read succeded and the correct element was read.
   * 
   * @param elementId
   *            This is the DBM element Id to read.
   * 
   * @param elementVersion
   *            This is the version of the element to read.
   * 
   * @param recordNumber
   *            This is the record number within the element to read. This must be 0
   *            if the element is not composed of records.
   * 
   * @return null if the read failed, otherwise the read data is returned as a byte array.
   */
  public function readReq( elementId, elementVersion, recordNumber )
  {
    var readBuffer;
    
    rctx.sendPrim( new DBM_READ_REQ( task.getEhId(),
                                     transactionId,
                                     elementId,
                                     elementVersion,
                                     recordNumber,
                                     0,0 ) );
    
    readBuffer = waitForReadCnf( elementId,
                                 elementVersion,
                                 recordNumber );
    
    return( readBuffer );
  }
  
  /**
   * This method waits for the read confirm, checks its parameters to make sure the
   * read succeded and the correct element was read.
   * 
   * @param elementId
   *            This is the DBM element Id to read.
   * 
   * @param elementVersion
   *            This is the version of the element to read.
   * 
   * @param recordNumber
   *            This is the record number within the element to read. This must be 0
   *            if the element is not composed of records.
   * 
   * @return null if the read failed, otherwise the read data is returned as a byte array.
   */
  public function waitForReadCnf( elementId, elementVersion, recordNumber )
  {
    var readBuffer;
    
    taskController.run( function() { return ((prim == null) || !(prim instanceof DBM_READ_CNF)); } );
    
    if( ( prim.getTransactionId() == transactionId ) &&
        ( prim.getElementId() == elementId ) &&
        ( prim.getElementVersion() == elementVersion ) &&
        ( prim.getRecordNumber() == recordNumber ) )
    {
      if( prim.getSuccess() == 0 )
      {
        pkg.output.writeln( "Element Read Error" );
      }
      
      /* Even if there is no data this is ok. It will just be a 0 length array */
      readBuffer = prim.getData();
    }
    else
    {
      pkg.output.writeln( "Wrong element info returned" );
      pkg.output.writeln( "Expected Element Id : Version : Record Number -> " +
                        elementId +":"+ elementVersion +":"+ recordNumber );
      
      pkg.output.writeln( "Received Element Id : Version : Record Number -> " +
                        prim.getElementId() +":"+ prim.getElementVersion() +":"+ prim.getRecordNumber() );
      
      readBuffer = null;
    }
      
    prim = null;
      
    return( readBuffer );
  }
  
  /**
   * This method sends the Read Request to the DBM and waits for the read confirm, checks
   * its parameters to make sure the read succeded and the correct element was read.
   * 
   * @param elementId
   *            This is the DBM element Id to write.
   * 
   * @param elementVersion
   *            This is the version of the element to write.
   * 
   * @param recordNumber
   *            This is the record number within the element to write. This must be 0
   *            if the element is not composed of records.
   * 
   * @param dataBuffer
   *            This is the data to write. This must be an array, each element of the array
   *            represents a byte to write to the element. If there are too many bytes then
   *            the dbm will confirm with an error.
   * 
   * @return null if the read failed, otherwise the read data is returned as a byte array.
   */
  public function writeReq( elementId, elementVersion, recordNumber, dataBuffer )
  {
    var success;
    
    rctx.sendPrim( new DBM_WRITE_REQ( task.getEhId(),
                                      transactionId,
                                      elementId,
                                      elementVersion,
                                      recordNumber,
                                      0,0,
                                      dataBuffer.length(),
                                      dataBuffer ) );
    
    success = waitForWriteCnf( elementId,
                               elementVersion,
                               recordNumber );
    
    return( success );
  }
  
  /**
   * This method waits for the read confirm, checks its parameters to make sure the
   * read succeded and the correct element was read.
   * 
   * @param elementId
   *            This is the DBM element Id written to.
   * 
   * @param elementVersion
   *            This is the version of the element written to.
   * 
   * @param recordNumber
   *            This is the record number within the element written to. This must be 0
   *            if the element is not composed of records.
   * 
   * @return true if the primitive indicates a success for the proper element, false
   *         otherwise.
   */
  public function waitForWriteCnf( elementId, elementVersion, recordNumber )
  {
    var success = true;
    
    taskController.run( function() { return ((prim == null) || !(prim instanceof DBM_WRITE_CNF)); } );
    
    if( ( prim.getTransactionId() == transactionId ) &&
        ( prim.getElementId() == elementId ) &&
        ( prim.getElementVersion() == elementVersion ) &&
        ( prim.getRecordNumber() == recordNumber ) )
    {
      if( prim.getSuccess() == 0 )
      {
        pkg.output.writeln( "Element Write Error" );
        success = false;
      }
    }
    else
    {
      pkg.output.writeln( "Wrong element info returned" );
      pkg.output.writeln( "Expected Element Id : Version : Record Number -> " +
                          elementId +":"+ elementVersion +":"+ recordNumber );
      
      pkg.output.writeln( "Received Element Id : Version : Record Number -> " +
                          prim.getElementId() +":"+ prim.getElementVersion() +":"+ prim.getRecordNumber() );
      
      success = false;
    }
      
    prim = null;
      
    return( success );
  }
  
  /**
   */
  function handlePrim( prim )
  {
    if( prim.getSourceNodeId() == nodeId )
    {
      this.prim = prim;
    }
    else
    {
      pkg.output.writeln( "NodeId mismatch Expecting : Received " +
                          nodeId + ":" + prim.getSourceNodeId() );
    }
  }
  
  
  // we want to recv certain primitives:
  taskController.addPrimHandlers( [ DBM_READ_CNF,
                                    DBM_WRITE_CNF,
                                    DBM_PASSWORD_CNF ],
                                  handlePrim );
}




/*
 *   Local Variables:
 *   tab-width: 2
 *   indent-tabs-mode: nil
 *   mode: java
 *   c-indentation-style: java
 *   c-basic-offset: 2
 *   eval: (c-set-offset 'statement-cont '0)
 *   eval: (c-set-offset 'substatement-open '0)
 *   eval: (c-set-offset 'case-label '+)
 *   eval: (c-set-offset 'inclass '+)
 *   eval: (c-set-offset 'inline-open '0)
 *   End:
 */
