/******************************************************************************
 *        Copyright Texas Instruments 2003. All Rights Reserved.
 */

/**
 * This "ping-pong" test sends an <code>ETS_MEM_READ_REQ</code> and then waits
 * for a <code>ETS_MEM_READ_CNF</code> primitive with a matching transaction-
 * id.  This checks for the routing system getting "stuck", where a primitive
 * has been sent by another <code>rctx</code> but other <code>rctx</code>s
 * listening for the primitive don't see it.  In this case sending another
 * primitive will usually unstick things, so often the problem goes unnoticed,
 * but in this test since no other primitive is sent until the previous one is
 * received, the problem can be detected.
 * <p>
 * The watchdog thread will detect if things have gotten stuck by detecting if 
 * the current transaction-id has not changed in more than 5secs. The watchdog 
 * thread will also make a performance estimate of the number of prims/sec
 * being sent, because it knows that each increment of <code>transId</code>
 * means two primitives have been sent.
 * <p>
 * If run with no active SUT connection, then this test will install it's own
 * prim-handler to send the <code>ETS_MEM_READ_CNF</code> in response to an
 * <code>ETS_MEM_READ_REQ</code>, so that this test can also be used to do a
 * stand-alone test of the routing system without any SUT.
 */
public function PingPongTest(args...)
  extends pkg.tf.TestCase("Ping Pong")
{
  /**
   * Run a ping-pong test for <code>count</code> iterations
   * 
   * @param count the number of iterations to make
   */
  private function pingPongTest(count)
  {
    var hasSut = services["esf connection mgr"].getConnections().hasNext();
    var currentTimeMillis = java.lang.System.currentTimeMillis;
    var transId = 0;
    var lastSendReqTimeMs = 0;
    var lastSendCnfTimeMs = 0;
    
    pkg.output.logInfoMsg("SUT present? " + hasSut);
    
    /**
     * The watchdog.  If the watchdog thread detects that too much time
     * has elapsed since the last time the transaction-id increased, it
     * will bump() things along.  It also calculates an estimate of the
     * number of prims/sec.
     */
    var watchDogThread = new java.lang.Thread( function() {
      
      while(true)
      {
        var lastTransId = transId;
        var timeMs = currentTimeMillis();
        
        java.lang.Thread.sleep(5000);
        
        if( transId == lastTransId )
        {
          var str;
          if( lastSendReqTimeMs > lastSendCnfTimeMs )
            str = "lastSendReqTimeMs=" + lastSendReqTimeMs;
          else
            str = "lastSendCnfTimeMs=" + lastSendCnfTimeMs;
          
          pkg.output.logError("transId has not changed in 5000ms, transId=" + transId + ", " + str + ", bumping...");
          
          services["esf routing"].bump();
        }
        else
        {
          var deltaTransId = (transId - lastTransId).castToInexactNumber();
          var deltaTimeMs  = (currentTimeMillis() - timeMs).castToInexactNumber();
          var primsPerMs   = (2.0 * deltaTransId) / deltaTimeMs;
          pkg.output.logInfoMsg("prims/sec:" + (primsPerMs * 1000.0).castToExactNumber());
        }
      }
      
    });
    
    /* This handler sends the CNF primitive back to the main thread which
     * is sitting in a send-and-wait loop
     */
    if(!hasSut)
    {
      var rctx = services["esf routing"].getRoutingContext();
      
      rctx.addPrimHandler( function(prim) {
                             lastSendCnfTimeMs = currentTimeMillis();
                             rctx.sendPrim( new ETS_MEM_READ_CNF(
                                              /* destination ehId */     0x101,
                                              /* transId */              prim.getTransId(),
                                              /* readStatus */           0,
                                              /* address */              0,
                                              /* addressType */          0,
                                              /* dataUnit */             0,
                                              /* dataUnitArrayCount */   0,
                                              []
                                            ) );
                           },
                           rctx.type(ETS_MEM_READ_REQ) );
    }
    
    /* This is the main send-and-wait loop:
     */
    {
      var rctx = services["esf routing"].getRoutingContext();
      rctx.setConstraint( rctx.noHistory() );
      
      try
      {
        watchDogThread.start();
        for( var i=0; i<count; i++ )
        {
          lastSendReqTimeMs = currentTimeMillis();
          
          var prim = new ETS_MEM_READ_REQ(
                       /* source ehId */          0x101,
                       /* transId */              ++transId,
                       /* address */              0,
                       /* addressType */          0,
                       /* dataUnit */             0,
                       /* numUnits */             0
                     );
          
          prim.putSequenceNumber(transId);  // so we can see whats going in logger window!
          rctx.sendAndWaitForPrim( prim, 
                                   function(prim) {
                                     if( prim.getTransId() == transId )
                                       return false;
                                     pkg.output.logWarning("unexpected transId, got " + prim.getTransId() + ", expected " + transId);
                                     return true;
                                   },
                                   rctx.type(ETS_MEM_READ_CNF) );
        }
      }
      finally
      {
        watchDogThread.interrupt();
      }
    }
  }
  
  /**
   * The entry point to the test is this method.
   */
  public function runTest()
  {
    pingPongTest(500);
  }
}

