/******************************************************************************
 *     Copyright Texas Instruments, Inc., 2002.  All Rights Reserved.
 */

/**
 * Runs a regression sort of test for the sut connection specified.
 * The test spawns 2 threads. "SendPrimThread" & "SutStateControlThread".
 * The "SendPrimThread"  sends a random no. of prims & waits for the cnfs. 
 * It is in a send & wait loop until the test is stopped.
 * The  "SutStateControlThread" is responsible for activating & deactivating the sut connection.
 */
public function DynamicSutRegressionTest(args...)
  extends pkg.tf.TestCase("DynamicSutRegressionTest")
{
  const var MIN_TO_MS_CONVERSION_FACTOR = (1000*60);

  var rnd = new java.util.Random();
  
  var testRunning = false;
  var sutConnectionName = null;
  
  var sendPrimThread = null;
  var sutStateControlThread = null;
  var testDur = 0;
  var registryNodeSubscriber =  null;
  
  var regNodePath = "/ESF/DynamicSutRegressionTest/TestDuration";
  if( ! registry.exists( regNodePath ))
  {
    var regNode = new ti.chimera.registry.Node( new java.lang.Integer(5),
                                                ti.chimera.registry.NodeContract.NUMBER_CONTRACT,
                                                "Duration of the test in minutes");
    registry.link(regNode, regNodePath);
  }

  var testDurationNodeSubscriber =  function( node, val)
                                    {
                                      var nodeval = node.getValue();
                                      if(nodeval != null )
                                        testDur = nodeval *  MIN_TO_MS_CONVERSION_FACTOR;
                                    };
  registry.subscribeToValue( regNodePath ,
                             ti.chimera.registry.NodeContract.NUMBER_CONTRACT,
                             testDurationNodeSubscriber
                            );

  /**
   * watch dog timer to stop the test after "testDur" ms
   *
   */
  var watchDogTimer = new(function() extends java.lang.Thread(){
    setName("DynamicSutRegressionTest-watchDogTimer");
    public function run()
    {
      try {
        sleep( testDur );
        stopTest();
      }
      catch(exp){/*do nothing */}
    }
  })();

  /*
   * overrides the base class method
   */
  public function runTest()
  {
    startTest();
  }
  
  /*
   * start the test.
   */
  function startTest()
  {

    if( !testRunning )
    {
      /*
       * registry notifications when connection is activated & deactivated.
       */
      registryNodeSubscriber =  new ( function() extends ti.chimera.registry.NodeSubscriber() {
        public function publish( node, value )
        {
          
          var nodeVal = node.getValue();
          
          if( nodeVal.getChildCount() > 0 )
          {
            var itr = nodeVal.getChildNames();
            sutConnectionName = itr.next();
            
            if( testRunning )
              sendPrimThread.connectionUp();
          }
          else
          {
            if( testRunning )
            sendPrimThread.connectionDown();
          }
        }})();
      
      registry.subscribeToValue( "/ESF/Connections",null, registryNodeSubscriber);    
      
      sendPrimThread = new SendPrimThread();
      sutStateControlThread = new SutStateControlThread();
      testRunning = true;
      sendPrimThread.start();
      sutStateControlThread.start();
      watchDogTimer.start();
      
      sutStateControlThread.join();
      sendPrimThread.join();
    }
  }
  
  /*
   * stop the test
   */
  function stopTest()
  {
    if( testRunning )
    {
      registry.unlink("/ESF/DynamicSutRegressionTest",true);
      registry.unsubscribeFromValue(registryNodeSubscriber);
      testRunning = false;
      
      sendPrimThread.interrupt();
      sutStateControlThread.interrupt();
    }
  }
  
  /*
   * SutStateControlThread : responsible for activating & deactivating the sut connection.
   * The amount of time duration, that this thread waits between activating & deactivating 
   * a sut connection, is randomly generated
   */
  public function SutStateControlThread() extends java.lang.Thread("sutStateControlThread")
  {
    setName("sutStateControlThread");
    
    public function run()
    {
      
      
      const var MIN_DELAY_TO_DISCONNECT = 10000;
      const var MAX_DELAY_TO_DISCONNECT = MIN_DELAY_TO_DISCONNECT * 6;
      const var MAX_DELAY_TO_CONNECT = 1000;
      
      var disconnectDelay = rnd.nextInt(MAX_DELAY_TO_DISCONNECT);
      var connectDelay = rnd.nextInt(MAX_DELAY_TO_CONNECT);
      
      while( testRunning )
      {
        try
        {
          connectDelay = rnd.nextInt(MAX_DELAY_TO_CONNECT);
          
          disconnectDelay = rnd.nextInt(MAX_DELAY_TO_DISCONNECT);
          if( disconnectDelay < MIN_DELAY_TO_DISCONNECT )
            disconnectDelay = MIN_DELAY_TO_DISCONNECT;
          
          java.lang.Thread.sleep(MAX_DELAY_TO_CONNECT);
          
          services["esf connection mgr"].activateConnection(sutConnectionName);
          
          if( !services["esf connection mgr"].isActive(sutConnectionName))
          {
            pkg.output.logError("Failed to establish connection [ " + sutConnectionName + " ]");
            break;
          }
          
          java.lang.Thread.sleep(disconnectDelay);
          services["esf connection mgr"].deactivateConnection(sutConnectionName);
        }
        catch(excp)
        {
          if( excp instanceof java.lang.InterruptedException ) 
          { 
            pkg.output.logError("Test interrupted...");
            break;
          }
          else
          {
            pkg.output.logError(excp.toString());
            break;
          }
        }
      }
      stopTest();
      pkg.output.logInfoMsg("SutStateControlThread exit...");
    }
  }
  
  /*
   * SendPrimThread : This thread sends a random no. of primitives and waits for the cnfs.
   * If the connection goes down then it waits for the connection to go up.
   *
   */
  public function SendPrimThread() extends java.lang.Thread("sendPrimThread") 
  {
    setName("sendPrimThread");  
    
    var sutNodeId = 0;
    var syncObject = new java.lang.Object();
    var rctx = services["esf routing"].getRoutingContext();
    var rcxLastTransId = 0;
    var trxLastTransId = 0;
    
    /*
     * connection up notification method.
     */
    public function connectionUp()
    {
      synchronized( syncObject )
      {
        pkg.output.logInfoMsg("Connection to [" + sutConnectionName + "] up...");
        reset();
        syncObject.notify();
      }
    }
    
    /*
     * connection down notification method.
     */
    public function connectionDown()
    {
      pkg.output.logInfoMsg("Connection to [" + sutConnectionName + "] down...");
      rctx.dispose();
      sendPrimThread.interrupt();     
    }
    
    /*
     * waits for the connection to be activated.
     */
    function waitForActiveConnection()
    {
      synchronized( syncObject )
      {
        try {
          syncObject.wait();
        }
        catch( excep ) {
          // u will land up here when the test is stopped.
          if( excep instanceof java.lang.InterruptedException ) 
            interrupt();
        } 
      }
    }

    /*
     * thread exit clean up.
     */
    function cleanUp()
    {
      rctx.dispose();
    }
    
    /*
     * reset the thread data.
     */
    function reset()
    {
      rcxLastTransId = 0;
      trxLastTransId = 0;
      
      sutNodeId = services["esf connection mgr"].nodeNameToId( sutConnectionName);
      
      rctx = services["esf routing"].getRoutingContext();
      rctx.addPrimHandler( function(prim) {  
                                           rcxLastTransId = prim.getTransId();
                                          },rctx.type( ETS_LOOPBACK_CNF ) );      
    }
    
    /*
     * Thread "run" method
     */
    public function run()
    {
      
      var SEND_PRIM_MAX_BURST_RATE = 200;
      var MAX_LOOP_DELAY = 1000;
      var data = new Array(1,2,3,4,5,6,7,8,9,10,11,12,13,14,15,16,17,18,19,20);
      
      var primsToSend = 0;
      
      if( sutConnectionName == null)
        waitForActiveConnection();
      
      reset();
      
      while( testRunning )
      {
        try
        {
          // if interrupted and connection is not active then wait for connection to activate.
          if( ( interrupted() ) && ( !services["esf connection mgr"].isActive(sutConnectionName)))
            waitForActiveConnection();
          
          primsToSend = rnd.nextInt(SEND_PRIM_MAX_BURST_RATE);
          
          for( var i=0; i<primsToSend; i++ )
          {
            var req = new ETS_LOOPBACK_REQ(
                                            0x0101, // source ehId 
                                            i,      // transId
                                            data    //data
                                          );
            req.putDestinationNodeId(sutNodeId);
            rctx.sendPrim( req );
            
            trxLastTransId = i;
          }
          
          pkg.output.logInfoMsg("Sent Reqs :" + trxLastTransId );
          
          while(( rcxLastTransId != trxLastTransId ) && ( testRunning ))
            java.lang.Thread.sleep(MAX_LOOP_DELAY);
          
          pkg.output.logInfoMsg("Recvied Cnfs :" + rcxLastTransId);
          
          rcxLastTransId = 0;
          trxLastTransId = 0;
        }
        catch( excep )
        {
          if( excep instanceof java.lang.InterruptedException ) 
          { 
            interrupt();
          }
          else
          {
            pkg.output.logError(excep.toString());
            break;
          }
        }
      }
      cleanUp(); 
      stopTest();             
      pkg.output.logInfoMsg("SendPrimThread exit...");
    }
  }
}



/*
 *   Local Variables:
 *   tab-width: 2
 *   indent-tabs-mode: nil
 *   mode: java
 *   c-indentation-style: java
 *   c-basic-offset: 2
 *   eval: (c-set-offset 'statement-cont '0)
 *   eval: (c-set-offset 'substatement-open '0)
 *   eval: (c-set-offset 'case-label '+)
 *   eval: (c-set-offset 'inclass '+)
 *   eval: (c-set-offset 'inline-open '0)
 *   End:
 */
