/*******************************************************************************\
*                                                                           	*
*  TEXAS INSTRUMENTS ISRAEL PROPRIETARY AND CONFIDENTIAL INFORMATION			*
*																				*
*  LEGAL NOTE: THE TRANSFER OF THE TECHNICAL INFORMATION IS BEING MADE			*
*	UNDER AN EXPORT LICENSE ISSUED BY THE ISRAELI GOVERNMENT AND THAT THE		*
*	APPLICABLE EXPORT LICENSE DOES NOT ALLOW THE TECHNICAL INFORMATION TO		*
*	BE USED FOR THE MODIFICATION OF THE BT ENCRYPTION OR THE DEVELOPMENT		*
*	OF ANY NEW ENCRYPTION. THE INFORMATION CAN BE USED FOR THE INTERNAL			*
*	DESIGN AND MANUFACTURE OF TI PRODUCTS THAT WILL CONTAIN THE BT IC.			*
*																				*
\*******************************************************************************/
/*******************************************************************************\
*
*   FILE NAME:      btl_pbaps_pbi.c
*
*   DESCRIPTION:    The btl_pbaps_pbi.c is a BTL internal interface between the BTL_PBAP and the 
*					BTHAL_PB that handles the phonebook objects.  This module is called by  BTL_PBAPS module to 
*					prepare the phonebook data for the GOEP in order to send it to the peer device. 
*					It calls the BTHAL_PB to retrieve the device's phonebook data. 
*
*   AUTHOR:         Yoni Shavit
*
\*******************************************************************************/

#include "btl_config.h"
#if BTL_CONFIG_PBAPS == BTL_CONFIG_ENABLED


/******************************************************************************/
#include <bthal_pb.h>
#include <btl_pbaps_pbi.h>
#include <pbap.h>
#include <utils.h>
#include <btl_defs.h>

BTL_LOG_SET_MODULE(BTL_LOG_MODULE_TYPE_PBAPS);
/****************************************************************************
 *
 * Section: Constants
 *
 ***************************************************************************/
 
/*-------------------------------------------------------------------------------
 * BTL_PBAPS_PB_MAX_DATA_CHUNK constant
 *
 *     Defines the maximum length of data that should be prepared for the different pull operations.
 *	   Based on the maximum OBEX packet size. 
 */
#define BTL_PBAPS_PB_MAX_DATA_CHUNK  		 OBEX_TX_BUFF_SIZE

/*--------------------------------------------------------------------------
 * XML Context states
 *
 *     Defines the state of the phonebook XML context
 */
#define BTL_PBAPS_PB_XML_BEGIN_DECL_STATE       0
#define BTL_PBAPS_PB_XML_COPY_BEGIN_DECL_STATE  1
#define BTL_PBAPS_PB_XML_INIT_STATE    			2
#define BTL_PBAPS_PB_XML_OBJECT_SETUP_STATE     3
#define BTL_PBAPS_PB_XML_NEXT_STATE     		4
#define BTL_PBAPS_PB_XML_END_DECL_STATE         5
#define BTL_PBAPS_PB_XML_COPY_END_DECL_STATE    6
#define BTL_PBAPS_PB_XML_COPY_OBJECT_STATE      7 /* BTL_PBAPS_PB_XML_COPY_OBJECT = 7-12 */

/*--------------------------------------------------------------------------
 * XML Context vectors maximum length
 *
 *     Defines the maximum length of the XML context vectors
 */
/* 3 digits to the number + 4 to '.vcf' + 1 for null termination */
#define BTL_PBAPS_PB_MAX_VECTOR_ID				 8
/* There are possibly five names in the vector, plus 4 ';' */
#define BTL_PBAPS_PB_MAX_VECTOR_NAME		 	(BTHAL_PB_MAX_ENTRY_NAME  * 5  + 4) 

/*---------------------------------------------------------------------------
 * XML Context vectors maximum number
 *
 *     Defines the maximum number of the XML context vectors 
 */
#define BTL_PBAPS_PB_NUM_VECTORS         6

/*--------------------------------------------------------------------------
 * XML Context strings
 *
 *     Strings definitions state for the use of the phonebook XML context
 */
#define BTL_PBAPS_PB_XML_BEGIN_STR_1  	    "<?xml version=\"1.0\" ?>\n<!-- This Folder Object generated by iAnywhere Blue SDK -->\n"
#define BTL_PBAPS_PB_XML_BEGIN_STR_2  	    "<!DOCTYPE vcard-listing SYSTEM \"vcard-listing.dtd\">\n<vcard-listing version=\"1.0\">\n"
#define BTL_PBAPS_PB_XML_END_STR 	        "</vcard-listing>\n"

#define BTL_PBAPS_PB_XML_BEGIN_STR          BTL_PBAPS_PB_XML_BEGIN_STR_1 BTL_PBAPS_PB_XML_BEGIN_STR_2

/*---------------------------------------------------------------------------
 * BtlPbapsPbPathFlags type
 *
 *     This type is used to identify the portion of path information that
 */
typedef U16 BtlPbapsPbPathFlags;

#define BTL_PBAPS_PB_PATH_TELECOM                 0x0001    /* Telecom path is used */
#define BTL_PBAPS_PB_PATH_SIM                     0x0002    /* SIM path is used */
#define BTL_PBAPS_PB_PATH_PHONEBOOK               0x0004    /* Phonebook path is used */
#define BTL_PBAPS_PB_PATH_INCOMING                0x0008    /* Incoming call path is used */
#define BTL_PBAPS_PB_PATH_OUTGOING                0x0010    /* Outgoing call path is used */
#define BTL_PBAPS_PB_PATH_MISSED                  0x0020    /* Missed call path is used */
#define BTL_PBAPS_PB_PATH_COMBINED                0x0040    /* Combined call path is used */

/* End of PbPathFlags */

/*---------------------------------------------------------------------------
 * BtlPbapsPbCurOper type
 *
 *     	Describes the current on going pull operation.
 */
typedef U8 BtlPbapsPbCurOper;

#define BTL_PBAPS_PB_OPER_NONE			  		   0
#define BTL_PBAPS_PB_OPER_PULL_PB         	       1      
#define BTL_PBAPS_PB_OPER_PULL_VCARD_LIST 	  	   2      
#define BTL_PBAPS_PB_OPER_PULL_ENTRY     		   3      
#define BTL_PBAPS_PB_OPER_PROVIDE_PARMS  		   4  
#define BTL_PBAPS_PB_OPER_REMOTE_ABORT   		   5
#define BTL_PBAPS_PB_OPER_USER_ABORT               6


/*---------------------------------------------------------------------------
 * BtlPbapsPbOperState type
 *
 *     	Describes the state of the different phonebook operation. 
 */
typedef U8 BtlPbapsPbOperState;

/* The operation hasn't started yet */
#define BTL_PBAPS_PB_OPER_STATE_IDLE	  	  0
/* A phonebook was opened */ 
#define BTL_PBAPS_PB_OPER_STATE_PB_OPENED     1
/* Data is needed */
#define BTL_PBAPS_PB_OPER_STATE_GET_DATA      2
/* Data was received  */
#define BTL_PBAPS_PB_OPER_STATE_DATA_RECEIVED 3
/* Done reading data, phonebook maybe closed or not */
#define BTL_PBAPS_PB_OPER_STATE_DONE_DATA     4


/*---------------------------------------------------------------------------
 * BtlPbapsPbXmlRetVal type
 *
 *     	A return value of the function BtlPbapsBuildXmlData
 */
typedef U8 BtlPbapsPbXmlRetVal;

#define BTL_PBAPS_PB_XML_STATUS_FAILED			   0
#define BTL_PBAPS_PB_XML_STATUS_PENDING			   1      
#define BTL_PBAPS_PB_XML_STATUS_DONE_DATA   	   2      
#define BTL_PBAPS_PB_XML_STATUS_MORE_DATA		   3  

/*---------------------------------------------------------------------------
 * BtlPbapsPbReturnState type
 *
 *     	Holds the error state of the device's phonebook when it is asynchronous. 
 *		If this variable indicate of an error, the context switch from the phonebook
 *		task to the stack task will cause a PBAP abort operation. 
 *		
 */
typedef U8 BtlPbapsPbReturnState;

#define BTL_PBAPS_PB_RETURN_STATE_SUCCESS			   0
#define BTL_PBAPS_PB_RETURN_STATE_ERROR		 	  	   1  
#define BTL_PBAPS_PB_RETURN_STATE_IDLE     			   2    
#define BTL_PBAPS_PB_RETURN_STATE_NO_RESOURCES  	   3 
#define BTL_PBAPS_PB_RETURN_STATE_NOT_FOUND			   4 

/****************************************************************************
 *
 * Section: Types
 *
 ***************************************************************************/

/* Forward declarations */
typedef struct _BtlPbapsPbPullData   	BtlPbapsPbPullData;
typedef struct _BtlPbapsPbXmlContext    BtlPbapsPbXmlContext;
typedef struct _BtlPbapsPbPullParams    BtlPbapsPbPullParams;

/********************************************************************************
 *
 * Data Structures
 *
 *******************************************************************************/

/*--------------------------------------------------------------------------
 * BtlPbapsPbPullData structure
 *
 *     Holds the needed global parameters for the different pull operations the BTL_PBAP_PB module handles.
 */
struct _BtlPbapsPbPullData 
{
	/* A phonebook handle */
	BthalPbHandle 		pbHandle; 

	/* Holds the current phonebook path in BTHAL terms */
	BthalPbPhonebookType bthalPbPath;

	/* Holds the current phonebook path in the PBAP virtual folder structure terms */
	char pbPath[BTL_PBAPS_PB_MAX_PATH_LEN];

	/* Holds a copy of the current phonebook path, for the pull phonebook operation use. It is needed since pull
	 * pull operation shouldn't change the permanent path. */
	char tempPath[BTL_PBAPS_PB_MAX_PATH_LEN];

	/* Used for the speciel case when a pull operation is aborted. In that case the correct path should be recovered
	 * in the beginning of the following operation. 
	 */
	BOOL tempPathFlag;

 	/* Number of entries in a phonebook */
	U16 entriesPbNum;
	
	/* An entry data received from the phonebook */
	U8 	entryDataBuf[BTHAL_PB_MAX_ENTRY_LEN]; 	  

	/* BtlPbapsEntryDataBuf length */
	U16     entryDataLen;	
	
	/* Points to the entryDataBuf, tracks the left overs from the filling of chunkDataBuf */
	U16   entryOffset;
	
	/* A data chunk that is built from the phonebook data. 
	 * used to be delivered immediately opun request 
	 */
	U8 	chunkDataBuf[BTL_PBAPS_PB_MAX_DATA_CHUNK];   

	/* Points to the chunkDataBuf, tracks its fillings */
	U16   chunkOffset; 

	/* An entry name used in pull vCard listing operation */
	BthalPbEntryName entryName;
		
	/* An entry ID, to be used in pull operations for tracking the data building progress */
	U16 entryId;

	/* Counts number of entries in pull phonebook data building */
	U16	entriesCount;

	/* Searching paramters for the vCard listing operation */
	BthalPbListSearch listSearch;

	/* Sorting paramters for the vCard listing operation */
	BthalPbListSortingOrder listSortOrder;

	/* The retreived list of sotred entries.  */
	BthalPbEntriesList entriesList;

	/* Holds the phonebook path of which the parameter 'entriesList' was built opun in 
	 * the pull vcard listing operation
	 * The folder of pull entry requests is expected to be identical to that path, in  
	 * order to map the indexes correctly. Otherwise, the operation will be aborted 
	 */
	BthalPbPhonebookType entriesListPath;

	/* Indicates whether we finished reading  */
	BOOL doneReading;
	
	/* Indicates whether an error in the phonebook has occurred (asych mode only) */
	BtlPbapsPbReturnState returnState;

	/* We have a race condition when closing the phonebook  asynchronously:
	 * What will happened first - phonebook will close, or last continue event will be called 
	 * in the BTL? We need to send the last continue only after the phonebook was closed, and therefore 
	 * using the following flag 
	 */
	BOOL sendContinue;

	/* Indicates whether each phonebook has been changed by the user since 
	 * the last time it was opened by the BTL. Each slot in the array reffers to a certain phonrbook. Used 
	 * for the Browsing feature client synchronization If the be aborted by the server. (In that case client has to 
	 * issue another pull vCard listing operation for update)
 	 */					  
	BOOL pbChanged;

  
} ;

/*--------------------------------------------------------------------------
 * BtlPbapsPbPullParams structure
 *
 *     Holds the needed global RX parameters for the different pull operations the BTL_PBAP_PB module handles.
 */
struct _BtlPbapsPbPullParams 
{
	/* Pull phonebook parameters */
	BtlPbapsPbParams pb;

	/* Pull vCard listing parameters */
	BtlPbapsPbFolderListingParams listing;
	
	/* Pull vCard listing parameters */
	BtlPbapsPbVcardEntryParams	entry;
  
} ;

/*--------------------------------------------------------------------------
 * BtlPbapsPbXmlContext structure
 *
 *     This structure is used in the building of the XML listing of the pull vCard listing operation.
 */
struct _BtlPbapsPbXmlContext {
    U8              state;
    U8             *vectors[BTL_PBAPS_PB_NUM_VECTORS];
	U8			    entryIdVec[BTL_PBAPS_PB_MAX_VECTOR_ID];
	U8				entryNameVec[BTL_PBAPS_PB_MAX_VECTOR_NAME];

} ;


/********************************************************************************
 *
 * Globals
 *
 *******************************************************************************/

/* Holds global data variables for the different pull operation */							  
static BtlPbapsPbPullData btlPbapsPbPullData;

/* Holds global RX params for the different pull operation */							  
static BtlPbapsPbPullParams btlPbapsPbPullParams;

/* Used for the vCard XML listing building in the pull vCard entry operation */
static BtlPbapsPbXmlContext btlPbapsPbXmlContext;

/* The PPBAP server session */
static PbapServerSession  *btlPbapsPbserverSession;															   

/* The current on going operation */
static BtlPbapsPbCurOper btlPbapsPbCurOper;

/* Describes the state of the different phonebook operation */
static BtlPbapsPbOperState btlPbapsPbOperState;

/********************************************************************************
 *
 * Internal function prototypes
 *
 *******************************************************************************/															

static void 				BtlPbapsPbCallback(const BthalPbCallbackParms *callbackParams);
static BOOL 				BtlPbapsStripExt(char *name, const char *ext);
static U8   				BtlPbapsPbVectorCopy(const U8 **sourceVector);
static void 				BtlPbapsPbHtoA(U32 val, U8 *out);
static BOOL 				BtlPbapsPbVerifyPath(char *path, BtlPbapsPbPathFlags *PbPath);
static BOOL					BtlPbapsPbCorrectPullListingPath();
static BOOL 				BtlPbapsPbVarifyCallbackParams(const BthalPbCallbackParms *callbackParams);
static BtlPbapsPbXmlRetVal  BtlPbapsBuildXmlData();
void						BtlPbapsPbProcessContinue();
U8 							BtlPbapsPbStrNCmp(const char *Str1, const char *Str2, U16 num);
U16 						BtlPbapsPbAxToI(char *hexStg);
void						BtlPbapResetEntryName(void);

/********************************************************************************
 *
 * Function definitions
 *
 *******************************************************************************/

/*-------------------------------------------------------------------------------
 * BtlPbapsPbInit()
 */
BtStatus BtlPbapsPbInit(PbapServerSession *server)
{
	BtStatus 		   status = BT_STATUS_SUCCESS;
	
	BTL_FUNC_START("BtlPbapsPbInit");

/* The BTL pb implemetation assumes that the maximum OBEX packet size is bigger 
 * than max entry size. An implemantation change is needed if this assumption is wrong
 * The BtlPbapsPbBuildEntryData function should be changed. Enjoy (-:
 */
#if BTL_PBAPS_PB_MAX_DATA_CHUNK <= BTHAL_PB_MAX_ENTRY_LEN
#error OBEX max packet length must be bigger than max entry size length
#endif

	btlPbapsPbserverSession = server;

	btlPbapsPbPullData.tempPath[0] = '\0';
	btlPbapsPbPullData.tempPathFlag = FALSE;
	btlPbapsPbPullData.entriesListPath = 0;
	/* Register the PBHAL_PB module */
	BTHAL_PB_Register(BtlPbapsPbCallback);
    
	BTL_FUNC_END();
	
	return (status);
}

/*-------------------------------------------------------------------------------
 * BtlPbapsPbStartOper()
 */
BtStatus BtlPbapsPbStartOper()
{
	BtStatus 		   status = BT_STATUS_SUCCESS;

	BTL_FUNC_START("BtlPbapsPbStartOper");
	BTL_VERIFY_ERR(BTL_PBAPS_PB_OPER_USER_ABORT != btlPbapsPbCurOper, BT_STATUS_BUSY, 
		          ("Failed start a pb operation - phone is closing from last aborted operationon"));
	
	/* Index counters init */
	btlPbapsPbPullData.entryOffset = 0;
	btlPbapsPbPullData.chunkOffset = 0;
	btlPbapsPbPullData.entriesCount = 0;

	/* States init */
	btlPbapsPbCurOper = BTL_PBAPS_PB_OPER_NONE;
	btlPbapsPbOperState = BTL_PBAPS_PB_OPER_STATE_IDLE;
	btlPbapsPbPullData.returnState = BTL_PBAPS_PB_RETURN_STATE_IDLE;
	btlPbapsPbXmlContext.state = BTL_PBAPS_PB_XML_BEGIN_DECL_STATE;

	/* Boolean variables init */
	btlPbapsPbPullData.doneReading = FALSE;
	btlPbapsPbPullData.sendContinue = FALSE;

	/* Buffers init */
	OS_MemSet(btlPbapsPbPullData.entryDataBuf, 0 , BTHAL_PB_MAX_ENTRY_LEN);
	OS_MemSet(btlPbapsPbPullData.chunkDataBuf, 0 , BTHAL_PB_MAX_ENTRY_LEN);
	OS_MemSet((U8*)&btlPbapsPbXmlContext, 0 , sizeof(BtlPbapsPbXmlContext));

	/* Path recovery, in case the previous pull operation was aborted */
    if (TRUE == btlPbapsPbPullData.tempPathFlag)
    {
	    /* Set the phonebook path back to what it was before the pull phonebook operation */
		status = BtlPbapsPbSetPbPath(btlPbapsPbPullData.tempPath, BTL_PBAPS_PB_SET_PATH_ABSOLUTE);
		btlPbapsPbPullData.tempPathFlag = FALSE;
    }
		

	BTL_FUNC_END();
	
	return (status);
	
}

/*-------------------------------------------------------------------------------
 * BtlPbapsPbBuildEntryData()
 */
BtStatus BtlPbapsPbBuildEntryData(BOOL stackTask)
{
	
    static U16		   entryXmlId;                /* The enrty ID given to it in the XML listing */
	U16				   entryPbId;                 /* The enrty ID in the device's phonebook */
	BthalStatus 	   bthalStatus;
	BtStatus 		   status = BT_STATUS_SUCCESS;
    ObStatus 		   obStatus;
	BOOL			   retVal;
	BOOL 	           goOn = TRUE;
	BOOL			   pbChanged;
		
	BTL_FUNC_START("BtlPbapsPbBuildEntryData");

	btlPbapsPbCurOper = BTL_PBAPS_PB_OPER_PULL_ENTRY;
	
	while (goOn == TRUE)
	{

		switch (btlPbapsPbOperState)
		{
			case BTL_PBAPS_PB_OPER_STATE_IDLE:

				
				if (btlPbapsPbPullData.doneReading == TRUE)
				{
					/* Data is ready, call continue to keep the requests flowing. */
			    	obStatus = PBAP_ServerContinue(btlPbapsPbserverSession);
					BTL_VERIFY_ERR(OB_STATUS_SUCCESS == obStatus, BT_STATUS_INTERNAL_ERROR, ("Server: PBAP server continue failure"));
					BTL_RET(BT_STATUS_SUCCESS);
				}
				/* Check whether the phonebook was changed */
				/* If the current phonebook was already checked using BTHAL_PB_WasPbChanged, don't do it 
				 * again, since it will reset the BTHAL_PB_WasPbChanged indicator 
				 */
				if (TRUE == BtlPbapsPbGetChangedStatus())
				{
                    BTL_LOG_DEBUG((""));
					status = BT_STATUS_IMPROPER_STATE;
					goOn = FALSE;
					break;
				}
				
				/* Otherwise ask the phonebook whether it was changed */
				bthalStatus = BTHAL_PB_WasPbChanged(btlPbapsPbPullData.bthalPbPath, (BTHAL_BOOL*)&pbChanged);
				BTL_VERIFY_ERR(BTHAL_STATUS_FAILED != bthalStatus, BT_STATUS_FAILED, ("BTHAL_PB_WasPbChanged failed"));

				if (TRUE == pbChanged)
				{
					BtlPbapsPbSetChangedStatus(TRUE);
					status = BT_STATUS_IMPROPER_STATE;
					goOn = FALSE;
					break;
				} 

				/* If the path doesn't identical to the path that the entries list 
				 * was built opun, the entry ID mapping will be incorrect. 
				 * Need to abort the operation.
				 */
				if (btlPbapsPbPullData.entriesListPath != btlPbapsPbPullData.bthalPbPath)
				{
					BTL_LOG_INFO(("Pull Entry Failed. The last pull vcard listing was performed from a different path."));
					status = BT_STATUS_IMPROPER_STATE;
					goOn = FALSE;
					break;
				}
				
				/* Prepare the entry index */
				retVal = BtlPbapsStripExt(btlPbapsPbPullParams.entry.entryName, ".vcf");
				BTL_VERIFY_ERR(TRUE == retVal, BT_STATUS_FAILED, ("BtlPbapsPbBuildEntryData failed"));
				
				entryXmlId = BtlPbapsPbAxToI(btlPbapsPbPullParams.entry.entryName);

				/* Check that the asked entry exists.
				 * Note: In case of an invalid entry name OS_AtoU16 returns 0, so the owner entry will be returned. */
				if (entryXmlId >= btlPbapsPbPullData.entriesList.entriesNum) 
				{
					BTL_LOG_DEBUG(("BtlPbapsPbBuildEntryData. Invalid entry number"));
					status = BT_STATUS_NOT_FOUND;
					goOn = FALSE;
					break;
				}
				
				/* Open a phonebook */
				bthalStatus = BTHAL_PB_OpenPb(btlPbapsPbPullData.bthalPbPath, &btlPbapsPbPullData.pbHandle);
				BTL_VERIFY_ERR(BTHAL_STATUS_FAILED != bthalStatus, BT_STATUS_FAILED, ("BTHAL_PB_OpenPb failed"));

				/* Check whether the pb is busy */
				if (BTHAL_STATUS_BUSY == bthalStatus)
				{
					BTL_LOG_DEBUG(("BTHAL_PB_OpenPb is busy"));
					status = BT_STATUS_BUSY;
					goOn = FALSE;
					break;
				}
				
				btlPbapsPbOperState = BTL_PBAPS_PB_OPER_STATE_PB_OPENED;

				if (bthalStatus == BTHAL_STATUS_PENDING)
				{
					status = BT_STATUS_PENDING;
					goOn = FALSE;
				}
				break;

			case BTL_PBAPS_PB_OPER_STATE_PB_OPENED:
				
				entryPbId = btlPbapsPbPullData.entriesList.entryId[entryXmlId]; 	
				
				bthalStatus = BTHAL_PB_GetEntryData(btlPbapsPbPullData.pbHandle,(BthalPbEntryId) entryPbId,
													btlPbapsPbPullParams.entry.format, btlPbapsPbPullParams.entry.filter,
													(BTHAL_U8*) btlPbapsPbPullData.chunkDataBuf, 
													(BthalPbVcfLen*)&btlPbapsPbPullData.entryDataLen);
				BTL_VERIFY_ERR(BTHAL_STATUS_FAILED != bthalStatus, BT_STATUS_FAILED, ("BTHAL_PB_GetEntryData failed"));
				
				btlPbapsPbOperState = BTL_PBAPS_PB_OPER_STATE_GET_DATA;

				if (bthalStatus == BTHAL_STATUS_PENDING)
				{
					status = BT_STATUS_PENDING;
					goOn = FALSE;
					
				}
				break;
				
			case BTL_PBAPS_PB_OPER_STATE_GET_DATA:
				
				btlPbapsPbPullData.chunkOffset = (U16)(btlPbapsPbPullData.chunkOffset + btlPbapsPbPullData.entryDataLen);
				btlPbapsPbPullData.doneReading = TRUE;
				/* If the phonebook returns synchronously, we finished */
				goOn = FALSE;
				
				/* We switch to data received state since more continue events may arrive if not all chunk will be copied 
				 * in one packet
				 */
				btlPbapsPbOperState = BTL_PBAPS_PB_OPER_STATE_DATA_RECEIVED;

				if (stackTask == TRUE)
				{
					/* Data is ready, call continue to keep the requests flowing. */
			    	obStatus = PBAP_ServerContinue(btlPbapsPbserverSession);
					BTL_VERIFY_ERR(OB_STATUS_SUCCESS == obStatus, BT_STATUS_FAILED, ("Server: PBAP server continue failure"));
				}
				else
				{
					btlPbapsPbPullData.returnState = BTL_PBAPS_PB_RETURN_STATE_SUCCESS;
					/* Sent event to return to the stack task */
					bthalStatus = BTHAL_OS_SendEvent(BTHAL_OS_TASK_HANDLE_STACK, OS_EVENT_STACK_PB_TASK);
					BTL_VERIFY_ERR(bthalStatus == BTHAL_STATUS_SUCCESS , BT_STATUS_FAILED, ("BTHAL_OS_SendEvent failed."));
				}
				break;
				
				
			case BTL_PBAPS_PB_OPER_STATE_DATA_RECEIVED:	
				 /* In case this function was called since more continue events have arrived since only part of chunk was transmitted 
				 * in one OBEX packet (only part of the chunk was copied in BtlPbapsPbGetData)
				 */
				btlPbapsPbPullData.doneReading = TRUE;
				/* If the phonebook returns synchronously, we finished */
				goOn = FALSE;
				if (stackTask == TRUE)
				{
					/* Data is ready, call continue to keep the requests flowing. */
			    	obStatus = PBAP_ServerContinue(btlPbapsPbserverSession);
					BTL_VERIFY_ERR(OB_STATUS_SUCCESS == obStatus, BT_STATUS_FAILED, ("Server: PBAP server continue failure"));
				}
				else
				{
					btlPbapsPbPullData.returnState = BTL_PBAPS_PB_RETURN_STATE_SUCCESS;
					/* Sent event to return to the stack task */
					bthalStatus = BTHAL_OS_SendEvent(BTHAL_OS_TASK_HANDLE_STACK, OS_EVENT_STACK_PB_TASK);
					BTL_VERIFY_ERR(bthalStatus == BTHAL_STATUS_SUCCESS , BT_STATUS_FAILED, ("BTHAL_OS_SendEvent failed."));
				}
				break;

				
			/* We have a race condition when closing the phonebook  asynchronously:
			 * What will happened first - phonebook will close, or last continue event will be called 
			 * in the BTL? The next case is handling the situation 
			 */
			case BTL_PBAPS_PB_OPER_STATE_DONE_DATA:
				/* In case this function was called as a result of the last continue event in btl_pbap, and the phonebook
				 * was already closed, don't need to build any more data
				 */
				if (btlPbapsPbPullData.sendContinue == TRUE)
				{
					/* Data is ready, call continue to keep the requests flowing. */
			    	obStatus = PBAP_ServerContinue(btlPbapsPbserverSession);
					BTL_VERIFY_ERR(OB_STATUS_SUCCESS == obStatus, BT_STATUS_FAILED, ("Server: PBAP server continue failure"));
					btlPbapsPbPullData.sendContinue = FALSE;
					btlPbapsPbOperState = BTL_PBAPS_PB_OPER_STATE_IDLE;
					
				}
				else
					btlPbapsPbPullData.sendContinue = TRUE;
				
				goOn = FALSE;
				break;
				
								
			default:
				BTL_ERR(BT_STATUS_FAILED, ("BtlPbapsPbBuildEntryData failed. Wrong operation state"));
				
		}
		
			
	}
	
	

	BTL_FUNC_END();
	
	return (status);
}


/*-------------------------------------------------------------------------------
 * BtlPbapsPbBuildPbData()
 */
BtStatus BtlPbapsPbBuildPbData(BOOL stackTask)
{
	BtStatus 		  		 status = BT_STATUS_SUCCESS;
	BthalStatus 	  		 bthalStatus;
	ObStatus 		  		 obStatus;
	U16				  	 	 remainingBufLen;
	BOOL					 goOn = TRUE;

	BTL_FUNC_START("BtlPbapsPbBuildPbData");


	btlPbapsPbCurOper = BTL_PBAPS_PB_OPER_PULL_PB;

	/* Build phonebook data */
	while (goOn == TRUE)
	{
		switch (btlPbapsPbOperState)
		{
			case BTL_PBAPS_PB_OPER_STATE_IDLE:

				/* In case we have data left from the last chuck building, first copy it to the chunk */
				if (btlPbapsPbPullData.entryOffset > 0)
				{
					OS_MemCopy(&btlPbapsPbPullData.chunkDataBuf[btlPbapsPbPullData.chunkOffset] , 
						       &btlPbapsPbPullData.entryDataBuf[btlPbapsPbPullData.entryOffset], 
				   	           (U32) (btlPbapsPbPullData.entryDataLen - btlPbapsPbPullData.entryOffset) );

					btlPbapsPbPullData.chunkOffset = (U16)(btlPbapsPbPullData.chunkOffset + (btlPbapsPbPullData.entryDataLen - btlPbapsPbPullData.entryOffset));
					btlPbapsPbPullData.entryOffset = 0;
				}

				btlPbapsPbOperState = BTL_PBAPS_PB_OPER_STATE_GET_DATA;
				break;
			
			case BTL_PBAPS_PB_OPER_STATE_GET_DATA:

				/* fill the chunk */
				if (  (btlPbapsPbPullData.entriesCount < btlPbapsPbPullParams.pb.maxListCount) && 
					     (btlPbapsPbPullData.entryId < btlPbapsPbPullData.entriesPbNum) &&
					     (btlPbapsPbPullData.chunkOffset < (U16)BTL_PBAPS_PB_MAX_DATA_CHUNK)        )
				{
					/* Read entry from the phonebook */
					bthalStatus = BTHAL_PB_GetEntryData(btlPbapsPbPullData.pbHandle, (BthalPbEntryId) btlPbapsPbPullData.entryId,
														btlPbapsPbPullParams.pb.format, btlPbapsPbPullParams.pb.filter,
														(BTHAL_U8*)btlPbapsPbPullData.entryDataBuf, (BthalPbVcfLen*)&btlPbapsPbPullData.entryDataLen);
					BTL_VERIFY_ERR(BTHAL_STATUS_FAILED != bthalStatus, BT_STATUS_FAILED, ("BTHAL_PB_GetEntryData failed"));
					BTL_VERIFY_ERR(BTHAL_STATUS_NOT_FOUND != bthalStatus, BT_STATUS_FAILED, ("BTHAL_PB_GetEntryData failed - entry not found"));
					
					btlPbapsPbOperState = BTL_PBAPS_PB_OPER_STATE_DATA_RECEIVED;
					if (bthalStatus == BTHAL_STATUS_PENDING)
						goOn = FALSE;

					break;
					
					
				} 
				else
				/* If the chunk is not full yet but all needed entries were read, we finished with this state */
				{
					btlPbapsPbPullData.doneReading = TRUE;
					goOn = FALSE;
					if (stackTask == TRUE)
					{
						/* Data is ready, call continue to keep the requests flowing. */
				    	obStatus = PBAP_ServerContinue(btlPbapsPbserverSession);
						BTL_VERIFY_ERR(OB_STATUS_SUCCESS == obStatus, BT_STATUS_FAILED, ("Server: PBAP server continue failure"));
					}
					else
					{
						btlPbapsPbPullData.returnState = BTL_PBAPS_PB_RETURN_STATE_SUCCESS;
						/* Sent event to return to the stack task */
						bthalStatus = BTHAL_OS_SendEvent(BTHAL_OS_TASK_HANDLE_STACK, OS_EVENT_STACK_PB_TASK);
						BTL_VERIFY_ERR(bthalStatus == BTHAL_STATUS_SUCCESS , BT_STATUS_FAILED, ("BTHAL_OS_SendEvent failed."));
					}
					break;

					
				}

			case BTL_PBAPS_PB_OPER_STATE_DATA_RECEIVED:	
				remainingBufLen = (U16)(BTL_PBAPS_PB_MAX_DATA_CHUNK - btlPbapsPbPullData.chunkOffset);

				btlPbapsPbPullData.entryId++;
				btlPbapsPbPullData.entriesCount++;

				if (remainingBufLen >= btlPbapsPbPullData.entryDataLen) 
				/* Chuck has enough memory for the whole entry */					
				{
					OS_MemCopy(&btlPbapsPbPullData.chunkDataBuf[btlPbapsPbPullData.chunkOffset],
							    btlPbapsPbPullData.entryDataBuf, (U32) btlPbapsPbPullData.entryDataLen);
					btlPbapsPbPullData.chunkOffset = (U16)(btlPbapsPbPullData.chunkOffset + btlPbapsPbPullData.entryDataLen);
					
										
				}
				else
				/* Chuck has enough memory for part of the entry only */
				{
					
					OS_MemCopy(&btlPbapsPbPullData.chunkDataBuf[btlPbapsPbPullData.chunkOffset], btlPbapsPbPullData.entryDataBuf, (U32) remainingBufLen);			
					btlPbapsPbPullData.entryOffset = remainingBufLen;
					btlPbapsPbPullData.chunkOffset = (U16)(btlPbapsPbPullData.chunkOffset + remainingBufLen);
					
				}
				
				if (btlPbapsPbPullData.chunkOffset == (U16)(BTL_PBAPS_PB_MAX_DATA_CHUNK))
				/* If we've filled the whole chunck, we finished with this state */
				{
										
					btlPbapsPbOperState = BTL_PBAPS_PB_OPER_STATE_IDLE;
					goOn = FALSE;
					if (stackTask == TRUE)
					{
						/* Data is ready, call continue to keep the requests flowing. */
				    	obStatus = PBAP_ServerContinue(btlPbapsPbserverSession);
						BTL_VERIFY_ERR(OB_STATUS_SUCCESS == obStatus, BT_STATUS_FAILED, ("Server: PBAP server continue failure"));
					}
					else
					{
						btlPbapsPbPullData.returnState = BTL_PBAPS_PB_RETURN_STATE_SUCCESS;
						/* Sent event to return to the stack task */
						bthalStatus = BTHAL_OS_SendEvent(BTHAL_OS_TASK_HANDLE_STACK, OS_EVENT_STACK_PB_TASK);
						BTL_VERIFY_ERR(bthalStatus == BTHAL_STATUS_SUCCESS , BT_STATUS_FAILED, ("BTHAL_OS_SendEvent failed."));
					}
					break;
					
					
				}
				else if (  (btlPbapsPbPullData.chunkOffset <= (U16)BTL_PBAPS_PB_MAX_DATA_CHUNK) && 
				    ( (btlPbapsPbPullData.entriesCount == btlPbapsPbPullParams.pb.maxListCount) || 
					  (btlPbapsPbPullData.entriesCount == btlPbapsPbPullData.entriesPbNum) ) )
				/* If  the chunk is not full yet but all needed entries were read, we finished with this state */
				{
					btlPbapsPbPullData.doneReading = TRUE;
					goOn = FALSE;
					btlPbapsPbOperState = BTL_PBAPS_PB_OPER_STATE_GET_DATA;
					if (stackTask == TRUE)
					{
						/* Data is ready, call continue to keep the requests flowing. */
				    	obStatus = PBAP_ServerContinue(btlPbapsPbserverSession);
						BTL_VERIFY_ERR(OB_STATUS_SUCCESS == obStatus, BT_STATUS_FAILED, ("Server: PBAP server continue failure"));
					}
					else
					{
						btlPbapsPbPullData.returnState = BTL_PBAPS_PB_RETURN_STATE_SUCCESS;
						/* Sent event to return to the stack task */
						bthalStatus = BTHAL_OS_SendEvent(BTHAL_OS_TASK_HANDLE_STACK, OS_EVENT_STACK_PB_TASK);
						BTL_VERIFY_ERR(bthalStatus == BTHAL_STATUS_SUCCESS , BT_STATUS_FAILED, ("BTHAL_OS_SendEvent failed."));
					}
					break;

					
				}
				else
				/* We didn't finish reading */
				{
					btlPbapsPbOperState = BTL_PBAPS_PB_OPER_STATE_GET_DATA;
				}
				
				break;

			/* We have a race condition when closing the phonebook  asynchronously:
			 * What will happened first - phonebook will close, or last continue event will be called 
			 * in the BTL? The next case is handling the situation 
			 */
			case BTL_PBAPS_PB_OPER_STATE_DONE_DATA:

				/* In case this function was called as a result of the last continue event in btl_pbap, and the phonebook
				 * was already closed, don't need to build any more data
				 */
				if (btlPbapsPbPullData.sendContinue == TRUE)
				{
					btlPbapsPbPullData.sendContinue = FALSE;
					btlPbapsPbOperState = BTL_PBAPS_PB_OPER_STATE_IDLE;
					/* Set the phonebook path back to what it was before the pull phonebook operation */
					status = BtlPbapsPbSetPbPath(btlPbapsPbPullData.tempPath, BTL_PBAPS_PB_SET_PATH_ABSOLUTE);
					btlPbapsPbPullData.tempPathFlag = FALSE;
					BTL_VERIFY_ERR(BT_STATUS_SUCCESS == status, BT_STATUS_FAILED, ("BtlPbapsPbSetPbPath failed"));
					
					if (stackTask == TRUE)
					{
						/* Data is ready, call continue to keep the requests flowing. */
				    	obStatus = PBAP_ServerContinue(btlPbapsPbserverSession);
						BTL_VERIFY_ERR(OB_STATUS_SUCCESS == obStatus, BT_STATUS_FAILED, ("Server: PBAP server continue failure"));
					}
					else
					{
						btlPbapsPbPullData.returnState = BTL_PBAPS_PB_RETURN_STATE_SUCCESS;
						/* Sent event to return to the stack task */
						bthalStatus = BTHAL_OS_SendEvent(BTHAL_OS_TASK_HANDLE_STACK, OS_EVENT_STACK_PB_TASK);
						BTL_VERIFY_ERR(bthalStatus == BTHAL_STATUS_SUCCESS , BT_STATUS_FAILED, ("BTHAL_OS_SendEvent failed."));
					}
					
				}
				else
					btlPbapsPbPullData.sendContinue = TRUE;
				
				goOn = FALSE;
				break;
				
			default:
				BTL_ERR(BT_STATUS_FAILED, ("BtlPbapsPbBuildPbData failed. Wrong operation state"));

		} /* End switch (btlPbapsPbOperState) */
		
	} /* End loop */

	BTL_FUNC_END();

	return (status);	
}


/*-------------------------------------------------------------------------------
 * BtlPbapsPbBuildFolderListingData()
 */
BtStatus BtlPbapsPbBuildFolderListingData(BOOL stackTask)
{
	BtStatus 		  		 status = BT_STATUS_SUCCESS;
	BthalStatus 	  		 bthalStatus;
	ObStatus 		  		 obStatus;
	BOOL					 goOn = TRUE;
	
	BTL_FUNC_START("BtlPbapsPbBuildFolderListingData");

	btlPbapsPbCurOper = BTL_PBAPS_PB_OPER_PULL_VCARD_LIST;
	
	while (goOn == TRUE)
	{

		switch (btlPbapsPbOperState)
		{
			case BTL_PBAPS_PB_OPER_STATE_IDLE:
				btlPbapsPbOperState = BTL_PBAPS_PB_OPER_STATE_GET_DATA;
				/* Drop to next case */
				
			case BTL_PBAPS_PB_OPER_STATE_GET_DATA:

				switch (btlPbapsPbXmlContext.state) 
				{
				    case BTL_PBAPS_PB_XML_BEGIN_DECL_STATE:
				        OS_MemSet((U8 *)&btlPbapsPbXmlContext, 0, sizeof(BtlPbapsPbXmlContext));
				        btlPbapsPbXmlContext.vectors[0] = (U8*)(BTL_PBAPS_PB_XML_BEGIN_STR);
				        btlPbapsPbXmlContext.vectors[1] = 0;

				        /* Drop through to next state */
				        btlPbapsPbXmlContext.state++;

				    case BTL_PBAPS_PB_XML_COPY_BEGIN_DECL_STATE:
				        /* Write XML Header and Folder Decl */
				        if (BtlPbapsPbVectorCopy((const U8 **)&btlPbapsPbXmlContext.vectors[0])) 
						{
				            /* Couldn't write complete declaration, were done (with this chunk only) */
							goOn = FALSE;
							if (stackTask == TRUE)
							{
								/* Data is ready, call continue to keep the requests flowing. */
						    	obStatus = PBAP_ServerContinue(btlPbapsPbserverSession);
								BTL_VERIFY_ERR(OB_STATUS_SUCCESS == obStatus, BT_STATUS_FAILED, ("Server: PBAP server continue failure"));
							}
							else
							{
								btlPbapsPbPullData.returnState = BTL_PBAPS_PB_RETURN_STATE_SUCCESS;
								/* Sent event to return to the stack task */
								bthalStatus = BTHAL_OS_SendEvent(BTHAL_OS_TASK_HANDLE_STACK, OS_EVENT_STACK_PB_TASK);
								BTL_VERIFY_ERR(bthalStatus == BTHAL_STATUS_SUCCESS , BT_STATUS_FAILED, ("BTHAL_OS_SendEvent failed."));
							}
				            break;
				        }
				        /* Drop through to next state */
				        btlPbapsPbXmlContext.state++;

				    default:
				        /* Write folder object body */
				        switch (BtlPbapsBuildXmlData())
				        {
				        	case BTL_PBAPS_PB_XML_STATUS_FAILED:
								BTL_ERR(BT_STATUS_FAILED, ("Failed building XML data"));
								/* break; */
							case BTL_PBAPS_PB_XML_STATUS_PENDING:
								status = BT_STATUS_PENDING;
								goOn = FALSE;
								break;
							case BTL_PBAPS_PB_XML_STATUS_DONE_DATA:
								/* Drop through to next XML state to finish XML */
				       			btlPbapsPbXmlContext.state = BTL_PBAPS_PB_XML_END_DECL_STATE;
								break;
			
							case BTL_PBAPS_PB_XML_STATUS_MORE_DATA: 
								/* Need to send the chunk, and only then and keep on building XML */
								goOn = FALSE;
								if (stackTask == TRUE)
								{
									/* Data is ready, call continue to keep the requests flowing. */
							    	obStatus = PBAP_ServerContinue(btlPbapsPbserverSession);
									BTL_VERIFY_ERR(OB_STATUS_SUCCESS == obStatus, BT_STATUS_FAILED, ("Server: PBAP server continue failure"));
								}
								else
								{
									btlPbapsPbPullData.returnState = BTL_PBAPS_PB_RETURN_STATE_SUCCESS;
									/* Sent event to return to the stack task */
									bthalStatus = BTHAL_OS_SendEvent(BTHAL_OS_TASK_HANDLE_STACK, OS_EVENT_STACK_PB_TASK);
									BTL_VERIFY_ERR(bthalStatus == BTHAL_STATUS_SUCCESS , BT_STATUS_FAILED, ("BTHAL_OS_SendEvent failed."));
								}
								break;
								
				        }
				        break;	
				                

				    case BTL_PBAPS_PB_XML_END_DECL_STATE:
				        /* Write XML End folder Decl */
				        btlPbapsPbXmlContext.vectors[0] = (U8*)(BTL_PBAPS_PB_XML_END_STR);
				        btlPbapsPbXmlContext.vectors[1] = 0;
				        /* Drop through to next state */
				        btlPbapsPbXmlContext.state++;

				    case BTL_PBAPS_PB_XML_COPY_END_DECL_STATE:
				        if (BtlPbapsPbVectorCopy((const U8 **)&btlPbapsPbXmlContext.vectors[0])) 
						{
							
				            /* Couldn't write complete declaration, were done (with this chunk only) */
							goOn = FALSE;
 							if (stackTask == TRUE)
							{
								/* Data is ready, call continue to keep the requests flowing. */
						    	obStatus = PBAP_ServerContinue(btlPbapsPbserverSession);
								BTL_VERIFY_ERR(OB_STATUS_SUCCESS == obStatus, BT_STATUS_FAILED, ("Server: PBAP server continue failure"));
							}
							else
							{
								btlPbapsPbPullData.returnState = BTL_PBAPS_PB_RETURN_STATE_SUCCESS;
								/* Sent event to return to the stack task */
								bthalStatus = BTHAL_OS_SendEvent(BTHAL_OS_TASK_HANDLE_STACK, OS_EVENT_STACK_PB_TASK);
								BTL_VERIFY_ERR(bthalStatus == BTHAL_STATUS_SUCCESS , BT_STATUS_FAILED, ("BTHAL_OS_SendEvent failed."));
							}
				            break;
				        }

				        /* We're done building XML data*/
						btlPbapsPbPullData.doneReading = TRUE;
						goOn = FALSE;
						if (stackTask == TRUE)
						{
							/* Data is ready, call continue to keep the requests flowing. */
					    	obStatus = PBAP_ServerContinue(btlPbapsPbserverSession);
							BTL_VERIFY_ERR(OB_STATUS_SUCCESS == obStatus, BT_STATUS_FAILED, ("Server: PBAP server continue failure"));
						}
						else
						{
							btlPbapsPbPullData.returnState = BTL_PBAPS_PB_RETURN_STATE_SUCCESS;
							/* Sent event to return to the stack task */
							bthalStatus = BTHAL_OS_SendEvent(BTHAL_OS_TASK_HANDLE_STACK, OS_EVENT_STACK_PB_TASK);
							BTL_VERIFY_ERR(bthalStatus == BTHAL_STATUS_SUCCESS , BT_STATUS_FAILED, ("BTHAL_OS_SendEvent failed."));
						}
			            break;
						
			    } /* End (btlPbapsPbXmlContext.state) */
				break;


				/* We have a race condition when closing the phonebook  asynchronously:
				 * What will happened first - phonebook will close, or last continue event will be called 
				 * in the BTL? The next case is handling the situation. 
				 */
				case BTL_PBAPS_PB_OPER_STATE_DONE_DATA:
					/* In case this function was called as a result of the last continue event in btl_pbap, and the phonebook
					 * was already closed, don't need to build any more data
					 */
					if (btlPbapsPbPullData.sendContinue == TRUE)
					{
						btlPbapsPbPullData.sendContinue = FALSE;
						btlPbapsPbOperState = BTL_PBAPS_PB_OPER_STATE_IDLE;
						btlPbapsPbXmlContext.state = BTL_PBAPS_PB_XML_BEGIN_DECL_STATE;
						/* Set the phonebook path back to what it was before the pull phonebook operation */
						status = BtlPbapsPbSetPbPath(btlPbapsPbPullData.tempPath, BTL_PBAPS_PB_SET_PATH_ABSOLUTE);
						btlPbapsPbPullData.tempPathFlag = FALSE;
						BTL_VERIFY_ERR(BT_STATUS_SUCCESS == status, BT_STATUS_FAILED, ("BtlPbapsPbSetPbPath failed"));
						
						/* Data is ready, call continue to keep the requests flowing. */
				    	obStatus = PBAP_ServerContinue(btlPbapsPbserverSession);
						BTL_VERIFY_ERR(OB_STATUS_SUCCESS == obStatus, BT_STATUS_FAILED, ("Server: PBAP server continue failure"));
						
						
						
					}
					else
						btlPbapsPbPullData.sendContinue = TRUE;
					
					goOn = FALSE;
					break;
					
				
			default:
				BTL_ERR(BT_STATUS_FAILED, ("BtlPbapsPbBuildPbData failed. Wrong operation state"));
				
		} /* End switch (btlPbapsPbOperState) */
				
		
	} /* End while(goOn == TRUE) */
	
	BTL_FUNC_END();
	
	return (status);
}

/*-------------------------------------------------------------------------------
 * BtlPbapsPbGetData()
 */
BtStatus BtlPbapsPbGetData(U8 *dataPtr, U16 *dataLen, BOOL *more)
{
	BtStatus 	status = BT_STATUS_SUCCESS;
	U16 	 	copyLen;
	U16		 	i;
	BthalStatus bthalStatus;
	BTL_FUNC_START("BtlPbapsGetPbData");
	
	BTL_VERIFY_ERR(0 != dataPtr, BT_STATUS_FAILED, ("BtlPbapsPbGetData failed. Null buffer pointer"));
	
	copyLen = (U16)min(*dataLen, btlPbapsPbPullData.chunkOffset);
	OS_MemCopy(dataPtr, btlPbapsPbPullData.chunkDataBuf, copyLen);

	*dataLen = copyLen;
	/* If not all the chunk was requested, fix the chuck for the next chuck request */
	for (i = 0; i < btlPbapsPbPullData.chunkOffset - *dataLen; i++)
		btlPbapsPbPullData.chunkDataBuf[i] = btlPbapsPbPullData.chunkDataBuf[*dataLen + i];
	if (i > 0)
	{
		btlPbapsPbPullData.chunkOffset = i;
		btlPbapsPbPullData.doneReading = FALSE;
	}
	else
	{
		btlPbapsPbPullData.chunkOffset = 0;
	}

	/* Do we have any more data to supply? */
	if (TRUE == btlPbapsPbPullData.doneReading)  
	{
		*more = FALSE;
		btlPbapsPbOperState = BTL_PBAPS_PB_OPER_STATE_DONE_DATA;
		bthalStatus = BTHAL_PB_ClosePb(btlPbapsPbPullData.pbHandle);
		BTL_VERIFY_ERR(BTHAL_STATUS_FAILED != bthalStatus, BT_STATUS_FAILED, ("BTHAL_PB_ClosePb failed"));
		
		/* If PB is synchronous, don't need to wait for the phnebook to close, before sending the last 
		 * OBEX packet 
		 */
		if (BTHAL_STATUS_SUCCESS == bthalStatus)
			btlPbapsPbPullData.sendContinue = TRUE;
				
	}
	else
	{
		*more = TRUE;
	}
	
	BTL_FUNC_END();

	return (status);
}

/*-------------------------------------------------------------------------------
 * BtlPbapsPbSetPbPath()
 */
BtStatus BtlPbapsPbSetPbPath(const char *path, BtlPbapsPbSetPathFlags flags)
{
	BtStatus 				 status = BT_STATUS_SUCCESS;
    BtlPbapsPbPathFlags  	 pbPathFlags;
    U16            			 i, pathLen, len = 0;
	BOOL 					 retVal;
	char 				 tempPbPath[BTL_PBAPS_PB_MAX_PATH_LEN];
	
	BTL_FUNC_START("BtlPbapsPbSetPbPath");

	/* input check */
	if (path != 0)
	{
		BTL_VERIFY_ERR( (OS_StrLen(path) < BTL_PBAPS_PB_MAX_PATH_LEN) , 
					BT_STATUS_FAILED, ("BtlPbapsPbSetPbPath failed. Invalid path"));
		
	}

	if (BTL_PBAPS_PB_SET_PATH_RESET == flags) 
	{
        /* Path Reset */
        OS_MemSet( (U8 *)btlPbapsPbPullData.pbPath, 0, BTL_PBAPS_PB_MAX_PATH_LEN);
    }
	else
	if (BTL_PBAPS_PB_SET_PATH_ABSOLUTE == flags) 
	{
		BTL_VERIFY_ERR((0 != path), 
					BT_STATUS_FAILED, ("BtlPbapsPbSetPbPath failed. Invalid path"));

		OS_StrCpy(tempPbPath, btlPbapsPbPullData.pbPath);
		OS_StrCpy(btlPbapsPbPullData.pbPath, path);
		btlPbapsPbPullData.pbPath[OS_StrLen(path)] = '\0';
	}
	else
	if ((BTL_PBAPS_PB_SET_PATH_LOCAL == flags) || (BTL_PBAPS_PB_SET_PATH_BACKUP == flags))
	{

		OS_StrCpy(tempPbPath, btlPbapsPbPullData.pbPath);
		
	    /* Length of current path */
	    len = OS_StrLen(btlPbapsPbPullData.pbPath);

	    /* Apply backup command if present. */
	    if (flags & BTL_PBAPS_PB_SET_PATH_BACKUP) 
		{

			BTL_VERIFY_ERR(0 != len, BT_STATUS_FAILED, ("Can't backup path. Path is root folder"));
			
	        while(len > 0) {
	            --len;
	            if ((btlPbapsPbPullData.pbPath[len] == '\\') ||
	                (btlPbapsPbPullData.pbPath[len] == '/')) {
	                btlPbapsPbPullData.pbPath[len] = 0;
	                break;
	            }
	            btlPbapsPbPullData.pbPath[len] = 0;
	        }
	    }

	    if ((OS_StrLen(path) > 0) && (len > 0)) {
	        /* Add a folder separator before adding the new path information */
	        btlPbapsPbPullData.pbPath[len++] = '/';
	    }

	    /* Append path at the end of the existing path - being sure to convert
	     * all path separators to use '\' instead of '/' 
	     */
	    pathLen = OS_StrLen(path);
	    for (i = 0; i < pathLen; i++) {
	        if (path[i] == '\\') 
			{
	            /* Convert path separator to '/' */
	            btlPbapsPbPullData.pbPath[len++] = '/';
	        } else 
			{
	            btlPbapsPbPullData.pbPath[len++] = path[i];
	        }
	    }
		btlPbapsPbPullData.pbPath[len] = '\0';
	}

	pbPathFlags = 0;
	
	/* Check that the resulting path is valid, and save the BTHAL path */
	retVal = BtlPbapsPbVerifyPath(btlPbapsPbPullData.pbPath, &pbPathFlags);

	if (TRUE == retVal)
	{
		/* The bthal path flags is a bit different from the localy path flags. */
	    btlPbapsPbPullData.bthalPbPath =  (pbPathFlags);
		if (pbPathFlags & BTL_PBAPS_PB_PATH_SIM)
		{
			btlPbapsPbPullData.bthalPbPath &= ~BTL_PBAPS_PB_PATH_TELECOM;
		}
	
	}
	else
	{
		OS_StrCpy(btlPbapsPbPullData.pbPath, tempPbPath);
		btlPbapsPbPullData.pbPath[OS_StrLen(tempPbPath)] = '\0';
		status = BT_STATUS_FAILED;
	}
	

	BTL_FUNC_END();

    return (status);
}

/*-------------------------------------------------------------------------------
 * BtlPbapsPbGetPbParams()
 */
BtStatus BtlPbapsPbGetPbParams(BOOL pullPbOper, BOOL stackTask)
{	
	BtStatus 		  		 status = BT_STATUS_SUCCESS;
	BthalStatus 	  		 bthalStatus;
	ObStatus				 obStatus;
	BOOL					 goOn = TRUE;
	static PbapSetPbParams 	 reqParams;
	BOOL					 retVal;
	BOOL 					 uselessBool;
	static BOOL				 sendPbSize;
	
	BTL_FUNC_START("BtlPbapsPbGetPbParams");
	
	btlPbapsPbCurOper = BTL_PBAPS_PB_OPER_PROVIDE_PARMS;

	while (goOn == TRUE)
	{

		switch (btlPbapsPbOperState)
		{
			case BTL_PBAPS_PB_OPER_STATE_IDLE:
				sendPbSize = FALSE;
				
				/* Init params */
				OS_MemSet( (U8*) &reqParams, 0 , sizeof(PbapSetPbParams));
								
				/* Save a copy of the path, in order to recreate it after the pull operation finishes */
				OS_StrCpy(btlPbapsPbPullData.tempPath, btlPbapsPbPullData.pbPath);
				btlPbapsPbPullData.tempPath[OS_StrLen(btlPbapsPbPullData.pbPath)] = '\0';
				btlPbapsPbPullData.tempPathFlag = TRUE;
				
				/* Set path, set start ID, check whether pb size should be sent */ 	
				if (TRUE == pullPbOper)
				{
					btlPbapsPbPullData.entryId = btlPbapsPbPullParams.pb.startOffset;
					retVal = BtlPbapsStripExt(btlPbapsPbPullParams.pb.pbName, ".vcf");
					
					/* Error check */
					if (FALSE == retVal)
					{
						BTL_LOG_DEBUG(("BtlPbapsStripExt failed"));
						status = BT_STATUS_INVALID_PARM;
						goOn = FALSE;
						break;
					}
					/* Set the new phonebook path */
					status = BtlPbapsPbSetPbPath(btlPbapsPbPullParams.pb.pbName, BTL_PBAPS_PB_SET_PATH_ABSOLUTE);

					/* Check whether phonebook size should be sent to client */
					if (0 == btlPbapsPbPullParams.pb.maxListCount)
						sendPbSize = TRUE;
				}	
				else
				{
					/* By calling BTHAL_PB_WasPbChanged we reset the phonebook validation at the BTHAL_PB */
					bthalStatus = BTHAL_PB_WasPbChanged(btlPbapsPbPullData.bthalPbPath, (BTHAL_BOOL*)&uselessBool);
					BTL_VERIFY_ERR(BTHAL_STATUS_FAILED != bthalStatus, BT_STATUS_FAILED, ("BTHAL_PB_WasPbChanged failed"));
					
					/* Clear the phonebook changes tracking variable */
					BtlPbapsPbSetChangedStatus(FALSE);
					
					btlPbapsPbPullData.entryId = btlPbapsPbPullParams.listing.listStartOffset;
					status = BtlPbapsPbSetPbPath(btlPbapsPbPullParams.listing.folderName, BTL_PBAPS_PB_SET_PATH_LOCAL);

					if (FALSE == BtlPbapsPbCorrectPullListingPath() )
					{
						status = BT_STATUS_IMPROPER_STATE;
						goOn = FALSE;
						break;
					}
					
					/* Check whether phonebook size should be sent to client */
					if (0 == btlPbapsPbPullParams.listing.maxListCount)
						sendPbSize = TRUE;
					
				}

				/* Error check */
				if (BT_STATUS_SUCCESS != status)
				{
					BTL_LOG_DEBUG(("BtlPbapsPbSetPbPath failed"));
					status = BT_STATUS_INVALID_PARM;
					goOn = FALSE;
					break;
				}
								
				/* Open a phonebook */
				bthalStatus = BTHAL_PB_OpenPb(btlPbapsPbPullData.bthalPbPath, &btlPbapsPbPullData.pbHandle);
				BTL_VERIFY_ERR(BTHAL_STATUS_FAILED != bthalStatus, BT_STATUS_FAILED, ("BTHAL_PB_OpenPb failed"));

				/* Check whether the pb is busy */
				if (BTHAL_STATUS_BUSY == bthalStatus)
				{
					BTL_LOG_DEBUG(("BTHAL_PB_OpenPb is busy"));
					status = BT_STATUS_BUSY;
					goOn = FALSE;
					break;
				}
							
				btlPbapsPbOperState = BTL_PBAPS_PB_OPER_STATE_PB_OPENED;

				if (bthalStatus == BTHAL_STATUS_PENDING)
				{
					status = BT_STATUS_PENDING;
					goOn = FALSE;
				}
				break;
				
			case BTL_PBAPS_PB_OPER_STATE_PB_OPENED:
				/* Ask for new missed calls num only if missed calls phonebook was requested by client */
				if (BTHAL_PB_PATH_MISSED & btlPbapsPbPullData.bthalPbPath)
				{
					/* Get new missed calls */
					bthalStatus =  BTHAL_PB_GetNewMissedCallsNum(btlPbapsPbPullData.pbHandle, &btlPbapsPbPullData.entriesPbNum);
					BTL_VERIFY_ERR(BTHAL_STATUS_FAILED != bthalStatus, BT_STATUS_FAILED, ("BTHAL_PB_GetNewMissedCallsNum failed"));
							
					if (bthalStatus == BTHAL_STATUS_PENDING)
					{
						status = BT_STATUS_PENDING;
						goOn = FALSE;
					}
				}
				
				btlPbapsPbOperState = BTL_PBAPS_PB_OPER_STATE_GET_DATA;
				break;
				
			case BTL_PBAPS_PB_OPER_STATE_GET_DATA:
				/* Set new missed calls num only if missed calls phonebook was requested by client */
				if (BTHAL_PB_PATH_MISSED & btlPbapsPbPullData.bthalPbPath)
					reqParams.newMissedCalls = (U8) btlPbapsPbPullData.entriesPbNum;

				/* Get number of entries in the phonebook */
				bthalStatus =  BTHAL_PB_GetPbEntriesNum(btlPbapsPbPullData.pbHandle, &btlPbapsPbPullData.entriesPbNum);

				if (bthalStatus == BTHAL_STATUS_PENDING)
				{
					status = BT_STATUS_PENDING;
					goOn = FALSE;
				}
				else if (bthalStatus == BTHAL_STATUS_SUCCESS)
				{
					BTL_VERIFY_ERR((btlPbapsPbPullData.entriesPbNum <= BTHAL_PB_MAX_ENTRIES_NUM), 
									BT_STATUS_NO_RESOURCES,
									("Number of entries (%d) exceeds max configured (%d)", 
									btlPbapsPbPullData.entriesPbNum, BTHAL_PB_MAX_ENTRIES_NUM));
				}
				else
				{
					BTL_ERR(BT_STATUS_FAILED, ("BTHAL_PB_GetPbEntriesNum failed"));
				}
									
				btlPbapsPbOperState = BTL_PBAPS_PB_OPER_STATE_DONE_DATA;
				break;

			case BTL_PBAPS_PB_OPER_STATE_DONE_DATA:

				if (btlPbapsPbPullData.entriesPbNum > BTHAL_PB_MAX_ENTRIES_NUM)
				{
					btlPbapsPbPullData.returnState = BTL_PBAPS_PB_RETURN_STATE_NO_RESOURCES;

					BTL_ERR(	BT_STATUS_NO_RESOURCES, 
								("Number of entries (%d) exceeds max configured (%d)", 
								btlPbapsPbPullData.entriesPbNum, BTHAL_PB_MAX_ENTRIES_NUM));
				}
				
				/* Set for phonebook size only if needed */
				if (TRUE == sendPbSize)
					reqParams.phonebookSize = btlPbapsPbPullData.entriesPbNum;

				/* Set required params accroding to client request */
				reqParams.flags = PBAP_PARAM_NONE;
				if (TRUE == sendPbSize)
					reqParams.flags |= (PBAP_PARAM_PB_SIZE);
				if (BTHAL_PB_PATH_MISSED & btlPbapsPbPullData.bthalPbPath)
					reqParams.flags |= (PBAP_PARAM_NEW_MISSED_CALLS);

				obStatus = PBAP_ServerSetPbParams(btlPbapsPbserverSession, &reqParams);
				BTL_VERIFY_ERR(OB_STATUS_SUCCESS == obStatus, BT_STATUS_FAILED, ("PBAP_ServerSetPbParams failed"));
				

				/* We finished with the provide params operation */
				btlPbapsPbOperState = BTL_PBAPS_PB_OPER_STATE_IDLE;
				btlPbapsPbCurOper = BTL_PBAPS_PB_OPER_NONE;
				goOn = FALSE;	        

				if (stackTask == TRUE)
				{
					/* Data is ready, call continue to keep the requests flowing. */
			    	obStatus = PBAP_ServerContinue(btlPbapsPbserverSession);
					BTL_VERIFY_ERR(OB_STATUS_SUCCESS == obStatus, BT_STATUS_FAILED, ("Server: PBAP server continue failure"));
				}
				else
				{
					btlPbapsPbPullData.returnState = BTL_PBAPS_PB_RETURN_STATE_SUCCESS;
					/* Sent event to return to the stack task */
					bthalStatus = BTHAL_OS_SendEvent(BTHAL_OS_TASK_HANDLE_STACK, OS_EVENT_STACK_PB_TASK);
					BTL_VERIFY_ERR(bthalStatus == BTHAL_STATUS_SUCCESS , BT_STATUS_FAILED, ("BTHAL_OS_SendEvent failed."));
				}
				break;
		
						
		}
	}
	BTL_FUNC_END();

    return (status);
}


/*-------------------------------------------------------------------------------
 * BtlPbapsPbSetPullPbParams()
 */
void BtlPbapsPbSetPullPbParams(PbapServerCallbackParms *Parms)
{
	U16  len;

	BTL_FUNC_START("BtlPbapsPbSetPullPbParams");
	
	OS_MemCopy((U8 *)btlPbapsPbPullParams.pb.filter.byte, (U8 *)&Parms->u.paramsRx.filter.byte, PBAP_FILTER_SIZE);
    btlPbapsPbPullParams.pb.format = Parms->u.paramsRx.format;
    btlPbapsPbPullParams.pb.startOffset = Parms->u.paramsRx.listStartOffset;
    btlPbapsPbPullParams.pb.maxListCount = Parms->u.paramsRx.maxListCount;

	if ((Parms->u.paramsRx.name) && ( (len = OS_StrLen((char*)Parms->u.paramsRx.name) ) > 0))
	/* In case name parameter was received. If not it will be updated in BtlPbapsPbUpdateNameParam 
	 * funciton 
	 */
	{
		if (BTL_PBAPS_PB_MAX_PATH_LEN <= len)
		{
			BTL_ERR_NORET(("BtlPbapsPbSetPullPbParams failed. Too long pb name"));
			return;
		}
		OS_StrnCpy(btlPbapsPbPullParams.pb.pbName, (char*)Parms->u.paramsRx.name, len);
		btlPbapsPbPullParams.pb.pbName[len] = '\0';
	}
	else
		btlPbapsPbPullParams.pb.pbName[0] = '\0';

	BTL_FUNC_END();
}

/*-------------------------------------------------------------------------------
 * BtlPbapsPbSetPullListParams()
 */
void BtlPbapsPbSetPullListParams(PbapServerCallbackParms *Parms)
{
	 U16  len;

	 BTL_FUNC_START("BtlPbapsPbSetPullListParams");
	 
	 btlPbapsPbPullParams.listing.order = Parms->u.paramsRx.order;
     btlPbapsPbPullParams.listing.searchAttrib = Parms->u.paramsRx.searchAttrib;
	 OS_StrnCpyUtf8(btlPbapsPbPullParams.listing.searchValue, Parms->u.paramsRx.searchValue, Parms->u.paramsRx.searchValueLen);
     btlPbapsPbPullParams.listing.searchValue[Parms->u.paramsRx.searchValueLen] = '\0';
     btlPbapsPbPullParams.listing.listStartOffset = Parms->u.paramsRx.listStartOffset;
     btlPbapsPbPullParams.listing.maxListCount = Parms->u.paramsRx.maxListCount;
 	
    if ((Parms->u.paramsRx.name) && ( (len = OS_StrLen((char*)Parms->u.paramsRx.name) ) > 0))
	/* In case name parameter was received. If not it will be updated in BtlPbapsPbUpdateNameParam 
	 * funciton 
	 */
	{
	     if (BTL_PBAPS_PB_MAX_PATH_LEN <= len)
		 {
			BTL_ERR_NORET(("BtlPbapsPbSetPullListParams failed. Too long pb name"));
			return;
		 }
	     OS_StrnCpy(btlPbapsPbPullParams.listing.folderName, (char*)Parms->u.paramsRx.name, len);
		 btlPbapsPbPullParams.listing.folderName[len] = '\0';
 	}
	else
	     btlPbapsPbPullParams.listing.folderName[0] = '\0';


	 BTL_FUNC_END();
		            
}

/*-------------------------------------------------------------------------------
 * BtlPbapsPbSetPullEntryParams()
 */
void BtlPbapsPbSetPullEntryParams(PbapServerCallbackParms *Parms)
{
	U16  len;
	
	BTL_FUNC_START("BtlPbapsPbSetPullEntryParams");
	
	OS_MemCopy((U8 *)&btlPbapsPbPullParams.entry.filter.byte, (U8 *)&Parms->u.paramsRx.filter.byte, sizeof(PbapVcardFilter));
	
    btlPbapsPbPullParams.entry.format = Parms->u.paramsRx.format;

	if ((Parms->u.paramsRx.name) && ( (len = OS_StrLen((char*)Parms->u.paramsRx.name) ) > 0))
	/* In case name parameter was received. If not it will be updated in BtlPbapsPbUpdateNameParam 
	 * funciton 
	 */
	{
	    if (BTL_PBAPS_PB_MAX_ENTRY_INDEX_LEN <= len)
		{
			BTL_ERR_NORET(("BtlPbapsPbSetPullListParams failed. Too long pb name"));
			return;
		}

	    OS_StrnCpy(btlPbapsPbPullParams.entry.entryName, (char*)Parms->u.paramsRx.name, len);
	    btlPbapsPbPullParams.entry.entryName[len] = '\0';
	}
	else
	    btlPbapsPbPullParams.entry.entryName[0] = '\0';
	
	/* We're in a beginning of pull entry operation. That's a good time to init params */
	btlPbapsPbPullData.doneReading = FALSE;
	btlPbapsPbPullData.sendContinue = FALSE;
	btlPbapsPbPullData.chunkOffset = 0;
	
	BTL_FUNC_END();
}

/*-------------------------------------------------------------------------------
 * BtlPbapsPbSetPullEntryParams()
 */
void BtlPbapsPbUpdateNameParam(PbapOp oper, char * nameParam)
{
	U16  	  len;
	char	 *stringPtr;
	BtStatus  status = BT_STATUS_SUCCESS;
	
	BTL_FUNC_START("BtlPbapsPbUpdateNameParam");

	switch (oper)
	{
		case PBAPOP_PULL_PHONEBOOK:
			/* Need to update name only if it wasn't received before */
			if (0 != OS_StrLen(btlPbapsPbPullParams.pb.pbName))
			{
				BTL_RET(BT_STATUS_SUCCESS);
			}
			len = OS_StrLen(nameParam);
			if (BTL_PBAPS_PB_MAX_PATH_LEN <= len)
			{
				BTL_ERR(BT_STATUS_SUCCESS,("BtlPbapsPbUpdateNameParam failed. Too long pb name"));
			}
			stringPtr = btlPbapsPbPullParams.pb.pbName;
			break;
			
		case PBAPOP_PULL_VCARD_LISTING:
			/* Need to update name only if it wasn't received before */
			if (0 != OS_StrLen(btlPbapsPbPullParams.listing.folderName)) 
			{
				BTL_RET(BT_STATUS_SUCCESS);
			}
			len = OS_StrLen(nameParam);
			if (BTL_PBAPS_PB_MAX_PATH_LEN <= len)
			{
				BTL_ERR(BT_STATUS_SUCCESS,("BtlPbapsPbUpdateNameParam failed. Too long pb name"));
			}
			stringPtr = btlPbapsPbPullParams.listing.folderName;
			break;
			
		case PBAPOP_PULL_VCARD_ENTRY:
			/* Need to update name only if it wasn't received before */
			if (0 != OS_StrLen(btlPbapsPbPullParams.entry.entryName))  
			{
				BTL_RET(BT_STATUS_SUCCESS);
			}
			len = OS_StrLen(nameParam);
			if (BTL_PBAPS_PB_MAX_ENTRY_INDEX_LEN <= len)
			{
				BTL_ERR(BT_STATUS_SUCCESS,("BtlPbapsPbUpdateNameParam failed. Too long pb name"));
			}
			stringPtr = btlPbapsPbPullParams.entry.entryName;
		/* We're in a beginning of pull entry operation. That's a good time to init params */
			btlPbapsPbPullData.doneReading = FALSE;
			btlPbapsPbPullData.sendContinue = FALSE;
			btlPbapsPbPullData.chunkOffset = 0;
			
			break;

		default:
			BTL_ERR(BT_STATUS_SUCCESS,("BtlPbapsPbUpdateNameParam failed. Unexpected operation"));

	}

	stringPtr[0] = '\0';
    OS_StrnCpy(stringPtr, nameParam, len);
    stringPtr[len] = '\0';

	BTL_FUNC_END();
}

/*-------------------------------------------------------------------------------
 * BtlPbapsPbInitParams()
 */
void BtlPbapsPbInitParams()
{	
	 BTL_FUNC_START("BtlPbapsPbInitParams");
	 
	 OS_MemSet((U8 *)&btlPbapsPbPullParams.pb, 0, sizeof(BtlPbapsPbParams));
	 btlPbapsPbPullParams.pb.maxListCount = 0xFFFF; /* Default, in case the App Params header is not sent */
	 OS_MemSet((U8 *)&btlPbapsPbPullParams.listing, 0, sizeof(BtlPbapsPbFolderListingParams));
	 btlPbapsPbPullParams.listing.maxListCount = 0xFFFF; /* Default, in case the App Params header is not sent */
	 OS_MemSet((U8 *)&btlPbapsPbPullParams.entry, 0, sizeof(BtlPbapsPbVcardEntryParams));
	 
	 BTL_FUNC_END();
}

/*-------------------------------------------------------------------------------
 * BtlPbapsPbCallback()
 *
 *		Internal callback for handling BTHAL_PB events in BTL layer.
 *		
 * Type:
 *		Synchronous
 *
 * Parameters:
 *		callbackParams [in] - Callback parameters of the event.
 *
 * Returns:
 *		void.
 */
static void BtlPbapsPbCallback(const BthalPbCallbackParms *callbackParams)
{
	BtStatus 			 status = BT_STATUS_SUCCESS;
	BthalPbCallbackParms params;
	BthalStatus			 bthalStatus;  

	BTL_FUNC_START("BtlPbapsPbCallback");
  
	if (FALSE == BtlPbapsPbVarifyCallbackParams(callbackParams))
	{
		status = BT_STATUS_FAILED;
		BTL_LOG_ERROR(("BtlPbapsPbVarifyCallbackParams failed"));
	}

	if (BT_STATUS_FAILED != status)
	{
		OS_MemCopy((U8 *) &params, (U8 *) callbackParams, sizeof(BthalPbCallbackParms));

		/* Verify no error in the phonebook */
		if (BTHAL_PB_EVENT_PB_ERROR == callbackParams->event)
		{
			status = BT_STATUS_FAILED;
			BTL_LOG_ERROR(("PB retruned with an error - pb failed."));
		}
	}
	
	if (BT_STATUS_FAILED != status)
	{
		switch (btlPbapsPbCurOper)
		{
			case BTL_PBAPS_PB_OPER_PULL_PB:
				status = BtlPbapsPbBuildPbData(FALSE);
				break;
			case BTL_PBAPS_PB_OPER_PULL_VCARD_LIST:
				status = BtlPbapsPbBuildFolderListingData(FALSE);
				break;
			case BTL_PBAPS_PB_OPER_PULL_ENTRY:
				status = BtlPbapsPbBuildEntryData(FALSE);
				break;
			case BTL_PBAPS_PB_OPER_PROVIDE_PARMS:
				/* First argument of BtlPbapsPbGetPbParams is don't care at this point */
				status = BtlPbapsPbGetPbParams(0, FALSE);
				break;
			case BTL_PBAPS_PB_OPER_REMOTE_ABORT:
				btlPbapsPbPullData.returnState = BTL_PBAPS_PB_RETURN_STATE_SUCCESS;
				/* Sent event to return to the stack task */
				bthalStatus = BTHAL_OS_SendEvent(BTHAL_OS_TASK_HANDLE_STACK, OS_EVENT_STACK_PB_TASK);
				BTL_VERIFY_ERR(bthalStatus == BTHAL_STATUS_SUCCESS , BT_STATUS_FAILED, ("BTHAL_OS_SendEvent failed."));
			case BTL_PBAPS_PB_OPER_USER_ABORT:
				btlPbapsPbCurOper = BTL_PBAPS_PB_OPER_NONE;
				break;
				
			default:
				status = BT_STATUS_FAILED;
				
		}
	}
	else
	{
		/* Reset the state of the operation */
		btlPbapsPbCurOper = BTL_PBAPS_PB_OPER_NONE;
	}

	/* In case of a BTHAL_PB failure, need to abort the PBAP operation */
	if (	(BT_STATUS_FAILED == status )  ||
			(BT_STATUS_NO_RESOURCES == status) || 
			(BT_STATUS_NOT_FOUND == status)			)
	{
		switch (status)
		{
			case BT_STATUS_NO_RESOURCES:
				btlPbapsPbPullData.returnState = BTL_PBAPS_PB_RETURN_STATE_NO_RESOURCES;
				break;
			case BT_STATUS_NOT_FOUND:
				btlPbapsPbPullData.returnState = BTL_PBAPS_PB_RETURN_STATE_NOT_FOUND;
				break;
			case BT_STATUS_FAILED:
			default:
				btlPbapsPbPullData.returnState = BTL_PBAPS_PB_RETURN_STATE_ERROR;
				break;
		}
		
		/* Sent event to return to the stack task */
		bthalStatus = BTHAL_OS_SendEvent(BTHAL_OS_TASK_HANDLE_STACK, OS_EVENT_STACK_PB_TASK);
		BTL_VERIFY_ERR(bthalStatus == BTHAL_STATUS_SUCCESS , BT_STATUS_FAILED, ("BTHAL_OS_SendEvent failed."));
	}

	BTL_FUNC_END();
	
}

/*---------------------------------------------------------------------------
 * BtlPbapsStripExt()
 *
 *
 *   Removes the specified extension from the string, if it is 
 *   present.
 *
 * Type:
 *		Synchronous 
 *
 * Parameters:
 *
 *		name - String to modify driectly
 *      ext - Extension (i.e. ".vcf" to remove)
 *
 * Return:    TRUE - extension was successfully stripped
 *            FALSE - extension was not present
 */
static BOOL BtlPbapsStripExt(char *name, const char *ext)
{
    U16 len, extLen;

    /* Measure lengths */
    len = OS_StrLen(name);
    extLen = OS_StrLen(ext);

    /* Strip if possible */
    if (len > extLen && 
        BtlPbapsPbStrNCmp(name + len - extLen, ext, extLen) == 0)
    {
        name[len - extLen] = 0;
        return TRUE;
    }

	return FALSE;
}

/*---------------------------------------------------------------------------
 * BtlPbapsBuildXmlData
 *
 *   This function builds the body of the XML vCard folder listing.
 *
 * Parameters:
 *	 void
 * 
 * Return: 
 *
 *   BtlPbapsPbXmlRetVal - See definitions.
 *
 */
static BtlPbapsPbXmlRetVal BtlPbapsBuildXmlData()
{
	BthalStatus   		    bthalStatus;
    BtStatus 				status = BT_STATUS_SUCCESS;
	U16						i;
	BtlPbapsPbXmlRetVal		retVal = BTL_PBAPS_PB_XML_STATUS_MORE_DATA;
	BOOL 				    goOn = TRUE;	

	BTL_FUNC_START("BtlPbapsBuildXmlData");

    while ( (btlPbapsPbPullData.chunkOffset < (U16)BTL_PBAPS_PB_MAX_DATA_CHUNK) && (goOn))
	{
        switch (btlPbapsPbXmlContext.state) 
		{
	        case BTL_PBAPS_PB_XML_INIT_STATE:
				/* Handle the special case when there are no more entries to pull */

				BTL_VERIFY_FATAL(	btlPbapsPbPullData.entryId < BTHAL_PB_MAX_ENTRIES_NUM, 
									BT_STATUS_INTERNAL_ERROR, 
									("Entry Id (%d) Out of Bounds (%d)", btlPbapsPbPullData.entryId, BTHAL_PB_MAX_ENTRIES_NUM));

				if (btlPbapsPbPullData.entryId >= btlPbapsPbPullData.entriesPbNum) 
				{
					btlPbapsPbPullData.entriesList.entriesNum = 0;
					retVal = BTL_PBAPS_PB_XML_STATUS_DONE_DATA;
					goOn = FALSE;
					break;
				}
				/* Setting parameters for the BTHAL_PB_BuildEntriesList function */
				btlPbapsPbPullData.listSearch.searchAttribute = btlPbapsPbPullParams.listing.searchAttrib;
				OS_StrCpyUtf8(btlPbapsPbPullData.listSearch.searchValue, btlPbapsPbPullParams.listing.searchValue);
				btlPbapsPbPullData.listSortOrder = btlPbapsPbPullParams.listing.order;

				/* Getting the needed list from the phonebook */
				bthalStatus = BTHAL_PB_BuildEntriesList(btlPbapsPbPullData.pbHandle, 
												btlPbapsPbPullData.listSearch, 
												btlPbapsPbPullData.listSortOrder,
												&btlPbapsPbPullData.entriesList);

	           	BTL_VERIFY_ERR(BTHAL_STATUS_FAILED != bthalStatus, BT_STATUS_FAILED, ("BTHAL_PB_BuildEntriesList failed"));
				
				/* If no search/sort is supported by the phonebook, just pull the entries by index order */
				if (BTHAL_STATUS_NOT_SUPPORTED == bthalStatus)
				{
					for(i = 0; i < 	btlPbapsPbPullData.entriesPbNum; i++)
						btlPbapsPbPullData.entriesList.entryId[i] = i;
					btlPbapsPbPullData.entriesList.entriesNum = btlPbapsPbPullData.entriesPbNum;
				}
				/* Save the path of which the entries list was built opun. 
				 * The following pull entry operations are expected to be from 
				 * that path 
				 */
				btlPbapsPbPullData.entriesListPath = btlPbapsPbPullData.bthalPbPath;

				btlPbapsPbXmlContext.state = BTL_PBAPS_PB_XML_NEXT_STATE;
	
				/* If we work asychronously */
				if (bthalStatus == BTHAL_STATUS_PENDING)
				{
					goOn = FALSE;
					retVal = BTL_PBAPS_PB_XML_STATUS_PENDING;
				}
				break;

	        case BTL_PBAPS_PB_XML_OBJECT_SETUP_STATE:

				/* Init vectors buffers */
				btlPbapsPbXmlContext.entryIdVec[0] = '\0';
				btlPbapsPbXmlContext.entryNameVec[0] = '\0';
				
				/* Setting the BtlPbapsXmlContext vector fileds */
				/* Setting XML line header vector */
				btlPbapsPbXmlContext.vectors[0] = (U8*)"   <card handle=\""; 

				/* Setting VCF entry number vector*/
				BtlPbapsPbHtoA( (U32)btlPbapsPbPullData.entryId, btlPbapsPbXmlContext.entryIdVec);
				OS_StrCat((char*)btlPbapsPbXmlContext.entryIdVec , ".vcf");
				btlPbapsPbXmlContext.vectors[1] = btlPbapsPbXmlContext.entryIdVec;

				/* Setting VCF name vector */
				OS_StrCpyUtf8(btlPbapsPbXmlContext.entryNameVec, btlPbapsPbPullData.entryName.lastName);
				OS_StrCatUtf8(btlPbapsPbXmlContext.entryNameVec, ";");
				OS_StrCatUtf8(btlPbapsPbXmlContext.entryNameVec, btlPbapsPbPullData.entryName.firstName);
				OS_StrCatUtf8(btlPbapsPbXmlContext.entryNameVec, ";");
				OS_StrCatUtf8(btlPbapsPbXmlContext.entryNameVec, btlPbapsPbPullData.entryName.middleName);
				OS_StrCatUtf8(btlPbapsPbXmlContext.entryNameVec, ";");
				OS_StrCatUtf8(btlPbapsPbXmlContext.entryNameVec, btlPbapsPbPullData.entryName.prefix);
				OS_StrCatUtf8(btlPbapsPbXmlContext.entryNameVec, ";");
				OS_StrCatUtf8(btlPbapsPbXmlContext.entryNameVec, btlPbapsPbPullData.entryName.suffix);

				btlPbapsPbXmlContext.vectors[2] = (U8*)"\" name = \"";
	            btlPbapsPbXmlContext.vectors[3] = btlPbapsPbXmlContext.entryNameVec;
				
	            /* Setting end of XML line */
			    btlPbapsPbXmlContext.vectors[4] = (U8*)"\"/>\n";
	            btlPbapsPbXmlContext.vectors[5] = 0;
	            btlPbapsPbXmlContext.state = BTL_PBAPS_PB_XML_COPY_OBJECT_STATE;
				btlPbapsPbPullData.entryId++;
				btlPbapsPbPullData.entriesCount++;
	            break;

	        default:
	            BTL_VERIFY_ERR((btlPbapsPbXmlContext.state >= BTL_PBAPS_PB_XML_COPY_OBJECT_STATE && 
							    btlPbapsPbXmlContext.state < BTL_PBAPS_PB_XML_COPY_OBJECT_STATE + BTL_PBAPS_PB_NUM_VECTORS), 
					BT_STATUS_FAILED, ("BtlPbapsBuildXmlData failed. Wrong XML context state"));


	            if (btlPbapsPbXmlContext.vectors[btlPbapsPbXmlContext.state - BTL_PBAPS_PB_XML_COPY_OBJECT_STATE]) 
				{
	                if (BtlPbapsPbVectorCopy((const U8 **)&btlPbapsPbXmlContext.vectors[btlPbapsPbXmlContext.state - BTL_PBAPS_PB_XML_COPY_OBJECT_STATE]))
					{
	                    break;
	                }
					btlPbapsPbXmlContext.state++;
					break;
				          
	            }
				/* We're done with this vector set.
				 * Now need to read another entry name, or finish 
				 */
				if (  (btlPbapsPbPullData.entriesCount == btlPbapsPbPullParams.listing.maxListCount) || 
				      (btlPbapsPbPullData.entryId == btlPbapsPbPullData.entriesPbNum)                ) 
				{ 
					goOn = FALSE;
					retVal = BTL_PBAPS_PB_XML_STATUS_DONE_DATA;
	                break;
				}
	            /* To read another entry name, drop into next state */

	        case BTL_PBAPS_PB_XML_NEXT_STATE:
				/* We must have another enrty in the list, otherwise we shouldn't have gotten here */
				/* getting an entry from the list */
				OS_MemSet( (U8 *) &btlPbapsPbPullData.entryName, 0, sizeof(BthalPbEntryName));
				bthalStatus = BTHAL_PB_GetListedEntryName(btlPbapsPbPullData.pbHandle, 
														  &btlPbapsPbPullData.entryName,
														  btlPbapsPbPullData.entriesList.entryId[btlPbapsPbPullData.entryId]);
				BTL_VERIFY_ERR(BTHAL_STATUS_FAILED != bthalStatus, BT_STATUS_FAILED, ("BTHAL_PB_BuildEntriesList failed"));
				BTL_VERIFY_ERR(BTHAL_STATUS_NOT_FOUND != bthalStatus, BT_STATUS_FAILED, ("BTHAL_PB_BuildEntriesList failed - entry not found"));
				btlPbapsPbXmlContext.state = BTL_PBAPS_PB_XML_OBJECT_SETUP_STATE;
				
	            if (bthalStatus == BTHAL_STATUS_PENDING)
				{
	                /* Done, no more records exist */
	                goOn = FALSE;
					retVal = BTL_PBAPS_PB_XML_STATUS_PENDING;
	            } 
				else if (bthalStatus == BTHAL_STATUS_SUCCESS)
	            {
	                /* An entry was found */
	                goOn = TRUE;
	            }
	            break;
        }
    }

    
	BTL_FUNC_END();
	
	if (status == BT_STATUS_FAILED)
			retVal =  BTL_PBAPS_PB_XML_STATUS_FAILED;

    return retVal;
}

/*---------------------------------------------------------------------------
 * BtlPbapsPbVectorCopy
 *
 * 	  Internal string copy function copies the entire source string or
 *    up to 'space' bytes into 'dest'. It moves both buffer pointers
 *    and decrements the buffer length as it copies.
 *
 * Parameters:
 *		dest [in/out] - Destenation buffer pointer to pointer.
 *	
 *		source [in/out] - Source buffer pointer to pointer.
 *
 *		space [in/out] - Max length to copy.
 *
 * Return:
 *		 Zero if entire string was copied. Non-zero otherwise
 *
 */
static U8 BtlPbapsPbVectorCopy(const U8 **sourceVector)
{
	BTL_FUNC_START("BtlPbapsPbVectorCopy");
    while ((0 != **sourceVector) && (btlPbapsPbPullData.chunkOffset < (U16)BTL_PBAPS_PB_MAX_DATA_CHUNK))
	{
        btlPbapsPbPullData.chunkDataBuf[btlPbapsPbPullData.chunkOffset]  = *(*sourceVector)++;
		btlPbapsPbPullData.chunkOffset++;
	}
			

	BTL_FUNC_END();
    return **sourceVector;
}

/*---------------------------------------------------------------------------
 *    BtlPbapsPbHtoA        
 *
 *   Converts an unsigned 32-bit value to a base-16 number string.
 *
 * Parameters:
 *		
 *		val [in] - Value to convert
 *
 *		out [out] - Output
 *
 * Return:    void.
 */
static void BtlPbapsPbHtoA(U32 val, U8 *out)
{
    U8  rem;
    U8  str[11], *dest = str;

	BTL_FUNC_START("BtlPbapsPbHtoA");
    if (val == 0)
        *dest++ = '0';

    while (val) {
        rem = (U8) (val%16);
        if (rem <= 9) {
            /* Handle the numbers 0-9 */
	        *dest++ = (U8)(rem + '0');
        } else {
            /* Handle the letters a-f */
            rem -= 10;
            *dest++ = (U8)(rem + 'a');
        }
        val = val/16;
    }

    while (--dest >= str)
        *out++ = *dest;

    *out = 0;

	BTL_FUNC_END();
}


/*---------------------------------------------------------------------------
 * BtlPbapsPbVerifyPath
 *
 * 	  Checks path information to verify that it corresponds to a
 *    valid PBAP virtual folder structure path.
 *
 * Parameters:
 *
 *		path [in] - Path input
 *	
 *		bthalPbPath [out] - The phonebook path in BTHAL_PB terms
 *
 * Return:
 *
 *		TRUE - path is valid
 *      FALSE - path is invalid
 *
 */
static BOOL BtlPbapsPbVerifyPath(char *path, BtlPbapsPbPathFlags *bthalPbPath)
{
    BOOL           		   valid = TRUE;
	char            	  *curPos;
    U16           	       len, strLen;
    BthalPbPhonebookType   localPath;
	
	BTL_FUNC_START("BtlPbapsPbVerifyPath");

    len = 0;
    localPath = *bthalPbPath;
	curPos = (char *)path;
    strLen = OS_StrLen(path);

	while ((strLen > 0) && (len <= strLen) && (valid) ) 
	{
		/* Make sure only the supported path information is attempted */
        if ((path[len] == '\\') ||
            (path[len] == '/') ||
            (len == strLen)) {
			/* Compare this section against our supported paths - 
			 * "telecom", "pb", "ich", "och", "mch", "cch" and
             * check for multiple paths of the same name as well.
			 */
            if ((BtlPbapsPbStrNCmp(curPos, "telecom", 7) == 0) && 
                ((localPath & (BTL_PBAPS_PB_PATH_TELECOM|BTL_PBAPS_PB_PATH_PHONEBOOK|BTL_PBAPS_PB_PATH_INCOMING|
                           BTL_PBAPS_PB_PATH_OUTGOING|BTL_PBAPS_PB_PATH_MISSED|BTL_PBAPS_PB_PATH_COMBINED)) == 0)) 
            {
                /* May occur after SIM path.
                 * Must occur prior to Pb, Ich, Och, Mch, and Cch paths. 
                 */
                localPath |= BTL_PBAPS_PB_PATH_TELECOM;
            } 
			else if ((BtlPbapsPbStrNCmp(curPos, "SIM1", 4) == 0) && (localPath == 0)) 
			{
                /* Must occur prior to all path information. */
                localPath |= BTL_PBAPS_PB_PATH_SIM;
            } 
			else if ((BtlPbapsPbStrNCmp(curPos, "pb", 2) == 0) &&
                       ((localPath & BTL_PBAPS_PB_PATH_TELECOM) != 0) &&
                       ((localPath & (BTL_PBAPS_PB_PATH_PHONEBOOK|BTL_PBAPS_PB_PATH_INCOMING|BTL_PBAPS_PB_PATH_OUTGOING|
                                  BTL_PBAPS_PB_PATH_MISSED|BTL_PBAPS_PB_PATH_COMBINED)) == 0)) 
            {
                /* Must occur after Telecom path.  SIM path may or may not exist. 
                 * Must never occur after Ich, Och, Mch, or Cch paths.
                 */
                localPath |= BTL_PBAPS_PB_PATH_PHONEBOOK;
            } 
			else if ((BtlPbapsPbStrNCmp(curPos, "ich", 3) == 0)  &&
                       ((localPath & BTL_PBAPS_PB_PATH_TELECOM) != 0) &&
                       ((localPath & (BTL_PBAPS_PB_PATH_PHONEBOOK|BTL_PBAPS_PB_PATH_INCOMING|BTL_PBAPS_PB_PATH_OUTGOING|
                                  BTL_PBAPS_PB_PATH_MISSED|BTL_PBAPS_PB_PATH_COMBINED)) == 0)) 
            {
                /* Must occur after Telecom path.  SIM path may or may not exist. 
                 * Must never occur after Pb, Och, Mch, or Cch paths.
                 */
                localPath |= BTL_PBAPS_PB_PATH_INCOMING;
            } else if ((BtlPbapsPbStrNCmp(curPos, "och", 3) == 0) &&
                       ((localPath & BTL_PBAPS_PB_PATH_TELECOM) != 0) &&
                       ((localPath & (BTL_PBAPS_PB_PATH_PHONEBOOK|BTL_PBAPS_PB_PATH_INCOMING|BTL_PBAPS_PB_PATH_OUTGOING|
                                  BTL_PBAPS_PB_PATH_MISSED|BTL_PBAPS_PB_PATH_COMBINED)) == 0)) 
            {
                /* Must occur after Telecom path.  SIM path may or may not exist. 
                 * Must never occur after Pb, Ich, Mch, or Cch paths.
                 */
                localPath |= BTL_PBAPS_PB_PATH_OUTGOING;
            } 
			else if ((BtlPbapsPbStrNCmp(curPos, "mch", 3) == 0) &&
                       ((localPath & BTL_PBAPS_PB_PATH_TELECOM) != 0) &&
                       ((localPath & (BTL_PBAPS_PB_PATH_PHONEBOOK|BTL_PBAPS_PB_PATH_INCOMING|BTL_PBAPS_PB_PATH_OUTGOING|
                                  BTL_PBAPS_PB_PATH_MISSED|BTL_PBAPS_PB_PATH_COMBINED)) == 0)) 
            {
                /* Must occur after Telecom path.  SIM path may or may not exist. 
                 * Must never occur after Pb, Ich, Och, or Cch paths.
                 */
                localPath |= BTL_PBAPS_PB_PATH_MISSED;
            } 
			else if ((BtlPbapsPbStrNCmp(curPos, "cch", 3) == 0) &&
                       ((localPath & BTL_PBAPS_PB_PATH_TELECOM) != 0) &&
                       ((localPath & (BTL_PBAPS_PB_PATH_PHONEBOOK|BTL_PBAPS_PB_PATH_INCOMING|BTL_PBAPS_PB_PATH_OUTGOING|
                                  BTL_PBAPS_PB_PATH_MISSED|BTL_PBAPS_PB_PATH_COMBINED)) == 0)) 
            {
                /* Must occur after Telecom path.  SIM path may or may not exist. 
                 * Must never occur after Pb, Ich, Och, or Mch, paths.
                 */
                localPath |= BTL_PBAPS_PB_PATH_COMBINED;
            } 
			else 
			{
                valid = FALSE;
            }

			/* Matches a supported path, this is okay */
			curPos = (char *)(path + len + 1);
        }		
		len++;
	}

    *bthalPbPath = localPath;

	BTL_FUNC_END();

	return valid;
}

static BOOL BtlPbapsPbVarifyCallbackParams(const BthalPbCallbackParms *callbackParams)
{
	BOOL retVal = TRUE;
	
	switch (callbackParams->event)
	{
		/* Yoni ^^ - Check event parameters. TBD */
	}	

	return retVal;
}

/*---------------------------------------------------------------------------
 * BtlPbapsPbCorrectPullListingPath
 *
 * 	  Checks whether the bthal pb path is a valid for a pull vCard listing operation.
 *	  If it is not valid, the opration should be aborted with the Not Acceptable return value. 
 *
 * Parameters:
 *
 *		None
 *
 * Return:
 *
 *		TRUE - path is a valid pull vCard listing path
 *      FALSE - path is an invalid pull vCard listing path
 *
 */
static BOOL BtlPbapsPbCorrectPullListingPath()
{
	
	switch (btlPbapsPbPullData.bthalPbPath)
	{
		/* Check whether the path is in one of the main pb folders */
		case 5:
		case 9:
		case 17:
		case 33:
		case 65:
			
		/* Check whether the path is in one of the SIM folders */	
		case 6:
		case 10:
		case 18:
		case 34:
		case 66:
			return TRUE;

		default:
			return FALSE;
		
	}
}

/*---------------------------------------------------------------------------
 * BtlPbapsPbStrNCmp()
 *
 *     Compares N first charachters of two strings for equality
 *
 * Parameters:
 *     Str1 - String to compare.
 *     Str2 - String to compare.
 *
 * Returns:
 *     Zero - If strings match.
 *     Non-Zero - If strings do not match.
 */
U8 BtlPbapsPbStrNCmp(const char *Str1, const char *Str2, U16 num)
{
	U16 i = 0;
	
    while ((*Str1 == *Str2) && (i < num)) {
        if (*Str1 == 0 || *Str2 == 0) {
            break;
        }
        Str1++;
        Str2++;
		i++;
    }

    /* Return zero on success, just like the ANSI strcmp() */
    if (i == num)
        return 0;
	if (*Str1 == *Str2)
		return 0;
	
    return 1;
}

/*---------------------------------------------------------------------------
 * BtlPbapsPbProcessContinue
 *
 * 	  	This funciton continues the PBAP operation in the stack task, and calls the server PBAP_ServerContinue
 *		function. It is being called in the context of the stack, after  an event was sent to the stack 
 *		in the context of the device's phonebook, using the BTHAL_OS_SendEvent function.
 *		In case of an error in the phonebook, the function invokes 		
 *		
 *
 * Parameters:
 *
 *		path [in] - Path input
 *	
 *		bthalPbPath [out] - The phonebook path in BTHAL_PB terms
 *
 * Return:
 *
 *		TRUE - path is valid
 *      FALSE - path is invalid
 *
 */
void BtlPbapsPbProcessContinue()
{	
	ObStatus				 obStatus = OB_STATUS_FAILED;
	BtStatus      		     status = BT_STATUS_SUCCESS;
	
	BTL_FUNC_START("BtlPbapsPbProcessContinue");
	
	
	/* Context switch to the stack task */
	switch (btlPbapsPbPullData.returnState)
	{
		case BTL_PBAPS_PB_RETURN_STATE_IDLE:
			break;
		case BTL_PBAPS_PB_RETURN_STATE_ERROR:
			btlPbapsPbPullData.returnState = BTL_PBAPS_PB_RETURN_STATE_IDLE;
			obStatus = PBAP_ServerAbort(btlPbapsPbserverSession, PBRC_SERVICE_UNAVAILABLE);
			/* Call continue to keep the requests flowing. */
			obStatus = PBAP_ServerContinue(btlPbapsPbserverSession);
			break;
			
		case BTL_PBAPS_PB_RETURN_STATE_NO_RESOURCES:
			btlPbapsPbPullData.returnState = BTL_PBAPS_PB_RETURN_STATE_IDLE;
			obStatus = PBAP_ServerAbort(btlPbapsPbserverSession, PBRC_NOT_ACCEPTABLE);
			/* Call continue to keep the requests flowing. */
			obStatus = PBAP_ServerContinue(btlPbapsPbserverSession);
			break;

		case BTL_PBAPS_PB_RETURN_STATE_NOT_FOUND:
			btlPbapsPbPullData.returnState = BTL_PBAPS_PB_RETURN_STATE_IDLE;
			obStatus = PBAP_ServerAbort(btlPbapsPbserverSession, PBRC_NOT_FOUND);
			/* Call continue to keep the requests flowing. */
			obStatus = PBAP_ServerContinue(btlPbapsPbserverSession);
			break;

		
		case BTL_PBAPS_PB_RETURN_STATE_SUCCESS:
			btlPbapsPbPullData.returnState = BTL_PBAPS_PB_RETURN_STATE_IDLE;
			/* Call continue to keep the requests flowing. */
			obStatus = PBAP_ServerContinue(btlPbapsPbserverSession);
			break;
	}
	
	BTL_VERIFY_ERR(OB_STATUS_SUCCESS == obStatus, BT_STATUS_INTERNAL_ERROR, ("Server: PBAP server continue failure"));	

	BTL_FUNC_END();
}

/*---------------------------------------------------------------------------
 * BtlPbapsPbAxToI
 *
 * 	  	Converts a four-character text string that
 *  	represents a hexadecimal number into an integer.		
 *		
 *
 * Parameters:
 *
 *		hexSt [in] - string to convert
 * Return:
 *
 *		converted integer. 0 in case of an error.
 *
 */
U16 BtlPbapsPbAxToI(char *hexStg) 
{
  U16 n = 0;         /* position in string */
  U16 m = 0;         /* position in digit[] to shift */
  U16 count;         /* loop index */
  U16 intValue = 0;  /* integer value of hex string */
  U16 digit[5];      /* hold values to convert */
  while (n < 4) 
  {
     if (hexStg[n]=='\0')
        break;
     if (hexStg[n] > 0x29 && hexStg[n] < 0x40 ) /* if 0 to 9 */
        digit[n] = (U16)(hexStg[n] & 0x0f);            /* convert to int */
     else if (hexStg[n] >='a' && hexStg[n] <= 'f') /* if a to f */
        digit[n] = (U16)((hexStg[n] & 0x0f) + 9);      /* convert to int */
     else if (hexStg[n] >='A' && hexStg[n] <= 'F') /* if A to F */
        digit[n] = (U16)((hexStg[n] & 0x0f) + 9);      /* convert to int */
     else break;
    n++;
  }
  count = n;
  m = (U16)(n - 1);
  n = 0;
  while(n < count) 
  {
     /* digit[n] is value of hex digit at position n */
     /* (m << 2) is the number of positions to shift */
     /* OR the bits into return value */
     intValue = (U16) (intValue | (digit[n] << (m << 2)));
     m--;   /* adjust the position to set */
     n++;   /* next digit to process */
  }
  return (intValue);
}

/*---------------------------------------------------------------------------
 * BtlPbapsPbSetChangedStatus
 */
void BtlPbapsPbSetChangedStatus(BOOL changedFlag)
{
	btlPbapsPbPullData.pbChanged = changedFlag;
}

/*---------------------------------------------------------------------------
 * BtlPbapsPbGetChangedStatus
 *
 */

BOOL BtlPbapsPbGetChangedStatus()
{
	return btlPbapsPbPullData.pbChanged;
}

/*---------------------------------------------------------------------------
 * BtlPbapsPbAbortOperation
 *
 */

void BtlPbapsPbAbortOperation(BOOL sendContinue)
{
	BthalStatus bthalStatus;
	ObStatus	obStatus;
	
	BTL_FUNC_START("BtlPbapsPbAbortOperation");

	if (TRUE == sendContinue)
	{
		btlPbapsPbCurOper = BTL_PBAPS_PB_OPER_REMOTE_ABORT;
	}
	else
	{
		btlPbapsPbCurOper = BTL_PBAPS_PB_OPER_USER_ABORT;
	}

	/* Close the phonebook */
	
	bthalStatus = BTHAL_PB_ClosePb(btlPbapsPbPullData.pbHandle);
	BTL_VERIFY_ERR_NO_RETVAR(BTHAL_STATUS_FAILED != bthalStatus, ("BTHAL_PB_ClosePb failed"));

	if (TRUE == sendContinue)
	{
		if (BTHAL_STATUS_SUCCESS == bthalStatus)
		/* In that case, we should call continue now */
		{
	    	obStatus = PBAP_ServerContinue(btlPbapsPbserverSession);
			BTL_VERIFY_ERR_NO_RETVAR(OB_STATUS_SUCCESS == obStatus, ("Server: PBAP server continue failure"));
		}
		/* Otherwise, wait for the phonebook to return, and only than call continue */
		BTL_VERIFY_ERR_NO_RETVAR(BTHAL_STATUS_PENDING == bthalStatus, ("BTHAL_PB_ClosePb failed - return wrong value"));
	}
		
	BTL_FUNC_END();
}


void	BtlPbapResetEntryName(void)
{
	btlPbapsPbPullParams.entry.entryName[0] = '\0';
}



#endif /* BTL_CONFIG_PBAPS == BTL_CONFIG_ENABLED*/

