/*
+-----------------------------------------------------------------------------
|  Project :  GSM-F&D (8411)
|  Modul   :  ACI
+-----------------------------------------------------------------------------
|  Copyright 2005 Texas Instruments Berlin
|                 All rights reserved.
|
|                 This file is confidential and a trade secret of Texas
|                 Instruments Inc.
|                 The receipt of or possession of this file does not convey
|                 any rights to reproduce or disclose its contents or to
|                 manufacture, use, or sell anything it may describe, in
|                 whole, or in part, without the specific written consent of
|                 Texas Instruments Berlin, AG.
+-----------------------------------------------------------------------------
|  Purpose :  This Modul holds the internal functions
|             for the binary AT command library
+-----------------------------------------------------------------------------
*/
#define _BAT_KERF_C_

/*==== INCLUDES =============================================================*/
#include <string.h>
#include <stdio.h>

#include "typedefs.h"
#include "gdd.h"
#include "gdd_sys.h"
#include "l2p_types.h"
#include "l2p.h"
#include "bat.h"
#include "bat_ctrl.h"
#include "bat_intern.h"

/*==== DEFINES ===========================================================*/

#define GET_INST(inst_hndl) ((T_BAT_instance_maintain*)\
        (*(ptr_bat_list + inst_hndl)))

#define GET_CLNT(inst_mt,clnt_id) ((T_BAT_client_maintain*)\
        ((U32)sizeof(T_BAT_client_maintain)*clnt_id + \
         (U32)inst_mt + (U32)sizeof(T_BAT_instance_maintain)))

/*==== GLOBAL VARS ===========================================================*/

GLOBAL U8 max_bat_num = 4; /* start the BAT Lib with maximum 4 instances */
GLOBAL T_BAT_instance_maintain **ptr_bat_list = NULL;

/*==== LOCAL VARS ===========================================================*/

/* This table will be removed once it is generated by the tool.
 * As long as SAPE and makcdg does not handle feature flags,
 * we do not use #ifdef ... #endif in this table
 */

typedef struct
{
  size_t              size;
} T_map_params_2_size;

static T_map_params_2_size params_2_size[] =
{
  {/* BAT_CMD_SET_PERCENT_ALS        = 0x0,  */ sizeof(T_BAT_cmd_set_percent_als)},
  {/* BAT_CMD_SET_PERCENT_BAND       = 0x1,  */ sizeof(T_BAT_cmd_set_percent_band)},
  {/* BAT_CMD_ANDC                   = 0x2,  */ sizeof(T_BAT_cmd_andc)},
  {/* BAT_CMD_SET_PLUS_CACM          = 0x3,  */ sizeof(T_BAT_cmd_set_plus_cacm)},
  {/* BAT_CMD_SET_PLUS_CAMM          = 0x4,  */ sizeof(T_BAT_cmd_set_plus_camm)},
  {/* BAT_CMD_SET_PLUS_CAOC          = 0x5,  */ sizeof(T_BAT_cmd_set_plus_caoc)},
  {/* BAT_CMD_SET_PERCENT_CBHZ       = 0x6,  */ sizeof(T_BAT_cmd_set_percent_cbhz)},
  {/* BAT_CMD_SET_PLUS_CBST          = 0x7,  */ sizeof(T_BAT_cmd_set_plus_cbst)},
  {/* BAT_CMD_SET_PERCENT_CCBS       = 0x8,  */ sizeof(T_BAT_cmd_set_percent_ccbs)},
  {/* BAT_CMD_SET_PLUS_CCFC          = 0x9,  */ sizeof(T_BAT_cmd_set_plus_ccfc)},
  {/* BAT_CMD_SET_PLUS_CCLK          = 0xa,  */ sizeof(T_BAT_cmd_set_plus_cclk)},
  {/* BAT_CMD_SET_PLUS_CCUG          = 0xb,  */ sizeof(T_BAT_cmd_set_plus_ccug)},
  {/* BAT_CMD_SET_PLUS_CCWA          = 0xc,  */ sizeof(T_BAT_cmd_set_plus_ccwa)},
  {/* BAT_CMD_SET_PLUS_CCWA_W        = 0xd,  */ sizeof(T_BAT_cmd_set_plus_ccwa_w)},
  {/* BAT_CMD_SET_PLUS_CFUN          = 0xe,  */ sizeof(T_BAT_cmd_set_plus_cfun)},
  {/* BAT_CMD_SET_PERCENT_CGAATT     = 0xf,  */ sizeof(T_BAT_cmd_set_percent_cgaatt)},
  {/* BAT_CMD_SET_PLUS_CGACT         = 0x10  */ sizeof(T_BAT_cmd_set_plus_cgact)},
  {/* BAT_CMD_SET_PLUS_CGANS         = 0x11  */ sizeof(T_BAT_cmd_set_plus_cgans)},
  {/* BAT_CMD_SET_PLUS_CGATT         = 0x12  */ sizeof(T_BAT_cmd_set_plus_cgatt)},
  {/* BAT_CMD_SET_PERCENT_CGCLASS    = 0x13  */ sizeof(T_BAT_cmd_set_percent_cgclass)},
  {/* BAT_CMD_SET_PLUS_CGCLASS       = 0x14  */ sizeof(T_BAT_cmd_set_plus_cgclass)},
  {/* BAT_CMD_SET_PLUS_CGDATA        = 0x15  */ sizeof(T_BAT_cmd_set_plus_cgdata)},
  {/* BAT_CMD_SET_PLUS_CGDCONT       = 0x16  */ sizeof(T_BAT_cmd_set_plus_cgdcont)},
  {/* BAT_CMD_SET_PERCENT_CGMM       = 0x17  */ sizeof(T_BAT_cmd_set_percent_cgmm)},
  {/* BAT_CMD_SET_PLUS_CGPADDR       = 0x18  */ sizeof(T_BAT_cmd_set_plus_cgpaddr)},
  {/* BAT_CMD_SET_PERCENT_CGPCO      = 0x19  */ sizeof(T_BAT_cmd_set_percent_cgpco)},
  {/* BAT_CMD_SET_PERCENT_CGPPP      = 0x1a  */ sizeof(T_BAT_cmd_set_percent_cgppp)},
  {/* BAT_CMD_SET_PLUS_CGQMIN        = 0x1b  */ sizeof(T_BAT_cmd_set_plus_cgqmin)},
  {/* BAT_CMD_SET_PLUS_CGQREQ        = 0x1c  */ sizeof(T_BAT_cmd_set_plus_cgqreq)},
  {/* BAT_CMD_SET_PLUS_CGSMS         = 0x1d  */ sizeof(T_BAT_cmd_set_plus_cgsms)},
  {/* BAT_CMD_SET_PERCENT_CHLD       = 0x1e  */ sizeof(T_BAT_cmd_set_percent_chld)},
  {/* BAT_CMD_SET_PERCENT_CHPL       = 0x1f  */ sizeof(T_BAT_cmd_set_percent_chpl)},
  {/* BAT_CMD_SET_PERCENT_CHPL_W     = 0x20  */ sizeof(T_BAT_cmd_set_percent_chpl_w)},
  {/* BAT_CMD_SET_PLUS_CIND          = 0x21  */ sizeof(T_BAT_cmd_set_plus_cind)},
  {/* BAT_CMD_SET_PLUS_CLAN          = 0x22  */ sizeof(T_BAT_cmd_set_plus_clan)},
  {/* BAT_CMD_SET_PLUS_CLCK          = 0x23  */ sizeof(T_BAT_cmd_set_plus_clck)},
  {/* BAT_CMD_SET_PLUS_CLIR          = 0x24  */ sizeof(T_BAT_cmd_set_plus_clir)},
  {/* BAT_CMD_SET_PLUS_CLVL          = 0x25  */ sizeof(T_BAT_cmd_set_plus_clvl)},
  {/* BAT_CMD_SET_PLUS_CMER          = 0x26  */ sizeof(T_BAT_cmd_set_plus_cmer)},
  {/* BAT_CMD_SET_PLUS_CMGC          = 0x27  */ sizeof(T_BAT_cmd_set_plus_cmgc)},
  {/* BAT_CMD_SET_PLUS_CMGD          = 0x28  */ sizeof(T_BAT_cmd_set_plus_cmgd)},
  {/* BAT_CMD_SET_PLUS_CMGL          = 0x29  */ sizeof(T_BAT_cmd_set_plus_cmgl)},
  {/* BAT_CMD_SET_PLUS_CMGL_W        = 0x2a  */ sizeof(T_BAT_cmd_set_plus_cmgl_w)},
  {/* BAT_CMD_SET_PLUS_CMGR          = 0x2b  */ sizeof(T_BAT_cmd_set_plus_cmgr)},
  {/* BAT_CMD_SET_PLUS_CMGR_W        = 0x2c  */ sizeof(T_BAT_cmd_set_plus_cmgr_w)},
  {/* BAT_CMD_SET_PLUS_CMGS          = 0x2d  */ sizeof(T_BAT_cmd_set_plus_cmgs)},
  {/* BAT_CMD_SET_PLUS_CMGW          = 0x2e  */ sizeof(T_BAT_cmd_set_plus_cmgw)},
  {/* BAT_CMD_SET_PLUS_CMOD          = 0x2f  */ sizeof(T_BAT_cmd_set_plus_cmod)},
  {/* BAT_CMD_SET_PLUS_CMSS          = 0x30  */ sizeof(T_BAT_cmd_set_plus_cmss)},
  {/* BAT_CMD_SET_PLUS_CMUT          = 0x31  */ sizeof(T_BAT_cmd_set_plus_cmut)},
  {/* BAT_CMD_SET_PLUS_CMUX          = 0x32  */ sizeof(T_BAT_cmd_set_plus_cmux)},
  {/* BAT_CMD_SET_PLUS_CNMA          = 0x33  */ sizeof(T_BAT_cmd_set_plus_cnma)},
  {/* BAT_CMD_SET_PLUS_CNMI          = 0x34  */ sizeof(T_BAT_cmd_set_plus_cnmi)},
  {/* BAT_CMD_SET_PERCENT_COPS       = 0x35  */ sizeof(T_BAT_cmd_set_percent_cops)},
  {/* BAT_CMD_SET_PLUS_COPS          = 0x36  */ sizeof(T_BAT_cmd_set_plus_cops)},
  {/* BAT_CMD_SET_PERCENT_COPS_W     = 0x37  */ sizeof(T_BAT_cmd_set_percent_cops_w)},
  {/* BAT_CMD_SET_PERCENT_CPALS      = 0x38  */ sizeof(T_BAT_cmd_set_percent_cpals)},
  {/* BAT_CMD_SET_PERCENT_CPALS_W    = 0x39  */ sizeof(T_BAT_cmd_set_percent_cpals_w)},
  {/* BAT_CMD_SET_PLUS_CPBF          = 0x3a  */ sizeof(T_BAT_cmd_set_plus_cpbf)},
  {/* BAT_CMD_SET_PLUS_CPBF_W        = 0x3b  */ sizeof(T_BAT_cmd_set_plus_cpbf_w)},
  {/* BAT_CMD_SET_PLUS_CPBR          = 0x3c  */ sizeof(T_BAT_cmd_set_plus_cpbr)},
  {/* BAT_CMD_SET_PLUS_CPBR_W        = 0x3d  */ sizeof(T_BAT_cmd_set_plus_cpbr_w)},
  {/* BAT_CMD_SET_PLUS_CPBS          = 0x3e  */ sizeof(T_BAT_cmd_set_plus_cpbs)},
  {/* BAT_CMD_SET_PLUS_CPBW          = 0x3f  */ sizeof(T_BAT_cmd_set_plus_cpbw)},
  {/* BAT_CMD_SET_PLUS_CPBW_W        = 0x40  */ sizeof(T_BAT_cmd_set_plus_cpbw_w)},
  {/* BAT_CMD_SET_PERCENT_CPCFU      = 0x41  */ sizeof(T_BAT_cmd_set_percent_cpcfu)},
  {/* BAT_CMD_SET_PERCENT_CPHS       = 0x42  */ sizeof(T_BAT_cmd_set_percent_cphs)},
  {/* BAT_CMD_SET_PLUS_CPIN          = 0x43  */ sizeof(T_BAT_cmd_set_plus_cpin)},
  {/* BAT_CMD_SET_PERCENT_CPMB       = 0x44  */ sizeof(T_BAT_cmd_set_percent_cpmb)},
  {/* BAT_CMD_SET_PERCENT_CPMBW      = 0x45  */ sizeof(T_BAT_cmd_set_percent_cpmbw)},
  {/* BAT_CMD_SET_PERCENT_CPMBW_W    = 0x46  */ sizeof(T_BAT_cmd_set_percent_cpmbw_w)},
  {/* BAT_CMD_SET_PERCENT_CPMB_W     = 0x47  */ sizeof(T_BAT_cmd_set_percent_cpmb_w)},
  {/* BAT_CMD_SET_PLUS_CPMS          = 0x48  */ sizeof(T_BAT_cmd_set_plus_cpms)},
  {/* BAT_CMD_SET_PERCENT_CPNUMS     = 0x49  */ sizeof(T_BAT_cmd_set_percent_cpnums)},
  {/* BAT_CMD_SET_PERCENT_CPNUMS_W   = 0x4a  */ sizeof(T_BAT_cmd_set_percent_cpnums_w)},
  {/* BAT_CMD_SET_PLUS_CPOL          = 0x4b  */ sizeof(T_BAT_cmd_set_plus_cpol)},
  {/* BAT_CMD_SET_PLUS_CPOL_W        = 0x4c  */ sizeof(T_BAT_cmd_set_plus_cpol_w)},
  {/* BAT_CMD_SET_PERCENT_CPRSM      = 0x4d  */ sizeof(T_BAT_cmd_set_percent_cprsm)},
  {/* BAT_CMD_SET_PLUS_CPUC          = 0x4e  */ sizeof(T_BAT_cmd_set_plus_cpuc)},
  {/* BAT_CMD_SET_PLUS_CPUC_W        = 0x4f  */ sizeof(T_BAT_cmd_set_plus_cpuc_w)},
  {/* BAT_CMD_SET_PERCENT_CPVWI      = 0x50  */ sizeof(T_BAT_cmd_set_percent_cpvwi)},
  {/* BAT_CMD_SET_PLUS_CPWD          = 0x51  */ sizeof(T_BAT_cmd_set_plus_cpwd)},
  {/* BAT_CMD_SET_PLUS_CRES          = 0x52  */ sizeof(T_BAT_cmd_set_plus_cres)},
  {/* BAT_CMD_SET_PLUS_CRLP          = 0x53  */ sizeof(T_BAT_cmd_set_plus_crlp)},
  {/* BAT_CMD_SET_PLUS_CRSM          = 0x54  */ sizeof(T_BAT_cmd_set_plus_crsm)},
  {/* BAT_CMD_SET_PLUS_CSAS          = 0x55  */ sizeof(T_BAT_cmd_set_plus_csas)},
  {/* BAT_CMD_SET_PLUS_CSCA          = 0x56  */ sizeof(T_BAT_cmd_set_plus_csca)},
  {/* BAT_CMD_SET_PLUS_CSCB          = 0x57  */ sizeof(T_BAT_cmd_set_plus_cscb)},
  {/* BAT_CMD_SET_PLUS_CSCS          = 0x58  */ sizeof(T_BAT_cmd_set_plus_cscs)},
  {/* BAT_CMD_SET_PLUS_CSIM          = 0x59  */ sizeof(T_BAT_cmd_set_plus_csim)},
  {/* BAT_CMD_SET_PLUS_CSMS          = 0x5a  */ sizeof(T_BAT_cmd_set_plus_csms)},
  {/* BAT_CMD_SET_PLUS_CSNS          = 0x5b  */ sizeof(T_BAT_cmd_set_plus_csns)},
  {/* BAT_CMD_SET_PLUS_CSTA          = 0x5c  */ sizeof(T_BAT_cmd_set_plus_csta)},
  {/* BAT_CMD_SET_PLUS_CSVM          = 0x5d  */ sizeof(T_BAT_cmd_set_plus_csvm)},
  {/* BAT_CMD_SET_PLUS_CTFR          = 0x5e  */ sizeof(T_BAT_cmd_set_plus_ctfr)},
  {/* BAT_CMD_SET_PERCENT_CTTY       = 0x5f  */ sizeof(T_BAT_cmd_set_percent_ctty)},
  {/* BAT_CMD_SET_PLUS_CTZU          = 0x60  */ sizeof(T_BAT_cmd_set_plus_ctzu)},
  {/* BAT_CMD_SET_PLUS_CUSD          = 0x61  */ sizeof(T_BAT_cmd_set_plus_cusd)},
  {/* BAT_CMD_SET_PLUS_CUSD_W        = 0x62  */ sizeof(T_BAT_cmd_set_plus_cusd_w)},
  {/* BAT_CMD_SET_PERCENT_CUST       = 0x63  */ sizeof(T_BAT_cmd_set_percent_cust)},
  {/* BAT_CMD_SET_PERCENT_CWUP       = 0x64  */ sizeof(T_BAT_cmd_set_percent_cwup)},
  {/* BAT_CMD_AT_D                   = 0x65  */ sizeof(T_BAT_cmd_at_d)},
  {/* BAT_CMD_SET_PERCENT_DATA       = 0x66  */ sizeof(T_BAT_cmd_set_percent_data)},
  {/* BAT_CMD_SET_PERCENT_DINF       = 0x67  */ sizeof(T_BAT_cmd_set_percent_dinf)},
  {/* BAT_CMD_SET_PLUS_DS            = 0x68  */ sizeof(T_BAT_cmd_set_plus_ds)},
  {/* BAT_CMD_AT_D_W                 = 0x69  */ sizeof(T_BAT_cmd_at_d_w)},
  {/* BAT_CMD_SET_PERCENT_EFRSLT     = 0x6a  */ sizeof(T_BAT_cmd_set_percent_efrslt)},
  {/* BAT_CMD_SET_PERCENT_EM         = 0x6b  */ sizeof(T_BAT_cmd_set_percent_em)},
  {/* BAT_CMD_SET_PLUS_FAP           = 0x6c  */ sizeof(T_BAT_cmd_set_plus_fap)},
  {/* BAT_CMD_SET_PLUS_FBO           = 0x6d  */ sizeof(T_BAT_cmd_set_plus_fbo)},
  {/* BAT_CMD_SET_PLUS_FBU           = 0x6e  */ sizeof(T_BAT_cmd_set_plus_fbu)},
  {/* BAT_CMD_SET_PLUS_FCC           = 0x6f  */ sizeof(T_BAT_cmd_set_plus_fcc)},
  {/* BAT_CMD_SET_PLUS_FCLASS        = 0x70  */ sizeof(T_BAT_cmd_set_plus_fclass)},
  {/* BAT_CMD_SET_PLUS_FCQ           = 0x71  */ sizeof(T_BAT_cmd_set_plus_fcq)},
  {/* BAT_CMD_SET_PLUS_FCR           = 0x72  */ sizeof(T_BAT_cmd_set_plus_fcr)},
  {/* BAT_CMD_SET_PLUS_FCT           = 0x73  */ sizeof(T_BAT_cmd_set_plus_fct)},
  {/* BAT_CMD_SET_PLUS_FEA           = 0x74  */ sizeof(T_BAT_cmd_set_plus_fea)},
  {/* BAT_CMD_SET_PLUS_FFC           = 0x75  */ sizeof(T_BAT_cmd_set_plus_ffc)},
  {/* BAT_CMD_SET_PLUS_FIE           = 0x76  */ sizeof(T_BAT_cmd_set_plus_fie)},
  {/* BAT_CMD_SET_PLUS_FIS           = 0x77  */ sizeof(T_BAT_cmd_set_plus_fis)},
  {/* BAT_CMD_SET_PLUS_FIT           = 0x78  */ sizeof(T_BAT_cmd_set_plus_fit)},
  {/* BAT_CMD_SET_PLUS_FLI           = 0x79  */ sizeof(T_BAT_cmd_set_plus_fli)},
  {/* BAT_CMD_SET_PLUS_FLO           = 0x7a  */ sizeof(T_BAT_cmd_set_plus_flo)},
  {/* BAT_CMD_SET_PLUS_FLP           = 0x7b  */ sizeof(T_BAT_cmd_set_plus_flp)},
  {/* BAT_CMD_SET_PLUS_FMS           = 0x7c  */ sizeof(T_BAT_cmd_set_plus_fms)},
  {/* BAT_CMD_SET_PLUS_FNS           = 0x7d  */ sizeof(T_BAT_cmd_set_plus_fns)},
  {/* BAT_CMD_SET_PLUS_FPA           = 0x7e  */ sizeof(T_BAT_cmd_set_plus_fpa)},
  {/* BAT_CMD_SET_PLUS_FPI           = 0x7f  */ sizeof(T_BAT_cmd_set_plus_fpi)},
  {/* BAT_CMD_SET_PLUS_FPS           = 0x80  */ sizeof(T_BAT_cmd_set_plus_fps)},
  {/* BAT_CMD_SET_PLUS_FPW           = 0x81  */ sizeof(T_BAT_cmd_set_plus_fpw)},
  {/* BAT_CMD_SET_PLUS_FRQ           = 0x82  */ sizeof(T_BAT_cmd_set_plus_frq)},
  {/* BAT_CMD_SET_PLUS_FSA           = 0x83  */ sizeof(T_BAT_cmd_set_plus_fsa)},
  {/* BAT_CMD_SET_PLUS_FSP           = 0x84  */ sizeof(T_BAT_cmd_set_plus_fsp)},
  {/* BAT_CMD_SET_PLUS_ICF           = 0x85  */ sizeof(T_BAT_cmd_set_plus_icf)},
  {/* BAT_CMD_SET_PLUS_IFC           = 0x86  */ sizeof(T_BAT_cmd_set_plus_ifc)},
  {/* BAT_CMD_SET_PLUS_IPR           = 0x87  */ sizeof(T_BAT_cmd_set_plus_ipr)},
  {/* BAT_CMD_SET_PERCENT_PBCF       = 0x88  */ sizeof(T_BAT_cmd_set_percent_pbcf)},
  {/* BAT_CMD_SET_PERCENT_PPP        = 0x89  */ sizeof(T_BAT_cmd_set_percent_ppp)},
  {/* BAT_CMD_SET_PERCENT_PVRF       = 0x8a  */ sizeof(T_BAT_cmd_set_percent_pvrf)},
  {/* BAT_CMD_SET_PERCENT_RDL        = 0x8b  */ sizeof(T_BAT_cmd_set_percent_rdl)},
  {/* BAT_CMD_SET_PERCENT_RDLB       = 0x8c  */ sizeof(T_BAT_cmd_set_percent_rdlb)},
  {/* BAT_CMD_SET_PERCENT_SATC       = 0x8d  */ sizeof(T_BAT_cmd_set_percent_satc)},
  {/* BAT_CMD_SET_PERCENT_SATCC      = 0x8e  */ sizeof(T_BAT_cmd_set_percent_satcc)},
  {/* BAT_CMD_SET_PERCENT_SATE       = 0x8f  */ sizeof(T_BAT_cmd_set_percent_sate)},
  {/* BAT_CMD_SET_PERCENT_SATR       = 0x90  */ sizeof(T_BAT_cmd_set_percent_satr)},
  {/* BAT_CMD_SET_PERCENT_SATT       = 0x91  */ sizeof(T_BAT_cmd_set_percent_satt)},
  {/* BAT_CMD_SET_PERCENT_SNCNT      = 0x92  */ sizeof(T_BAT_cmd_set_percent_sncnt)},
  {/* BAT_CMD_SET_PERCENT_VTS        = 0x93  */ sizeof(T_BAT_cmd_set_percent_vts)},
  {/* BAT_CMD_SET_PLUS_VTS           = 0x94  */ sizeof(T_BAT_cmd_set_plus_vts)},
  {/* BAT_CMD_SET_PLUS_WS46          = 0x95  */ sizeof(T_BAT_cmd_set_plus_ws46)},
  {/* BAT_CMD_SET_PERCENT_CPINF      = 0x96  */ sizeof(T_BAT_cmd_set_percent_cpinf)},
  {/* BAT_CMD_SET_PERCENT_CMGR       = 0x97  */ sizeof(T_BAT_cmd_set_percent_cmgr)},
  {/* BAT_CMD_SET_PERCENT_CMGL       = 0x98  */ sizeof(T_BAT_cmd_set_percent_cmgl)},
  {/* BAT_CMD_SET_PERCENT_CTZV       = 0x99  */ sizeof(T_BAT_cmd_set_percent_ctzv)},
#ifdef TI_PS_FF_AT_CMD_P_ECC
  {/* BAT_CMD_SET_PERCENT_ECC        = 0x9a  */ sizeof(T_BAT_cmd_set_percent_ecc)},
#else
  {/* BAT_CMD_SET_PERCENT_ECC        = 0x9a  */ 0}, /* In case flag is not enabled */
#endif /* TI_PS_FF_AT_CMD_P_ECC */

#ifdef SIM_PERS
  { /* BAT_CMD_SET_PERCENT_MEPD     = 0x9b */ sizeof(T_BAT_cmd_set_percent_mepd)},
#else
  { /* BAT_CMD_SET_PERCENT_MEPD     = 0x9b   */ 0}, /* In case flag is not enabled */
#endif

{ /* BAT_CMD_SET_PERCENT_CSQ      = 0x9c   */ sizeof(T_BAT_cmd_set_percent_csq)},
{/* BAT_CMD_SET_PERCENT_CNIV       = 0x9d  */ sizeof(T_BAT_cmd_set_percent_cniv)},
#ifdef REL99
  {/* BAT_CMD_SET_PERCENT_CMGRS      = 0x9e  */ sizeof(T_BAT_cmd_set_percent_cmgrs)},
#endif
{/* BAT_CMD_SET_PERCENT_SIMEF       = 0xa7   */ sizeof(T_BAT_cmd_set_percent_simef)}
  
};

/*
+----------------------------------------------------------------------------+
| PROJECT :                    MODULE  : BINARY AT COMMAND LIBRARY           |
| STATE   : code               ROUTINE : bat_change_client_state             |
+----------------------------------------------------------------------------+

PURPOSE : 
  This function changes the client state and traces the state change. 
*/

GLOBAL void bat_change_client_state (T_BAT_client_maintain   *client,
                                     T_BAT_client_state       new_state)
{
  char trcBuf[80];
  T_BAT_client_state  old_state = client->client_state;
  client->client_state = new_state;

  if (old_state EQ new_state)
  {
    return;
  }

  sprintf (trcBuf, "(client 0x%04X) state: ", (int)client);

  switch (old_state)
  {
    case (BAT_CLIENT_IDLE):
      strcat (trcBuf, "BAT_CLIENT_IDLE -> ");
      break;
    case (BAT_CLIENT_ACTIVATING):
      strcat (trcBuf, "BAT_CLIENT_ACTIVATING -> ");
      break;
    case (BAT_CLIENT_READY):
      strcat (trcBuf, "BAT_CLIENT_READY -> ");
      break;    
    case (BAT_CLIENT_BUSY):
      strcat (trcBuf, "BAT_CLIENT_BUSY -> ");
      break;
    case (BAT_CLIENT_SENDING):
      strcat (trcBuf, "BAT_CLIENT_SENDING -> ");
      break;
    case (BAT_CLIENT_SENDING_AND_BUSY):
      strcat (trcBuf, "BAT_CLIENT_SENDING_AND_BUSY -> ");
      break;
    default:
      strcat (trcBuf, "UNKNOWN ! ");
  }
  
  switch (new_state)
  {
    case (BAT_CLIENT_IDLE):
      strcat (trcBuf, "BAT_CLIENT_IDLE");
      break;
    case (BAT_CLIENT_ACTIVATING):
      strcat (trcBuf, "BAT_CLIENT_ACTIVATING");
      break;
    case (BAT_CLIENT_READY):
      strcat (trcBuf, "BAT_CLIENT_READY");
      break;
    case (BAT_CLIENT_BUSY):
      strcat (trcBuf, "BAT_CLIENT_BUSY");
      break;
    case (BAT_CLIENT_SENDING):
      strcat (trcBuf, "BAT_CLIENT_SENDING");
      break;
    case (BAT_CLIENT_SENDING_AND_BUSY):
      strcat (trcBuf, "BAT_CLIENT_SENDING_AND_BUSY");
      break;
    default:
      strcat (trcBuf, "UNKNOWN !");
  }

  BAT_TRACE_EVENT_P1("%s",trcBuf);
}
/*
+----------------------------------------------------------------------------+
| PROJECT :                    MODULE  : BINARY AT COMMAND LIBRARY           |
| STATE   : code               ROUTINE : bat_change_instance_state           |
+----------------------------------------------------------------------------+

PURPOSE : 
  This function changes the instance state and traces the state change. 
*/
GLOBAL void bat_change_instance_state (T_BAT_instance_maintain *instance, 
                                       T_BAT_instance_state     new_state)
{
  char trcBuf[80];
  T_BAT_instance_state  old_state = instance->instance_state;
  instance->instance_state = new_state;

  if (old_state EQ new_state)
  {
    return;
  }

  sprintf (trcBuf, "(instance 0x%08X) state: ", (int)instance);

  switch (old_state)
  {
    case (BAT_INSTANCE_IDLE):
      strcat (trcBuf, "BAT_INSTANCE_IDLE -> ");
      break;
    case (BAT_INSTANCE_ACTIVATING):
      strcat (trcBuf, "BAT_INSTANCE_ACTIVATING -> ");
      break;
    case (BAT_INSTANCE_READY):
      strcat (trcBuf, "BAT_INSTANCE_READY -> ");
      break;    
    case (BAT_INSTANCE_BUSY):
      strcat (trcBuf, "BAT_INSTANCE_BUSY -> ");
      break;
    default:
      strcat (trcBuf, "UNKNOWN ! ");
  }
  
  switch (new_state)
  {
    case (BAT_INSTANCE_IDLE):
      strcat (trcBuf, "BAT_INSTANCE_IDLE");
      break;
    case (BAT_INSTANCE_ACTIVATING):
      strcat (trcBuf, "BAT_INSTANCE_ACTIVATING");
      break;
    case (BAT_INSTANCE_READY):
      strcat (trcBuf, "BAT_INSTANCE_READY");
      break;
    case (BAT_INSTANCE_BUSY):
      strcat (trcBuf, "BAT_INSTANCE_BUSY");
      break;
    default:
      strcat (trcBuf, "UNKNOWN !");
  }

  BAT_TRACE_EVENT_P1("%s",trcBuf);
}

/*
+----------------------------------------------------------------------------+
| PROJECT :                    MODULE  : BINARY AT COMMAND LIBRARY           |
| STATE   : code               ROUTINE : bat_change_buffer_state             |
+----------------------------------------------------------------------------+

PURPOSE : 
  This function changes the buffer state and traces the state change. 
*/
GLOBAL void bat_change_buffer_state (T_BAT_instance_maintain *instance, 
                                     T_BAT_buf_state          new_state)
{
/*  char trcBuf[80];
  T_BAT_buf_state  old_state = instance->buffer.buf_st;*/
  instance->buffer.buf_st = new_state;
/*
  if (old_state EQ new_state)
  {
    return;
  }
  sprintf (trcBuf, "(instance 0x%08X) buffer state: ", (int)instance);

  switch (old_state)
  {
    case (BAT_BUF_EMPTY):
      strcat (trcBuf, "BAT_BUF_EMPTY -> ");
      break;
    case (BAT_BUF_FILLING):
      strcat (trcBuf, "BAT_BUF_FILLING -> ");
      break;
    case (BAT_BUF_FILLED):
      strcat (trcBuf, "BAT_BUF_FILLED -> ");
      break;
    default:
      strcat (trcBuf, "UNKNOWN ! ");
  }
  
  switch (new_state)
  {
    case (BAT_BUF_EMPTY):
      strcat (trcBuf, "BAT_BUF_EMPTY");
      break;
    case (BAT_BUF_FILLING):
      strcat (trcBuf, "BAT_BUF_FILLING");
      break;
    case (BAT_BUF_FILLED):
      strcat (trcBuf, "BAT_BUF_FILLED");
      break;
    default:
      strcat (trcBuf, "UNKNOWN !");
  }

  BAT_TRACE_EVENT_P1("%s",trcBuf);*/
}

/*
+----------------------------------------------------------------------------+
| PROJECT :                    MODULE  : BINARY AT COMMAND LIBRARY           |
| STATE   : code               ROUTINE : bat_init_global_params              |
+----------------------------------------------------------------------------+

PURPOSE : 
  This function initializes the global parameters. 
*/

GLOBAL T_BAT_return bat_init_global_params ()
{
  /* The first BAT instance, so init the params */
  if(ptr_bat_list EQ NULL)
  {
    void *bat_header = NULL;
    int header_size = max_bat_num * sizeof (T_BAT_instance_maintain *); 
    BAT_TRACE_EVENT("First BAT instance!");
    /* allocate memory to hold the instance pointer array */
    bat_header = gdd_sys_mem_malloc(header_size);
    /* init allocated memory */
    memset(bat_header, 0, header_size);
    
    ptr_bat_list = (T_BAT_instance_maintain **)bat_header;
  }

  return (BAT_OK);
}

/*
+----------------------------------------------------------------------------+
| PROJECT :                    MODULE  : BINARY AT COMMAND LIBRARY           |
| STATE   : code               ROUTINE : bat_deinit_global_params            |
+----------------------------------------------------------------------------+

PURPOSE : 
  This function deinitializes the global parameters. 
*/

GLOBAL T_BAT_return bat_deinit_global_params ()
{
  int i;
  for(i = 0; i < max_bat_num; i ++)
  {
    if (GET_INST (i) NEQ NULL)
    return (BAT_OK);
  }
    
  max_bat_num = 4;
  ptr_bat_list = NULL;

  return (BAT_OK);
}

/*
+----------------------------------------------------------------------------+
| PROJECT :                    MODULE  : BINARY AT COMMAND LIBRARY           |
| STATE   : code               ROUTINE : bat_deinit_global_params            |
+----------------------------------------------------------------------------+

PURPOSE : 
  This function checks if the global parameters have been init or deinit. If 
  is_init is TRUE then the init state should be checked otherwise the deinit 
  state should be checked. 
*/

GLOBAL BOOL bat_check_init_or_deinit (BOOL is_init)
{
  /*BAT Lib should be already init*/
  if (is_init AND ptr_bat_list EQ NULL)
  {
    return (FALSE);
  }
  /*BAT Lib should be already deinit*/
  if (!is_init AND ptr_bat_list NEQ NULL)
  {
    return (FALSE);
  }

  return (TRUE);
}

/*
+----------------------------------------------------------------------------+
| PROJECT :                    MODULE  : BINARY AT COMMAND LIBRARY           |
| STATE   : code               ROUTINE : bat_init_inatance_pointer           |
+----------------------------------------------------------------------------+

PURPOSE : 
  This function initializes a pointer in the pointer list. 
*/

GLOBAL T_BAT_return bat_init_instance_pointer (T_BAT_instance inst_hndl, 
                                               T_BAT_instance_maintain *inst_mt)
{ 
  *(ptr_bat_list + inst_hndl)= inst_mt;

  return (BAT_OK);
}

/*
+----------------------------------------------------------------------------+
| PROJECT :                    MODULE  : BINARY AT COMMAND LIBRARY           |
| STATE   : code               ROUTINE : bat_init_inatance_pointer           |
+----------------------------------------------------------------------------+

PURPOSE : 
  This function initializes a pointer in the pointer list. 
*/

GLOBAL T_BAT_return bat_deinit_instance_pointer (T_BAT_instance inst_hndl)
{ 
  *(ptr_bat_list + inst_hndl) = NULL;

  return (BAT_OK);
}

/*
+----------------------------------------------------------------------------+
| PROJECT :                    MODULE  : BINARY AT COMMAND LIBRARY           |
| STATE   : code               ROUTINE : check_if_all_instances_are_closed   |
+----------------------------------------------------------------------------+

PURPOSE : 
  This function initializes the global parameters. 
*/

GLOBAL BOOL bat_check_if_all_instances_are_closed()
{
  int i;
  for (i = 0; i < max_bat_num; i++)
  {
    if (GET_INST(i) NEQ NULL)
    {
      BAT_TRACE_ERROR ("bat_check_if_all_instances_are_closed(): not all the instances are closed!");
      return (FALSE);
    }
  }
  return (TRUE);
}

/*
+----------------------------------------------------------------------------+
| PROJECT :                    MODULE  : BINARY AT COMMAND LIBRARY           |
| STATE   : code               ROUTINE : check_if_all_clients_are_closed     |
+----------------------------------------------------------------------------+

PURPOSE : 
  This function checks if all the clients are closed. 
*/

GLOBAL BOOL bat_check_if_all_clients_are_closed(T_BAT_instance inst_hndl)
{
  int i;
  T_BAT_instance_maintain *inst_mt = GET_INST(inst_hndl);
  
  if (inst_mt)
  {
    for (i = 0; i < inst_mt->max_client_num; i++)
    {
      if ((GET_CLNT(inst_mt, i))->client_state NEQ BAT_CLIENT_IDLE)
      {
        BAT_TRACE_ERROR("bat_check_if_all_clients_are_closed(): Not all clients are closed!");
       return (FALSE);
      }
    }
  }
  return (TRUE);
}
/*
+----------------------------------------------------------------------------+
| PROJECT :                    MODULE  : BINARY AT COMMAND LIBRARY           |
| STATE   : code               ROUTINE : bat_init_global_params              |
+----------------------------------------------------------------------------+

PURPOSE : 
  This function gets the address from instance handle. 
*/

GLOBAL T_BAT_return bat_get_instance_from_instance_handle 
                                          (T_BAT_instance inst_hndl, 
                                           T_BAT_instance_maintain **inst_mt)
{
  if (inst_hndl > max_bat_num)
  {
    BAT_TRACE_ERROR("bat_get_instance_from_instance_handle(): instance searched for is out of bound!");
    return (BAT_ERROR);
  }
  *inst_mt = GET_INST(inst_hndl);
  if (*inst_mt EQ NULL)
  {
    BAT_TRACE_ERROR ("bat_get_instance_from_instance_handle(): no instance has been found!");
    return (BAT_ERROR);
  }
  return (BAT_OK);
}
/*
+----------------------------------------------------------------------------+
| PROJECT :                    MODULE  : BINARY AT COMMAND LIBRARY           |
| STATE   : code               ROUTINE : bat_get_new_instance                |
+----------------------------------------------------------------------------+

PURPOSE : 
  This function scans through the pointer list and passes the index of the first
  free instance pointer with the output parameter <instance>. 
*/

GLOBAL T_BAT_return bat_get_new_instance (T_BAT_instance *instance)
{
  int i = 0;

  do 
  {
    if (GET_INST(i)EQ NULL)
    {
      *instance = (T_BAT_instance) i;
      return (BAT_OK);
    }
    i++;
  }  while (i < max_bat_num);

  /* the allocated memory for the BAT header is full, we will allocate a bigger 
     block of memory and copy the older header array to the new memory block */
  if (i < BAT_INVALID_INSTANCE_HANDLE-4) /* maximum 256 instances can be created */
  {
    void *new_bat_header = NULL;
    U8 new_max_bat_num = max_bat_num + 4;
    int new_header_size = new_max_bat_num * sizeof(T_BAT_instance_maintain*);

    new_bat_header = gdd_sys_mem_malloc(new_header_size);
    memset(new_bat_header, 0, new_header_size);
    memcpy(new_bat_header, ptr_bat_list, max_bat_num*sizeof(T_BAT_instance_maintain*));
    
    /* free the old header list and point to the new header list */
    gdd_sys_mem_free(ptr_bat_list);
    ptr_bat_list = (T_BAT_instance_maintain **)new_bat_header;
    *instance = max_bat_num; /*the new instance handle should be the old max number+1*/
    max_bat_num = new_max_bat_num;
    BAT_TRACE_EVENT_P2("Current max BAT is %d, the header address is 0x%04X", 
                        new_max_bat_num, ptr_bat_list);
    return (BAT_OK);

  }

  BAT_TRACE_ERROR("bat_get_new_instance(): No free instance to create!");
  return (BAT_ERROR);
}

/*
+----------------------------------------------------------------------------+
| PROJECT :                    MODULE  : BINARY AT COMMAND LIBRARY           |
| STATE   : code               ROUTINE : bat_delete_instance                 |
+----------------------------------------------------------------------------+

PURPOSE : 
  This function checks if all the clients in the instance are closed, if not then
  an error will be returned. If yes, the pointer pointing to the instance in 
  pointer list will be set to NULL.   
*/

GLOBAL T_BAT_return bat_delete_instance (T_GDD_CON_HANDLE con_handle)
{
  int i;
  T_BAT_instance inst_hdle = BAT_INVALID_INSTANCE_HANDLE;
  T_BAT_client clnt_hndl = BAT_INVALID_CLIENT_HANDLE;
  T_BAT_instance_maintain *inst_mt = NULL;
  T_BAT_client_maintain *clnt_mt = NULL;

  BAT_TRACE_FUNCTION ("bat_delete_instance()");

  if (bat_get_instance_from_gdd_handle (con_handle, &inst_hdle, &inst_mt) EQ BAT_ERROR)
  {
    return (BAT_ERROR);
  }
  
  for (i = 0; i < inst_mt->max_client_num; i++)
  {
    bat_make_client_handle_from_gdd_and_client_id(con_handle, (U8)i, &clnt_hndl);
    bat_get_client_from_client_handle(clnt_hndl, &clnt_mt);

    if (clnt_mt->client_state NEQ BAT_CLIENT_IDLE)
    {
      BAT_TRACE_ERROR ("bat_delete_instance(): not all clients are closed");
      return (BAT_ERROR);
    }
  }
  /* check if the instanse  */
  if (bat_get_instance_from_instance_handle(inst_hdle, &inst_mt))
  {
    return (BAT_ERROR);
  }
  return (BAT_OK);
}


/*
+----------------------------------------------------------------------------+
| PROJECT :                    MODULE  : BINARY AT COMMAND LIBRARY           |
| STATE   : code               ROUTINE : bat_get_instance_from_gdd_handle    |
+----------------------------------------------------------------------------+

PURPOSE : 
  This function searches the instances and passes the address of the correct 
  instance through the output parameter.  
*/

T_BAT_return bat_get_instance_from_gdd_handle ( T_GDD_CON_HANDLE con_handle, 
                                                T_BAT_instance *instance_handle,
                                                T_BAT_instance_maintain **instance)

{
  int i;

  for (i = 0; i < max_bat_num; i++)
  {
    *instance = GET_INST(i);
    *instance_handle = (U8)i;

    if ((*instance)!= NULL && (*instance)->con_handle == con_handle)
    {
      return (BAT_OK);
    }
  }
  return (BAT_ERROR);
}

/*
+----------------------------------------------------------------------------+
| PROJECT :                    MODULE  : BINARY AT COMMAND LIBRARY           |
| STATE   : code               ROUTINE : bat_get_new_client                  |
+----------------------------------------------------------------------------+

PURPOSE : 
  This function searches in an instance until it reaches the first free client 
  (e,g. the Nth client). This instance handle and the client id (N in this case)
  will be used to create the client handle.  
*/

T_BAT_return bat_get_new_client( T_BAT_instance instance, T_BAT_client *client)
{
  int i;
  T_BAT_client_maintain *new_client = NULL;
  T_BAT_instance_maintain *inst_mt = GET_INST(instance);

  if (inst_mt EQ NULL)
  { 
    BAT_TRACE_ERROR ("bat_get_new_client(): the input instance is NULL.");
    return (BAT_ERROR);
  }

  for (i = 0; i<inst_mt->max_client_num; i++)
  {
    new_client = GET_CLNT(inst_mt,i);
    if (new_client->client_state EQ BAT_CLIENT_IDLE)
    {
      *client = MAKE_CLNT_HNDL(i, instance); 
      BAT_TRACE_EVENT_P1("bat_get_new_client(): new client is : 0x%04X", *client);
      return (BAT_OK);
    } 
  }
  BAT_TRACE_ERROR ("bat_get_new_client(): no client is found!");
  return (BAT_ERROR);
}

/*
+----------------------------------------------------------------------------+
| PROJECT :                    MODULE  : BINARY AT COMMAND LIBRARY           |
| STATE   : code               ROUTINE : bat_init_new_client                 |
+----------------------------------------------------------------------------+

PURPOSE : 
  This function searches in an instance until it reaches the first free client 
  (e,g. the Nth client). This instance handle and the client id (N in this case)
  will be used to create the client handle.  
*/

T_BAT_return bat_init_new_client (T_BAT_client client, 
                                    int(*response_cb)(T_BAT_client client, T_BAT_cmd_response *response),
                                    void(*signal_cb)(T_BAT_client client, T_BAT_signal))

{
  T_BAT_client_maintain *clnt_mt = NULL;

  if (bat_get_client_from_client_handle(client, &clnt_mt) EQ BAT_ERROR)
  {
    return (BAT_ERROR);
  }
  bat_change_client_state(clnt_mt, BAT_CLIENT_ACTIVATING);
  clnt_mt->signal_cb = signal_cb;
  clnt_mt->response_cb = response_cb;

  return (BAT_OK);
}

/*
+----------------------------------------------------------------------------+
| PROJECT :                    MODULE  : BINARY AT COMMAND LIBRARY           |
| STATE   : code     ROUTINE : bat_make_client_handle_from_gdd_and_client_id |
+----------------------------------------------------------------------------+

PURPOSE : 
  This function gets the <client_id> and the <con_handle> and then builds the 
  client handle out of them. The client handle will then passed through the 
  output parameter <client>. 
*/

T_BAT_return bat_make_client_handle_from_gdd_and_client_id (T_GDD_CON_HANDLE con_handle, 
                                                            U8 client_id,
                                                            T_BAT_client *client)
{
  T_BAT_instance_maintain *dummy = NULL;
  T_BAT_instance inst_hndl = BAT_BROADCAST_CHANNEL;

  bat_get_instance_from_gdd_handle(con_handle, &inst_hndl, &dummy);
  *client = MAKE_CLNT_HNDL(client_id, inst_hndl);
  return (BAT_OK);
}

/*
+----------------------------------------------------------------------------+
| PROJECT :                    MODULE  : BINARY AT COMMAND LIBRARY           |
| STATE   : code     ROUTINE : bat_make_client_handle_from_gdd_and_client_id |
+----------------------------------------------------------------------------+

PURPOSE : 
  This function gets the client maintain data from the client handle.
*/

T_BAT_return bat_get_client_from_client_handle (T_BAT_client clnt_hndl, 
                                                T_BAT_client_maintain **clnt_mt)
{
  U8 inst_hndl = GET_INST_HNDL_FROM_CLNT_HANDLE(clnt_hndl);
  U8 clnt_id = GET_CLNT_ID_FROM_CLNT_HANDLE(clnt_hndl);
  T_BAT_instance_maintain *inst_mt = NULL;
  
  if (inst_hndl >= max_bat_num)
  {
    BAT_TRACE_ERROR ("bat_get_client_from_client_handle(): instance handle out of bound.");
    return (BAT_ERROR);
  }
  inst_mt = GET_INST (inst_hndl);
  if (inst_mt EQ NULL OR (clnt_id >= inst_mt->max_client_num))
  {
    BAT_TRACE_ERROR ("bat_get_client_from_client_handle(): instance not found or client id out of bound!");
    return (BAT_ERROR);
  }
  *clnt_mt = GET_CLNT(inst_mt, clnt_id);
  if (*clnt_mt EQ NULL)
  {
    return (BAT_ERROR);
  }
  return (BAT_OK);
}


/*
+----------------------------------------------------------------------------+
| PROJECT :                    MODULE  : BINARY AT COMMAND LIBRARY           |
| STATE   : code     ROUTINE : bat_make_client_handle_from_gdd_and_client_id |
+----------------------------------------------------------------------------+

PURPOSE : 
  This function is used to process the received unsolicited code. 
*/

T_BAT_return bat_unsolicited_code_rcv(T_BAT_instance inst_hndl, T_BAT_cmd_response *rsp)
{
  T_BAT_instance_maintain *inst_mt = NULL;

  BAT_TRACE_FUNCTION ("bat_unsolicited_code_rcv()");

  bat_get_instance_from_instance_handle (inst_hndl, &inst_mt);

  if (inst_mt->unsolicited_result_cb NEQ NULL)
  {
    if (inst_mt->unsolicited_result_cb(MAKE_UNS_CLNT_HNDL(inst_hndl),rsp) EQ BAT_BUSY_RESOURCE)
    {
      inst_mt->buffer.dest = MAKE_UNS_CLNT_HNDL(inst_hndl);
      bat_change_buffer_state(inst_mt, BAT_BUF_FILLED);
      return (BAT_BUSY_RESOURCE);
    }  
  }

  bat_change_buffer_state(inst_mt, BAT_BUF_EMPTY);

  return (BAT_OK);
}

/*
+----------------------------------------------------------------------------+
| PROJECT :                    MODULE  : BINARY AT COMMAND LIBRARY           |
| STATE   : code     ROUTINE : bat_make_client_handle_from_gdd_and_client_id |
+----------------------------------------------------------------------------+

PURPOSE : 
  This function is used to process the received control information. 
*/

T_BAT_return bat_control_confirm_rcv(T_BAT_instance inst_hndl, T_BATC_confirm cnf)
{
  T_BAT_client clnt_hndl = BAT_INVALID_CLIENT_HANDLE;
  T_BAT_instance_maintain *inst_mt = NULL;
  T_BAT_client_maintain *clnt_mt = NULL;

  BAT_TRACE_FUNCTION ("bat_control_confirm_rcv()");

  if (bat_get_instance_from_instance_handle (inst_hndl, &inst_mt))
  {
    bat_change_buffer_state(inst_mt, BAT_BUF_EMPTY);

    return (BAT_ERROR);
  }

  switch (cnf.rsp_params)
  {
    case (BATC_MAX_CLIENTS_CNF):
    case (BATC_MAX_CLIENTS_REJ):
    {
     /* check the instance state */
      if (inst_mt->instance_state NEQ BAT_INSTANCE_ACTIVATING)
      {
        BAT_TRACE_ERROR ("bat_control_confirm_rcv(): A conf/rej for bat_new is received in wrong BAT instance state");
        bat_change_buffer_state(inst_mt, BAT_BUF_EMPTY);
        return (BAT_ERROR);
      }
      if (cnf.rsp_params EQ BATC_MAX_CLIENTS_CNF)
      {
        BAT_TRACE_EVENT ("bat_control_confirm_rcv(): New instance confirmed");
        bat_change_instance_state(inst_mt,BAT_INSTANCE_READY);
        inst_mt->instance_signal_cb(BAT_NEW_INSTANCE_SUCCEED);
      }
      else if (cnf.rsp_params EQ BATC_MAX_CLIENTS_REJ)
      {
        BAT_TRACE_EVENT ("bat_control_confirm_rcv(): New instance rejected!");
        bat_change_instance_state(inst_mt,BAT_INSTANCE_IDLE);
        if (inst_mt->config->adapter.gdd_if.gdd_disconnect ((T_GDD_CON_HANDLE)(inst_mt->con_handle)) != GDD_OK)
        {
          BAT_TRACE_ERROR ("GDD disconnect failed after bat_new() was rejected");
        }
        L2P_Remove (inst_hndl);
        inst_mt->instance_signal_cb(BAT_NEW_INSTANCE_FAIL);
      }
      break;
    }
    case (BATC_OPEN_CLIENT_CNF):
    case (BATC_OPEN_CLIENT_REJ):
    {
      BAT_TRACE_EVENT ("bat_control_confirm_rcv(): Open client confirmed/rejected!");
      if (inst_mt->instance_state < BAT_INSTANCE_READY) 
      {
        BAT_TRACE_ERROR ("bat_control_confirm_rcv(): A conf/rej is received in wrong BAT instance state");
        BAT_TRACE_EVENT_P1 ("bat_control_confirm_rcv(): current BAT instance state is: %d", inst_mt->instance_state);
        bat_change_buffer_state(inst_mt, BAT_BUF_EMPTY);

        return (BAT_ERROR);
      }

      clnt_hndl = MAKE_CLNT_HNDL(cnf.rsp.ptr_bat_open_client_cnf->client_id, inst_hndl);

      if (bat_get_client_from_client_handle(clnt_hndl, &clnt_mt))
      {
        bat_change_buffer_state(inst_mt, BAT_BUF_EMPTY);

        return (BAT_ERROR);
      }
      if (clnt_mt->client_state NEQ BAT_CLIENT_ACTIVATING)
      {
        BAT_TRACE_ERROR ("bat_control_confirm_rcv(): A confirmation for bat_open is received in wrong BAT client state");
        bat_change_buffer_state(inst_mt, BAT_BUF_EMPTY);

        return (BAT_ERROR);
      }
      if (cnf.rsp_params EQ BATC_OPEN_CLIENT_CNF)
      {
        BAT_TRACE_EVENT_P1("bat_control_confirm_rcv(): The confirmed clnt hndl is 0x%x", clnt_hndl);
        bat_change_client_state(clnt_mt, BAT_CLIENT_READY);
        clnt_mt->signal_cb(clnt_hndl, BAT_OPEN_CLIENT_SUCCEED);
      }
      else if (cnf.rsp_params EQ BATC_OPEN_CLIENT_REJ)
      {
        bat_change_client_state(clnt_mt, BAT_CLIENT_IDLE);
        clnt_mt->signal_cb(clnt_hndl, BAT_OPEN_CLIENT_FAIL);
      }
      break;
    }
    case (BATC_ABORT_COMMAND_CNF):
    case (BATC_ABORT_COMMAND_REJ):
    {
      BAT_TRACE_EVENT ("bat_control_confirm_rcv(): Abort command confirmed/rejected!");
      if (inst_mt->instance_state < BAT_INSTANCE_READY) 
      {
        BAT_TRACE_ERROR ("bat_control_confirm_rcv():  An abort conf/rej is received in wrong BAT instance state");
        BAT_TRACE_EVENT_P1 ("bat_control_confirm_rcv(): current BAT instance state is: %d", inst_mt->instance_state);
        bat_change_buffer_state(inst_mt, BAT_BUF_EMPTY);

        return (BAT_ERROR);
      }

      clnt_hndl = MAKE_CLNT_HNDL(cnf.rsp.ptr_bat_open_client_cnf->client_id, inst_hndl);

      if (bat_get_client_from_client_handle(MAKE_CLNT_HNDL(cnf.rsp.ptr_bat_open_client_cnf->client_id, 
                                                           inst_hndl), &clnt_mt))
      {
        bat_change_buffer_state(inst_mt, BAT_BUF_EMPTY);

        return (BAT_ERROR);
      }

      if (cnf.rsp_params EQ BATC_ABORT_COMMAND_CNF)
      {
        bat_change_client_state(clnt_mt, BAT_CLIENT_READY);
        clnt_mt->signal_cb(clnt_hndl, BAT_ABORT_COMMAND_SUCCEED);
      }

      else if (cnf.rsp_params EQ BATC_ABORT_COMMAND_REJ)
      {
        bat_change_client_state(clnt_mt, BAT_CLIENT_SENDING);
        clnt_mt->signal_cb(clnt_hndl, BAT_ABORT_COMMAND_FAIL);
      }
      break;
    }
    default:
    {
      bat_change_buffer_state(inst_mt, BAT_BUF_EMPTY);

      return (BAT_ERROR);
    }
  }
  bat_change_buffer_state(inst_mt, BAT_BUF_EMPTY);

  return (BAT_OK);

}

/*
+----------------------------------------------------------------------------+
| PROJECT :                    MODULE  : BINARY AT COMMAND LIBRARY           |
| STATE   : code     ROUTINE : bat_make_client_handle_from_gdd_and_client_id |
+----------------------------------------------------------------------------+

PURPOSE : 
  This function is used to process the received command response. 
*/

T_BAT_return bat_command_response_rcv (T_BAT_instance inst_hndl, U8 clnt_id, T_BAT_cmd_response *rsp)
{
  T_BAT_client clnt_hndl = MAKE_CLNT_HNDL(clnt_id, inst_hndl);
  T_BAT_instance_maintain *inst_mt = NULL;
  T_BAT_client_maintain *clnt_mt = NULL;

  BAT_TRACE_FUNCTION ("bat_command_response_rcv()");

  if (bat_get_client_from_client_handle ( MAKE_CLNT_HNDL(clnt_id, inst_hndl), &clnt_mt))
  {
    return (BAT_ERROR);
  }
  
  if (bat_get_instance_from_instance_handle(inst_hndl, &inst_mt))
  {
    return (BAT_ERROR);
  }

  if (clnt_mt->client_state < BAT_CLIENT_READY)
  {
    BAT_TRACE_EVENT ("bat_command_response_rcv(): Client is closed!");
    bat_change_buffer_state(inst_mt, BAT_BUF_EMPTY);
    return (BAT_OK);
  }

  /* if the rsp is a finial rsp. */
  if (rsp->ctrl_response <= BAT_RES_PLUS_EXT_ERROR)
  {
    BAT_TRACE_EVENT ("bat_command_response_rcv(): FINAL RESULT CODE");
    bat_change_client_state(clnt_mt, BAT_CLIENT_READY);
  }
  else
  {
    BAT_TRACE_EVENT ("bat_command_response_rcv(): INTERMEDIATE CODE");
  }

  /* 
   * call now the callback function of the client and
   * check whether the application was able to process the callback.
   * if not then the buffer will not be empty and the destination of
   * the buffer will be the busy client handle.
   */
  if (clnt_mt->response_cb(clnt_hndl, rsp) EQ BAT_BUSY_RESOURCE)
  {
    inst_mt->buffer.dest = clnt_hndl; 
    return (BAT_BUSY_RESOURCE); 
  }
  
  bat_change_buffer_state(inst_mt, BAT_BUF_EMPTY);
  return (BAT_OK);
}


/*
+----------------------------------------------------------------------------+
| PROJECT :                    MODULE  : BINARY AT COMMAND LIBRARY           |
| STATE   : code               ROUTINE : bat_send_cmd_data                   |
+----------------------------------------------------------------------------+

PURPOSE : 
  This function is used to send the BAT command to ACI (via GDD). 
*/

T_BAT_return bat_send_cmd_data (T_BAT_client clnt_hndl, T_BAT_cmd_send *data)
{
  U16          size    = 0;
  int          errVal  = -1;
  void        *dataPtr = NULL;
  T_L2P_STATUS ret;

  if (data->ctrl_params & 0x1000) /* 0x1xxx = T_BAT_no_parameter */
  {
    size = sizeof(T_BAT_no_parameter); 
    dataPtr = (void *)(data->params.ptr_set_percent_als);
  }
  else
  if (data->ctrl_params & 0x2000) /* 0x2xxx = T_BAT_custom */
  {
/* <!-- ********** until ccdgen is able to cope with ***********
   search for the above comment in bat.sap and remove it
   when ccdgen is able to work with bat.sap
  
    if (data->params.ptr_custom->c_buf > BAT_MAX_CUSTOM_CMD_LEN)
    {
      return (BAT_ERROR);
    }
    size    = data->params.ptr_custom->c_buf + sizeof(U16);
    dataPtr = (void *)(data->params.ptr_custom);
*/
    return(BAT_ERROR); /* remove this whenn cddgen is able to */
  }
  else
  {
    size = (U16)(params_2_size[data->ctrl_params].size);  
    dataPtr = (void *)(data->params.ptr_set_percent_als);
  }

  BAT_TRACE_EVENT_P2("bat_send_cmd_data(): cmd = 0x%04X, size = %i", data->ctrl_params, size);

  BAT_TRACE_BINDUMP(xxx_handle,TC_USER4,"BAT COMMAND PARAMETER",(U8 *)data->params.ptr_at_a,size); 
 
  /*L2P is called here to stuff the data*/
  ret = L2P_Send(GET_INST_HNDL_FROM_CLNT_HANDLE(clnt_hndl), 
                 GET_CLNT_ID_FROM_CLNT_HANDLE(clnt_hndl), 
                 data->ctrl_params, 
                 dataPtr, 
                 size, 
                 &errVal);
  
  switch (ret)
  {
    case (L2P_STAT_SUCCESS):
    {
      return(BAT_OK);
    }
    case (L2P_STAT_UNKNOWN_ERROR):
    {
      BAT_TRACE_EVENT("bat_send_cmd_data(): L2P UNKNOWN ERROR");
      break;
    }
    case (L2P_STAT_MSG_SEND_FAIL):
    {
      BAT_TRACE_EVENT("bat_send_cmd_data(): L2P FAILED TO SEND");
      break;
    }
    case (L2P_STAT_INVALID_PARAMETER):
    {
      BAT_TRACE_EVENT("bat_send_cmd_data(): L2P INVALID PARAMETER");
      break;
    }    
    case (L2P_STAT_NO_SUCH_BAT_ID):
    {
      BAT_TRACE_EVENT("bat_send_cmd_data(): L2P UNKNOWN CLIENT HANDLE");
      break;
    }    
    case (L2P_STAT_NO_SUCH_MUX_ID):
    {
      BAT_TRACE_EVENT("bat_send_cmd_data(): L2P UNKNOWN CLIENT ID");
      break;
    }    
    case (L2P_STAT_UNABLE_TO_GET_BUFFER):
    {
      BAT_TRACE_EVENT("bat_send_cmd_data(): L2P NO TRANSMIT BUFFER AVAILABLE");
      return(BAT_BUSY_RESOURCE); 
    }
    case (L2P_STAT_BAT_ID_ALREADY_EXISTS):
    {
      BAT_TRACE_EVENT("bat_send_cmd_data(): L2P BAT ID ALREADY EXISTS");
      break;
    }    
    case (L2P_STAT_UNEXPECTED_FRAME_RXD):
    {
      BAT_TRACE_EVENT("bat_send_cmd_data(): L2P UNEXPECTED FRAME RECEIVED");
      break; 
    }       
    case(L2P_STAT_CRC_FAIL):
    {
      BAT_TRACE_EVENT("bat_send_cmd_data(): L2P CRC ERROR");
      break;
    }
    default:
    {
      BAT_TRACE_EVENT("bat_send_cmd_data(): L2P UNDEFINED RETURN VALUE");
      break;
    }  
  }
  return(BAT_ERROR);
}

/*
+----------------------------------------------------------------------------+
| PROJECT :                    MODULE  : BINARY AT COMMAND LIBRARY           |
| STATE   : code               ROUTINE : bat_send_ctrl_data                  |
+----------------------------------------------------------------------------+

PURPOSE : 
  This function is used to send the control info to ACI (via GDD). 
*/

T_BAT_return bat_send_ctrl_data (T_BAT_instance inst_hndl, T_BATC_signal *data)
{
  int errVal = -1;
  U16        size = 0;

  BAT_TRACE_FUNCTION ("bat_send_ctrl_data()");

  switch (data->ctrl_params)
  {
    case (BATC_MAX_CLIENTS):
    case (BATC_OPEN_CLIENT):
    case (BATC_ABORT_CMD):
    case (BATC_CLOSE_CLIENT):
      size = sizeof(T_BATC_max_clients);
      break;

    default:
      break;
  }  

 /*L2P has to be called here to stuff the data*/
  if (L2P_STAT_UNABLE_TO_GET_BUFFER EQ L2P_Send (inst_hndl, BAT_CONTROL_CHANNEL, 
                            data->ctrl_params, (void *)(data->params.ptr_max_clients), size, &errVal))
  {
    return (BAT_BUSY_RESOURCE);
  }

  return (BAT_OK);
}
/*
+----------------------------------------------------------------------------+
| PROJECT :                    MODULE  : BINARY AT COMMAND LIBRARY           |
| STATE   : code     ROUTINE : bat_make_client_handle_from_gdd_and_client_id |
+----------------------------------------------------------------------------+

PURPOSE : 
  This function is used to get the size of the BAT cmd basing on the ctrl param. 
*/

T_BAT_return bat_get_cmd_size (T_BAT_ctrl_params cmd_ctrl, int *size)
{
  BAT_TRACE_FUNCTION ("bat_get_cmd_size()");
  
  *size = params_2_size[cmd_ctrl].size;

  return (BAT_OK);
}
