/*******************************************************************************\
*                                                                           	*
*  TEXAS INSTRUMENTS ISRAEL PROPRIETARY AND CONFIDENTIAL INFORMATION			*
*																				*
*  LEGAL NOTE: THE TRANSFER OF THE TECHNICAL INFORMATION IS BEING MADE			*
*	UNDER AN EXPORT LICENSE ISSUED BY THE ISRAELI GOVERNMENT AND THAT THE		*
*	APPLICABLE EXPORT LICENSE DOES NOT ALLOW THE TECHNICAL INFORMATION TO		*
*	BE USED FOR THE MODIFICATION OF THE BT ENCRYPTION OR THE DEVELOPMENT		*
*	OF ANY NEW ENCRYPTION. THE INFORMATION CAN BE USED FOR THE INTERNAL			*
*	DESIGN AND MANUFACTURE OF TI PRODUCTS THAT WILL CONTAIN THE BT IC.			*
*																				*
\*******************************************************************************/
/*******************************************************************************\
*
*   FILE NAME:      browser.c
*
*   DESCRIPTION:    This file contains the code for building folder browsing
*					objects.
*
*   AUTHOR:         Ilan Elias
*
\*******************************************************************************/

#include "btl_config.h"
#if BTL_CONFIG_FTPS ==   BTL_CONFIG_ENABLED

/*******************************************************************************/
#include "bthal_fs.h"
#include "btl_defs.h"
#include "btl_utils.h"
#include "browser.h"

BTL_LOG_SET_MODULE(BTL_LOG_MODULE_TYPE_FTP);

/* Internal Function Prototypes */
static BOOL ReadFolder(U8 **buff, U16 *len);
static U8   VectorCopy(U8 **dest, const U8 **source, U16 *space);
static void __itoa(U8 *dest, U32 val);
static BOOL	checkHideMode(const BtlUtf8 *oblName);

/****************************************************************************
 *
 * Constants
 *
 ****************************************************************************/
#define BEGIN_XML_DECL      "<?xml version=\"1.0\" ?>\n<!-- This Folder Object generated by XTNDAccess Blue SDK -->\n"
#define BEGIN_FOLDER_DECL   "<!DOCTYPE folder-listing SYSTEM \"obex-folder-listing.dtd\">\n<folder-listing version=\"1.0\">\n"
#define END_FOLDER_DECL     "</folder-listing>\n"

#define BEGIN_DECL          BEGIN_XML_DECL BEGIN_FOLDER_DECL
#define NUM_VECTORS         18

/* XML Browser States */
#define XB_BEGIN_DECL       0
#define XB_COPY_BEGIN_DECL  1
#define XB_BROWSER_INIT     2
#define XB_OBJECT_SETUP     3
#define XB_BROWSE_NEXT      4
#define XB_END_DECL         5
#define XB_COPY_END_DECL    6
#define XB_COPY_OBJECT      7 /* XB_COPY = 7-12 */

/****************************************************************************
 *
 * RAM data
 *
 ****************************************************************************/
struct _BrowserContext {
    U8              state;
    BtlUtf8*		vectors[NUM_VECTORS];
    U8              scratch[16];    /* Holds __itoa() converted string */
    const BtlUtf8   *folderName;
	/* hide object arrary */
	U8*				hideObjects[BTL_CONFIG_FTPS_MAX_HIDE_OBJECT];
	BOOL			hideMode;

    /* FS specific variables */
    BthalFsDirDesc  searchHandle;
	BthalFsStat		fileInfo;
	BtlUtf8 		*fileName;
} Browser = {0};

/*---------------------------------------------------------------------------
 *            BROWSER_Open
 *---------------------------------------------------------------------------
 *
 * Synopsis:  Initialize the Browser for a new folder.
 *            
 *
 * Return:    OBRC_SUCCESS on success.
 *
 */
ObexRespCode BROWSER_Open(const BtlUtf8 *Folder,U8*	hideObjects[BTL_CONFIG_FTPS_MAX_HIDE_OBJECT],BOOL hideMode)
{
	U8		index;
    Browser.state = XB_BEGIN_DECL;
	
    Browser.folderName = Folder;
	for(index = 0;index < BTL_CONFIG_FTPS_MAX_HIDE_OBJECT;index++)
	{
		Browser.hideObjects[index] = hideObjects[index];
	}

	Browser.hideMode = hideMode;

    return OBRC_SUCCESS;
}


/*---------------------------------------------------------------------------
 *            BROWSER_ReadFlex
 *---------------------------------------------------------------------------
 *
 * Synopsis:  This function builds the XML Folder object wrappers and calls
 *            the OS dependent ReadFolder() function to insert the folder 
 *            contents.
 *
 * Return:    OBRC_SUCCESS on success.
 *
 */
ObexRespCode BROWSER_ReadFlex(U8 *buff, U16 *rLen, BOOL *more)
{
    U16     len = *rLen;

    *more = TRUE;           /* Assume we have more to write */

    switch (Browser.state) {
    case XB_BEGIN_DECL:
        Browser.vectors[0] = (BtlUtf8*)BEGIN_DECL;
        Browser.vectors[1] = 0;

        /* Drop through to next state */
        Browser.state++;

    case XB_COPY_BEGIN_DECL:
        /* Write XML Header and Folder Decl */
        if (VectorCopy(&buff,(const U8 **) &Browser.vectors[0], &len)) {
            /* Couldn't write complete declaration, were done. */
            break;
        }
        /* Drop through to next state */
        Browser.state++;

    default:
        /* Write folder object body */
        if (ReadFolder(&buff, &len))
            break;

        /* Drop through to next state */
        Browser.state = XB_END_DECL;

    case XB_END_DECL:
        /* Write XML End folder Decl */
        Browser.vectors[0] = (BtlUtf8*)END_FOLDER_DECL;
        Browser.vectors[1] = 0;
        /* Drop through to next state */
        Browser.state++;

    case XB_COPY_END_DECL:
        if (VectorCopy(&buff,(const U8 **) &Browser.vectors[0], &len)) {
            /* Couldn't write complete declaration, were done. */
            break;
        }

        /* Were done */
        *more = FALSE;
        Browser.state = XB_BEGIN_DECL;
        break;
    }

    /* Compute how much we read into buff */
    Assert(*rLen >= len);
    (*rLen) = (U16)((*rLen) - len);

    return OBRC_SUCCESS;
}

/*---------------------------------------------------------------------------
 *            ReadFolder
 *---------------------------------------------------------------------------
 *
 * Synopsis:  This function handles the OS dependent portion of folder object
 *            building. Its purpose is to walk through the objects in the
 *            specified folder and write them out to the provided buffer.
 *
 * Return:    TRUE if there is more work to do. FALSE when done.
 *
 */
static BOOL ReadFolder(U8 **buff, U16 *len)
{
    static BtlDateAndTimeString cTime;
    static BtlDateAndTimeString mTime;
    static BtlDateAndTimeString aTime;

    static BtlPermissionString userPerm;
    static BtlPermissionString groupPerm;
    static BtlPermissionString otherPerm;

    BOOL found = TRUE;
	U16 length;
	U8	i;

    while (*len && found) {
        switch (Browser.state) {
        case XB_BROWSER_INIT:
            if (BTHAL_FS_OpenDir(Browser.folderName, &Browser.searchHandle) != BT_STATUS_HAL_FS_SUCCESS) {
                BTL_LOG_INFO(("Failed to start search\n"));
                return FALSE;
            }

            /* Go to to next state */
            Browser.state = XB_BROWSE_NEXT;
			break;

        case XB_OBJECT_SETUP:
            /*	Check if the object is in hide mode	*/
			if ( 0 != OS_StrLenUtf8(Browser.fileName))
			{
				/* Figure out what type of object we need to record */
				if (Browser.fileInfo.type == BTHAL_FS_FILE) {
					i = 1;
	                Browser.vectors[0] =(BtlUtf8*)"   <file name=\""; 
	                Browser.vectors[1] = Browser.fileName;
					
                    if ( 0 != (BTHAL_FS_STAT_MASK & BTHAL_FS_S_SIZE))
					{
						__itoa(Browser.scratch, Browser.fileInfo.size);
						Browser.vectors[++i] = (BtlUtf8*)"\" size = \"";
	                	Browser.vectors[++i] = Browser.scratch;
					}				
					
                    if ( 0 != (BTHAL_FS_STAT_MASK & BTHAL_FS_S_MTIME))
					{
						Browser.vectors[++i] = (BtlUtf8*)"\" modified = \""; 
                        BTL_UTILS_XmlBuildAttrDateAndTime (&Browser.fileInfo.mTime, (U8*)mTime);
						Browser.vectors[++i] = mTime;
					}
					
                    if ( 0 != (BTHAL_FS_STAT_MASK & BTHAL_FS_S_CTIME))
					{
						Browser.vectors[++i] = (BtlUtf8*)"\" created = \""; 
                        BTL_UTILS_XmlBuildAttrDateAndTime (&Browser.fileInfo.cTime, (U8*)cTime);
						Browser.vectors[++i] = cTime;
					}
					
                    if ( 0 != (BTHAL_FS_STAT_MASK & BTHAL_FS_S_ATIME))
					{
						Browser.vectors[++i] = (BtlUtf8*)"\" accessed = \""; 
                        BTL_UTILS_XmlBuildAttrDateAndTime (&Browser.fileInfo.aTime, (U8*)aTime);
						Browser.vectors[++i] = aTime;
					}

                    if ( 0 != (BTHAL_FS_STAT_MASK & BTHAL_FS_S_USERPERM))
                    {
                
						Browser.vectors[++i] = (BtlUtf8*)"\" user-perm = \""; 
						BTL_UTILS_XmlBuildPermissions (Browser.fileInfo.userPerm, (U8*)userPerm);
                        Browser.vectors[++i] = userPerm;
                    }

                    if ( 0 != (BTHAL_FS_STAT_MASK & BTHAL_FS_S_GROUPPERM))
					{
					
                        Browser.vectors[++i] = (BtlUtf8*)"\" group-perm = \""; 
						BTL_UTILS_XmlBuildPermissions (Browser.fileInfo.groupPerm, (U8*)groupPerm);
                        Browser.vectors[++i] = groupPerm;
                    }

                    if ( 0 != (BTHAL_FS_STAT_MASK & BTHAL_FS_S_OTHERPERM))
					{
                        Browser.vectors[++i] = (BtlUtf8*)"\" other-perm = \""; 
						BTL_UTILS_XmlBuildPermissions (Browser.fileInfo.otherPerm, (U8*)otherPerm);
                        Browser.vectors[++i] = otherPerm;
					}
					
	                Browser.vectors[++i] = (BtlUtf8*)"\"/>\n";
	                Browser.vectors[++i] = 0;
	                Browser.state = XB_COPY_OBJECT;
	            } 
	            else if (Browser.fileInfo.type == BTHAL_FS_DIR) {
	                /* Skip . and convert .. to parent-folder identifier. */
	                if (Browser.fileName[0] == '.' && Browser.fileName[1] == '.') {
	                    Browser.vectors[0] = (BtlUtf8*)"    <parent-folder/>\n";
	                    Browser.vectors[1] = 0;
	                    Browser.state = XB_COPY_OBJECT;
	                }
	                else if (Browser.fileName[0] != '.') {
						i = 1;
	                    Browser.vectors[0] =(BtlUtf8*) "   <folder name=\"";
	                    Browser.vectors[1] = Browser.fileName;
						
                        if ( 0 != (BTHAL_FS_STAT_MASK & BTHAL_FS_S_SIZE))
						{
							__itoa(Browser.scratch, Browser.fileInfo.size);
							Browser.vectors[++i] = (BtlUtf8*)"\" size = \"";
		                	Browser.vectors[++i] = Browser.scratch;
						}			
						
                        if ( 0 != (BTHAL_FS_STAT_MASK & BTHAL_FS_S_MTIME))
						{
							Browser.vectors[++i] = (BtlUtf8*)"\" modified = \""; 
							BTL_UTILS_XmlBuildAttrDateAndTime (&Browser.fileInfo.mTime, (U8*)mTime);
                            Browser.vectors[++i] = mTime;
						}
						
                        if ( 0 != (BTHAL_FS_STAT_MASK & BTHAL_FS_S_CTIME))
						{
							Browser.vectors[++i] = (BtlUtf8*)"\" created = \""; 
                            BTL_UTILS_XmlBuildAttrDateAndTime (&Browser.fileInfo.cTime, (U8*)cTime);
							Browser.vectors[++i] = cTime;
						}
						
                        if ( 0 != (BTHAL_FS_STAT_MASK & BTHAL_FS_S_ATIME))
						{
							Browser.vectors[++i] = (BtlUtf8*)"\" accessed = \""; 
                            BTL_UTILS_XmlBuildAttrDateAndTime (&Browser.fileInfo.aTime, (U8*)aTime);
							Browser.vectors[++i] = aTime;
						}

                         if ( 0 != (BTHAL_FS_STAT_MASK & BTHAL_FS_S_USERPERM))
						{
							Browser.vectors[++i] = (BtlUtf8*)"\" user-perm = \""; 
						    BTL_UTILS_XmlBuildPermissions (Browser.fileInfo.userPerm, (U8*)userPerm);
                            Browser.vectors[++i] = userPerm;
                        }

                        if ( 0 != (BTHAL_FS_STAT_MASK & BTHAL_FS_S_GROUPPERM))
						{
						
                            Browser.vectors[++i] = (BtlUtf8*)"\" group-perm = \""; 
						    BTL_UTILS_XmlBuildPermissions (Browser.fileInfo.groupPerm, (U8*)groupPerm);
                            Browser.vectors[++i] = groupPerm;
                        }

                        if ( 0 != (BTHAL_FS_STAT_MASK & BTHAL_FS_S_OTHERPERM))
						{
                            Browser.vectors[++i] = (BtlUtf8*)"\" other-perm = \""; 
						    BTL_UTILS_XmlBuildPermissions (Browser.fileInfo.otherPerm, (U8*)otherPerm);
                            Browser.vectors[++i] = otherPerm;
						}
					
	                    Browser.vectors[++i] =(BtlUtf8*) "\"/>\n";
	                    Browser.vectors[++i] = 0;
	                    Browser.state = XB_COPY_OBJECT;
	                }
	                else Browser.state = XB_BROWSE_NEXT;
	            } else {
	                Browser.state = XB_BROWSE_NEXT;
	            }
			}else {
	                Browser.state = XB_BROWSE_NEXT;
			}
            break;

        default:
            Assert(Browser.state >= XB_COPY_OBJECT && Browser.state < XB_COPY_OBJECT+NUM_VECTORS);

            if (Browser.vectors[Browser.state - XB_COPY_OBJECT]) {
                if (VectorCopy(buff, (const U8 **)&Browser.vectors[Browser.state - XB_COPY_OBJECT], len)) {
                    break;
                }
                Browser.state++;
                break;
            }
            /* Done with this vector set, drop into next state */

        case XB_BROWSE_NEXT:
			if ((BTHAL_FS_ReadDir(Browser.searchHandle, &Browser.fileName) != BT_STATUS_HAL_FS_SUCCESS) || 
				(BTHAL_FS_Stat(Browser.fileName, &Browser.fileInfo) != BT_STATUS_HAL_FS_SUCCESS))
			{
				found = FALSE;
			}
			else
			{				
				/* Check if the object is in hide mode and hide mode enable */
				if ( (TRUE == checkHideMode(Browser.fileName)) && Browser.hideMode)
					OS_StrCpyUtf8(Browser.fileName,"" );
					
				/* Save only object name (without whole path) */
				length = OS_StrLenUtf8(Browser.fileName);
				while (length > 0)
				{
					if (Browser.fileName[length-1] == BTHAL_FS_PATH_DELIMITER)
					{
						Browser.fileName = &(Browser.fileName[length]);
						break;
					}
					
					length--;
				}
			}
			
            Browser.state = XB_OBJECT_SETUP;
            break;
        }
    }

    if (!found)
    {
    	BTHAL_FS_CloseDir(Browser.searchHandle);
        return FALSE;
    }

    Assert(*len == 0);
    return TRUE;
}

/*---------------------------------------------------------------------------
 *            VectorCopy
 *---------------------------------------------------------------------------
 *
 * Synopsis:  This string copy function copies the entire source string or
 *            up to 'space' bytes into 'dest'. It moves both buffer pointers
 *            and decrements the buffer length as it copies.
 *
 * Return:    Zero if entire string was copied. Non-zero otherwise
 */
static U8 VectorCopy(U8 **dest, const U8 **source, U16 *space)
{
    while (**source && *space) {
        *(*dest)++ = *(*source)++;
        (*space)--;
    }

    return **source;
}

/*---------------------------------------------------------------------------
 *            __itoa
 *---------------------------------------------------------------------------
 *
 * Synopsis:  Converts an unsigned 32-bit value to a base-10 number string.
 *
 * Return:    void.
 */
static void __itoa(U8 *out, U32 val)
{
    U32	rem;
    U8  str[11], *dest = str;

    if (val == 0)
        *dest++ = '0';

    while (val) {
        rem = val%10;
        *dest++ = (U8)(rem + '0');
        val = val/10;
    }

    while (--dest >= str)
        *out++ = *dest;

    *out = 0;
}
/*---------------------------------------------------------------------------
 *            checkHideMode
 *---------------------------------------------------------------------------
 *
 * Synopsis:  check in the hide array for the objName .
 *
 * Return:    TRUE if found other FALSE .
 */
static BOOL	checkHideMode(const BtlUtf8 *objName)
{
	U8 index;

	for(index = 0;index < BTL_CONFIG_FTPS_MAX_HIDE_OBJECT;index++)
	{
		if(BTL_UTILS_FS_StrCmpUTf8(Browser.hideObjects[index],objName) == 0) 
			return TRUE;
	}
	return FALSE;
}



#endif /*BTL_CONFIG_FTPS ==   BTL_CONFIG_ENABLED*/


