#!/usr/bin/perl
#
# ===================================================================================
# This file converts the logical locationconstratins to physical location constraints 
# using the logical to physical location mapping table.
# ===================================================================================
# 
# Turn on strict error checking
use strict;
use English;
use Getopt::Long;


# Some Revision Control Stuff
my $HEADER   = '$Header:   /pvcs/quartus/devices/stratixhc/sw2ic_loc_convert.pl_   22.0.1.0   04 Feb 2006 17:21:54   max  $';
my $SOURCE   = "<UNKNOWN>";
my $REVISION = "<UNKNOWN>";
if ($HEADER =~ m/^\S+\s+(\S+)\s+(\S+)/) {
   $SOURCE   = $1;
   $REVISION = $2;
}

# Usage Information
my $USAGE = "
${PROGRAM_NAME} is a Perl script that converts logical location constraints to physical location constraints
using the logical to physical location mapping table.

Usage:   ${PROGRAM_NAME} [options]
Options: -help                    Show this help, then exit successfully
         -version                 Show version information, then exit successfully
         -mapping    \"filename\"   Specify the input location mapping file. 
         -output     \"filename\"   Specify the output/converted location constraints file. 
         -constraint \"filename\"   Specify the input location constraints file. 
         -fpga_die   1s80           Specift which FPGA Prototype device is used. - Really, it should come from the constraint file.
         -debug      yes            Enable debug mode. It is diabled by default.
Examples:
         perl ${PROGRAM_NAME} -constraint constraints.apc -mapping hc1s80_loc.map -fpga_die 1s80 -output constraints.pla

";

# hard-coded string for placement
my $placement_string = "dbSetCellInstPlacement _cell";

# set up defaults for command line arguments, filenames.
my $input_filename      = undef;
my $output_filename     = undef;
my $mapping_filename    = undef;
my $debug_filename      = undef;

# Define some die dependent variable for max_row & max_col
my $max_row = undef;
my $max_col = undef;

# Define a die type
my $fpga_die = undef;


# hash table for logical and physical coordinate mappings
my %LOCATION_TABLE = ();


# assume no debug for default
my $DEBUG = undef;


# Call our main routine and pass in command line arguments
&main (@ARGV);
exit (0);


sub main 
{
   
   my $print_help    = 0;
   my $print_version = 0;

   GetOptions 
   (  "mapping=s"  => \$mapping_filename,
      "output=s"  => \$output_filename,
      "constraint=s"  => \$input_filename,
      "fpga_die=s"  => \$fpga_die,
      "debug=s"  => \$DEBUG,
      "help"        => \$print_help,
      "version"     => \$print_version
   );

   if ($print_help) {
      &print_help ();
      exit (0);
   }

   if ($print_version) {
      &print_version ();
      exit (0);
   }


   # Input location constraint file must be defined. If not, give an error message, print help, and exit
   if (!defined ($input_filename) ) {
      &print_file_not_defined ( "input" );
      &print_help ();
      exit (0);
   }

   # Mapping file must be defined. If not, give an error message, print help, and exit
   if (!defined ($mapping_filename) ) {
      &print_file_not_defined ( "mapping" );
      &print_help ();
      exit (0);
   }

   # If debug output filename is not specified, give adefault nam, using die size
   if (!defined ($debug_filename) ) {
      $debug_filename = "${fpga_die}_mapping.xls";
   }

   # FPGA die must be defined. If not, give an error message, print help, and exit
   if (!defined ($fpga_die) ) {
      print ("Please specify the fpga die used in prototyping.\n");
      &print_help ();
      exit (0);
   }

   # Set up the max_row & max_col based on fpga die type
   elsif ($fpga_die eq "1s80" || $fpga_die eq "1S80" || $fpga_die eq "ep1s80" || $fpga_die eq "EP1S80" )
   {
      $max_row = 91;
      $max_col = 124;
   }
   elsif ($fpga_die eq "1s60" || $fpga_die eq "1S60" || $fpga_die eq "ep1s60" || $fpga_die eq "EP1S60" )
   {
      $max_row = 73;
      $max_col = 112;
   }
   # add 1S30/40 support
   elsif ($fpga_die =~ m/[ep]?1s40/i)
   {
      $max_row = 61;
      $max_col = 95;
   }   
   elsif ($fpga_die =~ m/[ep]?1s30/i)
   {
      $max_row = 57;
      $max_col = 84;
   }   
   # add 1S25 support
   elsif ($fpga_die =~ m/[ep]?1s25/i)
   {
      $max_row = 46;
      $max_col = 78;
   }
   else
   {
      die ("Error: unsupported fpga die type $fpga_die! \n\n");
   }


   if(defined($DEBUG))
   {
      print("\nInfo: Maximun number of (row, col) in $fpga_die is ($max_row, $max_col) \n");
   }


   # if the output file is not defined, give a default file name
   if (!defined ($output_filename) ) {
      $output_filename = "converted_constraints.pla";
   }

   # read in the mapping table
   &build_mapping_table();
   
   # read in the logical location constraints
   &convert_constraints();
   

   # It is done here
   print("\nInfo: --------------- done! ----------------\n");
   exit (0);
}


# ======================================================
# Read in the logical to physical location mapping table
# ======================================================
sub build_mapping_table
{

   # Define some local variables
   my $hash_key = undef;
   my $hash_value = undef;

   my $logical_x = undef;
   my $logical_y = undef;

   my $physical_x = undef;
   my $physical_y = undef;

   my $origin = undef;

   my $blk_type = undef;

   my $line = undef;
   my $line_num = undef;

   my $type_lab  = 0;
   my $type_seab = 0;
   my $type_meab = 0;
   my $type_mac  = 0;
   my $type_mram = 0;

   # Open the mapping file
   if(!open(MAPPING_FILE_HDL, "<$mapping_filename"))
   {
      die ("Error: cannot open $mapping_filename for reading");
   }

   if(defined($DEBUG))
   {
      print("\nInfo: Start reading the mapping file $mapping_filename.\n\n");
   }

   # get all mappings
   $line_num = 0;
   while ($line = <MAPPING_FILE_HDL>)
   {

      if ($line =~ m/(\S+)\s*(\S+)\s*(\S+)\s*(\S+)\s*(\S+)\s*(\S+)/)
      {

         $line_num++;

         # ignore the comment line
         ($line =~ m/^#/) && next;
         
         # Read in all the relavent fields
         $blk_type = $1;
         $logical_x = $2;
         $logical_y = $3;
         $physical_x = $4;
         $physical_y = $5;
         $origin = $6;

         # Check the block types and update the counters
         if ($blk_type eq "LAB")
         {
            $type_lab++;
         }
         elsif ($blk_type =~ m/SEAB|SMALL_EAB/)
         {
            $blk_type = "M512";
            $type_seab++;
         }
         elsif ($blk_type =~ m/MEAB|MEDIUM_EAB/)
         {
            $blk_type = "M4K";
            $type_meab++;
         }
         elsif ($blk_type eq "MAC")
         {
            $blk_type = "DSP";
            $type_mac++;
         }
         elsif ($blk_type =~ m/MRAM|MEGA_RAM/)
         {
            $blk_type = "MRAM";
            $type_mram++;
         }
         else
         {
            die ("Error: unsupported block type $blk_type\n");
         }


         # define a logical xy pair as key
         $hash_key = "${logical_x}_$logical_y";
         $hash_value = [$blk_type, $physical_x, $physical_y, $origin];

         # Check to see if the key already in the table
         if (exists($LOCATION_TABLE{$hash_key}))
         {
            die ("Error: duplicated {$logical_x, $logical_y} location in the table - check your table again!\n");
         }
         else
         {
            $LOCATION_TABLE{$hash_key} = $hash_value;
         }

      }

   }

   # print line number information
   print("\nInfo: $line_num entries were processed from the mapping table.\n");
   print("        $type_lab \t LAB blocks \n");
   print("        $type_seab \t SEAB blocks \n");
   print("        $type_meab \t MEAB blocks \n");
   print("        $type_mram \t MRAM blocks \n");
   print("        $type_mac \t MAC blocks \n");
   print("\n");


   # Close file
   close ( MAPPING_FILE_HDL );

}


# =============================================================
# Read in the constraints from logical location constraint file
# =============================================================
sub convert_constraints
{

   # Define some local variables
   my $hash_key = undef;
   my $hash_value = undef;
   my $logical_x = undef;
   my $logical_y = undef;
   my $physical_x = undef;
   my $physical_y = undef;

   my $string   = undef;

   my $line     = undef;
   my $line_num = undef;

   my $blk_type = undef;

   my $blk_fpga = undef;
   my $blk_hc   = undef;

   my $fpga_row = undef;
   my $fpga_col = undef;
   my $rba_row  = undef;
   my $rba_col  = undef;

   my $rba_name = undef;

   #Open the input constraint file
   if(!open(INPUT_FILE_HDL, "<$input_filename"))
   {
      die("Error: cannot open $input_filename for reading");
   }

   #Open the output constraint file
   if(!open(OUTPUT_FILE_HDL, ">$output_filename"))
   {
      die("Error: cannot open $output_filename for writing");
   }

   #Open the output constraint file
   if(!open(DEBUG_FILE_HDL, ">$debug_filename"))
   {
      die("Error: cannot open $debug_filename for writing");
   }

   print("\nInfo: start to read the input constraint file $input_filename.\n");

   # get all mappings
   $line_num = 0;
   while ($line = <INPUT_FILE_HDL>)
   {

      $line_num++;

      if ($line =~ m/s*\[([^_]+)_X(\d+)_Y(\d+)\]\s+\S+\s+\[([^_]+)_X(\d+)_Y(\d+)\s*/)
      {
         # then the line define at least one block
         $blk_fpga  = $1;
         $fpga_col  = $2;
         $fpga_row  = $3;
         $blk_hc    = $4;
         $logical_x = $5;
         $logical_y = $6;

         if(defined($DEBUG))
         {
            print ("Info: line $line_num ( $blk_fpga $fpga_row $fpga_col) to ( $blk_hc $logical_x $logical_y )\n");
         }

         # Check to make sure that FPGA type matches HC type
         if ($blk_fpga ne $blk_hc)
         {
            die ("Error: FPGA block type does not match HC in the constraint file $input_filename. \n");
         }

         # Convert the FPGA (row col) numbers to RBA (row col) numbers
         $rba_row = $max_row - $fpga_row;
         $rba_col = $max_col - $fpga_col;


         # Convert the fpga name to IC format and applying offsets
         if ($blk_fpga eq "LAB")
         {
         }
         elsif ($blk_fpga eq "M512")
         {
            $blk_fpga = "SEAB";
         }
         elsif ($blk_fpga eq "M4K")
         {
            $blk_fpga = "MEAB";
            $rba_col = $rba_col - 1;
         }
         elsif ($blk_fpga eq "DSP")
         {
            $blk_fpga = "MAC";
            $rba_row = $rba_row - 7;
            $rba_col = $rba_col - 1;
         }
         elsif ($blk_fpga eq "MRAM")
         {
            $rba_row = $rba_row - 12;
            $rba_col = $rba_col - 12;
         }
         else
         {
            die ("Error: unsupported block type $blk_fpga \n");
         }


         # Senity checks
         if ($rba_row < 0 || $rba_col < 0)
         {
            die ("Error: FPGA ($fpga_row, $fpga_col) in constraint file bigger than MAX ($max_row, $max_col) \n");
         }

         # Construct RBA Instance Name
         $rba_name = "R${rba_row}_C${rba_col}_$blk_fpga";

         if(defined($DEBUG))
         {
            print ("Info: line $line_num ( $blk_fpga $fpga_row $fpga_col) to $rba_name to HC ( $blk_hc $logical_x $logical_y )\n");
         }

         # Create coordinate pair
         $hash_key = "${logical_x}_$logical_y";

         if (exists($LOCATION_TABLE{$hash_key}))
         {
            @{$hash_value} = @{$LOCATION_TABLE{$hash_key}};

            if ( $blk_hc eq $hash_value->[0] )
            {
               $physical_x = $hash_value->[1];
               $physical_y = $hash_value->[2];
               $string     = $hash_value->[3];

               if (defined($DEBUG))
               {
                  print ("$placement_string '($physical_x $physical_y)\n");
               }

               # change to output file
               select (OUTPUT_FILE_HDL);

               printf("$placement_string \"$rba_name\" \"0\" \"$string\" \"origin\" '($physical_x $physical_y) \n");

               # change to debug output file
               select (DEBUG_FILE_HDL);

               print("${blk_fpga}_X${fpga_col}_Y${fpga_row}\t$fpga_col\t$fpga_row\t${blk_hc}_X${logical_x}_Y${logical_y}\t$logical_x\t$logical_y\t$rba_name\t$physical_x\t$physical_y\n");


               # change back to STDOUT 
               select(STDOUT);

               # Empty the table when a match block is placed in the location
               delete ( $LOCATION_TABLE{$hash_key} );

            }
            else
            {
               die ("Error: Block type $blk_hc does not match the type in TABLE location ($logical_x, $logical_y) \n");
            }

         }
         else
         {
            die ("Error: Location {$logical_x, $logical_y} not in the mapping table or has been occupied!\n");
         }

      }

   }

   # print line number information
   print("\nInfo: totally $line_num entries were processed from the constraint file.\n");

   # Close file
   close ( INPUT_FILE_HDL );
   close ( OUTPUT_FILE_HDL );

}


# ==============
# Print the help
# ==============
sub print_help 
{
# This routine prints the help screen.

   print ($USAGE);
}


# ======================
# Print an error message
# ======================
sub print_file_not_defined ( ) 
{
# This routine prints the help screen.
   my $file = $_[0];
   print ("Please specify the $file filename\n");
}


# ========================
# Print the version number
# ========================
sub print_version 
{
# This routine print version information.

   print ("${PROGRAM_NAME} (Revision ${REVISION} of ${SOURCE})\n");
}
