
#############################################################################
##  remote-compile.tcl
##
##  Design Space Explorer remote compilation script for parallel and
##  distributed exploration flows.
##
##  ALTERA LEGAL NOTICE
##  
##  This script is  pursuant to the following license agreement
##  (BY VIEWING AND USING THIS SCRIPT, YOU AGREE TO THE
##  FOLLOWING): Copyright (c) 2006-2007 Altera Corporation, San Jose,
##  California, USA.  Permission is hereby granted, free of
##  charge, to any person obtaining a copy of this software and
##  associated documentation files (the "Software"), to deal in
##  the Software without restriction, including without limitation
##  the rights to use, copy, modify, merge, publish, distribute,
##  sublicense, and/or sell copies of the Software, and to permit
##  persons to whom the Software is furnished to do so, subject to
##  the following conditions:
##  
##  The above copyright notice and this permission notice shall be
##  included in all copies or substantial portions of the Software.
##  
##  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
##  EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES
##  OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
##  NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT
##  HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY,
##  WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
##  FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
##  OTHER DEALINGS IN THE SOFTWARE.
##  
##  This agreement shall be governed in all respects by the laws of
##  the State of California and by the laws of the United States of
##  America.
##
##
##  CONTACTING ALTERA
##  
##  You can contact Altera through one of the following ways:
##  
##  Mail:
##     Altera Corporation
##     Applications Department
##     101 Innovation Drive
##     San Jose, CA 95134
##  
##  Altera Website:
##     www.altera.com
##  
##  Online Support:
##     www.altera.com/mysupport
##     
##  Troubshooters Website:
##     www.altera.com/support/kdb/troubleshooter
##  
##  Technical Support Hotline:
##     (800) 800-EPLD or (800) 800-3753
##        7:00 a.m. to 5:00 p.m. Pacific Time, M-F 
##     (408) 544-7000
##        7:00 a.m. to 5:00 p.m. Pacific Time, M-F 
##  
##     From other locations, call (408) 544-7000 or your local
##     Altera distributor.
##  
##  The mySupport web site allows you to submit technical service
##  requests and to monitor the status of all of your requests
##  online, regardless of whether they were submitted via the
##  mySupport web site or the Technical Support Hotline. In order to
##  use the mySupport web site, you must first register for an
##  Altera.com account on the mySupport web site.
##  
##  The Troubleshooters web site provides interactive tools to
##  troubleshoot and solve common technical problems.
##


#############################################################################
##  Additional Packages Required
package require cmdline
package require xml
package require ::quartus::dse::result
package require ::quartus::flow
package require ::quartus::misc
package require ::quartus::project


#############################################################################
##  Export Functions & Declare Global Namespace Variables


#############################################################################
##  Command Line Options for Script
set         tlist       "run-power.arg"
lappend     tlist       0
lappend     tlist       "Run Quartus II PowerPlay Power Analyzer as part of compile"
lappend function_opts $tlist

set         tlist       "hardcopy.arg"
lappend     tlist       0
lappend     tlist       "Run Quartus II Hardcopy flow"
lappend function_opts $tlist

set         tlist       "save-rpt-files.arg"
lappend     tlist       0
lappend     tlist       "Save the rpt files even if the compile succeeds"
lappend function_opts $tlist

set         tlist       "point.arg"
lappend     tlist       "#_required_#"
lappend     tlist       "DSE point we're compiling with this remote call"
lappend function_opts $tlist

set         tlist       "project.arg"
lappend     tlist       "#_required_#"
lappend     tlist       "Name of project we're compiling with this remote call"
lappend function_opts $tlist

set         tlist       "revision.arg"
lappend     tlist       "#_required_#"
lappend     tlist       "Name of revision we're compiling with this remote call"
lappend function_opts $tlist

set         tlist       "lower-priority.arg"
lappend     tlist       0
lappend     tlist       "Lower the priority of compiles"
lappend function_opts $tlist

set         tlist       "do-combined-analysis.arg"
lappend     tlist       0
lappend     tlist       "True if we are doing a combined analysis and need to return two result objects"
lappend function_opts $tlist

array set optshash [cmdline::getFunctionOptions ::quartus(args) $function_opts]

# Check that we got all our options
foreach opt [array names optshash] {
    post_message -type info "Check option $opt = $optshash($opt)"
    if {$optshash($opt) == "#_required_#"} {
        post_message -type error "Missing required option: -$opt"
        exit 1
    }
}

# Create a ::dse::result object that holds the settings
post_message "Creating result object for setting and data storage"
set resultobj [uplevel #0 ::quartus::dse::result #auto $optshash(project) $optshash(revision)]

# Load the XML for this result that contains all the settings.
# Skip the load if the point is "base".
set xmlfile "$optshash(point)-dse-result.xml"
if {![string equal $optshash(point) "base"]} {
    if {![file exists $xmlfile]} {
        post_message -type error "Missing required XML file: $xmlfile"
        exit 1
    }
    post_message "Loading result settings from $xmlfile"
    set xmlfh [open $xmlfile "r"]
    $resultobj loadXML $xmlfh
    close $xmlfh
}

# Make sure we have a base.qar file
if {![file exists base.qar]} {
    post_message -type error "Missing required Quartus II Archive file: base.qar"
    exit 1
}

post_message "Unarchiving base.qar"
project_restore -overwrite -update_included_file_info "base"
# Skip this part if the point is "base"
if {![string equal $optshash(point) "base"]} {
    # We can always use makeHardCopyRevision to apply the settings
    # to the project because we don't have to concern ourselves
    # with restoring the base results at the end since this compile
    # is being done on a unique instance of the project.
    post_message "Applying settings from result object to project"
    $resultobj makeHardCopyRevision
}
# Delete any existing .rpt files before we start
catch {file delete -force [glob -nocomplain -- {*.rpt}]}
post_message "Opening project $optshash(project) with revision $optshash(revision)"
project_open -revision $optshash(revision) $optshash(project)

post_message "Compling project"
set save_rpt_files $optshash(save-rpt-files)
if {[catch {execute_flow -compile} result]} {
    post_message -type error "Error compiling project"
    post_message -type error $result
    set save_rpt_files 1
} else {
    post_message "Compilation successful"
}
if {$optshash(run-power)} {
    post_message "Running PowerPlay power analyzer on project"
    if {[catch {execute_module -tool pow} result]} {
        post_message -type error "Error running quartus_pow on project"
        post_message -type error $result
        set save_rpt_files 1
    } else {
        post_message "Power analysis successful"
    }
}
post_message "Achiving project as $optshash(point).qar"
project_archive -include_outputs -overwrite $optshash(point).qar
project_close

# Should we save the report files?
if {$save_rpt_files} {
    set repat "$optshash(revision)\\.(\[a-z\]+)(\\.rpt.*)"
    foreach {f} [lsort [glob -nocomplain -- {*}]] {
        if {[regexp -nocase -- $repat $f => rtype ext]} {
            # Skip the .dse.rpt file
            if {![string equal -nocase $rtype "dse"]} {
                set newf "$optshash(revision).$optshash(point).${rtype}${ext}"
                catch {file copy -force $f $newf}
                lappend flist $newf
            }
        }
    }
    if {[llength $flist] > 0} {
        post_message -type warning "Output from failed compilation has been saved as:"
        foreach f $flist {
            post_message -type warning "   $f"
        }
    } else {
        post_message -type warning "No output from failed compilation was found"
    }
}

# Save the results back to the XML file
post_message "Parsing results from project"
if {$optshash(do-combined-analysis)} {
    set resultobj_fast [$resultobj duplicate]
    $resultobj_fast getResultsFromProject -model fast
    $resultobj_fast setCompiled 1
    set xmlfile "$optshash(point)-fast-dse-result.xml"
    post_message "Dumping parsed results back to $xmlfile"
    set xmlfh [open $xmlfile {WRONLY CREAT TRUNC}]
    $resultobj_fast dumpXML $xmlfh
    close $xmlfh

    $resultobj getResultsFromProject -model slow
    $resultobj setCompiled 1
    set xmlfile "$optshash(point)-dse-result.xml"
    post_message "Dumping parsed results back to $xmlfile"
    set xmlfh [open $xmlfile {WRONLY CREAT TRUNC}]
    $resultobj dumpXML $xmlfh
    close $xmlfh
} else {
    $resultobj getResultsFromProject
    $resultobj setCompiled 1
    post_message "Dumping parsed results back to $xmlfile"
    set xmlfh [open $xmlfile {WRONLY CREAT TRUNC}]
    $resultobj dumpXML $xmlfh
    close $xmlfh
}

post_message "Done"
exit 0

