
####################################################################################
## dse-stratixii-lib.tcl - v1.0
##
## A set of library routines used to power the Design Space Explorer script that
## are specific to the Altera Stratix II family of devices and its derivatives.
## 
##  ALTERA LEGAL NOTICE
##  
##  This script is  pursuant to the following license agreement
##  (BY VIEWING AND USING THIS SCRIPT, YOU AGREE TO THE
##  FOLLOWING): Copyright (c) 2006-2007 Altera Corporation, San Jose,
##  California, USA.  Permission is hereby granted, free of
##  charge, to any person obtaining a copy of this software and
##  associated documentation files (the "Software"), to deal in
##  the Software without restriction, including without limitation
##  the rights to use, copy, modify, merge, publish, distribute,
##  sublicense, and/or sell copies of the Software, and to permit
##  persons to whom the Software is furnished to do so, subject to
##  the following conditions:
##  
##  The above copyright notice and this permission notice shall be
##  included in all copies or substantial portions of the Software.
##  
##  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
##  EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES
##  OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
##  NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT
##  HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY,
##  WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
##  FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
##  OTHER DEALINGS IN THE SOFTWARE.
##  
##  This agreement shall be governed in all respects by the laws of
##  the State of California and by the laws of the United States of
##  America.
##
##
##  CONTACTING ALTERA
##  
##  You can contact Altera through one of the following ways:
##  
##  Mail:
##     Altera Corporation
##     Applications Department
##     101 Innovation Drive
##     San Jose, CA 95134
##  
##  Altera Website:
##     www.altera.com
##  
##  Online Support:
##     www.altera.com/mysupport
##     
##  Troubshooters Website:
##     www.altera.com/support/kdb/troubleshooter
##  
##  Technical Support Hotline:
##     (800) 800-EPLD or (800) 800-3753
##        7:00 a.m. to 5:00 p.m. Pacific Time, M-F 
##     (408) 544-7000
##        7:00 a.m. to 5:00 p.m. Pacific Time, M-F 
##  
##     From other locations, call (408) 544-7000 or your local
##     Altera distributor.
##  
##  The mySupport web site allows you to submit technical service
##  requests and to monitor the status of all of your requests
##  online, regardless of whether they were submitted via the
##  mySupport web site or the Technical Support Hotline. In order to
##  use the mySupport web site, you must first register for an
##  Altera.com account on the mySupport web site.
##  
##  The Troubleshooters web site provides interactive tools to
##  troubleshoot and solve common technical problems.
##

package provide ::quartus::dse::stratixii 1.0

#############################################################################
##  Additional Packages Required
package require ::quartus::dse::ccl 1.0
package require ::quartus::misc 1.0
package require cmdline


#############################################################################
##  Export Functions & Declare Global Namespace Variables
namespace eval ::quartus::dse::stratixii {
    namespace export is_valid_type
    namespace export get_valid_types
    namespace export get_description_for_type
    namespace export set_design_space
    namespace export has_quick_recipe_for
    namespace export get_quick_recipe_for

    # Declare Global Variables Here!
    # DO NOT EXPORT ANY OF THESE!
    
    variable pvcs_revision [regsub -nocase -- {\$revision:\s*(\S+)\s*\$} {$Revision:   22.0.1.1  $} {\1}]
    
    # NOTE: If you want this family to support additional exploration
    #       types you need to update this list! "Custom Space" should
    #       ALWAYS be LAST on this list please!
    variable valid_types [list "Seed Sweep" "Extra Effort Space" "Physical Synthesis Space" "Physical Synthesis with Retiming Space" "Extra Effort Space for Quartus II Integrated Synthesis Projects" "Physical Synthesis Space for Quartus II Integrated Synthesis Projects" "Physical Synthesis with Retiming Space for Quartus II Integrated Synthesis Projects" "Area Optimization Space" "Power Optimization Space" "Signature: Placement Effort Multiplier" "Signature: Netlist Optimizations" "Signature: Fast Fit" "Signature: Register Packing" "Custom Space"]
}


#############################################################################
##  Procedure:  is_valid_type
##
##  Arguments:
##      _t
##          The type you want to test for validity.
##
##  Description:
##      Tells you if a type of exploration is supported by this family or not.
##      If this type is supported then the function returns 1, otherwise it
##      returns 0.
proc ::quartus::dse::stratixii::is_valid_type {_t} {

    # Get our global list of valid exploration types
    variable valid_types

    foreach {t} $valid_types {
        if {[string equal $_t $t]} {
            return 1
        }
    }
    return 0
}


#############################################################################
##  Procedure:  get_valid_types
##
##  Arguments:
##      <none>
##
##  Description:
##      Returns a list of valid exploration types for this family.
proc ::quartus::dse::stratixii::get_valid_types {} {
    
    # Get our global list of valid exploration types
    variable valid_types
    return $valid_types
}


#############################################################################
##  Procedure:  get_description_for_type
##
##  Arguments:
##      _t
##
##  Description:
##      Returns a string that holds a paragraph description of the
##      type $_t. Returns an empty string if $_t is not a valid
##      type.
proc ::quartus::dse::stratixii::get_description_for_type {_t} {
    
    set debug_name "stratixii::get_description_for_type()"

    set retstr ""
    
    # You need to keep this up to date!
    switch -exact -- $_t {
        {Seed Sweep} {
            append retstr "A simple seed sweep."
        }
        {Extra Effort Space} {
            append retstr "A seed sweep plus an increase in the fitting effort level with "
            append retstr "and without register packing."
        }
        {Physical Synthesis Space} {
            append retstr "A seed sweep combined with various netlist optimization "
            append retstr "algorithms that do not move registers in your design "
            append retstr "and an increase in the fitting effort level with and "
            append retstr "without register packing."
        }
        {Physical Synthesis with Retiming Space} {
            append retstr "A seed sweep combined with various netlist optimization "
            append retstr "algorithms that may move registers in your design "
            append retstr "and an increase in the fitting effort level with and "
            append retstr "without register packing."
        }
        {Extra Effort Space for Quartus II Integrated Synthesis Projects} {
            append retstr "A seed sweep plus an increase in the fitting "
            append retstr "effort level with and without register packing "
            append retstr "and various Quartus II Integrated Synthesis optimizations."
        }
        {Physical Synthesis Space for Quartus II Integrated Synthesis Projects} {
            append retstr "A seed sweep combined with various netlist "
            append retstr "optimization algorithms that do not move registers in your "
            append retstr "design and an increase in the fitting effort level with and "
            append retstr "without register packing "
            append retstr "and various Quartus II Integrated Synthesis optimizations."
        }
        {Physical Synthesis with Retiming Space for Quartus II Integrated Synthesis Projects} {
            append retstr "A seed sweep combined with various netlist "
            append retstr "optimization algorithms that may move registers in your "
            append retstr "design and an increase in the fitting effort level with and "
            append retstr "without register packing "
            append retstr "and various Quartus II Integrated Synthesis optimizations."
        }
        {Area Optimization Space} {
            set retstr "A seed sweep combined with various synthesis settings "
            append retstr "and fitter register packing designed to minimize area. This "
            append retstr "exploration space should be run with an optimization goal that "
            append retstr "determines the best results based on area."
        }
        {Power Optimization Space} {
            set retstr "A seed sweep combined with various synthesis settings "
            append retstr "and fitter settings designed to minimize power consumption "
            append retstr "in your design. This exploration space should be run with "
            append retstr "an optimization goal that determines the best results based "
            append retstr "on power use in a design."
        }
        {Signature: Placement Effort Multiplier} {
			set retstr "An exploration of Fitter placement effort. Each of the following levels "
			append retstr "are explored for each seed you provide: 1.0, 3.0, 5.0, "
			append retstr  "and 10.0. Higher Placement Effort Multiplier values increase "
			append retstr  "compilation time, but may improve placement quality. This "
			append retstr  "exploration space should be run with an Exhaustive search method."
        }
        {Signature: Netlist Optimizations} {
            set retstr "An exploration of the available netlist optimization algorithms. "
            append retstr "Each of the following groups of netlist algorithms are applied "
            append retstr "for each seed you provide: Off, On and On with Register Moves. "
            append retstr "This exploration type should be run with an Exhaustive "
            append retstr "search method."
        }
        {Signature: Fast Fit} {
            set retstr "An exploration of the available fast fit options. "
            append retstr "Each of the following fast fit options are applied "
            append retstr "for each seed you provide: Off and On. "
            append retstr "This exploration type should be run with an Exhaustive "
            append retstr "search method."
        }
        {Signature: Register Packing} {
            set retstr "An exploration of the available register packing options. "
            append retstr "Each of the following register packing options are applied "
            append retstr "for each seed you provide: Off, Normal, Minimize Area, and Minimize Area with Chains. "
            append retstr "This exploration type should be run with an Exhaustive "
            append retstr "search method."
        }
        {Custom Space} {
            set retstr "Use a custom exploration space defined in a file. "
            append retstr "Use the \"Options -> Custom Space file...\" dialog to choose "
            append retstr "a file to load. Your seeds in the main window will be added to "
            append retstr "any existing seeds in the file."
        }   
        default {
            # Huh? How'd we get here?
            ::quartus::dse::ccl::dputs "${debug_name}: Error: No description for \"$_t\" found"
        }
    }

    return $retstr
}


#############################################################################
##  Procedure:  has_quick_recipe_for
##
##  Arguments:
##      _t
##
##  Description:
##      Checks this family library to see if it contains a quick
##      recipie for type _t. You can pass it anything at all for
##      _t but really it should be a string like "power" or
##      "area" or "speed: low".
##
##      The GUI uses this method to contruct the appropriate user
##      interface experience when ever a design is loaded into DSE.
##
##      Returns 1 if a quick recipie exists. 0 otherwise.
proc ::quartus::dse::stratixii::has_quick_recipe_for {_t} {
    set debug_name "stratixii::has_quick_recipe_for()"

    # This library supports the following quick reipes:
    switch -- $_t {
        "area" -
        "speed:low" -
        "speed:medium" -
        "speed:high" -
        "speed:highest" -
        "power" {
            set retval 1
        }
        default {
            ::quartus::dse::ccl::dputs "${debug_name}: There is no quick recipie for $_t in this library"
            set retval 0
        }
    }
    return $retval
}


#############################################################################
##  Procedure:  get_quick_recipe_for
##
##  Arguments:
##      -recipe <name>
##          The name of the recipie you want the secret sauce for
##
##      -qii-synthesis [0|1]
##          Set this to true if the user has indicated their design
##          takes advantage of QII native synthesis. It defaults
##          to 0 (or off).
##
##  Description:
##      If the recipe is supported by this library it returns a list
##      with four elements in it. The first element is the name of
##      the design space to load. The second element is optimization
##      goal method to use. The third element is the optimization
##      function to use. The fourth element is the search method
##      to use.
##
##      If this recipe is not supported it returns an empty list.
proc ::quartus::dse::stratixii::get_quick_recipe_for {args} {
    set debug_name "stratixii::get_quick_recipe_for()"

    set         tlist       "qii-synthesis.arg"
    lappend     tlist       0
    lappend     tlist       "True if the circuit is using QII integrated synthesis"
    lappend function_opts $tlist

    set         tlist       "recipe.arg"
    lappend     tlist       "unknown"
    lappend     tlist       "Name of the secret sauce recipe you seek"
    lappend function_opts $tlist

    array set optshash [cmdline::getFunctionOptions args $function_opts]

    # This library supports the following quick reipes:
    switch -- $optshash(recipe) {
        "area" {
            set s {Area Optimization Space}
            set a {Optimize for Area}
            set g {::quartus::dse::flows::simple_area_best_worst_analysis}
            set m {::quartus::dse::flows::accelerated_flow}
            set secret_sauce [list $s $a $g $m]
        }
        "speed:low" {
            set s {Seed Sweep}
            set a {Optimize for Speed}
            set g {::quartus::dse::flows::simple_slack_best_worst_analysis}
            set m {::quartus::dse::flows::exhaustive_flow}
            set secret_sauce [list $s $a $g $m]
        }
        "speed:medium" {
            if {$optshash(qii-synthesis)} {
                set s {Extra Effort Space for Quartus II Integrated Synthesis Projects}
            } else {
                set s {Extra Effort Space}
            }
            set a {Optimize for Speed}
            set g {::quartus::dse::flows::simple_slack_best_worst_analysis}
            set m {::quartus::dse::flows::accelerated_flow}
            set secret_sauce [list $s $a $g $m]
        }
        "speed:high" {
            if {$optshash(qii-synthesis)} {
                set s {Physical Synthesis Space for Quartus II Integrated Synthesis Projects}
            } else {
                set s {Physical Synthesis Space}
            }
            set a {Optimize for Speed}
            set g {::quartus::dse::flows::simple_slack_best_worst_analysis}
            set m {::quartus::dse::flows::accelerated_flow}
            set secret_sauce [list $s $a $g $m]
        }
        "speed:highest" {
            if {$optshash(qii-synthesis)} {
                set s {Physical Synthesis with Retiming Space for Quartus II Integrated Synthesis Projects}
            } else {
                set s {Physical Synthesis with Retiming Space}
            }
            set a {Optimize for Speed}
            set g {::quartus::dse::flows::simple_slack_best_worst_analysis}
            set m {::quartus::dse::flows::accelerated_flow}
            set secret_sauce [list $s $a $g $m]
        }
        "power" {
            set s {Power Optimization Space}
            set a {Optimize for Power}
            set g {::quartus::dse::flows::simple_power_best_worst_analysis}
            set m {::quartus::dse::flows::accelerated_flow}
            set secret_sauce [list $s $a $g $m]
        }
        default {
            ::quartus::dse::ccl::dputs "${debug_name}: There is no quick recipie for $_t in this library"
            set secret_sauce [list]
        }
    }
    return $secret_sauce
}


#############################################################################
##  Procedure:  set_design_space
##
##  Arguments:
##      dse_object
##          A pass-by-reference DSE object. This is the objec that will
##          have its space filled in with the appropriate points for
##          the requested space type. Nothing will be deleted in this
##          object. Points will only be added. A "base" point will
##          be added at (map,0) for you.
##
##      space_type
##          The name of the space you want created. Should match one
##          of the names you get from calling the get_valid_types()
##          function for this family.
##
##      custom_file
##          The name of a custom space file to load if the space is
##          a custom space. If you're not running a custom_space it
##          doesn't matter what you pass us for this value.
##
##      args
##          Additional key/value pairs you want to add. For every pair
##          you pass you'll get each point in the pre-determined space
##          with the pair. For example, if you pass:
##              {SEED 5}
##          As the $args list then you'll get all the points in the
##          space with SEED=5 as well. If you pass:
##              {SEED 5 SEED 10}
##          You'll get all the points with SEED=5 and then all the
##          points with SEED=10. Get it? All the points will show
##          up in the "fit" space. The "map" space is unavailable
##          to you from this interface because I'm expecting you'll
##          only ever use this to sweep SEED values.
##
##  Description:
##      Sets up all the values for the design space for us.
##
##      Returns true (1) if setup was successfull and your _dse_object
##      that you passed by reference has the space configured properly.
##      Returns false (0) if something went wrong.
proc ::quartus::dse::stratixii::set_design_space { designspace space_type custom_file args } {
    
    set debug_name "stratixii::set_design_space()"

    # Flatten arguments
    set args [join $args]

    # Make sure they're asking for a valid type
    if {![is_valid_type $space_type]} {
        ::quartus::dse::ccl::dputs "${debug_name}: Error: Type $space_type is not a valid space type"
        return 0
    }

    # Make sure $designspace is a dse object!
    if {![$designspace isa ::quartus::dse::designspace]} {
        ::quartus::dse::ccl::dputs "${debug_name}: Error: $designspace is not a ::quartus::dse::designspace object!"
        return 0
    }

    # Set the "base" point
    ::quartus::dse::ccl::dputs "${debug_name}: Adding base point at {map 0, fit 0}"
    $designspace addPoint map 0 {}
    $designspace addPoint fit 0 {}

    switch -exact $space_type {
        {Seed Sweep} {
            ::quartus::dse::ccl::dputs "${debug_name}: Creating Seed Sweep"

        }
        {Extra Effort Space} {
            ::quartus::dse::ccl::dputs "${debug_name}: Creating Extra Effort Space"

            # FIT: no register packing + increase in fitter effort level
            set options ""
            lappend options AUTO_PACKED_REGISTERS_STRATIXII OFF
            lappend options PLACEMENT_EFFORT_MULTIPLIER 3.0
            lappend options FITTER_EFFORT "STANDARD FIT"
            lappend options ROUTER_TIMING_OPTIMIZATION_LEVEL "MAXIMUM"
            lappend options ROUTER_LCELL_INSERTION_AND_LOGIC_DUPLICATION "ON"
            $designspace addPoint fit end $options
            unset options
            
            # FIT: register packing + increase in fitter effort level
            set options ""
            lappend options AUTO_PACKED_REGISTERS_STRATIXII NORMAL
            lappend options PLACEMENT_EFFORT_MULTIPLIER 3.0
            lappend options FITTER_EFFORT "STANDARD FIT"
            lappend options ROUTER_TIMING_OPTIMIZATION_LEVEL "MAXIMUM"
            lappend options ROUTER_LCELL_INSERTION_AND_LOGIC_DUPLICATION "ON"
            $designspace addPoint fit end $options
            unset options
            
            # FIT: no register packing
            set options ""
            lappend options AUTO_PACKED_REGISTERS_STRATIXII OFF
            lappend options FITTER_EFFORT "STANDARD FIT"
            lappend options ROUTER_TIMING_OPTIMIZATION_LEVEL "MAXIMUM"
            lappend options ROUTER_LCELL_INSERTION_AND_LOGIC_DUPLICATION "ON"
            $designspace addPoint fit end $options
            unset options
        }
        {Physical Synthesis Space} {
            ::quartus::dse::ccl::dputs "${debug_name}: Creating Physical Synthesis Space"

            # MAP: Advanced netlist optimizations
            set options ""
            lappend options ADV_NETLIST_OPT_SYNTH_GATE_RETIME OFF
            lappend options ADV_NETLIST_OPT_SYNTH_WYSIWYG_REMAP ON
            $designspace addPoint map end $options
            unset options

            # FIT: register packing + increase in fitter effort level + physical synthesis (no register moves)
            set options ""
            lappend options AUTO_PACKED_REGISTERS_STRATIXII NORMAL
            lappend options PHYSICAL_SYNTHESIS_COMBO_LOGIC ON
            lappend options PHYSICAL_SYNTHESIS_REGISTER_DUPLICATION OFF
            lappend options PHYSICAL_SYNTHESIS_REGISTER_RETIMING OFF
            lappend options PHYSICAL_SYNTHESIS_EFFORT EXTRA
            lappend options PLACEMENT_EFFORT_MULTIPLIER 3.0
            lappend options FITTER_EFFORT "STANDARD FIT"
            lappend options ROUTER_TIMING_OPTIMIZATION_LEVEL "MAXIMUM"
            lappend options ROUTER_LCELL_INSERTION_AND_LOGIC_DUPLICATION "ON"
            $designspace addPoint fit end $options
            unset options
        
            # FIT: register packing + increase in fitter effort level + physical synthesis (no register moves)
            set options ""
            lappend options AUTO_PACKED_REGISTERS_STRATIXII NORMAL
            lappend options PHYSICAL_SYNTHESIS_COMBO_LOGIC ON
            lappend options PHYSICAL_SYNTHESIS_REGISTER_DUPLICATION OFF
            lappend options PHYSICAL_SYNTHESIS_REGISTER_RETIMING OFF
            lappend options PHYSICAL_SYNTHESIS_EFFORT NORMAL
            lappend options PLACEMENT_EFFORT_MULTIPLIER 3.0
            lappend options FITTER_EFFORT "STANDARD FIT"
            lappend options ROUTER_TIMING_OPTIMIZATION_LEVEL "MAXIMUM"
            lappend options ROUTER_LCELL_INSERTION_AND_LOGIC_DUPLICATION "ON"
            $designspace addPoint fit end $options
            unset options
            
            # FIT: register packing + increase in fitter effort level + no physical synthesis (no register moves)
            set options ""
            lappend options AUTO_PACKED_REGISTERS_STRATIXII NORMAL
            lappend options PHYSICAL_SYNTHESIS_COMBO_LOGIC OFF
            lappend options PHYSICAL_SYNTHESIS_REGISTER_DUPLICATION OFF
            lappend options PHYSICAL_SYNTHESIS_REGISTER_RETIMING OFF
            lappend options PHYSICAL_SYNTHESIS_EFFORT NORMAL
            lappend options PLACEMENT_EFFORT_MULTIPLIER 3.0
            lappend options FITTER_EFFORT "STANDARD FIT"
            lappend options ROUTER_TIMING_OPTIMIZATION_LEVEL "MAXIMUM"
            lappend options ROUTER_LCELL_INSERTION_AND_LOGIC_DUPLICATION "ON"
            $designspace addPoint fit end $options
            unset options
                        
        }
        {Physical Synthesis with Retiming Space} {
            ::quartus::dse::ccl::dputs "${debug_name}: Creating Physical Synthesis with Retiming Space"

            # MAP: Advanced netlist optimizations
            set options ""
            lappend options ADV_NETLIST_OPT_SYNTH_GATE_RETIME ON
            lappend options ADV_NETLIST_OPT_SYNTH_WYSIWYG_REMAP ON
            $designspace addPoint map end $options
            unset options

            # FIT: register packing + increase in fitter effort level + physical synthesis (no register moves)
            set options ""
            lappend options AUTO_PACKED_REGISTERS_STRATIXII NORMAL
            lappend options PHYSICAL_SYNTHESIS_COMBO_LOGIC ON
            lappend options PHYSICAL_SYNTHESIS_REGISTER_DUPLICATION ON
            lappend options PHYSICAL_SYNTHESIS_REGISTER_RETIMING ON
            lappend options PHYSICAL_SYNTHESIS_EFFORT EXTRA
            lappend options PLACEMENT_EFFORT_MULTIPLIER 3.0
            lappend options FITTER_EFFORT "STANDARD FIT"
            lappend options ROUTER_TIMING_OPTIMIZATION_LEVEL "MAXIMUM"
            lappend options ROUTER_LCELL_INSERTION_AND_LOGIC_DUPLICATION "ON"
            $designspace addPoint fit end $options
            unset options
        
            # FIT: register packing + increase in fitter effort level + physical synthesis (no register moves)
            set options ""
            lappend options AUTO_PACKED_REGISTERS_STRATIXII NORMAL
            lappend options PHYSICAL_SYNTHESIS_COMBO_LOGIC ON
            lappend options PHYSICAL_SYNTHESIS_REGISTER_DUPLICATION ON
            lappend options PHYSICAL_SYNTHESIS_REGISTER_RETIMING ON
            lappend options PHYSICAL_SYNTHESIS_EFFORT NORMAL
            lappend options PLACEMENT_EFFORT_MULTIPLIER 3.0
            lappend options FITTER_EFFORT "STANDARD FIT"
            lappend options ROUTER_TIMING_OPTIMIZATION_LEVEL "MAXIMUM"
            lappend options ROUTER_LCELL_INSERTION_AND_LOGIC_DUPLICATION "ON"
            $designspace addPoint fit end $options
            unset options
            
            # FIT: register packing + increase in fitter effort level + no physical synthesis (no register moves)
            set options ""
            lappend options AUTO_PACKED_REGISTERS_STRATIXII NORMAL
            lappend options PHYSICAL_SYNTHESIS_COMBO_LOGIC OFF
            lappend options PHYSICAL_SYNTHESIS_REGISTER_DUPLICATION OFF
            lappend options PHYSICAL_SYNTHESIS_REGISTER_RETIMING OFF
            lappend options PHYSICAL_SYNTHESIS_EFFORT NORMAL
            lappend options PLACEMENT_EFFORT_MULTIPLIER 3.0
            lappend options FITTER_EFFORT "STANDARD FIT"
            lappend options ROUTER_TIMING_OPTIMIZATION_LEVEL "MAXIMUM"
            lappend options ROUTER_LCELL_INSERTION_AND_LOGIC_DUPLICATION "ON"
            $designspace addPoint fit end $options
            unset options            
        }
        {Extra Effort Space for Quartus II Integrated Synthesis Projects} {
            ::quartus::dse::ccl::dputs "${debug_name}: Creating Extra Effort Space for Quartus II Integrated Synthesis Projects"

            # MAP: Optimize for area
            set options ""
            lappend options STRATIXII_OPTIMIZATION_TECHNIQUE AREA
            lappend options ADV_NETLIST_OPT_SYNTH_GATE_RETIME OFF
            lappend options STATE_MACHINE_PROCESSING AUTO
            lappend options MUX_RESTRUCTURE ON
            $designspace addPoint map end $options
            unset options

            # MAP: Optimize for area
            set options ""
            lappend options STRATIXII_OPTIMIZATION_TECHNIQUE AREA
            lappend options ADV_NETLIST_OPT_SYNTH_GATE_RETIME OFF
            lappend options STATE_MACHINE_PROCESSING AUTO
            lappend options MUX_RESTRUCTURE OFF
            $designspace addPoint map end $options
            unset options

            # MAP: Optimize for speed
            set options ""
            lappend options STRATIXII_OPTIMIZATION_TECHNIQUE SPEED
            lappend options ADV_NETLIST_OPT_SYNTH_GATE_RETIME OFF
            lappend options STATE_MACHINE_PROCESSING AUTO
            lappend options MUX_RESTRUCTURE OFF
            $designspace addPoint map end $options
            unset options

            # MAP: Optimize for balanced
            set options ""
            lappend options STRATIXII_OPTIMIZATION_TECHNIQUE BALANCED
            lappend options ADV_NETLIST_OPT_SYNTH_GATE_RETIME OFF
            lappend options STATE_MACHINE_PROCESSING AUTO
            lappend options MUX_RESTRUCTURE OFF
            $designspace addPoint map end $options
            unset options

            # MAP: Optimize for speed
            set options ""
            lappend options STRATIXII_OPTIMIZATION_TECHNIQUE SPEED
            lappend options ADV_NETLIST_OPT_SYNTH_GATE_RETIME OFF
            lappend options STATE_MACHINE_PROCESSING "USER-ENCODED"
            lappend options MUX_RESTRUCTURE OFF
            $designspace addPoint map end $options
            unset options

            # FIT: no register packing + increase in fitter effort level
            set options ""
            lappend options AUTO_PACKED_REGISTERS_STRATIXII OFF
            lappend options PLACEMENT_EFFORT_MULTIPLIER 3.0
            lappend options FITTER_EFFORT "STANDARD FIT"
            lappend options ROUTER_TIMING_OPTIMIZATION_LEVEL "MAXIMUM"
            lappend options ROUTER_LCELL_INSERTION_AND_LOGIC_DUPLICATION "ON"
            $designspace addPoint fit end $options
            unset options
            
            # FIT: register packing + increase in fitter effort level
            set options ""
            lappend options AUTO_PACKED_REGISTERS_STRATIXII NORMAL
            lappend options PLACEMENT_EFFORT_MULTIPLIER 3.0
            lappend options FITTER_EFFORT "STANDARD FIT"
            lappend options ROUTER_TIMING_OPTIMIZATION_LEVEL "MAXIMUM"
            lappend options ROUTER_LCELL_INSERTION_AND_LOGIC_DUPLICATION "ON"
            $designspace addPoint fit end $options
            unset options
            
            # FIT: no register packing
            set options ""
            lappend options AUTO_PACKED_REGISTERS_STRATIXII OFF
            lappend options FITTER_EFFORT "STANDARD FIT"
            lappend options ROUTER_TIMING_OPTIMIZATION_LEVEL "MAXIMUM"
            lappend options ROUTER_LCELL_INSERTION_AND_LOGIC_DUPLICATION "ON"
            $designspace addPoint fit end $options
            unset options
        }
        {Physical Synthesis Space for Quartus II Integrated Synthesis Projects} {
            ::quartus::dse::ccl::dputs "${debug_name}: Creating Physical Synthesis Space for Quartus II Integrated Synthesis Projects"

            # MAP: Optimize for balanced
            set options ""
            lappend options STRATIXII_OPTIMIZATION_TECHNIQUE BALANCED
            lappend options ADV_NETLIST_OPT_SYNTH_GATE_RETIME OFF
            lappend options STATE_MACHINE_PROCESSING AUTO
            lappend options MUX_RESTRUCTURE OFF
            $designspace addPoint map end $options
            unset options

            # MAP: Optimize for speed
            set options ""
            lappend options STRATIXII_OPTIMIZATION_TECHNIQUE SPEED
            lappend options ADV_NETLIST_OPT_SYNTH_GATE_RETIME OFF
            lappend options STATE_MACHINE_PROCESSING "USER-ENCODED"
            lappend options MUX_RESTRUCTURE ON
            $designspace addPoint map end $options
            unset options
            
            # MAP: Optimize for speed
            set options ""
            lappend options STRATIXII_OPTIMIZATION_TECHNIQUE SPEED
            lappend options ADV_NETLIST_OPT_SYNTH_GATE_RETIME OFF
            lappend options STATE_MACHINE_PROCESSING AUTO
            lappend options MUX_RESTRUCTURE OFF
            $designspace addPoint map end $options
            unset options

            # MAP: Optimize for speed and disable removing duplicate logic
            set options ""
            lappend options STRATIXII_OPTIMIZATION_TECHNIQUE SPEED
            lappend options ADV_NETLIST_OPT_SYNTH_GATE_RETIME OFF
            lappend options STATE_MACHINE_PROCESSING "USER-ENCODED"
            lappend options MUX_RESTRUCTURE OFF
            lappend options REMOVE_DUPLICATE_LOGIC OFF
            $designspace addPoint map end $options
            unset options

            # MAP: Optimize for area
            set options ""
            lappend options STRATIXII_OPTIMIZATION_TECHNIQUE AREA
            lappend options ADV_NETLIST_OPT_SYNTH_GATE_RETIME OFF
            lappend options STATE_MACHINE_PROCESSING AUTO
            lappend options MUX_RESTRUCTURE ON
            $designspace addPoint map end $options
            unset options

            # FIT: register packing + increase in fitter effort level + physical synthesis (no register moves)
            set options ""
            lappend options AUTO_PACKED_REGISTERS_STRATIXII NORMAL
            lappend options PHYSICAL_SYNTHESIS_COMBO_LOGIC ON
            lappend options PHYSICAL_SYNTHESIS_REGISTER_DUPLICATION OFF
            lappend options PHYSICAL_SYNTHESIS_REGISTER_RETIMING OFF
            lappend options PHYSICAL_SYNTHESIS_EFFORT EXTRA
            lappend options PLACEMENT_EFFORT_MULTIPLIER 3.0
            lappend options FITTER_EFFORT "STANDARD FIT"
            lappend options ROUTER_TIMING_OPTIMIZATION_LEVEL "MAXIMUM"
            lappend options ROUTER_LCELL_INSERTION_AND_LOGIC_DUPLICATION "ON"
            $designspace addPoint fit end $options
            unset options
        
            # FIT: register packing + increase in fitter effort level + physical synthesis (no register moves)
            set options ""
            lappend options AUTO_PACKED_REGISTERS_STRATIXII NORMAL
            lappend options PHYSICAL_SYNTHESIS_COMBO_LOGIC ON
            lappend options PHYSICAL_SYNTHESIS_REGISTER_DUPLICATION OFF
            lappend options PHYSICAL_SYNTHESIS_REGISTER_RETIMING OFF
            lappend options PHYSICAL_SYNTHESIS_EFFORT NORMAL
            lappend options PLACEMENT_EFFORT_MULTIPLIER 3.0
            lappend options FITTER_EFFORT "STANDARD FIT"
            lappend options ROUTER_TIMING_OPTIMIZATION_LEVEL "MAXIMUM"
            lappend options ROUTER_LCELL_INSERTION_AND_LOGIC_DUPLICATION "ON"
            $designspace addPoint fit end $options
            unset options
            
            # FIT: register packing + increase in fitter effort level + no physical synthesis (no register moves)
            set options ""
            lappend options AUTO_PACKED_REGISTERS_STRATIXII NORMAL
            lappend options PHYSICAL_SYNTHESIS_COMBO_LOGIC OFF
            lappend options PHYSICAL_SYNTHESIS_REGISTER_DUPLICATION OFF
            lappend options PHYSICAL_SYNTHESIS_REGISTER_RETIMING OFF
            lappend options PHYSICAL_SYNTHESIS_EFFORT NORMAL
            lappend options PLACEMENT_EFFORT_MULTIPLIER 3.0
            lappend options FITTER_EFFORT "STANDARD FIT"
            lappend options ROUTER_TIMING_OPTIMIZATION_LEVEL "MAXIMUM"
            lappend options ROUTER_LCELL_INSERTION_AND_LOGIC_DUPLICATION "ON"
            $designspace addPoint fit end $options
            unset options
            
        }
        {Physical Synthesis with Retiming Space for Quartus II Integrated Synthesis Projects} {
            ::quartus::dse::ccl::dputs "${debug_name}: Creating Physical Synthesis with Retiming Space for Quartus II Integrated Synthesis Projects"

            # MAP: Optimize for balanced
            set options ""
            lappend options STRATIXII_OPTIMIZATION_TECHNIQUE BALANCED
            lappend options ADV_NETLIST_OPT_SYNTH_GATE_RETIME OFF
            lappend options STATE_MACHINE_PROCESSING AUTO
            lappend options MUX_RESTRUCTURE OFF
            $designspace addPoint map end $options
            unset options

            # MAP: Optimize for speed
            set options ""
            lappend options STRATIXII_OPTIMIZATION_TECHNIQUE SPEED
            lappend options ADV_NETLIST_OPT_SYNTH_GATE_RETIME OFF
            lappend options STATE_MACHINE_PROCESSING "USER-ENCODED"
            lappend options MUX_RESTRUCTURE ON
            $designspace addPoint map end $options
            unset options
            
            # MAP: Optimize for speed
            set options ""
            lappend options STRATIXII_OPTIMIZATION_TECHNIQUE SPEED
            lappend options ADV_NETLIST_OPT_SYNTH_GATE_RETIME OFF
            lappend options STATE_MACHINE_PROCESSING AUTO
            lappend options MUX_RESTRUCTURE OFF
            $designspace addPoint map end $options
            unset options

            # MAP: Optimize for speed and disable removing duplicate logic
            set options ""
            lappend options STRATIXII_OPTIMIZATION_TECHNIQUE SPEED
            lappend options ADV_NETLIST_OPT_SYNTH_GATE_RETIME OFF
            lappend options STATE_MACHINE_PROCESSING "USER-ENCODED"
            lappend options MUX_RESTRUCTURE OFF
            lappend options REMOVE_DUPLICATE_LOGIC OFF
            $designspace addPoint map end $options
            unset options

            # MAP: Optimize for area
            set options ""
            lappend options STRATIXII_OPTIMIZATION_TECHNIQUE AREA
            lappend options ADV_NETLIST_OPT_SYNTH_GATE_RETIME OFF
            lappend options STATE_MACHINE_PROCESSING AUTO
            lappend options MUX_RESTRUCTURE ON
            $designspace addPoint map end $options
            unset options

            # FIT: register packing + increase in fitter effort level + physical synthesis (no register moves)
            set options ""
            lappend options AUTO_PACKED_REGISTERS_STRATIXII NORMAL
            lappend options PHYSICAL_SYNTHESIS_COMBO_LOGIC ON
            lappend options PHYSICAL_SYNTHESIS_REGISTER_DUPLICATION ON
            lappend options PHYSICAL_SYNTHESIS_REGISTER_RETIMING ON
            lappend options PHYSICAL_SYNTHESIS_EFFORT EXTRA
            lappend options PLACEMENT_EFFORT_MULTIPLIER 3.0
            lappend options FITTER_EFFORT "STANDARD FIT"
            lappend options ROUTER_TIMING_OPTIMIZATION_LEVEL "MAXIMUM"
            lappend options ROUTER_LCELL_INSERTION_AND_LOGIC_DUPLICATION "ON"
            $designspace addPoint fit end $options
            unset options
        
            # FIT: register packing + increase in fitter effort level + physical synthesis (no register moves)
            set options ""
            lappend options AUTO_PACKED_REGISTERS_STRATIXII NORMAL
            lappend options PHYSICAL_SYNTHESIS_COMBO_LOGIC ON
            lappend options PHYSICAL_SYNTHESIS_REGISTER_DUPLICATION ON
            lappend options PHYSICAL_SYNTHESIS_REGISTER_RETIMING ON
            lappend options PHYSICAL_SYNTHESIS_EFFORT NORMAL
            lappend options PLACEMENT_EFFORT_MULTIPLIER 3.0
            lappend options FITTER_EFFORT "STANDARD FIT"
            lappend options ROUTER_TIMING_OPTIMIZATION_LEVEL "MAXIMUM"
            lappend options ROUTER_LCELL_INSERTION_AND_LOGIC_DUPLICATION "ON"
            $designspace addPoint fit end $options
            unset options
            
            # FIT: register packing + increase in fitter effort level + no physical synthesis (no register moves)
            set options ""
            lappend options AUTO_PACKED_REGISTERS_STRATIXII NORMAL
            lappend options PHYSICAL_SYNTHESIS_COMBO_LOGIC OFF
            lappend options PHYSICAL_SYNTHESIS_REGISTER_DUPLICATION OFF
            lappend options PHYSICAL_SYNTHESIS_REGISTER_RETIMING OFF
            lappend options PHYSICAL_SYNTHESIS_EFFORT NORMAL
            lappend options PLACEMENT_EFFORT_MULTIPLIER 3.0
            lappend options FITTER_EFFORT "STANDARD FIT"
            lappend options ROUTER_TIMING_OPTIMIZATION_LEVEL "MAXIMUM"
            lappend options ROUTER_LCELL_INSERTION_AND_LOGIC_DUPLICATION "ON"
            $designspace addPoint fit end $options
            unset options
            
        }
        {Area Optimization Space} {
            ::quartus::dse::ccl::dputs "${debug_name}: Creating Area Optimization Space"

            # MAP: Try wysiwyg remapping and mux restructuring with area mapping
            foreach {M} [list "OFF" "ON"] {
                foreach {N} [list "OFF" "ON"] {
                    set options ""
                    lappend options STRATIXII_OPTIMIZATION_TECHNIQUE AREA
                    lappend options MUX_RESTRUCTURE $M
                    lappend options ADV_NETLIST_OPT_SYNTH_WYSIWYG_REMAP $N
                    $designspace addPoint map end $options
                    unset options
                }
            }

            # FIT: regpacking minimize_area_with_chains
            set options ""
            lappend options AUTO_PACKED_REGISTERS_STRATIXII MINIMIZE_AREA_WITH_CHAINS
            $designspace addPoint fit end $options
            unset options
            
            # FIT: regpacking minimize area
            set options ""
            lappend options AUTO_PACKED_REGISTERS_STRATIXII MINIMIZE_AREA
            $designspace addPoint fit end $options
            unset options

            # FIT: regpacking normal
            set options ""
            lappend options AUTO_PACKED_REGISTERS_STRATIXII NORMAL
            $designspace addPoint fit end $options
            unset options
        }
        {Power Optimization Space} {
            ::quartus::dse::ccl::dputs "${debug_name}: Creating Power Optimization Space"

            foreach {mux} [list "OFF" "ON"] {
                foreach {synth} [list "Extra Effort" "Normal Compilation" "OFF"] {
                    foreach {tech} [list "Balanced" "Area"] {
                        set options ""
                        lappend options MUX_RESTRUCTURE $mux
                        lappend options OPTIMIZE_POWER_DURING_SYNTHESIS $synth
                        lappend options STRATIXII_OPTIMIZATION_TECHNIQUE $tech
                        $designspace addPoint map end $options
                        unset options
                    }
                }
            }

            foreach {power} [list "Extra Effort" "Normal Compilation" "OFF"] {
                set options ""
                lappend options OPTIMIZE_POWER_DURING_FITTING $power
                $designspace addPoint fit end $options
                unset options
            }

        }
        {Signature: Placement Effort Multiplier} {
            ::quartus::dse::ccl::dputs "${debug_name}: Creating Fitting Effort Level Signature"

            # "NORMAL" fitting effort
            $designspace addPoint fit end PLACEMENT_EFFORT_MULTIPLIER  1.0 FITTER_EFFORT {"STANDARD FIT"}

            # "MEDIUM" fitting effort
            $designspace addPoint fit end PLACEMENT_EFFORT_MULTIPLIER  3.0 FITTER_EFFORT {"STANDARD FIT"}

            # "HARD" fitting effort
            $designspace addPoint fit end PLACEMENT_EFFORT_MULTIPLIER  5.0 FITTER_EFFORT {"STANDARD FIT"}

            # "HARDER" fitting effort
            $designspace addPoint fit end PLACEMENT_EFFORT_MULTIPLIER  10.0 FITTER_EFFORT {"STANDARD FIT"}
        }
        {Signature: Register Packing} {
            ::quartus::dse::ccl::dputs "${debug_name}: Creating Register Packing Signature"

            # No register packing
            set options ""
            lappend options AUTO_PACKED_REGISTERS_STRATIXII OFF
            $designspace addPoint fit end $options
            unset options

            # Normal register packing
            set options ""
            lappend options AUTO_PACKED_REGISTERS_STRATIXII NORMAL
            $designspace addPoint fit end $options
            unset options

            # Area minimization register packing
            set options ""
            lappend options AUTO_PACKED_REGISTERS_STRATIXII MINIMIZE_AREA
            $designspace addPoint fit end $options
            unset options

            # Area minimization with chains register packing
            set options ""
            lappend options AUTO_PACKED_REGISTERS_STRATIXII MINIMIZE_AREA_WITH_CHAINS
            $designspace addPoint fit end $options
            unset options
        }
        {Signature: Fast Fit} {
            ::quartus::dse::ccl::dputs "${debug_name}: Creating Fast Fit Signature"

            # No fast fit
            $designspace addPoint fit end FAST_FIT_COMPILATION OFF FIT_ONLY_ONE_ATTEMPT OFF

            # Fast fit
            $designspace addPoint fit end FAST_FIT_COMPILATION ON FIT_ONLY_ONE_ATTEMPT ON
        }
        {Signature: Netlist Optimizations} {
            ::quartus::dse::ccl::dputs "${debug_name}: Creating Netlist Optimizations Signature"

            # OFF
            set options ""
            lappend options PHYSICAL_SYNTHESIS_COMBO_LOGIC OFF
            lappend options PHYSICAL_SYNTHESIS_REGISTER_DUPLICATION OFF
            lappend options PHYSICAL_SYNTHESIS_REGISTER_RETIMING OFF
            $designspace addPoint fit end $options
            unset options

            # ON
            set options ""
            lappend options PHYSICAL_SYNTHESIS_COMBO_LOGIC ON
            lappend options PHYSICAL_SYNTHESIS_REGISTER_DUPLICATION OFF
            lappend options PHYSICAL_SYNTHESIS_REGISTER_RETIMING OFF
            lappend options PHYSICAL_SYNTHESIS_EFFORT NORMAL
            $designspace addPoint fit end $options
            unset options

            # ON
            set options ""
            lappend options PHYSICAL_SYNTHESIS_COMBO_LOGIC ON
            lappend options PHYSICAL_SYNTHESIS_REGISTER_DUPLICATION OFF
            lappend options PHYSICAL_SYNTHESIS_REGISTER_RETIMING OFF
            lappend options PHYSICAL_SYNTHESIS_EFFORT EXTRA
            $designspace addPoint fit end $options
            unset options

            # ON w/Register Moves
            set options ""
            lappend options PHYSICAL_SYNTHESIS_COMBO_LOGIC ON
            lappend options PHYSICAL_SYNTHESIS_REGISTER_DUPLICATION ON
            lappend options PHYSICAL_SYNTHESIS_REGISTER_RETIMING ON
            lappend options PHYSICAL_SYNTHESIS_EFFORT NORMAL
            $designspace addPoint fit end $options
            unset options
        
            # ON w/Register Moves
            set options ""
            lappend options PHYSICAL_SYNTHESIS_COMBO_LOGIC ON
            lappend options PHYSICAL_SYNTHESIS_REGISTER_DUPLICATION ON
            lappend options PHYSICAL_SYNTHESIS_REGISTER_RETIMING ON
            lappend options PHYSICAL_SYNTHESIS_EFFORT EXTRA
            $designspace addPoint fit end $options
            unset options
        }
        {Custom Space} {
            ::quartus::dse::ccl::dputs "${debug_name}: Creating Custom Space"
        }
        default {
            # Huh? How'd we get here?
            ::quartus::dse::ccl::dputs "${debug_name}: Error: switch encountered an unknown type: $space_type"
            return 0
        }
    }

    # Return successfully
    return 1
}


