set pvcs_revision(qsta_helper) [regsub -nocase -- {\$revision:\s*(\S+)\s*\$} {$Revision:   22.0.1.8  $} {\1}]

# *************************************************************
#
# File: qsta_helper.tcl
#
# Usage: Internal Use. 
#
# Description: 
#        Used by STA to implement functionality that is 
#        easier done in Tcl
#
#
# *************************************************************

# ----------------------------------------------------------------
#
namespace eval qsta_utility {
#
# Description: Helper functions to implement TimeQuest functionality
#
# ----------------------------------------------------------------

}

# -------------------------------------------------
# -------------------------------------------------

proc qsta_utility::get_default_sdc_files {} {
	# 1.- Figure out the default SDC file or files
	#  Use the following priority
	#   - If QSF DEFAULT_SDC_FILE exist, return that
	#   - If QSF SDC_FILEs exist, return them
	#   - If <revision>.sdc exist, return that
	#
	# Return a list of SDC files
# -------------------------------------------------
# -------------------------------------------------

	set sdc_file_list ""
	set no_sdc_file 1

	# 1.- Check DEFAULT_SDC_FILE assignment
	set default_sdc_file [get_global_assignment -name DEFAULT_SDC_FILE]
	if [string compare $default_sdc_file ""] {
		lappend sdc_file_list $default_sdc_file
		set no_sdc_file 0
	}

	# 2.- Check any SDC_FILE assignments
	if { $no_sdc_file } {
		set file_asgn_col [get_all_global_assignments -name SDC_FILE]

		# Add each file to the list
		foreach_in_collection file_asgn $file_asgn_col {
			# Get all assignments returns an array of subarrays
			# The subarrays format in this case is:
			# {} {SDC_FILE} {<file_name>}
			set sdc_file [lindex $file_asgn 2]
			lappend sdc_file_list $sdc_file
			set no_sdc_file 0
		}
	}

	# 3.- Default to <rev>.sdc
	if { $no_sdc_file } {
		set revision [get_current_revision]
		msg_vdebug "Generating default SDC for Rev = $revision"
		lappend sdc_file_list "${revision}.sdc"
	}

	foreach file $sdc_file_list {
		msg_vdebug "SDC File: $file"
	}

	return $sdc_file_list
}

# -------------------------------------------------
# -------------------------------------------------

proc qsta_utility::read_default_sdc { migrate_if_needed } {
	# 1.- Figure out which SDC file to load
	#   - If it doesn't exist, try to create one based
	#     on QSF file and TAN report.
	# 2.- Load file
	#
	# If migrate_if_needed is 1, call QSF2SDC if no
	#   SDC is found
	#
	# Return Nothing
# -------------------------------------------------
# -------------------------------------------------

	set sdc_file_names [get_default_sdc_file_names]
	set no_sdc_file 1

	foreach file $sdc_file_names {
		msg_vdebug "SDC File: $file"
		if [file exist $file] {
			post_message -type info "Reading SDC File: $file"
			read_sdc $file
			set no_sdc_file 0
		} else {
			post_message -type critical_warning "SDC file not found: $file"
		}
	}

	if {$migrate_if_needed && $no_sdc_file} {
		set file  [lindex $sdc_file_names 0]

		post_message -type info "Attempting to generate SDC based on QSF and Timing Analysis Report"
		load_package flow
		if [catch {execute_module -tool sta -args "--qsf2sdc --disable_all_banners"} result] {
			# qsf2sdc does not have any non-IE error conditions
			msg_vdebug "$result"
		}

		if [file exist $file] {
			post_message -type info "Reading SDC File: $file"
			read_sdc $file
		} else {
			post_message -type error "Unable to source SDC file: $file"
		}
	}
}

# -------------------------------------------------
# -------------------------------------------------

proc qsta_utility::generate_tan_tables { } {
	# For each clock, generate TAN like tables:
	#     Clock Setup 
	#     Clock Hold
# -------------------------------------------------
# -------------------------------------------------

	set setup_domain_list [get_clock_domain_info -setup]

	# Report the Worst Case setup slack per clock
	foreach domain $setup_domain_list {
		set clk_name [lindex $domain 0]
		# Account for NOT_A_CLOCK clock domain
		if [string compare $clk_name ""] {
			report_timing -setup -panel_name "Clock Setup: $clk_name" -npaths 200 -summary -to_clock $clk_name
		}
		foreach domain $setup_domain_list {
			set clk_name [lindex $domain 0]
			report_timing -hold -panel_name "Clock Hold: $clk_name" -npaths 200 -summary -to_clock $clk_name
		}
	}
}

# -------------------------------------------------
# -------------------------------------------------

proc qsta_utility::generate_top_failures_per_clock { number_of_rows } {
	# For each clock, generate a table if the slack is negative.
	# Generate table with -napths <number_of_rows>
	# Check
	#     - Setup 
	#     - Hold
	#     - Recovery 
	#     - Removal
# -------------------------------------------------
# -------------------------------------------------

	set nothing_done 1

	# 1.- Report the Worst Case setup slack per clock
	set setup_domain_list [get_clock_domain_info -setup]
	foreach domain $setup_domain_list {
		set clk_name [lindex $domain 0]
		# Account for NOT_A_CLOCK clock domain
		if [string compare $clk_name ""] {
			set slack [lindex $domain 1]
			if { double($slack) < double(0.0) } {
				report_timing -setup -panel_name "Setup: $clk_name" -less_than_slack 0.0 -npaths 200 -summary -to_clock $clk_name
				set nothing_done 0
			}
		}
	}

	# 2.- Report the Worst Case hold slack per clock
	set hold_domain_list [get_clock_domain_info -hold]
	foreach domain $hold_domain_list {
		set clk_name [lindex $domain 0]
		# Account for NOT_A_CLOCK clock domain
		if [string compare $clk_name ""] {
			set slack [lindex $domain 1]
			if { double($slack) < double(0.0) } {
				report_timing -hold -panel_name "Hold: $clk_name" -less_than_slack 0.0 -npaths 200 -summary -to_clock $clk_name
				set nothing_done 0
			}
		}
	}

	# 3.- Report the Worst Case recovery slack per clock
	set recovery_domain_list [get_clock_domain_info -recovery]
	foreach domain $recovery_domain_list {
		set clk_name [lindex $domain 0]
		# Account for NOT_A_CLOCK clock domain
		if [string compare $clk_name ""] {
			set slack [lindex $domain 1]
			if { double($slack) < double(0.0) } {
				report_timing -recovery -panel_name "Recovery: $clk_name" -less_than_slack 0.0 -npaths 200 -summary -to_clock $clk_name
				set nothing_done 0
			}
		}
	}

	# 4.- Report the Worst Case removal slack per clock
	set removal_domain_list [get_clock_domain_info -removal]
	foreach domain $removal_domain_list {
		set clk_name [lindex $domain 0]
		# Account for NOT_A_CLOCK clock domain
		if [string compare $clk_name ""] {
			set slack [lindex $domain 1]
			if { double($slack) < double(0.0) } {
				report_timing -removal -panel_name "Removal: $clk_name" -less_than_slack 0.0 -npaths 200 -summary -to_clock $clk_name
				set nothing_done 0
			}
		}
	}

	if { $nothing_done } {
		post_message -type info "No failing paths found"
	}

}

# -------------------------------------------------
# -------------------------------------------------

proc qsta_utility::generate_all_histograms { } {
	# For each clock, call
	#     create_histogram
# -------------------------------------------------
# -------------------------------------------------

	set setup_domain_list [get_clock_domain_info -setup]

	# Report the Worst Case setup slack per clock
	foreach domain $setup_domain_list {
		set clk_name [lindex $domain 0]
		# Account for NOT_A_CLOCK clock domain
		if [string compare $clk_name ""] {
			create_slack_histogram -clock_name $clk_name -num_bins 30
		}
	}
}

# -------------------------------------------------
# -------------------------------------------------

proc qsta_utility::generate_all_summary_tables { } {
	# Generate all key summary reports
	#
# -------------------------------------------------
# -------------------------------------------------

	create_timing_summary -setup -panel_name "Summary (Setup)"
	create_timing_summary -hold -panel_name "Summary (Hold)"
	create_timing_summary -recovery -panel_name "Summary (Recovery)"
	create_timing_summary -removal -panel_name "Summary (Removal)"

	report_min_pulse_width -nworst 100 -panel_name "Minimum Pulse Width"

	report_clocks -panel_name "Clocks Summary"
}

# -------------------------------------------------
# -------------------------------------------------

proc qsta_utility::generate_clock_groups {} {

# Generate a <rev>.clk_grps.sdc file
# with a set_clock_groups command emulating
# TAN's behavior
# -------------------------------------------------
# -------------------------------------------------

	if {![timing_netlist_exist]} {
		post_message -type error "Netlist does not exist"
		return
	}

	set file_name "$::quartus(settings).clk_grps.sdc"
	if [catch {set outfile [open $file_name  w]} result] {
		post_message -info error "File $file_name could not be opened"
	} else {

		puts $outfile "###########################################################################"
		puts $outfile "#"
		puts $outfile "# Generated by  : [info script]"
		puts $outfile "# Using Quartus : $::quartus(version)"
		puts $outfile "#"
		puts $outfile "# Project       : $::quartus(project)"
		puts $outfile "# Revision      : $::quartus(settings)"
		puts $outfile "#"
		puts $outfile "# Date          : [clock format [clock seconds]]"
		puts $outfile "#"
		puts $outfile "###########################################################################"
		puts $outfile ""
		puts $outfile ""		

		# 1.- Get all master clocks
		foreach_in_collection clk [get_clocks *] {

			set type [get_clock_info -type $clk]
			set name [get_clock_info -name $clk]

			if {![catch {set master_clock [get_clock_info -master_clock $clk]}]} {
				if [string equal $master_clock ""] {
					# Filter out N/C domains, which represent Tco paths
					if [string compare $name "N/C"] {
						msg_vdebug "Master = $name"
						set clock_group($name) $name
					}
				}
			}
		}

		# 1.- Get all Generated clocks
		foreach_in_collection clk [get_clocks *] {

			set type [get_clock_info -type $clk]
			set name [get_clock_info -name $clk]

			if {![catch {set master_clock [get_clock_info -master_clock $clk]}]} {
				if [string compare $master_clock ""] {
					lappend clock_group($master_clock) $name
				}
			}
		}

		# 3.- Generate set_clock_group command

		puts $outfile "set_clock_groups -exclusive \\"

		set count 0
		foreach key [array names clock_group] {
			set master_clock $key
			incr count
			post_message -type extra_info "Group $count"
			foreach clock $clock_group($key) {
				post_message -type extra_info "   Clock: $clock"
			}

			puts $outfile " -group { $clock_group($key) } \\"
		}

		post_message -type info "Generated $file_name"

		post_message -type info "Use read_sdc $file_name to load"

		close $outfile
	}
}
