
set pvcs_revision(experiment) [regsub -nocase -- {\$revision:\s*(\S+)\s*\$} {$Revision:   22.0.1.2  $} {\1}]

# *************************************************************
#
# File: qsta_experiment.tcl
#
# Summary: Utilities used by quartus_sta <rev> --experiment flow
#
# Description:
#		These functions should be used by anybody running Quartus experiments as a
#       means to produce a consistent report from quartus_sta. This script is
#       the STA equivalent of the qtan_experiment.tcl script for TAN. The
#       results are printed to a file called {project}.experiment.sta
#
#       The intended use of this script is:
#         1) Run flow (i.e. map->fit->asm)
#         2) Run "quartus_sta --experiment_report to report the results of the
#            project.
#
# *************************************************************

# ----------------------------------------------------------------
#
namespace eval qsta_experiment {
#
# Description: Helper functions to implement TimeQuest functionality
#
# ----------------------------------------------------------------

}

# -------------------------------------------------
# -------------------------------------------------

proc qsta_experiment::write_entry { outfile msg } {

	# Write msg to outfile. If verbose, print msg
# -------------------------------------------------
# -------------------------------------------------

	puts $outfile "$msg"
	if { $::options(verbose) } {
		post_message -type info "$msg"
	}
}

# -------------------------------------------------
# -------------------------------------------------

proc qsta_experiment::create_experiment_summary_file { outfile } {

	# Function creates a .experiment.sta file
	# with key benchmark metrics like
	#       * "WC_MAX_CLOCK_SETUP_SLACK : <clock name> : <num>" - Worst case
	#         Max (Slow Model) Setup Slack is taken directly form the
	#         "Timing Analysis Summary" and is always accurate.
	#
	#       * "WC_MAX_CLOCK_HOLD_SLACK : <clock name> : <num>" - Worst case
	#         Max (Slow Model) Hold Slack is taken directly form the
	#         "Timing Analysis Summary" and is always accurate.
	#
	#       * "PER_CLK_KEEPER_TNS : <clock_name> : <num>" - The total negative slack of the
	#         given clock domain based on keepers.
	#
	#       * "PER_CLK_EDGE_TNS : <clock_name> : <num>" - The total negative slack of the
	#         given clock domain based on edges.
# -------------------------------------------------
# -------------------------------------------------



	# Start by iterating through the clock domain info list to get the slack,
	# keeper TNS and edge TNS per clock domain.

	set setup_domain_list [get_clock_domain_info -setup]
	set hold_domain_list [get_clock_domain_info -hold]

	# Report the Worst Case setup slack per clock
	foreach domain $setup_domain_list {
	    # Format is { <clk_name> <slack> <keeper tns> <edge tns> }
	    set clk_name [lindex $domain 0]
		set setup_slack [lindex $domain 1]
		set keeper_tns [lindex $domain 2]
		set edge_tns [lindex $domain 3]

		qsta_experiment::write_entry $outfile "WC_MAX_CLOCK_SETUP_SLACK : $clk_name : $setup_slack"
		qsta_experiment::write_entry $outfile "PER_CLK_KEEPER_TNS : $clk_name : $keeper_tns"
		qsta_experiment::write_entry $outfile "PER_CLK_EDGE_TNS : $clk_name : $edge_tns"
	}


	# Now do the same thing for hold
	foreach domain $hold_domain_list {
	    # Format is { <clk_name> <slack> <keeper tns> <edge tns> }
	    set clk_name [lindex $domain 0]
		set hold_slack [lindex $domain 1]

		qsta_experiment::write_entry $outfile "WC_MAX_CLOCK_HOLD_SLACK : $clk_name : $hold_slack"
	}

}

# -------------------------------------------------
# -------------------------------------------------
proc qsta_experiment::create_custom_tao_file {} {

	# Generates a custom .sta.tao file with more
	# detail information about the different clock
	# domains and I/O paths in the design
# -------------------------------------------------
# -------------------------------------------------


	# Write some standard panels
	create_timing_summary -panel_name "Summary (Setup)"
	create_timing_summary -hold -panel_name "Summary (Hold)"
	report_clocks -panel_name "Clocks Summary"
	report_clock_transfers -panel_name "Clock Transfers"

	# Write some global summaries
	report_timing -setup -from [all_registers] -to [all_outputs] -npaths 10 -from_clock * -to_clock * -panel_name "WC Clock Output Delay"
	report_timing -setup -from [all_inputs] -to [all_registers] -npaths 10 -from_clock * -to_clock * -panel_name "WC Clock Input Delay"
	report_timing -setup -from [all_inputs] -to [all_outputs] -npaths 10 -panel_name "WC TPD Delay"
	report_timing -setup -from [all_registers] -to [all_registers] -npaths 10 -from_clock * -to_clock * -panel_name "WC Reg-Reg Delay"
	report_timing -setup -panel_name "WC Delay"

	# Now loop for each clock
	#foreach_in_collection clk_name [get_clocks *]   <= Why doesn't this work?
	set setup_domain_list [get_clock_domain_info -setup]
	foreach domain $setup_domain_list {
	    # Format is { <clk_name> <slack> <keeper tns> <edge tns> }
	    set clk_name [lindex $domain 0]

		# We have the concept of a NOT_A_CLOCK which is used for cases 
		# where we have a TCO type constraint using mas_delay to an oport
		# Skip those
		if [string compare $clk_name ""] {
			report_timing -setup -from [all_registers] -to [all_registers] -npaths 10 -from_clock $clk_name -to_clock $clk_name -panel_name "Clock Setup: $clk_name"
			report_timing -hold -from [all_registers] -to [all_registers] -npaths 10 -from_clock $clk_name -to_clock $clk_name -panel_name "Clock Hold: $clk_name"

			report_timing -setup -from [all_inputs] -to [all_registers] -npaths 10 -from_clock $clk_name -to_clock $clk_name -panel_name "Clock Input Delay: $clk_name"
			report_timing -setup -from [all_registers] -to [all_inputs] -npaths 10 -from_clock $clk_name -to_clock $clk_name -panel_name "Clock Output Delay: $clk_name"
		}
	}

}
