set pvcs_revision(main) [regsub -nocase -- {\$revision:\s*(\S+)\s*\$} {$Revision:   22.0.1.6  $} {\1}]

# *************************************************************
#
# File: qsta_default_script.tcl
#
# Usage: quartus_sta --default [options] <project>
#
#		where [options] are described below. Search for available_options
#			
#
# Description: 
#		This script represents the default run of the new Static Timin
#
#
# *************************************************************

set builtin_dir [file dirname [info script]]

# ---------------------------------------------------------------
# Available User Options for:
#    quartus_sta --default [options] <project>
# ---------------------------------------------------------------

set available_options {
	{ rev.arg "#_ignore_#" "Revision name (QSF name)" }
	{ sdc.arg "#_ignore_#" "SDC File to load" }
	{ report_script.arg "\#_ignore_\#" "Custom Script" }
	{ read_xml.arg "\#_ignore_\#" "Read XML file instead of creating a netlist" }
	{ write_xml.arg "\#_ignore_\#" "Write XML file with netlist content" }
	{ qsf2sdc "Migrate QSF assignments. Generate SDC file" }
	{ risefall "Enable Rise/Fall Delays" }
	{ do_report_timing "Show the worst case path per clock (using report_timing)" }
	{ force_dat "Force Delay Annotation step" }
	{ post_map "Use the quartus_map output netlist" }
	{ post_asm "Use the quartus_asm output netlist (HCII only)" }
	{ verbose "Give additional information" }
	{ experiment_report "Run script to extract experiment friendly results" }
	{ file.arg "#_ignore_#" "Output File Name" }
}

# --------------------------------------
# Other Global variables
# --------------------------------------

# ------------------------------
# Load Required Quartus Packages
# ------------------------------
load_package sta_internal
load_package report
package require cmdline


# -------------------------------------------------
# -------------------------------------------------

proc create_or_read_netlist {} {
	# 
	# Based on options, either read an XML
	# representation of the Netlist or create
	# a new netlist
	# -------------------------------------------------
	# -------------------------------------------------

	global quartus
	global options

	if {$options(read_xml) != "#_ignore_#"} {
		post_message -type info "Reading: $options(read_xml)"

		if [catch {eval read_timing_netlist -no_report -xml $options(read_xml)} result] {
			post_message -type error "read_timing_netlist failed. Make sure $options(read_xml) exists"
			msg_vdebug $result
			qexit -error
		}
		
	} else {

		set create_netlist_args "-no_report"

		if {$options(risefall)} {
			post_message -type info "Using Rise/Fall Delays"
			append create_netlist_args " -risefall"
		}

		if {$options(force_dat)} {
			post_message -type info "Forcing Delay Annotation"
			append create_netlist_args " -force_dat"
		}

		if {$options(post_map)} {
			post_message -type info "Using post quartus_map netlist"
			append create_netlist_args " -post_map"
		}

		if {$options(post_asm)} {
			post_message -type info "Using post quartus_asm netlist"
			append create_netlist_args " -post_asm"
		}

		if [catch {eval create_timing_netlist $create_netlist_args} result] {

			# Remove "ERROR:" from message
			set error_message [string range $result 7 [string length $result]]
			# Remove new-line character
			set error_message [string trimright $error_message "\n"]

			post_message -type error $error_message
			qexit -error
		}
	}

	# If needed, write out an XML Netlist
	if {$options(write_xml) != "#_ignore_#"} {
		post_message -type info "Writing: $options(write_xml)"

		if [catch {eval write_timing_netlist -force -compress -xml $options(write_xml)} result] {
			post_message -type error "write_timing_netlist failed."
			msg_vdebug $result
			qexit -error
		}
	}			
}

# -------------------------------------------------
# -------------------------------------------------

proc generate_experiment_reports {} {
	# 
	# Call functions to generate a custom
	# experiment report(s)
# -------------------------------------------------
# -------------------------------------------------

	global quartus
	global options

	post_message -type info "Running Experiment Script"

	if [catch {set outfile [open $options(file)  w]} result] {
		post_message -info error "File $option(file) could not be opened"
		qexit -error
	}

	# Perform the reporting and print them to .experiment.sta
	qsta_experiment::create_experiment_summary_file $outfile
	
	# Perform the reporting desired and print it to the sta report
	# Need to create a new report first
	open_timing_report -new
	# Now add the report panels
	qsta_experiment::create_custom_tao_file			
	# Write the timing report
	write_timing_report -ascii

}

# -------------------------------------------------
# -------------------------------------------------

proc generate_summary_header { outfile } {
	# 
	# Add Header to <rev>.sta.summary file
# -------------------------------------------------
# -------------------------------------------------

	puts $outfile "------------------------------------------------------------"
	puts $outfile "TimeQuest Timing Analyzer Summary"
	puts $outfile "------------------------------------------------------------"
	puts $outfile ""
}

# -------------------------------------------------
# -------------------------------------------------

proc add_entry_to_summary { outfile type domain } {
	# 
	# Add domain info to Summary File
# -------------------------------------------------
# -------------------------------------------------

	set clock_name [lindex $domain 0]
	set slack [lindex $domain 1]
	set tns [lindex $domain 2]

	puts $outfile "Type  : $type '${clock_name}'"
	puts $outfile "Slack : $slack"
	puts $outfile "TNS   : $tns"
	puts $outfile ""
}

# -------------------------------------------------
# -------------------------------------------------

proc generate_compiler_reports {} {
	# 
	# Call functions to generate basic
	# reports during a QuartusII compilation:
	#   * A .sta.rpt with:
	#     - Summary (Setup)
	#     - Summary (Hold)
	#     - Messages
	#   * A .sta.summary with Setup information
# -------------------------------------------------
# -------------------------------------------------

	global quartus
	global options

	set summary_file "$options(rev).sta.summary"

	if [file exist $summary_file] {
		msg_vdebug "QSTA_DEFAULT_SCRIPT: About to delete $summary_file"
		file delete -force $summary_file
	}

	# Reports all clocks
	report_clocks

	# Create Summary
	set setup_domain_list [get_clock_domain_info -setup]

	if { $options(do_report_timing) } {
		# Report the Worst Case setup slack per clock
		foreach domain $setup_domain_list {
			set clk_name [lindex $domain 0]
			# Account for the NOT_A_CLOCK domain
			if [string compare $clk_name ""] {
				report_timing -setup -to_clock $clk_name
			}
		}
	}

	set hold_domain_list [get_clock_domain_info -hold]
	set recovery_domain_list [get_clock_domain_info -recovery]
	set removal_domain_list [get_clock_domain_info -removal]

	# Now lets create a Clock Summary in the Compiler report
	load_report

	if [catch {set summary_fileid [open $summary_file  w]} result] {
		post_message -info error "File $summary_file could not be opened"
		return
	}

	generate_summary_header $summary_fileid

	# Find TimeQuest Timing Analyzer Summary report
	# which is created by create_timing_netlist but
	# is empty by this point
	set panel {*TimeQuest*Summary*Setup*}
	set id [get_report_panel_id $panel]

	if {$id != -1} {
		msg_vdebug "QSTA_DEFAULT_SCRIPT: About to add rows to summary panel"
		# If panel exists, add a row per domain
		set grey_row 0
		foreach domain $setup_domain_list {
			set slack [lindex $domain 1]
			set domain [lrange $domain 0 2]
			set row_fcolor { blue blue blue }
			if { $slack < 0 } {
				set row_fcolor { red red red }
			}
			
			add_row_to_table -fcolors $row_fcolor -id $id $domain

			# Add entry to Summary file
			add_entry_to_summary $summary_fileid Setup $domain
		}
		# Save the changes to the report database
		save_report_database

	} else {
		# Otherwise print an error message
		post_message -type error "Table $panel does not exist."
	}

	set panel {*TimeQuest*Summary*Hold*}
	set id [get_report_panel_id $panel]

	if {$id != -1} {
		msg_vdebug "QSTA_DEFAULT_SCRIPT: About to add rows to summary panel"
		# If panel exists, add a row per domain
		set grey_row 0
		foreach domain $hold_domain_list {
			set slack [lindex $domain 1]
			set domain [lrange $domain 0 2]
			set row_fcolor { blue blue blue }
			if { $slack < 0 } {
				set row_fcolor { red red red }
			}
			
			add_row_to_table -fcolors $row_fcolor -id $id $domain

			# Add entry to Summary file
			add_entry_to_summary $summary_fileid Hold $domain
		}
		# Save the changes to the report database
		save_report_database

	} else {
		# Otherwise print an error message
		post_message -type error "Table $panel does not exist."
	}

	set panel {*TimeQuest*Summary*Recovery*}
	set id [get_report_panel_id $panel]

	if {$id != -1} {
		msg_vdebug "QSTA_DEFAULT_SCRIPT: About to add rows to summary panel"
		# If panel exists, add a row per domain
		set grey_row 0
		foreach domain $recovery_domain_list {
			set slack [lindex $domain 1]
			set domain [lrange $domain 0 2]
			set row_fcolor { blue blue blue }
			if { $slack < 0 } {
				set row_fcolor { red red red }
			}
			
			add_row_to_table -fcolors $row_fcolor -id $id $domain

			# Add entry to Summary file
			add_entry_to_summary $summary_fileid Recovery $domain
		}
		# Save the changes to the report database
		save_report_database

	} else {
		# Otherwise print an error message
		post_message -type error "Table $panel does not exist."
	}

	set panel {*TimeQuest*Summary*Removal*}
	set id [get_report_panel_id $panel]

	if {$id != -1} {
		msg_vdebug "QSTA_DEFAULT_SCRIPT: About to add rows to summary panel"
		# If panel exists, add a row per domain
		set grey_row 0
		foreach domain $removal_domain_list {
			set slack [lindex $domain 1]
			set domain [lrange $domain 0 2]
			set row_fcolor { blue blue blue }
			if { $slack < 0 } {
				set row_fcolor { red red red }
			}
			
			add_row_to_table -fcolors $row_fcolor -id $id $domain

			# Add entry to Summary file
			add_entry_to_summary $summary_fileid Removal $domain
		}
		# Save the changes to the report database
		save_report_database

	} else {
		# Otherwise print an error message
		post_message -type error "Table $panel does not exist."
	}

	puts $summary_fileid "------------------------------------------------------------"
}

# -------------------------------------------------
# -------------------------------------------------

proc main {} {
	# Script starts here
	# 1.- Process command-line arguments
	# 2.- Open project
	# 7.- Close Project
# -------------------------------------------------
# -------------------------------------------------

	global quartus
	global options



	# ---------------------------------
	# Print some useful infomation
	# ---------------------------------
	post_message -type info "----------------------------------------------------------"
	post_message -type info "----------------------------------------------------------"
	post_message -type info "[file tail [info script]] version: $::pvcs_revision(main)"
	post_message -type info "----------------------------------------------------------"
	post_message -type info "----------------------------------------------------------"

	# Check arguments
	# Need to define argv for the cmdline package to work
	set argv0 "quartus_sta -t [info script]"
	set usage "\[<options>\] <project_name>:"

	set argument_list $quartus(args)

	# Use cmdline package to parse options
	if [catch {array set options [cmdline::getoptions argument_list $::available_options]} result] {
		if {[llength $argument_list] > 0 } {
			# This is not a simple -? or -help but an actual error condition
			post_message -type error "Illegal Options"
			post_message -type error  [::cmdline::usage $::available_options $usage]
			qexit -error
		} else {
			post_message -type info  "Usage:"
			post_message -type info  [::cmdline::usage $::available_options $usage]
			qexit -success
		}
	}

	# If project was opened internally, then don't close project
	set project_already_opened [is_project_open]

	# cmdline::getoptions is going to modify the argument_list.
	# Note however that the function will ignore any positional arguments
	# We are only expecting one and only one positional argument (the project)
	# so give an error if the list has more than one element
	if {[llength $argument_list] == 1 } {

		# The first argument MUST be the project name
		set options(project_name) [lindex $argument_list 0]

		if [string compare [file extension $options(project_name)] ""] {
			set project_name [file rootname $options(project_name)]
		}

		set project_name [file normalize $options(project_name)]
	
		msg_vdebug  "Project = $project_name"

	} elseif {$project_already_opened == 1} {

		set project_name $::quartus(project)
		set options(rev) $::quartus(settings)

		msg_vdebug  "Project = $project_name (opened internally)"

	} else {

		post_message -type error "Project name is missing"
		post_message -type info [::cmdline::usage $::available_options $usage]
		post_message -type info "For more details, use \"quartus_sta --help=experiment_mode\""
		qexit -error
	}

	# Script may be called from Quartus or another script where the project
	# is already open
	if ![is_project_open] {

		# Create new project if needed and open
		if { ![project_exists $project_name] } {
			post_message -type error "Project $project_name does not exist"

		} else {

			# Get the revision name first if the user didn't give us one
			if {$options(rev) == "#_ignore_#"} {
				msg_vdebug "Opening Project: $project_name (Current Revision)" 
				if [catch {project_open $project_name -current_revision}] {
					post_message -type error "Project $options(project_name) (Current Revision) cannot be opened" 
					qexit -error
				}
				set options(rev) $::quartus(settings)
			} else {
				msg_vdebug "Opening Project: $project_name (Rev = $options(rev))" 
				if [catch {project_open $project_name -revision $options(rev)}] {
					post_message -type error "Project $options(project_name) ($options(rev).qsf) cannot be opened" 
					qexit -error
				}
			}
		}
	}

	# Check output file name
	if {$options(file) == "#_ignore_#"} {
		set options(file) "$options(rev).experiment.sta"
	}

	if [is_project_open] {

		post_message -type info "----------------------------------------------------------"

		if {$options(qsf2sdc)} {
			post_message -type info "Migrating assignments from quartus_tan"

			tan2sdc::convert $options(rev).sdc

			if {$project_already_opened == 0} {
				project_close
			}

			return
		}

		# Here is where the timing netlist is created
		# The function will either call:
		#   - create_timing_netlist <args>
		#   - read_timing_netlist <args>
		create_or_read_netlist

		# If an SDC is found, load it
		if {$options(sdc) == "#_ignore_#"} {
			# Call qsta_helper.tcl's read_default_sdc
			# 1 tells function to migrate if needed
			qsta_utility::read_default_sdc 1
		} else {

			if [file exist $options(sdc)] {
				post_message -type info "Reading SDC File: $options(sdc)"
				read_sdc $options(sdc)
			}
		}

		update_timing_netlist

		if {$options(experiment_report)} {

			# Internal Use Only
			generate_experiment_reports

			delete_timing_netlist

			if {$project_already_opened == 0} {
				project_close
			}

			return
		}

		# Here is where the .sta.rpt and .sta.summary
		# files are generated
		generate_compiler_reports

		if {$options(report_script) != "#_ignore_#"} {
			post_message -type info "Using custom script: $options(report_script)"

			source $options(report_script)
		}

		delete_timing_netlist

		if {$project_already_opened == 0} {
			project_close
		}
	}
}

# -------------------------------------------------
# -------------------------------------------------
main
# -------------------------------------------------
# -------------------------------------------------

