###################################################################################
#                                                                                 #
# File Name:    oaw_list_input_regs.tcl                                           #
#                                                                                 #
# Summary:      This script lists the input registers.                            #
# 																				  #
# Version:		Quartus II 5.0                                                    #
# 																				  #
# Note: 		This script is run from the Quartus Optimization Advisor.	      #
# 				This script is evaluated by quartus_tan executable.               #
# 				This script is passed in <project_name> and <revision_name> as    #
# 				arguments by default. 											  #
# 																				  #
# Author:		Jim Dong	(12/01/2004)							              #
#                                                                                 #
# Licensing:    This script is  pursuant to the following license agreement       #
#               (BY VIEWING AND USING THIS SCRIPT, YOU AGREE TO THE               #
#               FOLLOWING): Copyright (c) 2004 Altera Corporation, San Jose,      #
#               California, USA.  Permission is hereby granted, free of           #
#               charge, to any person obtaining a copy of this software and       #
#               associated documentation files (the "Software"), to deal in       #
#               the Software without restriction, including without limitation    #
#               the rights to use, copy, modify, merge, publish, distribute,      #
#               sublicense, and/or sell copies of the Software, and to permit     #
#               persons to whom the Software is furnished to do so, subject to    #
#               the following conditions:                                         #
#                                                                                 #
#               The above copyright notice and this permission notice shall be    #
#               included in all copies or substantial portions of the Software.   #
#                                                                                 #
#               THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,   #
#               EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES   #
#               OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND          #
#               NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT       #
#               HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY,      #
#               WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING      #
#               FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR     #
#               OTHER DEALINGS IN THE SOFTWARE.                                   #
#                                                                                 #
#               This agreement shall be governed in all respects by the laws of   #
#               the State of California and by the laws of the United States of   #
#               America.	 													  #
# 							                                                      #
###################################################################################

###################################################################################
##  Procedure:  list_all_input_regs
##
##  Arguments:  None
##
##  Description:
##      Find and return the input registers.
##
###################################################################################
proc list_all_input_regs {} {

	global quartus

	## Load necessary packages ##
	load_package advanced_timing
	
	set project_name ""
	set revision_name ""

	set project_name [lindex $quartus(args) 0]
	set revision_name [lindex $quartus(args) 1]
	
	# check if project_name is set
	if {$project_name == ""} {
		 return -code error "ERROR: No project name specified"
	}

	# check if revision_name is set
	if {$revision_name == ""} {
		set revision_name [get_current_revision $project_name]
	}
	
    # Open project
	project_open $project_name -revision $revision_name

	# Create Timing Netlist
	create_timing_netlist

	# Create a data structure with sources, delays, and clock
	# paths for each keeper.	
	create_p2p_delays

	# Set the data table header in the Optimization Advisor.
	puts "oaw_add_header_internal {{Node Name} {Node Type}}"

	# local variables
	set has_regs 0
	set node_type "input reg"

	# iterate through all the registers.
	foreach_in_collection node [get_timing_nodes -type reg] {

		# get register's name
		set node_name [get_timing_node_info -info name $node]

		# get the delay from the register
		set delays_from_keeper [get_delays_from_keepers $node]

		# iterate through each delay path
		foreach delay $delays_from_keeper {

			# get the source keeper's type
			set src_type [get_timing_node_info -info type [lindex $delay 0]]

			# if source keeper is a pin, means the current register
			# is an input register
			if {$src_type == "pin"} {
				set has_regs 1
				puts "oaw_add_row_internal {{$node_name} {$node_type}}"
				break
			}
		}
	}

	# if no registers found, display the not found message in the table
	if { $has_regs == 0 } {

		puts "oaw_add_row_internal {{No input registers found }}"
	}

	project_close

	return 0;
}

# Execute the procedure
list_all_input_regs
