# ***************************************************************
# ***************************************************************
#
# File:         vcs_mx.tcl
# Description:  Quartus Nativelink Simulation flow
#               This script is used by Quartus to launch Scirocco
#               tool for Vhdl simulation
#
# Version:      1.0
#
# Authors:      Altera Corporation
#
#               Copyright (c)  Altera Corporation 2003 - .
#               All rights reserved.
#
# ***************************************************************
# ***************************************************************

proc ::quartus::nativelinkflow::sim::compile_source_file {lib file} {
    set status 0
    if {[get_file_type $file] == "verilog"}  {
	set vlogan_cmd [resolve_tool_path "sim" "vlogan"]
	if {$vlogan_cmd == ""} {
	    nl_postmsg error "Error: Can't launch the VCS MX software -- the path to the location of the executables for the VCS MX software not specified."
	    nl_postmsg error "Error: You can specify the path in the EDA Tool Options page of the Options dialog box or using the Tcl command set_user_option."
	    set status 1
	} else {
	    set vcs_mx_cmd "$vlogan_cmd  "
	}
    } else {
	set vhdlan_cmd [resolve_tool_path "sim" "vhdlan"]
	if {$vhdlan_cmd == ""} {
	    nl_postmsg error "Error: Can't launch the VCS MX software -- the path to the location of the executables for the VCS MX software not specified."
	    nl_postmsg error "Error: You can specify the path in the EDA Tool Options page of the Options dialog box or using the Tcl command set_user_option."
	error ""  ""
	    set status 1
	} else {
	    set vcs_mx_cmd "$vhdlan_cmd  -nc"
	}
    }

    if {$status == 0} {
	if {$lib == ""} {
	    set lib "work"
	}

	#We should check if .synopsys_vss.setup contains definition of library $lib, 
	#if not then we should add one...
	if {![file exists $file]} {
	    nl_postmsg error "Error: Can't find file $file" 
	    set status 1
	} else {
	    nl_postmsg info "Info: Compiling file $file to library $lib"
	    nl_logmsg "Running Command: $vcs_mx_cmd -work $lib $file"
	    set cmd_status [catch {eval exec $vcs_mx_cmd -work $lib $file} result]
	    if { ($cmd_status == 0 ) || ([string equal $::errorCode NONE]) } {
		#Command succeeded but wrote something to stderr
		nl_postmsg info "Info: Compilation file $file was successful"
	    } else {
		nl_postmsg error "Error: Compilation of file $file was NOT successful" 
		foreach msg_line [split $result \n] {
		    if {$msg_line != ""} {
			nl_postmsg error "VCS MX: $msg_line"
		    }
		}
		set status 1
	    }
	}
    }
    return $status
}

proc ::quartus::nativelinkflow::sim::launch_vcsmx {args} {
    set status 0
    namespace import ::quartus::nativelinkflow::nl_postmsg
    namespace import ::quartus::nativelinkflow::get_sim_models_root_path
    namespace import ::quartus::nativelinkflow::create_work_dir
    namespace import ::quartus::nativelinkflow::resolve_tool_path

    set lib_path [get_sim_models_root_path]
    set cap [get_project_settings -cmp]
    set lang [lindex $args 0]
    set rtl_sim [lindex $args 1]
    set lib_file $lib_path

    set flow "$lang"
    # create .synopsys_vss.setup setup file
    set setupfilename ".synopsys_vss.setup"
    if [ catch {open $setupfilename w+} file_id ] {
	nl_postmsg  error "Error: Can't open file $setupfilename for writing"
    }

    set env(synopsys_sim_setup) "[pwd]/.synopsys_vss.setup"
    puts $file_id "-- Custom Settings --"
    puts $file_id "TIMEBASE = ps"
    puts $file_id "TIME_RESOLUTION = ps"
    puts $file_id ""
    puts $file_id "-- work directory --"
    close $file_id

    set lib_base_dir "vhdl_libs"
    if {$lang == "verilog"} {
	set lib_base_dir "verilog_libs"
    } else {
	set lib_base_dir "vhdl_libs"
    }
    if ![file isdirectory $lib_base_dir] {
	file mkdir $lib_base_dir
    }

    set libs [get_sim_models_required $lang $rtl_sim]
    foreach lib_info $libs {
	set lib_name [lindex $lib_info 0]
	set lib_sources [lindex $lib_info 1]

    	set lib_dir "$lib_base_dir/$lib_name"

	if ![file isdirectory $lib_dir] {
	    nl_postmsg info "Info: Creating directory $lib_dir"
	    create_work_dir $lib_dir
	}

	set file_id [open $setupfilename a+]
	puts $file_id "$lib_name : $lib_dir"
	close $file_id

	foreach source $lib_sources {
	    if {$status == 0} {
		set status [compile_source_file $lib_name $source]
	    }
	}
    }


    if {$status == 0} {
	set design_files_info [get_design_files $lang $rtl_sim]
	set work_lib "gate_work"
	if {$rtl_sim == "1"} {
	    set work_lib "rtl_work"
	} else {
	    set work_lib "gate_work"
	}
	if ![file isdirectory $work_lib] {
	    file mkdir $work_lib
	}

	set file_id [open $setupfilename a+]
	puts $file_id "WORK > $work_lib"
	puts $file_id "$work_lib : ./$work_lib"
	close $file_id

	foreach design_file_info $design_files_info {
	    set lib_name [lindex $design_file_info 0]
	    set lib_sources [lindex $design_file_info 1]
	    foreach source $lib_sources {
		if {$status == 0} {
		    set status [compile_source_file $lib_name $source]
		}
	    }
	}
    }

    if {$status == 0} {
	set tb_mode [get_testbench_mode "gate"]
	
	if {$tb_mode == "testbench"} {
	    set tb_info [get_testbench_info]
	    if {$tb_info == ""} {
		nl_postmsg error "Error: No Simulation Test Bench specified -- cannot continue NativeLink simulation"
		error "" ""
	    } else {
		set tb_files [lindex $tb_info 3]
		if {$tb_files == ""} {
		    nl_postmsg error "Error: Test Bench does not contain any stimulus files -- cannot continue NativeLink simulation"
		    set status 1
		}
		set testbench_module_name  [lindex $tb_info 0]
		if {$testbench_module_name == ""} {
		    nl_postmsg error "Error: Top level entity/module name for test bench not specified -- cannot continue NativeLink simulation"
		    set status 1
		}
	    }
	    if {$status == 0} {
		foreach file $tb_files {
		    if {$status == 0 } {
			set status [compile_source_file $lib_name $file]
		    }
		}
	    }

	    if {$status == 0} {
		set tb_module [string toupper $testbench_module_name]
		if [file exists "$work_lib/$tb_module\.sim"] {
		    set elab_cmd [resolve_tool_path "sim" "scs"]
		    if {$elab_cmd == ""} {
			nl_postmsg error "Error: VCX MX command $elab_cmd not found"
			set status 1
		    } else {
			set elab_cmd "$elab_cmd -nc -mhdl -debug_all"
			set launch_cmd "scsim -gui"
		    }
		    #set launch_cmd "scsim -RI"
		} else {
		    set elab_cmd [resolve_tool_path "sim" "vcs"]
		    if {$elab_cmd == ""} {
			nl_postmsg error "Error: VCX MX command $elab_cmd not found"
			set status 1
		    }
		    #set launch_cmd "simv"
		    set elab_cmd "$elab_cmd -nc -debug_all"
		    set launch_cmd "simv -gui"
		}


		set elab_cmd "$elab_cmd $work_lib.$testbench_module_name"

		if {$status == 0} {
		    #Elaborate design
		    nl_postmsg info "Info: Elaborating design $testbench_module_name"
		    nl_logmsg "Running Command: $elab_cmd"
		    set cmd_status [ catch {eval exec $elab_cmd} result]
		    if { ($cmd_status == 0 ) || ([string equal $::errorCode NONE]) } {
			nl_postmsg info "Info: Elaboration of design $testbench_module_name was successful" 
		    } else {
			if {[lindex $::errorCode 0] ne "NONE"} {
			    nl_postmsg error "Error: Elaboration of design $testbench_module_name was NOT successful"
			    foreach msg_line [split $result \n] {
				if {$msg_line != ""} {
				    nl_postmsg error "VCS MX: $msg_line"
				}
			    }
			    set status 1
			}
		    }
		}

		if {$status == 0} {
		    set sim_setup_script [lindex $tb_info 4]
		    nl_postmsg warning "sim_setup_script is $sim_setup_script"
		    if {$sim_setup_script == "" } {
			set sim_setup_script "vcsmx_sim.ucli"
			if [ catch {open "$sim_setup_script" w+} ucli_file_id ] {
			    nl_postmsg  error "Error: Can't open file $setupfilename for writing"
			    set status 1
			} else {
			    puts $ucli_file_id "run [lindex $tb_info 2]"
			    close $ucli_file_id
			}
		    } 
		}

		if {$status == 0} {
		    set launch_cmd "$launch_cmd -i $sim_setup_script"
		    nl_logmsg "Running Command: $launch_cmd"
		    set cmd_status [ catch {eval exec $launch_cmd} result]
		    if { ($cmd_status == 0 ) || ([string equal $::errorCode NONE]) } {
			nl_postmsg info "Info: Simulation of design $testbench_module_name was successful" 
		    } else {
			if {[lindex $::errorCode 0] ne "NONE"} {
			    nl_postmsg error "Error: Simulation of design $testbench_module_name was NOT successful"
			    foreach msg_line [split $result \n] {
				if {$msg_line != ""} {
				    nl_postmsg error "VCS MX: $msg_line"
				}
			    }
			    set status 1
			}
		    }
		}
	    }
	} else {
	    nl_postmsg info "Info: Successfully compiled the Simulation Libraries and sources for VCS MX "
	    nl_postmsg info "Info: To perform simulation using VCS MX you should"
	    nl_postmsg info "Info:  (1) Compile the testbench using the commands vhdlan and/or vlogan"
	    nl_postmsg info "Info:  (2) Elaborate the design using the command scs or vcs"
	    nl_postmsg info "Info:  (3) Launch simulator using command simv or scsim"

	}
    }
    return $status
}
