# ***************************************************************
# ***************************************************************
#
# File:         vcs.tcl
# Description:  Quartus Nativelink Simulation flow
#               This script is used by Quartus to launch VCS
#               tool for Verilog simulation
#
# Version:      1.0
#
# Authors:      Altera Corporation
#
#               Copyright (c)  Altera Corporation 2003 - .
#               All rights reserved.
#
# ***************************************************************
# ***************************************************************

proc ::quartus::nativelinkflow::sim::launch_vcs {args} {
    set status 0
    namespace import ::quartus::nativelinkflow::nl_postmsg
    namespace import ::quartus::nativelinkflow::get_sim_models_root_path
    namespace import ::quartus::nativelinkflow::create_work_dir

    set lib_path [get_sim_models_root_path]
    set cap [get_project_settings -cmp]
    #The language argument is not used, however it is included to make
    #all launch_* functions similar
    set lang [lindex $args 0]
    set rtl_sim [lindex $args 1]
    set lib_file $lib_path
    
    set vcs_options_file "$cap.vcs"
    if [catch {open $vcs_options_file w+} vcs_options_file_id] {
	nl_postmsg error "Error: Unable to open file $vcs_options_file"
	return 1
    }

    #atoms file
    puts -nonewline $vcs_options_file_id " +cli+1 -line "
    set libs [get_sim_models_required "verilog" $rtl_sim]
    foreach lib_info $libs {
	set lib_name [lindex $lib_info 0]
	set lib_sources [lindex $lib_info 1]
	foreach source $lib_sources {
	    puts -nonewline $vcs_options_file_id " -v $source"
	}
    }

    set design_files_info [get_design_files "verilog" $rtl_sim]
    foreach design_file_info $design_files_info {
	set lib_name [lindex $design_file_info 0]
	set lib_sources [lindex $design_file_info 1]
	foreach source $lib_sources {
	    puts -nonewline $vcs_options_file_id " $source"
	}
    }

    set tb_mode [get_testbench_mode "gate"]
    
    if {$tb_mode == "testbench"} {
	set tb_info [get_testbench_info]
        if {$tb_info == ""} {
	    nl_postmsg error "Error: No Simulation Test Bench specified -- cannot continue NativeLink simulation"
	    error "" ""
	} else {
	    set tb_files [lindex $tb_info 3]
	    if {$tb_files == ""} {
		nl_postmsg error "Error: Test Bench does not contain any stimulus files -- cannot continue NativeLink simulation"
		error "" ""
	    }
	    set testbench_module_name  [lindex $tb_info 0]
	    if {$testbench_module_name == ""} {
		nl_postmsg error "Error: Top level entity/module name for test bench not specified -- cannot continue NativeLink simulation"
		error "" ""
	    }
	}

	foreach file $tb_files {
	    puts -nonewline $vcs_options_file_id " $file"
	}

	if [is_glitch_filter_enabled] {
		puts -nonewline $vcs_options_file_id " +transport_int_delays +pulse_int_r/0 +pulse_int_e/0 +transport_path_delays +pulse_r/0 +pulse_e/0"
	}
	set sim_setup_script [lindex $tb_info 4]
	if {$sim_setup_script != ""} {
	    puts -nonewline $vcs_options_file_id " -i $sim_setup_script"
	}
	close $vcs_options_file_id

	nl_postmsg info "Info: Compiling and Simulating design $testbench_module_name"
	namespace import ::quartus::nativelinkflow::resolve_tool_path
	set vcs_cmd [resolve_tool_path "sim" "vcs"]
	if {$vcs_cmd == ""} {
	    nl_postmsg error "Error: Can't launch the VCS software -- the path to the location of the executables for the VCS software not specified."
	    nl_postmsg error "Error: You can specify the path in the EDA Tool Options page of the Options dialog box or using the Tcl command set_user_option."
	} else {
	    nl_logmsg "Running Command $vcs_cmd -RI -file $vcs_options_file" 
	    if [ catch {exec "$vcs_cmd" -RI -file $vcs_options_file } result] {
		nl_postmsg error "Error: Compilation and Simulation of design $testbench_module_name was NOT successful"
		foreach msg_line [split $result \n] {
		    if {$msg_line != ""} {
			nl_postmsg error "VCS: $msg_line"
		    }
		}
		set status 1
	    } else {
		nl_postmsg info "Info: Compilation and Simulation of test bench file $testbench_module_name was successful" 
	    }
	}
    } else {
	    close $vcs_options_file_id
    
	    nl_postmsg info "Info: Successfully wrote the options file $vcs_options_file for VCS"
	    nl_postmsg info "Info: To perform simulation using VCS you should"
	    nl_postmsg info "Info:  (1) Add the testbench filename to the options file $vcs_options_file"
	    nl_postmsg info "Info:  (2) Compile and simulate design using vcs -RI -file $vcs_options_file"
    }
    return $status
}
