# ***************************************************************
# ***************************************************************
#
# File:         qnativesimflow.tcl
# Description:  Quartus NativeLink Simulation flow
#               This script is used by Quartus to launch eda
#               simulation tools.
#
# Version:      1.0
#
# Authors:      Altera Corporation
#
#               Copyright (c)  Altera Corporation 2003 - .
#               All rights reserved.
#
# ***************************************************************
# *************************************************************** 

# All quartus related variables have prefix "q_"
# All EDA simulation tool related variables have prefix "s_"
# All globals in CAPS

# Global variables


# Global procedures

# Root namespace

namespace eval ::quartus::nativelinkflow::sim {


# Error codes

    variable ERROR_UNKNOWN_NETLIST_TYPE    5

# Error code messages

    variable ERROR_UNKNOWN_NETLIST_TYPE_STRING    "Error: Unknown EDA simulation netlist type"

# Other error/info messages

    variable ERROR_INIT_SIM_SETTINGS     "Error: Initialization of EDA simulation settings was NOT successful"
    variable ERROR_SOURCE_NL_SCRIPT      "Error: Sourcing NativeLink script"
    variable ERROR_ABSENT_NL_SCRIPT      "Error: Can't find NativeLink simulation tool script "
    variable ERROR_ABSENT_SIM_DIR        "Error: Can't find simulation directory "
    variable ERROR_OUTPUT_NETLISTER_NOT_RUN        "Error: Simulation output netlist writer for EDA simulation tool not run"
    variable ERROR_NL_FOR_SIMTOOL_NOT_SUPPORTED  "Error: NativeLink simulation flow not supported for current EDA simulation tool"

# Info  messages

    variable INFO_INIT_SIM_SETTINGS_SUCCESS     "Info: Initialization of EDA simulation settings was successful "
    variable INFO_END_SIM_TOOL_LAUNCH           "Info: Ending launching of EDA simulation tool"

    namespace export run_eda_simulation_tool ERROR* INFO*

    variable q_sim_tool        ""
    variable q_sim_lang        ""
    variable q_sim_output_file ""
    variable q_sim_sdf_file    ""
    variable q_sim_dir         ""
    variable q_vhd_version


    proc set_q_sim_environ       {} {}
    proc run_simulator           {} {}
    proc get_sim_tool_name       {} {}

    namespace eval modelsim {
	namespace export run_modelsim
	variable ::quartus

	proc run_modelsim {} {}

    }

    namespace eval modelsim-altera {
	namespace export run_modelsim-altera
	variable ::quartus

	proc run_modelsim-altera {} {}
    }

    namespace eval ncsim {
	namespace export run_ncsim
	variable ::quartus

	proc run_ncsim {} {}

    }

    namespace eval scsim {
	namespace export run_scsim
	variable ::quartus
	proc run_scsim {} {}

    }

    namespace eval vcs {
	namespace export run_vcs
	variable ::quartus

	proc run_vcs {} {}

    }

    namespace export get_sim_models_required
    namespace export get_design_files
    namespace export get_testbench_files
    namespace export get_testbench_mode	
    namespace export get_testbench_file
    namespace export get_testbench_name
    namespace export get_testbench_run_for
    namespace export get_design_instance_name
    namespace export get_command_script	
    namespace export is_vcd_generation_enabled
    namespace export is_glitch_filter_enabled
    namespace export goto_sim_dir	
    namespace export get_sim_dir	
    namespace export get_eda_writer_netlist_ext	
    namespace export launch_modelsim
    namespace export launch_modelsim_verilog
    namespace export launch_modelsim_vhdl
    namespace export launch_ncsim
    namespace export launch_vcs
    namespace export launch_vcsmx
    namespace export get_ip_info
    namespace export get_unencrypted_hdl_files
    namespace export qmap_successfully_completed
    namespace export backup_file
    namespace export get_file_type
    namespace export is_timing_simulation_on
}

############################################################################
##
proc ::quartus::nativelinkflow::sim::get_file_type {filename} {
##
############################################################################
    set verilog_ext ".v .vh .vlg .vo .vt"
    set vhdl_ext ".vhd .vhdl .vht .vho"
    set type "unknown"

    set ext [file extension $filename]
    if {[lsearch -exact $verilog_ext $ext] != -1} {
	set type "verilog"
    } elseif {[lsearch -exact $vhdl_ext $ext] != -1} {
	set type "vhdl"
    }
    return $type
}

#######################################################################
##
proc ::quartus::nativelinkflow::sim::get_sim_models_required {language rtl_sim} {
##
#######################################################################
    set files_list ""
    set lib_path [get_sim_models_root_path]
    set part_info [ get_nativelink_info "qeda_simulation.dat"]						
    if {$part_info == ""} {
	  nl_postmsg error "Error: Family not supported for NativeLink Simulation"
	  error "Family not supported"
    }
    set family_name [string tolower [lindex $part_info 0]]
    switch -regexp -- $language {
	(?i)^verilog$ 
	{
	   lappend files_list "\{$family_name\_ver\} \{$lib_path/$family_name\_atoms.v\}"
	   if {$rtl_sim == 1} {
		lappend files_list "\{lpm_ver\} \{$lib_path/220model.v\}"
		lappend files_list "\{altera_ver\} \{$lib_path/altera_primitives.v\}"
		lappend files_list "\{altera_mf_ver\} \{$lib_path/altera_mf.v\}"
		if ![regexp "apex" $family_name] {
		    lappend files_list "\{sgate_ver\} \{$lib_path/sgate.v\}"
		}
	    }
	    if [regexp "stratix(ii)?gx" $family_name] {
		if {$rtl_sim != 1} {
		    lappend files_list "\{lpm_ver\} \{$lib_path/220model.v\}"
		    lappend files_list "\{sgate_ver\} \{$lib_path/sgate.v\}"
		}
		if [regexp "stratixgx" $family_name] {
		    lappend files_list "\{altgxb_ver\} \{$lib_path/$family_name\_mf.v\}"
		    lappend files_list "\{$family_name\_gxb_ver\} \{$lib_path/$family_name\_hssi_atoms.v\}"
		} else {
		    #Stratix II - Verilog
		    lappend files_list "\{$family_name\_hssi_ver\} \{$lib_path/$family_name\_hssi_atoms.v\}"
		} 
	    }
	}
	(?i)^vhdl$ 
	{
	    lappend files_list "\{$family_name\} \{$lib_path/$family_name\_atoms.vhd $lib_path/$family_name\_components.vhd\}"
	    if {$rtl_sim == 1} {
		lappend files_list "\{lpm\} \{$lib_path/220pack.vhd $lib_path/220model.vhd\}"
		lappend files_list "\{altera\} \{$lib_path/altera_primitives_components.vhd $lib_path/altera_primitives.vhd\}"
		lappend files_list "\{altera_mf\} \{$lib_path/altera_mf_components.vhd $lib_path/altera_mf.vhd\}"
		if ![regexp "apex" $family_name] {
		    lappend files_list "\{sgate\} \{$lib_path/sgate_pack.vhd $lib_path/sgate.vhd\}"
		}
	    }
	    if [regexp "stratix(ii)?gx" $family_name] {
		if {$rtl_sim != 1} {
		    lappend files_list "\{lpm\} \{$lib_path/220pack.vhd $lib_path/220model.vhd\}"
		    lappend files_list "\{sgate\} \{$lib_path/sgate_pack.vhd $lib_path/sgate.vhd\}"
		}
		if [regexp "stratixgx" $family_name] {
		    lappend files_list "\{altgxb\} \{$lib_path/$family_name\_mf.vhd $lib_path/$family_name\_mf_components.vhd\}"
		    lappend files_list "\{$family_name\_gxb\} \{$lib_path/$family_name\_hssi_atoms.vhd $lib_path/$family_name\_hssi_components.vhd\}"
		} else {
		    #Stratix II - VHDL
		    lappend files_list "\{$family_name\_hssi\} \{$lib_path/$family_name\_hssi_components.vhd $lib_path/$family_name\_hssi_atoms.vhd\}"
		}
	    }
	}
    }
    return $files_list
}

#######################################################################
##
proc ::quartus::nativelinkflow::sim::get_design_files {language rtl_sim} {
##
##
#######################################################################
    set design_files_list ""
    if {$rtl_sim =="1"} {
        #Since RTL may contain both VHDL and Verilog design files, 
	#we would ignore the language.
	set file_list [get_unencrypted_hdl_files "mixed"]
	set ignored_files ""
	foreach file $file_list {
	    if {[lsearch -exact $ignored_files $file] == -1} {
		set full_file_name [get_file_info -info full_path -filename $file]
		array set list [get_ip_info $full_file_name]
		if {([array names list related_files] != "" ) && ($list(related_files) != "")} {
		    # All related_files to ignored_files list
		    set ignored_files "$ignored_files [split $list(related_files) ,]"
		}
		if {([array names list ipfs_file] != "" ) && ($list(ipfs_file) != "")} {
		    set path [file dirname $full_file_name]
		    lappend design_files_list "\{rtl_work\} \{${path}/$list(ipfs_file)\}"
		} else {
		    lappend design_files_list "\{rtl_work\} \{$full_file_name\}"
		}
	    }
	}
    } else {
	switch -regexp -- $language {
	    (?i)^verilog$ 
	    {
		set ext "vo";
	    }
	    (?i)^vhdl$ 
	    {
		set ext "vho";
	    }
	}
	set cap [get_project_settings -cmp]
	lappend design_files_list "\{gate_work\} \{${cap}.${ext}\}"
    }
    return $design_files_list
}

#######################################################################
##
proc ::quartus::nativelinkflow::sim::get_testbench_info {} {
##
#######################################################################
    set testbench_info ""
    #The sim_mode is not used - the significance of assignments has changed.
    set testbench [get_global_assignment -name EDA_NATIVELINK_SIMULATION_TEST_BENCH -section_id eda_simulation]
    if {$testbench == ""} {
	return $testbench_info
    }
    lappend testbench_info [get_global_assignment -name EDA_TEST_BENCH_MODULE_NAME -section_id $testbench]
    lappend testbench_info [get_global_assignment -name EDA_DESIGN_INSTANCE_NAME -section_id $testbench]
    lappend testbench_info [get_global_assignment -name EDA_TEST_BENCH_RUN_SIM_FOR -section_id $testbench]

    set tb_files ""
    set tb_files_col [get_all_global_assignments -name EDA_TEST_BENCH_FILE -section_id $testbench]
    foreach_in_collection asgn $tb_files_col {

	#if the testbench file has no path then use the project path.
	set tb_file [ convert_filepath_to_tclstyle [lindex $asgn 2 ]]

	if { [string compare -nocase $::tcl_platform(platform) "windows"] == 0} {
	    if ![regexp -nocase {^[a-z]\:/} $tb_file] {
		set tb_file "[get_project_dir]/$tb_file"
	    }
	} else {
	    if {![regexp {^/} $tb_file]} {
		set tb_file "[get_project_dir]/$tb_file"
	    }
	}
	lappend tb_files $tb_file
    }
    lappend testbench_info $tb_files

    set tb_script [get_global_assignment -name EDA_NATIVELINK_SIMULATION_SETUP_SCRIPT -section_id eda_simulation]
    if {$tb_script != "" } {
	if { [string compare -nocase $::tcl_platform(platform) "windows"] == 0} {
	    if ![regexp -nocase {^[a-z]\:/} $tb_script] {
		set tb_script "[get_project_dir]/$tb_script"
	    }
	} else {
	    if {![regexp {^/} $tb_script]} {
		set tb_script "[get_project_dir]/$tb_script"
	    }
	}
    }
    lappend testbench_info $tb_script
    return $testbench_info
}

#######################################################################
##
proc ::quartus::nativelinkflow::sim::backup_file {filename} {
##
#######################################################################
	#if current filename is <name>.bak<num> then backup file is called
	# <name>.bak<num++>, otherwise backbup file is called <name>.bak
    if [file exists $filename] {
	set bkp_file_name "$filename.bak"
	if ![file exists $bkp_file_name] {
	    set bkp_file_name ${filename}.bak
	} else {
	    for {set bkp_idx 1 } {$bkp_idx <= 10 } {incr bkp_idx} {
		if ![file exists ${bkp_file_name}$bkp_idx] {
		    break
		}
	    }  
	    set bkp_file_name ${bkp_file_name}$bkp_idx
	}
	file copy -force $filename $bkp_file_name
	nl_postmsg warning "Warning: File $filename already exists - backing up current file as $bkp_file_name"
    }
}

#######################################################################
##
proc ::quartus::nativelinkflow::sim::qmap_successfully_completed {} {
##
#######################################################################
    package require ::quartus::report
    set return_val 1
    set cmp_status ""

    if [catch {load_report}] {
	set return_val 0
    } else {
	set panel_id [get_report_panel_id {Analysis & Synthesis||Analysis & Synthesis Summary}]
	if {$panel_id != -1} {
	    set cmp_status [get_report_panel_data -row 0 -col 1 -id $panel_id]
	} else {
	    set panel_id [get_report_panel_id {Analysis & Synthesis||Analysis & Elaboration Summary}]
	    if {$panel_id != -1} {
		set cmp_status [get_report_panel_data -row 0 -col 1 -id $panel_id]
	    } else {
		#Failed
		set return_val 0
	    }
	}
	unload_report
    }
    if {$return_val != 0} {
	#check if cmp_status is "Failed - ..." if so set return_val to 0
	if ![regexp "^Successful - (.)*" $cmp_status] {
	    set return_val 0
	}
    }
    return $return_val
}

#######################################################################
##
proc ::quartus::nativelinkflow::sim::get_unencrypted_hdl_files {language} {
##
#######################################################################
    set file_list {}
    set unencrypted_compiled_file_list {}
    variable ::quartus
    #HDL files in <qinstall>/libraries/megafunctions should be ignored
    set quartus_mega_lib_path "[file dirname $quartus(binpath)]"
    set quartus_mega_lib_path "[file join $quartus_mega_lib_path libraries]"
    set quartus_mega_lib_path "[file join $quartus_mega_lib_path megafunctions]"
    switch -regexp -- $language {
	(?i)^verilog$
	{
	    set file_list [get_files -type verilog]
	}
	(?i)^vhdl$
	{
	    set file_list [get_files -type vhdl]
	}
	(?i)^mixed$
	{
	    set file_list [get_files -type verilog]
	    set vhdl_file_list [get_files -type vhdl]

    	    if {$vhdl_file_list != ""} {
		if {$file_list != ""} {
		    set file_list [concat $file_list "$vhdl_file_list"]
		} else {
		    set file_list "$vhdl_file_list"
		}
	    }
	}
    }

    foreach file $file_list {
	set full_file_name [get_file_info -filename $file -info full_path]
	set file_path [file dirname $full_file_name]
	if {$file_path != $quartus_mega_lib_path} {
	    if [get_file_info -filename $full_file_name -info compiled_status] {
		if {!([get_file_info -info is_encrypted -filename $full_file_name] \
		    || [get_file_info -info encrypted_submodule_file -filename $file]) } {
		    lappend unencrypted_compiled_file_list $file
		}
	    }
	}
    }
    return $unencrypted_compiled_file_list
}

#######################################################################
##
proc ::quartus::nativelinkflow::sim::get_ip_info {filename} {
##
#######################################################################
    set ip_file 0
    set mega_wiz_file 0
    set related_files ""
    set ipfs_files ""
    set return_val ""

    if {[get_file_type $filename] eq "vhdl"} {
	set comment "--"
    } else {
	set comment "//"
    }
    if [ catch { open $filename r} file_id ] {
	nl_postmsg error "Error : Can't open file -- $filename"
    }
    while  {![eof $file_id ]} {
	#Check to see if this is IP file.
	#IP files will have first 2 lines as follows
	# //megafunciton  wizard: ...
	# //GENERATION: XML
	#Skip empty lines
	set line [gets $file_id]
	if [regexp "^( 	)*$" $line] {
	    continue
	}
	#If the first non-empty line is a megafunction line, this is either 
	#an IP file or megafunction file. The IP files will
        #have GENERATION: XML as following line
	if [regexp -nocase "^${comment} megafunction wizard\: \%(.*)\%" $line] {
	    set line [gets $file_id]
	    if [regexp "^${comment} GENERATION\: XML" $line] {
		set ip_file 1
		break
	    } elseif [regexp -nocase "^${comment} GENERATION ?\: ?STANDARD" $line] {
		set mega_wiz_file 1
	    break
	    } 
	} elseif [regexp -nocase "^${comment}( )?Generated by (.+)Altera, IP Toolbench (.+)" $line ] {
	    set ip_file 1
	    break
	}
    }

    if {($mega_wiz_file == "1") || ($ip_file == "1")} {
	#if this was an IP file then look for related files.
	# and ipfs files information
	while  {![eof $file_id ]} {
	    set line [gets $file_id]
	    if [regexp -nocase "^${comment} ?RELATED_FILES ?\: ?(\[^;\]*)(;)?" $line junk file_list] {
		regsub -all " " $file_list {} file_list
		set related_files $file_list
		lappend return_val "related_files"
		lappend return_val $related_files
	    } elseif [regexp -nocase "^${comment} ?IPFS_FILES ?\: ?(\[^;\]*)(;)?" $line junk file_list] {
		regsub -all " " $file_list {} file_list
		set ipfs_files $file_list
		lappend return_val "ipfs_file"
	        lappend return_val $ipfs_files
	    } else {
		continue
	    }
	}
    }

    if {($ip_file == "1") && ($related_files == "") && ($ipfs_files == "")} {
	post_message -type error "Error: File $filename is generated using an older version of MegaCore which is not supported by RTL NativeLink Simulation" -file $filename
	nl_logmsg "Error: File $filename is generated using older version of MegaCore which is not supported by RTL NativeLink Simulation"
	nl_postmsg error "Error: Regenerate the IP files using latest version of MegaCores for RTL NativeLink Simulation flow to function correctly"
	error "" "Regenerate the IP files using latest version of MegaCores for RTL NativeLink Simulation flow to function correctly" 
    }

    if [ catch {close $file_id} err ] {
	set savedCode $errorCode
	set savedInfo $errorInfo
	error "" $savedInfo $savedCode
    }
    
    return $return_val
}

#######################################################################
##
proc ::quartus::nativelinkflow::sim::get_testbench_mode {sim_mode} {
##
#######################################################################
    set mode "none"
    set sim_mode "gate"
    if {[string compare -nocase $sim_mode "gate"] == 0} {
	set qsf_asgn_name EDA_TEST_BENCH_ENABLE_STATUS
    } else {
	set qsf_asgn_name EDA_RTL_SIM_MODE
    }
    set mode [lindex [ get_global_assignment -name $qsf_asgn_name -section_id eda_simulation ] 0]

    if {[string compare -nocase $mode "NOT_USED"] == 0} {
	set mode "none"
    } elseif {[string compare -nocase $mode "TEST_BENCH_MODE"] == 0} {
	set mode "testbench"
    } elseif {[string compare -nocase $mode "COMMAND_MACRO_MODE"] == 0}  {
    	set mode "script"
    }
    return $mode
}

#######################################################################
##
proc ::quartus::nativelinkflow::sim::get_testbench_file {sim_mode} {
##
#######################################################################
    namespace import ::quartus::nativelinkflow::convert_filepath_to_tclstyle
    if {[string compare -nocase $sim_mode "gate"] == 0} {
	set qsf_asgn_name EDA_TEST_BENCH_FILE_NAME
    } else {
	set qsf_asgn_name EDA_RTL_TEST_BENCH_FILE_NAME
    }
    set tb_file_name [ lindex [ get_global_assignment -name $qsf_asgn_name -section_id eda_simulation ] 0]
    
    #if the testbench file has no path then use the project path.
    set tb_file_name [ convert_filepath_to_tclstyle $tb_file_name ]

    if { [string compare -nocase $::tcl_platform(platform) "windows"] == 0} {
	if ![regexp -nocase {^[a-z]\:/} $tb_file_name] {
	    set tb_file_name "../../$tb_file_name"
	}
    } else {
	if {![regexp {^/} $tb_file_name]} {
	    set tb_file_name "../../$tb_file_name"
	}
    }

    return $tb_file_name
}

#######################################################################
##
proc ::quartus::nativelinkflow::sim::get_testbench_name {sim_mode} {
##
#######################################################################
    if {[string compare -nocase $sim_mode "gate"] == 0} {
	set qsf_asgn_name EDA_TEST_BENCH_ENTITY_MODULE_NAME
    } else {
	set qsf_asgn_name EDA_RTL_TEST_BENCH_NAME
    }
    set tb_name [ lindex [ get_global_assignment -name $qsf_asgn_name -section_id eda_simulation ] 0]
    return $tb_name
}

#######################################################################
##
proc ::quartus::nativelinkflow::sim::get_testbench_run_for {sim_mode} {
##
#######################################################################
    set run_for "-all"

    if {[string compare -nocase $sim_mode "gate"] == 0} {
	set qsf_asgn_name EDA_TEST_BENCH_RUN_FOR
    } else {
	set qsf_asgn_name EDA_RTL_TEST_BENCH_RUN_FOR
    }

    set run_for [get_global_assignment -name $qsf_asgn_name -section_id eda_simulation]
    if {[string compare $run_for  ""] == 0} {
	set run_for "-all"
    } else {
	regsub -all " " $run_for {} run_for 
    }
     return [string tolower $run_for]
}

#######################################################################
##
proc ::quartus::nativelinkflow::sim::get_design_instance_name {sim_mode} {
    #Sim mode will always be gate level?
##
#######################################################################
     set dsgn_inst_name [ lindex [ get_global_assignment -name {EDA_TEST_BENCH_DESIGN_INSTANCE_NAME} -section_id eda_simulation ] 0]
     return $dsgn_inst_name
}

#######################################################################
##
proc ::quartus::nativelinkflow::sim::get_command_script {sim_mode} {
##
#######################################################################
    if {[string compare -nocase $sim_mode "gate"] == 0} {
	set qsf_asgn_name EDA_SIMULATION_RUN_SCRIPT
    } else {
	set qsf_asgn_name EDA_RTL_SIMULATION_RUN_SCRIPT
    }

    set script_file_name [ lindex [ get_global_assignment -name $qsf_asgn_name -section_id eda_simulation ] 0]	

    set script_file_name [ convert_filepath_to_tclstyle $script_file_name ]

    if { [string compare -nocase $::tcl_platform(platform) "windows"] == 0} {
	if ![regexp -nocase {^[a-z]\:/} $script_file_name] {
	    set script_file_name "../../$script_file_name"
	}
    } else {
	if {![regexp {^/} $script_file_name]} {
	    set script_file_name "../../$script_file_name"
	}
    }
    return $script_file_name
}

#######################################################################
##
proc ::quartus::nativelinkflow::sim::is_vcd_generation_enabled {} {
##
#######################################################################
    return [is_setting_enabled vcd_gen]
}

#######################################################################
##
proc ::quartus::nativelinkflow::sim::is_glitch_filter_enabled {} {
##
#######################################################################
    return [is_setting_enabled glitch_filter]
}

#######################################################################
##
proc ::quartus::nativelinkflow::sim::is_setting_enabled {setting} {
##
#######################################################################
    set enabled 0
    variable	setting_name_to_qsf {
		    {{glitch_filter} {EDA_ENABLE_GLITCH_FILTERING}}
		    {{vcd_gen} {EDA_WRITE_NODES_FOR_POWER_ESTIMATION}}
		}

    foreach item $setting_name_to_qsf {
	set old_name [lindex $item 0]
	if {[string compare -nocase $old_name $setting ] == 0} {
	    set value [ get_global_assignment -name [lindex $item 1] -section_id eda_simulation ]
	    if {[string compare -nocase $value "OFF"] != 0} {
		set enabled 1
	    }
	    break
	}
    }
    return $enabled
}

#######################################################################
##
proc ::quartus::nativelinkflow::sim::get_sim_dir {tool_name} { 
##
#######################################################################
    array set tool_dir_map   {
       modelsim	        modelsim 
       modelsim-altera 	modelsim 
       vcs		vcs
       ncsim		ncsim
       scsim		scsim 
    }
    set default_output_dir "[get_project_dir]/simulation/$tool_dir_map($tool_name)"

    set eda_sim_writer_output_dir [get_global_assignment -name EDA_NETLIST_WRITER_OUTPUT_DIR -section_id eda_simulation]
    if {$eda_sim_writer_output_dir == ""} {
	    set eda_sim_writer_output_dir "$default_output_dir"
    }
    return $eda_sim_writer_output_dir
}

#######################################################################
##
##
proc ::quartus::nativelinkflow::sim::goto_sim_dir {tool_name} {
##
##
#######################################################################
    set return_val 0
    set path_dirs "simulation"

    array set tool_dir_map   {
       modelsim	        modelsim 
       modelsim-altera 	modelsim 
       vcs		vcs
       ncsim		ncsim
       scsim		scsim 
    }

    set sim_dir [get_sim_dir $tool_name]
    if [file exists $sim_dir] {
	if [catch {cd $sim_dir} err] {
	    nl_postmsg error "Error: $err"
	    set return_val 1
	    break
	}
    } else {
	if [catch {file mkdir $sim_dir } err ] {
	    nl_postmsg error "Error: Can't create directory [pwd]/$sim_dir"
	    set return_val 1
	} else {
	    if [catch {cd $sim_dir} err] {
		nl_postmsg error "Error: $err"
		set return_val 1
		break
	    }
	}
    }
    return $return_val
}

#######################################################################
##
proc ::quartus::nativelinkflow::sim::get_eda_writer_netlist_ext {language} {
##
#######################################################################
    set ext "vo"
    if {$language == "vhdl" } {
	set ext "vho"
    } 
    return $ext
}

#######################################################################
##
proc ::quartus::nativelinkflow::sim::get_eda_writer_sdf_ext {language} {
##
#######################################################################
    set ext "_verilog.sdo"
    if {$language == "vhdl" } {
	set ext "_vhdl.sdo"
    } 
    return $ext
}

#######################################################################
##
proc ::quartus::nativelinkflow::sim::get_sim_tool_name {} {
##
#######################################################################
    array set tool_map   {
       modelsim        modelsim 
       modelsim-altera modelsim-altera 
       vcs             vcs
       nc-verilog      ncsim
       nc-vhdl         ncsim
       {vcs mx}        scsim 
    }
    
    set qsf_tool_name [ get_global_assignment -name {EDA_SIMULATION_TOOL} ] 
    #QSF allows short versions of Tool names. 
    #First convert these to standard lowercase names
    set standard_tool_name [convert_to_standard_name $qsf_tool_name]
    set standard_tool_name [string tolower $standard_tool_name]

    #remove everything in braces (), i.e tool names 
    # ModelSim(Verilog) will become modelsim
    if ![regexp -nocase {([^(]+) \(([a-z ]+)\)} $standard_tool_name full_match tool_name language] {
	set tool_name $standard_tool_name
    }

    if {[array names tool_map $tool_name] == "" }  {
	set sim_tool ""
    } else {
	set sim_tool $tool_map($tool_name)
    }
    return $sim_tool
}

#######################################################################
##
proc ::quartus::nativelinkflow::sim::set_q_sim_environ {} {
##
#######################################################################

    variable q_sim_lang
    variable q_sim_tool
    variable q_sim_output_file
    variable q_sim_sdf_file
    variable q_vhd_version
    variable q_sim_dir

    set rev_name [get_project_settings -cmp]
    set q_sim_tool [get_sim_tool_name]

    array set tool_dir_map   {
       modelsim	        modelsim 
       modelsim-altera 	modelsim 
       vcs		vcs
       ncsim		ncsim
       scsim		scsim 
    }

    if { [string compare -nocase $::tcl_platform(platform) "windows"] != 0} {
	set q_sim_dir "simulation/$tool_dir_map($q_sim_tool)"
    } else {	
	set q_sim_dir "simulation\\$tool_dir_map($q_sim_tool)"
    }

    set q_vhd_version "OFF"
    
    if {[string compare $q_sim_tool "<None>"]} { 
	set q_sim_lang [ string tolower [lindex [ get_global_assignment -name {EDA_OUTPUT_DATA_FORMAT} -section_id eda_simulation ] 0 ] ]
	set q_sim_output_file "${rev_name}\.[get_eda_writer_netlist_ext $q_sim_lang]" 
	set q_sim_sdf_file    "$rev_name\_[get_eda_writer_sdf_ext $q_sim_lang]"
	#return -code $ERROR_UNKNOWN_NETLIST_TYPE
    }
    return;
}

#######################################################################
##
proc ::quartus::nativelinkflow::sim::run_eda_simulation_tool {
	rtl_sim
} {
##
#######################################################################

    set status 0
    variable INFO_INIT_SIM_SETTINGS_SUCCESS 

    variable ERROR_INIT_SIM_SETTINGS 
    variable ERROR_UNKNOWN_NETLIST_TYPE_STRING
    variable ERROR_ABSENT_SIM_DIR
    variable ERROR_OUTPUT_NETLISTER_NOT_RUN
    variable ERROR_NL_FOR_SIMTOOL_NOT_SUPPORTED

    variable q_sim_tool        
    variable q_sim_lang        
    variable q_sim_output_file 
    variable q_sim_sdf_file    
    variable q_sim_dir         
    variable q_vhd_version

    namespace import ::quartus::nativelinkflow::get_project_dir

    set part_info [ get_nativelink_info "qeda_simulation.dat"]						
    if {$part_info == ""} {
	  nl_postmsg error "Error: Family not supported for NativeLink Simulation"
	  error "Family not supported"
    }
    set family_name [string tolower [lindex $part_info 0]]
    if [catch { set_q_sim_environ } result] {
	nl_postmsg error "$ERROR_INIT_SIM_SETTINGS : $result"
	return 1
    }

    if {$q_sim_tool == ""} {
	nl_postmsg error "$ERROR_NL_FOR_SIMTOOL_NOT_SUPPORTED $q_sim_tool"
	set ::errorCode 1
	set ::errorInfo "$ERROR_NL_FOR_SIMTOOL_NOT_SUPPORTED $q_sim_tool"
	return 1
    }


    if {$rtl_sim == 1} {

        # if NativeLink EDA Synthesis is enabled, then disable NativeLink RTL simulation.
        set nl_syn [get_global_assignment -name EDA_RUN_TOOL_AUTOMATICALLY -section_id eda_design_synthesis]

	if {[string compare -nocase $nl_syn "ON"] == 0} {
		nl_postmsg error "Error: RTL Simulation using NativeLink is not supported when EDA Synthesis using NativeLink is enabled"
		error "" "EDA RTL Simulation using NativeLink is not supported wwhen EDA Synthesis using NativeLink is enabled"
	}
	    
	set sim_mode  "RTL"
	#if design uses mixed language, error out
	set verilog_files [get_unencrypted_hdl_files "verilog"]
	set vhdl_files [get_unencrypted_hdl_files "vhdl"]

	set file_list ""
	if {$q_sim_lang == "vhdl"} {
	    set file_list $vhdl_files
	} else {
	    set file_list $verilog_files
	}
	#Remove files ignored due to IPFS flow from the list of language files
	set ignored_files ""
	foreach file $file_list {
	    set full_file_name [get_file_info -info full_path -filename $file]
	    array set list [get_ip_info $full_file_name]
	    if {([array names list related_files] != "" ) && ($list(related_files) != "")} {
		# All related_files to ignored_files list
		set ignored_files "$ignored_files [split $list(related_files) ,]"
	        set ix [lsearch -exact $ignored_files $file]
	    	if {$ix >= 0} {
		    set ignored_files [lreplace $ignored_files $ix $ix]
		}
	    }
	}
	foreach ignored_file $ignored_files {
	    set ix [lsearch -exact $verilog_files $ignored_file] 
	    if {$ix >= 0} {
		set verilog_files [lreplace $verilog_files $ix $ix]
	    }
	    set ix [lsearch -exact $vhdl_files ignored_file] 
	    if {$ix >= 0} {
		set vhdl_files [lreplace $vhdl_files $ix $ix]
	    }
	}

       #if one of verilog_files and vhdl_files is null, we will switch formats here.
       #This would ensure that we do not use mixex mode simulation

	if ![qmap_successfully_completed] {
	    nl_postmsg error "Error: Run Analysis and Elaboration successfully before starting RTL NativeLink Simulation"
	    error "" "Analysis and Synthesis should be completed successfully before starting RTL NativeLink Simulation"
	} elseif {($vhdl_files == "") && ($verilog_files == "")} {
	    nl_postmsg error "Error: NativeLink did not detect any HDL files in the project"
	    error "" "NativeLink did not detect any HDL files in the project"
	} elseif {$verilog_files == ""} {
	    nl_postmsg info "Info: NativeLink has detected VHDL design -- VHDL simulation models will be used"
	    set q_sim_lang "vhdl"
	} elseif {$vhdl_files == ""} {
	    nl_postmsg info "Info: NativeLink has detected Verilog design -- Verilog simulation models will be used"
	    set q_sim_lang "verilog"
	} else {
	    nl_postmsg info "Info: NativeLink has detected a mixed Verilog and VHDL design -- $q_sim_lang simulation models will be used"
	}

    } else {
	set sim_mode  "Gate"
    }
    nl_logmsg "\n========= EDA Simulation Settings =====================\n"
    nl_logmsg "Sim Mode              :  $sim_mode"
    nl_logmsg "Family                :  $family_name"
    nl_logmsg "Quartus root          :  $::quartus(binpath)"
    nl_logmsg "Quartus sim root      :  $::quartus(eda_libpath)sim_lib"
    nl_logmsg "Simulation Tool       :  $q_sim_tool"
    nl_logmsg "Simulation Language   :  $q_sim_lang"
    if [regexp -nocase VHDL $q_sim_lang] {
	if [regexp -nocase OFF $q_vhd_version] {
	    nl_logmsg "Version               :  93"
	} else {
	    nl_logmsg "Version               :  87"
	}
    }
    nl_logmsg "Sim Output File       :  $q_sim_output_file"
    nl_logmsg "Sim SDF file          :  $q_sim_sdf_file"
    nl_logmsg "Sim dir               :  $q_sim_dir"
    nl_logmsg "\n=======================================================\n"

    #save original directory
    set orig_dir [pwd]
    if {[catch {goto_sim_dir $q_sim_tool} result]} {
	set status 1
	return 1
    }

    set rev_name [get_project_settings -cmp]
    if {[catch { 
	    namespace import $q_sim_tool\::* ; 
	    set status [run_$q_sim_tool $q_sim_lang $rev_name $q_vhd_version $family_name $q_sim_output_file $sim_mode]
    } ] } {
	cd $orig_dir
	set status 1
    } else {
	cd $orig_dir
    }
    return $status
}


#######################################################################
##
proc ::quartus::nativelinkflow::sim::modelsim::run_modelsim {
	language 
        cap      
	vhd_ver 
	family 
	output_file
	sim_mode
} {
##
#######################################################################
    nl_postmsg info "Info: Starting NativeLink simulation with ModelSim software"

    set status 0
    variable ::quartus

    set nativelink_tool_script "$quartus(nativelink_tclpath)modelsim.tcl"

    set pnsp [namespace parent]

    namespace import ${pnsp}::INFO* ${pnsp}::ERROR*
    foreach x [concat [info vars ${pnsp}::INFO*] [info vars ${pnsp}::ERROR*]] {
	variable $x
    }


    if { [ file exists $nativelink_tool_script ] == 0} {
	nl_postmsg error "$ERROR_ABSENT_NL_SCRIPT $nativelink_tool_script !"
    } else {
	if { [ catch { source $nativelink_tool_script } result ]  } {
	    nl_postmsg error "$ERROR_SOURCE_NL_SCRIPT $result"
	} else {
	    nl_logmsg "Sourced tool specific script -- $nativelink_tool_script"
	}
    }

    namespace import ${pnsp}::*

    set launch_args "$language"
    #Compile Simulation Altera libraries for Modelsim SE
    lappend launch_args "1"

    if {[string compare -nocase $sim_mode "rtl"] == 0} {
	lappend launch_args "1"
    } else {
        # Check if [get_ exists otherwise error out
	set cap [get_project_settings -cmp]
    	set ext [get_eda_writer_netlist_ext $language]
    	if ![file exists ${cap}.${ext} ] {
	    nl_postmsg error "Error: Gate Level Simulation Netlist not found -- run EDA NetList Writer to generate Gate Level simulation netlist"
	    set status 1
	}
	lappend launch_args "0" 
    }

    if {$status == 0} {
	if [catch {eval launch_modelsim $launch_args} status ] {
	    nl_postmsg error "Error: NativeLink Simulation failed"
	    set status 1
	} 
    }
    if {$status == 0} {
	nl_postmsg info "Info: NativeLink Simulation succeded"
    }
    return $status
}

#######################################################################
##
proc ::quartus::nativelinkflow::sim::modelsim-altera::run_modelsim-altera {
	language 
        cap      
	vhd_ver 
	family 
	output_file
	sim_mode
} {
##
#######################################################################
    nl_postmsg info "Info: Starting NativeLink simulation with ModelSim-Altera software"

    set status 0
    variable ::quartus
    set nativelink_tool_script "$quartus(nativelink_tclpath)modelsim.tcl"

    set pnsp [namespace parent]
    namespace import ${pnsp}::INFO* ${pnsp}::ERROR*

    foreach x [concat [info vars ${pnsp}::INFO*] [info vars ${pnsp}::ERROR*] ] {
	variable $x
    }

    if { [ file exists $nativelink_tool_script ] == 0} {
	nl_postmsg error "$ERROR_ABSENT_NL_SCRIPT $nativelink_tool_script!"
    } else {
	if { [ catch { source $nativelink_tool_script } result ]  } {
	    nl_postmsg error "$ERROR_SOURCE_NL_SCRIPT $result"
	} else {
	    nl_logmsg "Sourced tool specific script -- $nativelink_tool_script"
	}
    }

    namespace import ${pnsp}::*

    set launch_args "$language"
    
    #Do not compile Altera Simulation libraries for Modelsim-Altera
    lappend launch_args "0"

    if {[string compare -nocase $sim_mode "rtl"] == 0} {
	lappend launch_args "1"
    } else {
	set cap [get_project_settings -cmp]
    	set ext [get_eda_writer_netlist_ext $language]
    	if ![file exists ${cap}.${ext} ] {
	    nl_postmsg error "Error: Gate Level Simulation Netlist not found -- run EDA NetList Writer to generate Gate Level simulation netlist"
	    set status 1
	}
	lappend launch_args "0" 
    }

    if {$status == 0} {
	if [catch {eval launch_modelsim $launch_args} status ] {
	    nl_postmsg error "Error: NativeLink Simulation failed"
	    set status 1
	}
    }
    if {$status == 0} {
	nl_postmsg info "Info: NativeLink Simulation succeded"
    }
    return $status
}

#######################################################################
##
proc ::quartus::nativelinkflow::sim::ncsim::run_ncsim {
	language 
        cap      
	vhd_ver 
	family 
	output_file
	sim_mode
} {
##
#######################################################################
    nl_postmsg info "Info: Starting NativeLink simulation with NcSim software"

    set status 0
    variable ::quartus
    set nativelink_tool_script "$quartus(nativelink_tclpath)ncsim.tcl"

    set pnsp [ namespace parent ]
    namespace import ${pnsp}::INFO* ${pnsp}::ERROR*

    foreach x [concat [info vars ${pnsp}::INFO*] [info vars ${pnsp}::ERROR*]] {
	variable $x
    }

    if { [ file exists $nativelink_tool_script ] == 0} {
	nl_postmsg error "$ERROR_ABSENT_NL_SCRIPT $nativelink_tool_script !"
    } else {
	if { [ catch { source $nativelink_tool_script } result ]  } {
	    nl_postmsg error "$ERROR_SOURCE_NL_SCRIPT $result"
	} else {
	    nl_logmsg "Sourced NativeLink script -- $nativelink_tool_script"
	}
    }
    
    namespace import ${pnsp}::*  

    set launch_args "$language"
    
    if {[string compare -nocase $sim_mode "rtl"] == 0} {
	lappend launch_args "1"
    } else {
	# Check if [get_ exists otherwise error out
	set cap [get_project_settings -cmp]
    	set ext [get_eda_writer_netlist_ext $language]
    	if ![file exists ${cap}.${ext} ] {
	    nl_postmsg error "Error: Gate Level Simulation Netlist not found -- run EDA NetList Writer to generate Gate Level simulation netlist"
	    set status 1
	}
	lappend launch_args "0"
    }

    if {$status == 0}  {
	if [catch {eval launch_ncsim $launch_args } status ] {
	    nl_postmsg error "Error: NativeLink Simulation failed"
	    set status 1
	}
    }
    if {$status == 0}  {
	nl_postmsg info "Info: NativeLink Simulation succeded"
    }
    return $status
}

#######################################################################
##
proc ::quartus::nativelinkflow::sim::scsim::run_scsim {
	language 
        cap      
	vhd_ver 
	family 
	output_file
	sim_mode
} {
##
#######################################################################
    nl_postmsg info "Info: Starting NativeLink simulation with VCS MX software"

    set status 0
    variable ::quartus
    set nativelink_tool_script "$quartus(nativelink_tclpath)vcs_mx.tcl"

    set pnsp [namespace parent]
    namespace import ${pnsp}::INFO* ${pnsp}::ERROR*

    foreach x [concat [info vars ${pnsp}::INFO*] [info vars ${pnsp}::ERROR*]] {
	variable $x
    }

    if { [ file exists $nativelink_tool_script ] == 0} {
	nl_postmsg error "$ERROR_ABSENT_NL_SCRIPT $nativelink_tool_script !"
    } else {
	if { [ catch { source $nativelink_tool_script } result ]  } {
	    nl_postmsg error "$ERROR_SOURCE_NL_SCRIPT $result"
	} else {
	    nl_logmsg "Sourced tool specific script -- $nativelink_tool_script"
	}
    }
    
    namespace import ${pnsp}::*  

    set launch_args "$language"

    if {[string compare -nocase $sim_mode "rtl"] == 0} {
	lappend launch_args "1"
    } else {
	# Check if [get_ exists otherwise error out
	set cap [get_project_settings -cmp]
    	set ext [get_eda_writer_netlist_ext $language]
    	if ![file exists ${cap}.${ext} ] {
	    nl_postmsg error "Error: Gate Level Simulation Netlist not found -- run EDA NetList Writer to generate Gate Level simulation netlist"
	    set status 1
	}
	lappend launch_args "0"
    }

    if {$status == 0}  {
	if [catch {eval launch_vcsmx $launch_args } status ] {
	    nl_postmsg error "Error: NativeLink Simulation failed"
	    set status 1
	}
    }
    if {$status == 0} {
	nl_postmsg info "Info: NativeLink Simulation succeded"
    }
    return $status
}

#######################################################################
##
proc ::quartus::nativelinkflow::sim::vcs::run_vcs {
	language 
        cap      
	vhd_ver 
	family 
	output_file
	sim_mode
} {
##
#######################################################################
    nl_postmsg info "Info: Starting NativeLink simulation with VCS software"

    set status 0
    variable ::quartus

    set nativelink_tool_script "$quartus(nativelink_tclpath)vcs.tcl"

    set pnsp [ namespace parent ]
    namespace import ${pnsp}::INFO* ${pnsp}::ERROR*

    foreach x [concat [info vars ${pnsp}::INFO*] [info vars ${pnsp}::ERROR*]] {
	variable $x
    }

    if { [ file exists $nativelink_tool_script ] == 0} {
	nl_postmsg error "$ERROR_ABSENT_NL_SCRIPT $nativelink_tool_script !"
    } else {
	if { [ catch { source $nativelink_tool_script } result ]  } {
	    nl_postmsg error "$ERROR_SOURCE_NL_SCRIPT $result"
	} else {
	    nl_logmsg "Sourced tool specific script -- $nativelink_tool_script"
	}
    }

    namespace import ${pnsp}::*  

    set launch_args "$language"
    if {[string compare -nocase $sim_mode "rtl"] == 0} {
	lappend launch_args "1"
    } else {
	# Check if [get_ exists otherwise error out
	set cap [get_project_settings -cmp]
    	set ext [get_eda_writer_netlist_ext $language]
    	if ![file exists ${cap}.${ext} ] {
	    nl_postmsg error "Error: Gate Level Simulation Netlist not found -- run EDA NetList Writer to generate Gate Level simulation netlist"
	    set status 1
	}
	lappend launch_args "0"
    }

    if {$status == 0}  {
	if [catch {eval launch_vcs $launch_args } status ] {
	    nl_postmsg error "Error: NativeLink Simulation failed"
	    set status 1
	}
    }
    if {$status == 0}  {
	nl_postmsg info "Info: NativeLink Simulation succeded"
    }
    return $status
}

############################################################################
##
##
proc ::quartus::nativelinkflow::sim::is_timing_simulation_on {} {
##
############################################################################
    set result 0 
    set timing_sim [ get_global_assignment -name {EDA_GENERATE_FUNCTIONAL_NETLIST} -section_id eda_simulation ]
    if {$timing_sim ne "ON"} {
	set result 1 
    }
    
    return $result
}
