# ***************************************************************
# ***************************************************************
#
# File:         ncsim_verilog.tcl
# Description:  Quartus Nativelink Simulation flow
#               This script is used by Quartus to launch NcSim
#               tool for Verilog simulation
#
# Version:      1.0
#
# Authors:      Altera Corporation
#
#               Copyright (c)  Altera Corporation 2003 - .
#               All rights reserved.
#
# ***************************************************************
# ***************************************************************

proc ::quartus::nativelinkflow::sim::compile_source_file {lib file} {
    set status 0
    set ncsim_cmd ""
    if {[get_file_type $file] == "verilog"}  {
	set ncvlog_cmd [resolve_tool_path "sim" "ncvlog"]
        if {$ncvlog_cmd == ""} {
	    nl_postmsg error "Error: Can't launch the NcSim software. Path to the location of the executables for the NcSim software is not specified"
	    nl_postmsg error "Error: You can specify the path in the EDA Tool Options page of the Options dialog box or using the Tcl command set_user_option."
	   set status 1
	}
	set ncsim_cmd "\"$ncvlog_cmd\" -nocopyright -nowarn -messages -append_log"
    } else {
	set ncvhdl_cmd [resolve_tool_path "sim" "ncvhdl"]
        if {$ncvhdl_cmd == ""} {
	    nl_postmsg error "Error: Can't launch the NcSim software. Path to the location of the executables for the Ncsim software is not specified"
	    nl_postmsg error "Error: You can specify the path in the EDA Tool Options page of the Options dialog box or using the Tcl command set_user_option."
	   set status 1
	}
	set ncsim_cmd "\"$ncvhdl_cmd\" -v93 -nocopyright -nowarn -messages -append_log"
    }

    if {$status == 0} {
	if {$lib == ""} {
	    set lib "work"
	}

	#We should check if cds.lib contains definition of library $lib, 
	#if not then we should add one...
	if {![file exists $file]} {
	    nl_postmsg error "Error: Can't find file $file" 
	    set status 1
	} else {
	    nl_postmsg info "Info: Compiling file $file to library $lib"
	    nl_logmsg "Running command $ncsim_cmd -work $lib $file"
	    if [catch {eval exec $ncsim_cmd -work $lib $file} result] {
		nl_postmsg error "Error: Compilation of file $file was NOT successful" 
		foreach msg_line [split $result \n] {
		    if {$msg_line != ""} {
			nl_postmsg error "NcSim: $msg_line"
		    }
		}
		set status 1
	    } else {
		nl_postmsg info "Info: Compilation file $file was successful"
	    }
	}
    }
    return $status
}

proc ::quartus::nativelinkflow::sim::launch_ncsim {args} {
    set status 0
    namespace import ::quartus::nativelinkflow::get_sim_models_root_path
    namespace import ::quartus::nativelinkflow::create_work_dir
    namespace import ::quartus::nativelinkflow::resolve_tool_path
    #if {![file exists ncsim_work]} {
	#nl_postmsg info "Info: Creating directory ncsim_work for Verilog simulation"
	#create_work_dir ncsim_work
    #} else {
	    #nl_postmsg info "Info: Directory ncsim_work already exists"
    #}
    set lib_path [get_sim_models_root_path]
    set cap [get_project_settings -cmp]
    set lang [lindex $args 0]
    set rtl_sim [lindex $args 1]
    set tb_mode [lindex $args 2]
    set tb_args [lrange $args 4 end]
    set sdo_cmd_file ""

    set ncsim_cmd [resolve_tool_path "sim" "ncsim"]
    set ncelab_cmd [resolve_tool_path "sim" "ncelab"]
    set nclaunch_cmd [resolve_tool_path "sim" "nclaunch"]
    if {($ncsim_cmd == "" ) || ($ncelab_cmd == "") || ($nclaunch_cmd == "")} {
	    nl_postmsg error "Error: Can't launch the NcSim software. Path to the location of the executables for the NcSim software is not specified"
	nl_postmsg error "Error: You can specify the path in the EDA Tool Options page of the Options dialog box or using the Tcl command set_user_option."
	set status 1
    }

    if {$status == 0} {
	set lib_base_dir "vhdl_libs"
	if {$lang == "verilog"} {
	    set lib_base_dir "verilog_libs"
	} else {
	    set lib_base_dir "vhdl_libs"
	}
	if ![file isdirectory $lib_base_dir] {
	    file mkdir $lib_base_dir
	}

	# create cds.lib config file
	if [file exists cds.lib] {
	    file delete -force cds.lib
	}
	
	if [file exists hdl.var] {
	    file delete -force hdl.var
	}

	if {![file exists cds.lib] } {
	    set cdsfile "cds.lib"
	    set cdslib [open $cdsfile w+]
	    puts $cdslib "SOFTINCLUDE \$\{CDS_INST_DIR\}/tools/inca/files/cdsvhdl.lib"
	    close $cdslib
	    nl_postmsg info "Info: Created cds.lib"
	} else {
	    nl_postmsg info "Info: cds.lib already exists"
	}
    
	# create hdl.var config file
	if {![file exists hdl.var]} {
	    set hdlfile "hdl.var"
	    set hdlvar [open $hdlfile w+]
	    puts $hdlvar "SOFTINCLUDE \$\{CDS_INST_DIR\}/tools/inca/files/hdlvlog.var"
	    puts $hdlvar "DEFINE LIB_MAP ( + => work )"
	    puts $hdlvar   "DEFINE NCSIMRC         ( \$\{CDS_INST_DIR\}/tools/inca/files/ncsimrc, \~/.ncsimrc )"
	    puts $hdlvar "DEFINE VERILOG_SUFFIX (.v, .vt, .vlg, .vo)"
	    puts $hdlvar "DEFINE VHDL_SUFFIX (.vhd, .vht, .vhdl, .vho)"
	    close $hdlvar
	    nl_postmsg info "Info: Created hdl.var"
	} else {
	    nl_postmsg info "Info: hdl.var already exists"
	}

	set libs [get_sim_models_required $lang $rtl_sim]
	foreach lib_info $libs {
	    set lib_name [lindex $lib_info 0]
	    set lib_sources [lindex $lib_info 1]
	    set lib_dir "$lib_base_dir/$lib_name"

	    if ![file isdirectory $lib_dir] {
		nl_postmsg info "Info: Creating directory $lib_dir"
		create_work_dir $lib_dir
	    }

	    set cdslib [open $cdsfile a+]
	    puts $cdslib "DEFINE $lib_name ./$lib_dir"
	    close $cdslib
	    foreach source $lib_sources {
		if {$status == 0} {
		    set status [compile_source_file $lib_name $source]
		}
	    }
	} 
    }
    
    if { $status == 0} {
	set design_files_info [get_design_files $lang $rtl_sim]
	set work_lib "gate_work"
	if {$rtl_sim == "1"} {
	    set work_lib "rtl_work"
	} else {
	    set work_lib "gate_work"
	}
	if ![file isdirectory $work_lib] {
	    file mkdir $work_lib
	}
	set cdslib [open $cdsfile a+]
	puts $cdslib "DEFINE $work_lib ./$work_lib"
	close $cdslib

	foreach design_file_info $design_files_info {
	    set lib_name [lindex $design_file_info 0]
	    set lib_sources [lindex $design_file_info 1]
	    foreach source $lib_sources {
		if {$status == 0} {
		    set status [compile_source_file $lib_name $source]
		}
	    }
	}
    }

    if {$status == 0} {
	set tb_mode [get_testbench_mode "gate"]
	if {$tb_mode == "testbench" } {
	    set tb_info [get_testbench_info]
	    set tb_files [lindex $tb_info 3]
	    foreach file $tb_files {
		if {$status == 0} {
		    set status [compile_source_file $lib_name $file]
		}
	    }
	    if {$status == 0} {
		if {($rtl_sim == 0) && ($lang == "vhdl")} {
		    if [is_timing_simulation_on ] {
			set sdo_file "${cap}_vhd.sdo"
			nl_postmsg info "Info: Compiling SDO file $sdo_file"
			nl_logmsg "Running command ncsdfc $sdo_file"
			if [catch {exec ncsdfc $sdo_file } result] {
			    nl_postmsg error "Error: Compilation of design file $src was NOT successful"
			    foreach msg_line [split $result \n] {
				if {$msg_line != ""} {
				    nl_postmsg error "NcSim: $msg_line"
				}
			    }
			    set status 1
			} else {
			    nl_postmsg info "Info: Compilation of SDO file $sdo_file was successful"
			    set tb_design_inst_name [lindex $tb_info 1]
			    #generate_sdf_command_file
			    set sdo_cmd_file "sdf_cmd_file"
			    set sdo_cmd_file_id [open $sdo_cmd_file w]
			    puts $sdo_cmd_file_id "COMPILED_SDF_FILE = \"${sdo_file}.X\","
			    puts $sdo_cmd_file_id "SCOPE = \:$tb_design_inst_name,"
			    puts $sdo_cmd_file_id "MTM_CONTROL = \"TYPICAL\","
			    puts $sdo_cmd_file_id "SCALE_FACTORS = \"1.0:1.0:1.0\","
			    puts $sdo_cmd_file_id "SCALE_TYPE = \"FROM_MTM\";"
			    close $sdo_cmd_file_id
			}
		    }
		}
	    }
	    

	    if {$status == 0} {
		#Open up access to all objects/signals, this may be may ncsim slower, however we do not know which signals the user wants to view.
		set access_file_name "$cap\_acess.af"
		set access_file_id [open $access_file_name w]
		puts $access_file_id "PATH ... +rwc"
		#Ideally we do not want to expose everything, however some versions of 
		#ncsim have bugs and do not expose signals, hence we add everything to 
		#access list
		#puts $access_file_id "PATH $testbench_module_name.<WIRE> +rwc"
		#puts $access_file_id "PATH $testbench_module_name.* +rwc"
		close $access_file_id
		
		set testbench_module_name  [lindex $tb_info 0]
		set ncelab_cmd "\"$ncelab_cmd\" -timescale 1ps/1ps"

		if {$sdo_cmd_file != ""} {
		    set ncelab_cmd "$ncelab_cmd -SDF_CMD_FILE $sdo_cmd_file"
		}

		if {($rtl_sim == 0) && ([is_glitch_filter_enabled] == "1")} {
		    set ncelab_cmd "$ncelab_cmd -pulse_r 0 -pulse_e 0 -intermod_path"
		}
		set ncelab_cmd "$ncelab_cmd -work $work_lib -afile $access_file_name $work_lib.$testbench_module_name"
		nl_logmsg "Running command $ncelab_cmd"
		if [ catch {eval exec $ncelab_cmd} result] {
		    nl_postmsg error "Error: Elaboration of design $work_lib.$testbench_module_name was NOT successful"
		    foreach msg_line [split $result \n] {
			if {$msg_line != ""} {
			    nl_postmsg error "NcSim: $msg_line"
			}
		    }
		    set status 1
		} else {
		    nl_postmsg info "Info: Elaboration of design $work_lib.$testbench_module_name was successful"
		}	
	    }

	    if {$status == 0} {
	        set sim_setup_script [lindex $tb_info 4]
		if {$sim_setup_script == ""} {
		    set ncsim_script_file "$cap\_run.tcl"
		    set ncsim_script_file_id [open $ncsim_script_file w]
		    if {($rtl_sim == 0) && ([is_vcd_generation_enabled] == "1")} {
			puts $ncsim_script_file_id "source ${cap}_dump_all_vcd_nodes.tcl"
		    }
		    puts $ncsim_script_file_id "run [lindex $tb_info 2]"
		    close $ncsim_script_file_id
		} else {
		    set ncsim_script_file $sim_setup_script
		}

		nl_logmsg "Running command $ncsim_cmd -gui -input $ncsim_script_file $testbench_module_name"
		if [ catch {exec "$ncsim_cmd" -gui -input $ncsim_script_file $testbench_module_name } result] {
		    nl_postmsg error "Error: Simulation of design $testbench_module_name was NOT successful"
		    foreach msg_line [split $result \n] {
			if {$msg_line != ""} {
			    nl_postmsg error "NcSim: $msg_line"
			}
		    }
		    set status 1
		}
	    }

	} else {
	    if {$status == 0} {
		nl_logmsg "Running $nclaunch_cmd"
		exec "$nclaunch_cmd" &
	    }
	}
    }
    return $status
}
