## ***************************************************************
## ***************************************************************
##
## File:         modelsim_verilog.tcl
## Description:  Quartus Nativelink Simulation flow
##               This script is used by Quartus to launch ModelSim 
##               tool for Verilog simulation
##
##  ALTERA LEGAL NOTICE
##  
##  This script is  pursuant to the following license agreement
##  (BY VIEWING AND USING THIS SCRIPT, YOU AGREE TO THE
##  FOLLOWING): Copyright (c) 2006-2007 Altera Corporation, San Jose,
##  California, USA.  Permission is hereby granted, free of
##  charge, to any person obtaining a copy of this software and
##  associated documentation files (the "Software"), to deal in
##  the Software without restriction, including without limitation
##  the rights to use, copy, modify, merge, publish, distribute,
##  sublicense, and/or sell copies of the Software, and to permit
##  persons to whom the Software is furnished to do so, subject to
##  the following conditions:
##  
##  The above copyright notice and this permission notice shall be
##  included in all copies or substantial portions of the Software.
##  
##  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
##  EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES
##  OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
##  NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT
##  HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY,
##  WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
##  FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
##  OTHER DEALINGS IN THE SOFTWARE.
##  
##  This agreement shall be governed in all respects by the laws of
##  the State of California and by the laws of the United States of
##  America.
##
##  CONTACTING ALTERA
##  
##  You can contact Altera through one of the following ways:
##  
##  Mail:
##     Altera Corporation
##     Applications Department
##     101 Innovation Drive
##     San Jose, CA 95134
##  
##  Altera Website:
##     www.altera.com
##  
##  Online Support:
##     www.altera.com/mysupport
##     
##  Troubshooters Website:
##     www.altera.com/support/kdb/troubleshooter
##  
##  Technical Support Hotline:
##     (800) 800-EPLD or (800) 800-3753
##        7:00 a.m. to 5:00 p.m. Pacific Time, M-F 
##     (408) 544-7000
##        7:00 a.m. to 5:00 p.m. Pacific Time, M-F 
##  
##     From other locations, call (408) 544-7000 or your local
##     Altera distributor.
##  
##  The mySupport web site allows you to submit technical service
##  requests and to monitor the status of all of your requests
##  online, regardless of whether they were submitted via the
##  mySupport web site or the Technical Support Hotline. In order to
##  use the mySupport web site, you must first register for an
##  Altera.com account on the mySupport web site.
##  
##  The Troubleshooters web site provides interactive tools to
##  troubleshoot and solve common technical problems.
##
## ***************************************************************
## ***************************************************************

#This function creates the modelsim script <revision_name>_run_msim.do
#
#
#
proc ::quartus::nativelinkflow::sim::gen_msim_script {args} {
    set status 0
    namespace import ::quartus::nativelinkflow::get_sim_models_root_path
    set lib_path [get_sim_models_root_path]
    set cap [get_project_settings -cmp]
    set lang  [lindex $args 0] 
    set compile_libs [lindex $args 1]
    set rtl_sim [lindex $args 2]
    set vsim_cmd "vsim -t 1ps"
    #language comes as an argument..
    set process_sgate_lib 1

    if {$rtl_sim == 1} {
	set sim_mode "rtl"
    } else {
	set sim_mode "gate"
    }
    set quartus_version $::quartus(version);
    set msim_do_file_name "$cap\_run_msim_${sim_mode}_${lang}.do"
    if [ file exists $msim_do_file_name ] {
	backup_file $msim_do_file_name
    }

    if [ catch { open $msim_do_file_name w } msim_do_file ] {
	nl_postmsg  error "Error: Can't open file $msim_do_file_name: $file_id"
	set msim_do_file_name ""
    } else { 

	puts $msim_do_file "transcript on"

	set libs [get_sim_models_required $lang $rtl_sim]
	if {$compile_libs == "1"} {
	    # Compile libraries for SE version

	    if {$lang == "verilog"} {
		puts $msim_do_file "if !\[file isdirectory verilog_libs\] {"
		puts $msim_do_file "\tfile mkdir verilog_libs"
		puts $msim_do_file "}\n"
	    } else {
		puts $msim_do_file "if !\[file isdirectory vhdl_libs\] {"
		puts $msim_do_file "\tfile mkdir vhdl_libs"
		puts $msim_do_file "}\n"
	    }

	    foreach lib_info $libs {
		set lib_name [lindex $lib_info 0]
		set lib_sources [lindex $lib_info 1]
		if [regexp "_ver$" $lib_name] {
		    #verilog library
		    puts $msim_do_file "vlib verilog_libs/$lib_name"
		    puts $msim_do_file "vmap $lib_name verilog_libs/$lib_name"
		    foreach source_file $lib_sources  {
			puts $msim_do_file "vlog -work $lib_name $source_file"
		    }
		    puts $msim_do_file ""
		    set vsim_cmd "$vsim_cmd -L $lib_name"
		} else {
		    #vhdl library
		    puts $msim_do_file "vlib vhdl_libs/$lib_name"
		    puts $msim_do_file "vmap $lib_name vhdl_libs/$lib_name"
		    foreach source_file $lib_sources  {
			puts $msim_do_file "vcom -work $lib_name $source_file"
		    }
		    puts $msim_do_file ""
		    set vsim_cmd "$vsim_cmd -L $lib_name"
		}
	    }

	} else {
	    foreach lib_info $libs {
		set lib_name [lindex $lib_info 0]
		    set vsim_cmd "$vsim_cmd -L $lib_name"
	    }
	}
	set design_files_info [get_design_files $lang $rtl_sim]

	if {$rtl_sim == "1"} {
	    puts $msim_do_file "if {\[file exists rtl_work\]} {"
	    puts $msim_do_file "\tvdel -lib rtl_work -all"
	    puts $msim_do_file "}"

	    puts $msim_do_file "vlib rtl_work"
	    puts $msim_do_file "vmap work rtl_work\n"
	} else {
	    puts $msim_do_file "if {\[file exists gate_work\]} {"
	    puts $msim_do_file "\tvdel -lib gate_work -all"
	    puts $msim_do_file "}"

	    puts $msim_do_file "vlib gate_work"
	    puts $msim_do_file "vmap work gate_work\n"
	}

        foreach design_file_info $design_files_info {
	    set lib_name [lindex $design_file_info 0]
	    set lib_sources [lindex $design_file_info 1]
            if {($lib_name != "rtl_work") && ($lib_name != "gate_work") } {
		puts $msim_do_file "vlib $lib_name"
		puts $msim_do_file "vmap $lib_name $lib_name"
	    }
	    foreach source_file $lib_sources  {
		if {[get_file_type $source_file] == "verilog" } {
		    puts $msim_do_file "vlog -work $lib_name $source_file"
		} else {
		    puts $msim_do_file "vcom -work $lib_name $source_file"
		}
	    }
	}
	puts $msim_do_file ""

	if {$rtl_sim == "1"} {
	    set lib_name "rtl_work"
	} else {
	    set lib_name "gate_work"
	}
	set tb_mode [get_testbench_mode "gate"]
	switch -regexp -- $tb_mode {
	    (?i)testbench
	    {
		set tb_info [get_testbench_info]
	        if {$tb_info == ""} {
		    nl_postmsg error "Error: No Simulation Test Bench specified -- cannot continue NativeLink simulation"
		    set status 1
		} else {
		    set testbench_module_name  [lindex $tb_info 0]
		    if {$testbench_module_name == ""} {
			nl_postmsg error "Error: Top level entity/module name for test bench not specified -- cannot continue NativeLink simulation"
			set status 1
		    }
		    set tb_files [lindex $tb_info 3]
		    if {$tb_files == ""} {
			nl_postmsg error "Error: Test Bench does not contain any stimulus files -- cannot continue NativeLink simulation"
			set status 1
		    }
		}
		if {$status == 0} {
		    foreach file $tb_files {
			if {[get_file_type $file] == "verilog" } {
			    puts $msim_do_file "vlog -work $lib_name $file"
			} else {
			    puts $msim_do_file "vcom -work $lib_name $file"
			}
		    }

		    if {$rtl_sim == "0"} {
			if {[is_glitch_filter_enabled] == "1"} {
			    set vsim_cmd "$vsim_cmd +transport_int_delays +transport_path_delays"
			}
			if {$lang == "vhdl"} {
			    if [is_timing_simulation_on] {
				set sdo_file "$cap\_vhd.sdo"
				set tb_design_inst_name [lindex $tb_info 1]
				if {$tb_design_inst_name == ""} {
				    nl_postmsg critical_warning "Test Bench instance name not speficied -- unable to apply timing file $sdo_file -- functional simulation will be performed"
				} else {
				    set vsim_cmd "$vsim_cmd -sdftyp /$tb_design_inst_name=$sdo_file"
				}
			    }
			}
		    }
		
		    set vsim_cmd "$vsim_cmd work.$testbench_module_name"
		    puts $msim_do_file "\n$vsim_cmd\n"

		    if {($rtl_sim == 0) && ([is_vcd_generation_enabled] == "1")} {
			set vcd_script_file "${cap}_dump_all_vcd_nodes.tcl"
			puts $msim_do_file "source $vcd_script_file"
		    }

		    set sim_setup_script [lindex $tb_info 4]
		    if {$sim_setup_script == "" } {
			puts $msim_do_file "add wave *"
			puts $msim_do_file "view structure"
			puts $msim_do_file "view signals"
			set run_time [lindex $tb_info 2]
			set rt_value [lindex $run_time 0]
			set rt_units [lindex $run_time 1]
			if {$rt_units == "s"} {
			    set rt_units "sec"
			}
			puts $msim_do_file "run $rt_value $rt_units"
		    } else {
			puts $msim_do_file "do $sim_setup_script"
		    }
		}
	    }
	    (?i)script
	    {
		puts $msim_do_file "do [get_command_script gate]"
	    }
	}
	close $msim_do_file
    }
    
    if {$status != 0 } {
	set msim_do_file_name ""
    }
    return $msim_do_file_name
}

#This function launches the modelsim executable
proc ::quartus::nativelinkflow::sim::launch_modelsim {args} {
    set err_cnt 0
    set script_args ""
    set msim_do_file_name ""
    namespace import ::quartus::nativelinkflow::resolve_tool_path
    set vsim_cmd [resolve_tool_path "sim" "vsim"]
    set msim_cmd [resolve_tool_path "sim" "modelsim"]
    if { [string compare -nocase $::tcl_platform(platform) "windows"] == 0} {
	set msim_exe "\"$msim_cmd\""
    } else {
	set msim_exe "\"$vsim_cmd\""
    }

    if {[lindex $args 1] == "1"} {
	set tool "ModelSim"
    } else {
	set tool "ModelSim-Altera"
    }

    if {$vsim_cmd == ""} {
	nl_postmsg error "Error: Can't launch the $tool software -- the path to the location of the executables for the $tool software not specified."
	nl_postmsg error "Error: You can specify the path in the EDA Tool Options page of the Options dialog box or using the Tcl command set_user_option."
	error ""  ""
    } 

    #First detect if correct version of Modelsim is in the users path
    if [ catch {exec $vsim_cmd -version} version_str] {
	nl_postmsg error "Error: Can't launch the $tool software -- the path to the location of the executables for the $tool software not specified."
	error "ModelSim Software not found - $version_str" "ModelSim Software not found - $version_str"
    } else {
	if [ regexp "ModelSim ALTERA" $version_str] {
	    set found_tool "ModelSim-Altera"
	} else {
	    set found_tool "ModelSim"
	}
	if {$found_tool != $tool} {
	    nl_postmsg error "Error: You selected $tool as Simulation Software in EDA Tool Settings, however NativeLink found $found_tool in the path -- correct path or change EDA Tool Settings and try again"
	    error "Expecing $tool, found $found_tool" "Expecting $tool, found $found_tool"
	}
    }

    if [ catch {eval gen_msim_script $args} msim_do_file_name] {
	error "$::errorCode" "$::errorInfo"
    } else {
	post_message -type info "Info: Generated ModelSim script file [pwd]/$msim_do_file_name" -file "[pwd]/$msim_do_file_name"
	if {$msim_do_file_name == ""} {
	    error "$::errorCode" "$::errorInfo"
	} else {
	    file delete -force msim_transcript
	    if [ catch { set pipe_id [open "|$msim_exe -i -l msim_transcript -do $msim_do_file_name" r]} ] {
		set savedCode $::errorCode
		set savedInfo $::errorInfo
		nl_postmsg error "Error: Can't launch $tool Simulation software ---  make sure the path environment variable points to the location of simulation software"
		error "" $savedInfo $savedCode
	    }
	    set tscr_fpos 0
	    set exit_loop 0
	    while { 1 } {
		gets $pipe_id line
		if { [eof $pipe_id] } {
		    set exit_loop 1
		}

		if {$exit_loop == "1"} {
		    break
		}
		set sleep_flag 0
		after 100 { set sleep_flag 1 }
		vwait sleep_flag
	    }
	    if [catch { open "msim_transcript" r } tscr_fid] {
		nl_postmsg error "Can't launch $tool - make sure the tool is setup correctly"
		error "" "Can't open transcript file msim_transcript"
	    } else {
		nl_postmsg info "Probing transcript"
		seek $tscr_fid $tscr_fpos
		while {1} {
		    gets $tscr_fid line
		    if { [eof $tscr_fid] == 0 } {
			if { [regexp {# \*\* Error:} $line] } {
			    nl_postmsg error "$tool Error: $line"
			    incr err_cnt;
			} elseif { [regexp {# \*\* Warning:} $line] } {
			    nl_postmsg warning "$tool Warning: $line"
			} else {
			    nl_postmsg info "$tool Info: $line"
			}
		    } else {
			break
		    }
		}
		set tscr_fpos [tell $tscr_fid]
		close $tscr_fid
	    }
	    set sleep_flag 0
	    after 100 { set sleep_flag 1 }
	    vwait sleep_flag
	    if [catch {close $pipe_id} errstream ] {
		set savedCode $::errorCode
		set savedInfo $::errorInfo
		if { [regexp {CHILDSTATUS ([1-9]+) ([1-9]+)} $savedCode match pid status]} {
		    if {$status != "46"} {
			nl_postmsg error "Error: Encountered an error while running $tool Simulation software "
			incr err_cnt
			error "" $savedInfo $savedCode
		    }
		} else {
			nl_postmsg error "Error: Encountered an error while running $tool Simulation software "
			incr err_cnt
			error "" $savedInfo $savedCode
		}
	    }
	}
    }
    return $err_cnt 
}
