# ----------------------------------------------------------------
#
namespace eval qsf_checklist {
#
# Description:	Provides information on HCII assignments
#				that requires verification.
#
# ----------------------------------------------------------------

	# ************************************************************
	#
	# Following array contains information about the assignments.
	# The format is:
	#
	#	-----------------
	#	key
	#	-----------------
	#
	#	  <qsf assignment>
	#
	#	-----------------
	#	values (in order)
	#	-----------------
	#
	#     <applicable flow type>     : <all|hcii_to_sii>
	#     <applicable device family> : <all|stratixii|hardcopyii>
	#     <recommended value>        : <value>
	#     <ui name>                  : <value>
	#     <master recommendation #>  : <value>
	#     <slave recommendation #>   : <value>
	#
	# ************************************************************

	variable all "all"
	variable hcii_to_sii "hcii_to_sii"
	variable sii "stratixii"
	variable hcii "hardcopyii"
	variable on "ON"
	variable off "OFF"
	variable empty ""

	variable global_assignments [list \
\
	ENABLE_DRC_SETTINGS						[list $all $all $on "Enable Design Assistant" R18 R4] \
	FLOW_DISABLE_ASSEMBLER					[list $all $all $off "Disable Assembler" R60 R40] \
	STA_MODE								[list $all $hcii $off $empty RXX RXX] \
	ENABLE_RECOVERY_REMOVAL_ANALYSIS		[list $all $all $on $empty R19 R5] \
	FLOW_ENABLE_TIMING_CONSTRAINT_CHECK		[list $all $all $on $empty R20 R6] \
	DO_COMBINED_ANALYSIS					[list $all $all $on $empty R21 R7] \
	REPORT_IO_PATHS_SEPARATELY				[list $all $all $on $empty R22 R8] \
	ENABLE_CLOCK_LATENCY					[list $all $all $on $empty R23 R9] \
	CUT_OFF_IO_PIN_FEEDBACK					[list $all $all $on $empty R61 R41] \
	CUT_OFF_READ_DURING_WRITE_PATHS			[list $all $all $on $empty R61 R41] \
	CUT_OFF_CLEAR_AND_PRESET_PATHS			[list $all $all $on $empty R61 R41] \
	DEFAULT_HOLD_MULTICYCLE					[list $all $all "SAME AS MULTICYCLE" $empty R61 R41] \
	INCREMENTAL_COMPILATION					[list $all $all $off $empty R25 R11] \
	TSU_REQUIREMENT							[list $all $all $empty $empty R62 R42] \
	TCO_REQUIREMENT							[list $all $all $empty $empty R62 R42] \
	TPD_REQUIREMENT							[list $all $all $empty $empty R62 R42] \
	TH_REQUIREMENT							[list $all $all $empty $empty R62 R42] \
	MIN_TCO_REQUIREMENT						[list $all $all $empty $empty R62 R42] \
	MINIMUM_TPD_REQUIREMENT					[list $all $all $empty $empty R62 R42] \
	RESERVE_ALL_UNUSED_PINS					[list $all $all "AS INPUT TRI-STATED WITH WEAK PULL-UP" $empty R64 R44] \
	OPTIMIZE_HOLD_TIMING					[list $all $all "ALL PATHS" $empty R65 R45] \
	EDA_FORMAL_VERIFICATION_TOOL			[list $all $all "<None>" $empty R66 R46] \
\
	]

	variable instance_assignments [list \
\
	CLOCK_ENABLE_MULTICYCLE					[list $all $all $empty $empty R63 R43] \
	CLOCK_ENABLE_MULTICYCLE_HOLD			[list $all $all $empty $empty R63 R43] \
	CLOCK_ENABLE_SOURCE_MULTICYCLE			[list $all $all $empty $empty R63 R43] \
	CLOCK_ENABLE_SOURCE_MULTICYCLE_HOLD		[list $all $all $empty $empty R63 R43] \
	INVERTED_CLOCK							[list $all $all $empty $empty R63 R43] \
	MAX_CLOCK_ARRIVAL_SKEW					[list $all $all $empty $empty R63 R43] \
	MAX_DATA_ARRIVAL_SKEW					[list $all $all $empty $empty R63 R43] \
	MAX_DELAY								[list $all $all $empty $empty R63 R43] \
	MIN_DELAY								[list $all $all $empty $empty R63 R43] \
	MIN_TCO_REQUIREMENT						[list $all $all $empty $empty R63 R43] \
	TCO_REQUIREMENT							[list $all $all $empty $empty R63 R43] \
	TH_REQUIREMENT							[list $all $all $empty $empty R63 R43] \
	TSU_REQUIREMENT							[list $all $all $empty $empty R63 R43] \
	VIRTUAL_CLOCK_REFERENCE					[list $all $all $empty $empty R63 R43] \
	VIRTUAL_PIN								[list $all $all $empty $empty R27 R13] \
	USE_CLK_FOR_VIRTUAL_PIN					[list $all $all $empty $empty R27 R13] \
	STP_VIRTUAL_PIN							[list $all $all $empty $empty R27 R13] \
	STP_VIRTUAL_PIN_CLK_SOURCE				[list $all $all $empty $empty R27 R13] \
	MAX_FANOUT								[list $all $all $empty $empty R28 R14] \
\
	]

	variable checklist
	array set checklist {}

	namespace export \
	begin \
	end \
	get_assignment \
	is_hardcopyii_first \
	is_check_required_for_hardcopyii \
	is_check_required_for_stratixii \
	is_global \
	is_instance \
	get_recommendation \
	get_ui_name

	# -------------------------------------------------
	# -------------------------------------------------
	proc initialize {} {
		# Initializes the checklist array if necessary.
	# -------------------------------------------------
	# -------------------------------------------------
		variable global_assignments
		variable instance_assignments
		variable checklist

		if {[array size checklist] == 0} {
				# initialize table
			set i 0
			foreach {j k} $global_assignments {
				set checklist($i) [list $j global $k]
				incr i
			}
			foreach {j k} $instance_assignments {
				set checklist($i) [list $j instance $k]
				incr i
			}
		}
	}

	# -------------------------------------------------
	# -------------------------------------------------
	proc begin {} {
		# Returns the begin iterator.
	# -------------------------------------------------
	# -------------------------------------------------
		initialize
		return 0
	}

	# -------------------------------------------------
	# -------------------------------------------------
	proc end {} { 
		# Returns the end iterator.
	# -------------------------------------------------
	# -------------------------------------------------
		variable checklist
		initialize
		return [array size checklist]
	}

	# -------------------------------------------------
	# -------------------------------------------------
	proc get_assignment {i} {
		# Returns the assignment name.
	# -------------------------------------------------
	# -------------------------------------------------
		variable checklist
		return [lindex $checklist($i) 0] 
	}

	# -------------------------------------------------
	# -------------------------------------------------
	proc is_global {i} {
		# Returns true if it's a global assignment.
	# -------------------------------------------------
	# -------------------------------------------------
		variable checklist
		return [expr {[string compare [lindex $checklist($i) 1] global] == 0}]
	}

	# -------------------------------------------------
	# -------------------------------------------------
	proc is_instance {i} {
		# Returns true if it's an instance assignment.
	# -------------------------------------------------
	# -------------------------------------------------
		variable checklist
		return [expr {[string compare [lindex $checklist($i) 1] instance] == 0}]
	}

	# -------------------------------------------------
	# -------------------------------------------------
	proc is_hardcopyii_first {i} {
		# Returns true if assignment needs to be checked
		# only in the HCII to SII migration flow.
	# -------------------------------------------------
	# -------------------------------------------------
		variable checklist
		variable hcii_to_sii
		return [expr {[string compare [lindex [lindex $checklist($i) 2] 0] $hcii_to_sii] == 0}]
	}

	# -------------------------------------------------
	# -------------------------------------------------
	proc is_check_required_for_hardcopyii {i} {
		# Returns true if assignment needs to
		# be checked for HardCopy II revision.
	# -------------------------------------------------
	# -------------------------------------------------
		variable checklist
		variable all
		variable hcii
		set device_family [lindex [lindex $checklist($i) 2] 1]
		return [expr {[string compare $device_family $all] == 0 || [string compare $device_family $hcii] == 0}]
	}

	# -------------------------------------------------
	# -------------------------------------------------
	proc is_check_required_for_stratixii {i} {
		# Returns true if assignment needs to
		# be checked for Stratix II revision.
	# -------------------------------------------------
	# -------------------------------------------------
		variable checklist
		variable all
		variable sii
		set device_family [lindex [lindex $checklist($i) 2] 1]
		return [expr {[string compare $device_family $all] == 0 || [string compare $device_family $sii] == 0}]
	}

	# -------------------------------------------------
	# -------------------------------------------------
	proc get_recommendation {i} {
		# Returns the recommended value for the
		# assignment.
	# -------------------------------------------------
	# -------------------------------------------------
		variable checklist
		return [lindex [lindex $checklist($i) 2] 2]
	}

	# -------------------------------------------------
	# -------------------------------------------------
	proc get_ui_name {i} {
		# Returns the special UI name for
		# the assignment.
	# -------------------------------------------------
	# -------------------------------------------------
		variable checklist
		return [lindex [lindex $checklist($i) 2] 3]
	}

	# -------------------------------------------------
	# -------------------------------------------------
	proc get_master_rkey {i} {
		# Returns the master recommendation key for
		# the assignment.
	# -------------------------------------------------
	# -------------------------------------------------
		variable checklist
		return [lindex [lindex $checklist($i) 2] 4]
	}

	# -------------------------------------------------
	# -------------------------------------------------
	proc get_slave_rkey {i} {
		# Returns the slave recommendation key for
		# the assignment.
	# -------------------------------------------------
	# -------------------------------------------------
		variable checklist
		return [lindex [lindex $checklist($i) 2] 5]
	}
}
