set pvcs_revision(main) [regsub -nocase -- {\$revision:\s*(\S+)\s*\$} {$Revision:   22.0.1.1  $} {\1}]

# *************************************************************
#
# File:        hcii_archive.tcl
#
# Usage:       quartus_cdb --hcii_archive [<options>] <project>
#
# Authors:     Peter Wu
#
#              Copyright (c) Altera Corporation 2005 - .
#              All rights reserved.
#
# *************************************************************

# -------------------------------------------------
# Available User Options for:
#    quartus_cdb --hcii_archive [<options>] <project>
# -------------------------------------------------
set available_options {
	{ revision.arg "#_ignore_#" "Option to specify the revision name" }
	{ output.arg "#_ignore_#" "Option to specify the output file name" }
}

# -------------------------------------------------
# Other Global variables
# -------------------------------------------------
array set info_map {}
	# If project was opened internally, then don't close project
set project_already_opened [is_project_open]

# -------------------------------------------------
# Load Required Packages
# -------------------------------------------------
package require cmdline
load_package database_manager
load_package device
load_package flow
load_package report

# -------------------------------------------------
# -------------------------------------------------
proc ipc_restrict_percent_range {min max} {
	# Update progress bar range
# -------------------------------------------------
# -------------------------------------------------

	global quartus

	if {$quartus(ipc_mode)} {
		puts "restrict_percent_range -min $min -max $max"
	}
}

# -------------------------------------------------
# -------------------------------------------------
proc ipc_set_percent_range {low high} {
	# Update progress bar range
# -------------------------------------------------
# -------------------------------------------------

	global quartus

	if {$quartus(ipc_mode)} {
		puts "set_percent_range -low $low -high $high"
	}
}

# -------------------------------------------------
# -------------------------------------------------
proc ipc_report_status {percent} {
	# Update progress bar
# -------------------------------------------------
# -------------------------------------------------

	global quartus

	if {$quartus(ipc_mode)} {
		puts "report_status $percent"
	}
}

# -------------------------------------------------
# -------------------------------------------------
proc is_user_entered {value} {
	# Determines if user entered the value
# -------------------------------------------------
# -------------------------------------------------

	return [expr [string compare $value "#_ignore_#"] ? 1 : 0]
}

# -------------------------------------------------
# -------------------------------------------------
proc display_banner {} {
	# Display start banner
# -------------------------------------------------
# -------------------------------------------------

	post_message -type info "----------------------------------------------------------"
	post_message -type info "----------------------------------------------------------"
	post_message -type info "[file tail [info script]] version $::pvcs_revision(main)"
	post_message -type info "----------------------------------------------------------"
	post_message -type info "----------------------------------------------------------"

	##---- 0% - 18% progress ----##
	ipc_set_percent_range 0 18

	ipc_report_status 0
}

# -------------------------------------------------
# -------------------------------------------------
proc process_options {} {
	# Process user entered options
# -------------------------------------------------
# -------------------------------------------------

	global info_map
	global project_already_opened
	global quartus

	# Define argv0 for the cmdline package to work properly
	set argv0 "quartus_cdb --hcii_archive"
	set usage "\[<options>\] <project_name>:"

	set argument_list $quartus(args)

	# Use cmdline package to parse options
	if [catch {array set info_map [cmdline::getoptions argument_list $::available_options]} result] {
		if {[llength $argument_list] > 0} {
			# This is not a simple -? or -help but an actual error condition
			post_message -type error "Found illegal or missing options"
		}
		post_message -type error [::cmdline::usage $::available_options $usage]
		post_message -type error "For more details, use \"quartus_cdb --help=hcii_archive\""
		return 0
	}

	# cmdline::getoptions modifies the argument_list.
	# However, it ignores positional arguments.
	# One and only one positional argument (the project) is expected.
	# Issue an error if otherwise.
	if {[llength $argument_list] == 1} {

		set info_map(project) [lindex $argument_list 0]

		if {[string length [file extension $info_map(project)]] > 0} {
			set $info_map(project) [file rootname $info_map(project)]
		}

		set $info_map(project) [file normalize $info_map(project)]

	} elseif {$project_already_opened == 1} {

		set info_map(project) $quartus(project)
		set info_map(revision) $quartus(settings)

	} else {

		if {[llength $argument_list] == 0} {
			post_message -type error "Project name is missing"
		} else {
			post_message -type error "More than one project name specified: $argument_list"
		}
		post_message -type info [::cmdline::usage $::available_options $usage]
		post_message -type info "For more details, use \"quartus_cdb --help=hcii_archive\""
		return 0
	}

	ipc_report_status 40

	return 1
}

# -------------------------------------------------
# -------------------------------------------------
proc open_project {} {
	# Open project if necessary
# -------------------------------------------------
# -------------------------------------------------

	global info_map
	global project_already_opened
	global quartus

		# Script may be called from Quartus
		# or
		# another script where the project was already opened
	if {$project_already_opened == 0} {

		# Open the project and create one if necessary
		if {[project_exists $info_map(project)]} {

			msg_vdebug "Opening project: $info_map(project) (revision = $info_map(revision))" 

			if {[is_user_entered $info_map(revision)]} {
				if [catch {project_open $info_map(project) -revision $info_map(revision)}] {
					post_message -type error "Project $info_map(project) (using $info_map(revision).qsf) could not be opened"
					return 0
				}
			} else {
				if [catch {project_open $info_map(project) -current_revision}] {
					post_message -type error "Project $info_map(project) (using the current revision's .qsf file) could not be opened"
					return 0
				}
				set info_map(revision) $quartus(settings)
			}
		} else {

			msg_vdebug "Creating project: $info_map(project) (revision = $info_map(revision))" 

			if {[is_user_entered $info_map(revision)]} {
				project_new $info_map(project) -revision $info_map(revision)
			} else {
				project_new $info_map(project)
				set info_map(revision) [get_current_revision]
			}
		}
	}

	if {[is_project_open] == 0} {

		post_message -type error "Failed to open project: $info_map(project)"
		return 0
	}

	msg_vdebug "project   = $info_map(project)"
	msg_vdebug "revision  = $info_map(revision)"

	ipc_report_status 100

	return 1
}

# -------------------------------------------------
# -------------------------------------------------
proc close_project {} {
	# Close project if necessary
# -------------------------------------------------
# -------------------------------------------------

	ipc_report_status 100

	if {$::project_already_opened == 0} {

		# Close project before exiting
		project_close
	}
}

# -------------------------------------------------
# -------------------------------------------------
proc update_report {success} {
	# Creates table if necessary
# -------------------------------------------------
# -------------------------------------------------

	global info_map

	catch load_report

	set table "Archive HardCopy II Handoff Files||Files Generated"
	set table_id [get_report_panel_id $table]
	if {$table_id == -1} {
   		set table_id [create_report_panel -table $table]
		add_row_to_table -id $table_id [list "File Name"]
	}

	if {$success} {

		set handoff_qar $info_map(output)
	} else {

		set handoff_qar "N/A"
	}

	add_row_to_table -id $table_id [list $handoff_qar]

	catch save_report_database
	catch unload_report
}

# -------------------------------------------------
# -------------------------------------------------
proc is_export_db_good {directory revision} {
	# Check exported database files for revision
# -------------------------------------------------
# -------------------------------------------------

	set success 0

	if {[file isdirectory $directory]} {

		set pwd [pwd]

		if [catch {cd $directory} result] {
			msg_vdebug $result
		} else {
			set log $revision.export_db_files
			if {[file exists $log]} {

				if [catch {open $log r} logid] {
					msg_vdebug "Couldn't open the file \"$log\" for reading"
				} else {
					set success 1
					set db_info ""
					array set check_revision {} 

					foreach fname [glob -nocomplain -- $revision.*] {
						set fname [string tolower $fname]
						set check_revision($fname) 1
					}

					while {[gets $logid fname] >= 0} {
						if {[string length $fname] > 0} {
							if {[string compare -nocase [file extension $fname] ".db_info"] == 0} {
								set db_info $fname
							}
							set fname [string tolower $fname]
							if {[llength [array get check_revision $fname]] > 0} {
							} else {
								msg_vdebug "Couldn't find the file \"[file join $directory $fname]\""
								set success 0
							}
						}
					}

					close $logid

					if {$success} {

						if {[string length $db_info] > 0} {
							# Now, check if the db_info file is consistent with the current
							# Quartus II software version.
							set success [is_database_compatible $db_info]
							if {!$success} {
								msg_vdebug "Database version for directory \"$directory\" is not consistent with the current version of Quartus II software"
							}
						} else {
							# no db_info file was found.
							msg_vdebug "Couldn't find the file \"[file join $directory $revision.db_info]\""
							set success 0
						}
					}
				}
			} else {
				msg_vdebug "Couldn't find the file \"$log\""
			}

			# return to original directory
			cd $pwd
		}
	} else {
		msg_vdebug "\"$directory\" is not a directory"
	}

	return $success
}

# -------------------------------------------------
# -------------------------------------------------
proc create_archive {revision qar low high} {
	# Create archive for the revision
# -------------------------------------------------
# -------------------------------------------------

	global info_map

	set success 1

		# set the revision as current, if necessary
	set prev_revision [get_current_revision]
	if {[string compare $prev_revision $revision] == 0} {
		set prev_revision ""
	} else {
		set_current_revision $revision
	}

	##---- low% - low+18% progress ----##
	ipc_set_percent_range $low [expr $low + 18]

	set ini_args ""
	if {[string compare $info_map(fpga) $revision] == 0} {

		# fpga revision
		post_message -type info "Initializing archive for Stratix II revision \"$revision\""

		set export_db [get_global_assignment -name BAK_EXPORT_DIR]
	} else {

		# hcii revision
		post_message -type info "Initializing archive for HardCopy II revision \"$revision\""

		set ini_args " --ini=qatm_force_export=on --ini=qatm_rcf_support_for_post_asm=on"
		set export_db [get_global_assignment -name HCII_OUTPUT_DIR]
	}

	if {$success} {

			# alway run export database

		#if {[is_export_db_good $export_db $revision]} {

		#	ipc_report_status 100
		#} else {

				# create export database since it doesn't exist
			post_message -type info "Exporting version compatible database"
			if [catch {execute_module -tool cdb -args "$ini_args --export_database=$export_db --disable_all_banners"} result] {
				set success 0
			}
		#}
	}

	##---- low+18% - high% progress ----##
	ipc_set_percent_range [expr $low + 18] $high

	if {$success} {

			# include version compatible database
		if {[catch {project_archive $qar -disable_export_database -version_compatible_database} result] || ![file exists $qar]} {

			if {[string length $result] > 0} {
				post_message -type error $result
			}
			post_message -type error "Failed to generate $qar"
			set success 0
		}

		ipc_report_status 100
	}

		# Reset revision to the previous one
	if {[string length $prev_revision] > 0} {
		set_current_revision $prev_revision
	}

	return $success
}

# -------------------------------------------------
# -------------------------------------------------
proc is_device_part_of {revision expected_family} {
	# Check if device exists in the expected family
# -------------------------------------------------
# -------------------------------------------------

	set success 0

	set device [get_global_assignment -name DEVICE]
	if [catch {set part_family [lindex [get_part_info $device -family] 0]} result] {
		post_message -type error "Found illegal device part \"$device\" -- make sure the DEVICE assignment is set to a legal device part name inside $revision.qsf"
	} else {
		if {[string compare $part_family $expected_family] == 0} {

			set success 1
		} else {

			post_message -type error "Device part \"$device\" does not belong to the $expected_family family. Make sure the DEVICE assignment is set to a $expected_family device part name inside $revision.qsf."
		}
	}

	return $success
}

# -------------------------------------------------
# -------------------------------------------------
proc generate_hcii_handoff_report {} {
	# Check if HardCopy II Handoff Report was run
# -------------------------------------------------
# -------------------------------------------------

	global info_map

	post_message -type info "Reviewing HardCopy II Handoff Report"

	foreach revision [list $info_map(hcii) $info_map(fpga)] {

		set success 1
		set hcii_report_count 0

			# set the revision as current, if necessary
		set prev_revision [get_current_revision]
		if {[string compare $prev_revision $revision] == 0} {
			set prev_revision ""
		} else {
			set_current_revision $revision
		}

		if [catch {load_report}] {
		} else {

			# Get Report panel
			set panel_name "*HardCopy II Handoff Report Summary"
			set panel_id [get_report_panel_id $panel_name]

			if {$panel_id == -1} {
				# panel not found
			} else {

				# found the hcii report
				incr hcii_report_count

				# Get the number of rows
				set row_cnt [get_number_of_rows -id $panel_id]

				msg_vdebug [get_report_panel_row -row 0 -id $panel_id]
				set type [get_report_panel_data -row 0 -col 0 -id $panel_id]
				set data [get_report_panel_data -row 0 -col 1 -id $panel_id]

				if {[string match *Failed* $data]} {

					set success 0
				}
			}

			unload_report
		}

		if {$hcii_report_count == 0 || $success == 0} {

			if {$success} {

				# Generate HardCopy II Handoff Report
				post_message -type info "Generating HardCopy II Handoff Report for revision $revision"
			} else {

				# Generate HardCopy II Handoff Report
				post_message -type info "Generating HardCopy II Handoff Report again for revision $revision due to previous errors"
			}

			if [catch {execute_module -tool cdb -args "--hcii_review --disable_all_banners"} result] {

				set success 0
			} else {

				set success 1
			}
		}

			# Reset revision to the previous one
		if {[string length $prev_revision] > 0} {
			set_current_revision $prev_revision
		}

		if {!$success} {

			break;
		}
	}

	return $success
}

# -------------------------------------------------
# -------------------------------------------------
proc hcii_archive {} {
	# Open project if necessary
# -------------------------------------------------
# -------------------------------------------------

	global info_map

	set project $info_map(project)
	set revision $info_map(revision)
	set handoff_qar $info_map(output)
	set success 1

		##---- 18% - 23% progress ----##
	ipc_set_percent_range 18 23

	set companion [get_global_assignment -name COMPANION_REVISION_NAME]
	if {[string length $companion] == 0} {

		post_message -type error "Companion revision is not defined by revision \"$revision\". Check $revision.qsf to see if COMPANION_REVISION_NAME assignment exists. If not, create a companion revision before proceeding."
		set success 0
	} elseif {![revision_exists $companion]} {

		post_message -type error "Companion revision $companion does not belong to the current project $project. Check if $companion.qsf exists. If not, create the companion revision before proceeding."
		set success 0
	}
	msg_vdebug "companion = $companion"
	set info_map(companion) $companion

	if {$success} {

			# Family name is already checked by hcii_review.tcl
		catch {set family [get_dstr_string -family [get_global_assignment -name FAMILY]]}
		set stratixii [get_dstr_string -family "stratixii"]
		if {[string compare $family $stratixii] == 0} {

			set info_map(fpga) $revision
			set info_map(hcii) $companion
		} else {

			set info_map(hcii) $revision
			set info_map(fpga) $companion
		}

		msg_vdebug "SII  = $info_map(fpga)"
		msg_vdebug "HCII = $info_map(hcii)"

	#	if {[string compare $info_map(hcii) $revision] != 0} {
	#
	#		post_message -type error "HardCopy II Handoff Files can only be archived from the HardCopy II companion revision \"$companion\""
	#		set success 0
	#	}

		ipc_report_status 70

		if {![generate_hcii_handoff_report]} {

			set success 0
		}

		ipc_report_status 100
	}

	if {$success} {

		set revision_qar "$revision.hcii_archive.qar"
		set companion_qar "$companion.hcii_archive.qar"
		if {![is_user_entered $handoff_qar]} {

			set handoff_qar "$info_map(hcii)_handoff.qar"
			set info_map(output) $handoff_qar
		}
		set handoff_qar_list "$handoff_qar.files"

		msg_vdebug "revision_qar  = $revision_qar"
		msg_vdebug "companion_qar = $companion_qar"
		msg_vdebug "handoff_qar   = $handoff_qar"

			# remove previous files
		if {[file exists $revision_qar]} {
			file delete -force $revision_qar
		}
		if {[file exists $companion_qar]} {
			file delete -force $companion_qar
		}
		if {[file exists $handoff_qar_list]} {
			file delete -force $handoff_qar_list
		}
		if {[file exists $handoff_qar]} {
			file delete -force $handoff_qar
		}

			# create version compatible databases
			# and then archive
		if {![create_archive $info_map(hcii) $revision_qar 23 57]} {

			set success 0
		} elseif {![create_archive $info_map(fpga) $companion_qar 57 91]} {

			set success 0
		}
	}

	if {$success} {

		# -------------------------------------------------
		# Archive HardCopy II
		# -------------------------------------------------

			# create list of qar files to archive
		set of [open $handoff_qar_list "w"]
		puts $of $revision_qar
		puts $of $companion_qar
		close $of

			# create hcii archive
		##---- 91% - 100% progress ----##
		ipc_set_percent_range 91 100
		ipc_report_status 30
		if {[catch {project_archive $handoff_qar -general_archive -file_list $handoff_qar_list} result] || ![file exists $handoff_qar]} {

			if {[string length $result] > 0} {
				post_message -type error $result
			}
			post_message -type error "Failed to generate $handoff_qar"
			set success 0
		}

		post_message -type info "Generated archive \"$handoff_qar\" for HardCopy II Handoff Files"

			# delete temporary files
		file delete -force $handoff_qar_list
		file delete -force $revision_qar
		file delete -force ${revision_qar}log
		file delete -force $companion_qar
		file delete -force ${companion_qar}log
	}

		# update the report database
	update_report $success

	if {$success} {

		ipc_report_status 95
	}

	return $success
}

# -------------------------------------------------
# -------------------------------------------------
proc main {} {
	# Script starts here
	# 1.- Process command-line arguments
# -------------------------------------------------
# -------------------------------------------------

#---------
	ipc_restrict_percent_range 5 91
#---------

	display_banner

	if {[process_options] && [open_project]} {

		hcii_archive

		close_project
	}

#---------
	ipc_restrict_percent_range 91 100
#---------
}

# -------------------------------------------------
# -------------------------------------------------
main
# -------------------------------------------------
# -------------------------------------------------
