# *************************************************************
#
# Filename:    HARDCOPY_MSGS.TCL
#
# Description: This file is used to post messages
#              for hardcopy Tcl scripts.
#
#              Copyright (c) Altera Corporation 1997 - 2006
#              All rights reserved.
#
# *************************************************************


# ---------------------------------------------------------------------------
#
namespace eval hardcopy_msgs {
#
# Description:	Define the namespace.
#
# Warning:		All defined variables should not be accessed.
#				Use defined accessors instead.
#
# ---------------------------------------------------------------------------

		# Map of message handles
	variable msgs
	array set msgs {}
		# Handle -> <help id> <formatted message> <raw message> <expected number of arguments>
	set msgs(E_ILLEGAL_MIG_DEVICE) [list EHARDCOPY_ILLEGAL_MIG_DEVICE "Illegal DEVICE_TECHNOLOGY_MIGRATION_LIST = %s" "Illegal DEVICE_TECHNOLOGY_MIGRATION_LIST = %1!s!" 1]
	set msgs(E_ILLEGAL_SOURCE_DEVICE) [list EHARDCOPY_ILLEGAL_SOURCE_DEVICE "Source device %s is not valid -- make sure you select a valid device" "Source device %1!s! is not valid -- make sure you select a valid device" 1]
	set msgs(E_ILLEGAL_SOURCE_FAMILY) [list EHARDCOPY_ILLEGAL_SOURCE_FAMILY "Source device family %s is not supported -- make sure the family is either a Stratix II or HardCopy II source device" "Source device family %1!s! is not supported -- make sure the family is either a Stratix II or HardCopy II source device" 1]
	set msgs(E_NO_COPY_DEST_DIRECTORY) [list EHARDCOPY_NO_COPY_DEST_DIRECTORY "Couldn't copy \\\"%s\\\" to \\\"%s\\\" because the destination file is an existing directory" "Couldn't copy \\\"%1!s!\\\" to \\\"%2!s!\\\" because the destination file is an existing directory" 2]
	set msgs(E_NO_COPY_PERMISSION) [list EHARDCOPY_NO_COPY_PERMISSION "Couldn't copy \\\"%s\\\" to \\\"%s\\\" -- make sure you have permission to write to the destination file" "Couldn't copy \\\"%1!s!\\\" to \\\"%2!s!\\\" -- make sure you have permission to write to the destination file" 2]
	set msgs(E_NO_COPY_SRC_DIRECTORY) [list EHARDCOPY_NO_COPY_SRC_DIRECTORY "Couldn't copy \\\"%s\\\" to \\\"%s\\\" because the source file has the same name as a directory" "Couldn't copy \\\"%1!s!\\\" to \\\"%2!s!\\\" because the source file has the same name as a directory" 2]
	set msgs(E_SAME_REVISION) [list EHARDCOPY_SAME_REVISION "COMPANION_REVISION_NAME cannot be the same as the current revision name" "COMPANION_REVISION_NAME cannot be the same as the current revision name" 0]
	set msgs(E_SRC_FILE_MISSING) [list EHARDCOPY_SRC_FILE_MISSING "Could not copy \\\"%s\\\" to \\\"%s\\\" because the source file does not exist" "Could not copy \\\"%1!s!\\\" to \\\"%2!s!\\\" because the source file does not exist" 2]
	set msgs(E_TURN_OFF_QIC) [list EHARDCOPY_TURN_OFF_QIC "Cannot migrate project with Incremental Compilation enabled -- turn Incremental Compilation off and then recompile before performing migration." "Cannot migrate project with Incremental Compilation enabled -- turn Incremental Compilation off and then recompile before performing migration." 0]
	set msgs(E_UNDEFINED_MIG_DEVICE) [list EHARDCOPY_UNDEFINED_MIG_DEVICE "Your current revision has DEVICE_TECHNOLOGY_MIGRATION_LIST undefined -- make sure you select a valid migration device" "Your current revision has DEVICE_TECHNOLOGY_MIGRATION_LIST undefined -- make sure you select a valid migration device" 0]
	set msgs(E_WRONG_FAMILY_MIG_DEVICE) [list EHARDCOPY_WRONG_FAMILY_MIG_DEVICE "Your current revision has DEVICE_TECHNOLOGY_MIGRATION_LIST set to %s which does not belong to the %s device family" "Your current revision has DEVICE_TECHNOLOGY_MIGRATION_LIST set to %1!s! which does not belong to the %2!s! device family" 2]
	set msgs(I_COPIED_FILE) [list IHARDCOPY_COPIED_FILE "Copied \\\"%s\\\" to \\\"%s\\\"" "Copied \\\"%1!s!\\\" to \\\"%2!s!\\\"" 2]
	set msgs(I_COPIED_FILE_GENERIC) [list IHARDCOPY_COPIED_FILE_GENERIC "Copied \\\"%s\\\" to \\\"%s\\\"" "Copied \\\"%1!s!\\\" to \\\"%2!s!\\\"" 2]
	set msgs(I_GENERATING_CHECKSUMS) [list IHARDCOPY_GENERATING_CHECKSUMS "Generating file checksum values" "Generating file checksum values" 0]
	set msgs(I_MIGRATING_BACK_ANNOTATION) [list IHARDCOPY_MIGRATING_BACK_ANNOTATION "Migrating Back-Annotated Assignments" "Migrating Back-Annotated Assignments" 0]
	set msgs(I_MIGRATING_GLOBAL) [list IHARDCOPY_MIGRATING_GLOBAL "Migrating Global Assignments" "Migrating Global Assignments" 0]
	set msgs(I_MIGRATING_INSTANCE) [list IHARDCOPY_MIGRATING_INSTANCE "Migrating Instance Assignments" "Migrating Instance Assignments" 0]
	set msgs(I_MIGRATING_PARAMETER) [list IHARDCOPY_MIGRATING_PARAMETER "Migrating User Parameters" "Migrating User Parameters" 0]
	set msgs(I_MIGRATING_PIN_LOCATION) [list IHARDCOPY_MIGRATING_PIN_LOCATION "Migrating Pin Assignments" "Migrating Pin Assignments" 0]
	set msgs(I_OVERRIDE_REVISION) [list IHARDCOPY_OVERRIDE_REVISION "Overwriting existing Target Revision named %s" "Overwriting existing Target Revision named %1!s!" 1]
	set msgs(I_READ_BACK_ANNOTATION) [list IHARDCOPY_READ_BACK_ANNOTATION "Reading assignments from back-annotation" "Reading assignments from back-annotation" 0]
	set msgs(I_READ_PIN_LOCATIONS) [list IHARDCOPY_READ_PIN_LOCATIONS "Reading Pin locations from compiler database" "Reading Pin locations from compiler database" 0]
	set msgs(I_READ_QSF) [list IHARDCOPY_READ_QSF "Reading Quartus II Settings File assignments" "Reading Quartus II Settings File assignments" 0]
	set msgs(I_SCRIPT) [list IHARDCOPY_SCRIPT "Using Script \\\"%s\\\"" "Using Script \\\"%1!s!\\\"" 1]
	set msgs(I_SEPARATOR) [list IHARDCOPY_SEPARATOR "----------------------------------------------------------" "----------------------------------------------------------" 0]
	set msgs(I_SKIPPING_BACK_ANNOTATION) [list IHARDCOPY_SKIPPING_BACK_ANNOTATION "Skipping resource back-annoation because INI hcii_migration_dont_use_asl=on" "Skipping resource back-annoation because INI hcii_migration_dont_use_asl=on" 0]
	set msgs(I_SKIPPING_GLOBAL) [list IHARDCOPY_SKIPPING_GLOBAL "Not migrating global assignment named %s" "Not migrating global assignment named %1!s!" 1]
	set msgs(I_SKIPPING_INSTANCE) [list IHARDCOPY_SKIPPING_INSTANCE "Not migrating instance assignment named %s" "Not migrating instance assignment named %1!s!" 1]
	set msgs(I_SOURCE_DEVICE) [list IHARDCOPY_SOURCE_DEVICE "Source Device = %s" "Source Device = %1!s!" 1]
	set msgs(I_SOURCE_FAMILY) [list IHARDCOPY_SOURCE_FAMILY "Source Family = %s" "Source Family = %1!s!" 1]
	set msgs(I_SOURCE_REVISION) [list IHARDCOPY_SOURCE_REVISION "Using Source Revision named %s" "Using Source Revision named %1!s!" 1]
	set msgs(I_TARGET_DEVICE) [list IHARDCOPY_TARGET_DEVICE "Target Device = %s" "Target Device = %1!s!" 1]
	set msgs(I_TARGET_FAMILY) [list IHARDCOPY_TARGET_FAMILY "Target Family = %s" "Target Family = %1!s!" 1]
	set msgs(I_TARGET_REVISION) [list IHARDCOPY_TARGET_REVISION "Using Target Revision named %s" "Using Target Revision named %1!s!" 1]
	set msgs(W_DEST_FILE_EXISTS) [list WHARDCOPY_DEST_FILE_EXISTS "Not copying \\\"%s\\\" to \\\"%s\\\" because the destination file already exists" "Not copying \\\"%1!s!\\\" to \\\"%2!s!\\\" because the destination file already exists" 2]
	set msgs(W_FAIL_BACK_ANNOTATION) [list WHARDCOPY_FAIL_BACK_ANNOTATION "Resource allocation cannot be back annotated -- a successful full compile is required" "Resource allocation cannot be back annotated -- a successful full compile is required" 0]
	set msgs(W_FAIL_RESOURCE_ALLCATION) [list WHARDCOPY_FAIL_RESOURCE_ALLCATION "Resource allocation cannot be migrated from the compiled database" "Resource allocation cannot be migrated from the compiled database" 0]
	set msgs(W_NOT_COMPILED) [list WHARDCOPY_NOT_COMPILED "Revision %s has not been compiled" "Revision %1!s! has not been compiled" 1]
	set msgs(W_NOT_COMPILED_WITH_FAMILY) [list WHARDCOPY_NOT_COMPILED_WITH_FAMILY "Revision %s has not been compiled for device family %s" "Revision %1!s! has not been compiled for device family %2!s!" 2]
	set msgs(W_PIN_NOT_MIGRATED) [list WHARDCOPY_PIN_NOT_MIGRATED "Pin Locations will not be migrated from the compiled database" "Pin Locations will not be migrated from the compiled database" 0]

		# Map of message types
	variable msg_types
	array set msg_types { \
		CW	{ critical_warning	"Critical Warning" } \
		E	{ error				Error } \
		EI	{ extra_info		"Extra Info" } \
		I	{ info				Info } \
		W	{ warning			Warning } \
	}
	
	# -------------------------------------------------
	# -------------------------------------------------
	#
	# Exported API's:
	#
	#  - post			- posts the message "Critical Warning: Hello World" given the msg handle "CW_HELLO_WORLD" (and optional variable sized arguments)
	#  - get_user_type	- returns "Critical Warning"
	#  - get_text		- returns "Hello World"
	#
	# Hidden API's:
	#
	#  - get_debug_type	- returns "critical_warning" used by "post_message -type <debug msg type>"
	#  - is_legal		- exits script program with error if handle is not defined in the .msg file
	#  - get_help_id	- returns the help id associated with GUI online help page
	#
	# -------------------------------------------------
	# -------------------------------------------------
	namespace export post get_user_type get_text internal_error
	
	# -------------------------------------------------
	# -------------------------------------------------
	proc internal_error {msg} {
		# Returns:
		#   An Internal Error.
	# -------------------------------------------------
	# -------------------------------------------------
		return -code error "\n!------- Internal Error -------!\n$msg\n!------- Internal Error -------!\n"
	}
	
	# -------------------------------------------------
	# -------------------------------------------------
	proc is_legal {handle} {
		# Returns:
		#   The actual handle.
		#
		#   The actual handle will differ when the user
		#   specifies the handle as a CW (Critical Warning)
		#   or EI (Extra Info). In this case, we
		#   need to convert them into I and W, respectively.
		#
		#   Script will exit if the handle is not
		#   a legally defined handle.
	# -------------------------------------------------
	# -------------------------------------------------
	
		variable msgs
	
		set orignal_handle $handle
	
		set underscore [string first "_" $handle]
		set type [string range $handle 0 [expr $underscore - 1]]
		if {[string compare $type EI] == 0} {
			set handle "I_[string range $handle [expr $underscore + 1] end]"
		} elseif {[string compare $type CW] == 0} {			
			set handle "W_[string range $handle [expr $underscore + 1] end]"
		}
	
		if {[string compare [array names msgs $handle] ""] == 0} {
			internal_error "Illegal handle: $orignal_handle"
		}
	
		return $handle
	}
	
	# -------------------------------------------------
	# -------------------------------------------------
	proc get_debug_type {handle for_ostream} {
		# Returns:
		#   The message type for use internally
		#   by "post_message -type <debug msg type>"
		#   For example, the debug msg type
		#   "critical_warning"
		#   is equivalent to the user msg type
		#   "Critical Warning".
	# -------------------------------------------------
	# -------------------------------------------------
	
		variable msg_types
	
		is_legal $handle
	
		set type [string range $handle 0 [expr [string first "_" $handle] - 1]]
	
		return [lindex $msg_types($type) 0]
	}
	
	
	# -------------------------------------------------
	# -------------------------------------------------
	proc get_user_type {handle for_ostream} {
		# Returns:
		#   The message type for pretty printing.
		#   This returns the string you see prepended
		#   to every user message.
		#   For example, the user msg type
		#   "Critical Warning"
		#   is equivalent to the debug msg type
		#   "critical_warning".
	# -------------------------------------------------
	# -------------------------------------------------
	
		variable msg_types
	
		is_legal $handle
	
		set type [string range $handle 0 [expr [string first "_" $handle] - 1]]
	
		return [lindex $msg_types($type) 1]
	}
	
	# -------------------------------------------------
	# -------------------------------------------------
	proc get_help_id {handle} {
		# Returns:
		#   The help id associated with Quartus II
		#   online help page.
	# -------------------------------------------------
	# -------------------------------------------------
	
		variable msgs
	
		set handle [is_legal $handle]
	
		return [lindex $msgs($handle) 0]
	}
	
	# -------------------------------------------------
	# -------------------------------------------------
	proc get_raw_text {handle} {
		# Returns:
		#   The raw text containing containing
		#   placeholders such as %1!s!
	# -------------------------------------------------
	# -------------------------------------------------
	
		variable msgs
	
		set handle [is_legal $handle]
	
		return [lindex $msgs($handle) 2]
	}
	
	# -------------------------------------------------
	# -------------------------------------------------
	proc get_expected_args_count {handle} {
		# Returns:
		#   The expected number of arguments
		#   for the handle.
	# -------------------------------------------------
	# -------------------------------------------------
	
		variable msgs
	
		set handle [is_legal $handle]
	
		return [lindex $msgs($handle) 3]
	}
	
	# -------------------------------------------------
	# -------------------------------------------------
	proc _get_text {handle args_list} {
		# Returns:
		#   The actual text users see when the
		#   message is posted
		#   (e.g. "Hello World")
		#   -- minus the prepending
		#   user msg type (returned by
		#   proc "get_user_type") and ": "
		#   (e.g. minus the "Info: " portion).
	# -------------------------------------------------
	# -------------------------------------------------
	
		variable msgs
	
		set handle [is_legal $handle]
	
		set tmp_msg	[lindex $msgs($handle) 1]
		set ret_msg	""
		set i		0
		set left	0
		while {1} {
			set right [string first "%s" $tmp_msg $left]
			if {$right == -1} {
				append ret_msg [string range $tmp_msg $left [string length $tmp_msg]]
				break
			}
			append ret_msg [string range $tmp_msg $left [expr $right - 1]] [lindex $args_list $i]
			set left [expr $right + 2]
			incr i
		}
	
		return $ret_msg
	}
	
	# -------------------------------------------------
	# -------------------------------------------------
	proc get_text {handle args} {
		# Returns:
		#   Same as "_get_text" except
		#   that this command can take a variable
		#   number of arguments.
	# -------------------------------------------------
	# -------------------------------------------------
	
		return [_get_text $handle $args]
	}
	
	# -------------------------------------------------
	# -------------------------------------------------
	proc post {handle args} {
		# Returns:
		#   Nothing.
		#   Posts the user message associated to
		#   the handle and optional variable-sized arguments.
		# Note:
		#   If you wish to post "Extra Info" or "Critical Warning"
		#   types, just prepend with "EI" or "CW" instead
		#   of the "I" and "W" types marked in your .msg file.
	# -------------------------------------------------
	# -------------------------------------------------
	
		set cmd "post_message \"[_get_text $handle $args]\" \
							  -raw_text \"[get_raw_text $handle]\" \
							  -type \"[get_debug_type $handle 0]\" \
							  -help_id \"[get_help_id $handle]\""
	
		set expected [get_expected_args_count $handle]
		set actual 0
		foreach i $args { append raw_args "$i "; incr actual }
		if {$expected != $actual} {
			internal_error "Expected $expected argument[expr {$expected == 1 ? "" : "s"}] for message \"$handle\" but got $actual instead"
		} elseif {$actual > 0} {
			append cmd " -raw_args \"$raw_args\""
		}
	
		eval $cmd
	}
}

