::dtw::add_version_date {$Date:   15 Feb 2006 13:10:04  $}

##############################################################################
#
# File Name:    dtw_misc_panel.tcl
#
# Summary:      This TK script is a simple Graphical User Interface to
#               generate timing requirements for DDR memory interfaces
#
# Licencing:
#               ALTERA LEGAL NOTICE
#               
#               This script is  pursuant to the following license agreement
#               (BY VIEWING AND USING THIS SCRIPT, YOU AGREE TO THE
#               FOLLOWING): Copyright (c) 2006-2007 Altera Corporation, San Jose,
#               California, USA.  Permission is hereby granted, free of
#               charge, to any person obtaining a copy of this software and
#               associated documentation files (the "Software"), to deal in
#               the Software without restriction, including without limitation
#               the rights to use, copy, modify, merge, publish, distribute,
#               sublicense, and/or sell copies of the Software, and to permit
#               persons to whom the Software is furnished to do so, subject to
#               the following conditions:
#               
#               The above copyright notice and this permission notice shall be
#               included in all copies or substantial portions of the Software.
#               
#               THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
#               EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES
#               OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
#               NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT
#               HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY,
#               WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
#               FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
#               OTHER DEALINGS IN THE SOFTWARE.
#               
#               This agreement shall be governed in all respects by the laws of
#               the State of California and by the laws of the United States of
#               America.
#
#               
#
# Usage:
#
#               You can run this script from a command line by typing:
#                     quartus_sh --dtw
#
###############################################################################

# ----------------------------------------------------------------
#
namespace eval dtw_misc {
#
# Description: Namespace to encapsulate the Miscellaneous Data panel
#
# ----------------------------------------------------------------
	variable s_misc_data_types
	variable s_data_array_list
    variable s_data_enum_list

    variable s_time_units
    set s_time_units {"ms" "us" "ns" "ps"}

    variable s_percent_with_time_units
    set s_percent_with_time_units {"%" "ms" "us" "ns" "ps"}

    variable s_timing_model "combined_fast_and_slow"

	# Source everything to get the latest version date
	source ${::dtw::s_dtw_dir}dtw_extract_tco.tcl
}

# ----------------------------------------------------------------
#
proc dtw_misc::panel { misc_window next_button args } {
#
# Description: Show the wizard panel for entering misc parameters
#
# ----------------------------------------------------------------
	variable s_misc_data_types
    variable s_time_units
    variable s_percent_with_time_units

	# Format of entries:
	# <variable_name> <display_text> <optional> <data units> <default unit index> <default data value> <entry label>
	set s_misc_data_types [list \
		[list sys_clk_max_tco "Estimated slowest tco to the CK/CK# system clock output pins" 0 $s_time_units @2 "" ""] \
		[list sys_clk_slow_min_tco "Estimated slow model minimum tco to the CK/CK# system clock output pins" 0 $s_time_units @2 "" ""] \
		[list sys_clk_fast_max_tco "Estimated fast model maximum tco to the CK/CK# system clock output pins" 0 $s_time_units @2 "" ""] \
		[list sys_clk_min_tco "Estimated fastest tco to the CK/CK# system clock output pins" 0 $s_time_units @2 "" ""] \
		[list fb_clk_max_tco "Estimated slowest tco to the feedback clock output pin" 0 $s_time_units @2 "" ""] \
		[list fb_clk_slow_min_tco "Estimated slow model min tco to the feedback clock output pin" 0 $s_time_units @2 "" ""] \
		[list fb_clk_fast_max_tco "Estimated fast model max tco to the feedback clock output pin" 0 $s_time_units @2 "" ""] \
		[list fb_clk_min_tco "Estimated fastest tco to the feedback clock output pin" 0 $s_time_units @2 "" ""] \
		[list pll_dc_distortion "Duty cycle distortion error of DDIO output clocks with PLL" 0 $s_percent_with_time_units @0 "5" "+/-"] \
		[list fpga_tOUTJITTER "PLL jitter" 0 $s_time_units @2 "0.125" "+/-"] \
		[list fpga_tPLL_COMP_ERROR "PLL compensation error" 0 $s_time_units @2 "0.100" "+/-"] \
		[list fpga_tPLL_PSERR "PLL phase shift error" 0 $s_time_units @2 "0.030" "+/-"] \
		[list fpga_tCLOCK_SKEW_ADDER "Clock skew adder (skew between two dedicated clock networks feeding I/O banks on the same side of the FPGA)" 0 $s_time_units @2 "0.050" "+/-"] \
		[list fpga_tSHIFT_ERROR "DQS phase shift error" 0 $s_time_units @2 "0.038" "+/-"] \
		[list fpga_tJITTER "Estimated DQS phase jitter" 0 $s_time_units @2 "0.150" "+/-"] \
		[list fpga_tSKEW "Estimated DQS bus skew" 0 $s_time_units @2 "0.035" "+/-"] \
		]

	# TAN doesn't support intra-die variations yet....
    # [list ocv "On-die delay variation" 0 "%" @0 "33"]

	frame $misc_window -borderwidth 12
	labelframe ${misc_window}.data_frame -text "FPGA timing parameters" -labelanchor nw -pady 2
	label ${misc_window}.data_frame.label -text "Note that the following parameters should be adjusted after the first full compilation to match what is reported by the Timing Analyzer.  Use the \"Extract tcos\" button to adjust the tcos after the first compile so that accurate latencies are used for the read input clocks." -anchor w -justify left -pady 4 -padx 4
	foreach data_type $s_misc_data_types {
		set data_enum [lindex $data_type 0]
		set data_label [lindex $data_type 1]
		set data_units [lindex $data_type 3]
		set data_units_default [lindex $data_type 4]
		set data_default [lindex $data_type 5]
		set entry_label [lindex $data_type 6]
		::dtw::dtw_data_entry::dtw_data_entry ${misc_window}.data_frame.${data_enum}_frame $data_label $data_units $data_units_default 4 $data_default 8 "" $entry_label "[namespace code validate_float] %P %V"
	}
	frame ${misc_window}.data_frame.button_frame -borderwidth 4
	Button ${misc_window}.data_frame.button_frame.set_defaults_button -text "Defaults" -helptext "Set default values for all parameters" -command "[namespace code on_set_defaults] $misc_window $next_button" -pady 2
	Button ${misc_window}.data_frame.button_frame.extract_tcos_button -text "Extract tcos" -helptext "Set the tco parameters according to the results of the previous compilation" -command "[namespace code on_extract_tcos] $misc_window $next_button" -width 15 -pady 2

	pack ${misc_window}.data_frame -side top -fill both -expand 0
	pack ${misc_window}.data_frame.label -side top -fill x -anchor w
	pack ${misc_window}.data_frame.button_frame -side top -fill x
	pack ${misc_window}.data_frame.button_frame.set_defaults_button -side left -padx 4 -pady 4 -anchor w
	pack ${misc_window}.data_frame.button_frame.extract_tcos_button -side right -padx 4 -pady 4 -anchor e
	foreach data_type $s_misc_data_types {
		set data_enum [lindex $data_type 0]		
		set data_optional [lindex $data_type 2]
		pack ${misc_window}.data_frame.${data_enum}_frame -side top -fill x -expand 0
		if {$data_optional == 0} {
			::dtw::dtw_data_entry::bind_to_entry ${misc_window}.data_frame.${data_enum}_frame <KeyRelease> "[namespace code on_entry_keyrelease] $misc_window $next_button"
		}
	}
	labelframe ${misc_window}.radio_frame
	radiobutton ${misc_window}.radio_frame.fast_and_slow_radio -anchor w -justify left -text "Both fast and slow timing model tcos (easiest to use, but will generate very conservative constraints because the entire range of tcos cannot exist on the same device)" -variable [namespace which -variable s_timing_model] -value "combined_fast_and_slow" -command "[namespace code on_radio] $misc_window $next_button"
	radiobutton ${misc_window}.radio_frame.slow_only_radio -anchor w -justify left -text "Slow timing model tcos (will require a separate DTW and timing analysis iteration with Fast timing model tcos)" -variable [namespace which -variable s_timing_model] -value "slow" -command "[namespace code on_radio] $misc_window $next_button"
	radiobutton ${misc_window}.radio_frame.fast_only_radio -anchor w -justify left -text "Fast timing model tcos (should only be used for final timing analysis, not for constraining the Quartus II fitter)" -variable [namespace which -variable s_timing_model] -value "fast" -command "[namespace code on_radio] $misc_window $next_button"
	pack ${misc_window}.radio_frame.fast_and_slow_radio -side top -fill x -padx 8
	pack ${misc_window}.radio_frame.slow_only_radio -side top -fill x -padx 8
	pack ${misc_window}.radio_frame.fast_only_radio -side top -fill x -padx 8
	pack ${misc_window}.radio_frame -side top -fill x -pady 4
	set radio_width 20
	bind ${misc_window}.radio_frame.fast_and_slow_radio <Configure> "${misc_window}.radio_frame.fast_and_slow_radio configure -wraplength \[expr %w - $radio_width\]"
	bind ${misc_window}.radio_frame.slow_only_radio <Configure> "${misc_window}.radio_frame.slow_only_radio configure -wraplength \[expr %w - $radio_width\]"
	bind ${misc_window}.radio_frame.fast_only_radio <Configure> "${misc_window}.radio_frame.fast_only_radio configure -wraplength \[expr %w - $radio_width\]"

	bind $misc_window <Map> "[namespace code update_next_state] $misc_window $next_button"
	bind ${misc_window}.data_frame.label <Configure> "${misc_window}.data_frame.label configure -wraplength %w"

	return $misc_window
}

# ----------------------------------------------------------------
#
proc dtw_misc::save_data {misc_window data_array_name} {
#
# Description: Save the data in this panel in the data_array
#
# ----------------------------------------------------------------
	variable s_data_enum_list
	upvar 1 $data_array_name data_array
    variable s_timing_model

	# Data in the panel is saved in array:
	# set data_array(data_enum0) = {value0}
	# set data_array(data_enum1) = {value1}
	# ...
	set mode_name [::dtw::dtw_timing::get_ddr_interface_mode data_array]
	foreach data_enum $s_data_enum_list {
		set data_value [::dtw::dtw_data_entry::get_entry_text ${misc_window}.data_frame.${data_enum}_frame]
		if {$data_value != ""} {
			set data_array($data_enum) [::dtw::dtw_data_entry::get_data ${misc_window}.data_frame.${data_enum}_frame]
		}
	}
	set data_array(timing_model) $s_timing_model

	return
}

# ----------------------------------------------------------------
#
proc dtw_misc::load_data {misc_window data_array_name} {
#
# Description: Load the data in this panel from the data_array
#
# ----------------------------------------------------------------
	variable s_misc_data_types
	variable s_data_array_list
	variable s_data_enum_list
	variable s_timing_model
	upvar $data_array_name data_array

	set s_data_array_list [array get data_array]
	set mode_name [::dtw::dtw_timing::get_ddr_interface_mode data_array]
	set s_data_enum_list [list]
	if {[::dtw::dtw_device_get_family_parameter $data_array(family) ${mode_name}_misc_parameters s_data_enum_list] == 0} {
		::dtw::dtw_device_get_family_parameter "_default" ${mode_name}_misc_parameters s_data_enum_list
	}
	set const_term_list [list]
	if {[::dtw::dtw_device_get_family_parameter $data_array(family) const_terms const_term_list] == 0} {
		::dtw::dtw_device_get_family_parameter "_default" const_terms const_term_list
	}
	# Remove all const enums
	array set const_term_array $const_term_list
	set const_enum_list [array names const_term_array]
	foreach const_enum $const_enum_list {
		set const_index [lsearch -exact $s_data_enum_list $const_enum]
		if {$const_index != -1} {
			set s_data_enum_list [lreplace $s_data_enum_list $const_index $const_index]
		}
	}

	set user_term_list [list]
	if {[::dtw::dtw_device_get_family_parameter $data_array(family) user_terms user_term_list] == 0} {
		::dtw::dtw_device_get_family_parameter "_default" user_terms user_term_list
	}
	array set user_term_array $user_term_list

	foreach data_type $s_misc_data_types {
		set data_enum [lindex $data_type 0]
		if {[lsearch -exact $s_data_enum_list $data_enum] == -1} {
			pack forget ${misc_window}.data_frame.${data_enum}_frame
		} else {
			if {[array names user_term_array -exact $data_enum] != ""} {
				::dtw::dtw_data_entry::set_user_term_label_text ${misc_window}.data_frame.${data_enum}_frame "($user_term_array($data_enum))"
			} else {
				::dtw::dtw_data_entry::set_user_term_label_text ${misc_window}.data_frame.${data_enum}_frame "(${data_enum})"
			}
			if {[array names data_array -exact $data_enum] == ""} {
				set value [get_default_parameter data_array $data_enum]
			} else {
				set value $data_array($data_enum)
			}
			if {$value != ""} {
				::dtw::dtw_data_entry::set_data ${misc_window}.data_frame.${data_enum}_frame $value
			}
		}
	}
	if {[array names data_array -exact timing_model] != ""} {
		set s_timing_model $data_array(timing_model)
	}

	if {$data_array(device) != ""} {
		${misc_window}.data_frame.button_frame.set_defaults_button configure -text "Defaults for [string toupper $data_array(device)]"
	} else {
		# No defaults available for unknown devices
		${misc_window}.data_frame.button_frame.set_defaults_button configure -state disabled
	}

	::dtw::dtw_device_get_family_parameter "_default" $data_array(memory_type)_user_terms mem_user_term_list
	array set mem_user_term $mem_user_term_list
	if {[lsearch -exact $s_data_enum_list "fb_clk_max_tco"] == -1 && [lsearch -exact $s_data_enum_list "fb_clk_max_tco"] == -1 } {
		${misc_window}.radio_frame configure -text "Generate $mem_user_term(read_dqs) input clock latencies using"
	} else {
		${misc_window}.radio_frame configure -text "Generate fedback/$mem_user_term(read_dqs) input clock latencies using"
	}
	::dtw::dtw_data_entry::set_label ${misc_window}.data_frame.sys_clk_max_tco_frame "Estimated slowest tco to the $mem_user_term(ck)/$mem_user_term(ckn) system clock output pins"
	::dtw::dtw_data_entry::set_label ${misc_window}.data_frame.sys_clk_slow_min_tco_frame "Estimated slow model minimum tco to the $mem_user_term(ck)/$mem_user_term(ckn) system clock output pins"
	::dtw::dtw_data_entry::set_label ${misc_window}.data_frame.sys_clk_fast_max_tco_frame "Estimated fast model maximum tco to the $mem_user_term(ck)/$mem_user_term(ckn) system clock output pins"
	::dtw::dtw_data_entry::set_label ${misc_window}.data_frame.sys_clk_min_tco_frame "Estimated fastest tco to the $mem_user_term(ck)/$mem_user_term(ckn) system clock output pins"
	if {[lsearch -exact $s_data_enum_list "sys_clk_max_tco"] == -1 && [lsearch -exact $s_data_enum_list "sys_clk_min_tco"] == -1 && [lsearch -exact $s_data_enum_list "fb_clk_max_tco"] == -1 && [lsearch -exact $s_data_enum_list "fb_clk_max_tco"] == -1 } {
		pack forget ${misc_window}.data_frame.label
		pack forget ${misc_window}.data_frame.button_frame.extract_tcos_button
		pack forget ${misc_window}.radio_frame
	} elseif {$::dtw::s_auto_extract_tcos} {
		do_extract_tcos $misc_window
	}

	return
}

# ----------------------------------------------------------------
#
proc dtw_misc::validate_float {number validation_type} {
#
# Description: Get the data in this panel
#
# ----------------------------------------------------------------
	if {$number == "." && $validation_type != "focusout"} {
		set valid 1
	} else {
		set valid [string is double $number]
	}
	return $valid
}

# ----------------------------------------------------------------
#
proc dtw_misc::on_set_defaults { misc_window next_button } {
#
# Description: Sets all parameters to their default values
#
# ----------------------------------------------------------------
    variable s_data_enum_list
	variable s_data_array_list

	array set data_array $s_data_array_list
	foreach data_enum $s_data_enum_list {
		set default_value [get_default_parameter data_array $data_enum]
		::dtw::dtw_data_entry::set_data ${misc_window}.data_frame.${data_enum}_frame $default_value
	}

	update_next_state $misc_window $next_button
}

# ----------------------------------------------------------------
#
proc dtw_misc::on_extract_tcos { misc_window next_button } {
#
# Description: "Extract tcos" button processing
#
# ----------------------------------------------------------------
	do_extract_tcos $misc_window
	update_next_state $misc_window $next_button
	::dtw::msg_wait_end $misc_window
}

# ----------------------------------------------------------------
#
proc dtw_misc::do_extract_tcos { misc_window } {
#
# Description: Extract tcos based on previously compiled fit
#
# ----------------------------------------------------------------
	variable s_data_array_list
	array set data_array $s_data_array_list

	set tmp_file_name "$data_array(output_filename).tmp"

	package require ::quartus::dtw_dwz

	::dtw::msg_wait_begin "Extracting tcos...." $misc_window
	::quartus::dtw_dwz::write_dwz data_array "$tmp_file_name"

	# Run the extract_tco script under quartus_tan
	if {[catch "exec quartus_tan -t ${::dtw::s_dtw_dir}dtw_extract_tco.tcl \"$tmp_file_name\"" tco_result]} {
		puts $tco_result
	} else {
		puts $tco_result
		::quartus::dtw_dwz::read_dwz data_array "$tmp_file_name"

		# set the UI values
		set extracted_enums_list [list sys_clk_max_tco sys_clk_slow_min_tco sys_clk_fast_max_tco sys_clk_min_tco]
		if {$data_array(is_clk_fedback_in)} {
			lappend extracted_enums_list fb_clk_max_tco fb_clk_slow_min_tco fb_clk_fast_max_tco fb_clk_min_tco
		}
		foreach data_enum $extracted_enums_list {
			set value $data_array($data_enum)
			::dtw::dtw_data_entry::set_data ${misc_window}.data_frame.${data_enum}_frame $value
		}
	}
	::dtw::msg_wait_end $misc_window
}

# ----------------------------------------------------------------
#
proc dtw_misc::on_entry_keyrelease { misc_window next_button } {
#
# Description: Get the data in this panel
#
# ----------------------------------------------------------------
	update_next_state $misc_window $next_button
}

# ----------------------------------------------------------------
#
proc dtw_misc::update_next_state { misc_window next_button } {
#
# Description: Tells if the "Next" button should be enabled and changes its
#              state accordingly.
#
# ----------------------------------------------------------------
	variable s_misc_data_types
	variable s_data_enum_list
	variable s_timing_model

	if {$s_timing_model == "slow"} {
		set optional_list [list sys_clk_fast_max_tco sys_clk_min_tco fb_clk_fast_max_tco fb_clk_min_tco]
	} elseif {$s_timing_model == "fast"} {
		set optional_list [list sys_clk_max_tco sys_clk_slow_min_tco fb_clk_max_tco fb_clk_slow_min_tco]
	} else {
		set optional_list [list sys_clk_slow_min_tco sys_clk_fast_max_tco fb_clk_slow_min_tco fb_clk_fast_max_tco]
	}

	set enable 1
	foreach data_type $s_misc_data_types {
		set data_enum [lindex $data_type 0]
		if {[lsearch -exact $s_data_enum_list $data_enum] != -1} {
			set data_optional [lindex $data_type 2]
			set data_value [::dtw::dtw_data_entry::get_entry_text ${misc_window}.data_frame.${data_enum}_frame]
			if {[lsearch -exact $optional_list $data_enum] != -1} {
				::dtw::dtw_data_entry::set_labels_state ${misc_window}.data_frame.${data_enum}_frame disabled
				set is_optional 1
			} else {
				::dtw::dtw_data_entry::set_labels_state ${misc_window}.data_frame.${data_enum}_frame normal
				set is_optional 0
			}
			if {$data_value != "" || $data_optional == 1} {
				# data good
			} else {
				if {$is_optional} {
					# data optional
				} else {
					# data required - disable Next button
					set enable 0
				}
			}
		}
	}
	if {$next_button != "" } {
		if {$enable == 0} {
			$next_button configure -state disabled
		} else {
			$next_button configure -state normal
		}
	}
	return $enable
}

# ----------------------------------------------------------------
#
proc dtw_misc::on_next { misc_window } {
#
# Description: Handles the "Next" button event
#
# ----------------------------------------------------------------
	if {[update_next_state $misc_window ""]} {
		set result "next"
	} else {
		::dtw::msg_o "Error" "Missing required data"
		set result "none"
	}
	return $result
}

# ----------------------------------------------------------------
#
proc dtw_misc::get_default_parameter { data_array_name data_enum } {
#
# Description: Sets defaults for all parameters
#
# ----------------------------------------------------------------
	upvar 1 $data_array_name data_array

	set default_value ""
	if {$data_enum == "sys_clk_slow_min_tco" || $data_enum == "fb_clk_slow_min_tco"} {
		set data_enum "sys_clk_max_tco"
	} elseif {$data_enum == "sys_clk_fast_max_tco" || $data_enum == "fb_clk_fast_max_tco"} {
		set data_enum "sys_clk_min_tco"
	}
	if {$data_enum == "sys_clk_min_tco" || $data_enum == "fb_clk_min_tco"} {
		::dtw::dtw_device_get_timing_parameter $data_array(family) "m" $data_enum default_value
	} elseif {$data_enum == "fpga_tCLOCK_SKEW_ADDER"} {
		set default_value ""
		if {[::dtw::dtw_device_get_family_parameter $data_array(family) fpga_tCLOCK_SKEW_ADDER_$data_array(density) default_value] == 0} {
			::dtw::dtw_device_get_family_parameter $data_array(family) fpga_tCLOCK_SKEW_ADDER default_value
		}
	} else {
		if {[::dtw::dtw_device_get_timing_parameter $data_array(family) $data_array(speed_grade) $data_enum default_value] == 0} {
			::dtw::dtw_device_get_family_parameter $data_array(family) $data_enum default_value
		}
	}
	return [split $default_value]
}

# ----------------------------------------------------------------
#
proc dtw_misc::validate_positive_int { number validation_type } {
#
# Description: Validate the number is a positive integer
#
# Returns: true if it is, false otherwise
#
# ----------------------------------------------------------------
	if {$number == "" || ([string is integer $number] && $number > 0)} {
		set valid 1
	} else {
		set valid 0
	}

	return $valid
}

# ----------------------------------------------------------------
#
proc dtw_misc::on_radio {misc_window next_button} {
#
# Description: Called when a radio button is selected
#
# Returns:     nothing
#
# ----------------------------------------------------------------
	update_next_state $misc_window $next_button
	return
}
