if {[namespace exists ::dtw]} {
	::dtw::add_version_date {$Date:   05 Feb 2006 03:13:32  $}
}

##############################################################################
#
# File Name:    dtw_extract_tco.tcl
#
# Summary:      This TK script is a simple Graphical User Interface to
#               generate timing requirements for DDR memory interfaces
#
# Licencing:
#               ALTERA LEGAL NOTICE
#               
#               This script is  pursuant to the following license agreement
#               (BY VIEWING AND USING THIS SCRIPT, YOU AGREE TO THE
#               FOLLOWING): Copyright (c) 2006-2007 Altera Corporation, San Jose,
#               California, USA.  Permission is hereby granted, free of
#               charge, to any person obtaining a copy of this software and
#               associated documentation files (the "Software"), to deal in
#               the Software without restriction, including without limitation
#               the rights to use, copy, modify, merge, publish, distribute,
#               sublicense, and/or sell copies of the Software, and to permit
#               persons to whom the Software is furnished to do so, subject to
#               the following conditions:
#               
#               The above copyright notice and this permission notice shall be
#               included in all copies or substantial portions of the Software.
#               
#               THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
#               EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES
#               OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
#               NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT
#               HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY,
#               WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
#               FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
#               OTHER DEALINGS IN THE SOFTWARE.
#               
#               This agreement shall be governed in all respects by the laws of
#               the State of California and by the laws of the United States of
#               America.
#
#               
#
# Usage:
#
#               You can run this script from a command line by typing:
#                     quartus_sh --dtw
#
###############################################################################

package require ::quartus::dtw_msg
package require ::quartus::dtw_dwz

# ----------------------------------------------------------------
#
namespace eval dtw_extract_tco {
#
# Description: Top-level namespace for the auto-detection code
#
# ----------------------------------------------------------------
	source ${quartus(tclpath)}packages/dtw/dtw_msg.tcl
	namespace import ::quartus::dtw_msg::*

	source ${quartus(tclpath)}packages/dtw/dtw_dwz.tcl
	namespace import ::quartus::dtw_dwz::*
}

# ----------------------------------------------------------------
#
proc dtw_extract_tco::extract { argv } {
#
# Description: Extracts tcos
#              First argument is the dwz file name
#
# ----------------------------------------------------------------
	set dwz_file [lindex $argv 0]

	init_tk
	package require ::quartus::timing 1.2
	package require ::quartus::report 2.1
	package require ::quartus::advanced_timing 1.2

	array set data_array [list]
	read_dwz data_array $dwz_file
	set project_name [file tail "$data_array(project_path)"]
	set revision_name $data_array(project_revision)
	set success 1

	# Open project
	set project_is_open [is_project_open]
	if {$project_is_open == 0} {
		project_open $project_name -revision $revision_name
	}

	if {[was_fit_successful]} {
		# Extract from the Fast Timing Model first
		create_timing_netlist -fast_model
		create_p2p_delays
		set nodes_list [concat $data_array(ck_list) $data_array(ckn_list) $data_array(clk_sys)]
		if {$data_array(is_clk_fedback_in)} {
			lappend nodes_list $data_array(clk_feedback_out)
		}
		create_tdb_arrays $nodes_list tdb_array
		set clk_sys_id $tdb_array($data_array(clk_sys))

		get_min_max_tcos tdb_array $clk_sys_id [concat $data_array(ck_list) $data_array(ckn_list)] min_tco max_tco
		if {$min_tco != [get_illegal_delay_value]} {
			set data_array(sys_clk_min_tco) "$min_tco ns"
		} else {
			set data_array(sys_clk_min_tco) ""
			puts "Cannot find fast min tco for CK/CK# clocks"
		}
		if {$max_tco != [get_illegal_delay_value]} {
			set data_array(sys_clk_fast_max_tco) "$max_tco ns"
		} else {
			set data_array(sys_clk_fast_max_tco) ""
			puts "Cannot find fast max tco for CK/CK# clocks"
		}
		unset min_tco
		unset max_tco

		if {$data_array(is_clk_fedback_in)} {
			get_min_max_tcos tdb_array $clk_sys_id [list $data_array(clk_feedback_out)] fb_min_tco fb_max_tco
			if {$fb_min_tco != [get_illegal_delay_value]} {
				set data_array(fb_clk_min_tco) "$fb_min_tco ns"
			} else {
				set data_array(fb_clk_min_tco) ""
				puts "Cannot find fast min tco for feedback output clock"
			}
			if {$fb_max_tco != [get_illegal_delay_value]} {
				set data_array(fb_clk_fast_max_tco) "$fb_max_tco ns"
			} else {
				set data_array(fb_clk_fast_max_tco) ""
				puts "Cannot find fast max tco for feedback output clock"
			}
		}
		delete_timing_netlist

		# Extract from the Slow Timing Model
		create_timing_netlist
		create_p2p_delays

		get_min_max_tcos tdb_array $clk_sys_id [concat $data_array(ck_list) $data_array(ckn_list)] min_tco max_tco
		if {$min_tco != [get_illegal_delay_value]} {
			set data_array(sys_clk_slow_min_tco) "$min_tco ns"
		} else {
			set data_array(sys_clk_slow_min_tco) ""
			puts "Cannot find slow min tco for CK/CK# clocks"
		}
		if {$max_tco != [get_illegal_delay_value]} {
			set data_array(sys_clk_max_tco) "$max_tco ns"
		} else {
			set data_array(sys_clk_max_tco) ""
			puts "Cannot find slow max tco for CK/CK# clocks"
		}
		if {$data_array(is_clk_fedback_in)} {
			get_min_max_tcos tdb_array $clk_sys_id [list $data_array(clk_feedback_out)] fb_min_tco fb_max_tco
			if {$fb_min_tco != [get_illegal_delay_value]} {
				set data_array(fb_clk_slow_min_tco) "$fb_min_tco ns"
			} else {
				set data_array(fb_clk_slow_min_tco) ""
				puts "Cannot find slow min tco for feedback output clock"
			}
			if {$fb_max_tco != [get_illegal_delay_value]} {
				set data_array(fb_clk_max_tco) "$fb_max_tco ns"
			} else {
				set data_array(fb_clk_max_tco) ""
				puts "Cannot find slow max tco for feedback output clock"
			}
		}
		delete_timing_netlist
	} else {
		msg_o "Sorry" "Cannot extract tcos until you have a successfully compiled fit."
		set success 0
	}
	# All done - Clean up
	if {$project_is_open == 0} {
		project_close
	}

	if {$success} {
		# Auto-detect successful
		write_dwz data_array $dwz_file
		set return_code 0
	} else {
		set return_code 1
	}
	return $return_code
}

# ----------------------------------------------------------------
#
proc dtw_extract_tco::create_tdb_arrays { name_list tdb_array_name } {
#
# Description: Create a map of interesting TDB node ids.  This is an
#              array mapping node names to ids.
#
# ----------------------------------------------------------------
	upvar 1 $tdb_array_name tdb_array

	array set tdb_array [list]
	foreach_in_collection node_id [get_timing_nodes -type keeper] {
		set node_type [get_timing_node_info -info type $node_id]
		set node_name [get_timing_node_info -info name $node_id]
		if {($node_type == "pin" || $node_type == "clk") && [lsearch -exact $name_list $node_name] != -1} {
			set tdb_array($node_name) $node_id
		}
	}
}

# ----------------------------------------------------------------
#
proc dtw_extract_tco::compute_max_tco { clk_id out_id } {
#
# Description: Given a clk and output, returns the tco
#
# ----------------------------------------------------------------
	set ck_delays [get_delays_from_clocks $out_id]
	set tco [get_illegal_delay_value]
	foreach delay_set $ck_delays {
		if {[lindex $delay_set 0] == $clk_id} {
			set tco [lindex $delay_set 1]
			break
		}
	}
	if {$tco != [get_illegal_delay_value]} {
		set clk_latency [get_timing_node_info -info clock_latency $clk_id]
		if {[regexp -nocase -- {(-?[0-9]+\.?[0-9]* [a-z]+)(-?[0-9]+\.?[0-9]* [a-z]+)} $clk_latency -> late_latency early_latency]} {
			set tco_list [split $tco]
			set latency_list [split $late_latency]
			if {[lindex $tco_list 1] != "ns" || [lindex $latency_list 1] != "ns"} {
				error "Unexpected time units from ::quartus::timing API"
			}
			set tco [expr [lindex $tco_list 0] + [lindex $latency_list 0]]
		}
	}
	return $tco
}

# ----------------------------------------------------------------
#
proc dtw_extract_tco::compute_min_tco { clk_id out_id } {
#
# Description: Given a clk and output, returns the tco in ns
#
# ----------------------------------------------------------------
	set ck_delays [get_delays_from_clocks $out_id]
	set tco [get_illegal_delay_value]
	foreach delay_set $ck_delays {
		if {[lindex $delay_set 0] == $clk_id} {
			set tco [lindex $delay_set 2]
			break
		}
	}
	if {$tco != [get_illegal_delay_value]} {
		set clk_latency [get_timing_node_info -info clock_latency $clk_id]
		if {[regexp -nocase -- {(-?[0-9]+\.?[0-9]* [a-z]+)(-?[0-9]+\.?[0-9]* [a-z]+)} $clk_latency -> late_latency early_latency]} {
			set tco_list [split $tco]
			set latency_list [split $early_latency]
			if {[lindex $tco_list 1] != "ns" || [lindex $latency_list 1] != "ns"} {
				error "Unexpected time units from ::quartus::timing API"
			}
			set tco [expr [lindex $tco_list 0] + [lindex $latency_list 0]]
		}
	}
	return $tco
}

# ----------------------------------------------------------------
#
proc dtw_extract_tco::get_min_max_tcos { tdb_array_name clk_id output_list min_tco_name max_tco_name } {
#
# Description: Get the min and max tcos for the given outputs through the given
#              clock
#
# ----------------------------------------------------------------
	upvar 1 $tdb_array_name tdb_array
	upvar 1 $min_tco_name min_tco
	upvar 1 $max_tco_name max_tco

	set min_tco [get_illegal_delay_value]
	set max_tco [get_illegal_delay_value]
	foreach output_pin $output_list {
		set output_pin_id $tdb_array($output_pin)
		set output_pin_min_tco [compute_min_tco $clk_id $output_pin_id]
		if {$min_tco == [get_illegal_delay_value] || $output_pin_min_tco < $min_tco} {
			set min_tco $output_pin_min_tco
		}
		set output_pin_max_tco [compute_max_tco $clk_id $output_pin_id]
		if {$max_tco == [get_illegal_delay_value] || $output_pin_max_tco > $max_tco} {
			set max_tco $output_pin_max_tco
		}
	}
}

# ----------------------------------------------------------------
#
proc dtw_extract_tco::was_fit_successful { } {
#
# Description: Tells if the previous fit attempt was successful or not
#
# ----------------------------------------------------------------
	set fit_ok 0

	if {[catch "load_report" res]} {
		# No reports to load, so fit was not successful
		puts "No previous compile detected"
	} else {
		set fitter_summary_id [get_report_panel_id {Fitter||Fitter Summary}]
		if {$fitter_summary_id != -1} {
			set fitter_status [split [lindex [get_report_panel_row -id $fitter_summary_id -row 0] 1]]
			if {[lindex $fitter_status 0] == "Successful"} {
				set fit_ok 1
			} else {
				puts "Fitter not Successful"
			}
		} else {
			puts "No previous fit detected"
		}
		unload_report
	}

	return $fit_ok
}

if {[namespace exists ::dtw] == 0} {
	dtw_extract_tco::extract $quartus(args)
}
