::dtw::add_version_date {$Date:   15 Feb 2006 13:07:24  $}

##############################################################################
#
# File Name:    dtw_clocks_panel.tcl
#
# Summary:      This TK script is a simple Graphical User Interface to
#               generate timing requirements for DDR memory interfaces
#
# Licencing:
#               ALTERA LEGAL NOTICE
#               
#               This script is  pursuant to the following license agreement
#               (BY VIEWING AND USING THIS SCRIPT, YOU AGREE TO THE
#               FOLLOWING): Copyright (c) 2006-2007 Altera Corporation, San Jose,
#               California, USA.  Permission is hereby granted, free of
#               charge, to any person obtaining a copy of this software and
#               associated documentation files (the "Software"), to deal in
#               the Software without restriction, including without limitation
#               the rights to use, copy, modify, merge, publish, distribute,
#               sublicense, and/or sell copies of the Software, and to permit
#               persons to whom the Software is furnished to do so, subject to
#               the following conditions:
#               
#               The above copyright notice and this permission notice shall be
#               included in all copies or substantial portions of the Software.
#               
#               THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
#               EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES
#               OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
#               NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT
#               HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY,
#               WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
#               FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
#               OTHER DEALINGS IN THE SOFTWARE.
#               
#               This agreement shall be governed in all respects by the laws of
#               the State of California and by the laws of the United States of
#               America.
#
#               
#
# Usage:
#
#               You can run this script from a command line by typing:
#                     quartus_sh --dtw
#
###############################################################################

# ----------------------------------------------------------------
#
namespace eval dtw_clocks {
#
# Description: Namespace to encapsulate the Read Clocks panel
#
# ----------------------------------------------------------------
    variable s_time_units
    set s_time_units {"ms" "us" "ns" "ps" "MHz" "GHz"}
    set s_phase_units {"ms" "us" "ns" "ps" "deg"}

	variable s_clocks_data_types
	variable s_tCK
	# List of data_types:
	# <variable_name> <display_text> <indent> <rows> <optional> <widget> <name browser node type> <default name browser filter> <validation proc> <combobox list> <default combobox index> <default value>
	variable VARIABLE_NAME 0
	variable DISPLAY_TEXT 1
	variable INDENT 2
	variable ROWS 3
	variable OPTIONAL 4
	variable WIDGET 5
	variable NODE_TYPE 6
	variable NAME_BROWSER_FILTER 7
	variable VALIDATION_PROC 8
	variable COMBOBOX_LIST 9
	variable DEFAULT_COMBOBOX_INDEX 10
	variable DEFAULT_VALUE 11

	variable s_project_name ""
	variable s_revision_name ""
	variable s_use_hardware_dqs

	variable s_memory_type "ddr"
	variable s_is_clk_fedback_in 0
	variable s_use_source_synchronous_pll 0
	variable s_use_dcfifo 0

	variable s_data_enum_list [list]
	variable s_forget_list [list]

	variable s_widgets
}

# ----------------------------------------------------------------
#
proc dtw_clocks::update_data_enum_list {} {
#
# Description: Determines the valid data enums for the chosen mode
#
# ----------------------------------------------------------------
	variable s_use_hardware_dqs
	variable s_memory_type
	variable s_is_clk_fedback_in
	variable s_use_source_synchronous_pll
	variable s_use_dcfifo
	variable s_data_enum_list
	variable s_forget_list

	set s_data_enum_list [list clk_sys clk_pll_in pll_input_freq pll_mult pll_div]
	set s_forget_list [list]
	if {$s_memory_type == "ddr"} {
		lappend s_forget_list use_source_synchronous_pll use_dcfifo 
		lappend s_data_enum_list clk_resync resync_cycle resync_phase is_clk_fedback_in
		if {$s_use_hardware_dqs == 1} {
			# DQS hardware capture
			lappend s_data_enum_list clk_read_postamble postamble_cycle postamble_phase
			if {$s_is_clk_fedback_in == 1} {
				# Resync to Fedback PLL, Resync2 to System PLL 
				lappend s_data_enum_list clk_feedback_out clk_fedback_in clk_resync2 resync_sys_cycle resync_sys_phase inter_postamble_cycle inter_postamble_phase
			} else {
				# Resync to System PLL
			}
		} else {
			lappend s_forget_list clk_read_postamble postamble_cycle postamble_phase inter_postamble_cycle inter_postamble_phase
			if {$s_is_clk_fedback_in == 1} {
				# Capture with Fedback PLL, Resync to System PLL
				lappend s_data_enum_list clk_feedback_out clk_fedback_in clk_resync2 resync_sys_cycle resync_sys_phase
			} else {
				# Capture to System PLL
			}
		}
	} elseif {$s_memory_type == "qdr2" || $s_memory_type == "rldram2"} {
		lappend s_forget_list is_clk_fedback_in clk_feedback_out clk_fedback_in clk_read_postamble postamble_cycle postamble_phase inter_postamble_cycle inter_postamble_phase
		if {$s_use_hardware_dqs == 1} {
			lappend s_forget_list use_source_synchronous_pll clk_resync2 resync_sys_cycle resync_sys_phase
			lappend s_data_enum_list use_dcfifo
			# DQS hardware capture
			if {$s_use_dcfifo == 1} {
				# No resync
			} else {
				# Resync to System PLL
				lappend s_data_enum_list clk_resync resync_cycle resync_phase
			}
		} else {
			lappend s_data_enum_list use_source_synchronous_pll
			if {$s_use_source_synchronous_pll == 1} {
				lappend s_data_enum_list use_dcfifo
				if {$s_use_dcfifo == 1} {
					# Capture with Fedback PLL capture, no resync
					lappend s_data_enum_list clk_resync resync_phase
				} else {
					# Capture with Fedback PLL capture, Resync to System PLL
					lappend s_data_enum_list clk_resync resync_phase clk_resync2 resync_sys_cycle resync_sys_phase
				}
			} else {
				# Capture to System PLL
				lappend s_data_enum_list clk_resync resync_cycle resync_phase
			}
		}
	}
}

# ----------------------------------------------------------------
#
proc dtw_clocks::is_data_type_with_units { data_type } {
#
# Description: Tell whether or not the given data type has a units combobox
#
# ----------------------------------------------------------------
	variable COMBOBOX_LIST
	return [expr "[llength $data_type] > $COMBOBOX_LIST"]
}

# ----------------------------------------------------------------
#
proc dtw_clocks::panel { clocks_window next_button args } {
#
# Description: Show the wizard panel for entering clocks
#
# ----------------------------------------------------------------
	variable s_time_units
	variable s_phase_units
	variable s_clocks_data_types
	variable VARIABLE_NAME
	variable DISPLAY_TEXT
	variable INDENT
	variable ROWS
	variable OPTIONAL
	variable WIDGET
	variable NODE_TYPE
	variable NAME_BROWSER_FILTER
	variable VALIDATION_PROC
	variable COMBOBOX_LIST
	variable DEFAULT_COMBOBOX_INDEX
	variable DEFAULT_VALUE
	variable s_widgets

	# Format of entries:
	# <variable_name> <display_text> <indent> <lines> <optional> <widget> <name browser node type> <default name browser filter> <validation proc> <combobox list> <default combobox index> <default value>
	
	# If <optional> is 0, entry is required to enable the wizard's Next button
	# If <optional> is 1, entry is not required
	set s_clocks_data_types [list \
		[list clk_sys "Name of the PLL output driving the CK/CK# system clock pin(s)" 0 2 0 "pll_2_sys_output_wire_widget" comb "*pll*"] \
		[list clk_pll_in "Name of the above PLL's input pin" 20 1 0 "pll_inclk_pin_widget" input "*"] \
		[list pll_input_freq "PLL input pin frequency" 20 1 0 "" "" "" validate_float $s_time_units @2 ""] \
		[list pll_mult "PLL frequency multiplier of CK clock" 20 1 0 "pll_2_sys_output_wire_widget" "" "" validate_positive_int] \
		[list pll_div "PLL frequency divider of CK clock" 20 1 0 "pll_2_sys_output_wire_widget" "" "" validate_positive_int] \
		[list use_source_synchronous_pll "Captured with a second PLL fed by the CQ clock" 0 1 1 "" "" ""] \
		[list use_dcfifo "Use a dual-clock FIFO to re-synchronize read data from the Q capture registers" 0 1 1 "" "" ""] \
		[list clk_resync "Name of the PLL output re-synchronizing read data from the DQ capture registers" 0 2 0 "pll_2_resync_wire_widget" comb "*pll*"] \
		[list resync_cycle "Resynchronize captured read data in cycle" 20 1 0 "pll_2_resync_wire_widget" "" "" validate_int] \
		[list resync_phase "with phase shift, including inversion" 50 1 0 "pll_2_resync_wire_widget" "" "" validate_float $s_phase_units @4 "<default>"] \
		[list is_clk_fedback_in "Re-synchronized with a second PLL using a fed-back clock" 20 1 1 "" "" ""] \
		[list clk_feedback_out "Name of the feedback clock output pin" 50 1 0 "feedback_output_pin_widget" output "*"] \
		[list clk_fedback_in "Name of the fedback clock input pin" 50 1 0 "fedback_clk_pin_widget" input "*"] \
		[list clk_resync2 "Name of the System PLL output re-synchronizing second stage read data" 50 2 0 "pll_2_resync2_clk_wire_widget" comb "*pll*"] \
		[list resync_sys_cycle "Resynchronize captured read data to above System PLL clock output in cycle" 50 1 0 "pll_2_resync2_clk_wire_widget" "" "" validate_int] \
		[list resync_sys_phase "with phase shift, including inversion" 80 1 0 "pll_2_resync2_clk_wire_widget" "" "" validate_float $s_phase_units @4 "<default>"] \
		[list clk_read_postamble  "Name of the PLL output driving the read post-amble reset control clock (optional)" 0 2 1 "" comb "*pll*"] \
		[list postamble_cycle "Post-amble reset control clock in cycle (optional)" 20 1 1 "" "" "" validate_int] \
		[list postamble_phase "with phase shift, including inversion (optional)" 80 1 1 "" "" "" validate_float $s_phase_units @4 ""] \
		[list inter_postamble_cycle "Post-amble reset on system cycle (optional)" 20 1 1 "" "" "" validate_int] \
		[list inter_postamble_phase "with phase shift, including inversion (optional)" 80 1 1 "" "" "" validate_float $s_phase_units @4 ""]]

	frame $clocks_window -borderwidth 12
	labelframe ${clocks_window}.data_frame -text "PLL Clocks for Reading from Memory" -labelanchor nw -pady 2
	set data_frame ${clocks_window}.data_frame

	foreach data_type $s_clocks_data_types {
		set data_enum [lindex $data_type $VARIABLE_NAME]
		set data_label [lindex $data_type $DISPLAY_TEXT]
		set browser_node_type [lindex $data_type $NODE_TYPE]
		frame ${data_frame}.${data_enum}_frame -padx 8 -pady 1
		label ${data_frame}.${data_enum}_frame.label -text $data_label
		if {$data_enum == "is_clk_fedback_in" || $data_enum == "use_dcfifo" || $data_enum == "use_source_synchronous_pll"} {
			# Special case
			checkbutton ${data_frame}.${data_enum}_frame.checkbutton -variable [namespace which -variable s_$data_enum] -command "[namespace code on_check] $clocks_window $next_button"
		} elseif {$browser_node_type != ""} {
			# Instance Name
			entry ${data_frame}.${data_enum}_frame.entry -width 20
			Button ${data_frame}.${data_enum}_frame.browser_button -text "..." -helptext "Select name in the name browser" -width 3 -command "[namespace code on_name_browser] $clocks_window $next_button \"$data_type\""
		} else {
			# Numbers only
			if {[is_data_type_with_units $data_type]} {
				set data_units [lindex $data_type $COMBOBOX_LIST]
				set data_units_default [lindex $data_type $DEFAULT_COMBOBOX_INDEX]
				set data_value_default [lindex $data_type $DEFAULT_VALUE]
				entry ${data_frame}.${data_enum}_frame.entry -width 8 -justify right
				${data_frame}.${data_enum}_frame.entry insert 0 $data_value_default
				ComboBox ${data_frame}.${data_enum}_frame.units -width 4 -values $data_units -editable 0
				${data_frame}.${data_enum}_frame.units setvalue $data_units_default
			} else {
				entry ${data_frame}.${data_enum}_frame.entry -width 6 -justify right
			}
			set validation_proc [lindex $data_type $VALIDATION_PROC]
			if {$validation_proc != ""} {
				${data_frame}.${data_enum}_frame.entry configure -validate all -validatecommand "[namespace code $validation_proc] %P %V"
			}
		}
	}
	set canvas_window ${clocks_window}.canvas
	
	foreach data_type $s_clocks_data_types {
		set data_enum [lindex $data_type $VARIABLE_NAME]
		set data_indent [lindex $data_type $INDENT]
		set data_rows [lindex $data_type $ROWS]
		set browser_node_type [lindex $data_type $NODE_TYPE]
		if {$data_enum == "is_clk_fedback_in" || $data_enum == "use_dcfifo" || $data_enum == "use_source_synchronous_pll"} {
			grid columnconfigure ${data_frame}.${data_enum}_frame 0 -weight 0 -minsize $data_indent
			grid columnconfigure ${data_frame}.${data_enum}_frame 1 -weight 0
			grid columnconfigure ${data_frame}.${data_enum}_frame 2 -weight 1
			grid configure ${data_frame}.${data_enum}_frame.checkbutton -row 0 -column 1 -sticky w
			grid configure ${data_frame}.${data_enum}_frame.label -row 0 -column 2 -sticky w	
		} elseif {$data_rows == 2} {
			grid columnconfigure ${data_frame}.${data_enum}_frame 0 -weight 0 -minsize $data_indent
			grid columnconfigure ${data_frame}.${data_enum}_frame 1 -weight 1
			grid rowconfigure ${data_frame}.${data_enum}_frame 0 -weight 1
			grid rowconfigure ${data_frame}.${data_enum}_frame 1 -weight 1

			grid configure ${data_frame}.${data_enum}_frame.label -row 0 -column 1 -columnspan 2 -sticky w
			grid configure ${data_frame}.${data_enum}_frame.entry -row 1 -column 1 -sticky ew
			if {$browser_node_type != ""} {
				grid columnconfigure ${data_frame}.${data_enum}_frame 2 -weight 0
				grid configure ${data_frame}.${data_enum}_frame.browser_button -row 1 -column 2 -sticky e
			}
		} else {
			grid columnconfigure ${data_frame}.${data_enum}_frame 0 -weight 0 -minsize $data_indent
			grid columnconfigure ${data_frame}.${data_enum}_frame 1 -weight 1
			grid columnconfigure ${data_frame}.${data_enum}_frame 2 -weight 0
			grid configure ${data_frame}.${data_enum}_frame.label -row 0 -column 1 -sticky w
			grid configure ${data_frame}.${data_enum}_frame.entry -row 0 -column 2 -sticky e
			if {$browser_node_type != ""} {
				grid columnconfigure ${data_frame}.${data_enum}_frame 3 -weight 0
				grid configure ${data_frame}.${data_enum}_frame.browser_button -row 0 -column 3 -sticky e
			} elseif {[is_data_type_with_units $data_type]} {
				grid columnconfigure ${data_frame}.${data_enum}_frame 3 -weight 0
				grid configure ${data_frame}.${data_enum}_frame.units -row 0 -column 3 -sticky e
			}
		}
		pack ${data_frame}.${data_enum}_frame -side top -fill x -expand 1
		if {[winfo exists ${data_frame}.${data_enum}_frame.entry]} {
			bind ${data_frame}.${data_enum}_frame.entry <KeyRelease> "[namespace code on_entry_keyrelease] $clocks_window $next_button" 
		}
	}
	pack ${clocks_window}.data_frame -side top -fill both -expand 0

	bind $clocks_window <Map> "[namespace code update_next_state] $clocks_window $next_button"

	foreach data_type $s_clocks_data_types {
		set data_enum [lindex $data_type $VARIABLE_NAME]
		set data_widget [lindex $data_type $WIDGET]
		set browser_node_type [lindex $data_type $NODE_TYPE]
		if {$data_widget != ""} {
			bind ${data_frame}.${data_enum}_frame.entry <FocusIn> "[namespace code select_widget] ${data_frame}.${data_enum}_frame.entry $canvas_window $data_widget"
			bind ${data_frame}.${data_enum}_frame.entry <ButtonPress> "[namespace code select_widget] ${data_frame}.${data_enum}_frame.entry $canvas_window $data_widget"
			bind ${data_frame}.${data_enum}_frame.entry <FocusOut> "[namespace code deselect_widget] ${data_frame}.${data_enum}_frame.entry $canvas_window $data_widget"
			if {$browser_node_type != ""} {
				bind ${data_frame}.${data_enum}_frame.browser_button <FocusIn> "[namespace code select_widget] ${data_frame}.${data_enum}_frame.browser_button $canvas_window $data_widget"
				bind ${data_frame}.${data_enum}_frame.browser_button <FocusOut> "[namespace code deselect_widget] ${data_frame}.${data_enum}_frame.browser_button $canvas_window $data_widget"
			}
		}
	}

	return $clocks_window
}

# ----------------------------------------------------------------
#
proc dtw_clocks::save_data {clocks_window data_array_name} {
#
# Description: Save the data in this panel in the data_array
#
# ----------------------------------------------------------------
	variable s_clocks_data_types
	upvar $data_array_name data_array
	variable VARIABLE_NAME

	# Data in the panel is saved in array:
	# set data_array(data_enum0) = {value0}
	# set data_array(data_enum1) = {value1}
	# ...
	foreach data_type $s_clocks_data_types {
		set data_enum [lindex $data_type $VARIABLE_NAME]
		set data_value [get_data_value $clocks_window $data_enum]
		if {$data_enum == "is_clk_fedback_in" || $data_enum == "use_dcfifo" || $data_enum == "use_source_synchronous_pll"} {
			set state [${clocks_window}.data_frame.${data_enum}_frame.checkbutton cget -state]
		} else {
			set state [${clocks_window}.data_frame.${data_enum}_frame.entry cget -state]
		}
		if {$state == "normal"} {
			if {[is_data_type_with_units $data_type]} {
				set data_units [${clocks_window}.data_frame.${data_enum}_frame.units cget -text]
				set data_array($data_enum) [list $data_value $data_units]
			} else {
				set data_array($data_enum) $data_value
			}
		} else {
			array unset data_array $data_enum
		}
	}
	return
}

# ----------------------------------------------------------------
#
proc dtw_clocks::load_data {clocks_window data_array_name} {
#
# Description: Load the data in this panel from the data_array
#
# ----------------------------------------------------------------
	variable s_project_name
	variable s_revision_name
	variable s_clocks_data_types
	variable VARIABLE_NAME
	variable s_is_clk_fedback_in
	variable s_use_source_synchronous_pll
	variable s_use_dcfifo
	variable s_use_hardware_dqs
	variable s_memory_type
	variable s_tCK
	upvar $data_array_name data_array

	set s_project_name [file tail "$data_array(project_path)"]
	set s_revision_name $data_array(project_revision)
	set s_use_hardware_dqs $data_array(use_hardware_dqs)
	set s_memory_type $data_array(memory_type)
	::dtw::dtw_device_get_family_parameter "_default" ${s_memory_type}_user_terms mem_user_term_list
	array set mem_user_term $mem_user_term_list
	set tCK_var $mem_user_term(tCK_var)
	set s_tCK [expr "[::dtw::dtw_timing::get_ns $data_array($tCK_var)]"]

	set data_frame ${clocks_window}.data_frame
	foreach data_type $s_clocks_data_types {
		set data_enum [lindex $data_type $VARIABLE_NAME]
		if {[array names data_array -exact $data_enum] != ""} {
			set data_value [lindex $data_array($data_enum) 0]
			if {$data_enum == "is_clk_fedback_in" || $data_enum == "use_dcfifo" || $data_enum == "use_source_synchronous_pll"} {
				set s_$data_enum $data_value
			} else {
				${data_frame}.${data_enum}_frame.entry delete 0 end
				${data_frame}.${data_enum}_frame.entry insert 0 $data_value
			}
			if {[is_data_type_with_units $data_type]} {
				set data_units [lindex $data_array($data_enum) 1]
				${clocks_window}.data_frame.${data_enum}_frame.units configure -text $data_units
			}
		}
	}

	${data_frame}.clk_sys_frame.label configure -text "Name of the PLL output driving the $mem_user_term(ck_ckn) system clock pin(s)"
	${data_frame}.pll_mult_frame.label configure -text "PLL frequency multiplier of $mem_user_term(ck_ckn) clock"
	${data_frame}.pll_div_frame.label configure -text "PLL frequency divider of $mem_user_term(ck_ckn) clock"
	${data_frame}.clk_resync_frame.label configure -text "Name of the PLL output re-synchronizing read data from the $mem_user_term(read_dq) capture registers"

	if {$s_use_hardware_dqs == 0} {
		${data_frame}.clk_resync_frame.label configure -text "Name of the PLL output capturing read data into the $mem_user_term(read_dq) capture registers"
		${data_frame}.resync_cycle_frame.label configure -text "Capture read data in cycle"
		${data_frame}.clk_resync2_frame.label configure -text "Name of the System PLL output re-synchronizing read data from the $mem_user_term(read_dq) capture registers"
	}
	on_check $clocks_window ""
	return
}

# ----------------------------------------------------------------
#
proc dtw_clocks::on_entry_keyrelease { clocks_window next_button } {
#
# Description: Handles the keyboard event of an entry control
#
# ----------------------------------------------------------------
	update_next_state $clocks_window $next_button
}

# ----------------------------------------------------------------
#
proc dtw_clocks::can_go_to_next_state { clocks_window } {
#
# Description: Tells if the "Next" button should move to the next panel
#
# ----------------------------------------------------------------
	variable s_clocks_data_types
	variable s_is_clk_fedback_in
	variable VARIABLE_NAME
	variable OPTIONAL

	set enable 1
	foreach data_type $s_clocks_data_types {
		set data_enum [lindex $data_type $VARIABLE_NAME]
		set data_optional [lindex $data_type $OPTIONAL]
		set data_value [get_data_value $clocks_window $data_enum]
		if {$data_enum == "is_clk_fedback_in" || $data_enum == "use_dcfifo" || $data_enum == "use_source_synchronous_pll"} {
			set state [${clocks_window}.data_frame.${data_enum}_frame.checkbutton cget -state]
		} else {
			set state [${clocks_window}.data_frame.${data_enum}_frame.entry cget -state]
		}
		if {$state == "normal"} {
			if {$data_value != "" || $data_optional == 1} {
				# data good
			} elseif {$data_optional == 0} {
				# disable Next button
				set enable 0
			} else {
				error "Unknown optional value"
			}
		}
	}
	if {$enable == 1} {
		set postamble_data_enums [list clk_read_postamble postamble_cycle postamble_phase inter_postamble_cycle inter_postamble_phase]

		# If any postamble parameters are entered, then all other
		# postamble parameters are required
		set do_postamble_analysis 0
		foreach data_enum $postamble_data_enums {
			set state [${clocks_window}.data_frame.${data_enum}_frame.entry cget -state]
			if {$state == "normal" && [get_data_value $clocks_window $data_enum] != ""} {
				set do_postamble_analysis 1
			}
		}
		if {$do_postamble_analysis == 1} {
			foreach data_enum $postamble_data_enums {
				set state [${clocks_window}.data_frame.${data_enum}_frame.entry cget -state]
				if {$state == "normal" && [get_data_value $clocks_window $data_enum] == ""} {
					set enable 0
				}
			}
		}
	}
	return $enable
}

# ----------------------------------------------------------------
#
proc dtw_clocks::update_next_state { clocks_window next_button } {
#
# Description: Tells if the "Next" button should be enabled and changes its
#              state accordingly
#
# ----------------------------------------------------------------
	if {[can_go_to_next_state $clocks_window] == 0} {
		$next_button configure -state disabled
	} else {
		$next_button configure -state normal
	}
	return
}

# ----------------------------------------------------------------
#
proc dtw_clocks::on_name_browser { clocks_window next_button data_type } {
#
# Description: Get clocks from the name browser
#
# ----------------------------------------------------------------
	variable s_project_name
	variable s_revision_name
	variable VARIABLE_NAME
	variable NODE_TYPE
	variable NAME_BROWSER_FILTER

	set data_enum [lindex $data_type $VARIABLE_NAME]
	set data_node_type [lindex $data_type $NODE_TYPE]
	set data_filter [lindex $data_type $NAME_BROWSER_FILTER]
	set data_value [${clocks_window}.data_frame.${data_enum}_frame.entry get]
	if {$data_value != ""} {
		set clock_list [list $data_value]
	} else {
		set clock_list [list]
	}
	::dtw::dtw_name_browser::get_names_from_browser $s_project_name $s_revision_name $data_node_type post_synthesis "" clock_list $data_filter 1
	${clocks_window}.data_frame.${data_enum}_frame.entry delete 0 end
	${clocks_window}.data_frame.${data_enum}_frame.entry insert 0 [lindex $clock_list 0]
	update_next_state $clocks_window $next_button
}

# ----------------------------------------------------------------
#
proc dtw_clocks::validate_positive_int { number validation_type } {
#
# Description: Validate the number is a positive integer
#
# Returns: true if it is, false otherwise
#
# ----------------------------------------------------------------
	if {$number == "" || ([string is integer $number] && $number > 0)} {
		set valid 1
	} else {
		set valid 0
	}

	return $valid
}

# ----------------------------------------------------------------
#
proc dtw_clocks::validate_int { number validation_type } {
#
# Description: Validate the number is a positive integer
#
# Returns: true if it is, false otherwise
#
# ----------------------------------------------------------------
	if {$number == "" || ($number == "-" && $validation_type != "focusout") || [string is integer $number]} {
		set valid 1
	} else {
		set valid 0
	}

	return $valid
}

# ----------------------------------------------------------------
#
proc dtw_clocks::validate_float { number validation_type } {
#
# Description: Validate the number is a float
#
# Returns: true if it is, false otherwise
#
# ----------------------------------------------------------------
	if {$number == "" || ($number == "." && $validation_type != "focusout")} {
		set valid 1
	} else {
		set valid [string is double $number]
	}
	return $valid
}

# ----------------------------------------------------------------
#
proc dtw_clocks::on_check { clocks_window next_button } {
#
# Description: Called when the user clicks a checkbutton
#
# ----------------------------------------------------------------
	variable s_clocks_data_types
	variable VARIABLE_NAME
	variable NODE_TYPE
	variable s_data_enum_list
	variable s_forget_list

	# Update s_data_enum_list and s_forget_list
	update_data_enum_list
	
	set data_frame ${clocks_window}.data_frame
	foreach data_enum $s_forget_list {
		pack forget ${data_frame}.${data_enum}_frame
	}
	foreach data_type $s_clocks_data_types {
		set data_enum [lindex $data_type $VARIABLE_NAME]
		set browser_node_type [lindex $data_type $NODE_TYPE]

		if {[lsearch -exact $s_data_enum_list $data_enum] != -1} {
			set state "normal"
		} else {
			set state "disabled"
		}

		if {$data_enum == "is_clk_fedback_in" || $data_enum == "use_dcfifo" || $data_enum == "use_source_synchronous_pll"} {
			${data_frame}.${data_enum}_frame.checkbutton configure -state $state
		} elseif {$browser_node_type != ""} {
			${data_frame}.${data_enum}_frame.entry configure -state $state
			${data_frame}.${data_enum}_frame.browser_button configure -state $state
		} else {
			if {[is_data_type_with_units $data_type]} {
				${data_frame}.${data_enum}_frame.entry configure -state $state
				${data_frame}.${data_enum}_frame.units configure -state $state
			} else {
				${data_frame}.${data_enum}_frame.entry configure -state $state
			}
		}
	}
	update_drawing ${clocks_window}.canvas

	if {$next_button != ""} {
		update_next_state $clocks_window $next_button
	}
}

# ----------------------------------------------------------------
#
proc dtw_clocks::update_drawing { canvas_window } {
#
# Description: Called to update the drawing
#
# ----------------------------------------------------------------
	variable s_memory_type
	
	::dtw::dtw_device_get_family_parameter "_default" ${s_memory_type}_user_terms mem_user_term_list
	array set mem_user_term $mem_user_term_list

	if {$s_memory_type == "ddr"} {
		variable s_is_clk_fedback_in
		variable s_use_hardware_dqs
		variable s_widgets

		set s_widgets [::dtw::dtw_main::dtw_circuit::draw_circuit $canvas_window $s_use_hardware_dqs $s_is_clk_fedback_in 0 0 $mem_user_term(ck) $mem_user_term(read_dqs) $mem_user_term(read_dq) $mem_user_term(inverted_capture)]
	} elseif {$s_memory_type == "qdr2" || $s_memory_type == "rldram2"} {
		variable s_use_source_synchronous_pll
		variable s_use_hardware_dqs
		variable s_widgets
		variable s_use_dcfifo

		set s_widgets [::dtw::dtw_main::dtw_circuit::draw_circuit $canvas_window $s_use_hardware_dqs 0 $s_use_source_synchronous_pll $s_use_dcfifo $mem_user_term(ck) $mem_user_term(read_dqs) $mem_user_term(read_dq) $mem_user_term(inverted_capture)]
	} else {
		# No drawing
		set s_widgets [list]
	}
}

# ----------------------------------------------------------------
#
proc dtw_clocks::get_data_value { clocks_window data_enum } {
#
# Description: Called when the user clicks a checkbutton
#
# ----------------------------------------------------------------
	variable s_is_clk_fedback_in
	variable s_use_source_synchronous_pll
	variable s_use_dcfifo

	if {$data_enum == "is_clk_fedback_in"} {
		set data_value $s_is_clk_fedback_in
	} elseif {$data_enum == "use_source_synchronous_pll"} {
		set data_value $s_use_source_synchronous_pll
	} elseif {$data_enum == "use_dcfifo"} {
		set data_value $s_use_dcfifo
	} else {
		set data_value [${clocks_window}.data_frame.${data_enum}_frame.entry get]
	}
	return $data_value
}

# ----------------------------------------------------------------
#
proc dtw_clocks::select_widget {data_window canvas_window widget_enum} {
#
# Description: Select the given widget
#
# ----------------------------------------------------------------
	variable s_widgets
	array set widget_array $s_widgets
	if {[$data_window cget -state] == "normal"} {
		if {[array names widget_array -exact $widget_enum] != ""} {
			set widget_id [lindex $widget_array($widget_enum) 0]
			if {[llength $widget_array($widget_enum)] > 1} {
				# Select polygon
				$canvas_window itemconfigure $widget_id -outline blue -width 3
			} else {
				# Select line
				$canvas_window itemconfigure $widget_id -fill blue -width 3
			}
		} else {
			puts "Can't select widget $widget_enum"
		}
	}
}

# ----------------------------------------------------------------
#
proc dtw_clocks::deselect_widget {data_window canvas_window widget_enum} {
#
# Description: Un-select the given widget
#
# ----------------------------------------------------------------
	variable s_widgets
	array set widget_array $s_widgets
	if {[array names widget_array -exact $widget_enum] != ""} {
		set widget_id [lindex $widget_array($widget_enum) 0]
		if {[llength $widget_array($widget_enum)] > 1} {
			# Deselect polygon
			$canvas_window itemconfigure $widget_id -outline black -width 1
		} else {
			# Deselect line
			$canvas_window itemconfigure $widget_id -fill black -width 1
		}
	} else {
		if {[$data_window cget -state] == "normal"} {
			puts "Can't deselect widget $widget_enum"
		}
	}
}

# ----------------------------------------------------------------
#
proc dtw_clocks::on_next { clocks_window } {
#
# Description: Handles the "next" button event
#
# ----------------------------------------------------------------
	variable s_tCK

	if {[can_go_to_next_state $clocks_window]} {
		# Make sure the PLL settings (frequency multiplier and divider) make sense
		set pll_inclk_value [get_data_value $clocks_window "pll_input_freq"]
		set pll_mult_value [get_data_value $clocks_window "pll_mult"]
		set pll_div_value [get_data_value $clocks_window "pll_div"]
		set pll_inclk_units [${clocks_window}.data_frame.pll_input_freq_frame.units cget -text]
		set pll_inclk_raw [list $pll_inclk_value $pll_inclk_units]

		set inclk_period [::dtw::dtw_timing::get_ns $pll_inclk_raw]
		set outclk_period [expr "$inclk_period * $pll_div_value / $pll_mult_value"]
		if {[expr "abs($outclk_period - $s_tCK)/$s_tCK"] > .01} {
			::dtw::msg_o "Error: Inconsistent PLL Settings" "PLL output clock period $outclk_period ns is not close to the tCK clock period $s_tCK ns.  Make sure the correct PLL input frequency, multiplier, and divider have been entered."
			set result "none"
		} else {
			set result "next"
		}
	} else {
		::dtw::msg_o "Error" "Missing required PLL clock info"
		set result "none"
	}
	return $result
}

# ----------------------------------------------------------------
#
namespace eval dtw_circuit {
#
# Description: Namespace to encapsulate the Memory Data panel
#
# ----------------------------------------------------------------
	variable pll_cx 50
	variable pll_cy 40
	variable dff_cx 45
	variable dff_cy 50
	variable pin_cx 75
	variable pin_cy 14
}

# ----------------------------------------------------------------
#
proc dtw_circuit::draw_circuit { canvas_window use_hardware_dqs use_feedback use_source_synchronous_pll use_dcfifo ck_string dqs_string dq_string is_inverted_capture} {
#
# Description: Draws a diagram of the resync circuit w/ feedback
#
# ----------------------------------------------------------------
	variable pll_cx
	variable pll_cy
	variable dff_cx
	variable dff_cy
	variable pin_cy

	set pin_x 200
	set memory_x 80
	set capture_x [expr "$pin_x + 30"]
	set capture_y 20
	set dqs_y [expr "$capture_y + $dff_cy * 2 / 3"]
	set feedback_pll_x [expr "$pin_x + 30"]
	set feedback_pll_y [expr "$capture_y + $dff_cy + 5"]
	set system_pll_x $feedback_pll_x
	set system_pll_y [expr "$feedback_pll_y + $pll_cy + 2 * $pin_cy + 5"]
	set resync_x [expr "$capture_x + $dff_cx + 50"]
	set resync_y $capture_y
	set resync2_x [expr "$resync_x + $dff_cx + 50"]
	set resync2_y $capture_y
	set sys_clk_pin_y [expr "$system_pll_y - $pin_cy/2"]
	set feedback_out_pin_y [expr "$sys_clk_pin_y - $pin_cy - 5"]
	if {$use_dcfifo == 1} {
		set resync_string "Dual-\nClock\nFIFO"
	} else {
		set resync_string "Resync\nDFF"
	}

	create_canvas $canvas_window

	canvas_fpga $canvas_window [expr "$pin_x - 40"]

	array set widget_array [list]
	set widget_array(memory_widget) [canvas_memory $canvas_window $memory_x $sys_clk_pin_y $ck_string $dqs_y $dqs_string]

	set widget_array(sys_pll_widget) [canvas_pll $canvas_window $system_pll_x $system_pll_y "System\nPLL"]
	set widget_array(pll_inclk_pin_widget) [canvas_input_pin $canvas_window $pin_x [expr $system_pll_y + $pll_cy / 2] "clk"]
	set widget_array(pll_inclk_wire_widget) [canvas_wire_xyx $canvas_window $widget_array(pll_inclk_pin_widget) "out" $widget_array(sys_pll_widget) "inclk" 0 0.5]

	if {$use_hardware_dqs || $use_source_synchronous_pll} {
		set widget_array(dqs_pin_widget) [canvas_input_pin $canvas_window $pin_x $dqs_y $dqs_string]
		set widget_array(board_dqs_wire_widget) [canvas_wire_xyx $canvas_window $widget_array(memory_widget) $dqs_string $widget_array(dqs_pin_widget) "in" 0 0.5]
	}
	if {$use_hardware_dqs} {
		set widget_array(capture_dff_widget) [canvas_dff $canvas_window $capture_x $capture_y "$dq_string read\ncapture"]
		if {$use_feedback} {
			set widget_array(resync_dff_widget) [canvas_dff $canvas_window $resync_x $resync_y "$resync_string 1"]
			set widget_array(resync2_dff_widget) [canvas_dff $canvas_window $resync2_x $resync2_y "$resync_string 2"]
			set widget_array(resync_2_resync2_wire_widget) [canvas_wire_xyx $canvas_window $widget_array(resync_dff_widget) "q" $widget_array(resync2_dff_widget) "d" 0 .5]
		} else {
			set widget_array(resync_dff_widget) [canvas_dff $canvas_window $resync_x $resync_y $resync_string]
		}
		set widget_array(dqs_2_capture_wire_widget) [canvas_wire_xyx $canvas_window $widget_array(dqs_pin_widget) "out" $widget_array(capture_dff_widget) "clk" $is_inverted_capture .5]
		set widget_array(capture_2_resync_wire_widget) [canvas_wire_xyx $canvas_window $widget_array(capture_dff_widget) "q" $widget_array(resync_dff_widget) "d" 0 .5]
	} else {
		# non-DQS mode
		if {$use_feedback || $use_source_synchronous_pll} {
			set widget_array(resync_dff_widget) [canvas_dff $canvas_window $resync_x $resync_y "$dq_string read\ncapture"]
			set widget_array(resync2_dff_widget) [canvas_dff $canvas_window $resync2_x $resync2_y $resync_string]
			set widget_array(resync_2_resync2_wire_widget) [canvas_wire_xyx $canvas_window $widget_array(resync_dff_widget) "q" $widget_array(resync2_dff_widget) "d" 0 .5]
		} else {
			set widget_array(resync_dff_widget) [canvas_dff $canvas_window $resync_x $resync_y "$dq_string read\ncapture"]
		}
	}

	if {$use_feedback || ($use_hardware_dqs == 0 && $use_source_synchronous_pll)} {
		set widget_array(resync_pll_widget) [canvas_pll $canvas_window $feedback_pll_x $feedback_pll_y "Fedback\nPLL"]
		if {$use_feedback} {
			set widget_array(feedback_output_pin_widget) [canvas_output_pin $canvas_window $pin_x $feedback_out_pin_y "feedback_out"]
			set widget_array(fedback_clk_pin_widget) [canvas_input_pin $canvas_window $pin_x [expr $feedback_pll_y + $pll_cy / 2] "fedback_in"]
			set widget_array(fedback_clk_2_pll_wire_widget) [canvas_wire_xyx $canvas_window $widget_array(fedback_clk_pin_widget) "out" $widget_array(resync_pll_widget) "inclk" 0 0.5]
			set widget_array(fedback_clk_wire_widget) [canvas_wire_xyx $canvas_window $widget_array(feedback_output_pin_widget) "out" $widget_array(fedback_clk_pin_widget) "in" 0 -2.0]
			set widget_array(pll_2_feedback_output_wire_widget) [canvas_wire_xyx $canvas_window $widget_array(sys_pll_widget) "c0" $widget_array(feedback_output_pin_widget) "in" 0 .2]
		} elseif {$use_source_synchronous_pll} {
			set widget_array(fedback_clk_2_pll_wire_widget) [canvas_wire_xyx $canvas_window $widget_array(dqs_pin_widget) "out" $widget_array(resync_pll_widget) "inclk" 0 0.5]
		}

		set widget_array(pll_2_resync_wire_widget) [canvas_wire_xyx $canvas_window $widget_array(resync_pll_widget) "c0" $widget_array(resync_dff_widget) "clk" 0 .5]
		if {$use_dcfifo} {
			set widget_array(pll_2_fifo_clk_wire_widget) [canvas_wire_xyx $canvas_window $widget_array(resync_pll_widget) "c0" $widget_array(resync2_dff_widget) "clk" 0 .8]
		} else {
			set widget_array(pll_2_resync2_clk_wire_widget) [canvas_wire_xyx $canvas_window $widget_array(sys_pll_widget) "c1" $widget_array(resync2_dff_widget) "clk" 0 .8]
		}
	} else {
		if {$use_hardware_dqs && $use_dcfifo} {
			set widget_array(pll_2_fifo_wire_widget) [canvas_wire_xyx $canvas_window $widget_array(dqs_pin_widget) "out" $widget_array(resync_dff_widget) "clk" 0 .5]
		} else {
			set widget_array(pll_2_resync_wire_widget) [canvas_wire_xyx $canvas_window $widget_array(sys_pll_widget) "c1" $widget_array(resync_dff_widget) "clk" 0 .5]
		}
	}

	set widget_array(sys_output_pin_widget) [canvas_output_pin $canvas_window $pin_x $sys_clk_pin_y "sys_clk"]
	set widget_array(pll_2_sys_output_wire_widget) [canvas_wire_xyx $canvas_window $widget_array(sys_pll_widget) "c0" $widget_array(sys_output_pin_widget) "in" 0 .2]
	set widget_array(board_ck_wire_widget) [canvas_wire_xyx $canvas_window $widget_array(memory_widget) $ck_string $widget_array(sys_output_pin_widget) "out" 0 0.1]

	pack $canvas_window -side top -fill both -expand 1

	set result [array get widget_array]
	return $result
}

# ----------------------------------------------------------------
#
proc dtw_circuit::canvas_pll { canvas_window x y text } {
#
# Description: Creates a PLL on the specified canvas at (x,y)
#
# ----------------------------------------------------------------
	variable pll_cx
	variable pll_cy

	set cx $pll_cx
	set cy $pll_cy
	set arrow_size 6
	set right [expr "$x + $cx"]
	set bottom [expr "$y + $cy"]
	set port_list [construct_ports [list "inclk"] [list "c0" "c1"] $x $right $y $cy]

	set result [list [$canvas_window create rectangle $x $y $right $bottom -outline black -fill white] $port_list]

	set clk_pos [get_port_position $result "inclk"]
	set clk_x [lindex $clk_pos 0]
	set clk_y [lindex $clk_pos 1]
	$canvas_window create line $clk_x [expr $clk_y - $arrow_size] [expr $clk_x + $arrow_size] $clk_y $clk_x [expr $clk_y + $arrow_size]

	set pll_text_widget [$canvas_window create text [expr $x + 6] [expr $y + $cy / 2] -text $text -anchor w -justify left]

	return $result
}

# ----------------------------------------------------------------
#
proc dtw_circuit::canvas_fpga { canvas_window x } {
#
# Description: Draws an FPGA border on the specified canvas at x
#
# ----------------------------------------------------------------
	set color LightBlue

	set result [$canvas_window create rectangle $x 0 2000 2000 -outline $color -fill $color]
	$canvas_window create text [expr $x + 10] 1 -text "FPGA (Read Data Re-synchronization Circuitry)" -anchor nw -justify left

	return $result
}

# ----------------------------------------------------------------
#
proc dtw_circuit::canvas_memory { canvas_window x ck_y ck_string dqs_y dqs_string} {
#
# Description: Creates a memory at x
#
# ----------------------------------------------------------------
	set color gray60
	set port_size [expr 3*($ck_y - $dqs_y)]
	set port_pos [expr $dqs_y - $port_size / 3]
	set ports [construct_ports [list] [list $dqs_string $ck_string] 0 $x $port_pos $port_size]


	set result [list [$canvas_window create rectangle 0 0 $x 2000 -fill $color] $ports]
	$canvas_window create text [expr $x - 10] 1 -text "Memory" -anchor ne -justify right
	label_port $canvas_window $result $dqs_string e
	label_port $canvas_window $result $ck_string e

	return $result
}

# ----------------------------------------------------------------
#
proc dtw_circuit::construct_ports { inputs outputs in_x out_x y cy } {
#
# Description: Creates a port list
#
# ----------------------------------------------------------------
	set result [list]
	set inputs_p_1 [expr [llength $inputs] + 1]
	set i 1
	foreach in_port $inputs {
		lappend result [list "$in_port" $in_x [expr $y + $cy * $i / $inputs_p_1]]
		incr i
	}
	set i 1
	set outputs_p_1 [expr [llength $outputs] + 1]
	foreach out_port $outputs {
		lappend result [list "$out_port" $out_x [expr $y + $cy * $i / $outputs_p_1]]
		incr i
	}
	return $result
}

# ----------------------------------------------------------------
#
proc dtw_circuit::canvas_dff { canvas_window x y label} {
#
# Description: Creates a DFF on the specified canvas at (x,y)
#
# ----------------------------------------------------------------
	variable dff_cx
	variable dff_cy

	set cx $dff_cx
	set cy $dff_cy
	set arrow_size 6
	set right [expr $x+$cx]
	set bottom [expr $y+$cy]
	set port_list [construct_ports [list "d" "clk"] [list "q"] $x $right $y $cy]

	set result [list [$canvas_window create rectangle $x $y $right $bottom -outline black -fill white] $port_list]
	set clk_pos [get_port_position $result "clk"]
	set clk_x [lindex $clk_pos 0]
	set clk_y [lindex $clk_pos 1]
	$canvas_window create line $clk_x [expr $clk_y - $arrow_size] [expr $clk_x + $arrow_size] $clk_y $clk_x [expr $clk_y + $arrow_size]

	set dff_text_widget [$canvas_window create text [expr $x + $cx / 2] [expr $y + $cy / 2] -text "$label" -anchor center -justify center]
	return $result
}

# ----------------------------------------------------------------
#
proc dtw_circuit::canvas_input_pin { canvas_window x y label } {
#
# Description: Creates an input pin on the specified canvas at (x,y)
#
# ----------------------------------------------------------------
	variable pin_cx
	variable pin_cy
	set cx $pin_cx
	set cy $pin_cy
	set midx [expr $x - $cy / 2]
	set midy [expr $y + $cy / 2]
	set left [expr $x - $cx]
	set bottom [expr $y + $cy / 2]
	set top [expr $y - $cy / 2]
	set port_list [construct_ports [list "in"] [list "out"] $left $x $y 0]

	set result [list [$canvas_window create polygon $x $y $midx $midy $left $bottom $left $top $midx $top -fill white -outline black] $port_list]
	set pin_text_widget [$canvas_window create text [expr $x - $cx / 2] $y -text "$label" -anchor center]

	return $result
}

# ----------------------------------------------------------------
#
proc dtw_circuit::canvas_output_pin { canvas_window x y label } {
#
# Description: Creates an output pin on the specified canvas at (x,y)
#
# ----------------------------------------------------------------
	variable pin_cx
	variable pin_cy
	set cx $pin_cx
	set cy $pin_cy
	set left [expr $x - $cx]
	set midx [expr $left + $cy / 2]
	set right [expr $x]
	set top [expr $y - $cy / 2]
	set midy [expr $y]
	set bottom [expr $y + $cy / 2]
	set port_list [construct_ports [list "in"] [list "out"] $x $left $y 0]

	set result [list [$canvas_window create polygon $right $top $right $bottom $midx $bottom $left $midy $midx $top -fill white -outline black] $port_list]
	set pin_text_widget [$canvas_window create text [expr $x - $cx / 2] $y -text "$label" -anchor center]

	return $result
}

# ----------------------------------------------------------------
#
proc dtw_circuit::get_port_position { widget port } {
#
# Description: Gets the widget's port x,y position
#
# Returns: List of result [list x y]
#
# ----------------------------------------------------------------
	set pos [list 0 0]
	# Ports are stored as [list port x y ...]
	foreach widget_port [lindex $widget 1] {
		if {[lindex $widget_port 0] == $port} {
			set pos [list [lindex $widget_port 1] [lindex $widget_port 2]]
		}
	}
	if {$pos == [list 0 0]} {
		error "Bad port $port on widget $widget"
	}
	return $pos
}

# ----------------------------------------------------------------
#
proc dtw_circuit::canvas_wire_xyx { canvas_window src_widget out_port dest_widget in_port bubble kink } {
#
# Description: Connects the two ports with a wire line
#
# ----------------------------------------------------------------
	set src_pos [get_port_position $src_widget $out_port]
	set dest_pos [get_port_position $dest_widget $in_port]
	set src_x [lindex $src_pos 0]
	set src_y [lindex $src_pos 1]
	set dest_x [lindex $dest_pos 0]
	set dest_y [lindex $dest_pos 1]
	set x_distance [expr "abs($dest_x - $src_x)"]
	if {$x_distance < 10} {
		set x_distance 10
	}
	set mid_x [expr "$src_x + $x_distance * $kink"]
	if {$bubble} {
		set bubble_size 6
		set bubble_left [expr "$dest_x - $bubble_size"]
		set bubble_top [expr "$dest_y - $bubble_size / 2"]
		set bubble_right $dest_x
		set bubble_bottom [expr "$dest_y + $bubble_size / 2"]
		$canvas_window create oval $bubble_left $bubble_top $bubble_right $bubble_bottom
		set dest_x $bubble_left
	}
	return [$canvas_window create line $src_x $src_y $mid_x $src_y $mid_x $dest_y $dest_x $dest_y]
}

# ----------------------------------------------------------------
#
proc dtw_circuit::create_canvas { canvas_window } {
#
# Description: Creates the circuit drawing canvas
#
# ----------------------------------------------------------------
	# Make sure there's no previous canvas
	pack forget $canvas_window
	destroy $canvas_window

	canvas $canvas_window -background gray80
	return $canvas_window
}

# ----------------------------------------------------------------
#
proc dtw_circuit::label_port { canvas_window widget port anchor } {
#
# Description: Labels the port on the widget
#
# ----------------------------------------------------------------
	set port_pos [get_port_position $widget $port]
	$canvas_window create text [lindex $port_pos 0] [lindex $port_pos 1] -text $port -anchor $anchor
}

