if {[namespace exists ::dtw]} {
	::dtw::add_version_date {$Date:   23 Mar 2006 14:35:54  $}
}

##############################################################################
#
# File Name:    dtw_auto_detect.tcl
#
# Summary:      This TK script is a simple Graphical User Interface to
#               generate timing requirements for DDR memory interfaces
#
# Licencing:
#               ALTERA LEGAL NOTICE
#               
#               This script is  pursuant to the following license agreement
#               (BY VIEWING AND USING THIS SCRIPT, YOU AGREE TO THE
#               FOLLOWING): Copyright (c) 2006-2007 Altera Corporation, San Jose,
#               California, USA.  Permission is hereby granted, free of
#               charge, to any person obtaining a copy of this software and
#               associated documentation files (the "Software"), to deal in
#               the Software without restriction, including without limitation
#               the rights to use, copy, modify, merge, publish, distribute,
#               sublicense, and/or sell copies of the Software, and to permit
#               persons to whom the Software is furnished to do so, subject to
#               the following conditions:
#               
#               The above copyright notice and this permission notice shall be
#               included in all copies or substantial portions of the Software.
#               
#               THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
#               EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES
#               OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
#               NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT
#               HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY,
#               WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
#               FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
#               OTHER DEALINGS IN THE SOFTWARE.
#               
#               This agreement shall be governed in all respects by the laws of
#               the State of California and by the laws of the United States of
#               America.
#
#               
#
# Usage:
#
#               You can run this script from a command line by typing:
#                     quartus_sh --dtw
#
###############################################################################

package require ::quartus::dtw_msg
package require ::quartus::dtw_dwz

# ----------------------------------------------------------------
#
namespace eval dtw_auto_detect {
#
# Description: Top-level namespace for the auto-detection code
#
# ----------------------------------------------------------------
	variable s_dtw_dir ${quartus(tclpath)}apps/dtw/

	# Force auto-loading packages to load immediately by sourcing them,
    # otherwise their namespaces will be empty and nothing will import.
    # This is a common Tcl 8.0 bug documented in
    # http://www.wjduquette.com/tcl/namespaces.html
	source ${quartus(tclpath)}packages/dtw/dtw_msg.tcl
	namespace import ::quartus::dtw_msg::*

	source ${quartus(tclpath)}packages/dtw/dtw_dwz.tcl
	namespace import ::quartus::dtw_dwz::*

	source ${quartus(tclpath)}apps/dtw/dtw_device.tcl
	namespace import dtw_device::*
}

# ----------------------------------------------------------------
#
proc dtw_auto_detect::create_tdb_arrays { tdb_input_keeper_array_name tdb_output_array_name } {
#
# Description: Create a map of interesting TDB node ids.  This is an
#              array mapping node names to ids.
#
# ----------------------------------------------------------------
	upvar 1 $tdb_input_keeper_array_name tdb_input_keeper_array
	upvar 1 $tdb_output_array_name tdb_output_array

	array set tdb_input_keeper_array [list]
	array set tdb_output_array [list]
	foreach_in_collection node_id [get_timing_nodes -type keeper] {
		set node_name [get_timing_node_info -info name $node_id]
		set node_type [get_timing_node_info -info type $node_id]
		set number_of_fanout [llength [get_timing_node_info -info fanout_edges $node_id]]
		if {$number_of_fanout == 0 && $node_type != "reg"} {
			set tdb_output_array($node_name) $node_id
		} else {
			set tdb_input_keeper_array($node_name) $node_id
		}
	}
}

# ----------------------------------------------------------------
#
proc dtw_auto_detect::is_auto_detectable { } {
#
# Description: Tells whether or not the current project is for this family
#
# ----------------------------------------------------------------
	set family [get_dtw_family]
	set detectable 0
	dtw_device_get_family_parameter $family "is_supported" detectable

	return $detectable
}

# ----------------------------------------------------------------
#
proc dtw_auto_detect::extract_postamble_phases { data_array_name postamble_registers_list read_postamble_clk_id } {
#
# Description: Get the post-amble and inter postamble phases
#
# ----------------------------------------------------------------
	upvar 1 $data_array_name data_array

	if {$read_postamble_clk_id != -1} {
		if {[array names data_array -exact postamble_phase] != ""} {
			set data_array(postamble_phase) [get_implemented_phase $data_array(postamble_phase) $read_postamble_clk_id]
		} else {
			set data_array(postamble_phase) [get_implemented_phase "" $read_postamble_clk_id]
		}

		if {$data_array(use_hardware_dqs) && [array names data_array -exact is_clk_fedback_in] != "" && $data_array(is_clk_fedback_in) == 1} {
			# Detect the phase of the intermediate postamble register (in the
			# system clock domain) used to transfer the postamble reset signal
			# to the fedback postamble clock domain

			set postamble_control_reg [lindex $postamble_registers_list 0]
			array set inter_postamble_reg_array [list]
			traverse_fanin_up_to_depth $postamble_control_reg is_node_type_reg synch inter_postamble_reg_array 2
			set inter_postamble_registers [array names inter_postamble_reg_array]

			if {[llength $inter_postamble_registers] > 0} {
				get_clock $inter_postamble_registers "System PLL postamble" inter_postamble_clk_id
				if {$inter_postamble_clk_id != -1} {
					set inter_postamble_phase [get_timing_node_info -info phase_only $inter_postamble_clk_id]

					# Grab the inversion of inter_postamble_phase from one of the
					# intermediate postamble registers
					set inter_postamble_reg_id [lindex $inter_postamble_registers 0]
					if {[get_timing_node_info -info is_clock_inverted $inter_postamble_reg_id]} {
						set data_array(inter_postamble_phase) "[expr $inter_postamble_phase + 180] deg"
					} else {
						set data_array(inter_postamble_phase) "$inter_postamble_phase deg"
					}
				}
			}
		}
	} else {
		array unset data_array postamble_phase
		array unset data_array postamble_cycle
		array unset data_array inter_postamble_phase
		array unset data_array inter_postamble_cycle
	}
}

# ----------------------------------------------------------------
#
proc dtw_auto_detect::extract_stratixii_postamble_registers { data_array_name tdb_array_name } {
#
# Description: Look for the post-amble register
#
# ----------------------------------------------------------------
	upvar 1 $data_array_name data_array
	upvar 1 $tdb_array_name tdb_array

	array set dqs_postamble [list]

	# In Stratix II, the post-amble register is the registered input feeding
	# the DQS input's comb node that's also clocked by the comb node
	foreach dqs $data_array(dqs_list) {
		if {[array names tdb_array -exact $dqs] != ""} {
			set dqs_node_id $tdb_array($dqs)
			set dqs_fanout_list [get_timing_node_fanout $dqs_node_id]
			if {[llength $dqs_fanout_list] == 1} {
				set comb_node_id [lindex [lindex $dqs_fanout_list 0] 0]
				set comb_node_fanin_list [get_timing_node_fanin -type asynch $comb_node_id]
				set found_postamble_reg ""
				set comb_node_fanin_list_length [llength $comb_node_fanin_list]
				for {set fanin_index 0} {$fanin_index < $comb_node_fanin_list_length && $found_postamble_reg == ""} {incr fanin_index} {
					set comb_node_fanin [lindex [lindex $comb_node_fanin_list $fanin_index] 0]
					if {[get_timing_node_info -info type $comb_node_fanin] == "reg"} {
						set clock_edges [get_timing_node_info -info clock_edges $comb_node_fanin]
						foreach clock_edge $clock_edges {
							if {[get_timing_edge_info -info src_node $clock_edge] == $comb_node_id} {
								set found_postamble_reg [get_timing_node_info -info name $comb_node_fanin]
								puts "Auto-detect found post-amble register $found_postamble_reg for DQS $dqs"
								set dqs_postamble($dqs) [list $found_postamble_reg]
							}
						}
					}
				}
			}
		}
	}
	set data_array(dqs_postamble_list) [array get dqs_postamble]

	# Get the read post-amble control clk
	# 1. get the post-amble control register(s) from the post-amble enable regs
	array set postamble_control_reg_array [list]
	set dqs_with_postamble [array names dqs_postamble]
	foreach dqs $dqs_with_postamble {
		set postamble_reg [lindex $dqs_postamble($dqs) 0]
		set postamble_reg_id $tdb_array($postamble_reg)
		traverse_fanin_up_to_depth $postamble_reg_id is_node_type_reg asynch postamble_control_reg_array 2
	}
	# 2. get the post-amble control clk
	set postamble_control_registers [array names postamble_control_reg_array]
	set data_array(clk_read_postamble) [get_clock $postamble_control_registers "read post-amble control" read_postamble_clk_id]
	extract_postamble_phases data_array $postamble_control_registers $read_postamble_clk_id
}

# ----------------------------------------------------------------
#
proc dtw_auto_detect::is_register_clocked_by_clk { reg_id clk_id depth } {
#
# Description: Look for the resync clocks
#
# ----------------------------------------------------------------
	array set clk_array [list]
	traverse_fanin_up_to_depth $reg_id is_node_input_clk clock clk_array $depth
	puts "$reg_id is clocked by [array get clk_array]"
	if {[array names clk_array -exact $clk_id] == ""} {
		set result 0
	} else {
		set result 1
	}

	return $result
}

# ----------------------------------------------------------------
#
proc dtw_auto_detect::get_implemented_phase { ip_phase clk_id } {
#
# Description: Adjust the IP's phase shift to the implemented PLL phase
#
# ----------------------------------------------------------------
	set clk_phase [get_timing_node_info -info phase_only $clk_id]
	if {$ip_phase == ""} {
		set adjusted_phase "$clk_phase deg"
	} elseif {[lindex $ip_phase 1] == "deg"} {
		set ip_phase_number [lindex $ip_phase 0]
		if {$ip_phase_number == "<default>"} {
			set adjusted_phase "$clk_phase deg"
		} else {
			puts "Adjusted phase $ip_phase to"
			set adjusted_phase "[expr $clk_phase + 180*round(($ip_phase_number - $clk_phase)/180.0)] deg"
			puts "$adjusted_phase"
		}
	} else {
		set adjusted_phase $ip_phase
	}
	return $adjusted_phase
}

# ----------------------------------------------------------------
#
proc dtw_auto_detect::extract_resync_clocks { data_array_name tdb_array_name } {
#
# Description: Look for the resync clocks
#
# ----------------------------------------------------------------
	upvar 1 $data_array_name data_array
	upvar 1 $tdb_array_name tdb_array

	array set resync_registers_array [list]
	set resync1_clk_id -1
	array set dqs_dq $data_array(dqs_dq_list)
	if {$data_array(use_hardware_dqs)} {
		# Resync after initial capture with DQS clock
		foreach dqs $data_array(dqs_list) {
			if {[array names tdb_array -exact $dqs] != ""} {
				set dqs_id $tdb_array($dqs)
				set capture_registers_list [list]
				foreach dq $dqs_dq($dqs) {
					if {[array names tdb_array -exact $dq] != ""} {
						set dq_id $tdb_array($dq)
						traverse_fanout_to_clock_domain_transfer_to_depth $dq_id $dqs_id resync_registers_array 5
					}
				}
			}
		}
		set node_type "read re-synchronization"
	} else {
		# Resync immediately
		foreach dqs $data_array(dqs_list) {
			foreach dq $dqs_dq($dqs) {
				if {[array names tdb_array -exact $dq] != ""} {
					set dq_id $tdb_array($dq)
					traverse_fanout_to_registers $dq_id resync_registers_array 2
				}
			}
		}
		set node_type "read capture"
	}

	set resync_registers_list [array names resync_registers_array]
	set data_array(clk_resync) [get_clock $resync_registers_list $node_type resync1_clk_id]
	if {$resync1_clk_id != -1} {
		if {[array names data_array -exact resync_phase] != ""} {
			set data_array(resync_phase) [get_implemented_phase $data_array(resync_phase) $resync1_clk_id]
		} else {
			set data_array(resync_phase) [get_implemented_phase "" $resync1_clk_id]
		}
	}
	if {[array names data_array -exact is_clk_fedback_in] != "" && $data_array(is_clk_fedback_in) == 1 && $data_array(clk_resync) != ""} {
		# Get resync2 clk
		array set resync2_registers_array [list]
		foreach resync_reg $resync_registers_list {
			traverse_fanout_to_clock_domain_transfer_to_depth $resync_reg $resync1_clk_id resync2_registers_array 3
		}
		set resync2_registers_list [array names resync2_registers_array]
		set data_array(clk_resync2) [get_clock $resync2_registers_list "second stage read re-synchronization" resync2_clk_id]
		if {$resync2_clk_id != -1} {
			if {[array names data_array -exact resync_sys_phase] != ""} {
				set data_array(resync_sys_phase) [get_implemented_phase $data_array(resync_sys_phase) $resync2_clk_id]
			} else {
				set data_array(resync_sys_phase) [get_implemented_phase "" $resync2_clk_id]
			}
		}
	}
}

# ----------------------------------------------------------------
#
proc dtw_auto_detect::extract_stratixii_info { data_array_name tdb_input_keeper_array_name tdb_output_array_name user_string_array_name} {
#
# Description: Extract from a Stratix II-style netlist
#
# ----------------------------------------------------------------
	upvar 1 $data_array_name data_array
	upvar 1 $tdb_input_keeper_array_name tdb_input_keeper_array
	upvar 1 $tdb_output_array_name tdb_output_array
	upvar 1 $user_string_array_name user_string

	if {$data_array(memory_type) == "ddr" && $data_array(use_hardware_dqs)} {
		# Post-amble registers and clk
		extract_stratixii_postamble_registers data_array tdb_input_keeper_array
	}

	extract_ck_dqs_dq_resync_pll_clocks data_array tdb_input_keeper_array tdb_output_array user_string
}

# ----------------------------------------------------------------
#
proc dtw_auto_detect::extract_ck_dqs_dq_resync_pll_clocks { data_array_name tdb_input_keeper_array_name tdb_output_array_name user_string_array_name} {
#
# Description: Extract CK/CK#, DQS/DQ, and resync PLL output clocks
#
# ----------------------------------------------------------------
	upvar 1 $data_array_name data_array
	upvar 1 $tdb_input_keeper_array_name tdb_input_keeper_array
	upvar 1 $tdb_output_array_name tdb_output_array
	upvar 1 $user_string_array_name user_string

	# CK/CK# output clock
	set ck_ckn_list [concat $data_array(ck_list) $data_array(ckn_list)]
	set data_array(clk_sys) [get_output_clock $ck_ckn_list tdb_output_array "$user_string(ck_ckn) output"]

	# System PLL info (inclk, period, mult, div)
	extract_system_pll_info data_array tdb_input_keeper_array

	# Resync registers and clks
	if {$data_array(memory_type) == "ddr" || $data_array(use_dcfifo) == 0 || ($data_array(use_hardware_dqs) == 0 && $data_array(use_source_synchronous_pll) == 1)} {
		extract_resync_clocks data_array tdb_input_keeper_array
	}

	# DQS output clock
	if {$data_array(memory_type) == "ddr"} {
		set data_array(clk_dqs_out) [get_output_clock $data_array(dqs_list) tdb_output_array "$user_string(dqs_write) write"]
	} elseif {$data_array(memory_type) == "qdr2" || $data_array(memory_type) == "rldram2"} {
		# Same as K/K# clock
	}

	# DQ output clock
	dtw_device_get_family_parameter "_default" $data_array(memory_type)_user_terms mem_user_term_list
	array set mem_user_term $mem_user_term_list

	if {$mem_user_term(read_dqs) == $mem_user_term(write_dqs)} {
		set write_dqs_list $data_array(dqs_list)
	} else {
		set write_dqs_list $data_array(ck_list)
	}
	array set dqs_dq $data_array(dqs_dq_list)
	set dq_list [list]
	foreach dqs $write_dqs_list {
		foreach dq $dqs_dq($dqs) {
			lappend dq_list $dq
		}
	}
	set data_array(clk_dq_out) [get_output_clock $dq_list tdb_output_array "$user_string(dq_write) write"]

	# Find the Address/Control clock
	if {[array names data_array -exact addr_ctrl_list] != ""} {
		set addr_ctrl_list $data_array(addr_ctrl_list)
	} elseif {[array names data_array -exact addr_list] != ""} {
		set addr_ctrl_list $data_array(addr_list)
		if {[array names data_array -exact ctrl_list] != ""} {
			lappend addr_ctrl_list $data_array(addr_list)
		}
	}
	set data_array(clk_addr_ctrl_out) [get_output_clock $addr_ctrl_list tdb_output_array "Address/Control"]
}

# ----------------------------------------------------------------
#
proc dtw_auto_detect::get_clock { dest_id_list node_type {clock_id_name ""} } {
#
# Description: Look for the clock of the given nodes
#
# ----------------------------------------------------------------
	if {$clock_id_name != ""} {
		upvar 1 $clock_id_name clock_id
	}
	set clock_id -1

	array set clk_array [list]
	foreach node_id $dest_id_list {
		traverse_fanin_up_to_depth $node_id is_node_type_clk clock clk_array 3
	}
	if {[array size clk_array] == 1} {
		set clock_id [lindex [array names clk_array] 0]
		set clk [get_timing_node_info -info name $clock_id]
		puts "Auto-detect found $node_type clock $clk"
	} elseif {[array size clk_array] > 1} {
		puts "Warning: Found more than 1 clock driving the $node_type.  Enter the $node_type clock manually."
		set clk ""
	} else {
		set clk ""
		puts "Auto-detect could not find $node_type clock"
	}

	return $clk
}

# ----------------------------------------------------------------
#
proc dtw_auto_detect::get_output_clock { ddio_output_pin_list tdb_array_name pin_type } {
#
# Description: Look for the output clocks of the given pins
#
# ----------------------------------------------------------------
	upvar 1 $tdb_array_name tdb_array
	
	set output_id_list [list]
	foreach output_pin $ddio_output_pin_list {
		if {[array names tdb_array -exact $output_pin] != ""} {
			lappend output_id_list $tdb_array($output_pin)
		}
	}
	return [get_clock $output_id_list $pin_type]
}

# ----------------------------------------------------------------
#
proc dtw_auto_detect::traverse_fanin_up_to_depth { node_id match_command edge_type results_array_name depth} {
#
# Description: Recurses through the timing netlist starting from the given
#              node_id through edges of type edge_type to find nodes
#              satisfying match_command.
#              Recursion depth is bound to the specified depth.
#              Adds the resulting TDB node ids to the results_array.
#
# ----------------------------------------------------------------
	upvar 1 $results_array_name results

	if {$depth < 0} {
		error "Internal error: Bad timing netlist search depth"
	}
	set fanin_edges [get_timing_node_info -info ${edge_type}_edges $node_id]
	set number_of_fanin_edges [llength $fanin_edges]
	for {set i 0} {$i != $number_of_fanin_edges} {incr i} {
		set fanin_edge [lindex $fanin_edges $i]
		set fanin_id [get_timing_edge_info -info src_node $fanin_edge]
		if {$match_command == "" || [eval $match_command $fanin_id] != 0} {
			set results($fanin_id) 1
		} elseif {$depth == 0} {
			# Max recursion depth
		} else {
			traverse_fanin_up_to_depth $fanin_id $match_command $edge_type results [expr "$depth - 1"]
		}
	}
}

# ----------------------------------------------------------------
#
proc dtw_auto_detect::traverse_fanout_down_to_depth { node_id match_command results_array_name depth} {
#
# Description: Recurses through the timing netlist starting from the given
#              node_id through edges of type edge_type to find nodes
#              satisfying match_command.
#              Recursion depth is bound to the specified depth.
#              Adds the resulting TDB node ids to the results_array.
#
# ----------------------------------------------------------------
	upvar 1 $results_array_name results

	if {$depth < 0} {
		error "Internal error: Bad timing netlist search depth"
	}
	set fanout_edges [get_timing_node_info -info fanout_edges $node_id]
	set number_of_fanout_edges [llength $fanout_edges]
	for {set i 0} {$i != $number_of_fanout_edges} {incr i} {
		set fanout_edge [lindex $fanout_edges $i]
		set fanout_id [get_timing_edge_info -info dst_node $fanout_edge]
		if {$match_command == "" || [eval $match_command $fanout_id] != 0} {
			set results($fanout_id) 1
		} elseif {$depth == 0} {
			# Max recursion depth
		} else {
			traverse_fanout_down_to_depth $fanout_id $match_command results [expr "$depth - 1"]
		}
	}
}

# ----------------------------------------------------------------
#
proc dtw_auto_detect::traverse_fanout_to_registers { node_id results_array_name depth} {
#
# Description: Recurses through the timing netlist starting from the given
#              node_id to find nodes
#              satisfying.
#              Recursion depth is bound to the specified depth.
#              Adds the resulting TDB node ids to the results_array.
#
# ----------------------------------------------------------------
	upvar 1 $results_array_name results
	traverse_fanout_down_to_depth $node_id is_node_type_reg results $depth
}

# ----------------------------------------------------------------
#
proc dtw_auto_detect::traverse_fanout_to_clock_domain_transfer_to_depth { node_id clk_id results_array_name depth} {
#
# Description: Recurses through the timing netlist starting from the given
#              node_id to find registers that aren't driven by the given clk_id.
#              Recursion depth is bound to the specified depth.
#              Adds the resulting TDB node ids to the results_array.
#
# ----------------------------------------------------------------
	upvar 1 $results_array_name results

	if {$depth < 0} {
		error "Internal error: Bad timing netlist search depth"
	}
	set fanout_edges [get_timing_node_info -info fanout_edges $node_id]
	set number_of_fanout_edges [llength $fanout_edges]
	for {set i 0} {$i != $number_of_fanout_edges} {incr i} {
		set fanout_edge [lindex $fanout_edges $i]
		set fanout_id [get_timing_edge_info -info dst_node $fanout_edge]
		set node_type [get_timing_node_info -info type $fanout_id]
		set domain_transfer 0
		if {$node_type == "reg"} {
			array unset clk_array
			array set clk_array [list]
			traverse_fanin_up_to_depth $fanout_id is_node_type_clk clock clk_array $depth
			if {[array size clk_array] == 1 && [lindex [array names clk_array] 0] != $clk_id} {
				set domain_transfer 1
				set results($fanout_id) 1
			}
		}
		if {$depth == 0} {
			# Max recursion depth
		} elseif {$domain_transfer == 0} {
			traverse_fanout_to_clock_domain_transfer_to_depth $fanout_id $clk_id results [expr "$depth - 1"]
		}
	}
}

# ----------------------------------------------------------------
#
proc dtw_auto_detect::is_node_type_keeper { node_id } {
#
# Description: Given a node, tells whether or not it is a bidir or input
#
# ----------------------------------------------------------------
	set node_type [get_timing_node_info -info type $node_id]
	if {$node_type == "pin" || $node_type == "reg" || $node_type == "clk"} {
		set result 1
	} else {
		set result 0
	}
	return $result
}

# ----------------------------------------------------------------
#
proc dtw_auto_detect::is_node_type_reg { node_id } {
#
# Description: Given a node, tells whether or not it is a reg
#
# ----------------------------------------------------------------
	set node_type [get_timing_node_info -info type $node_id]
	if {$node_type == "reg"} {
		set result 1
	} else {
		set result 0
	}
	return $result
}

# ----------------------------------------------------------------
#
proc dtw_auto_detect::is_node_type_clk { node_id } {
#
# Description: Given a node, tells whether or not it is a clk
#
# ----------------------------------------------------------------
	set node_type [get_timing_node_info -info type $node_id]
	if {$node_type == "clk"} {
		set result 1
	} else {
		set result 0
	}
	return $result
}

# ----------------------------------------------------------------
#
proc dtw_auto_detect::is_node_input_clk { node_id } {
#
# Description: Given a node, tells whether or not it is a input clk pin
#
# ----------------------------------------------------------------
	set node_type [get_timing_node_info -info type $node_id]
	set number_of_fanin [llength [get_timing_node_info -info clock_edges $node_id]]
	if {$number_of_fanin == 0 && ($node_type == "pin" || $node_type == "clk")} {
		set result 1
	} else {
		set result 0
	}
	return $result
}

# ----------------------------------------------------------------
#
proc dtw_auto_detect::extract_system_pll_info { data_array_name tdb_array_name } {
#
# Description: Get the System PLL info for PLL with the given PLL output clock
#
# ----------------------------------------------------------------
	upvar 1 $data_array_name data_array
	upvar 1 $tdb_array_name tdb_array
	set pll_output $data_array(clk_sys)
	
	if {[array names tdb_array -exact $pll_output] != ""} {
		set pll_output_id $tdb_array($pll_output)
		array set pll_inclk_array [list]
		traverse_fanin_up_to_depth $pll_output_id is_node_input_clk clock pll_inclk_array 2
		if {[array size pll_inclk_array] == 1} {
			set input_id [lindex [array names pll_inclk_array] 0]
			set data_array(clk_pll_in) [get_timing_node_info -info name $input_id]
			puts "Auto-detect found PLL inclk $data_array(clk_pll_in)"

			set pll_inclk_info [get_timing_node_info -info clock_info $input_id]
			set pll_outclk_info [get_timing_node_info -info clock_info $pll_output_id]
			array set pll_inclk_info_array [join $pll_inclk_info]
			array set pll_outclk_info_array [join $pll_outclk_info]
			if {[array names pll_inclk_info_array -exact PERIOD] != ""} {
				puts "Auto-detect found PLL inclk period $pll_inclk_info_array(PERIOD)"
				set data_array(pll_input_freq) $pll_inclk_info_array(PERIOD)
			}
			if {[array names pll_inclk_info_array -exact MULITPLY] != ""} {
				puts "Auto-detect found CK/CK# divide factor $pll_outclk_info_array(MULITPLY)"
				# Note that TAN clock mult/div is for the period, while users
				# expect mult/div for the input frequency.  So we swap them.
				set data_array(pll_div) $pll_outclk_info_array(MULITPLY)
			} elseif {[array names pll_inclk_info_array -exact MULTIPLY] != ""} {
				puts "Auto-detect found CK/CK# divide factor $pll_outclk_info_array(MULTIPLY)"
				set data_array(pll_div) $pll_outclk_info_array(MULTIPLY)
			}
			if {[array names pll_inclk_info_array -exact DIVIDE] != ""} {
				puts "Auto-detect found CK/CK# multiply factor $pll_outclk_info_array(DIVIDE)"
				set data_array(pll_mult) $pll_outclk_info_array(DIVIDE)
			}
		} else {
			puts "Auto-detect missed PLL inclk for $pll_output"
		}
	}
}

# ----------------------------------------------------------------
#
proc dtw_auto_detect::extract_cycloneii_info { data_array_name tdb_input_keeper_array_name tdb_output_array_name user_string_array_name} {
#
# Description: Extract from a Cyclone II netlist
#
# ----------------------------------------------------------------
	upvar 1 $data_array_name data_array
	upvar 1 $tdb_input_keeper_array_name tdb_input_keeper_array
	upvar 1 $tdb_output_array_name tdb_output_array
	upvar 1 $user_string_array_name user_string

	if {$data_array(memory_type) == "ddr" && $data_array(use_hardware_dqs)} {
		# Post-amble registers and clk
		extract_cycloneii_postamble_registers data_array tdb_input_keeper_array
	}

	extract_ck_dqs_dq_resync_pll_clocks data_array tdb_input_keeper_array tdb_output_array user_string
}

# ----------------------------------------------------------------
#
proc dtw_auto_detect::extract_cycloneii_postamble_registers { data_array_name tdb_array_name } {
#
# Description: Look for the post-amble register
#
# ----------------------------------------------------------------
	upvar 1 $data_array_name data_array
	upvar 1 $tdb_array_name tdb_array

	array set dqs_postamble [list]

	# There are 2 methods to deal with postamble in Cyclone II:
	# 1) CLKCTRL solution
	# In Cyclone II, the post-amble register is the register fed by the DQS
	# that feeds 1 fake output pin (enable input is a fake output pin in TDB)

	# 2) Soft solution (Stratix I-style)
	# The post-amble register is the falling-edge DQS triggered register that
	# feeds another falling-edge DQS triggered register

	# Collect real output pins so we can figure out what's fake
	set outputs_collection [get_names -filter "*" -node_type pin -observable_type post_synthesis]
	set outputs_list [list]
	foreach_in_collection name_id $outputs_collection {
		set node_type [get_name_info -info node_type $name_id]
		if {$node_type == "bidir" || $node_type == "output"} {
			set output_name [get_name_info -info full_path $name_id]
			lappend outputs_list $output_name
		}
	}

	# For each DQS, traverse fan-outs to their registers and from there, check
	# if the register feeds a fake output pin.  If so, the register is a
	# post-amble register.
	set found 0
	foreach dqs $data_array(dqs_list) {
		if {[array names tdb_array -exact $dqs] != ""} {
			set dqs_node_id $tdb_array($dqs)
			array unset dqs_fanout_registers_array
			array set dqs_fanout_registers_array [list]
			traverse_fanout_to_registers $dqs_node_id dqs_fanout_registers_array 4
			foreach reg_id [array names dqs_fanout_registers_array] {
				set fanout_list [get_timing_node_fanout $reg_id]
				if {[llength $fanout_list] == 1} {
					set fanout_node_id [lindex [lindex $fanout_list 0] 0]
					set fanout_node_type [get_timing_node_info -info type $fanout_node_id]
					if {$fanout_node_type == "pin"} {
						set fanout_node_name [get_timing_node_info -info name $fanout_node_id]
						if {[lsearch -exact $outputs_list $fanout_node_name] == -1} {
							set found_postamble_reg [get_timing_node_info -info name $reg_id]
							set dqs_postamble($dqs) [list $found_postamble_reg]
							puts "Auto-detect found post-amble enable register $found_postamble_reg feeding the clkctrl of DQS $dqs"
							set found 1
						}
					}
				}
			}
		}
	}
	if {$found == 0} {
		# Look for soft postamble-reg
		# For each DQS, traverse fan-outs to their falling-edge triggered
		# registers and from there, check if the register feeds another
		# falling-edge DQS triggered register.  If so, the first register is a
		# post-amble enable register.
		if {[array names tdb_array -exact $dqs] != ""} {
			set dqs_node_id $tdb_array($dqs)
			array unset dqs_fanout_registers_array
			array set dqs_fanout_registers_array [list]
			traverse_fanout_to_registers $dqs_node_id dqs_fanout_registers_array 4
			foreach reg1_id [array names dqs_fanout_registers_array] {
				if {[get_timing_node_info -info is_clock_inverted $reg1_id]} {
					# Note that there may be delay buffers in the reg-reg path
					traverse_fanout_to_registers $reg1_id reg2_registers_array 4
					foreach reg2_id [array names reg2_registers_array] {
						if {[get_timing_node_info -info is_clock_inverted $reg2_id] && [is_register_clocked_by_clk $reg2_id $dqs_node_id 4]} {
							set found_postamble_reg [get_timing_node_info -info name $reg1_id]
							set dqs_postamble($dqs) [list $found_postamble_reg]
							puts "Auto-detect found post-amble enable register $found_postamble_reg for DQS $dqs"
							set found 1
						}
					}
				}
			}
		}
	}
	set data_array(dqs_postamble_list) [array get dqs_postamble]

	# Get the read post-amble control clk
	# 1. get the post-amble control register(s) from the post-amble enable regs
	array set postamble_control_reg_array [list]
	set dqs_with_postamble [array names dqs_postamble]
	foreach dqs $dqs_with_postamble {
		set postamble_reg [lindex $dqs_postamble($dqs) 0]
		set postamble_reg_id $tdb_array($postamble_reg)
		traverse_fanin_up_to_depth $postamble_reg_id is_node_type_reg asynch postamble_control_reg_array 2
	}
	# 2. get the post-amble control clk
	set postamble_control_registers [array names postamble_control_reg_array]
	set data_array(clk_read_postamble) [get_clock $postamble_control_registers "read post-amble control" read_postamble_clk_id]
	extract_postamble_phases data_array $postamble_control_registers $read_postamble_clk_id
}

# ----------------------------------------------------------------
#
proc dtw_auto_detect::detect_dqs_bus_clock_setting { data_array_name } {
#
# Description: Looks for a clock setting assignment on the read DQS bus
#              and sets the data_array parameter dqs_bus_clock_setting if found
#
# ----------------------------------------------------------------
	upvar 1 $data_array_name data_array
	set dqs0_name [lindex $data_array(dqs_list) 0]
	set array_bracket_index [string first "\[" $dqs0_name]
	if {$dqs0_name != "" && $array_bracket_index > 0} {
		set dqs_bus_name [string range $dqs0_name 0 [expr "$array_bracket_index - 1"]]
		if {[catch {get_instance_assignment -name CLOCK_SETTINGS -to $dqs_bus_name} setting]} {
			# Failed to find clock setting
		} elseif {$setting != ""} {
			set data_array(dqs_bus_clock_setting) "set_instance_assignment -name CLOCK_SETTINGS $setting -to $dqs_bus_name"
		}
	}
 
}

# ----------------------------------------------------------------
#
proc dtw_auto_detect::auto_detect { argv } {
#
# Description: Auto-detects timing netlist elements
#              First argument is the dwz file name
#
# ----------------------------------------------------------------
	set dwz_file [lindex $argv 0]

	init_tk
	package require ::quartus::timing 1.2
	package require ::quartus::report 2.1
	package require ::quartus::advanced_timing 1.2

	array set data_array [list]
	read_dwz data_array $dwz_file
	set project_name [file tail "$data_array(project_path)"]
	set revision_name $data_array(project_revision)

	# Open project
	set project_is_open [is_project_open]
	if {$project_is_open == 0} {
		project_open $project_name -revision $revision_name
	}

	if {[is_auto_detectable]} {
		set success 1
		# If required, ask the user to do an initial timing netlist
		if {[timing_netlist_exist] == 0} {
			if {[file exists "./db/${revision_name}.map.cdb"] == 0} {
				set run_map 1
			} else {
				set run_map 0
			}
			package require ::quartus::flow 1.0
			if {$run_map} {
				puts "Executing quartus_map...."
				if {[catch "execute_module -tool map" map_result]} {
					puts $map_result
					puts "quartus_map failed...."
					set success 0
				} else {
					puts $map_result
					puts "quartus_map successful."
				}
			}
			if {$success} {
				puts "Executing quartus_tan...."
				set clock_latency_mode [get_global_assignment -name ENABLE_CLOCK_LATENCY]
				if {$clock_latency_mode != "ON"} {
					# We need clock latency on to detect clock phases
					set_global_assignment -name ENABLE_CLOCK_LATENCY ON
				}
				if {[file exists "./db/${revision_name}.cmp.cdb"] == 0} {
					puts "Running in post-map mode"
					create_timing_netlist -post_map
				} else {
					puts "Running in post-fit mode"
					create_timing_netlist
				}
				if {$clock_latency_mode != "ON"} {
					# restore the user's clock latency mode
					set_global_assignment -name ENABLE_CLOCK_LATENCY $clock_latency_mode
				}				
			}
		} else {
			read_timing_netlist
		}

		# Extract the timing netlist
		create_tdb_arrays tdb_input_keeper_array tdb_output_array
		set device_family [get_dtw_family]

		array set user_string [list]
		if {$data_array(memory_type) == "ddr" || $data_array(memory_type) == "rldram2"} {
			set user_string(ck_ckn) "CK/CK#"
			set user_string(dqs_write) "DQS"
			set user_string(dq_write) "DQ/DM"
		} elseif {$data_array(memory_type) == "qdr2"} {
			set user_string(ck_ckn) "K/K#"
			set user_string(dqs_write) "CQ/CQ#"
			set user_string(dq_write) "D/BWSN"
		} else {
			set user_string(ck_ckn) "Clock"
			set user_string(dqs_write) "QK/QK#"
			set user_string(dq_write) "DQ"
		}

		detect_dqs_bus_clock_setting data_array
		if {$device_family == "stratix ii"} {
			extract_stratixii_info data_array tdb_input_keeper_array tdb_output_array user_string
		} elseif {$device_family == "cyclone ii"} {
			extract_cycloneii_info data_array tdb_input_keeper_array tdb_output_array user_string
		} else {
			error "Unknown family"
		}
	} else {
		puts "Warning: [get_dtw_family] family is unsupported for auto-detection"
		set success 0
	}
	# All done - Clean up
	if {$project_is_open == 0} {
		project_close
	}

	# Return result is written to the dwz file
	set data_array(auto_detect_result) $success
	write_dwz data_array $dwz_file
	return
}

if {[namespace exists ::dtw] == 0} {
	dtw_auto_detect::auto_detect $quartus(args)
}
