use strict; 
use ptf_parse;
use hdl_common;

sub main 
{
    my $switches = parse_args (@ARGV);

    # extract interesting command line switches
    my $system_directory = $switches->{system_directory};
    my $class_name = $switches->{target_class_name};
    my $module_name = $switches->{target_module_name};
    my $top_module_name = $switches->{top_module_name};

    # create some strings we'll need
    my $gen_script_file = $system_directory . "/" . $class_name . "/mk_" . $class_name . ".pl";
    my $PTF_file_name = $system_directory . "/" . $switches->{system_name} . ".ptf";

    # read in ptf file
    my $ptf = &new_ptf_from_file ($PTF_file_name);
    die ("Error: Unable to read PTF file ($PTF_file_name)!") if ($ptf eq "");

    # find our module
    my $module = &get_child_by_path ($ptf,"SYSTEM\/MODULE $module_name",0);
    die ("Error: No such module ($module_name) in file $PTF_file_name!") if ($module eq "");

    # get the list of HDL files for this module, in the string we like
    my @HDL_list = get_HDLfiles_from_module ($module);
    for (my $i=0; $i < scalar @HDL_list; $i++)
    {
        $HDL_list[$i] =~ /^.*\/(.*)$/; # no directory name
        $HDL_list[$i] = $1;
    }
    my $class_file_list = join "\",\"" , @HDL_list; # comma separated list
    $class_file_list = "\"" . $class_file_list . "\""; # surrounded by quotes

    # determine what to do with the file list
    my $class_file_mode = "";
    my $instantiate = get_data_by_path ($module, 
        "SYSTEM_BUILDER_INFO/Instantiate_In_System_Module");
    my $simulate = get_data_by_path ($module,
        "WIZARD_SCRIPT_ARGUMENTS/Simulate_Imported_HDL");
    if ($instantiate && $simulate)
    {
        $class_file_mode = "simulation_and_quartus";
    }
    elsif ($instantiate)
    {
        $class_file_mode = "quartus_only";
    }

    # standard program string declarations
    my $gen_header = qq[
# |
# | Generator script for class: $class_name
# |     This program was machine generated,
# |     and is intended to be a starting point
# | 

use strict;             # keeps us honest
use generator_library;  # includes all the code we'll need???

# |
# | generator_enable_mode: This call controls the amount of information 
# | displayed/logged while this generator script is running.  
# |     terse - minimal out
# |     verbose - lots of output 
# |

generator_enable_mode ("terse");

# | 
# | generator_begin: This call initialize the generator library
# | 

generator_begin (\@ARGV);

];

    my $gen_copy_files = qq[
# | Make a wrapper for the user's HDL:
generator_make_module_wrapper($simulate, "$top_module_name");

# |
# | generator_copy_files_and_set_system_ptf: this call causes the 
# | source files in this class to be copied to the appropriate
# | directory.
# |
# | The second parameter is the list of files
# | 
# | The first parameter specifies how the files will be used:
# | quartus_only 
# |         files will be used only for quartus synthesis
# | simulation_only 
# |         files will be used only for simulation
# | simulation_and_quartus 
# |         files will be used for quartus synthesis and simulation
# | precompiled_simulation_files 
# |         directories containing precomipled simulation models
# |
#
generator_copy_files_and_set_system_ptf ("$class_file_mode", 
                ($class_file_list));

];

    my $gen_footer = qq[
# | 
# | generator_end: This call completes the generation for this class

generator_end ();

# |
# | Exit with a successful return code
# | 

exit (0);

];


    open GF, ">$gen_script_file" or die "Unable to open output generator script";

    print GF $gen_header;

    if ($class_file_mode)
    {
        print GF $gen_copy_files;
    }

    print GF $gen_footer;

    close GF;
}

main;
1;

