##Copyright (C) 2004 Altera Corporation
##Any megafunction design, and related net list (encrypted or decrypted),
##support information, device programming or simulation file, and any other
##associated documentation or information provided by Altera or a partner
##under Altera's Megafunction Partnership Program may be used only to
##program PLD devices (but not masked PLD devices) from Altera.  Any other
##use of such megafunction design, net list, support information, device
##programming or simulation file, or any other related documentation or
##information is prohibited for any other purpose, including, but not
##limited to modification, reverse engineering, de-compiling, or use with
##any other silicon devices, unless such use is explicitly licensed under
##a separate agreement with Altera or a megafunction partner.  Title to
##the intellectual property, including patents, copyrights, trademarks,
##trade secrets, or maskworks, embodied in any such megafunction design,
##net list, support information, device programming or simulation file, or
##any other related documentation or information provided by Altera or a
##megafunction partner, remains with Altera, the megafunction partner, or
##their respective licensors.  No other licenses, including any licenses
##needed under any third party's intellectual property, are provided herein.
##Copying or modifying any file, or portion thereof, to which this notice
##is attached violates this copyright.

package altera_avalon_onchip_memory2;
use Exporter;
@ISA = Exporter;
@EXPORT = qw(
    &make_mem
);


use strict;
use europa_all;
use wiz_utils;
use format_conversion_utils;

sub validate_options
{
  my ($Opt, $project, $SBI) = @_;

  # Make the module name available in a handy format.
  $Opt->{name} = $project->_target_module_name();

  # Boolean variables specify what kind of I/O we have:
  #
  validate_parameter({hash    => $Opt,
                       name    => "Writeable",
                       type    => "boolean",
                       default => 1,
                      });

  if (!is_computer_acceptable_bit_width($SBI->{Data_Width}))
  {
    ribbit(
      "ERROR:  Parameter validation failed.\n" .
      "  Parameter 'Data_Width' (= $SBI->{Data_Width})\n" .
      "  is not an allowed value.\n"
    );
  }

  # Create fictitious (derived) Option "Address_Span" from existing
  #   "Size_Value" and "Size_Multiple" Opt:
  validate_parameter({hash    => $Opt,
                       name    => "Size_Multiple",
                       type    => "integer",
                       allowed => [1,1024],
                       default => 1,
                      });
  
  validate_parameter({hash    => $Opt,
                       name    => "Size_Value",
                       type    => "integer",
                      });

  $Opt->{Address_Span} = $Opt->{Size_Multiple} * 
                             $Opt->{Size_Value};

  # Write this parameter back into the ptf file.
  $project->WSA()->{Address_Span} = $Opt->{Address_Span};
  
  # This is probably irrelevant, but it could be helpful someday.
  # Write the WSA/Size assignment, for HAL use.
  $project->WSA()->{Size} = $Opt->{Address_Span};

  validate_parameter({hash    => $Opt,
                       name    => "ram_block_type",
                       type     => "string",
                       allowed => ["M512", 
                                   "M4K", 
                                   "M-RAM"],
                       default  => "M4K",
                      });

  #### Derived parameters:
  $Opt->{num_lanes}     = $SBI->{Data_Width} / 8;
  $Opt->{make_individual_byte_lanes} = $Opt->{ram_block_type} eq 'M512' && $Opt->{num_lanes} > 1;
  $Opt->{Address_Width} = $SBI->{Address_Width};
  $Opt->{Data_Width}    = $SBI->{Data_Width};

  # The depth of the memory, in words, is handy to know.  
  # This is all-too-easily confused with the size of the RAM, in
  # bytes.  Not the same thing at all.
  my $byte_width = int($SBI->{Data_Width} / 8);
  $Opt->{num_words} = ceil ($Opt->{Address_Span} / $byte_width);

  # The address-bus can be wider than required by the span.  This allows
  # memories to occupy more address-space than the actual, physically-
  # implemented memory block.
  validate_parameter({hash    => $SBI,
                       name    => "Address_Width",
                       type    => "integer",
                       range   => [&Bits_To_Encode($Opt->{num_words}-1), 32],
                       });

  $Opt->{base_addr_as_number} = $SBI->{Base_Address};
  $Opt->{base_addr_as_number} = oct($Opt->{base_addr_as_number})
    if ($Opt->{base_addr_as_number} =~ /^0/);
      
  validate_parameter({hash     => $SBI,
                       name     => "Read_Latency",
                       type     => "integer",
                       range    => [1, 2],
                       default  => 0,
                       });
  
  $Opt->{lang} =
    $project->system_ptf()->{WIZARD_SCRIPT_ARGUMENTS}{hdl_language};
  validate_parameter({hash    => $Opt,
                       name    => "lang",
                       type     => "string",
                       allowed => ["verilog", 
                                   "vhdl",],
                       default  => "verilog",
                      });
                       
  # Sekrit option.  Keep out!  If set, the placeholder memory contents 
  # file memory is full of random data, rather than 0.
  $Opt->{set_rand_contents} = $project->WSA()->{set_rand_contents};

  # Add a couple of handy references for make_placeholder_contents_files.
  $Opt->{Base_Address} = $SBI->{Base_Address};
  $Opt->{Address_Span} = $SBI->{Address_Span};
  
  # If the user selected a non-default initialization file but failed to specify an 
  # initialization file name OR if one doesn't exist in a legacy (pre 5.1) PTF,
  # then set the name to be the default (module name)
  $Opt->{init_contents_file} = $Opt->{name} if ($Opt->{init_contents_file} eq '');
}  
  

# Try to estimate usage.  Gets the wrong answer, I think,
# when byte lanes are used.
sub report_usage
{
  my $Opt = shift;
  
  print STDERR "  $Opt->{name} memory usage summary:\n";
  my %trimatrix_bits = (
    M512  => 512,
    M4K   => 4096,
    'M-RAM' => 512*1024,
  );
  my $bits_consumed = $Opt->{num_words} * $Opt->{Data_Width};
  my $block_granularity = $trimatrix_bits{$Opt->{ram_block_type}};
  if ($Opt->{make_individual_byte_lanes})
  {
    $block_granularity *= $Opt->{num_lanes};
  }

  print STDERR "$Opt->{num_words} words, $Opt->{Data_Width} bits wide ($bits_consumed bits) ";
  print STDERR "(@{[ceil($bits_consumed / $trimatrix_bits{$Opt->{ram_block_type}})]} $Opt->{ram_block_type} blocks.\n";
}

################################################################
# make_mem
#
# Given a name and a hashful of options, builds an e_module object
# which implements an onchip-memory peripheral.
#
################################################################
sub make_mem
{
  my ($module, $project) = (@_);
  my $Opt = &copy_of_hash ($project->WSA());
  my $SBI1 = &copy_of_hash ($project->SBI("s1"));

  $Opt->{name} = $module->name();

  validate_options($Opt, $project, $SBI1);

  if ($Opt->{dual_port})
  {
    my $SBI2 = &copy_of_hash ($project->SBI("s2"));

    validate_options($Opt, $project, $SBI2);
  }

  $project->do_makefile_target_ptf_assignments(
      's1',
      ['dat', 'hex', 'sym', ],
      $Opt,
  );
    

  # report_usage($Opt);
  
  instantiate_memory($module, $Opt, $project);

  # Since signal-widths don't "bubble" through assignments, specify 
  # address-port width explicitly:
  e_port->new({within    => $module,
               name      => "address",
               width     => $Opt->{Address_Width},
               direction => "in",
              });

  # Our slave-port name is taken from the SLAVE section which 
  # is named in our class.ptf file.
  e_avalon_slave->new({within => $module,
                       name   => "s1",
                       sideband_signals => [ "clken" ],
                       type_map => {debugaccess => 'debugaccess',},
                       });

  my %slave_2_type_map = reverse
  (
     clken      => "clken2",
     address    => "address2",
     readdata   => "readdata2",
     chipselect => "chipselect2",
     write      => "write2",
     writedata  => "writedata2",
     byteenable => "byteenable2",
     debugaccess => 'debugaccess',
  );

  e_avalon_slave->new({within => $module,
                       name   => "s2",
                       sideband_signals => [ "clken" ],
                       type_map => \%slave_2_type_map,
                    });
}

sub instantiate_memory
{
  my ($module, $Opt, $project) = @_;

  my $marker = e_default_module_marker->new($module);

  my $SBI1 = $project->SBI("s1");
  my $SBI2 = $project->SBI("s2");

  # Basic ports that all memory types have:
  e_port->adds(
    ['clk',        1,                     'in'],
    ['address',    $Opt->{Address_Width}, 'in'],
    ['readdata',   $Opt->{Data_Width},    'out'],
  );

  e_port->adds({name => "clken", width => 1, direction => "in",
    default_value => "1'b1"});

  # Basic input port, output port maps for all memory types:
  my $in_port_map = {
    clock0      => 'clk',
    clocken0    => 'clken',
    address_a   => 'address',
    wren_a      => 'wren',
    data_a      => 'writedata',
  };

  my $out_port_map = 
    ($SBI1->{Read_Latency} == 1) ? 
      { q_a => 'readdata' } :
      { q_a => 'readdata_ram' };

  # Pipeline read data if required.
  if ($SBI1->{Read_Latency} > 1) {
    e_register->add(
      {out => ["readdata", $Opt->{Data_Width}],            
       in => "readdata_ram",  
       enable => "clken",
       delay => ($SBI1->{Read_Latency} - 1),
      }
    );
  }

  # Whether RAM or ROM, the memory gets these ports:
  e_port->adds(
    ['chipselect', 1,                     'in'],
    ['write',      1,                     'in'],
    ['writedata',  $Opt->{Data_Width},    'in'],
  );

  if ($Opt->{Writeable})
  {
    e_assign->add(['wren', and_array('chipselect', 'write')]);
  }
  else
  {
    # "ROM".  Make a RAM which can be written when debugaccess is set
    # (unless hardcopy).
    e_port->adds(
      ['debugaccess', 1, 'in'],
    );
    if ($project->is_hardcopy_compatible())
    {
      # Hardcopy is targeted, so wire the write-enable inactive and let
      # Quartus figure out that this is really a ROM (and can have
      # an init_file).
      e_assign->add(['wren', 0]);
    }
    else
    {
      e_assign->add(['wren', and_array('chipselect', 'write', 'debugaccess')]);
    }
  }

  my $parameter_map = {
    operation_mode            => qq("SINGLE_PORT"),
    width_a                   => $Opt->{Data_Width},
    widthad_a                 => $Opt->{Address_Width},
    numwords_a                => $Opt->{num_words},
    lpm_type                  => qq("altsyncram"),
    byte_size                 => 8,
    outdata_reg_a             => qq("UNREGISTERED"),
    read_during_write_mode_mixed_ports => qq("DONT_CARE"),
    ram_block_type            => qq("$Opt->{ram_block_type}"),
  };

  # More than one byte lane? Add ports as needed.
  if ($Opt->{num_lanes} > 1)
  {
    # This module needs a byteenable input.
    e_port->adds(["byteenable", $Opt->{num_lanes},     "in" ],);
    if ($Opt->{ram_block_type} eq 'M512')
    {
      # Don't mention anything byteenable-related in the instantiation
      # of the byte-lane M512s, or Quartus gets angry.
    }
    else
    {
      $in_port_map->{byteena_a} = 'byteenable';
      $parameter_map->{width_byteena_a} = $Opt->{num_lanes};
    }
  }

  if ($Opt->{dual_port})
  {
    #add a bunch of new ports;
    e_port->adds(
      ['address2',    $Opt->{Address_Width}, 'in'],
      ['readdata2',   $Opt->{Data_Width},    'out'],
    );

    e_port->adds({name => "clken2", width => 1, direction => "in",
      default_value => "1'b1"});

    # Additional ports for dual-port ROM or RAM.
    $in_port_map->{clock1}      = 'clk';
    $in_port_map->{clocken1}    = 'clken2';
    $in_port_map->{address_b}   = 'address2';
    $in_port_map->{wren_b}      = 'wren2';
    $in_port_map->{data_b}      = 'writedata2';

    $out_port_map->{q_b} = 
      ($SBI2->{Read_Latency} == 1) ? 'readdata2' : 'readdata2_ram';

    # Pipeline read data if required.
    if ($SBI2->{Read_Latency} > 1) {
      e_register->add(
        {out => ["readdata2", $Opt->{Data_Width}],            
         in => "readdata2_ram",  
         enable => "clken2",
         delay => ($SBI2->{Read_Latency} - 1),
        }
      );
    }

    # Additional ports and signals for dual-port RAM.
    e_signal->adds(
      ['wren2', 1],
      ['write2', 1],
      ['chipselect2', 1],
      ['writedata2',  $Opt->{Data_Width}],
    );

    if ($Opt->{num_lanes} > 1)
    {
      e_signal->adds(
        ['byteenable', $Opt->{Data_Width} / 8],
        ['byteenable2', $Opt->{Data_Width} / 8],
      );
      $in_port_map->{byteena_b} = 'byteenable2';
      $parameter_map->{width_byteena_b} = $Opt->{num_lanes};
    }

    if ($Opt->{Writeable})
    {
      e_assign->add(['wren2', and_array('chipselect2', 'write2')]);
    }
    else
    {
      if ($project->is_hardcopy_compatible())
      {
        # Hardcopy is targeted, so wire the write-enable inactive and let
        # Quartus figure out that this is really a ROM (and can have
        # an init_file).
        e_assign->add(['wren2', 0]);
      }
      else
      {
        e_assign->add(['wren2', and_array('chipselect2', 'write2', 'debugaccess')]);
      }
    }

    # Parameters particular to dual-port memory.
    # Port A always uses clock 0 so no *_reg_a parameters exist or are needed.
    $parameter_map->{operation_mode} = qq("BIDIR_DUAL_PORT");
    $parameter_map->{width_b} = $Opt->{Data_Width};
    $parameter_map->{widthad_b} = $Opt->{Address_Width};
    $parameter_map->{numwords_b} = $Opt->{num_words};
    $parameter_map->{outdata_reg_b} = qq("UNREGISTERED");
    $parameter_map->{byteena_reg_b} = qq("CLOCK1");
    $parameter_map->{indata_reg_b} = qq("CLOCK1");
    $parameter_map->{address_reg_b} = qq("CLOCK1");
    $parameter_map->{wrcontrol_wraddress_reg_b} = qq("CLOCK1");
  }
  else
  {
    # Single-port RAM/ROM.
  }

  # Right now compilation and simulation versions of the altsyncram
  # have only very small differences, which show up in the parameters.
  
  # I could avoid the tags (have a single e_blind_instance) as follows:
  # For VHDL: specify the absolute path to the contents file, so that 
  # both Modelsim and Quartus can find it with no path issues.
  # For Verilog, use two levels of `ifdef:
  # if MODEL_TECH
  #   if NO_PLI
  #     dat file (in simulation directory)
  #   else
  #     ../hex file (up one level, in quartus directory)
  # else
  #   hex file (in quartus directory)
  # end
  # 
  # The decree is that there shall be no absolute path names in
  # files - so, stick with different compilation and simulation
  # instantiations.

  my $sim_parameter_map = {%$parameter_map};


  my $hdl_file_info = $Opt->{target_info}->{hex};
  my $sim_file_info = $Opt->{target_info}->{dat};
  
  if ($Opt->{make_individual_byte_lanes})
  {
    # Complicated instantiation for M512, which lacks byte enables.
    
    # Modify the parameter map and the input port map as needed, 
    # given that we're creating byte-lanes rather than a single
    # monolithic memory instantiation.
    $parameter_map->{width_a} = 8;
    $sim_parameter_map->{width_a} = 8;
    
    for my $lane (0 .. $Opt->{num_lanes} - 1)
    {
      e_assign->add(["write_lane$lane", and_array('wren', "byteenable\[$lane\]")]);
      $in_port_map->{wren_a} = "write_lane$lane";
      $in_port_map->{data_a} = sprintf("writedata[%d : %d]", ($lane + 1) * 8 - 1, $lane * 8);
      
      # SPR 188154 : if num of lanes > 1, out_port map is not checked for Read_latency
      $out_port_map->{q_a} = ($SBI1->{Read_Latency} == 1) ? 
      sprintf("readdata[%d : %d]", ($lane + 1) * 8 - 1, $lane * 8) :
      sprintf("readdata_ram[%d : %d]", ($lane + 1) * 8 - 1, $lane * 8);

      set_init_file_parameters(
        $Opt,
        $parameter_map,
        $sim_parameter_map,
        $hdl_file_info,
        $sim_file_info,
        $project,
      );

      # Create a synthesis-tagged memory module for this lane.
      e_blind_instance->add({
        tag    => 'synthesis',
        name   => "the_altsyncram_$lane",
        module => 'altsyncram',
        in_port_map => $in_port_map,
        out_port_map => $out_port_map,
        parameter_map => $parameter_map,
      });

      # Create a simulation-tagged memory module for this lane.
      e_blind_instance->add({
        tag    => 'simulation',
        name   => "the_altsyncram_$lane",
        module => 'altsyncram',
        in_port_map => $in_port_map,
        out_port_map => $out_port_map,
        parameter_map => $sim_parameter_map,
        use_sim_models => 1,
      });
    }
  }
  else
  {
    set_init_file_parameters(
      $Opt,
      $parameter_map,
      $sim_parameter_map,
      $hdl_file_info,
      $sim_file_info,
      $project,
    );

    if ($Opt->{ram_block_type} =~ /M-RAM/ )
    {
      # If this is an M-RAM that allows simulation contents, lie to the sim
      # model about the block type (Quartus' overzealous sim model doesn't
      # allow initialization for M-RAMs, so we have to fool it).
      $sim_parameter_map->{ram_block_type} = qq("M4K");
    }

    # Create a synthesis-tagged memory module.
    e_blind_instance->add({
      tag    => 'synthesis',
      name   => 'the_altsyncram',
      module => 'altsyncram',
      in_port_map => $in_port_map,
      out_port_map => $out_port_map,
      parameter_map => $parameter_map,
    });

    # Create a simulation-tagged memory module.
    e_blind_instance->add({
      tag    => 'simulation',
      name   => 'the_altsyncram',
      module => 'altsyncram',
      in_port_map => $in_port_map,
      out_port_map => $out_port_map,
      parameter_map => $sim_parameter_map,
      use_sim_models => 1,
    });
  }
  return $module;
}

sub set_init_file_parameters
{
  my (
    $Opt,
    $parameter_map,
    $sim_parameter_map,
    $hdl_file_info,
    $sim_file_info,
    $project,
  ) = @_;

  if ($hdl_file_info)
  {
    my $rec = shift @{$hdl_file_info->{targets}};
    # Explicitly perform altsyncram parameter map for the init file
    
    my $stub = $Opt->{init_contents_file};
    
    #if we're not targetting M512s, simple as
    my $hex = "$stub.hex";
    
    #otherwise append the byte lane index
    if($Opt->{make_individual_byte_lanes})
    {
      # Retrieve the byte-lane index
      split "lane", $rec->{ptf_key};
      $hex = $stub."_lane".$_[1].".hex";
    }
    
    $parameter_map->{init_file} = qq("$hex");
  }

  if ($sim_file_info)
  {
    my $rec = shift @{$sim_file_info->{targets}};
    # Explicitly perform altsyncram simulation parameter map for the init file
    my $stub = $Opt->{init_contents_file};
    
    #if we're not targetting M512s, simple as
    my $dat = "$stub.dat";
    my $hex = "$stub.hex";
    
    #otherwise append the byte lane index
    if($Opt->{make_individual_byte_lanes})
    {
      # Retrieve the byte-lane index
      split "lane", $rec->{ptf_key};
      $dat = $stub."_lane".$_[1].".dat";
      $hex = $stub."_lane".$_[1].".hex";
    }
    
    my $file;
    
    $file = join ("\n",'',
              '`ifdef NO_PLI',
              qq("$dat"),
              '`else',
              qq("../$hex"),
              "\`endif\n");

    if ($Opt->{lang} =~ /vhdl/i)
    {
      $file = qq("../$hex");
    }
    
    $sim_parameter_map->{init_file} = $file;
  }

  if ($Opt->{ram_block_type} =~ /M-RAM/ )
  {
    # If I set the init file field for this memory earlier,
    # but it's an M-RAM, fix that now.
    $parameter_map->{init_file} = qq("UNUSED");
  }
  
  # If we're targeting hardcopy, no init file for RAM.
  if ($Opt->{Writeable} && $project->is_hardcopy_compatible())
  {
    $parameter_map->{init_file} = qq("UNUSED");
    $sim_parameter_map->{init_file} = qq("UNUSED");
  }
}
