#Copyright (C)1991-2002 Altera Corporation
#Any megafunction design, and related net list (encrypted or decrypted),
#support information, device programming or simulation file, and any other
#associated documentation or information provided by Altera or a partner
#under Altera's Megafunction Partnership Program may be used only to
#program PLD devices (but not masked PLD devices) from Altera.  Any other
#use of such megafunction design, net list, support information, device
#programming or simulation file, or any other related documentation or
#information is prohibited for any other purpose, including, but not
#limited to modification, reverse engineering, de-compiling, or use with
#any other silicon devices, unless such use is explicitly licensed under
#a separate agreement with Altera or a megafunction partner.  Title to
#the intellectual property, including patents, copyrights, trademarks,
#trade secrets, or maskworks, embodied in any such megafunction design,
#net list, support information, device programming or simulation file, or
#any other related documentation or information provided by Altera or a
#megafunction partner, remains with Altera, the megafunction partner, or
#their respective licensors.  No other licenses, including any licenses
#needed under any third party's intellectual property, are provided herein.
#Copying or modifying any file, or portion thereof, to which this notice
#is attached violates this copyright.






################
# em_onchip_memory.pl
#
#
# This Perl-script is the "Generator_Program"
# for the SOPC-Builder component class "altera_avalon_ram".
#
# This generator program is very similar to the one you'll find 
# in the "altera_avalon_uart" library directory.  If you are interested 
# in reading an overlong comment which describes the operation 
# of, and philosophy behind, the standard set of generator programs,
# I strongly suggest that you review the uart's generator program.
# It is in a file named "mk_uart.pl" 
#
#
#
# Stratix Tri-Matrix Memory Block Usage for Automatic Setting.
#
# The GUI of the class.ptf file decides up front which memory block type will
# be used by the component.  The Perl file just follows the class.ptf's
# decision.
#
# The class.ptf follows these rules:
#
#   ROMs
# 
#     M512:  For memory sizes <=256 bytes
#     M4K:   For >=257 bytes
#     M-RAM: never
# 
#     Rationale:  You get to choose between M512s and M4Ks (M-RAMs never can be
#     ROMs).  In Stratix, M4K blocks use about 2.5x the Si area of M512 blocks.
#     And so really the cutover is between 128 bytes (1024 bits) and 256 bytes
#     (2048 bits).  (2*M512_area < M4K_area < 4*M512_area.)  But since M512s
#     are so numerous, I've pushed the cutover value up to favor their use for
#     256-byte ROMs.
# 
# 
#   RAM (blank)
# 
#     M512:  For memory sizes <=256 bytes
#     M4K:   For >= 257 bytes and < 20K bytes.
#     M-RAM: For > 20K bytes
# 
#     Rationale:  All types of Tri-Matrix memory blocks are available for use.
#     The cutover from M512 to M4K is the same as with ROMs.  M-RAMs are about
#     40x the Si area of M4K blocks.  The cutover is at 40 M4Ks, or 20,480
#     bytes.
#     
# 
#   RAM (initialized)
# 
#     M512:  never
#     M4K:   always
#     M-RAM: never
# 
#     Rationale:  M-RAMs cannot be initialized, and so they are out of the
#     running.  M512s are more difficult to initialize, and I don't want to
#     take the time to do this now.  The reason is that when M512s are used
#     for RAMs they are split into byte lanes, and I don't want to take the
#     time now to do that data conversion on the initialization files.  This
#     leaves M4Ks, which fortunately support initialization.
#        
# 
#   The generator needs to distinguish between memory blocks that will get
#   implemented as byte lanes (4 LEs for byte enables and 4 memory
#   instantiations) and those that get implemented as single blocks.  There is
#   an advantage to selecting one over the other, but the wizard doesn't need
#   to see this.
# 
#   The wizard displays which memory block type is going to be used by the
#   Automatic setting, but it doesn't say how many blocks will be used.  It may
#   depend on how the M4Ks are used.  For example, if someone wants 1,536 bytes
#   of memory, then that will take 3 M4Ks (single instantiation) or 4 M4Ks
#   (byte lanes).  I cannot predict what Quartus will do in these cases.
# 


use strict;
use europa_all;
use format_conversion_utils;
use wiz_utils;

#START:

my $project = e_project->new (@ARGV);
&make_mem ($project->top(), $project);
$project->output();

# DONE!

################################################################
# Validate_Memory_Options
#
# Checks all my PTF-parameters to be sure they're good.
#
################################################################
sub Validate_Memory_Options
{
  my ($Options, $SBI, $project) = (@_);

  # Oh, for goodness sake:  It's just nice to know our name:
  $Options->{name} = $project->_target_module_name();

  # Boolean variables specify what kind of I/O we have:
  #
  &validate_parameter ({hash    => $Options,
                        name    => "Writeable",
                        type    => "boolean",
                        default => 1,
                       });


#  &validate_parameter ({hash    => $Options,
#                        name    => "Contents",
#                        allowed => ["blank", 
#                                    "germs", 
#                                    "default_build", 
#                                    "custom_build"],
#                       });

  # Blank ROMs are legal, but suspicious.
  $Options->{is_blank}  = ($Options->{CONTENTS}{srec}{Kind} =~ /^blank/i) ? 
      "1" : "0";

  $Options->{is_file}   = 
      ($Options->{CONTENTS}{srec}{Kind} =~ /^textfile/i) ? 
      "1" : "0";

  my $textfile = $Options->{CONTENTS}{srec}{Textfile_Info};
  
  #turn bar/foo.srec relative path into an absolute one if needed
  my $system_directory = $project->_system_directory();
  $textfile =~ s/^(\w+)(\\|\/)/$system_directory$2$1$2/;
  #turn foo.srec to absolute path
  $textfile =~ s/^(\w+)$/$system_directory\/$1/;

  $Options->{textfile}   = $textfile;

  &validate_parameter ({hash     => $Options,
                        name     => "is_blank",
                        requires => "Writeable",
                        severity => "warning",
                        message  => "Blank ROM:  Legal, but suspicious.",
                       });


  # Figger out where our contents are coming from:

  my $Initfile = $Options->{name} . "_contents.srec";

  #
  # if not absolute path, prepend the project directory
  #

  if($Initfile !~ /^([a-zA-Z]:)?\/.*$/) # not start with X:/ or just slash...
  {
    $Initfile = $project->_system_directory() . "/" . $Initfile;
  }

  $Options->{Initfile} = $Initfile;

  # We only accept .mif- and .srec-files (or just blankness)
  &ribbit ("Memory-initialization files must be either .mif or .srec.\n",
           " not '$Options->{Initfile}'\n")
    unless $Options->{Initfile} =~ /\.(srec|mif)$/i;

  &validate_parameter ({hash     => $Options,
                        name     => "Shrink_to_fit_contents",
                        type     => "boolean",
                        default  => 0,
                     });

  $Options->{not_blank} = !$Options->{is_blank};
  &validate_parameter ({hash     => $Options,
                        name     => "Shrink_to_fit_contents",
                        requires => "not_blank",
                        message  => "How the heck can I shrink to fit blank?",
                       });

  &validate_parameter ({hash    => $SBI,
                        name    => "Data_Width",
                        type    => "integer",
                        allowed => [8, 16, 32, 64, 128,],
                       });

  # Create fictitious (derived) Option "Address_Span" from existing
  #   "Size_Value" and "Size_Multiple" options:
  &validate_parameter ({hash    => $Options,
                        name    => "Size_Multiple",
                        type    => "integer",
                        allowed => [1,1024],
                        default => 1,
                       });
  
  &validate_parameter ({hash    => $Options,
                        name    => "Size_Value",
                        type    => "integer",
                       });

  $Options->{Address_Span} = $Options->{Size_Multiple} * 
                             $Options->{Size_Value};

  # We actually -do- want to write this one (and only one) derived
  # parameter back into the PTF-file:
  $project->WSA()->{Address_Span} = $Options->{Address_Span};

  # Parameters for Stratix-specific memories.
  &validate_parameter ({hash     => $Options,
                        name     => "use_altsyncram",
                        type     => "boolean",
                        default  => 0,
                     });

  &validate_parameter ({hash    => $Options,
                        name    => "use_ram_block_type",
                        type     => "string",
                        allowed => ["M512", 
                                    "M4K", 
                                    "M-RAM"],
                        default  => "M4K",
                       });

  # Optional parameter in case we need to work around a bug.
  &validate_parameter ({hash    => $Options,
                        name    => "altsyncram_maximum_depth",
                        type    => "integer",
                        default  => 0,
                       });

  #### Derived parameters:
  # Number of byte-lanes.
  # 
  # When is a byte-lane not 8 bits wide?  When this is a ROM, and 
  # we pedantically think of the entire thing as one big "lane."
  # This makes RAM/ROM processing elsewhere more uniform.
  #
  $Options->{lane_width}    = $Options->{Writeable} ? 8 : $SBI->{Data_Width};
  $Options->{num_lanes}     = $SBI->{Data_Width}/$Options->{lane_width};
  $Options->{Address_Width} = $SBI->{Address_Width};
  $Options->{Data_Width}    = $SBI->{Data_Width};

  # The depth of the memory, in words, is handy to know.  
  # This is all-too-easily confused with the size of the RAM, in
  # bytes.  Not the same thing at all.
  my $byte_width = int($SBI->{Data_Width} / 8);
  $Options->{depth} = ceil ($Options->{Address_Span} / $byte_width);

  # The address-bus can be wider than required by the span.  This allows
  # memories to occupy more address-space than the actual, physically-
  # implemented memory block.
  &validate_parameter ({hash    => $SBI,
                        name    => "Address_Width",
                        type    => "integer",
                        range   => [&Bits_To_Encode($Options->{depth}-1), 32],
                        });

  # Annoyingly, Perl treats the SBI/Base_Address assignment as a
  # string if it starts with "0x:"
  $Options->{base_addr_as_number} = $SBI->{Base_Address};
  $Options->{base_addr_as_number} = hex ($Options->{base_addr_as_number})
      if ($Options->{base_addr_as_number} =~ /^0x/i);
      
  # We can create memories with read latency, and setup, wait and hold cycles.
  # This is probably only useful for debugging.
  &validate_parameter ({hash     => $SBI,
                        name     => "Read_Latency",
                        type     => "integer",
                        range    => [0, 10],
                        message  => "My, what a large Read_Latency you have.",
                        severity => "warning",
                        default  => 0,
                        });
  
  &validate_parameter ({hash     => $SBI,
                        name     => "Setup_Time",
                        type     => "integer",
                        range    => [0, 10],
                        message  => "My, what a large Setup_Time you have.",
                        severity => "warning",
                        default  => 0,
                        });
  &validate_parameter ({hash     => $SBI,
                        name     => "Hold_Time",
                        type     => "integer",
                        range    => [0, 10],
                        message  => "My, what a large Hold_Time you have.",
                        severity => "warning",
                        default  => 0,
                        });
  &validate_parameter ({hash     => $SBI,
                        name     => "Read_Wait_States",
                        type     => "integer",
                        range    => [0, 10],
                        message  =>
                          "My, what a lot of Read_Wait_States you have.",
                        severity => "warning",
                        default  => 0,
                        });
  &validate_parameter ({hash     => $SBI,
                        name     => "Write_Wait_States",
                        type     => "integer",
                        range    => [0, 10],
                        message  =>
                          "My, what a lot of Write_Wait_States you have.",
                        severity => "warning",
                        default  => 0,
                        });
                        
  $SBI->{no_latent_read} = $SBI->{Read_Latency} == 0;
  &validate_parameter ({hash     => $SBI,
                        name     => "Setup_Time",
                        requires => "no_latent_read",
                        message  => "setup time not allowed with latent read.",
                        });
  &validate_parameter ({hash     => $SBI,
                        name     => "Hold_Time",
                        requires => "no_latent_read",
                        message  => "hold time not allowed with latent read.",
                        });
  &validate_parameter ({hash     => $SBI,
                        name     => "Read_Wait_States",
                        requires => "no_latent_read",
                        message  =>
                          "read wait states not allowed with latent read.",
                        });
  &validate_parameter ({hash     => $SBI,
                        name     => "Write_Wait_States",
                        requires => "no_latent_read",
                        message  =>
                          "write wait states not allowed with latent read.",
                        });

  ################
  # Software-only mode.
  #   This script can get called in a mode where it only rebuilds
  #   the software-conents of the memory, without changing any of the 
  #   hardware.  This is set by the ARGV (command-line) parameter
  #   --software_only, and is thus not really a parameter in the same
  #   sense as our other options.  Still, it has consequence, and
  #   needs to be validated:
  # 
  warn ("Software-only modification for shrink-to-fit memory.\n".
        "  Contents may or may not still fit in hardware memory.\n")
      if ($project->_software_only() && $Options->{Shrink_to_fit_contents});
}

################################################################
# make_mem
#
# Given a name and a hashful of options, builds an e_module object
# which implements an onchip-memory peripheral PIO peripheral.
#
################################################################
sub make_mem
{
  my ($module, $project) = (@_);
  my $Opt = &copy_of_hash ($project->WSA());
  my $SBI = &copy_of_hash ($project->SBI("s1"));
  $Opt->{name} = $module->name();

  if ($Opt->{dual_port})
  {
     #sdk uses biggest base address to calculate memory so we might as
     #well too.
     my $SBI2 = &copy_of_hash ($project->SBI("s2"));
     if (eval($SBI2->{Base_Address}) > eval($SBI->{Base_Address}))
     {
        $SBI = $SBI2;
     }
  }

  &Validate_Memory_Options ($Opt, $SBI, $project);

  # First things first:  If the contents need to be built from
  # a process, then build them.
  # Note that, once the contents are built, the "depth" parameter
  # in $Opt might change if the memory is shrink-to-fit.
  #
  &process_mem_contents ($module->name(), $Opt, $SBI);

  if ($Opt->{use_altsyncram}) {
    &instantiate_altsyncram($module, $Opt, $project);
  } else {
    # Decide on the right starting-depth and minimum chunk-size.
    # This is a subtle art, as the comment atop this function will
    # describe:
    #
    &Figure_Out_Best_Depth ($Opt);

    # To support byte-wide writes, we need to segregate each individual
    # byte-lane into its own memory with its own write-signal:

    my @d_lane_list = ();
    my @q_lane_list = ();

    for (my $i = $Opt->{num_lanes} - 1; $i >= 0; $i--) {
      my $lane_d_sig = e_signal->new (["d_lane_$i", $Opt->{lane_width}]);
      my $lane_q_sig = e_signal->new (["q_lane_$i", $Opt->{lane_width}]);
      push (@d_lane_list, $lane_d_sig->name());
      push (@q_lane_list, $lane_q_sig->name());

      my $write_sig  = "0";   # Start with non-writeable assumption.
      if ($Opt->{Writeable}) {
        $write_sig = e_signal->new (["write_lane_$i", 1]);

        # Go through some shenanigans to hide bit indices from Leo when
        # byte-enable is only one bit wide:
        my $lane_wbe =  ($Opt->{num_lanes} == 1)    ? 
                            "writebyteenable"       : 
                            "writebyteenable\[$i\]" ;

        e_assign->new({within => $module,
                      lhs    => $write_sig,
                      rhs    => $lane_wbe,
                      });

        $module->add_contents 
          (e_signal->new (["writebyteenable", $Opt->{num_lanes}]));
      }

      # We use a recursive generator to divide the memory into 
      # powers-of-two "chunks."  Each chunk needs a select-signal from
      # its parent.  We start the recursion off with a "dummy"
      # always-one chunk-selector:
      #
      my $whole_mem_select_signal = e_signal->new (["select_all_chunks", 1,0,1]);
      e_assign->new ({within => $module,
                      lhs    => $whole_mem_select_signal,
                      rhs    => 1
                    });

      # Mif-files all start at zero, so we deduct this module's 
      # base-address from the address of this chunk.
      #
      my $initfile_base_address = 
          ($Opt->{Initfile} =~ /\.mif/i) ? 0 : $Opt->{base_addr_as_number};

      $module->add_contents
        ($lane_d_sig,
        $lane_q_sig,
        &make_chunked_memory ({name           => $module->name() . "_lane$i",
                                Writeable      => $Opt->{Writeable},
                                write          => $write_sig,
                                wrclock        => "clk",
                                d              => $lane_d_sig,
                                "q"            => $lane_q_sig,
                                outer_select   => $whole_mem_select_signal,
                                width          => $Opt->{lane_width},
                                depth          => $Opt->{depth},
                                Initfile       => $Opt->{Initfile},
                                address_width  => $SBI->{Address_Width},
                                lane           => $i,
                                num_top_lanes  => $Opt->{num_lanes},
                                base           => $initfile_base_address,
                                bytes_per_word => $SBI->{Data_Width} / 8,
                                Read_Latency   => $SBI->{Read_Latency},
                                project        => $project,
                                }),
        );
    }

    # Aggregate all lanes into read/write data signals.
    e_assign->new
      ({within => $module,
        lhs    => e_port->new(["readdata", $SBI->{Data_Width}, "out"]),
        rhs    => &concatenate(@q_lane_list),
      });

    e_assign->new
      ({within => $module,
        lhs    => &concatenate(@d_lane_list),
        rhs    => e_port->new(["writedata", $SBI->{Data_Width}, "in"]),
      }) if $Opt->{Writeable};
  }

  # Since signal-widths don't "bubble" through assignments, specify 
  # address-port width explicitly:
  e_port->new({within    => $module,
               name      => "address",
               width     => $SBI->{Address_Width},
               direction => "in",
              });

  # Our slave-port name is taken from the SLAVE section which 
  # is named in our class.ptf file.
  my $s1_args = {
     within => $module, 
     name   => "s1",
     sideband_signals => [ "clken" ],
  };

  e_avalon_slave->new($s1_args);

  my %slave_2_type_map = reverse
  (
     address    => "address2",
     readdata   => "readdata2",
     chipselect => "chipselect2",
     write      => "write2",
     writedata  => "writedata2",
     byteenable => "byteenable2",
     clken      => "clken2",
  );

  my $s2_args = {
     within => $module,
     name   => "s2",
     type_map => \%slave_2_type_map,
     sideband_signals => [ "clken" ],
  };

  e_avalon_slave->new($s2_args);

  return $module;
}

################################################################
# make_chunked_memory
# 
# We're smarter than Quartus! (or so it would seem).   In Quartus, if 
# you ask for a 6Kbyte memory, it gives you an 8Kbyte memory.  It
# always gives you an lpm-ram/rom object which is the size you
# requested...rounded-up to the next higher power-of-two words.  That
# can sometimes mean you buy a lot more memory than you ask for.
# ESB-limited applications, which are real and common, can ill afford
# this kind of wasteage.  
#
# Thanks to Rod Frazer for suggesting this approach.  A few more of
# these and we might like him as much as we like Marc Gaucheron!
#
# This routine is recursive, and needs these named arguments:
#  * name:         The name of this particular chunk
#  * Writeable:    Boolean.  Says whether to build RAMs or ROMs.
#  * d, q, write:  The d-inputs, q-outputs, and write-enable (signals).
#  * width:        How many bits wide to make this memory.
#  * depth:        The size, in words, of this chunk.
#  * file:         The init-file from whence to get init-data.
#  * lane:         An integer indicating which lane this is (for contents).
#  * base:         The absolute base-address (byte-address) of this chunk.
#
# This routine returns a *list* of "things" (e_ram/e_roms, e_assigns, e_muxes,
# etc).  If you happened to request a memory which is a power-of-two deep,
# your list will consist of one e_ram/e_rom instance.  If you request an
# "odd" depth, you'll get a list of instances-and-interconnections 
# that optimally "builds" the memory you asked-for out of even-sized mems.
#
################################################################
sub make_chunked_memory
{
  my ($arg) = (@_);

  # Guarantee that we don't build tiny-little chunks (e.g. depth 1 or 5).
  $arg->{depth} = max($arg->{minimum_chunk_size}, $arg->{depth});

  # Recursion end-condition:  If the user happens to ask for a power-of-two.
  if (&is_power_of_two($arg->{depth})) {
    return &make_one_memory_chunk ($arg);
  }

  # Build a composite memory:  Power-of-two size, plus leftovers:
  # We call the power-of-two part "chunk," and the rest "leftover".
  # Both chunk and leftover need their own separate write-enable 
  # and q-output signals.  They also get their own select-signal,
  # decoded from the address, which is used to mux the q's and 
  # as a term in the write-enables.
  #
  my $chunk_size         = &next_lower_power_of_two ($arg->{depth});
  my $leftover_size      = $arg->{depth} - $chunk_size;
  my $chunk_addr_bits    = &Bits_To_Encode($chunk_size - 1);
  my $leftover_base      = $arg->{base} + $chunk_size * $arg->{bytes_per_word};
  my $chunk_name         = $arg->{name} . "_chunk_$chunk_size";
  my $leftover_name      = $arg->{name} . "_leftover";

  my $chunk_select_sig    = e_signal->new(["select_$chunk_name",       1]);
  my $chunk_mux_select_sig= e_signal->new(["mux_select_$chunk_name",   1]);
  my $leftover_select_sig = e_signal->new(["select_$leftover_name",    1]);
  my $chunk_write_sig     = e_signal->new(["write_$chunk_name",        1]);
  my $leftover_write_sig  = e_signal->new(["write_$leftover_name",     1]);
  my $chunk_q_sig         = e_signal->new(["q_$chunk_name",    $arg->{width}]);
  my $leftover_q_sig      = e_signal->new(["q_$leftover_name", $arg->{width}]);

  &ribbit ("Odd.  leftover size < 0 ($arg->{depth} - $chunk_size\n") 
    if $leftover_size < 0;

  # The arguments I pass down to myself (recursively) look alot like the
  # COPIES OF the arguments I got, with a few critical substitutions:
  my %chunk_arg    = %{$arg};
  my %leftover_arg = %{$arg};

  $chunk_arg{name}             = $chunk_name;
  $chunk_arg{write}            = $chunk_write_sig;
  $chunk_arg{"q"}              = $chunk_q_sig;
  $chunk_arg{depth}            = $chunk_size;
  $chunk_arg{address_width}    = &Bits_To_Encode ($chunk_size - 1);
  $chunk_arg{outer_select}     = $chunk_select_sig;
  
  $leftover_arg{name}          = $leftover_name;
  $leftover_arg{write}         = $leftover_write_sig;
  $leftover_arg{"q"}           = $leftover_q_sig;
  $leftover_arg{depth}         = $leftover_size;
  $leftover_arg{address_width} = &Bits_To_Encode ($leftover_size - 1);
  $leftover_arg{base}          = $leftover_base;
  $leftover_arg{outer_select}  = $leftover_select_sig;

  # Extract the address-bits which are too high for this chunk.  Use
  # them to determine if the chunk is selected, or if the leftovers 
  # are selected.  Any nonzero value in the high-bits indicates the 
  # leftovers are selected.
  my @result = ();
  push 
    (@result, 
     e_signal->news ({name         => "$chunk_name\_lo_addr",
                      width        => $chunk_addr_bits,
                      never_export => 1,
                     },
                     {name         => "$chunk_name\_hi_addr",
                      width        => $arg->{address_width} - $chunk_addr_bits,
                      never_export => 1,
                     },
                    ),

     e_assign->new (["{$chunk_name\_hi_addr, $chunk_name\_lo_addr}", 
                     "address"]),
    );

  push (@result, 
    $chunk_select_sig,
    $leftover_select_sig,
    $chunk_q_sig,
    $leftover_q_sig,

    # the chunk-part is selected if:
    # The composite memory, of which we are a part, is selected, and
    # None of the high address bits are true.
    e_assign->news 
      ({lhs => $chunk_select_sig,
        rhs => "(~|($chunk_name\_hi_addr)) && ". $arg->{outer_select}->name(),
       },
       { lhs => $leftover_select_sig,
         rhs => "( |($chunk_name\_hi_addr)) &&". $arg->{outer_select}->name(),
       }),

    e_mux->new ({lhs     => $arg->{"q"},
                 table   => [$chunk_mux_select_sig => $chunk_q_sig],
                 default => $leftover_q_sig,
                }),
  );

  # This is nasty trickiness.  If the memory has read-latency, then you 
  # need to delay the chunk-selector mux-control signal to agree with the 
  # data coming out of the memories.  Which means you want to select the 
  # chunk that they asked for N times ago, where N is the read-latency
  # of the memory.  Easy enough:
  #
  if ($arg->{Read_Latency}) {
     push (@result, 
           e_register->new ({out    => $chunk_mux_select_sig,
                             in     => $chunk_select_sig->name(),
                             delay  => $arg->{Read_Latency},
                             enable => "1'b1",
                          }),
           );
  } else { 
     push (@result,
           e_assign->new ({lhs => $chunk_mux_select_sig,
                           rhs => $chunk_select_sig->name(),
                        }),
           );
  }
                               
  # We can go to a lot of trouble to avoid generating this thing when 
  # it's not used, or else we can just let Leo eat it.  
  # I choose the latter, but I have
  # to prevent it from "bubbling":
  #
  $leftover_select_sig->never_export(1);

  # Only emit write-enable logic if the memory is writeable:
  #
  if ($arg->{Writeable}) {
    push (@result,

      e_assign->new 
        ({lhs => $chunk_write_sig,
          rhs => $chunk_select_sig->name() . "&&" . $arg->{write}->name(),
        }),

      # Note:  Leftover select-signal is not even generated unless
      #         the memory is writeable.  Fascinating.
      e_assign->new 
        ({lhs => $leftover_write_sig,
          rhs => $leftover_select_sig->name() . "&&" . $arg->{write}->name(),
         }),
    );
  }

  push (@result,
    # The even-power-of-two-sized memory chunk:
    &make_chunked_memory (\%chunk_arg),

    # The leftover memory chunk:
    &make_chunked_memory (\%leftover_arg),
  );
  return @result;
}

################################################################
# make_one_memory_chunk
#
# Function that actually does the memory-creation work for 
# "make_chunked_memory."  Takes all the same args, except all we
# do is create a memory--not some composite of multiple memories.
#
# This is little more than just building an e_ram/e_rom object, 
# unless the memory has an initialization-file.  Then we have to
# "grab" the right part of the initialization file, tease it apart
# from the rest of the data, and create the assoicated MIF-file.
#
################################################################
sub make_one_memory_chunk
{
  my ($arg) = (@_);

  # This is only supposed to get called for things that are even
  # powers-of-two in size:
  &ribbit ("Memory-chunk created which is not a power-of-two size")
    if !is_power_of_two($arg->{depth});

  # Extract the proper portion of the address, assign to dedicated signal:
  #
  my $addr_bits   = &Bits_To_Encode ($arg->{depth} - 1);
  my $address_sig = e_signal->new ([$arg->{name} . "_address", $addr_bits]);

  # Create a mif-file for this chunk by extracting the appropriate
  # portion of the specified contents:
  #
  my $chunk_mif_filename = &create_mif_file_for_chunk ($arg);

  my @result = ();
  push (@result, e_assign->new ([$address_sig, "address"]));

  my $comment = 
    "Atomic memory instance: $arg->{name}\n" . 
    "  Size: $arg->{depth}(words) X $arg->{width}(bits)\n";

  # If these signals with properly-declared widths aren't added
  # explicitly to the module, I claim the widths will be lost and 
  # bogus default signals will be added instead.  
  push (@result, $arg->{"q"},  $arg->{d});

  if ($arg->{Writeable}) {
    push (@result,
      e_ram->new({name     => $arg->{name},
                  comment  => $comment,
                  port_map => {
                               wren      => $arg->{write},
                               wrclock   => $arg->{wrclock},
                               data      => $arg->{d},
                               rdaddress => $address_sig,
                               wraddress => $address_sig,
                               "q"       => $arg->{"q"},
                               # rdclken   => "1'b1",
                              },
                  mif_file => $chunk_mif_filename,
                  Read_Latency      => $arg->{Read_Latency},
                  })
         );
  } else {
    push (@result,
      e_rom->new({name     => $arg->{name},
                  comment  => $comment,
                  port_map => {
                               address   => $address_sig,
                               "q"       => $arg->{"q"},
                              },
                  mif_file => $chunk_mif_filename,
                  Read_Latency      => $arg->{Read_Latency},
                 })
         );
  }
  return @result;
}

################################################################
# Figure_Out_Best_Depth
#
# Chunkifying your memory saves you ESBs by not making memories
# which are senselessly too big.  But it costs you LEs, because
# we need to mux outputs from all the chunks.
#
# This trade-off tends to argue strongly against making lots of 
# tiny-little chunks in a foolish attempt to make a memory which 
# is -exactly- as big as you asked for, at huge expense, when a 
# slightly-larger memory could have (perhaps) been made in a single
# chunk.
#
# To guard against potential computer-generated stupidness, we make
# some common-sense trade-offs. 
#
# First: The accuracy we strive for is expressed as a -percentage- 
# of the total memory size.  If you're making a 3K memory,  1K of wasted
# ESBs might be a pretty big deal to you.  If you're building a 
# 253K memory, burning-off 3K is no sweat, and certainly not worth 
# the (horrific) cost in complexity and LEs.
#
# So we arbitrarily set the "target" memory-size accuracy at 14%.
# Thus, the smallest chunk we dane to make is 14% of the size of your
# memory.  Note that this always limits the number of chunks you get
# to 3, because of "math" (cool, huh?).  Also, the minimum chunk-size
# is rounded-up to the nearest even multiple of some rock-bottom 
# minimum (128).
# 
# Either of these parameters can be overridden from the WSA section 
# of the PTF, in the unlikely event that you are some sort of mutant.
#
# The result of this function is that it sets the $Opt->{depth} and 
# $Opt->{minimum_chunk_size} parameters to give you a reasonable 
# common-sense outcome without unduly-many chunks.
#
################################################################
sub Figure_Out_Best_Depth
{
  my ($Opt) = (@_);

  if ($Opt->{minimum_chunk_size} eq "")  # If user-set, leave it alone.
  {
    $Opt->{minimum_chunk_size} = int (max (128, $Opt->{depth} * 0.07));
  }

  # Minimum chunk-size must always be a power of two:
  $Opt->{minimum_chunk_size} = 
    &next_higher_power_of_two($Opt->{minimum_chunk_size});

  # Round depth up to even power-of-two (or next even multiple of 
  # chunk size), whichever is smaller:
  my $next_higher_power_of_two = &next_higher_power_of_two($Opt->{depth});
  my $num_total_chunks = ceil ($Opt->{depth} / $Opt->{minimum_chunk_size});
  my $next_chunk_multiple = $num_total_chunks * $Opt->{minimum_chunk_size};
  $Opt->{depth} = min ($next_higher_power_of_two, $next_chunk_multiple);
}

################################################################
# create_mif_file_for_chunk
# 
# Create a mif-file for a "chunk" (address-range and lane) of memory
# by extracting the appropriate sub-portion of the specified 
# contents.
#
# The incoming $arg-hash is a copy of (reference to) all the parameters 
# passed to &make_chunked_memory.
#
################################################################
sub create_mif_file_for_chunk
{
  my ($arg) = (@_);
  
  
  my $mif_filename = $arg->{name} . ".mif";
  my $mif_pathname = $arg->{project}->_system_directory() . "/" . 
                     $mif_filename;
  my $chunk_addr_span = $arg->{depth} * $arg->{bytes_per_word};

  # Set up to run nios-convert -twice-.  Once for the mif-file, and
  # then again for the dat-file:
  # For whatever reason, the source-file's argument name is "0" and
  # the destination-file argument name is "1".  How...numerical.

  my $chunk_offset_into_file = $arg->{base};
  if ($arg->{Initfile} =~ /\.mif$/i) { 
     $chunk_offset_into_file -= $arg->{Base_Address};
  }
  
  my $convert_args = {0            => $arg->{Initfile},
                      "1"          => $mif_pathname,
                      comments     => "0",
                      lane         => $arg->{lane},
                      lanes        => $arg->{num_top_lanes},
                      width        => $arg->{width},
                      address_low  => $arg->{base},
                      address_high => $arg->{base} + $chunk_addr_span - 1,
                   };

  $convert_args->{oformat}  = "mif";
  $convert_args->{destfile} = $mif_pathname;

  &fcu_convert ($convert_args);

  return $mif_filename;
}

################################################################
# create_mif_file_for_altsyncram
# 
################################################################
sub create_mif_file_for_altsyncram
{
  my ($Opt, $project) = (@_);
  
  my $mif_filename = $Opt->{name} . ".mif";
  my $mif_pathname = $project->_system_directory() . "/" . 
                     $mif_filename;

  # Set up to run nios-convert for the mif-file.
  # It appears to be run automatically for the dat-file.
  # For whatever reason, the source-file's argument name is "0" and
  # the destination-file argument name is "1".  How...numerical.

  my $convert_args = {0            => $Opt->{Initfile},
                      "1"          => $mif_pathname,
                      comments     => "0",
                      width        => $Opt->{Data_Width},
                   };

  $convert_args->{oformat}  = "mif";
  $convert_args->{destfile} = $mif_pathname;

  &fcu_convert ($convert_args);

  return $mif_filename;
}

################################################################
# create_hex_file_for_altsyncram
# 
################################################################
sub create_hex_file_for_altsyncram
{
  my ($Opt, $project) = (@_);
  
  
  my $hex_filename = $Opt->{name} . ".hex";
  my $hex_pathname = $project->simulation_directory() . "/" . 
                     $hex_filename;

  # Set up to run nios-convert for the mif-file.
  # It appears to be run automatically for the dat-file.
  # For whatever reason, the source-file's argument name is "0" and
  # the destination-file argument name is "1".  How...numerical.

  my $convert_args = {0            => $Opt->{Initfile},
                      "1"          => $hex_pathname,
                      comments     => "0",
                      width        => $Opt->{Data_Width},
                   };

  $convert_args->{oformat}  = "hex";
  $convert_args->{destfile} = $hex_pathname;

  &fcu_convert ($convert_args);

  #For modelsim, . is really ..
  $hex_pathname =~ s/^(\.[\\\/])/\.$1/s;
  return $hex_filename;
}

################################################################
# run_contents_process
#
# "System"s the content_creation_program, and deals with all the 
# error-reporting hooha.
#
# As a courtesy, we check to make sure the users' chosen
# contents-file is actually modified when we run this process.
# It sure would be suspicious if it weren't. 
#
# IMPORTANT NOTE: IT SHOULD BE $Opt->{Initfile} not $Opt->{initfile}
# I'M NOT CHANGING IT SINCE NOBODY RUNS THIS SUBROUTINE
################################################################
sub run_contents_process
{
  my ($mem_name, $Opt) = (@_);

  my $original_file_date = &get_file_modified_date ($Opt->{initfile});

                        type     => "boolean",
                        requires => "content_creation_program",

  my $return_code = &System_Win98_Safe ($Opt->{content_creation_program});
  if ($return_code != 0) {
    &ribbit ("Error executing content-creation program.\n",
             "  On-chip memory named: '$mem_name'\n",
             "  Command was:          '$Opt->{content_creation_program}\n",
             "  Error was:            '$?\n");
  }

  my $later_file_date = &get_file_modified_date($Opt->{initfile});
  if ($later_file_date <= $original_file_date) {
    &ribbit ("Suspicious content-creation process for on-chip memory.\n",
             " Initialization-file did not appear to get modified.\n",
             "  On-chip memory named: '$mem_name'\n",
             "  Command was:          '$Opt->{content_creation_program}\n",
             "  Initialization file:  '$Opt->{initfile}\n",
             "  Error was:            '$?\n");
  }
}

################################################################
# process_mem_contents
#
# Runs the "content_creation_program," if any.  Figures-out the 
# size of the file, and makes sure the memory is big enough.  
# If you've selected the "shrink_to_fit" option, it 
# goes in and monkeys with the $Opt->{depth} parameter to make the 
# minimally-sufficient memory.
#
################################################################
sub process_mem_contents
{
  my ($mem_name, $Opt, $SBI) = (@_);

  my @arg_o_list = ($Opt, $SBI, $project);

  # Make the srec file if "blank" or "file" is selected.  We do it here
  # because it's possible for the user to select a blank ram file in a
  # system with no cpu and thus no sdk generation.  This is a bit
  # redudant with the current sdk generation, but whatever, dude.

  my $base_address = $Opt->{base_addr_as_number};
  my $end_address = $base_address + $Opt->{Address_Span} - 1;

  if ($Opt->{is_blank} || $Opt->{is_file})
  {      
     my $convert_hash = {
        1            => $Opt->{Initfile},
        address_low  => $base_address,
        address_high => $end_address,
     };

     $convert_hash->{0} = $Opt->{textfile}
     if ($Opt->{is_file});

     &fcu_convert($convert_hash);
  }

  # Figure-out the address-span of the initialization file,
  # and see if it's too big (or outside the address-range of) the 
  # on-chip memory.
  my ($size, $lo_addr, $hi_addr) =
    &fcu_get_address_range ($Opt->{Initfile});

  # Most MIF-files assume a base address of 0, no matter where we think
  # they're mapped.    If we encounter a MIF-file with a base of zero,
  # Adjust lo_addr and hi_addr accordingly. Warn the user about what
  # we're doing, though.
  # 
  # If the MIF-file actually has a nonzero base-address, then I guess
  # the customer knows what they're doing, so we accept whatever they
  # say.
  #
  if ( ($Opt->{Initfile} =~ /.mif$/i) && ($lo_addr == 0) ) {
     $lo_addr += $Opt->{base_addr_as_number};
     $hi_addr += $Opt->{base_addr_as_number};
  }

  &ribbit ("Contents for memory named '$mem_name' contains data outside\n",
           " the memory's specified address-range\n",
           "   Contents file (init_file) is: $Opt->{Initfile}\n",
           "      Contains data between addresses [0x",
           sprintf("%X,",$lo_addr), "..0x", sprintf("%X",$hi_addr), "]",
           " (", sprintf ("%.3g", $size / 1024.0), "Kbytes)\n",
           "   Specified address range for memory is\n",
           "                                      [0x",
           sprintf("%X",$Opt->{base_addr_as_number}), "..0x", 
           sprintf("%X",$end_address), "]",
           " (", sprintf ("%.3g", $Opt->{Address_Span} / 1024.0), 
           "Kbytes)\n")
    if (($lo_addr < $Opt->{base_addr_as_number}) ||
        ($hi_addr > $end_address               )  );
  if ($Opt->{Shrink_to_fit_contents}) {
    $Opt->{depth} = ceil ($size / ($SBI->{Data_Width} / 8));
  }
}

sub figger_sdk_dir
{
   my ($project) = (@_);
   return $project->_system_directory() . "/".
          $project->_system_name()      . "_sdk";
}

################################################################
# run_default_build_process
#
# So...the user has a bunch of c- or s- files, and wants us to use
# the default-build process (or, perhaps, they have an 
# srec- or mif-file).  We can tell which by the boolean option
# "run_nios_build", which was set up where the filename arguments
# were validated.
# 
################################################################
sub run_default_build_process
{
  my ($Opt, $SBI, $project) = (@_);

  # Don't have to do anything if user specified a pre-made MIF or SREC file.
  return unless ($Opt->{run_nios_build});

  # Shucks.  I guess we have to do something.  How 'bout running
  # the much-maligned "nios-build" utility?

  my $sdk_dir = &figger_sdk_dir($project);

  my $cmd = "";
     $cmd .= $project->_sopc_directory();
     $cmd .= '\bin\nios-build';
     $cmd .= " -b $SBI->{Base_Address}";
     $cmd .= " -o $Opt->{Initfile}";
     $cmd .= " -sdk_directory=$sdk_dir";
     $cmd .= " $Opt->{extra_build_flags}";

  # Build files specified in WSA 
  foreach my $fname (split(/\s*,\s*/, $Opt->{Source_files})) {
     $cmd .= " $fname";
  }

  print STDERR ("about to try this: '$cmd'\n");
  my $error_code = 
    &Run_Command_In_Unix_Like_Shell ($project->_sopc_directory(), $cmd);
  &ribbit ("Default build-process failed.\n".
           " Attempted command: ($cmd).\n") 
    if $error_code != 0;
}

sub run_custom_build_process
{
  my ($Opt, $SBI, $project) = (@_);

  my $cmd = $Opt->{Custom_build_command};
  
  print STDERR ("about to try this: '$cmd'\n");
  my $error_code = 
    &Run_Command_In_Unix_Like_Shell ($project->_sopc_directory(), $cmd);
  &ribbit ("Custom build-process failed.\n".
           " Attempted command: ($cmd).\n" .
           "error code: '$error_code'") 
    if $error_code != 0;
}


################################################################
# instantiate_altsyncram
#
# Creates a memory and instantiates it in parameter $module.
#
# This function may be used for either RAMs or ROMs, and supports 
# 8, 16, or 32 bits wide and byte enables.
# 
# I'm sorry for the complicated code, but it was the most modular way of
# supporting so many parameters.
#
################################################################
sub instantiate_altsyncram
{
    my ($module, $Opt, $project) = (@_);

    # This is a fine kettle of fish.  Almost always, 
    # we create mif and dat files for this memory.  But if it's a
    # stratix-style M-RAM and the user hasn't explicitly requested the
    # contents file to be allowed for simulation purposes, don't 
    # create contents files.
    my $contents_file = $Opt->{name};

    if ($Opt->{use_ram_block_type} =~ /M-RAM/ && 
      !$Opt->{allow_mram_sim_contents_only_file})
    {
      $contents_file = '';
    }

    if ($contents_file)
    {
      $Opt->{mif_file} = &create_mif_file_for_altsyncram ($Opt, $project);
      $Opt->{hex_file} = &create_hex_file_for_altsyncram ($Opt, $project);
      $Opt->{dat_file} = $Opt->{name} . ".dat";
    }                       

    my $marker = e_default_module_marker->new ($module);

    e_port->adds(
        ["clk",        1,                     "in" ],
        ["address",    $Opt->{Address_Width}, "in" ],
        ["readdata",   $Opt->{Data_Width},    "out"],
        );

    e_port->adds({name => "clken", width => 1, direction => "in",
      default_value => "1'b1"});

    #For modelsim, . is really ..
    $contents_file =~ s/^(\.[\\\/])/\.$1/s;

    # account for whether it's a RAM or ROM.  add ports and dpram parameters as
    # needed.
    if ($Opt->{Writeable}) {      # it's a RAM
      e_assign->add (["chipselect_and_write", "chipselect & write"]);
      e_port->adds(
        ["chipselect", 1,                     "in" ],
        ["write",      1,                     "in" ],
        ["writedata",  $Opt->{Data_Width},    "in" ],
        );

    } else {                      # it's a ROM
      # er... um... nothing else needs to happen, actually.
    }

    # More than one byte lane? add ports and dpram as needed.
    my $lane_uniqueness_ref = {};
    if ($Opt->{num_lanes} > 1) {
        e_port->adds(
            ["byteenable", $Opt->{num_lanes},     "in" ],
            );
        $lane_uniqueness_ref = {
            ram_block_type       => qq(") . $Opt->{use_ram_block_type} . qq("),
            maximum_depth        => $Opt->{altsyncram_maximum_depth},
            port_map => {
                byteenable => "byteenable",
            },
        };
    } else {    # only one byte lane.
        # no additional work needed.
    }

    if ($Opt->{dual_port}) {
       #########################
       # TRUE DUAL PORT MEMORY #
       # 2 read/write ports    #
       #########################

       #add a bunch of new ports;
       e_port->adds(["address2",    $Opt->{Address_Width}, "in" ],
                    ["readdata2",   $Opt->{Data_Width},    "out" ],
                    );

       e_port->adds({name => "clken2", width => 1, direction => "in",
         default_value => "1'b1"});

       my $in_port_map = {
                address_a  => 'address',
                address_b  => 'address2',
                clock0     => 'clk',
                clock1     => 'clk',
                clocken0   => 'clken',
                clocken1   => 'clken2',
             };

       my $out_port_map = {
                q_a       => 'readdata',
                q_b       => 'readdata2',
             };

       my $wren;
       my $wren2;
       my $be2;
       my $be;
       if ($Opt->{Writeable})
       {
          e_signal->adds([write2 => 1],
                         [chipselect2 => 1],
                         ["writedata2",  $Opt->{Data_Width}],
                         [byteenable  => ($Opt->{Data_Width} / 8)],
                         [byteenable2 => ($Opt->{Data_Width} / 8)],
                         );

          e_assign->adds([[wren => 1], 'chipselect_and_write'],
                         [[wren2 => 1], &and_array("chipselect2","write2")],
                         );

          $in_port_map->{byteena_a} = 'byteenable';
          $in_port_map->{byteena_b} = 'byteenable2';
          $in_port_map->{wren_a}    = 'wren';
          $in_port_map->{wren_b}    = 'wren2';
          $in_port_map->{data_a}    = 'writedata',
          $in_port_map->{data_b}    = 'writedata2',
       }

       my $num_words = $Opt->{depth} || die ("no depth");
       my $block_type = $Opt->{use_ram_block_type};

       my $parameter_map = {
          operation_mode            => qq("BIDIR_DUAL_PORT"),
          width_a                   => $Opt->{Data_Width},
          widthad_a                 => $Opt->{Address_Width},
          numwords_a                => $num_words,
          width_b                   => $Opt->{Data_Width},
          widthad_b                 => $Opt->{Address_Width},
          numwords_b                => $num_words,
          lpm_type                  => qq("altsyncram"),
          width_byteena_a           => $Opt->{num_lanes},
          width_byteena_b           => $Opt->{num_lanes},
          byte_size                 => 8,
          outdata_reg_a             => qq("UNREGISTERED"),
          outdata_aclr_a            => qq("NONE"),
          outdata_reg_b             => qq("UNREGISTERED"),
          indata_aclr_a             => qq("NONE"),
          wrcontrol_aclr_a          => qq("NONE"),
          address_aclr_a            => qq("NONE"),
          byteena_aclr_a            => qq("NONE"),
          indata_reg_b              => qq("CLOCK1"),
          address_reg_b             => qq("CLOCK1"),
          wrcontrol_wraddress_reg_b => qq("CLOCK1"),
          indata_aclr_b             => qq("NONE"),
          wrcontrol_aclr_b          => qq("NONE"),
          address_aclr_b            => qq("NONE"),
          byteena_reg_b             => qq("CLOCK1"),
          byteena_aclr_b            => qq("NONE"),
          outdata_aclr_b            => qq("NONE"),
          ram_block_type            => '"'.$Opt->{use_ram_block_type}.'"',
          intended_device_family    => qq("Stratix"),
       };

       if ((!$project->is_hardcopy_compatible() || !$Opt->{Writeable}) &&
         ($Opt->{use_ram_block_type} eq 'M4K' ||
           ($Opt->{use_ram_block_type} eq 'M-RAM' && 
            $Opt->{allow_mram_sim_contents_only_file})))
       {
          #everything gets instantiated the same in quartus
          my %comp_parameter = %$parameter_map;

          # Don't want to pass an init_file to an M-RAM.
          if ($Opt->{use_ram_block_type} eq 'M4K') {
            $comp_parameter{init_file} = '"'.$Opt->{mif_file}.'"';
          }

          e_blind_instance->add({
             tag            => 'compilation',
             use_sim_models => 1,
             name           => 'the_altsyncram',
             module         => 'altsyncram',
             in_port_map    => $in_port_map,
             out_port_map   => $out_port_map,
             parameter_map  => \%comp_parameter,
          });

          #now it gets interesting for simulation
          my %sim_parameter = %$parameter_map;
          my $sim_file;
          if ($project->language() eq 'vhdl')
          {
             $sim_file = '"'.$Opt->{hex_file}.'"';
          }
          elsif ($project->language() eq 'verilog')
          {
             $sim_file = join ("\n",'',
                               '`ifdef NO_PLI',
                               '"'.$Opt->{dat_file}.'"',
                               '`else',
                               '"'.$Opt->{hex_file}.'"',
                               "\`endif\n");
          }
          else
          {
             die "unknown language ($project->language())\n";
          }

          # Can only get here if an M4K simulation model is required.
          # This occurs when the user asks for an M4K or when they
          # ask for an M-RAM and explicitly allow the contents file
          # to be specified for simulation only.
          $sim_parameter{ram_block_type} = '"M4K"';

          $sim_parameter{init_file} = $sim_file;
          e_blind_instance->add({
             tag            => 'simulation',
             use_sim_models => 1,
             name           => 'the_altsyncram',
             module         => 'altsyncram',
             in_port_map    => $in_port_map,
             out_port_map   => $out_port_map,
             parameter_map  => \%sim_parameter,
          });
       }
       else
       {
          e_blind_instance->add({
             use_sim_models => 1,
             name   => 'the_altsyncram',
             module => 'altsyncram',
             in_port_map => $in_port_map,
             out_port_map => $out_port_map,
             parameter_map => $parameter_map,
          });
       }
    } else {
       #############################
       # SIMPLE DUAL PORT MEMORY   #
       # 1 read port, 1 write port #
       #############################

       # regardless of parameters, there are certain keys that all dprams need.
       my $basic_dpram_hashref = {
           name                 => $Opt->{name} . "_memory_array",
           stratix_style_memory => 1,
           data_width           => $Opt->{Data_Width},
           address_width        => $Opt->{Address_Width},
           write_pass_through   => 0,
           contents_file        => $contents_file,
           allow_mram_sim_contents_only_file => 
             $Opt->{allow_mram_sim_contents_only_file},
           port_map => {
               # write port mappings are dealt with below.
               # read port map.
               rdclock   => "clk",
               rdclken   => 'clken',
               rdaddress => "address",
               q         => "readdata",
               },
       };

       if ($Opt->{Writeable}) {      # it's a RAM
          my $write_port_mappings = {
                wrclock   => "clk",
                wrclken   => "clken",
                data      => "writedata",
                wren      => "chipselect_and_write",
                wraddress => "address",
          };

          map {     # map all write ports into both 
            $basic_dpram_hashref->{port_map}{$_} = $$write_port_mappings{$_};
          } keys (%$write_port_mappings);
       }

       # take those byte-lane uniquenesses (above) and write 'em into the
       # dpram_hash.
       foreach my $key (keys (%$lane_uniqueness_ref)) {
         if (ref ($lane_uniqueness_ref->{$key}) eq "HASH")  {
           # if there's additional port_map work to be done, we need to add
           # ports within an already-made port_map. 
           # Here's a second layer of mapping!  (oh-m-gawd!)
           foreach my $key2 (keys (%{$lane_uniqueness_ref->{$key}})) {
             $basic_dpram_hashref->{$key}{$key2} = 
                           $lane_uniqueness_ref->{$key}{$key2};
           }
         } else {    # key is not a hash; it's just a humdrum key.
           $basic_dpram_hashref->{$key} = $lane_uniqueness_ref->{$key};
         }
       }

       # and finally, serve the dpram we've been preparing.
       e_dpram->add( $basic_dpram_hashref );
    }
    return $module;
}

