/******************************************************************************
*                                                                             *
* License Agreement                                                           *
*                                                                             *
* Copyright (c) 2003 Altera Corporation, San Jose, California, USA.           *
* All rights reserved.                                                        *
*                                                                             *
* Permission is hereby granted, free of charge, to any person obtaining a     *
* copy of this software and associated documentation files (the "Software"),  *
* to deal in the Software without restriction, including without limitation   *
* the rights to use, copy, modify, merge, publish, distribute, sublicense,    *
* and/or sell copies of the Software, and to permit persons to whom the       *
* Software is furnished to do so, subject to the following conditions:        *
*                                                                             *
* The above copyright notice and this permission notice shall be included in  *
* all copies or substantial portions of the Software.                         *
*                                                                             *
* THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR  *
* IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,    *
* FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE *
* AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER      *
* LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING     *
* FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER         *
* DEALINGS IN THE SOFTWARE.                                                   *
*                                                                             *
* This agreement shall be governed in all respects by the laws of the State   *
* of California and by the laws of the United States of America.              *
*                                                                             *
******************************************************************************/

#include <string.h>
#include <fcntl.h>
#include <errno.h>
#include <limits.h>

#include <sys/stat.h>

#include "sys/alt_dev.h"
#include "sys/alt_irq.h"
#include "sys/alt_alarm.h"
#include "sys/ioctl.h"
#include "alt_types.h"

#include "altera_avalon_jtag_uart_regs.h"
#include "altera_avalon_jtag_uart.h"

#include "sys/alt_log_printf.h"

#ifndef ALTERA_AVALON_JTAG_UART_SMALL

/* ----------------------------------------------------------------------- */
/* ---------------------- ROUTINES FOR FAST DRIVER ----------------------- */
/*
 *
 */

/*
 *
 */

int alt_avalon_jtag_uart_read(alt_fd* fd, char * buffer, int space)
{
  alt_avalon_jtag_uart_dev* dev = (alt_avalon_jtag_uart_dev*) fd->dev; 
  char * ptr = buffer;

  alt_irq_context context;
  unsigned int n;

  /*
   * When running in a multi threaded environment, obtain the "read_lock"
   * semaphore. This ensures that reading from the device is thread-safe.
   */
  ALT_SEM_PEND (dev->read_lock, 0);

  while (space > 0)
  {
    unsigned int in, out;

    /* Read as much data as possible */
    do
    {
      in  = dev->rx_in;
      out = dev->rx_out;

      if (in >= out)
        n = in - out;
      else
        n = ALT_AVALON_JTAG_UART_BUF_LEN - out;

      if (n == 0)
        break; /* No more data available */

      if (n > space)
        n = space;

      memcpy(ptr, dev->rx_buf + out, n);
      ptr   += n;
      space -= n;

      dev->rx_out = (out + n) % ALT_AVALON_JTAG_UART_BUF_LEN;
    }
    while (space > 0);

    /* If we read any data then return it */
    if (ptr != buffer)
      break;

    /* If in non-blocking mode then return error */
    if (fd->fd_flags & O_NONBLOCK)
      break;

    /* TODO: in the multi-threaded context this should be waiting on an event */
    /* Spin until more data arrives or until host disconnects */
    while (in == dev->rx_in && dev->host_inactive < dev->timeout)
      ; 
    
    if (in == dev->rx_in)
      break;
  }

  /*
   * Now that access to the circular buffer is complete, release the read
   * semaphore so that other threads can access the buffer.
   */

  ALT_SEM_POST (dev->read_lock);

  if (ptr != buffer)
  {
    /* If we read any data then there is space in the buffer so enable interrupts */
    context = alt_irq_disable_all();
    dev->irq_enable |= ALTERA_AVALON_JTAG_UART_CONTROL_RE_MSK;
    IOWR_ALTERA_AVALON_JTAG_UART_CONTROL(dev->base, dev->irq_enable);
    alt_irq_enable_all(context);
  }

  if (ptr != buffer)
    return ptr - buffer;
  else if (fd->fd_flags & O_NONBLOCK)
    return -EWOULDBLOCK;
  else
    return -EIO;
}

/*
 *
 */

int alt_avalon_jtag_uart_write(alt_fd* fd, const char * ptr, int count)
{
  /* Remove warning at optimisation level 03 by seting out to 0 */
  unsigned int in, out=0;
  unsigned int n;
  alt_irq_context context;

  alt_avalon_jtag_uart_dev* dev = (alt_avalon_jtag_uart_dev*) fd->dev;
  const char * start = ptr;

  /*
   * When running in a multi threaded environment, obtain the "write_lock"
   * semaphore. This ensures that writing to the device is thread-safe.
   */
  ALT_SEM_PEND (dev->write_lock, 0);

  do
  {
    /* Copy as much as we can into the transmit buffer */
    while (count > 0)
    {
      /* We need a stable value of the out pointer to calculate the space available */
      in  = dev->tx_in;
      out = dev->tx_out;

      if (in < out)
        n = out - 1 - in;
      else if (out > 0)
        n = ALT_AVALON_JTAG_UART_BUF_LEN - in;
      else
        n = ALT_AVALON_JTAG_UART_BUF_LEN - 1 - in;

      if (n == 0)
        break;

      if (n > count)
        n = count;

      memcpy(dev->tx_buf + in, ptr, n);
      ptr   += n;
      count -= n;

      dev->tx_in = (in + n) % ALT_AVALON_JTAG_UART_BUF_LEN;
    }

    /*
     * TODO: if interrupts are disabled then we could transmit here, we only need to enable
     * interrupts if there is no space left in the FIFO
     */

    /* For now kick the interrupt routine every time to make it transmit the data */
    context = alt_irq_disable_all();
    dev->irq_enable |= ALTERA_AVALON_JTAG_UART_CONTROL_WE_MSK;
    IOWR_ALTERA_AVALON_JTAG_UART_CONTROL(dev->base, dev->irq_enable);
    alt_irq_enable_all(context);

    /* If there is any data left then either return now or block until some has been sent */
    /* TODO: test whether there is anything there while doing this and delay for at most 2s. */
    if (count > 0)
    {
      if (fd->fd_flags & O_NONBLOCK)
        break;

      /* TODO: in the multi-threaded context this should be waiting on an event */

      /* Wait for data to be removed from buffer.  Once the interrupt routine
      ** has removed some data then we will be able to insert some more. */
      while (out == dev->tx_out && dev->host_inactive < dev->timeout)
        ;

      if (out == dev->tx_out)
        break;
    }
  }
  while (count > 0);

  /*
   * Now that access to the circular buffer is complete, release the write
   * semaphore so that other threads can access the buffer.
   */
  ALT_SEM_POST (dev->write_lock);

  if (ptr != start)
    return ptr - start;
  else if (fd->fd_flags & O_NONBLOCK)
    return -EWOULDBLOCK;
  else
    return -EIO; /* Host not connected */
}

/*
 * Interrupt routine
 */ 

static void alt_avalon_jtag_uart_irq(void* context, alt_u32 id)
{
  alt_avalon_jtag_uart_dev* dev = (alt_avalon_jtag_uart_dev*) context;
  unsigned int base = dev->base;

  /* ALT_LOG - see altera_hal/HAL/inc/sys/alt_log_printf.h */ 
  ALT_LOG_JTAG_UART_ISR_FUNCTION(base, dev);

  /* TODO: turn on events where appropriate */

  for ( ; ; )
  {
    unsigned int control = IORD_ALTERA_AVALON_JTAG_UART_CONTROL(base);

    /* Return once nothing more to do */
    if ((control & (ALTERA_AVALON_JTAG_UART_CONTROL_RI_MSK | ALTERA_AVALON_JTAG_UART_CONTROL_WI_MSK)) == 0)
      break;

    if (control & ALTERA_AVALON_JTAG_UART_CONTROL_RI_MSK)
    {
      /* process a read irq.  Start by assuming that there is data in the
       * receive FIFO (otherwise why would we have been interrupted?)
       */
      unsigned int data = 1 << ALTERA_AVALON_JTAG_UART_DATA_RAVAIL_OFST;

      for ( ; ; )
      {
        /* Check whether there is space in the buffer.  If not then we must not
         * read any characters from the buffer as they will be lost.
         */
        unsigned int next = (dev->rx_in + 1) % ALT_AVALON_JTAG_UART_BUF_LEN;
        if (next == dev->rx_out)
          break;

        /* Try to remove a character from the FIFO and find out whether there
         * are any more characters remaining.
         */
        data = IORD_ALTERA_AVALON_JTAG_UART_DATA(base);
        
        if ((data & ALTERA_AVALON_JTAG_UART_DATA_RVALID_MSK) == 0)
          break;

        dev->rx_buf[dev->rx_in] = (data & ALTERA_AVALON_JTAG_UART_DATA_DATA_MSK) >> ALTERA_AVALON_JTAG_UART_DATA_DATA_OFST;
        dev->rx_in = (dev->rx_in + 1) % ALT_AVALON_JTAG_UART_BUF_LEN;
      }

      if (data & ALTERA_AVALON_JTAG_UART_DATA_RAVAIL_MSK)
      {
        /* If there is still data available here then the buffer is full 
         * so turn off receive interrupts until some space becomes available.
         */
        dev->irq_enable &= ~ALTERA_AVALON_JTAG_UART_CONTROL_RE_MSK;
        IOWR_ALTERA_AVALON_JTAG_UART_CONTROL(base, dev->irq_enable);
      }
    }

    if (control & ALTERA_AVALON_JTAG_UART_CONTROL_WI_MSK)
    {
      /* process a write irq */
      unsigned int space = (control & ALTERA_AVALON_JTAG_UART_CONTROL_WSPACE_MSK) >> ALTERA_AVALON_JTAG_UART_CONTROL_WSPACE_OFST;

      while (space > 0 && dev->tx_out != dev->tx_in)
      {
        IOWR_ALTERA_AVALON_JTAG_UART_DATA(base, dev->tx_buf[dev->tx_out]);

        dev->tx_out = (dev->tx_out + 1) % ALT_AVALON_JTAG_UART_BUF_LEN;
        space--;
      }

      if (space > 0)
      {
        /* If we don't have any more data available then turn off the TX interrupt */

        dev->irq_enable &= ~ALTERA_AVALON_JTAG_UART_CONTROL_WE_MSK;
        IOWR_ALTERA_AVALON_JTAG_UART_CONTROL(dev->base, dev->irq_enable);
      }
    }
  }
}

/*
 *
 */

int alt_avalon_jtag_uart_ioctl (alt_fd* fd, int req, void* arg)
{
  alt_avalon_jtag_uart_dev* dev = (alt_avalon_jtag_uart_dev*) fd->dev;
  int rc = -ENOTTY;

  switch (req)
  {
  case TIOCSTIMEOUT:
    /* Set the time to wait until assuming host is not connected */
    if (dev->timeout != INT_MAX)
    {
      int timeout = *((int *)arg);
      dev->timeout = (timeout >= 2 && timeout < INT_MAX) ? timeout : INT_MAX - 1;
      rc = 0;
    }
    break;

  case TIOCGCONNECTED:
    /* Find out whether host is connected */
    if (dev->timeout != INT_MAX)
    {
      *((int *)arg) = (dev->host_inactive < dev->timeout) ? 1 : 0;
      rc = 0;
    }
    break;

  default:
    break;
  }

  return rc;
}


/*
 * Timeout routine is called every second
 */

alt_u32 alt_avalon_jtag_uart_timeout(void * context) 
{
  unsigned int control;

  alt_avalon_jtag_uart_dev* dev = (alt_avalon_jtag_uart_dev *) context;

  control = IORD_ALTERA_AVALON_JTAG_UART_CONTROL(dev->base);

  if (control & ALTERA_AVALON_JTAG_UART_CONTROL_AC_MSK)
  {
    IOWR_ALTERA_AVALON_JTAG_UART_CONTROL(dev->base, dev->irq_enable | ALTERA_AVALON_JTAG_UART_CONTROL_AC_MSK);
    dev->host_inactive = 0;
  }
  else if (dev->host_inactive < INT_MAX - 2)
    dev->host_inactive++;

  return alt_ticks_per_second();
}

/* Driver initialisation code.  Register interrupts and start a timer
 * which we can use to check whether the host is there.
 */

void alt_avalon_jtag_uart_init(alt_avalon_jtag_uart_dev* dev, int base, int irq)
{
  ALT_FLAG_CREATE(&dev->events, 0);
  ALT_SEM_CREATE(&dev->read_lock, 1);
  ALT_SEM_CREATE(&dev->write_lock, 1);

  /* enable read interrupts at the device */
  dev->irq_enable = ALTERA_AVALON_JTAG_UART_CONTROL_RE_MSK;

  IOWR_ALTERA_AVALON_JTAG_UART_CONTROL(base, dev->irq_enable); 
  
  /* register the interrupt handler */
  if (alt_irq_register(irq, dev, alt_avalon_jtag_uart_irq) >= 0)
  {
    /* make the device available to the system */
    alt_dev_reg(&dev->dev);
  }

  /* Register an alarm to go off every second to check for presence of host */
  dev->host_inactive = 0;

  if (alt_alarm_start(&dev->alarm, alt_ticks_per_second(), &alt_avalon_jtag_uart_timeout, dev) < 0)
  {
    /* If we can't set the alarm then record "don't know if host present" 
     * and behave as though the host is present.
     */
    dev->timeout = INT_MAX;
  }

  /* ALT_LOG - see altera_hal/HAL/inc/sys/alt_log_printf.h */ 
  ALT_LOG_JTAG_UART_ALARM_REGISTER(dev, base);

}

#else

/* ----------------------------------------------------------------------- */
/* ----------------------- ROUTINES FOR SMALL DRIVER --------------------- */

/* Read routine.  The small version blocks until it has at least one byte
 * available, it then returns as much as is immediately available without
 * waiting any more.  It's performance will be very poor without
 * interrupts.
 */

int alt_avalon_jtag_uart_read(alt_fd* fd, char * buffer, int space)
{
  alt_avalon_jtag_uart_dev * dev = (alt_avalon_jtag_uart_dev*) fd->dev; 
  unsigned int base = dev->base;

  char * ptr = buffer;
  char * end = buffer + space;

  while (ptr < end)
  {
    unsigned int data = IORD_ALTERA_AVALON_JTAG_UART_DATA(base);

    if (data & ALTERA_AVALON_JTAG_UART_DATA_RVALID_MSK)
      *ptr++ = (data & ALTERA_AVALON_JTAG_UART_DATA_DATA_MSK) >> ALTERA_AVALON_JTAG_UART_DATA_DATA_OFST;
    else if (ptr != buffer)
      break;
    else if(fd->fd_flags & O_NONBLOCK)
      break;   
    
  }

  if (ptr != buffer)
    return ptr - buffer;
  else if (fd->fd_flags & O_NONBLOCK)
    return -EWOULDBLOCK;
  else
    return -EIO;
}

/* Write routine.  The small version blocks when there is no space to write
 * into, so it's performance will be very bad if you are writing more than
 * one FIFOs worth of data.  But you said you didn't want to use interrupts :-)
 */

int alt_avalon_jtag_uart_write(alt_fd* fd, const char * ptr, int count)
{
  alt_avalon_jtag_uart_dev * dev = (alt_avalon_jtag_uart_dev*) fd->dev; 
  unsigned int base = dev->base;

  const char * end = ptr + count;

  while (ptr < end)
    if ((IORD_ALTERA_AVALON_JTAG_UART_CONTROL(base) & ALTERA_AVALON_JTAG_UART_CONTROL_WSPACE_MSK) != 0)
      IOWR_ALTERA_AVALON_JTAG_UART_DATA(base, *ptr++);

  return count;
}

#endif
