#Copyright (C)1991-2002 Altera Corporation
#Any megafunction design, and related net list (encrypted or decrypted),
#support information, device programming or simulation file, and any other
#associated documentation or information provided by Altera or a partner
#under Altera's Megafunction Partnership Program may be used only to
#program PLD devices (but not masked PLD devices) from Altera.  Any other
#use of such megafunction design, net list, support information, device
#programming or simulation file, or any other related documentation or
#information is prohibited for any other purpose, including, but not
#limited to modification, reverse engineering, de-compiling, or use with
#any other silicon devices, unless such use is explicitly licensed under
#a separate agreement with Altera or a megafunction partner.  Title to
#the intellectual property, including patents, copyrights, trademarks,
#trade secrets, or maskworks, embodied in any such megafunction design,
#net list, support information, device programming or simulation file, or
#any other related documentation or information provided by Altera or a
#megafunction partner, remains with Altera, the megafunction partner, or
#their respective licensors.  No other licenses, including any licenses
#needed under any third party's intellectual property, are provided herein.
#Copying or modifying any file, or portion thereof, to which this notice
#is attached violates this copyright.



use europa_all;
use em_epcs;
use em_spi;

# This module lives in ../altera_avalon_cfi_flash/.
# Because component generator functions have every other component's directory
# in @INC, the module is found.
use refdes_check;

use strict;

my $slave_name = get_slave_name();


sub get_contents_file_name
{
  my $project = shift;
  my $extension = shift;
  
  my $top_module = $project->top();
  my $top_level_module_name = $top_module->name();
  
  # If an extension was requested, make sure it starts
  # with '.'.
  if ($extension && $extension !~ /^\./)
  {
    $extension = '.' . $extension;
  }
  return "$top_level_module_name\_boot_rom$extension";
}

sub copy_and_convert_contents_file
{
  my $project = shift;
  
  my $top_module = $project->top();
  my $top_level_module_name = $top_module->name();
  my $SBI = $project->SBI($slave_name);
  
  # The actual data for the ROM in this component belongs to the mastering
  # CPU.  
  # Find all CPU masters of this component
  my @master_list = $project->get_my_cpu_masters_through_bridges(
      $top_level_module_name,
      $slave_name
    );
    
  # Verify that all such masters are the same class - otherwise, it's an error.
  my %master_classes =
    map {
      ($project->system_ptf()->{"MODULE $_"}->{class}, 1)
    } @master_list;

  # Use the first CPU master, unless overridden below.
  my $chosen_cpu = "";
  if (keys %master_classes < 1)
  {
    #
    #  No CPU! we'll generate zero'd contents later.
    #  But for now, scold the poor user. dvb05
    #
    print STDERR (
      "Warning: $top_level_module_name has no mastering CPU,\n" .
      "         so its boot rom will be blank.\n");
  }

  if (keys %master_classes == 1)
  {
    # easiest clearest case: exactly one cpu. yay.
    $chosen_cpu = $master_list[0];
  }

  if (keys %master_classes > 1)
  {
    $chosen_cpu = $master_list[0];

    print STDERR (
      "Warning: CPU masters of different classes for component " .
      "$top_level_module_name: ",
      keys %master_classes, "\n"
    );
    print STDERR ("assuming that these different CPU masters are code-compatible, " .
      "or that only one will actually execute code from this component.\n"
    );
    
    # Prefer a master that resets from this component.
    for (@master_list)
    {
      if (hex($project->get_cpu_reset_address($_)) ==
        hex($SBI->{Base_Address}))
      {
        $chosen_cpu = $_;
      }
    }
    print STDERR "using contents from master '$chosen_cpu'\n";
  }

  require "format_conversion_utils.pm";

  # args contains all parameters common to all output file types.
  my $args =
  {
    comments     => "0",
    width        => $SBI->{Data_Width},
    address_low  => 0,
    address_high => get_code_size($project) - 1,
  };

  # Find the contents file
  # (or leave "infile" unassigned to generate zeroes)

  if($chosen_cpu ne "")
  {
    #
    # IF there's actually a CPU chosen, then we blithely
    # assume that we can find an appropriate slug of
    # boot loader code. (If you can't, you should just
    # not assign the "infile" argument...)
    #
    my $chosen_cpu_class =
      $project->system_ptf()->{"MODULE $chosen_cpu"}->{class};
    my $cpu_wsa = $project->WSA($chosen_cpu);

    my %find_component_dir_context;
    $find_component_dir_context{system_directory} = $project->__system_directory();
    my $boot_copier_dir = find_component_dir(\%find_component_dir_context,'', $chosen_cpu_class);
    my $boot_copier_srec;

    # Stratix II has its own boot copier with its own file name
    if(uc($project->system_ptf()->{WIZARD_SCRIPT_ARGUMENTS}->{device_family})
       eq "STRATIXII")
    {
      $boot_copier_srec = "$boot_copier_dir/$cpu_wsa->{Boot_Copier_EPCS_Stratix_II}";
    }
    else
    {
      $boot_copier_srec = "$boot_copier_dir/$cpu_wsa->{Boot_Copier_EPCS}";
    }

    $$args{infile} = $boot_copier_srec;
  }
  
  # Convert to output files, copy to destinations.
  

  # The hash references within contents_file_spec list the
  # parameters that differ among the output file types.
  # Warning: the assumption is made below that "oformat"
  # is the same as the desired file extension.
  my @contents_file_spec = (
    {
      oformat => 'hex',
      outfile => $project->__system_directory() . "/",
    },
  );
  
  # Add the filename, with appropropriate extension, to hash element
  # 'outfile' (output_file).
  map {
    $_->{outfile} .= get_contents_file_name($project, $_->{oformat})
  } @contents_file_spec;
  
  
  for my $file_spec (@contents_file_spec)
  {
    my %specific_args = %$args;
    for (keys %$file_spec)
    {
      $specific_args{$_} = $file_spec->{$_};
    }
    
    format_conversion_utils::fcu_convert(\%specific_args);
  }
}



# -------------------------------------------------------------
# MAIN

if (!@ARGV)
{
  # Make a class.ptf.
  make_epcs();
}
else
{
  # Goal: create a component which contains an SPI master, and wire its 
  # ports to a tornado_spiblock atom.  Use the existing SPI generator to
  # create the SPI master.
  #
  # Problem: the SPI generator makes an SPI master whose module name is the
  # top-level component name.  The code below calls the SPI generator to
  # create the SPI master module, then renames that module so that it can
  # be used as a component within the top-level (component) module, along
  # with the tornado_spiblock atom.
  my $project = make_spi(@ARGV);

  my $error = refdes_check::check($project);
  if ($error)
  {
    print STDERR "\nERROR:\n$error\n";
    ribbit();
  }
    
  my $SBI     = &copy_of_hash($project->SBI($slave_name));

  my $top_module = $project->top();
  
  # I'll base various names of things on this component's top module name,
  # even though I fully expect it to always be 'epcs' at this point.
  my $top_level_module_name = $top_module->name();

  # I must update, to fill in the data base.
  $top_module->update();

  # Get the SPI ports.
  my @inner_ports = $top_module->get_object_names("e_port");

  # Rename the SPI module.
  my $new_name = $top_level_module_name . "_sub";
  my $inner_mod = $project->module_hash()->{$top_level_module_name};
  $inner_mod->name($new_name);
  $project->module_hash()->{$new_name} = $inner_mod;
  delete $project->module_hash()->{$top_level_module_name};
  
  # Make a new module, with the original top module name.
  my $module = e_module->new({
    name => $top_level_module_name,
    project => $project,
  });

  $module->add_contents(
    e_instance->new({
      module => $new_name,
    }),
  );

  # All inner module non-export ports become new top module ports--
  #   with the following specific exceptions:
  # 
  #     chipselect   -- The SPI module is only selected in the
  #                     high-half of this address range.
  # 
  #     readdata     -- Needs to be multiplexed 
  #
  #     writedata    -- The outer module's data is wider (32 bits).
  #
  #     address      -- The outer module's address is wider.
  #
  my @port_list = ();
  my %spi_port_names_by_type;

  foreach my $port_name (@inner_ports)
  {
    my $port = $top_module->get_object_by_name($port_name);

    ribbit() if not $port;
    ribbit() if not ref($port) eq "e_port";

    # Below, we might be interested in knowing the names of various
    # types.  Write them in a handy hash:
    #
    $spi_port_names_by_type{$port->type()} = $port_name;

    next if ($port->type() eq ''          ) || 
            ($port->type() eq 'address'   ) ||
            ($port->type() eq 'chipselect') ||
            ($port->type() eq 'writedata' ) || 
            ($port->type() eq 'readdata'  )  ;

    
    push @port_list, e_port->new({
        name => $port->name(),
        width => $port->width(), 
        direction => $port->direction(),
        type => $port->type(),
      });
  }

  # Now give the outer-module the "special" ports from above:
  push (@port_list, e_port->news(
    {name      => "address",    
     type      => "address",
     width     => $SBI->{Address_Width},
     direction => "input", 
    },
    {name      => "writedata",    
     type      => "writedata",    
     width     => 32,
     direction => "input", 
    },
    {name      => "readdata",    
     type      => "readdata",    
     width     => 32,
     direction => "output", 
    },
  ));

  $module->add_contents(@port_list);

  # Make an avalon slave port.
  my %type_map = ();
  map {$type_map{$_->name()} = $_->type()} @port_list;
  
  # My slave control port.
  $module->add_contents(
    e_avalon_slave->new({
      name => 'epcs_control_port',
      type_map => \%type_map,
    })
  );

  # Create a wrapper module for the tornado_spiblock.  Why?  We need
  # to write this module into its own file, so that VHDL can cope with
  # the simulation model.
  my $tspi_name = 'tornado_' . $top_level_module_name . '_atom';
  my $tspi_module = e_module->new({
    name => $tspi_name,
    project => $project,
  });
  $tspi_module->do_black_box(1);
  
  # This module definition will go in its own separate simulation and
  # compilation files.  Make the file name the same as the module, 
  # so Quartus can find it.
  
  $tspi_module->add_contents(
    e_port->new(['dclkin', 1, 'input',]),
    e_port->new(['scein', 1, 'input',]),
    e_port->new(['sdoin', 1, 'input',]),
    e_port->new(['oe', 1, 'input',]),
    e_port->new(['data0out', 1, 'output',]),
    e_blind_instance->new({
      tag => 'synthesis',
      name => 'the_tornado_spiblock',
      module => 'tornado_spiblock',
      in_port_map => {
        dclkin => 'dclkin',
        scein => 'scein',
        sdoin => 'sdoin',
        oe => 'oe',
      },
      out_port_map => {
        data0out => 'data0out',
      },
    }),
    # Simple simulation contents, because (I think) e_module::to_vhdl() won't
    # output an empty simulation module.
    e_assign->new({
      tag => 'simulation',
      lhs => 'data0out',
      rhs => 'sdoin | scein | dclkin | oe',
    }),
  );
  
  # Instantiate a Tornado SPI widget.
  $module->add_contents(
    e_instance->new({
      module => $tspi_name,
      port_map => {
        dclkin => 'SCLK',
        scein => 'SS_n',
        sdoin => 'MOSI',
        # For now, always drive oe low (active).  Someday someone
        # might want to control the oe drive level from software.
        # Note that oe must be drive low to enable driving on the device
        # output pins, contrary to what the documentation
        # (tornado_config_doc.doc) says.
        oe => "1'b0",
        data0out => 'MISO',
      },
    }),
  );

  #Instantiate our boot-copier ROM.
  if ((&Bits_To_Encode(get_code_size($project) - 1) - 2) > ($SBI->{Address_Width} - 1))
  {
     my $addr_bits = $SBI->{Address_Width};
     ribbit ("EPCS Boot copier program (@{[get_code_size($project)]}) too big for  address-range (($addr_bits)");
  }

  my $rom_data_width = $SBI->{Data_Width};
  my $bytes_per_word = $rom_data_width / 8;
  my $rom_address_width = &Bits_To_Encode(get_code_size($project) / $bytes_per_word - 1);
  my $rom_parameter_map = {
    init_file                 => qq("@{[get_contents_file_name($project, 'hex')]}"),
    operation_mode            => qq("ROM"),
    width_a                   => $SBI->{Data_Width},
    widthad_a                 => $rom_address_width,
    numwords_a                => get_code_size($project) / $bytes_per_word,
    lpm_type                  => qq("altsyncram"),
    byte_size                 => 8,
    outdata_reg_a             => qq("UNREGISTERED"),
    read_during_write_mode_mixed_ports => qq("DONT_CARE"),
  };

  # nifty how ROMs only have two inputs and one output:
  my $rom_in_port_map  = { address_a => sprintf("address[%d : 0]", $rom_address_width - 1),
                           clock0    => 'clk'          };
  my $rom_out_port_map = { q_a       => 'rom_readdata' };

  $module->add_contents(
    e_blind_instance->new({                        
      tag           => 'synthesis',
      name          => 'the_boot_copier_rom',
      module        => 'altsyncram',
      in_port_map   => $rom_in_port_map,
      out_port_map  => $rom_out_port_map,
      parameter_map => $rom_parameter_map,
   })
  );

  # 1) The init_file setting for this simulated ROM is wrong: it needs to have
  #    the usual `ifdef NO_PLI song and dance.
  # 2) The contents files for this simulated ROM should be the standard elf2dat/elf2hex
  #    file that an onchip memory would have.  This will contain the appropriate
  #    jump-to-start code (assuming that the reset address is to this ROM. The simulation
  #    hex file must not conflict with the hardware-target hex file, so be sure to override
  #    e_project::do_makefile_target_ptf_assignments()'s notion of the contents filename,
  #    and set init_file appropriately.
  if ($project->language() eq 'verilog')
  {
    $rom_parameter_map->{init_file} =
      join ("\n",'',
        '`ifdef NO_PLI',
        qq("@{[get_contents_file_name($project, 'dat')]}"),
        '`else',
        qq("@{[get_contents_file_name($project, 'hex')]}"),
        "\`endif\n");
  }
  else
  {
    $rom_parameter_map->{init_file} =
      qq("@{[get_contents_file_name($project, 'hex')]}");
  }
  $module->add_contents(
    e_blind_instance->new({                        
      tag           => 'simulation',
      name          => 'the_boot_copier_rom',
      module        => 'altsyncram',
      in_port_map   => $rom_in_port_map,
      out_port_map  => $rom_out_port_map,
      parameter_map => $rom_parameter_map,
      use_sim_models => 1,
   })
  );

  copy_and_convert_contents_file($project);

  # Generate the chipselect-signal for the SPI module.
  # While we're here, assign the address & writedata going into the 
  # module, too.
  my $address_msb = (&Bits_To_Encode(get_code_size($project) - 1) - 1) + 1;
  $address_msb -= 2;    # Word address --> byte address
  $module->add_contents (
      e_assign->new ({
         lhs => $spi_port_names_by_type {"chipselect"},
         rhs => "chipselect && (address \[ $address_msb \] )",
      }),
      e_assign->new ({
         lhs => $spi_port_names_by_type {"address"},
         rhs => "address",
      }),
      e_assign->new ({
         lhs => $spi_port_names_by_type {"writedata"},
         rhs => "writedata",
      }),
  );

  # Now we have to build a multiplexer for readdata
  # 
  my $spi_chipselect = $spi_port_names_by_type {"chipselect"};
  my $spi_readdata   = $spi_port_names_by_type {"readdata"  };
  $module->add_contents (
      e_signal->new({name => 'rom_readdata', width => $rom_data_width,}),
      e_assign->new ({
         lhs => "readdata",
         rhs => "$spi_chipselect ? $spi_readdata : rom_readdata",
      })
  );

  $project->add_module($module);  
  $project->top($module);

  # Ensure that register_offset tracks any change made
  # in the hardware.
  #
  # Note: Perform this update before
  # calling do_makefile_target_ptf_assignments below;
  # apparently it is dependant on the correctness of this
  # PTF setting.
  $project->system_ptf()->{"MODULE $top_level_module_name"}->
    {WIZARD_SCRIPT_ARGUMENTS}->{register_offset} =
    sprintf("0x%X", get_code_size($project));

  my @targets = ('flashfiles', 'dat', 'hex', 'programflash', 'sym');

  $project->do_makefile_target_ptf_assignments(
    'epcs_control_port',
    \@targets,

    # Override the default contents file name.
    {name => get_contents_file_name($project)},
  );


  $project->output();
}

