=head1 NAME

mk_clock_crossing.pl - Avalon bus clock crossing adaptor generator

=head1 VERSION

1.1

=head1 SYNOPSIS

This script generates the avalon bus clock crossing adaptor bridging
masters and slaves in two asynchronous clock domains.

=head1 DESCRIPTION

=head2 Overview

The purpose of the adaptor is to provide a safe, birectional
avalon bus bridge between two clock domains.  Filtering out of
metastability conditions via a bidirectional handshake protocol
ensures that data traversing this bridge will not be corrupted.
The bridge has a master and a slave interface. The master interface
consists of: master_wait_request(in), master_read(out), master_write(out),
master_writedata(out), master_byteenable(out), master_address(out) 
and master_readdata(in).
The slave interface
consists of: slave_waitrequest, slave_read(in), slave_write(in),
slave_writedata(in), slave_byteenable(in), slave_address(in) 
and slave_readdata(out).
Outgoing data toward the slave peripheral is address, writedata,
and byteenable and read/write control.
Incoming data from the slave is readdata and wait request.
All timing paths between the two clock domains
are cut using the Altera Quartus CUT synthesis attribute.

There are four handshake handshake tokens.  Read request and write request tokens
are passed from the slave state machine over the clock boundary to
the master state machine.  Read done and write done tokens are passed
back from the master to the slave state machine.  Tokens are encoded
on signal edges, not levels.  This is important to note since we need
to avoid using pulses as tokens, as these can disappear in synchronizers
when metastable events occur and the pulse is shorter in time than two 
periods of the receiving clock domain.  Using edges as tokens guarantees 
that this circuit works with any ratio of slave clock to master clock
frequencies.

=head2 Examples

Put an example here!

=cut

use europa_all;
use strict;

my $proj = e_project->new(@ARGV);

# grab system PTF parameter dictating false path handling
my $system_name = $proj->{_system_name};


my $dont_cut_false_timing_paths =
  $proj->{ptf_hash}{"SYSTEM $system_name"}{WIZARD_SCRIPT_ARGUMENTS}{dont_cut_false_timing_paths};

# define some bit types
my $bit1 = 1;
my $bit3 = 3;

# define the FSM state encoding for both slave and master
# one hot i.e. 3'b100, 3'b010, 3'b001
my @FSM_states = &one_hot_encoding ($bit3);
my ($IDLE, $READ_WAIT, $WRITE_WAIT) = @FSM_states;


&make_proj($proj);
$proj->output();



sub make_proj
{
   my $project = shift;

   my $top = $project->top();  # $top is top level module object
   $top->comment("Clock Domain Crossing Adapter" . $top->name() . "\n\n");

   my $thing = e_default_module_marker->new($top);

   my @special_avalon_signals =
       (
        [clk => 1],
        [waitrequest => 1],
        [reset_n => 1],
        );

   my $read_fifo_depth = 0;

   push (@special_avalon_signals, [readdatavalid => 1])
       if ($read_fifo_depth);

   my @write_and_control_signals = &get_write_and_control_signals($project);
   my @readdata_signals  = &get_readdata_signals($project);
   my @signals = (@write_and_control_signals,
                  @readdata_signals,
                  @special_avalon_signals,
		 [endofpacket => 1]);

   my $slave_type_map = { map {"slave_".$_->[0] => $_->[0] }
                          @signals};

   my $master_type_map = { map {"master_".$_->[0] => $_->[0] }
                           (@signals,[chipselect => 1])};

   e_avalon_slave->add({
			name => 'in',
			type_map => $slave_type_map,
			SBI_section => {Maximum_Pending_Read_Transactions => $read_fifo_depth},
		       });

   e_avalon_master->add({
			 name => 'out',
			 type_map => $master_type_map,
			});

   my $slave_SBI  = $project->module_ptf()->{"SLAVE in"}{SYSTEM_BUILDER_INFO};

   e_signal->adds(map {["master_".$_->[0],
                        $_->[1],0,0,0]} @signals);
   e_signal->adds(map {["slave_" .$_->[0],
                        $_->[1],0,0,0]} @signals);


   # explicit port declarations
    e_port->adds([master_read => 1, "output"],
                 [master_write => 1, "output"],
                 [slave_write => 1, "input"],
                 [slave_read => 1, "input"],
                 [slave_waitrequest => 1, "output"]);

   # create local sub module references
   &define_bit_pipe($project);

   &define_edge_to_pulse($project);
   &define_slave_FSM($project);
   &define_master_FSM($project);

   &add_slave_logic($project);
   &add_master_logic($project);
   &add_eop_logic($project);

   &clock_cross_readdata_signals($project,@readdata_signals);
   &clock_cross_write_and_control_signals($project,@write_and_control_signals);
}


sub add_eop_logic {
  my $project = shift;

  my $top_name = $project->top->name();

  my $name = "endofpacket";
  my $width = 1;

  if ($dont_cut_false_timing_paths eq "1") {
    e_assign->add([["slave_$name" => $width],
		   ["master_$name" => $width]]);
  } else {
    e_instance->add({
		   name => "$name\_bit_pipe",
			  module => $project->top()->name()."_bit_pipe",
			  port_map => {data_out => "slave_$name",
				       data_in => "master_$name",
				       clk1 => "slave_clk",
				       reset_clk1_n => "slave_reset_n",
				       clk2 => "master_clk",
				       reset_clk2_n => "master_reset_n"
				      }
		  });
  }
}

sub add_slave_logic {
  my $project = shift;

  my $top_name = $project->top->name();

  e_synchronizer->add({
			  name => "$top_name\_master_read_done_sync",
			  port_map => {data_in  => "master_read_done",
				       data_out => "master_read_done_sync",
				       clk      => "slave_clk",
				       reset_n  => "slave_reset_n",
				      }
			});
  e_synchronizer->add({
			  name => "$top_name\_master_write_done_sync",
			  port_map => {data_in  => "master_write_done",
				       data_out => "master_write_done_sync",
				       clk      => "slave_clk",
				       reset_n  => "slave_reset_n",
				      }
			});

  e_instance->add({
		   name => "read_done_edge_to_pulse",
		   module => $project->top()->name()."_edge_to_pulse",
		   port_map => {
				"clock"     => "slave_clk",
				"reset_n"   => "slave_reset_n",
				"data_in"   => "master_read_done_sync",
				"data_out"  => "master_read_done_token",
			       }
		  });
  e_instance->add({
		   name => "write_done_edge_to_pulse",
		   module => $project->top()->name()."_edge_to_pulse",
		   port_map => {
				"clock"     => "slave_clk",
				"reset_n"   => "slave_reset_n",
				"data_in"   => "master_write_done_sync",
				"data_out"  => "master_write_done_token",
			       }
		  });
  e_instance->add({
		   name => "slave_FSM",
		   module => $project->top()->name()."_slave_FSM",
		   port_map => {
				slave_read => "slave_read",
				slave_write => "slave_write",
				master_read_done_token => "master_read_done_token",
				master_write_done_token => "master_write_done_token",
				slave_waitrequest => "slave_waitrequest",
				slave_read_request => "slave_read_request",
				slave_write_request => "slave_write_request"
			       }
		  });
}

sub add_master_logic {
  my $project = shift;

  my $top_name = $project->top->name();

  e_synchronizer->add({
			  name => "$top_name\_slave_read_request_sync",
			  port_map => {data_in  => "slave_read_request",
				       data_out => "slave_read_request_sync",
				       clk      => "master_clk",
				       reset_n  => "master_reset_n",
				      }
			});
  e_synchronizer->add({
			  name => "$top_name\_slave_write_request_sync",
			  port_map => {data_in  => "slave_write_request",
				       data_out => "slave_write_request_sync",
				       clk      => "master_clk",
				       reset_n  => "master_reset_n",
				      }
			});
  e_instance->add({
		   name => "read_request_edge_to_pulse",
		   module => $project->top()->name()."_edge_to_pulse",
		   port_map => {
				"clock" => "master_clk",
				"reset_n" => "master_reset_n",
				"data_in" => "slave_read_request_sync",
				"data_out" => "slave_read_request_token"
			       }
		  });
  e_instance->add({
		   name => "write_request_edge_to_pulse",
		   module => $project->top()->name()."_edge_to_pulse",
		   port_map => {
				"clock" => "master_clk",
				"reset_n" => "master_reset_n",
				"data_in" => "slave_write_request_sync",
				"data_out" => "slave_write_request_token"
			       }
		  });
  e_instance->add({
		   name => "master_FSM",
		   module => $project->top()->name()."_master_FSM",
		   port_map => {
				master_waitrequest => "master_waitrequest",
				slave_read_request_token => "slave_read_request_token",
				slave_write_request_token => "slave_write_request_token",
				master_read => "master_read",
				master_write => "master_write",
				master_read_done => "master_read_done",
				master_write_done => "master_write_done"
			       }
		  });
}



sub clock_cross_readdata_signals {
   my ($project, @readdata_signals) = @_;

   my @master_concatenation = ();
   my @slave_concatenation = ();
   my $width = 0;

   foreach my $signal (@readdata_signals) {
     my ($name, $width) = @$signal;

     if ($name eq 'readdata') {
       e_register->add({out    => ["slave_$name\_p1" => $width],
			in     => ["master_$name" => $width],
			enable => "master_read & ~master_waitrequest",
			clock  => 'master_clk',
			reset => 'master_reset_n',
		       });
       if ($dont_cut_false_timing_paths eq "1") {
	 # don't instantiate redundant 2nd flop and deal with the consequences
	 # of constrained false timing paths
	 e_assign->add([["slave_$name" => $width],
			["slave_$name\_p1" => $width]]);
       } else {
	 # add a second readdata flop stage in the slave_clk domain in 
	 # order to provide a destination for cutting the false timing path
	 e_register->add({
			  out => ["slave_$name" => $width],
			  in => ["slave_$name\_p1" => $width],
			  clock => 'slave_clk',
			  reset => 'slave_reset_n',
			  enable => "1",
			  cut_from_timing => "1",
			 });
       }
     } elsif ($name ne 'waitrequest') {
	 e_assign->add([["slave_$name" => $width],
			["master_$name" => $width]]);
     }
   }
 }

sub clock_cross_write_and_control_signals {

   my ($project,@write_and_control_signals) = @_;

   my @master_concatenation = ();
   my @slave_concatenation = ();
   my $width = 0;

   foreach my $signal (@write_and_control_signals) {
     my ($name, $width) = @$signal;

     if (($name ne 'read') && ($name ne 'write')) {
       if ($dont_cut_false_timing_paths eq "1") {
	 # don't instantiate redundant flops and deal with the consequences
	 # of constrained false timing paths
	 e_assign->add([["master_$name" => $width],
			["slave_$name" => $width]]);
       } else {
	 # instantiate bus pipeline stages for cutting timing paths
         e_register->add({
                          out => "slave_$name\_d1",
                          in => "slave_$name",
                          clock => "slave_clk",
	                  reset => "slave_reset_n",
                          enable => "1",
                          cut_to_timing => 1,
                          preserve_register => 1,
			 }),
         e_register->add({
                          out => "master_$name",
                          in => "slave_$name\_d1",
                          clock => "master_clk",
                          reset => "master_reset_n",
                          enable => "1",
                          preserve_register => 1,
			 })
       }
     }
   }
 }


sub get_write_and_control_signals {
   my $project = shift;

   my $master_SBI = $project->module_ptf()->{"MASTER out"}{SYSTEM_BUILDER_INFO};
   my @write_and_control_signals = ();

   my $data_width       = $master_SBI->{Data_Width} ||  &ribbit ("bad data width");
   my $use_byteenables  = ($data_width > 8) || ($master_SBI->{Address_Alignment} eq 'dynamic');

   my $address_width    = $master_SBI->{Address_Width};

   # Native address width is narrower than byte address width.
   my $nativeaddress_width =
     $master_SBI->{Address_Width} - log2($master_SBI->{Data_Width} / 8);

   push (@write_and_control_signals,
         [writedata => $data_width],
         [address   => $address_width],
         [read      => 1],
         [write     => 1],
         [nativeaddress => $nativeaddress_width],
         );

   if ($use_byteenables)
   {
      my $byteenable_width = $data_width / 8;
      push (@write_and_control_signals,
            [byteenable => $byteenable_width]);
   }

   my $max_size = 0; #$master_ptf->{Maximum_Burst_Size};
   my $burstcount_width = ($max_size)? &log2($max_size):0;
   if ($burstcount_width)
   {
      push (@write_and_control_signals,
            [burstcount => $burstcount_width]);
   }
   return @write_and_control_signals;
}

sub get_readdata_signals
{
   my $project = shift;

   my $Data_Width = $project->module_ptf()->{"SLAVE in"}
   {SYSTEM_BUILDER_INFO}{Data_Width} || &ribbit ("bad datawidth");
   return [readdata => $Data_Width];
}



# ------------------------------------------------------------------------------
# Define the slave side FSM module
# ------------------------------------------------------------------------------
sub define_slave_FSM {
    my $project = shift;
    my $module = e_module->new({name => $project->top()->name()."_slave_FSM"});

    # register the module in the project DB for later instancing
    $project->add_module($module);

    $module->add_contents (
	   # explicitly declare the ports
	   e_port->news(
		[slave_read => $bit1, "input"],
                [slave_write => $bit1, "input"],
                [master_read_done_token => $bit1, "input"],
                [master_write_done_token => $bit1, "input"],
                [slave_waitrequest => $bit1, "output"],
		[slave_read_request => $bit1, "output"],
		[slave_write_request => $bit1, "output"]
		       ),

	   # declare handshake signals which go to the master
	   e_signal->news
		   ([slave_read_request => $bit1],
		    [slave_write_request => $bit1]
		   ),

	   # latch the handshake signals
	   e_register->news
		   ({out => "slave_read_request",
		     in => "next_slave_read_request",
		     clock => "slave_clk",
		     reset => "slave_reset_n",
		     enable => undef,
		     async_value => 0}
		   ),
	   e_register->news
		   ({out => "slave_write_request",
		     in => "next_slave_write_request",
		     clock => "slave_clk",
		     reset => "slave_reset_n",
		     enable => undef,
		     async_value => 0}
		   ),

	   # declare FSM state and next state vectors
	   e_signal->news
		   ([slave_state      => $bit3],
		    [next_slave_state => $bit3],
		   ),

	   # latch the state vector; goto IDLE state upon reset
	   e_register->news
		   ({out => "slave_state",
		     in => "next_slave_state",
		     clock => "slave_clk",
		     reset => "slave_reset_n",
		     enable => undef,
		     async_value => $IDLE}
		   ),

	   e_process->new({
		   clock   => undef,  # this is a purely combinatorial block
		   contents => [

		       # default - don't update these handshake control signals unless
                       # you're in a specific state

                       # *** defaults don't work properly with current e_process.pm
                       # *** because non-blocking assigns are used in ALL always blocks
                       # *** regardless of whether they are combinatorial or clocked!
                       # *** Temporarily work around this issue by not using default assigns.				
# 		       e_assign->new({out => "next_slave_read_request",
# 		                      in  => "slave_read_request"
# 				     }),
# 		       e_assign->new({out => "next_slave_write_request",
# 		                      in  => "slave_write_request"
# 				     }),

	               # combinatorial case switch defines state transitions
 	               # and output control signals
		       e_case->new ({
		           switch   => "slave_state",
			   parallel => 1,
			   default_sim => 0,
			   contents => {

	                   $IDLE => [
				    e_if->new({
				        comment  => "read request: go from IDLE state to READ_WAIT state",
					condition=> "slave_read",
					then     => [
						     "next_slave_state" => $READ_WAIT,
						     "slave_waitrequest" => 1,
						     "next_slave_read_request" => "!slave_read_request",
						     "next_slave_write_request" => "slave_write_request"
						    ],
					elsif => {
						  condition => "slave_write",
						  then => [
							   "next_slave_state" => $WRITE_WAIT,
							   "slave_waitrequest" => 1,
							   "next_slave_read_request" => "slave_read_request",
							   "next_slave_write_request" => "!slave_write_request"
							   ],
						  # no requests, so stay put
						  else => [
							   "next_slave_state" => "slave_state",
							   "slave_waitrequest" => 0,
							   "next_slave_read_request" => "slave_read_request",
							   "next_slave_write_request" => "slave_write_request"
							  ]
						  },
					      })
				    ],
                           $READ_WAIT => [
                                    e_if->new({
				        comment  => "stay in READ_WAIT state until master passes read done token",
	 			        condition=> "master_read_done_token",
					then     => [
						     "next_slave_state" => $IDLE,
						     "slave_waitrequest" => 0,
						    ],
					else     => [
						     "next_slave_state" => $READ_WAIT,
						     "slave_waitrequest" => 1,
						    ]
					      }),
				     # defaults
				     e_assign->new({out => "next_slave_read_request",
						    in  => "slave_read_request"
						   }),
				     e_assign->new({out => "next_slave_write_request",
						    in  => "slave_write_request"
						   })
				    ],
                           $WRITE_WAIT => [
                                    e_if->new({
				        comment  => "stay in WRITE_WAIT state until master passes write done token",
	 			        condition=> "master_write_done_token",
					then     => [
						     "next_slave_state" => $IDLE,
						     "slave_waitrequest" => 0,
						    ],
					else     => [
						     "next_slave_state" => $WRITE_WAIT,
						     "slave_waitrequest" => 1,
						    ]
					      }),
				     # defaults
				     e_assign->new({out => "next_slave_read_request",
						    in  => "slave_read_request"
						   }),
				     e_assign->new({out => "next_slave_write_request",
						    in  => "slave_write_request"
						   })

				    ],
			   default=> [
				      "next_slave_state" => $IDLE,
				      "slave_waitrequest" => 0,
				      "next_slave_read_request" => "slave_read_request",
				      "next_slave_write_request" => "slave_write_request"
				     ]
			   } # case contents
		    }), # e_case
		], # add contents
	  }) # e_process
    ) # module
} # sub define_slave_FSM

# ------------------------------------------------------------------------------
# Define the master side FSM module
# ------------------------------------------------------------------------------
sub define_master_FSM {
    my $project = shift;
    my $module = e_module->new({name => $project->top()->name()."_master_FSM"});

    # register the module in the project DB for later instancing
    $project->add_module($module);

    $module->add_contents (
	   # explicitly declare the ports
	   e_port->news(
		[master_waitrequest => $bit1, "input"],
                [slave_read_request_token => $bit1, "input"],
                [slave_write_request_token => $bit1, "input"],
		[master_read => $bit1, "output"],
                [master_write => $bit1, "output"],
                [master_read_done => $bit1, "output"],
                [master_write_done => $bit1, "output"]
		       ),

	   # declare handshake signals which go back to the slave
	   e_signal->news
		   ([master_read_done => $bit1],
		    [master_read_done => $bit1]
		   ),

	   # latch the handshake signals
	   e_register->news
		   ({out => "master_read_done",
		     in => "next_master_read_done",
		     clock => "master_clk",
		     reset => "master_reset_n",
		     enable => undef,
		     async_value => 0}
		   ),
	   e_register->news 
		   ({out => "master_write_done",
		     in => "next_master_write_done",
		     clock => "master_clk",
		     reset => "master_reset_n",
		     enable => undef,
		     async_value => 0}
		   ),

	   # declare control signals
	   e_signal->news
		   ([master_read => $bit1],
		    [master_read => $bit1]
		   ),

	   # latch the handshake signals
	   e_register->news 
		   ({out => "master_read",
		     in => "next_master_read",
		     clock => "master_clk",
		     reset => "master_reset_n",
		     enable => undef,
		     async_value => 0}
		   ),
	   e_register->news 
		   ({out => "master_write",
		     in => "next_master_write",
		     clock => "master_clk",
		     reset => "master_reset_n",
		     enable => undef,
		     async_value => 0}
		   ),

	   # declare FSM state and next state vectors
	   e_signal->news
		   ([master_state      => $bit3],
		    [next_master_state => $bit3],
		   ),

	   # latch the state vector; goto IDLE state upon reset
	   e_register->news
		   ({out => "master_state",
		     in => "next_master_state",
		     clock => "master_clk",
		     reset => "master_reset_n",
		     enable => undef,
		     async_value => $IDLE}
		   ),

	   e_process->new({
		   clock   => undef,  # this is a purely combinatorial block
		   contents=> [

		       # default - don't update these handshake control signals unless
                       # you're in a specific state

                       # *** defaults don't work properly with current e_process.pm
                       # *** because non-blocking assigns are used in ALL always blocks
                       # *** regardless of whether they are combinatorial or clocked!
                       # *** Temporarily work around this issue by not using default assigns.				
# 		       e_assign->new({out => "next_master_read",
# 		                      in  => "master_read"
# 				     }),
# 		       e_assign->new({out => "next_master_write",
# 		                      in  => "master_write"
# 				     }),
# 		       e_assign->new({out => "next_master_read_done",
# 		                      in  => "master_read_done"
# 				     }),
# 		       e_assign->new({out => "next_master_write_done",
# 		                      in  => "master_write_done"
# 				     }),

	               # combinatorial case switch defines state transitions
 	               # and output control signals
		       e_case->new ({
		           switch   => "master_state",
			   parallel => 1,
			   default_sim => 0,
			   contents => {

	                   $IDLE => [
				    e_if->new({
				        comment  => "if read request token from slave then goto READ_WAIT state",
					condition=> "slave_read_request_token",
					then     => [
						     "next_master_state" => $READ_WAIT,
						     "next_master_read" => 1,
						     "next_master_write" => 0
						    ],
					elsif => {
						  condition => "slave_write_request_token",
						  then => [
							   "next_master_state" => $WRITE_WAIT,
							   "next_master_read" => 0,
							   "next_master_write" => 1
							   ],
						  else => [
							   "next_master_state" => "master_state",
							   "next_master_read" => 0,
							   "next_master_write" => 0
							  ]
						  },
					      }),
				     # don't ever touch done flags in the IDLE state
				     e_assign->new({out => "next_master_read_done",
						    in  => "master_read_done"
						   }),
				     e_assign->new({out => "next_master_write_done",
						    in  => "master_write_done"
						   })
				    ],
                           $READ_WAIT => [
                                    e_if->new({
				        comment  => "stay in READ_WAIT state until master wait is deasserted",
	 			        condition=> "!master_waitrequest",
					then     => [
						     # done - move from READ_WAIT state back to IDLE
						     "next_master_state" => $IDLE,
						     "next_master_read_done" => "!master_read_done",
						     "next_master_read" => 0
						    ],
					else     => [
						     # stay in WRITE_WAIT state until we're done
						     "next_master_state" => $READ_WAIT,
						     "next_master_read_done" => "master_read_done",
						     "next_master_read" => "master_read"
						    ]
					      }),
				     # don't ever touch write signals in the READ_WAIT state
				     e_assign->new({out => "next_master_write_done",
						    in  => "master_write_done"
						   }),
				     e_assign->new({out => "next_master_write",
						    in  => 0
						   })
				    ],
                           $WRITE_WAIT => [
                                    e_if->new({
				        comment  => "stay in WRITE_WAIT state until slave wait is deasserted",
	 			        condition=> "!master_waitrequest",
					then     => [
						     # done - move from write wait state back to idle
						     "next_master_state" => $IDLE,
						     "next_master_write" => 0,
						     "next_master_write_done" => "!master_write_done"
						    ],
					else     => [
						     # stay in write wait state
						     "next_master_state" => $WRITE_WAIT,
						     "next_master_write" => "master_write",
						     "next_master_write_done" => "master_write_done"
						    ]
					      }),
				     # don't ever touch read signals in the READ_WAIT state
				     e_assign->new({out => "next_master_read_done",
						    in  => "master_read_done"
						   }),
				     e_assign->new({out => "next_master_read",
						    in  => 0
						   })
				    ],
			   default=> [
				      # should never get here!
				      "next_master_state" => $IDLE,
				      "next_master_write" => 0,
				      "next_master_write_done" => "master_write_done",
				      "next_master_read" => 0,
				      "next_master_read_done" => "master_read_done"
				     ]
			   } # case contents
		    }), # e_case
		], # add contents
	  }), # e_process
    ) # module
} # sub define_master_FSM

# ------------------------------------------------------------------------------
# Create handshake edge_to_pulse module which generates a one cycle pulse on
# positive or negative input data transitions.
# ------------------------------------------------------------------------------
sub define_edge_to_pulse {
   my $project = shift;

   my $module = e_module->new({name => $project->top()->name()."_edge_to_pulse"});

    # register the module in the project DB for later instancing
    $project->add_module($module);

    $module->add_contents (
	e_register->new({
			 out => "data_in_d1",
			 in => "data_in",
			 clock => "clock",
			 reset => "reset_n",
			 enable => "1"
			}),
        e_assign->new({
		       out => "data_out",
		       in  => "data_in ^ data_in_d1"
		      })
    );
}


# ------------------------------------------------------------------------------
# Create an 1-bit wide 2-stage pipeline whose sole purpose is to define a
# source register in clock domain 1 and a destination register in clock domain 2.
# The timing path between the source and destination registers is cut.
# ------------------------------------------------------------------------------
sub define_bit_pipe {
    my $project = shift;
    my $module = e_module->new({name => $project->top()->name()."_bit_pipe"});

    # register the module in the project DB for later instancing
    $project->add_module($module);

    $module->add_contents (
         e_register->new({
                          out => "data_in_d1",
                          in => "data_in",
                          clock => "clk1",
	                  reset => "reset_clk1_n",
                          enable => "1",
                          cut_to_timing => 1,
                          preserve_register => 1,
			 }),
         e_register->new({
                          out => "data_out",
                          in => "data_in_d1",
                          clock => "clk2",
                          reset => "reset_clk2_n",
                          enable => "1",
                          preserve_register => 1,
			 })
    );
}

=head1 BUGS AND LIMITATIONS

We need a better way to cut timing paths on signal nets. Currently
only flop and chip IOs can be used as -to and -from timing path
endpoints. Module ports cannot be used since they may disappear
during the synthesis flattening process.  This forces us to
gratuitously instantiate flops soley to satisfy the static timing
analyzer.

=head1 SEE ALSO

e_synchronizer.pm

=head1 AUTHOR

Orion and Paul

=head2 History

Added false path timing assertions. (PS 10-5-04).
Swapped e_clock_crossing.pm with e_synchronizer.pm module (PS 01-12-05)

=head1 COPYRIGHT

Copyright (c) 2004-2005, Altera Corporation. All Rights Reserved.

=cut
