use europa_all;
use europa_utils;
use strict;

sub make_burst_adapter
{
  # Simple syntax check is invoked with no parameters.
  return if !@_;

  my $project = e_project->new(@_);
 
  make_adapter($project);

  $project->output();
}

sub validate_options($)
{
  my $opt = shift;
 
  # These required parameters should have been set by the
  # burst adapter insertion code.
  validate_parameter({
    hash    => $opt,
    name    => "master_data_width",
    type    => "int",
    allowed_values => [8, 16, 32, 64, 128, ],
  });

  validate_parameter({
    hash    => $opt,
    name    => "master_interleave",
    type    => "bool",
  });

  validate_parameter({
    hash    => $opt,
    name    => "slave_interleave",
    type    => "bool",
  });

  validate_parameter({
    hash    => $opt,
    name    => "dynamic_slave",
    type    => "bool",
  });
}

sub assign_options($$$)
{
  my ($project, $module, $opt) = @_;

  # Signals added to this list will show up in wave_presets.do.
  $opt->{extra_sim_signals} = [];

  my $adapter_slave_SBI = $project->SBI('upstream');
  $opt->{data_width} = $adapter_slave_SBI->{Data_Width};

  # Address width is the upstream master's byte-address width.  I work with
  # byte addresses internally.
  $opt->{byteaddr_width} = $adapter_slave_SBI->{Address_Width} + log2($opt->{data_width} / 8);
  $opt->{nativeaddr_width} = $adapter_slave_SBI->{Address_Width};

  # Try to make the world safe for native slaves.
  $opt->{ceil_data_width} =
    round_up_to_next_computer_acceptable_bit_width($opt->{data_width});
    
  $opt->{downstream_addr_shift} = log2($opt->{ceil_data_width} / 8);
  $opt->{dbs_shift} = log2($opt->{master_data_width} / $opt->{ceil_data_width});
  
  # Grab the Maximum Burst Count from up & down stream sides, figure out the
  # bit-width of this size, and then calculate the difference. This will 
  # control how many transfers encompass a "downstream burst", and loading
  # the low-bits of the upstream burstcount.  
  my $upstream_SBI = $project->module_ptf()->{"SLAVE upstream"}{SYSTEM_BUILDER_INFO};
  $opt->{upstream_max_burstcount} = $upstream_SBI->{Maximum_Burst_Size};
  ribbit("Maximum_Burst_Size must be integer power of 2; is: $opt->{upstream_max_burstcount}\n")
    unless is_power_of_two($opt->{upstream_max_burstcount});

  my $downstream_SBI = $project->module_ptf()->{"MASTER downstream"}{SYSTEM_BUILDER_INFO};
  $opt->{downstream_max_burstcount} = $downstream_SBI->{Maximum_Burst_Size} || 1;
  ribbit("Maximum_Burst_Size must be integer power of 2; is: $opt->{downstream_max_burstcount}\n")
    unless is_power_of_two($opt->{downstream_max_burstcount});

  $opt->{downstream_burstcount_width} = 1 + &log2($opt->{downstream_max_burstcount});
  
  # Burst adapters are slave-data-width.  The upstream master may or may not be slave
  # data width.
  $opt->{upstream_burstcount_width} = 1 + &log2($opt->{upstream_max_burstcount});
  
  # dbs-adjusted burstcount width may become larger due to dbs, but it should
  # never become smaller (since writes need to be done 1 narrow master word
  # at a time in the case of negative dbs).
  $opt->{dbs_upstream_burstcount_width} = $opt->{upstream_burstcount_width};
  $opt->{dbs_upstream_burstcount_width} += $opt->{dbs_shift} if ($opt->{dbs_shift} > 0);
}

sub make_interfaces($$$)
{
  my ($project, $module, $opt) = @_;
  # Avalon signals common to upstream (slave) and downstream (master).
  # Generally these signals pass from slave to master or vv., sometimes
  # with some processing.
  my @avalon_slave_signals;
  my @avalon_master_signals;
  
  my @avalon_signals = 
  (
    {name => 'writedata', width => $opt->{data_width}},
    {name => 'readdata',  width => $opt->{data_width}},
    {name => 'readdatavalid'},
    {name => 'write'},
    {name => 'read'},
    {name => 'waitrequest'},
    {name => 'byteenable', width => $opt->{ceil_data_width} / 8},
  );

  # All those signals have types identical to 'name'.
  map {$_->{type} = $_->{name}} @avalon_signals;

  for my $sig (@avalon_signals)
  {
    my %sig_copy = %$sig;

    $sig->{name} = 'upstream_' . $sig->{name};
    push @avalon_slave_signals, $sig;
    $sig_copy{name} = 'downstream_' . $sig_copy{name};
    push @avalon_master_signals, \%sig_copy;
  }

  # Avalon signals for upstream slave only
  push @avalon_slave_signals,  (
    {name => 'upstream_address', width => $opt->{byteaddr_width}, type => 'byteaddress',},
    
    {name => 'upstream_burstcount', width => $opt->{upstream_burstcount_width}, type => 'burstcount',},
  );

  push @avalon_slave_signals,  (
    # I don't actually need a native address on the slave side, but having one makes
    # SOPC Builder happy.
    {name => 'upstream_nativeaddress', width => $opt->{nativeaddr_width}, type => 'address',},
  );
  e_assign->adds({
    lhs => {name => "sync_nativeaddress", never_export => 1,},
    rhs => "|upstream_nativeaddress",
  });

  # Avalon signals for downstream master only
  push @avalon_master_signals, (
    {name => 'downstream_address', width => $opt->{nativeaddr_width}, type => 'address',},
    {name => 'downstream_burstcount', width => $opt->{downstream_burstcount_width}, type => 'burstcount',},
    {name => 'downstream_arbitrationshare', width => $opt->{dbs_upstream_burstcount_width}, type => 'arbitrationshare',},
  );

  # Set some options in all generic master/slave signals.
  my @export_signals = qw(
    upstream_waitrequest
    upstream_readdatavalid
    downstream_write
    downstream_read
    downstream_burstcount
  );

  for my $sig (@avalon_slave_signals, @avalon_master_signals)
  {
    $sig->{copied} = 1;
    $sig->{never_export} = 0;
    
    $sig->{export} =  0;
    if (grep {$sig->{name} eq $_} @export_signals)
    {
      $sig->{export} =  1;
    }
  }

  e_signal->adds(@avalon_slave_signals, @avalon_master_signals);

  # Sprout master & slave ports
  my $slave_type_map = { map {$_->{name} => $_->{type}}
                         @avalon_slave_signals};
  my $master_type_map = { map {$_->{name} => $_->{type}}
                         (@avalon_master_signals)};

  e_avalon_master->add
  ({
    name => 'downstream',
    type_map => $master_type_map,
  });

  my $upstream_slave = e_avalon_slave->add
  ({
    name => 'upstream',
    type_map => $slave_type_map,
  });
}

sub make_counters($$$)
{
  my ($project, $module, $opt) = @_;

  # Atomic counter:
  # Counts in units of downstream data width, from
  #  0.  Counting is done in increments of 1 on writes;
  # in increments of downstream_burstcount on reads.
  # This counter is used to generate the signals
  #  
  #  upstream_burstdone
  #  downstream_burstdone
  #
  if (!$opt->{dynamic_slave})
  {
    # A note on native slaves:
    # A very common case is a bursting master connected to a non-bursting native
    # slave, where the master has no plans to burst to the slave (e.g. nios2 bursting
    # data master connected to UART, timer, et al).  Rather than enforce a cycle of
    # latency for this connection, as is done in the dynamic slave case, it seems kind
    # to provide a stripped-down adapter.  The current implementation always does bursts
    # of size 1 to native slaves (bursting or non-bursting). The result is an adapter
    # containing a single counter (width: $opt->{upstream_burstcount_width} - 1, since it
    # counts from <burstcount - 1> down to 0), and some wires.
    
    # Why not do larger bursts to bursting native slaves?  First of all, I'm not convinced
    # any such animal will ever exist - what advantage would it provide?  The only real
    # reason to burst to a native slave that I can come up with is to lock up arbitration
    # during the transfer. This benefit is provided by the simple burst-1  adapter (see
    # downstream_arbitrationshare), and because the transactions_remaining counter can be
    # a decrementer rather than a subtractor, it should be cheaper/faster.
    e_register->add({
      out => {name => 'transactions_remaining', width => $opt->{upstream_burstcount_width} - 1,},
      in => 
        # Start of a read: init the counter...
        "(upstream_read & ~upstream_waitrequest) ? (upstream_burstcount - 1) : " .
        # ... as each downstream read burst is accepted, decrement the
        # counter by downstream_burstcount (which happens to be a constant 1).
        "(downstream_read & ~downstream_waitrequest & (|transactions_remaining)) ? (transactions_remaining - downstream_burstcount) : " .
        "transactions_remaining",
      enable => 1,
    });
    push @{$opt->{extra_sim_signals}}, qw(
      transactions_remaining
    );
    return;
  }
  
  e_register->add({
    out => ['atomic_counter', $opt->{downstream_burstcount_width}],
    in => 'downstream_burstdone ? 0 : p1_atomic_counter',
    enable => '(downstream_read | downstream_write) & ~downstream_waitrequest',
    async_value => 0,
  });
  push @{$opt->{extra_sim_signals}}, qw(
    atomic_counter
  );

  # Transactions-remaining counter.  This counter is initialized to
  # upstream_burstcount (adjusted for dbs, if necessary) at the start
  # of a burst.  It decrements in steps of size downstream_burstcount
  # as each burst is accepted by the downstream slave.

  # Handy signals that go active upon completion of a read or write
  # subburst.
  e_assign->adds(
    {
      lhs => "read_update_count",
      
      # s/current_upstream_read/downstream_read/ ?
      rhs => "current_upstream_read & ~downstream_waitrequest",
    },
    {
      lhs => "write_update_count",
      rhs => "current_upstream_write & downstream_write & downstream_burstdone",
    },
    {
      lhs => "update_count",
      rhs => "read_update_count | write_update_count",
    },
  );

  # register with flow-through mux.  I think it should be possible to
  # just use the registered signal, but watch out for spurious 
  e_assign->add
  ({
    lhs => ['transactions_remaining', $opt->{dbs_upstream_burstcount_width}],
    rhs => "(state_idle & (upstream_read | upstream_write)) ? " .
      "dbs_adjusted_upstream_burstcount : transactions_remaining_reg",
  });
  
  e_register->add({
    out => ['transactions_remaining_reg', $opt->{dbs_upstream_burstcount_width}],
    in =>
      "(state_idle & (upstream_read | upstream_write)) ? dbs_adjusted_upstream_burstcount : " .
      "update_count ? transactions_remaining_reg - downstream_burstcount : " .
      "transactions_remaining_reg",
    enable => '1',
  });
  push @{$opt->{extra_sim_signals}}, qw(
    transactions_remaining
    transactions_remaining_reg
  );

  # data_counter counts down from dbs_adjusted_upstream_burstcount.
  # Decrement whenever the downstream slave delivers read
  # data.  This counter prevents transition to state_idle when more 
  # read data is expected.
  e_register->add({
    out => {name => 'data_counter', width => $opt->{dbs_upstream_burstcount_width}},
    in =>
      "state_idle & upstream_read & ~upstream_waitrequest ?  dbs_adjusted_upstream_burstcount : " .
      "downstream_readdatavalid ? data_counter - 1 : " .
      "data_counter",
    enable => 1,
  });

  push @{$opt->{extra_sim_signals}}, qw(
    data_counter
  );
}

sub make_downstream_burstcount($$$)
{
  my ($project, $module, $opt) = @_;
  if (!$opt->{dynamic_slave})
  {
    e_assign->add({
      lhs => e_signal->new({
        name => 'downstream_burstcount',
        width => $opt->{downstream_burstcount_width},
        never_export => 1,
      }),
      rhs => 1,
    });
    
    return;
  }
  
  # Interleaved slaves make burstcount complex, so here's a lengthy
  # screed on the subject of interleaved bursts:

  # Interleaved slaves are just like sequential slaves, except that they return
  # data within a burst in a different order, which depends on the burst base
  # address. This table illustrates the difference, for various values of
  # burst-base, for a slave with max-burst = 8.  The table elements show the
  # address offset of the word in the interleaved slave's hypothetical internal
  # sequentially-addressed memory space.
  #
  # sequence #     burst base address
  #                0  1  2  3  4  5  6  7
  #               +-----------------------+
  #      0       | 0  1  2  3  4  5  6  7 |
  #      1       | 1  0  3  2  5  4  7  6 |
  #      2       | 2  3  0  1  6  7  4  5 |
  #      3       | 3  2  1  0  7  6  5  4 |
  #      4       | 4  5  6  7  0  1  2  3 |
  #      5       | 5  4  7  6  1  0  3  2 |
  #      6       | 6  7  4  5  2  3  0  1 |
  #      7       | 7  6  5  4  3  2  1  0 |
  #              +------------------------+
  #      N         8  7  6  5  4  3  2  1
  #  
  #  The pattern is generated by XOR'ing the burst base address with
  # the sequence number.
  #
  # The burst adapter allows connection of a sequential master to either
  # sequential or interleaved slaves.  Sequential slaves are easy, especially
  # considering that it's assumed that slaves, when asked to burst across a 
  # slave burst boundary, do not wrap.  Subbursts are done simply by making
  # as many max-slave-size bursts as possible, and finishing off with a smaller
  # burst when necessary.
  #
  # With an interleaved slave, more care is required.  Three distinct phases
  # occur:
  # 1) do the maximum bursts possible that are compatible with sequential addressing,
  #   until a slave burst boundary is reached; 
  # 2) do max bursts as necessary (each based at a burst boundary);
  # 3) finally finish up with a smaller-than-max burst (based at a burst boundary)
  #
  # Any combination of the three phases involving at least one phase is possible.
  # 
  # Interesting implementation note: the difference between sequential and
  # interleaved bursting is that in for sequential slaves, phase 1) is omitted.
  #
  # For phase 1, what determines the maximum subburst size?
  # let interleave_onset =
  #   min(burstcount, <slave max burst size> - <offset of burst start from burst boundary>)
  # then 0 <= N <= slave max burst - 1, and the maximum burst that's compatible
  # with sequential bursting is given by the least-significant set-bit of N.
  #
  # An example of phase 1 may be helpful:
  #
  # slave max burst = 8
  # 
  # master bursts 5 @ address 4
  # interleave_onset = 8 - 4 = 4
  # burst 4 @ address 4: 4, 5, 6, 7
  # burst 1 @ address 8: 8
  #
  # master bursts 13 @ address 1
  # interleave_onset = 8 - 1 = 7
  # burst 1 @ address 1: 1
  # burst 2 @ address 2: 2, 3
  # burst 4 @ address 4: 4, 5, 6, 7
  # burst 6 @ address 8: 8, 9, 10, 11, 12, 13
  #
  # master bursts 23 @ address 2
  # interleave_onset = 8 - 2 = 6
  # burst 2 @ address 2: 2, 3
  # burst 4 @ address 4: 4, 5, 6, 7
  # burst 8 @ address 8: 8, 9, 10, 11, 12, 13, 14, 15
  # burst 8 @ address 16: 16, 17, 18, 19, 20, 21, 22, 23
  # burst 1 @ address 24: 24
  #
  # master burst 3 @ address 1
  # interleave_onset= 8 - 1 = 7
  # burst 1 @ address 1: 1
  # burst 2 @ address 2: 2, 3
  # burst 4 @ address 4: 4, 5, 6, 7 (skipped)
  #
  # master burst 3 @ address 4
  # interleave_onset  8 - 4 =  4
  # burst 4 @ address 4: 4, 5, 6, 7 (last is skipped)
  #
  # Elegant possible implementation:
  # At the start of each subburst, the burstcount value for the subburst is
  # chosen. The subburst base address is known (the initial address on the first
  # subburst, or the previous burst + the previous burstcount).
  # If a value burst_agenda is initialized as (slave_max_burst | (-offset within burst))
  #
  #   burst_agenda <= (slave_max_burst | (-offset within burst))
  #
  #   (the offset within the burst of the start of the transaction, with the slave
  # max burst value OR'ed in).
  # 
  # and is updated at the last cycle of each subburst as
  #
  #   burst_agenda <= slave_burst_mask & ~burstcount;
  #   
  #  (clear the bit corresponding to the previous burstcount, unless it was
  # a max-slave-burst burstcount; in that case, leave the bit set)
  # 
  # and
  #
  #  max_burst_size = burst_agenda & (~burst_agenda + 1)
  # 
  #  (find the lsb-most set bit within burst_agenda - this represents the largest
  #  burst that can be done (unless less than that amount would complete the burst)
  #
  # then the current subburst burstcount is
  # 
  # (transactions_remaining > (max_burst_size)) ? max_burst_size : transactions_remaining;
  # 
  # burst_agenda basically allows phases 1 and 2 to be merged into one
  # unified phase.  For the sequential-slave case, simply set the (-(offset within
  # burst)) term to 0, which should allow burst_agenda to be optimized out to a
  # constant (with value slave_max_burst).
  #
  # For linear slaves which wrap on their burst boundary, modify max_burst_size to be
  # the amount required to reach the next slave burst boundary (don't forget to also
  # modify downstream_arbitrationshare for reads).
  # 
  if (($opt->{downstream_max_burstcount} > 1) && $opt->{slave_interleave})
  {
    # interleave_onset contains a value in [0, downstream_max_burstcount - 1].
    # Extract the portion of the master address corresponding to the offset within
    # a slave burst boundary.
    #
    # Masters get bursts aligned with their own data width.  Ignore master address
    # lsbs within a master word, because dynamic bus sizing logic in the master arb
    # can set these bits spuriously.
    #
    my $interleave_onset_width = log2($opt->{downstream_max_burstcount});
    my $slave_burst_master_address_span;
    
    # [$span_msb : $span_lsb] is the range within the master address which
    # defines the slave burst boundary.
    my $span_lsb = log2($opt->{ceil_data_width} / 8);
    my $span_msb = $span_lsb + $interleave_onset_width - 1;
    my @interleave_bits;
    for my $i ($span_lsb .. $span_msb)
    {
      # Select a master address bit, but zero out any bits within
      # the master's data width -- they can toggle spuriously due
      # to master arb dbs shenanigans.
      unshift @interleave_bits,
        ($i < log2($opt->{master_data_width} / 8)) ?
          "1'b0" : "upstream_address[$i]"
    }
    my $upstream_address_wrt_slave_burst_boundary = concatenate(@interleave_bits);

    # Interleave onset computed as (-offset within burst)
    # can contain more set bits than will actually be represented by downstream
    # bursts.  Example:
    # slave max burst = 64
    # master burst 3 @ address 1
    # interleave onset: 64 - 1 = 63 = 0x3f
    # burst 1 @ address 1
    # burst 2 @ address 2
    #
    # (the pattern appears to be: if the burst lies entirely within the low half
    # of a burst boundary, then sum-of-bits(-offset within burst) will overestimate
    # the number of downstream bursts.
    
    # Mask off unnecessary bits of (-offset within burst) by finding the top address
    # of the requested burst, setting all bits below the MS-set bit, and using that
    # as an AND mask.
    
    # Compute the last (linear) address within the requested burst...
    e_assign->add({
      lhs => {name => "mask_top_burst_address", width => $opt->{downstream_burstcount_width},},
      rhs => "$upstream_address_wrt_slave_burst_boundary + dbs_adjusted_upstream_burstcount - 1",
    });
    
    # ... bit-reverse that address...
    e_assign->add({
      lhs => {name => "mask_reversed_top_burst_address", width => $opt->{downstream_burstcount_width},},
      rhs => concatenate(
        map {"mask_top_burst_address[$_]"} (0 .. -1 + $opt->{downstream_burstcount_width})
      ),
    });
    
    # ... find the least-significant set-bit of the reversed expression...
    e_assign->add({
      lhs => {name => "mask_reversed_bottom_bit", width => $opt->{downstream_burstcount_width},},
      rhs => "mask_reversed_top_burst_address & (~mask_reversed_top_burst_address+ 1)",
    });
    
    # ... reverse back...
    e_assign->add({
      lhs => {name => "mask_unreversed_bottom_bit", width => $opt->{downstream_burstcount_width},},
      rhs => concatenate(
        map {"mask_reversed_bottom_bit[$_]"} (0 .. -1 + $opt->{downstream_burstcount_width})
      )
    });
    
    # ... fill in low bits to form the mask.
    e_assign->add({
      lhs => {name => "mask_interleave", width => $opt->{downstream_burstcount_width},},
      rhs => "mask_unreversed_bottom_bit | (mask_unreversed_bottom_bit - 1)",
    });

    push @{$opt->{extra_sim_signals}}, qw(
      mask_top_burst_address
      mask_reversed_top_burst_address
      mask_reversed_bottom_bit
      mask_unreversed_bottom_bit
      mask_interleave
    );

    e_assign->add({
      lhs => {name => "initial_interleave_onset", width => $interleave_onset_width,},
      rhs => "-($upstream_address_wrt_slave_burst_boundary) & " .
        "(mask_interleave)"
    });
    my $p1_interleave_onset = 
      "(state_idle & (upstream_read | upstream_write)) ? (initial_interleave_onset) :
      update_count ? (interleave_onset & ~downstream_burstcount) : " .
      "interleave_onset";
      
    # Add up all the bits of the interleave_onset value.
    my $p1_interleave_onset_bitcount =
      join(" + ", map {"initial_interleave_onset[$_]"} (0 .. -1 + $interleave_onset_width));
    e_register->add({
      out => {name => "interleave_onset", width => $interleave_onset_width,},
      in => $p1_interleave_onset,
      enable => 1,
    });
    e_register->add({
      out => {name => "interleave_onset_bitcount", width => ceil(1 + log2($interleave_onset_width)),},
      in => $p1_interleave_onset_bitcount,
      enable => "state_idle & (upstream_read | upstream_write)",
    });

    e_assign->adds(
      [
        {name => "burst_agenda", width => $opt->{downstream_burstcount_width},},
        "($opt->{downstream_max_burstcount} | interleave_onset)"
      ],
      [
        {name => "max_burst_size", width => $opt->{downstream_burstcount_width},}, 
        "(burst_agenda & (~burst_agenda + 1))"
      ]
    );

    push @{$opt->{extra_sim_signals}}, qw(
      max_burst_size
      burst_agenda
      initial_interleave_onset
      interleave_onset
      interleave_onset_bitcount
    );
  }
  else
  {
    # downstream_maxburstcount == 1 or sequential-burst slave.
    # Burst in slave maxburstcount chunks (with a partial end-burst, if needed).
    e_assign->add([
      {name => "max_burst_size", width => $opt->{downstream_burstcount_width},},
      "$opt->{downstream_max_burstcount}"
    ]);
  }

  # Parcel out bursts to the downstream slave of no more than the allowed size
  # until the number of transactions requested by the upstream master has been
  # satisfied.
  my $downstream_burstcount = 
    "((transactions_remaining > max_burst_size) ? max_burst_size : " .
    "transactions_remaining)";

  if ($opt->{dbs_shift} < 0)
  {
    # Negative dbs writes must only do bursts of 1 (narrow bursts within
    # a wide slave data word are not supported by burst slaves - they'd be
    # incrementing their internal address counter on each narrow write within
    # the burst, spreading the narrow data over multiple wide slave addresses).
    $downstream_burstcount = "current_upstream_read ? ($downstream_burstcount) : 1";
  }

  e_assign->add({
    lhs => e_signal->new({
      name => 'downstream_burstcount',
      width => $opt->{downstream_burstcount_width},
      never_export => 1,
    }),
    rhs => $downstream_burstcount,
  });
}

sub make_downstream_arbitrationshare($$$)
{
  my ($project, $module, $opt) = @_;

  if (!$opt->{dynamic_slave})
  {
    # Read or write, native slaves see a sequence of bursts of
    # size 1. Lock up arbitration during the upstream burst.
    e_assign->add({
      lhs => 'downstream_arbitrationshare',
      rhs => "upstream_burstcount",
    });

    return;
  }
  
  # Arbitration share on write is simply the same value sent by the upstream
  # burst master (dbs-adjusted if positive dbs, but not if negative - negative dbs
  # write bursts do exactly as many writes as the narrow master requests).
  #
  # On read, we have to take into account that each "transaction" in a read
  # burst actually contains downstream_burstcount transactions.  What about
  # upstream_burstcount values  which don't divide nicely?  Request one arb
  # share downstream, for the final sub-burst, in that case.
  #
  # Another way of thinking about the godawful expression below is that it's
  # ceil(upstream master burstcount (dbs adjusted) / max slave burstcount). 
  # Upstream_master_burstcount varies at runtime, so the "division" must occur
  # at runtime, but fortunately max slave burstcount is constant and an integer
  # power of two, so the division is just a right-shift.
  
  # Special case: if downstream_max_burstcount is 1, just pass the master
  # burstcount along.
  my $read_arbshare_expression = "dbs_adjusted_upstream_burstcount";
  if ($opt->{downstream_max_burstcount} > 1)
  {
    my $ds_burstcount_bits = log2($opt->{downstream_max_burstcount});

    # Restrict the size of ds_burstcount_bits, or it'll lead to indexing
    # beyond the size of dbs_adjusted_upstream_burstcount for slaves with
    # large max burstcount.
    $ds_burstcount_bits = $opt->{dbs_upstream_burstcount_width}
      if ($ds_burstcount_bits > $opt->{dbs_upstream_burstcount_width});

    my $ds_select = ($ds_burstcount_bits > 1) ? "@{[$ds_burstcount_bits - 1]} : 0" : "0";
    
    # Interleaved and sequential arbitrationshare differ only in the
    # existence of the signal 'interleave_onset' and the count of its
    # 1-bits.  Make 0-value versions of those signals for the
    # sequential case.
    my $interleave_onset_bitcount;
    my $interleave_onset;
    if ($opt->{slave_interleave})
    {
      $interleave_onset_bitcount = "interleave_onset_bitcount";
      $interleave_onset = "interleave_onset";
    }
    else
    {
      $interleave_onset_bitcount = 0;
      $interleave_onset = 0;
    }
    e_assign->add([
      {name => "interleave_end", width => $opt->{dbs_upstream_burstcount_width},},
      "(dbs_adjusted_upstream_burstcount > $interleave_onset) ? (dbs_adjusted_upstream_burstcount - $interleave_onset) : 0",
    ]);
    
    $read_arbshare_expression = "$interleave_onset_bitcount + " .
      "(interleave_end >> $ds_burstcount_bits) + " .
      "|(interleave_end[$ds_select])";
  }
    
  e_assign->add({
    lhs => 'downstream_arbitrationshare',
    rhs => "current_upstream_read ? ($read_arbshare_expression) : dbs_adjusted_upstream_burstcount",
  });

  push @{$opt->{extra_sim_signals}}, qw(
    downstream_arbitrationshare
  );
}

sub make_burstdone_signals($$$)
{
  my ($project, $module, $opt) = @_;
  
  if (!$opt->{dynamic_slave})
  {
    return;
  }

  e_assign->adds(
    # When is an upstream burst done?  During the burst, atomic_counter counts
    # from 0 up to downstream_burstsize - 1, repeatedly.  On the final downstream burst,
    # atomic_counter will reach transactions_remaining.  Note that atomic_counter is
    # narrower than transactions_remaining.  In other words, the upper bits of
    # transactions_remaining will be 0 when upstream_burstdone is true.
    #
    # Note asymmetry: transactions_remaining is compared against (atomic_counter + 1) on a write.
    {
      lhs => 'upstream_burstdone',
      rhs => "current_upstream_read ? " .
        "(transactions_remaining == downstream_burstcount) & downstream_read & ~downstream_waitrequest : " .
        "(transactions_remaining == (atomic_counter + 1)) & downstream_write & ~downstream_waitrequest",
    },
    {
      lhs => {name => 'p1_atomic_counter', width => $opt->{downstream_burstcount_width}},
      rhs => "atomic_counter + (downstream_read ? downstream_burstcount : 1)",
    },
    {
      lhs => 'downstream_burstdone',
      rhs => '(downstream_read | downstream_write) & ~downstream_waitrequest & (p1_atomic_counter == downstream_burstcount)',
    }
  );
  
  push @{$opt->{extra_sim_signals}}, qw(
    upstream_burstdone
    downstream_burstdone
  );
}

sub make_downstream_address($$$)
{
  my ($project, $module, $opt) = @_;

  if (!$opt->{dynamic_slave})
  {
    e_assign->add({
      lhs => "downstream_address",
      rhs => "current_upstream_address",
    });

    return;
  }

  if ($opt->{dynamic_slave})
  {
    # For negative dbs, only increment the write offset when we're about to cross a slave word boundary.
    my @write_transaction_increment_and_terms =
      ("downstream_write", "~downstream_waitrequest", "downstream_burstdone");
    if ($opt->{dbs_shift} < 0)
    {
      # Add the downstream_byteenable msb to the list of enable terms.
      push @write_transaction_increment_and_terms,
        "downstream_byteenable[@{[-1 + $opt->{ceil_data_width} / 8]}]";
    }
    my $write_transaction_increment_enable = and_array(@write_transaction_increment_and_terms);

    e_register->add({
      out => ['write_address_offset', $opt->{dbs_upstream_burstcount_width}],
      in =>
        "state_idle & upstream_write ? 0 : " .
        "($write_transaction_increment_enable) ? write_address_offset + downstream_burstcount : " .
        "write_address_offset",
      enable => 1,
    });

    e_register->add({
      out => ['read_address_offset', $opt->{dbs_upstream_burstcount_width}],
      in =>
        "state_idle & upstream_read ? 0 : " .
        "(downstream_read & ~downstream_waitrequest) ? read_address_offset + downstream_burstcount : " .
        "read_address_offset",
      enable => 1,
    });

  }
  else
  {
    # For native slaves, make placeholder versions of a handful of signals.
    e_assign->adds(
      [{name => 'read_address_offset', never_export => 1,}, 0,],
      [{name => 'write_address_offset', never_export => 1,}, 0,],
    );
  }

  e_assign->add({
    lhs => ['address_offset', $opt->{dbs_upstream_burstcount_width}],
    rhs => 'current_upstream_read ? read_address_offset : write_address_offset',
  });
  
  push @{$opt->{extra_sim_signals}}, qw(
    read_address_offset
    address_offset
    write_address_offset
  );

  # The upstream address is assumed to be aligned on an upstream native word-size
  # boundary. This module's master arb will drive address lsbs with a counting
  # sequence in the case of positive dbs, so mask off those lower address bits.
  my $address_selection;
  if ($opt->{dynamic_slave} && ($opt->{dbs_shift} > 0))
  {
    my $master_aligned_0_bits = log2($opt->{master_data_width} / 8);
    $address_selection = 
      sprintf("{current_upstream_address[%d:%d], %d'b%s}",
        $opt->{byteaddr_width} - 1,
        $master_aligned_0_bits,
        $master_aligned_0_bits,
        '0' x $master_aligned_0_bits
      );
  }
  else
  {
    $address_selection = 'current_upstream_address';
  }

  # You may be surprised to learn that the downstream master interface's
  # address is a slave-datawidth-aligned address, _not_ the usual master
  # byte address.
  #
  # Create a native-aligned address by shifting the address base, then
  # adding the already-aligned address_offset.

  # Note: the intermediate signal "downstream_address_base" is necessary to
  # avoid a problem in VHDL: A_SRL() isn't happy with a concatenation of
  # signals (which $address_selection is, sometimes) as first parameter.
  e_assign->add({
    lhs => {name => "downstream_address_base", width => $opt->{byteaddr_width},},
    rhs => $address_selection,
  });
  
  my $adjusted_address_offset = 'address_offset';
  if ($opt->{downstream_addr_shift} > 0)
  {
    $adjusted_address_offset = "{$adjusted_address_offset, " .
      sprintf(
        "%d'b%s}", 
        $opt->{downstream_addr_shift}, '0' x $opt->{downstream_addr_shift}
      );
  }
  e_assign->add({
    lhs => "downstream_address",
    rhs => "downstream_address_base + $adjusted_address_offset", 
  });
}

sub make_downstream_read($$$)
{
  my ($project, $module, $opt) = @_;

  if (!$opt->{dynamic_slave})
  {
    e_assign->add([
      'downstream_read',
      'upstream_read | (|transactions_remaining)',
    ]);

    return;
  }
  
  e_register->adds(
    {
      out => 'downstream_read',
      enable => "~downstream_read | ~downstream_waitrequest",
      in =>
        # set...
        "state_idle & upstream_read ? 1 : " .
        # ... reset...
        "(transactions_remaining == downstream_burstcount) ? 0 : " .
        # ... hold.
        "downstream_read",
    },
  );
}

sub make_upstream_readdata($$$)
{
  my ($project, $module, $opt) = @_;

  if (!$opt->{dynamic_slave})
  {
    # Simple logic for native slaves.
    e_register->adds 
    (
      # These registers didn't seem to be needed by pabst 1753, but for some
      # reason the CPU data master needs them for uart readdata. I note that
      # in pabst 1753, readdatavalid pulses don't begin arriving from the slave
      # until the cycle after downstream_waitrequest has dropped, but for the
      # CPU data master and the uart, waitrequest dropping and readdatavalid
      # are coincident. These registers separate the two.
      {out => "upstream_readdatavalid", in => "downstream_readdatavalid", enable => 1,},
      {out => "upstream_readdata", in => "downstream_readdata", enable => 1,},
    );
      
    push @{$opt->{extra_sim_signals}}, qw(downstream_readdatavalid upstream_readdatavalid);

    return;
  }

  if ($opt->{dbs_shift} >= 0)
  {
    e_assign->adds 
    (
      ["upstream_readdatavalid", "downstream_readdatavalid"],
      ["upstream_readdata", "downstream_readdata"],   
    );

    # fifo_empty is needed in various places, but
    # only really exists for negative dbs.  Generate a dummy
    # version here.
    e_assign->add([{name => 'fifo_empty', never_export => 1,}, 1]);
    e_assign->add([{name => 'p1_fifo_empty', never_export => 1,}, 1]);
  }
  else
  {
    # Dynamic negative-dbs reads require special attention.
    # Method:
    # Execute a sequence of downstream burst reads, of the max size, until
    # the upstream burst count is satisfied.
    # Because the wide slave can likely provide data faster than the narrow
    # master can accept it, we need a FIFO.  The downstream read data writes
    # into the fifo; the upstream readdata reads from the fifo, with a counter
    # (0 .. $last_segment) and logic to replicate the designated segment on
    # upstream_readdata.
    # 
    my $num_segments = $opt->{ceil_data_width} / $opt->{master_data_width};
    my $last_segment = $num_segments - 1;
    
    my $negative_dbs_bits = -$opt->{dbs_shift};
    my $negative_dbs_base = log2($opt->{master_data_width} / 8);
    my $upstream_negative_dbs_address_index =
      $negative_dbs_bits == 1 ? "$negative_dbs_base" :
      sprintf("%d : %d", $negative_dbs_bits - 1 + $negative_dbs_base, $negative_dbs_base);
    my $readdatavalid_counter = "negative_dbs_rdv_counter";
    my $readdatavalid_counter_enable = "fifo_datavalid";
    e_register->add({
      out => {name => $readdatavalid_counter, width => log2($num_segments),},
      in => "(state_idle & upstream_read & ~upstream_waitrequest) ? upstream_address[$upstream_negative_dbs_address_index] : $readdatavalid_counter_enable ? $readdatavalid_counter + 1 : $readdatavalid_counter",
      enable => 1,
    });

    # Fifo writedata: downstream_readdata
    # Fifo write: downstream_readdatavalid
    # Fifo read: upstream_segment_counter == $last_segment
    # upstream_segment_counter enable: fifo_datavalid
    e_assign->adds(
      ["fifo_read", "~fifo_empty && (($readdatavalid_counter == $last_segment) || ((full_width_rdv_counter + 1) == current_upstream_burstcount))"],
      ["fifo_write", "downstream_readdatavalid"],
      [{name => "fifo_wr_data", width => $opt->{data_width}}, "downstream_readdata"],
    );

    my $fifo_numwords = ceil($opt->{upstream_max_burstcount} * 2**$opt->{dbs_shift});

    # ... oh, and if the downstream burst could straddle a wide slave word boundary, the
    # fifo needs to be one location deeper.  Sadly, this fifo only supports
    # power-of-two depths.
    $fifo_numwords *= 2;

    my $fifo_module = e_fifo->new({
      device_family => $module->project()->device_family(),
      name_stub => $module->name(),
      data_width => $opt->{data_width},
      fifo_depth => $fifo_numwords,
      implement_as_esb => 0,
      full_port => 1,
      empty_port => 1,
      p1_empty_port => 1,
      flush => "flush_fifo",
    });
    
    e_assign->add(["flush_fifo", "1'b0"]);
    e_signal->add({name => "fifo_empty", never_export => 1, });
    e_instance->add({
      module => $fifo_module,
      port_map => {
        inc_pending_data => "fifo_write",
        clk_en => "1'b1",
      },
    });

    # This FIFO should never overflow.
    my $condition = "fifo_full && fifo_write";
    my $then = [
      e_sim_write->new({
        show_time => 1,
        spec_string => "simulation assertion failed: " . $module->name() . ": illegal write into full fifo.",
      }),
      e_stop->new(),
    ];
    e_process->add({
      contents => [
        e_if->new({
          condition => $condition,
          then => $then,
        }),
      ],
      tag => 'simulation',
    });

    push @{$opt->{extra_sim_signals}}, qw(fifo_read fifo_full fifo_empty p1_fifo_empty fifo_write fifo_datavalid fifo_rd_data fifo_wr_data);

    # This counter resets on acceptance of a new read burst, and counts actual
    # readdatavalid pulses delivered to the master.
    e_register->add({
      out => {name => "full_width_rdv_counter", width => $opt->{upstream_burstcount_width},},
      enable => 1,
      in => "(state_idle & upstream_read & ~upstream_waitrequest) ? 0 : " .
        "upstream_readdatavalid ? full_width_rdv_counter + 1 : " .
        "full_width_rdv_counter",
      # in => "full_width_rdv_counter + 1",
      # enable => "state_idle | upstream_readdatavalid",
      # sync_reset => "state_idle",
    });
    push @{$opt->{extra_sim_signals}}, "full_width_rdv_counter";

    # Upstream readdatavalid is set if the fifo contains valid data. The negative dbs counter
    # ensures that only the requested portions of a partial slave data word are validated
    # upstream.
    e_assign->adds(["upstream_readdatavalid", "$readdatavalid_counter_enable"]);

    my @u_read_table;
    for my $i (0 .. -1 + $num_segments)
    {
      my $segment_select =
        sprintf("[%d : %d]", -1 + ($i + 1) * $opt->{master_data_width}, $i * $opt->{master_data_width});
      push @u_read_table, "($i == $readdatavalid_counter)", "{" . join(", ", ("fifo_rd_data$segment_select") x $num_segments) . "}"
    }
   
    e_mux->add({
      lhs => "upstream_readdata",
      table => [@u_read_table],
    });

    push @{$opt->{extra_sim_signals}}, ($readdatavalid_counter);
  }

  push @{$opt->{extra_sim_signals}}, qw(downstream_readdatavalid upstream_readdatavalid);
}

sub make_downstream_write($$$)
{
  my ($project, $module, $opt) = @_;
  if (!$opt->{dynamic_slave})
  {
    e_assign->adds 
    (
      ["downstream_write", "upstream_write & !downstream_read"],
      ["downstream_byteenable", "upstream_byteenable"],
    );
    return;
  }
  
  if ($opt->{dbs_shift} >= 0)
  {
    e_register->add({
      out => 'downstream_write_reg',
      enable => "~downstream_write_reg | ~downstream_waitrequest",
      in =>
        # First upstream write access after idle:"
        "state_idle & upstream_write ? 1 : " .
        # Subsequent bursts from upstream:"
        # "state_busy & upstream_write & ~upstream_waitrequest ? 1 : " .
        # End of write burst, with no follow-on:"
        "((transactions_remaining == downstream_burstcount) & downstream_burstdone) ? 0 : " .
        # nothing happening.
        "downstream_write_reg",
    });
    e_assign->adds 
    (
      ["downstream_write", "downstream_write_reg & upstream_write & !downstream_read",],
      ["downstream_byteenable", "upstream_byteenable"],
    );
  }
  else
  {
    # Negative dynamic-bus-sized bursts require special handling.
    # Method: pass wide upstream data through, with appropriate
    # byteeanble.
    
    # downstream_bytenable: set to upstream_byteeanble at the start
    # of a burst, and rotates left in segment-chunks on each successive
    # downstream write.
    # 
    # Example:
    # slave data width=32, master data width=8, burst of 4 @ 0
    # offset   downstream be
    # 0          {0, 0, 0, 1}
    # 1          {0, 0, 1, 0}
    # 2          {0, 1, 0, 0}
    # 3          {1, 0, 0, 0}
    
    # slave data width=32, master data width=16, burst of 2 @ 0
    # offset   downstream be
    # 0          {0, 0, 1, 1}
    # 1          {1, 1, 0, 0}
    
    # slave data width=16, master data width=8, burst of 4 @ 0
    # offset   downstream be
    # 0          {0, 1}
    # 1          {1, 0}
    # 2          {0, 1} 
    # 3          {1, 0}
    
    # Make a list of all the individual downstream byteenable bits.
    my @shifted_byteenable;
    for my $index (0 .. -1 + $opt->{ceil_data_width} / 8)
    {
      unshift @shifted_byteenable, "downstream_byteenable\[$index]";
    }
    
    # Shift the list appropriately to the master data width.
    for my $index (0 .. -1 + $opt->{master_data_width} / 8)
    {
      my $bit = shift @shifted_byteenable;
      push @shifted_byteenable, $bit;
    }

    e_register->add({
      out => {name => "downstream_byteenable", width => $opt->{ceil_data_width} / 8, export => 1,},
      in =>
        "state_idle ? upstream_byteenable : " .
        "(downstream_write & ~downstream_waitrequest) ? @{[concatenate(@shifted_byteenable)]} : " .
        "downstream_byteenable",
      enable => 1,
    });

    e_assign->add(["downstream_write", "state_busy & upstream_write & !downstream_read"]);
  }
}

sub make_downstream_writedata($$$)
{
  my ($project, $module, $opt) = @_;

  e_assign->add(["downstream_writedata", "upstream_writedata"]);
}

sub make_state_machine($$$)
{
  my ($project, $module, $opt) = @_;

  if (!$opt->{dynamic_slave})
  {
    return;
  }

  # Read state machine.  The burst adapter can hold one pending read transaction, but must
  # halt further upstream reads or writes until that pending transaction is complete (including
  # emptying the fifo, in the case of negative dbs).
  # Future optimization: for negative dbs, allow the next read to proceed as soon as the
  # fifo has enough space, given that it's emptying at top speed.
  e_assign->adds(
    {
      lhs => 'p1_state_idle',
      rhs =>
      "state_idle & ~upstream_read & ~upstream_write | " .
      "state_busy & (data_counter == 0) & p1_fifo_empty & ~pending_upstream_read & ~pending_upstream_write",
    },
    {
      lhs => 'p1_state_busy',
      rhs =>
      "state_idle & (upstream_read | upstream_write) | " .
      "state_busy & (~(data_counter == 0) | ~p1_fifo_empty | pending_upstream_read | pending_upstream_write)",
    },
  );

  e_assign->adds(
    {
      lhs => 'enable_state_change',
      rhs => "~(downstream_read | downstream_write) | ~downstream_waitrequest",
    }
  );
  e_register->adds(
    {
      out => 'pending_upstream_read_reg',
      enable => 1,
      sync_set => "upstream_read & state_idle",
      sync_reset => $opt->{dbs_shift} < 0 ? "downstream_readdatavalid" : "upstream_burstdone",
      priority => "set",
    },
    {
      out => 'pending_upstream_write_reg',
      enable => 1,
      sync_set => "upstream_write & (state_idle | ~upstream_waitrequest)",
      sync_reset => "upstream_burstdone",
    },
    {
      out => 'state_idle',
      in => 'p1_state_idle',
      enable => "enable_state_change",
      async_value => 1,
    },
    {
      out => 'state_busy',
      in => 'p1_state_busy',
      enable => "enable_state_change",
      async_value => 0,
    },
  );
  e_assign->adds(
    [
      "pending_upstream_read",
      "pending_upstream_read_reg",
    ],
    [
      "pending_upstream_write",
      "pending_upstream_write_reg & ~upstream_burstdone",
    ],
    [
      "pending_register_enable",
      "state_idle | ((upstream_read | upstream_write) & ~upstream_waitrequest)"
    ],
  );

  push @{$opt->{extra_sim_signals}}, qw(
    state_idle
    state_busy
    pending_register_enable
    pending_upstream_read
    pending_upstream_read_reg
    pending_upstream_write
    pending_upstream_write_reg
  );
}

sub make_dbs_adjusted_upstream_burstcount($$$)
{
  my ($project, $module, $opt) = @_;
  
  if (!$opt->{dynamic_slave})
  {
    return;
  }

  my $adjusted_upstream_burstcount;
  if (!$opt->{dynamic_slave} || ($opt->{dbs_shift} == 0))
  {
    # Native slave or no-dbs: no adjustment is necessary.
    $adjusted_upstream_burstcount = "upstream_burstcount"
  }
  elsif ($opt->{dbs_shift} > 0)
  {
    # Write or read, a wide master's burstcount counts for more than a slave's.
    $adjusted_upstream_burstcount = concatenate(
      "upstream_burstcount",
      sprintf("%d'b%s", $opt->{dbs_shift}, '0' x $opt->{dbs_shift})
    );
  }
  else
  {
    # Negative-dbs.
    
    # Calculate the range of _byte_ addresses which spans the requested data,
    # quantized to slave-data-width addresses.
    my $slave_mask_bits = (1 << $opt->{downstream_addr_shift}) - 1;
    my $master_shift = log2($opt->{master_data_width} / 8);
    my $upstream_byte_burst_count = concatenate(
      "upstream_burstcount",
      $master_shift ? sprintf("%d'b%s", $master_shift, '0' x $master_shift) : ''
    );

    # Optimization: do I really need the quantized... signals to be full
    # address width?
    e_assign->add([
      {
        name => "quantized_burst_base",
        width => $opt->{byteaddr_width}
      },
      "upstream_address & ~$slave_mask_bits"
    ]);
    e_assign->add([
      {name => "quantized_burst_limit", width => $opt->{byteaddr_width}},
      "((upstream_address + $upstream_byte_burst_count - 1) | $slave_mask_bits) + 1"
    ]);

    e_assign->add([
      {name => "negative_dbs_read_expression", width => $opt->{dbs_upstream_burstcount_width}},
      "(quantized_burst_limit - quantized_burst_base) >> " . log2($opt->{ceil_data_width} / 8)
    ]);

    push @{$opt->{extra_sim_signals}}, qw(
      quantized_burst_base
      quantized_burst_limit
      negative_dbs_read_expression
    );
    
    # Each read from the wide slave counts for (slave data width / master data width)
    # master burstcount units, so divide (rounding up).
    # Narrow writes are performed one at a time, so no adjustment is necessary.
    $adjusted_upstream_burstcount = 
      "upstream_read ? negative_dbs_read_expression : upstream_burstcount";
  }

  e_assign->adds(
    {
      lhs => {name => 'dbs_adjusted_upstream_burstcount', width => $opt->{dbs_upstream_burstcount_width},},
      rhs => "pending_register_enable ? read_write_dbs_adjusted_upstream_burstcount : registered_read_write_dbs_adjusted_upstream_burstcount",
    },
    {
      lhs => {name => 'read_write_dbs_adjusted_upstream_burstcount', width => $opt->{dbs_upstream_burstcount_width},},
      rhs => $adjusted_upstream_burstcount,
    },
  );

  e_register->adds(
    {
      out => {name => 'registered_read_write_dbs_adjusted_upstream_burstcount', width => $opt->{dbs_upstream_burstcount_width},},
      in => "read_write_dbs_adjusted_upstream_burstcount",
      enable => 'pending_register_enable',
    },
  );

  push @{$opt->{extra_sim_signals}}, qw(
    read_write_dbs_adjusted_upstream_burstcount
    registered_read_write_dbs_adjusted_upstream_burstcount
    dbs_adjusted_upstream_burstcount
  );
}

sub make_current_upstream_signals($$$)
{
  my ($project, $module, $opt) = @_;
  
  if (!$opt->{dynamic_slave})
  {
    e_register->adds(
      {
        out => {name => 'registered_upstream_address', width => $opt->{byteaddr_width}, never_export => 1,},
        in => 'upstream_address',
        enable => '~|transactions_remaining',
      },
    );
    e_assign->adds(
      {
        lhs => {name => 'current_upstream_address', width => $opt->{byteaddr_width}, never_export => 1,},
        rhs => "~|transactions_remaining ? upstream_address : registered_upstream_address",
      },
    );

    push @{$opt->{extra_sim_signals}}, qw(
      registered_upstream_address
      current_upstream_address
    );

    return;
  }

  e_register->adds(
    {
      out => 'registered_upstream_read',
      in => 'upstream_read',
      enable => 'pending_register_enable',
    },
    {
      out => 'registered_upstream_write',
      in => 'upstream_write',
      enable => 'pending_register_enable',
    },
    {
      out => {name => 'registered_upstream_burstcount', width => $opt->{upstream_burstcount_width},},
      in => 'upstream_burstcount',
      enable => 'pending_register_enable',
    },
    {
      out => {name => 'registered_upstream_address', width => $opt->{byteaddr_width},},
      in => 'upstream_address',
      enable => 'pending_register_enable',
    },
  );

  e_assign->adds(
    {
      lhs => 'current_upstream_read',
      rhs => "registered_upstream_read & !downstream_write",
    },
    {
      lhs => 'current_upstream_write',
      rhs => "registered_upstream_write",
    },
    {
      lhs => {name => 'current_upstream_address', width => $opt->{byteaddr_width},},
      rhs => "registered_upstream_address",
    },
    # The other two "current" signals are registered from upstream signals.  
    # current_upstream_burstcount needs to be available one cycle early, because
    # it feeds into transactions_remaining and transactions_remaining_reg.
    # Do the usual lookahead using the register enable and register input value.
    {
      lhs => {
        name => 'current_upstream_burstcount',
        width => $opt->{upstream_burstcount_width},
        never_export => 1,
      },
      rhs => "pending_register_enable ? upstream_burstcount : registered_upstream_burstcount",
    },
  );
  
  push @{$opt->{extra_sim_signals}}, qw(
    current_upstream_read
    current_upstream_burstcount
    current_upstream_address
    registered_upstream_read
    registered_upstream_address
  );
}

sub make_upstream_waitrequest($$$)
{
  my ($project, $module, $opt) = @_;

  if (!$opt->{dynamic_slave})
  {
    # Simple logic for native slaves.
    e_assign->adds([
      'upstream_waitrequest',
      'downstream_waitrequest | (|transactions_remaining)',
    ]);

    return;
  }

  e_assign->adds 
  (
    ["upstream_read_run", "state_idle & upstream_read"],
    ["upstream_write_run", "state_busy & upstream_write & "
     ."~downstream_waitrequest & !downstream_read",],
    ["upstream_waitrequest", 
      "(upstream_read | current_upstream_read) ? ~upstream_read_run : " .
      "current_upstream_write ? ~upstream_write_run : " .
       "1"
     ],
  );    
}

sub set_sim_wave_signals
{
  my ($project, $module, $opt) = @_;

  if ($project->system_ptf()->{WIZARD_SCRIPT_ARGUMENTS}->{do_burst_adapter_sim_wave})
  {
    my @sim_signals = qw(
        downstream_address
        downstream_burstcount
        downstream_byteenable
        downstream_read
        downstream_readdata
        downstream_waitrequest
        downstream_write
        downstream_writedata

        upstream_address
        upstream_burstcount
        upstream_byteenable
        upstream_read
        upstream_readdata
        upstream_waitrequest
        upstream_write
        upstream_writedata
    );
    
    push @sim_signals, @{$opt->{extra_sim_signals}};
    
    my %uniq = map {$_ => 1} @sim_signals;

    set_sim_ptf(
      [sort keys %uniq],
      $module,
      $project);
  }
}

sub set_sim_ptf
{
  my ($signals, $module, $project) = @_;
  
  # Signals which match any of the following regexps should be
  # radix hex.
  my @bus_signals = qw(
    address
    data$
  );
  
  my $module_name = $module->name();
  my $sys_ptf = $project->system_ptf();
  my $mod_ptf = $sys_ptf->{"MODULE $module_name"};
  $mod_ptf->{SIMULATION} = {} if (!defined($mod_ptf->{SIMULATION}));
  $mod_ptf->{SIMULATION}->{DISPLAY} = {} if (!defined($mod_ptf->{SIMULATION}->{DISPLAY})); 
  
  my $sig_ptf = $mod_ptf->{SIMULATION}->{DISPLAY};

  my $signum = 0;
  my $tag;
  for my $sig (@$signals)
  {
    my $tag = to_base_26($signum);
    my $radix;
    my $format;
    my $name;
    
    if ($sig =~ /Divider\s*(.*)/)
    {
      $name = $1;
      $format = 'Divider';
      $radix = '';
    }
    else
    {
      $name = $sig;
      $radix = 'hexadecimal';
      $format = 'Logic';
      if (grep {$sig =~ /$_/} @bus_signals)
      {
        $format = 'Literal';
      }
    }

    $sig_ptf->{"SIGNAL $tag"} = {name => $name, radix => $radix, format => $format};
    $signum++;  
  }
}

sub make_adapter
{
  my $project = shift;
  my $module = $project->top();
  my $marker = e_default_module_marker->new($module);
  my $opt = copy_of_hash($project->WSA());

  assign_options($project, $module, $opt);
  validate_options($opt);

  # In the module comment, print out each parameter (but skip
  # non-ref parameters, for brevity).
  my @printable_keys = grep {"" eq ref($opt->{$_})} keys %$opt;
  $module->comment(
    $module->comment() .
    "\nBurst adapter parameters:\n" .
    join("\n", (map {"$_: $opt->{$_}"} sort @printable_keys)) .
    "\n\n"
  );
  
  make_interfaces($project, $module, $opt);
  make_burstdone_signals($project, $module, $opt);
  make_dbs_adjusted_upstream_burstcount($project, $module, $opt);
  make_state_machine($project, $module, $opt);
  make_current_upstream_signals($project, $module, $opt);
  make_counters($project, $module, $opt);
  make_downstream_burstcount($project, $module, $opt);
  make_downstream_arbitrationshare($project, $module, $opt);
  make_downstream_address($project, $module, $opt);
  make_downstream_read($project, $module, $opt);
  make_upstream_readdata($project, $module, $opt);
  make_downstream_write($project, $module, $opt);
  make_downstream_writedata($project, $module, $opt);
  make_upstream_waitrequest($project, $module, $opt);

  # Add simulation wave signals, if enabled.
  set_sim_wave_signals($project, $module, $opt);
} 

1;

