#!/bin/perl -w
use strict;

# This script converts a POF file into a comma-separated list of numbers
# suitable for inclusion in a C source file.
#
# The program has three tasks:
# 1) strip off header bytes
# 2) bit-reverse POF data
# 3) strip off trailer bytes.
#

sub bit_reverse
{ 
  my $b = shift;

  return
      (($b & 0x80) ? 0x01 : 0) |
      (($b & 0x40) ? 0x02 : 0) |
      (($b & 0x20) ? 0x04 : 0) |
      (($b & 0x10) ? 0x08 : 0) |
      (($b & 0x08) ? 0x10 : 0) |
      (($b & 0x04) ? 0x20 : 0) |
      (($b & 0x02) ? 0x40 : 0) |
      (($b & 0x01) ? 0x80 : 0);
  
}

sub get_data
{
  my $fh = shift;
  my $data;

  my $data_length = read $fh, $data, 512;
  return () if $data_length == 0;
  
  my @values = split //, $data;
  
  # Convert binary characters to their numeric ASCII values.
  map {$_ = ord} @values;
  
  return @values;
}

my $pof = shift @ARGV;
print STDERR "converting '$pof' to C array syntax\n";

open POF, $pof or die "Can't open input file '$pof'\n";

binmode POF;

# Get info from the header
my @values = get_data(\*POF);
if (@values == 0)
{
  print STDERR "Can't read input file '$pof'\n";
  exit -1;
}

# Skip the header (all bytes before the first 0xFF byte)
my $value;
my $i;
my $header_length;
for $i (0 .. @values - 1)
{
  $value = $values[$i];

  if ($value == 0xFF)
  {
    $header_length = $i;
    last;
  }
}

if ($value != 0xFF)
{
  print STDERR "Failed to parse pof file '$pof' (no 0xFF in header)\n";
  exit -1;
}

# Find the length within the post-header, pre-data section.
# Assumption: the length is the 4 bytes found after the
# 0x6A marker byte:
# FF FF FF FF FF FF 6A D6 FF 40 00 0C 42 16 00 6A
#                   m              b0 b1 b2 b3                             
# This seems fragile!
my $index_6A;
for my $j ($header_length .. @values -1)
{
  $value = $values[$j];
  
  if ($value == 0x6A)
  {
    $index_6A = $j;
    last;
  }
}
if ($value != 0x6A)
{
  print STDERR "Failed to find 0x6A marker in pof file '$pof'\n";
  exit -1;
}

my $length_in_bits =
  $values[$index_6A + 5] |
  ($values[$index_6A + 6] << 8) |
  ($values[$index_6A + 7] << 16) |
  ($values[$index_6A + 8] << 24);

my $length = $length_in_bits / 8;

# Print the data.
my $start_index = $header_length;
my $byte_count = 0;
BUFFER: while (1)
{
  for my $i ($start_index .. @values - 1)
  {
    my $value = $values[$i];
    printf "0x%02X, ", bit_reverse($value);

    $byte_count++;
    
    if (($byte_count % 16) == 0)
    {
      printf("\n");
    }
    # Printed enough bytes yet?
    last BUFFER if ($byte_count > $length);
  }
  
  @values = get_data(\*POF);
  last if @values == 0; # Reached EOF.

  $start_index = 0;
}
printf("\n");

close POF;
