// ASMI Registers
typedef volatile struct
  {
  int np_asmirxdata;       // Read-only, 1-16 bit
  int np_asmitxdata;       // Write-only, same width as rxdata
  int np_asmistatus;       // Read-only, 9-bit
  int np_asmicontrol;      // Read/Write, 9-bit
  int np_asmireserved;     // reserved
  int np_asmislaveselect;  // Read/Write, 1-16 bit, master only
  int np_asmiendofpacket;  // Read/write, same width as txdata, rxdata.
  } np_asmi;

// ASMI Status Register Bits
enum
  {
  np_asmistatus_eop_bit  = 9,
  np_asmistatus_e_bit    = 8,
  np_asmistatus_rrdy_bit = 7,
  np_asmistatus_trdy_bit = 6,
  np_asmistatus_tmt_bit  = 5,
  np_asmistatus_toe_bit  = 4,
  np_asmistatus_roe_bit  = 3,

  np_asmistatus_eop_mask  = (1 << 9),
  np_asmistatus_e_mask    = (1 << 8),
  np_asmistatus_rrdy_mask = (1 << 7),
  np_asmistatus_trdy_mask = (1 << 6),
  np_asmistatus_tmt_mask  = (1 << 5),
  np_asmistatus_toe_mask  = (1 << 4),
  np_asmistatus_roe_mask  = (1 << 3),
  };

// ASMI Control Register Bits
enum
  {
  np_asmicontrol_sso_bit   = 10,
  np_asmicontrol_ieop_bit  = 9,
  np_asmicontrol_ie_bit    = 8,
  np_asmicontrol_irrdy_bit = 7,
  np_asmicontrol_itrdy_bit = 6,
  np_asmicontrol_itoe_bit  = 4,
  np_asmicontrol_iroe_bit  = 3,

  np_asmicontrol_sso_mask   = (1 << 10),
  np_asmicontrol_ieop_mask  = (1 << 9),
  np_asmicontrol_ie_mask    = (1 << 8),
  np_asmicontrol_irrdy_mask = (1 << 7),
  np_asmicontrol_itrdy_mask = (1 << 6),
  np_asmicontrol_itoe_mask  = (1 << 4),
  np_asmicontrol_iroe_mask  = (1 << 3),
  };


//ASMI memory definitions
#if na_asmi_64K
#define na_asmi_bulk_size   (0x2000)
#define na_asmi_sector_size (na_asmi_bulk_size >> 2)
#define na_asmi_page_size   0x20

#elif na_asmi_1M
#define na_asmi_bulk_size   (0x20000)
#define na_asmi_sector_size (na_asmi_bulk_size >> 2)
#define na_asmi_page_size   0x100

#elif na_asmi_4M
#define na_asmi_bulk_size   (0x80000)
#define na_asmi_sector_size (na_asmi_bulk_size >> 3)
#define na_asmi_page_size   0x100
#endif

//ASMI memory instructions
#define na_asmi_read    (unsigned char)0x03
#define na_asmi_write   (unsigned char)0x02
#define na_asmi_wren    (unsigned char)0x06
#define na_asmi_wrdi    (unsigned char)0x04
#define na_asmi_rdsr    (unsigned char)0x05
#define na_asmi_wrsr    (unsigned char)0x01
#define na_asmi_se      (unsigned char)0xd8
#define na_asmi_be      (unsigned char)0xc7
#define na_asmi_dp      (unsigned char)0xb9

//ASMI memory status register bit masks
#if (na_asmi_64K) || (na_asmi_1M)
#define na_asmi_bp      (unsigned char)0xc
#else
#define na_asmi_bp      (unsigned char)0x1c
#endif
#define na_asmi_wel     (unsigned char)0x2
#define na_asmi_wip     (unsigned char)0x1

//ASMI function error codes
#define na_asmi_success                 0
#define na_asmi_err_device_not_present  1
#define na_asmi_err_device_not_ready    2
#define na_asmi_err_timedout            3
#define na_asmi_err_write_failed        4
#define na_asmi_invalid_config          5

//ASMI protection masks
#define na_asmi_protect_none        0
#if (na_asmi_64K) || (na_asmi_1M)
#define na_asmi_protect_top_quarter 0x4
#define na_asmi_protect_top_half    0x8
#define na_asmi_protect_all         0xc
#else
#define na_asmi_protect_top_eighth  0x4
#define na_asmi_protect_top_quarter 0x8
#define na_asmi_protect_top_half    0xc
#define na_asmi_protect_all         0x10
#endif

//ASMI macros
//returns the protect bits shifted into the lsbs
#define nm_asmi_prot_sect(t) ((t & na_asmi_bp) >> 2)

//ASMI library routines
//

extern unsigned char nr_asmi_read_status ();
extern unsigned long nr_asmi_lowest_protected_address();
extern int nr_asmi_write_status (unsigned char value);
extern int nr_asmi_protect_region (int bpcode);
extern int nr_asmi_read_byte (unsigned long address, unsigned char *data);
extern int nr_asmi_write_byte (unsigned long address, unsigned char data);
extern int nr_asmi_erase_sector (unsigned long address);
extern int nr_asmi_erase_bulk ();
extern int nr_asmi_read_buffer (unsigned long address, int length, unsigned char *data);
extern int nr_asmi_write_page (unsigned long address, int length, unsigned char *data);
extern int nr_asmi_write_buffer (unsigned long address, int length, unsigned char *data);
extern int nr_asmi_address_past_config (unsigned long *addr);
        
