#Copyright (C)2001-2003 Altera Corporation
#Any megafunction design, and related net list (encrypted or decrypted),
#support information, device programming or simulation file, and any other
#associated documentation or information provided by Altera or a partner
#under Altera's Megafunction Partnership Program may be used only to
#program PLD devices (but not masked PLD devices) from Altera.  Any other
#use of such megafunction design, net list, support information, device
#programming or simulation file, or any other related documentation or
#information is prohibited for any other purpose, including, but not
#limited to modification, reverse engineering, de-compiling, or use with
#any other silicon devices, unless such use is explicitly licensed under
#a separate agreement with Altera or a megafunction partner.  Title to
#the intellectual property, including patents, copyrights, trademarks,
#trade secrets, or maskworks, embodied in any such megafunction design,
#net list, support information, device programming or simulation file, or
#any other related documentation or information provided by Altera or a
#megafunction partner, remains with Altera, the megafunction partner, or
#their respective licensors.  No other licenses, including any licenses
#needed under any third party's intellectual property, are provided herein.
#Copying or modifying any file, or portion thereof, to which this notice
#is attached violates this copyright.








use ptf_parse;

################################################################
# PTF_Translate_Old_Version
#
# Sometimes we will find ourselves in the unfortunate position
# of reading a PTF-file generated by an older version of the 
# kit.  This function is a pre-filter which is called before anything
# else happens to the PTF-file. 
#
# This script has these jobs:
#
#   1) Update all the signal-type names to their new version.
#
#   2) Make sure every module has the following required assignments
#      in its SYSTEM_BUILDER_INFO section:
#         - Data_Width
#         - Address_Width
#
#   3) Make sure every module has a correct "class" assignment at its
#      top-level.
#
################################################################
my $DEBUG = 0;
my %avalon_role_translation;
$avalon_role_translation {reset_n}                  = "resetn";
$avalon_role_translation {read_data}                = "readdata";
$avalon_role_translation {write_data}               = "writedata";
$avalon_role_translation {r_wn}                     = "writen";
$avalon_role_translation {w_rn}                     = "readn";
$avalon_role_translation {be_n}                     = "byteenablen";
$avalon_role_translation {chip_select}              = "chipselect";
$avalon_role_translation {wait_request}             = "waitrequest";
$avalon_role_translation {bus_wait}                 = "waitrequest";
$avalon_role_translation {registered_chip_select_n} = "registeredselectn";
$avalon_role_translation {irq}                      = "irq";
$avalon_role_translation {address}                  = "address";

my %master_type_translation;
$master_type_translation {clk}            = "master_input_clk";
$master_type_translation {reset_n}        = "master_input_resetn";
$master_type_translation {bus_wait}       = "master_input_waitrequest";
$master_type_translation {mem_is_32_bits} = "master_input_memis32bits";
$master_type_translation {ifetch}         = "master_output_ifetch";
$master_type_translation {data_to_cpu}    = "master_input_readdata";
$master_type_translation {data_from_cpu}  = "master_output_writedata";
$master_type_translation {mem_addr}       = "master_output_address";
$master_type_translation {mem_wr_n}       = "master_output_writen";
$master_type_translation {mem_rd_n}       = "master_output_readn";
$master_type_translation {mem_be_n}       = "master_output_byteenablen";
$master_type_translation {irq}            = "master_input_irq";
$master_type_translation {irq_number}     = "master_input_irqnumber";

my %class_translation_from_1_0;
$class_translation_from_1_0 {nios}       = "jnioswizard";
$class_translation_from_1_0 {uart}       = "juartwizard";
$class_translation_from_1_0 {pio}        = "jpiowizard";
$class_translation_from_1_0 {timer}      = "jtimerwizard";
$class_translation_from_1_0 {onchip_rom} = "jramwizard";
$class_translation_from_1_0 {onchip_ram} = "jramwizard";


my %class_translation_from_jw = (
  jnioswizard             => "altera_nios",
  juartwizard             => "altera_avalon_uart",
  jpiowizard              => "altera_avalon_pio",
  jtimerwizard            => "altera_avalon_timer",
  jramwizard              => "altera_avalon_onchip_memory",
  ji2cwizard              => "altera_avalon_i2c",
  jspiwizard              => "altera_avalon_spi",
  jusersocketwizard       => "altera_avalon_user_defined_interface",
  j2xidt71v016sawizard    => "altera_nios_dev_board_sram32",
  jam29lv800b_smallwizard => "altera_nios_dev_board_flash_small",
  jam29lv800bwizard       => "altera_nios_dev_board_flash",
  jidt71v016_smallwizard  => "altera_nios_dev_board_sram16",
);

my $PTF_CHANGES = 0;


sub PTF_Translate_From_Version_1_0
{
   my ($ptf_filename, $db_PTF_File) = (@_);

   my $db_Sys       = &PTF_Get_Required_Child_By_Path ($db_PTF_File, "SYSTEM");
   my $db_Sys_WSA   = &PTF_Get_Required_Child_By_Path ($db_Sys,
                                                    "WIZARD_SCRIPT_ARGUMENTS");

   &Rename_Old_edf_File($ptf_filename);

   printf STDERR "****PTF_Translate_Old_Version:
     Updating old-format PTF-file: $ptf_filename\n" if $DEBUG;

   $PTF_CHANGES += &PTF_Update_Boolean_Values ($db_Sys);

   # Get some old-tyme global parameters which we now want to poke
   # into the CPU:
   my $reset_module   = &get_data_by_path($db_Sys_WSA, "reset_module");
   my $reset_offset   = &get_data_by_path($db_Sys_WSA, "reset_offset");
   my $vecbase_module = &get_data_by_path($db_Sys_WSA, "vecbase_module");
   my $vecbase_offset = &get_data_by_path($db_Sys_WSA, "vecbase_offset");
   my $mainmem_module = &get_data_by_path($db_Sys_WSA, "mainmem_module");

   &add_child_data ($db_Sys_WSA, "datamem_module", $mainmem_module);

   my $num_modules = &get_child_count ($db_Sys);
   for (my $module_index = 0; $module_index < $num_modules; $module_index++)
   {
      my $db_Module = &get_child ($db_Sys, $module_index);
      next if &get_name ($db_Module) ne "MODULE";   # ignore non-modules.

      ################
      # Blast HDL_INFO  file-list assignments.
      # Any files that we need will be created by the new class's
      # generator_program.
      #
      my $db_HDLI = &get_child_by_path($db_Module,"HDL_INFO");
      &add_child_data ($db_HDLI, "Simulation_HDL_Files", "");
      &add_child_data ($db_HDLI, "Synthesis_HDL_Files", "");

      my $old_module_type = &get_data_by_path ($db_Module,
                                               "HDL_INFO/Module_Type");

      my $is_master = $old_module_type eq "nios";

      my $db_Types  = &get_child_by_path($db_Module,"PORT_WIRING/TYPE");
      my $db_Widths = &get_child_by_path($db_Module,"PORT_WIRING/WIDTH");
      my $db_WSA    = &get_child_by_path($db_Module,"WIZARD_SCRIPT_ARGUMENTS");
      my $db_SBI    = &get_child_by_path($db_Module,"SYSTEM_BUILDER_INFO");

      my $address_width = "--unknown--";
      my $data_width    = "--unknown--";
      my $has_reg_sel   = 0;
      my $has_irq       = 0;
      my $has_tri_state = 0;

      ################
      # Poke-in some Nios CPU-specific parameters, if this is the CPU:
      if ($is_master) {
        &add_child_data ($db_WSA, "reset_module",   $reset_module);
        &add_child_data ($db_WSA, "vecbase_module", $vecbase_module);
        &add_child_data ($db_WSA, "reset_offset",   $reset_offset);
        &add_child_data ($db_WSA, "vecbase_offset", $vecbase_offset);
      }

      ################ 
      # Translate all the old avalon-roles into new avalon-roles.
      #
      my $num_ports = &get_child_count ($db_Types);
      for (my $port_index = 0; $port_index < $num_ports; $port_index++)
      {
         my $db_Assignment = &get_child ($db_Types, $port_index);
         my $port_name = &get_name ($db_Assignment);
         my $type      = &get_data_by_path ($db_Types,  $port_name);
         my $width     = &get_data_by_path ($db_Widths, $port_name);

         my $new_type = $type;

         if ($old_module_type eq "nios" &&
             ($new_type =~ /(input|output)/ ))
         {
            # The old nios module had a very strange port-wiring section,
            # where no special types were given for the master's signals.
            # We have to use a special table to give the master some 
            # port-types, based on the (known) port-names on the Nios:
            #
            $new_type = $master_type_translation {$port_name};
         }

         foreach $old_role (keys(%avalon_role_translation))
         {
            my $new_role = $avalon_role_translation{$old_role};
            $new_type =~ s/$old_role$/$new_role/;
         }
         $data_width    = $width if $new_type =~ /data$/;
         $address_width = $width if $new_type =~ /_address$/;
         $has_irq       = 1      if $new_type =~ /_irq$/;
         $has_reg_sel   = 1      if $new_type =~ /_registeredselectn$/;
         $has_tri_state = 1      if $new_type =~ /_data$/;

         $new_type =~ s/_shared_off_chip_/_shared_/;

         if ($new_type ne $type)
         {
            print STDERR "$port_name: obsolete port type $type found.\n" if $DEBUG;
            print STDERR "   automatically replaced by new type $new_type.\n" if $DEBUG;
            $PTF_CHANGES++;

            &add_child_data ($db_Types, $port_name, $new_type) ;
         }
      }

      ################
      # Upgrade to new PORT-section style.
      #
      &PTF_Update_Port_Wiring_Section($db_Module);

      ################
      # Use "HDL_INFO/Module_Type" assignment to figure-out new "class".
      #
      # We use the "Module_Type" assignment.  Note that the old
      # "usersocket" megawizard allowed the user to type -anything-
      # into this field--so any Module_Type we don't recognize must 
      # be a user-socket.
      #
      my $class = &get_data_by_path ($db_Module, "class");
      if (!$class)
      {
         $class = $class_translation_from_1_0{$old_module_type};
         #JWIZ NAME CHANGE
         $class = "jusersocketwizard" if !$class;

         &add_child_data ($db_Module, "class", $class);
         print STDERR "Added 'class' assignment $class to obsolete MODULE\n" if $DEBUG;
         $PTF_CHANGES++;
      }

      ################ 
      # Add absolutely-required sections to PTF:
      #
      &PTF_Addit_And_Warnem($db_SBI, "Data_Width",    $data_width);
      &PTF_Addit_And_Warnem($db_SBI, "Address_Width", $address_width);
      &PTF_Addit_And_Warnem($db_SBI, "Is_Enabled",    1);
      &PTF_Addit_And_Warnem($db_SBI, "Is_Bus_Master", $is_master);
      &PTF_Addit_And_Warnem($db_SBI, "Date_Modified",  scalar(localtime));
      &PTF_Addit_And_Warnem($db_SBI, "Instantiate_In_System_Module", 1);

      if ((!$is_master)) { # Checks for slaves-only:
        &PTF_Addit_And_Warnem($db_SBI,
                              "Uses_Tri_State_Data_Bus", $has_tri_state);
        &PTF_Addit_And_Warnem($db_SBI,
                              "Uses_Registered_Select_Signal",  $has_reg_sel);
        &PTF_Addit_And_Warnem($db_SBI, "IRQ_Number",              "N/A");
        &PTF_Addit_And_Warnem($db_SBI, "Has_IRQ",                 $has_irq);
        &PTF_Addit_And_Warnem($db_SBI, "Address_Alignment",       "native");
      }

      ###############
      # Force halfword alignment to native.
      &PTF_Changeit_And_Warnem($db_SBI, 
                               "Address_Alignment",
                               'halfword',
                               "native");

      ################
      # Here are some module-specific updates:
      #
      if (($old_module_type eq "onchip_rom")) {
        # All old ROMs were 16 bits wide, so size = depth * 2
        my $size = &get_data_by_path ($db_WSA, "depth") * 2;
        my $file = &get_data_by_path  ($db_WSA, "contents");
        # All old rom-files were copied to the local directory.
        # use them there:
        $file =~ s/.*?([^\\\/]+)$/$1/;  # Strip-off leading path

        &PTF_Addit_And_Warnem ($db_WSA, "Writeable",          0  );
        &PTF_Addit_And_Warnem ($db_WSA, "Contents",        "file");
        &PTF_Addit_And_Warnem ($db_WSA, "Initfile",        $file );
        &PTF_Addit_And_Warnem ($db_SBI, "Is_Memory_Device",    1 );
        &PTF_Addit_And_Warnem ($db_SBI, "Address_Span",    $size );
      }
      if (($old_module_type eq "onchip_ram")) {
        # All old ROMs were 16 bits wide, so size = depth * 2
        my $size = &get_data_by_path ($db_WSA, "depth") * ($data_width/8);
        &PTF_Addit_And_Warnem ($db_WSA, "Writeable",           1);
        &PTF_Addit_And_Warnem ($db_WSA, "Contents",      "blank");
        &PTF_Addit_And_Warnem ($db_SBI, "Is_Memory_Device",   1 );
        &PTF_Addit_And_Warnem ($db_SBI, "Address_Span",    $size );
      }

      if (($old_module_type eq "uart")) {
        # Uarts are the only old "printable devices".
        &PTF_Addit_And_Warnem ($db_SBI, "Is_Printable_Device", 1 );
      }

      if (($class eq "jusersocketwizard")) {
        # Whatever ports are in this PTF-file, the user gets to keep.
        &PTF_Addit_And_Warnem ($db_WSA, "keep_legacy_ports",   1);
        &PTF_Addit_And_Warnem ($db_SBI, "Tri_State_Data_Bus",
                               "shared_off_chip") if $has_tri_state;
        &PTF_Addit_And_Warnem ($db_SBI, "Is_Memory_Device",    1 );
      }

      ################
      # And some other oddball system-level things:
      if ($freq = &get_data_by_path($db_WSA, "clock_freq")) {
        &PTF_Addit_And_Warnem ($db_Sys_WSA, "clock_freq", $freq);
      }

      if ($has_tri_state) {
        &PTF_Addit_And_Warnem ($db_Sys_WSA,
                               "Principal_Tri_State_Data_Bus",
                               "shared_off_chip");
      }
   }

}

sub PTF_Translate_Old_Version
{
   my ($ptf_filename) = (@_);

   $PTF_CHANGES = 0;

   my $db_PTF_File  = &PTF_New_Required_Ptf_From_File ($ptf_filename);
   my $db_Sys       = &PTF_Get_Required_Child_By_Path ($db_PTF_File, "SYSTEM");
   my $db_Sys_WSA   = &PTF_Get_Required_Child_By_Path ($db_Sys,
                                                    "WIZARD_SCRIPT_ARGUMENTS");

   ################
   # Decide if this is an "old" PTF at-all.
   # Because the version number was not in the PTF for release 1.0 of
   # the kit, we have to use an heuristic:  If the SYSTEM module
   # doesn't have a clock frequency, then it must be old.
   #
   &PTF_Translate_From_Version_1_0 ($ptf_filename, $db_PTF_File)
     if !&get_data_by_path ($db_Sys_WSA, "clock_freq");


   my $num_modules = &get_child_count ($db_Sys);
   for (my $module_index = 0; $module_index < $num_modules; $module_index++)
   {
      my $db_Module = &get_child ($db_Sys, $module_index);
      next if &get_name ($db_Module) ne "MODULE";   # ignore non-modules.

      my $class_name = &get_data_by_path($db_Module, "class");
      my $updated_name = $class_translation_from_jw{$class_name};
      next unless $updated_name;
      
      &add_child_data ($db_Module, "class", $updated_name);
      $PTF_CHANGES++;
    }

   if ($PTF_CHANGES != 0)
   {
      print STDERR "PTF_Translate_Old_Version:
        Made $PTF_CHANGES changes to obsolete PTF file $ptf_filename\n" if $DEBUG;
      &write_ptf_file ($db_PTF_File) or die "Couldn't write PTF File ";
    } else {
      # print STDERR "PTF_Translate_Old_Version:  No changes made.\n";
    }
}

################################################################
# PTF_Addit_And_Warnem
#
# Add a missing assignment to a (presumably-old) PTF-file 
# if it isn't already there.  If we have to add it, then  emit a
# scolding.
#
# Return "1" if we had to alter the PTF, "0" otherwise.
#
################################################################
sub PTF_Addit_And_Warnem
{
  my ($ptf_ref, $path, $value) = (@_);

  if (&get_data_by_path ($ptf_ref, $path) eq "") {
    &add_child_data($ptf_ref, $path, $value);
    my $name = &get_data ($ptf_ref);
    print STDERR "Added assignment ($path=$value) to old PTF file $name.\n" if $DEBUG;
    $PTF_CHANGES++;
    return 1;
  }

  return 0;

}

################################################################
# PTF_Changeit_And_Warnem
#
# Add a missing assignment to a (presumably-old) PTF-file 
# if it isn't already there.  If we have to add it, then  emit a
# scolding.
#
# Return "1" if we had to alter the PTF, "0" otherwise.
#
################################################################
sub PTF_Changeit_And_Warnem
{
  my ($ptf_ref, $path, $old_value, $new_value) = (@_);

  my $unchanged_old_value = $old_value;
  $old_value =~ s/(\W)/\\$1/g;

  if (&get_data_by_path ($ptf_ref, $path) =~ /$old_value/i) {
    &add_child_data($ptf_ref, $path, $new_value);
    my $name = &get_data ($ptf_ref);
    print STDERR "Changed assignment ($path=$unchanged_old_value) to $new_value.\n" if $DEBUG;
    $PTF_CHANGES++;
    return 1;
  }

  return 0;

}

################################################################
# PTF_Update_Boolean_Values
#
# Recursively loops through the PTF and all its children.
# If the value of any leaf-level assignment is "TRUE" or "FALSE"
# or "yEs" or "no" or some such, then we update to the new value:
# 1/0.
################################################################
sub PTF_Update_Boolean_Values
{
  my ($ptfRef) = (@_);

   my $num_corrections = 0;
   my $num_children = &get_child_count ($ptfRef);
   for (my $child_index = 0; $child_index < $num_children; $child_index++) {
      my $child_ptf = &get_child ($ptfRef, $child_index);

      if (&get_child_count($child_ptf) != 0) {
        $num_corrections += &PTF_Update_Boolean_Values ($child_ptf);
        next;
      }

      # We only want to correct boolean values in SBI and WSA sections
      # (better safe than sorry)

      my $section_type = &get_name ($ptfRef);
      next unless ($section_type eq "WIZARD_SCRIPT_ARGUMENTS") ||
                  ($section_type eq "SYSTEM_BUILDER_INFO"    )  ;

      my $child_name = &get_name ($child_ptf);
      my $value      = &get_data ($child_ptf);
      if (($value =~ /^TRUE$/i) ||
          ($value =~ /^yes$/i )  )
        {
          print STDERR "Replacing $child_name value '$value' with '1'\n" if $DEBUG;
          &add_child_data($ptfRef, $child_name, "1");
          $num_corrections++;
          next;
        }
      if (($value =~ /^FALSE$/i) ||
          ($value =~ /^no$/i   )  )
        {
          print STDERR "Replacing $child_name value '$value' with '0'\n" if $DEBUG;
          &add_child_data($ptfRef, $child_name, "0");
          $num_corrections++;
          next;
        }
   }
  return $num_corrections;
}

################################################################
# PTF_Update_Port_Wiring_Section
#
#  PORT_WIRING section: how it used to be.
#
# In the old days, all the WIDTHS of the ports were together in 
# a WIDTH section.  The assignment-names were the port-names, and 
# the values were the widths.  
#
# All the TYPES were together in the TYPES section.  The
# assignment-names were (again, and redundantly) the port-names, and
# the values were a twisted mutation of a little descriptor which
# contained a number of attributes like "avalon role" and "direction"
# embedded in it.
#
#  PORT_WIRING section:  how it is.
#
# Now each port has its own PORT section, where the assignment names are
# its attributes, and the values are the values.  Just like it should
# be.
#
# This function converts the old into the new, including parsing-apart
# the descriptor-string into useful data.  When it's done, it deletes
# the old WIDTH and TYPE sections, to avoid confusion.
#
################################################################
sub PTF_Update_Port_Wiring_Section
{
  my ($db_Module) = (@_);
  my $changes = 0;

  my $db_Port_Type  = &get_child_by_path($db_Module,"PORT_WIRING/TYPE");
  my $db_Port_Width = &get_child_by_path($db_Module,"PORT_WIRING/WIDTH");

  $db_Port_Type or
    die ("PTF_Update_Port_Wiring_Section: no 'TYPE' section.");
  $db_Port_Width or
    die ("PTF_Update_Port_Wiring_Section: no 'TYPE' section.");

  my $num_ports = &get_child_count ($db_Port_Type);
  for ($port_index = 0; $port_index < $num_ports; $port_index++)
  {
    my %port_hash = ();  # put all PORT's assignments into here.

    my $port_name      = &get_name (&get_child ($db_Port_Type, $port_index));
    $port_hash{width}  = &get_data_by_path ($db_Port_Width,   $port_name);
    my $old_type       = &get_data_by_path ($db_Port_Type,    $port_name);

    $old_type =~
           /(master|internal|external)_(input|output|inout)_?(shared_)?(.*)/
             or die "Update: Port $port_name has malformed type: $old_type";

    $port_hash{direction}   = $2;
    $port_hash{is_shared}   = $3 ne "";
    $port_hash{avalon_role} = $4;

    &delete_child($db_module, "PORT_WIRING/PORT $port_name");

    print "Adding PORT section: $port_name\n" if $DEBUG;
    foreach $assignment_name (keys(%port_hash)) {
      next if $port_hash{$assignment_name} eq "";

      my $value = $port_hash{$assignment_name};

      &add_child_data
        ($db_Module, "PORT_WIRING/PORT $port_name/$assignment_name",  $value);

    }
  }

  &delete_child ($db_Module, "PORT_WIRING/TYPE");
  &delete_child ($db_Module, "PORT_WIRING/WIDTH");
}

################################################################################
# Rename_Old_edf_File
# If leonardo was run on nios 1.0, there's a $ptf.edf
# lying around that will totally hose the user. 1.1 kit updates
# will not take effect in the synthesis design because quartus
# will find the .edf file before it finds the wrapper.  The
# solution is to rename the edf file to something safe and warn
# the user.

sub Rename_Old_edf_File
{
   my ($ptf_filename) = @_;

   my $malignent_edf_file = $ptf_filename;
   $malignent_edf_file =~ s/\.ptf$/\.edf/;

   return unless (-e $malignent_edf_file);
   my $benign_edf_file = "$malignent_edf_file\_old";
   while (-e "$benign_edf_file"){$benign_edf_file =
                                     "${benign_edf_file}_old";}
   print STDERR "Renaming $malignent_edf_file to $benign_edf_file\n" if $DEBUG;
   if(rename $malignent_edf_file,$benign_edf_file)
   {
      print STDERR "Renamed $malignent_edf_file to $benign_edf_file\n" if $DEBUG;
   }
   else
   {
      if ($DEBUG)
      {
         print STDERR "Rename Failed ($!).  Quartus will try to use\n";
         print STDERR "$malignent_edf_file instead of your design unless\n";
         print STDERR "it is specifically in the Quartus Project File list\n";
      }
   }
}

#################################################################
# Set_Avalon_Defaults
#
# Takes the port list from every module in db_Sys and sets default
# width and direction for each port that does not have it defined.
# this function should not be called in 1.1

sub Set_Avalon_Defaults
{
   die "Set_Avalon_Defaults you should not call this function\n";
   #do not use this function
   my ($db_Sys,
       $db_PTF_File) = @_;

   warn "in SAD\n";
   my $default_hash = &Get_Avalon_Requirement_Table;

   my @db_Module_Array = &get_first_children_of_type($db_Sys,
                                                     "MODULE");
   warn "dbma is @db_Module_Array\n";
   my $Mod;
   my %Sys_Hash;
   my $Sys = \%Sys_Hash;

   foreach $db (@db_Module_Array)
   {
      my $db_SBI = &PTF_Get_Required_Child_By_Path ($db,
                                                    "SYSTEM_BUILDER_INFO");
      $Mod = &PTF_Build_Hash_From_Section ($db_SBI);
      &PTF_Check_Bool ($Mod, "Is_Bus_Master",0);
      &PTF_Eval       ($Mod, "Address_Width"  );
      &PTF_Eval       ($Mod, "Data_Width"     );

      $$Mod{name} = &get_data ($db);
      $Sys_Hash{master_data_width} = $$Mod{Data_Width};
      last if $Mod->{Is_Bus_Master};
   }

   die "Set_Avalon_Defaults, No Master Found\n"
       unless ($$Mod{name});
   warn "SAD found master $$Mod{name}\n";

   #now we know which module is master
   #loop through again and get port data for each module
   my $default_direction;
   foreach $db_Mod (@db_Module_Array)
   {
      my $module_name = &get_data($db_Mod);
      print STDERR "port $module_name\n" if $DEBUG;

      $default_direction = $default_hash->{required_slave_dir};      
      $default_direction = $default_hash->{required_master_dir}
      if ($module_name eq $$Mod{name});

      my $db_Port_Wiring = &get_child_by_path($db_Mod,"PORT_WIRING",0) or die 
          "Set_Avalon_Defaults, No PORT_WIRING SECTION FOUND FOR\n"
              ."MODULE $module_name\n";

      my @db_Ports = &get_first_children_of_type($db_Port_Wiring,
                                                 "PORT") or die
          "Set_Avalon_Defaults, No PORTS FOUND in PORT_WIRING SECTION FOR\n"
              ."MODULE $module_name\n";

      foreach $db_Port (@db_Ports)
      {
         my $port_name = &get_data($db_Port);
         my $avalon_role = &get_data_by_path($db_Port,"avalon_role");
         if (!$avalon_role) 
         {print STDERR "  did not find avalon role for port $port_name\n" if $DEBUG;next}

         my $direction = &get_data_by_path($db_Port,"direction");
         if (!$direction) 
         {
            $direction = $default_direction->{$avalon_role};
            print STDERR "  did not find direction for port $port_name ($avalon_role)\n" if $DEBUG;
            print STDERR "  going with $direction\n" if $DEBUG;
            my $test = &get_child_by_path($db_Port,"direction",1);
            print STDERR "so here is direction $test\n" if $DEBUG;
            &add_child_data($db_Port,"direction",$direction);
            my $after_test = &get_data_by_path($db_Port,"direction");
            print STDERR "after assigning direction, $after_test\n" if $DEBUG;
         }
         #or
         #$default_direction->{$avalon_role};

         my $W = &get_data_by_path($db_Port,"width");# or 
             if (!$W)
             {
                eval ($default_hash->{width_default}{$avalon_role});
                print STDERR "  did not find direction for port $port_name ($avalon_role)\n" if $DEBUG;
                print STDERR "  going with $W\n" if $DEBUG;
                &add_child_data($db_Port,"width",$W);
                my $test = &get_child_by_path($db_Port,"width",1);
                print STDERR "so here is width $test\n" if $DEBUG;
                my $after_test = &get_data_by_path($db_Port,"width");
                print STDERR "after assigning width, $after_test\n" if $DEBUG;
             }                
      }
      
   }
   #&write_ptf_file ($db_PTF_File) or die
   #"ERROR Set_Avalon_Defaults, could not write ptf file ($!)\n";
}

1;    # Modules must say "1"--mustn't they?




