#Copyright (C)2001-2003 Altera Corporation
#Any megafunction design, and related net list (encrypted or decrypted),
#support information, device programming or simulation file, and any other
#associated documentation or information provided by Altera or a partner
#under Altera's Megafunction Partnership Program may be used only to
#program PLD devices (but not masked PLD devices) from Altera.  Any other
#use of such megafunction design, net list, support information, device
#programming or simulation file, or any other related documentation or
#information is prohibited for any other purpose, including, but not
#limited to modification, reverse engineering, de-compiling, or use with
#any other silicon devices, unless such use is explicitly licensed under
#a separate agreement with Altera or a megafunction partner.  Title to
#the intellectual property, including patents, copyrights, trademarks,
#trade secrets, or maskworks, embodied in any such megafunction design,
#net list, support information, device programming or simulation file, or
#any other related documentation or information provided by Altera or a
#megafunction partner, remains with Altera, the megafunction partner, or
#their respective licensors.  No other licenses, including any licenses
#needed under any third party's intellectual property, are provided herein.
#Copying or modifying any file, or portion thereof, to which this notice
#is attached violates this copyright.


# ex:set tabstop=4:
# ex:set expandtab:





# ---------------------------
# Each Section consists of a
# Type, a Name (which can be
# be blank), and an unordered
# list of Elements.
#
# Each Element can be either
# a Name and a Value, or another
# Section.
#
# An Element list is an associative
# array where the key is the name of
# a value, or the name of a module kind.
#
# For a value, the value at that key is just the value.
#
# FOr a module kind, the value is another associative
# array whose keys are the module names, and whose
# contents are each element lists.
#
# PTFHash ->
#	{
#	name => name of assignment, or kind of section if section, or name of file if outer
#	data => value of assignment, or name of section if section
#	kind => "section" or "file" (or nothing means "assignment")
#	section => reference to an array of PTFHash if section
#	}
#

package ptf_parse;
use strict;
use Exporter;

@ptf_parse::ISA = ("Exporter");
@ptf_parse::EXPORT = qw(
			new_ptf_from_file
			new_ptf_from_string
			new_ptf_file
			new_ptf
			get_child_count
			get_data
			get_name
			set_data
			set_data_by_path
			set_name
			get_child
			get_child_by_name
			get_child_by_path
			get_first_children_of_type
			delete_child
			get_data_by_path
			add_child_data
			add_child
			ptf_indent
			write_ptf_file
			get_number_by_path
			ptf_parse_args
			ptf_get_switch
			dateTime
			readFile
			writeFile
			copyFile
			ptf_PrintRef
             		ptf_print_status
			pp_ptf_to_string
		   enable_file_header
	     		);

# NOT exported:
#			childrenToString
#			indentSprint
#			aNumber
#			copyDirContents
#			killDirectory


my $g_mg_comment_prefix_const = ":-:";	# comments labeled #//# are generated by this library
my $g_mg_comment_prefix = $g_mg_comment_prefix_const;




# -----------------------------
# dequote(x)
# 
# strip off endquotes, if thats what they are
#

sub dequote
	{
	my ($x) = (@_);
	my $h;
	my $t;

	$h = substr($x,0,1);
	$t = substr($x,-1,1);

	if($h eq $t and
			($h eq "'" or $h eq '"'))
		{
		$x = substr($x,1,length($x) - 2);
		}
	
	return $x;
	}

# +---------------------------
# | string hideEscapees(escaped_string)
# | string revealEscapees(hidden_string)
# | string reescapeEscapees(hidden_string)
# |
# | There are three states a string might be in:
# |  Escaped, Hidden, and Revealed.
# |
# | Escaped means the special characters are in place.
# | Hidden means we have replaced them with boring-but-wrong strings
# | Revealed means we have no escape chars, just the data
# |
# | Examples:
# |
# | Escaped:  I said, \"I like this code.\"
# | Hidden:   I said, __dquot_12345__I like this code.__dquot_12345__
# | Revealed: I said, "I like this code."
# |
# | When users of this library access ptf entries, they
# | should get the "Revealed" form.
# |
# | When we read a PTF file, we expect "Escaped" form, and when
# | we write out a PTF file, we emit it again in "Escaped" form.
# |
# | Nobody should ever see the "Hidden" form, except a routine
# | or two in this library.
# |

my $dUnique = time;
my $dEscapePairs =
    [
    '\\\\\\" --> __dquote_' . $dUnique . '__ --> " --> \"',
    '\\\\\\\\ --> __bslash_' . $dUnique . '__ --> \\ --> \\\\'
    ];

sub hideOrRevealEscapees($$)
#
# operations: 0, escaped -> hidden
#             1, hidden -> revealed
#             2, hidden -> escaped
    {
    my ($s,$operation) = (@_);

    foreach my $e (@$dEscapePairs)
        {
        my ($escaped,$hidden,$revealed,$reescaped) = split(/ --> /,$e);
        if($operation == 0)
            {
            $s =~ s/$escaped/$hidden/gs;
            }
        elsif($operation == 1)
            {
            $s =~ s/$hidden/$revealed/gs;
            }
        else
            {
            $s =~ s/$hidden/$reescaped/gs;
            }
        }

    return $s;
    }

sub hideEscapees($)
    {
    return hideOrRevealEscapees(shift,0);
    }

sub revealEscapees($)
    {
    return hideOrRevealEscapees(shift,1);
    }

sub escapeEscapees($)
    {
    return hideOrRevealEscapees(shift,2);
    }



# ----------------------------
# arrayRef ptf_ReadSectionElements(string)
#
# returns a reference to
# an array of elements.

sub ptf_ReadSectionElements($)
    {
    my $data = shift;
    my @elementList;
    
    while($data)
        {
        # | pattern to match left _or_ right side of assignment
        # | allowing quoted & unquoted bits,
        # | assuming whitespace ok in pattern
        my $as_na =
            '\"[^\"]*?\"' .      # dquot, anything-but, and a dquot.
            "|" .                #    OR
            "\\'[^\\\']*?\\\'" . # squot, anything-but, and a squot
            "|" .                #    OR
            '[^\"\{\}\=\;]*?';   # anything but a curlybrace, quot, eq, or semi

        #
        # Find the next thing, which is either
        # a name="value"; pair, or a SECTION name {} block.
        # or a comment
        #

        if($data =~
                /^        # beginning of string
                \s*        # misc white space
                \#        # comment character
                /sx)
            {
            my %newElement;
            my $comment;

            #
            # Looks like a freestanding comment, try to find it.
            #

            ($comment,$data) = pull_leading_comments($data);
            $newElement{comment} = $comment if $comment ne "";
            push(@elementList,\%newElement);
            }

        elsif($data =~
                /^        # beginning of string
                \s*        # any leading whitespace
                (${as_na})    # the name
                \s*        # intervening white space
                \=        # the equal sign
                \s*        # more intervening white space
                (${as_na})    # the value (including quotes maybe!)
                \s*        # there could be white space here
                ;        # terminating semi-colon
                (.*)        # everything else
                /sx)
            {
            #
            # It's a name=value, each side possibly quoted
            #
                my $elementName;
                my $elementData;
                my $elementComment;
                my %newElement;

                $data = $3;
                $elementName = $1;
                $elementData = $2;

                $elementName = dequote($elementName);
                $elementData = dequote($elementData);

                ($elementComment,$data) = pull_leading_comments($data);

                $newElement{comment} = $elementComment if $elementComment ne "";
                $newElement{name} = $elementName;
                $newElement{data} = $elementData;

                push(@elementList,\%newElement);
            }

        elsif($data =~
                /^        # beginning of string
                ([^;\{]*)    # the element, up to the next ; or left curly
                \{        # section beginning
                (.*)        # and the whole rest of the string.
                $/sx)
            {
            #
            # It's a section, since it ends with a left-curly
            #

            my $sectionKind;
            my $sectionName;
            my $sectionComment;    # comment on same line, if any
            my $sectionKindName;
            my $sectionContents;

            $data = $2;
            $sectionKindName = $1;

            #
            # At this point, sectionKindName may have some number of
            # comments between the section kind and name, and the left curly
            # so we glom them all into "sectionComment".
            #

            if($sectionKindName =~
                    /^        # beginning of string
                    ([^\#]*)    # something that's not a comment
                    (\#.*)        # the comment
                    $/sx)        # no more
                {
                ($sectionComment,$sectionKindName) = pull_leading_comments($2);
                $sectionKindName = $1 . $sectionKindName;
                }

            if($sectionKindName =~ /^\s*(.*?)\s*$/s)
                {
                $sectionKind = $1;
                $sectionName = "";

                #
                # See if the section has a name or no...
                #

                if($sectionKind =~ /^\s*(\S*?)\s+(\S*?)\s*$/)
                    {
                    $sectionKind = $1;
                    $sectionName = $2;
                    }

                $sectionKindName = $sectionKind . " " . $sectionName;

                #
                # Find matching curlybrace ending
                #
                    {
                    my $braceDepth = 1;
                    my $dataLen = length($data);
                    my $i;
                    my $ch;
                    my %newElement;

                    $i = 0;
                    while ($i < $dataLen and $braceDepth > 0)
                        {
                        $ch = substr($data,$i,1);
                        $braceDepth++ if($ch eq "{");
                        $braceDepth-- if($ch eq "}");
                        $i++;
                        }

                    # Bust in half, neither half gets the last right curlybrace 
                    $sectionContents = substr($data,0,$i-1);
                    $data = substr($data,$i);


                    #
                    # Recursively descend, and get the tree
                    # for this section, and add it to the list.
                    #

                    $newElement{name} = $sectionKind;
                    $newElement{data} = $sectionName;
                    $newElement{comment} = $sectionComment if $sectionComment ne "";
                    $newElement{section} = ptf_ReadSectionElements($sectionContents);

                    push(@elementList,\%newElement);
                    }
                }
            }
        else
            {
            $data = "";
            }
        }

    #
    # Return reference to this whole tree.
    #

    return \@elementList;
    }






# ----------------------------
# Pull Leading Comments
#
# Return a \n delimited list of any comments
# which adorn the beginning of the string passed in.
# return a list of two items: (comments, string-without-comments)
#

sub pull_leading_comments
    {
	my $instring = shift;
	my $comments_out;


	while($instring =~
			/^			# beginning
			[\s^\n]*\#[ \t]*	# whitespace before & after comment character
			([^\n]*?)		# the comment, up to the newline
			\s*\n\s*		# whitespace and newline and whitespace (gone!)
			(.*)			# the rest of everything
			$/sx)
		{
		my $this_comment = $1;
		my $rest = $2;

		$instring = $rest;

		next if ($this_comment =~ /^$g_mg_comment_prefix/);

		$comments_out .= "# " . $this_comment . "\n";
		}

	return ($comments_out,$instring);
	}

# ----------------------------
# ptf new_ptf_from_file
#
# returns a ptf hashref

sub new_ptf_from_file
	{
	my $fileName = shift;
	my $fileContents;

	$fileContents = readFile($fileName);
	return (&new_ptf_from_string($fileName,$fileContents));
	}

# ----------------------------
# ptf new_ptf_from_string
#
# returns a ptf hashref

sub new_ptf_from_string
	{
	my $name = shift;
	my $string = shift;
	my %ptfHash;

	return "" if($string eq "");

    #
    # Un-escape any supported escaped characters
    #

    $string = hideEscapees($string);

	#
	# Populate top level of the ptf-ref
	#

	$ptfHash{name} = $name;
	$ptfHash{kind} = "file";
	$ptfHash{section} = ptf_ReadSectionElements($string);

	return \%ptfHash;
	}

# -----------------------------
# ptf new_ptf_file(fileName)
#
# returns a ptf hashref

sub new_ptf_file
	{
	my $fileName = shift;
	my %ptfHash;

	$ptfHash{name} = $fileName;
	$ptfHash{kind} = "file";

	return \%ptfHash;
	}

# -----------------------------
# ptf new_ptf(name,data)
#
# returns a ptf hashref with the
# new assignment ptf. (It becomes
# a "section" if you add any children
# to it.)

sub new_ptf
	{
	my %ptfHash;
	my $name = shift;
	my $data = shift;

	$ptfHash{name} = $name;
	$ptfHash{data} = $data;
	return \%ptfHash;
	}

# -----------------------------
# get_child_count(ptf_ref *ref [,child_name])
#
# if child_name is provided, count up only
# those children that match. For example,
# get_child_count(ptf_ref,"MODULE")
#
sub get_child_count
	{
	my ($ptfRef,$child_name) = @_;

    return 0 if ref($ptfRef) ne "HASH"; # wrong type? return 0, safely.

	my $childCount = 0;
	my $sectionRef = $$ptfRef{section};

	if($child_name eq "")
		{
		# |
		# | main case -- count all the chillun
		# |

		$childCount = $#$sectionRef + 1;
		}
	else
		{
		# |
		# | More complicated case: count only those
		# | chillun whose name matches the 2nd argumen
		# |

		my $i;
		my $child_ref;

		for($i = 0; $i <= $#$sectionRef; $i++)
			{
			$child_ref = $$sectionRef[$i];
			$childCount++ if($$child_ref{name} eq $child_name);
			}
		}
	
	return $childCount;
	}

sub get_data
	{
	my $ptfRef = shift;

	return revealEscapees($$ptfRef{data});
	}

sub get_name
	{
	my $ptfRef = shift;

	return $$ptfRef{name};
	}

sub set_data
	{
	my $ptfRef = shift;
	my $new_data = shift;

	$$ptfRef{data} = $new_data;
	return;
	}

sub set_data_by_path
	{
	my ($ptf_ref,$path,$data) = (@_);
	my $child_ref;

	$child_ref = get_child_by_path($ptf_ref,$path,1);
	set_data($child_ref,$data);
	}

sub set_name
	{
	my $ptfRef = shift;
	my $new_name = shift;

	$$ptfRef{name} = $new_name;
	return;
	}

# -----------------------------
# get_child(ptf_ref *ref,int index [,child_name])
#
# get the n'th child. if the optional
# child_name is provided, get the n'th one
# with that name. Like, get_child(3,"MODULE).
#

sub get_child($$;$)
	{
	my ($ptfRef,$index,$child_name) = @_;
	my $key;
	my $sectionRef;

	$sectionRef = $$ptfRef{section};

	if($child_name eq "")
		{
		return $$sectionRef[$index];
		}
	else
		{
		# |
		# | More complicated case, return the n'th
		# | with the matching child_name
		# |

		my $i;
		my $child_ref;

		for($i = 0; $i <= $#$sectionRef; $i++)
			{
			$child_ref = $$sectionRef[$i];
			if($$child_ref{name} eq $child_name)
				{
				return $child_ref if($index-- == 0);
				}
			}
		}
	}

# --------------------------------
# get_child_by_name(ptfRef, childName, [returnIndexToo])
#
# childName = "*": return first child
# childName = "fish": return child named "fish"
# childName = "fish blue": return child section type "fish", name "blue"
#

sub get_child_by_name($$$)
	{
	my $ptfRef = shift;
	my $name = shift;
	my $returnIndexToo = shift;

	my $sectionRef;
	my $childRef;

	my $nameFront;
	my $nameBack;
	my $i;

    return undef if !$ptfRef;  # null children for null parents

	($nameFront,$nameBack) = split(/ /,$name,2);

	if($$ptfRef{section})
		{
		$sectionRef = $$ptfRef{section};
		}
	else
		{
		return "";
		}

	for($i = 0; $i <= $#$sectionRef; $i++)
		{
		$childRef = $$sectionRef[$i];

		# easy case
		last if ($name eq "*");

		# harder case for named section
		last if ($nameFront eq $$childRef{name}
				and ($nameBack eq "" or $nameBack eq $$childRef{data}));

		$childRef = "";
		}
	return ($childRef,$i) if $returnIndexToo;
	return $childRef;
	}


# --------------------------------
# sub get_child_by_path(ptfRef, path [, buildIfAbsent[, returnParentToo])
#
# path is slash-separated of names as above
#
sub get_child_by_path($$;$$)
	{
	my $ptfRef = shift;
	my $path = shift;
	my $buildIfAbsent = shift;
	my $returnParentToo = shift;	# if true, return 2 element list

	my $pathFront;
	my $pathBack;
	my $childRef;
	my $parentRef;
	my $index;

	# In case path is null, or nearly so
	$childRef = $ptfRef;

	while($path ne "" and $path ne "/")	# paths dont really have a
                                        # leading slash... but.
		{
		($pathFront,$pathBack) = split(/\//,$path,2);
		($childRef,$index) = get_child_by_name($ptfRef,$pathFront,1);
		if($childRef eq "" and $buildIfAbsent)
			{
			my %child;
			my $sectionRef;

			# Build new child, then take its reference

				{
				my ($childName,$childData) = split(/ /,$pathFront,2);

				$child{name} = $childName;
				$child{data} = $childData;
				$childRef = \%child;
				}

			if(not $$ptfRef{section})
				{
				$$ptfRef{section} = [];
				}
			$sectionRef = $$ptfRef{section};
			push(@$sectionRef,$childRef);
			}
		return "" if($childRef eq "");

		$path = $pathBack;
		$parentRef = $ptfRef;
		$ptfRef = $childRef;
		}
	return ($parentRef,$childRef,$index) if $returnParentToo;
	return $childRef;
	}

# --------------------------------
# sub get_first_children_of_type(ptfRef, type)
#
# returns an array of all ptfRef's first children
# that have type ($type);

sub get_first_children_of_type
{
   my ($db_parent,$type) = @_;
   my $num_children = &get_child_count ($db_parent);   

   my @child_array;

   for (my $child_index = 0; 
        $child_index < $num_children; 
        $child_index++)
   {
      my $db_child = &get_child ($db_parent,$child_index);
      next if &get_name ($db_child) ne "$type";
      push (@child_array,$db_child);
   }
   return (@child_array);
}

# +--------------------------------
# | sub delete_child(ptfRef, path)
# |
# | path is slash-separated of names as above
# |


# | ascend a path, and delete any dead-hull sections in it

sub delete_empty_parts_of_path($$)
	{
	my ($ptf_ref,$path) = (@_);

	while($path)
		{

		my ($parent_ref,$child_ref,$index) = get_child_by_path($ptf_ref,$path,0,1);

		# |
		# | path -> nonempty section? delete it
		# |

		if($parent_ref and $$child_ref{section})
			{
			my $child_section_array_ref = $$child_ref{section};
			my $parent_section_array_ref = $$parent_ref{section};


			if($child_section_array_ref and $#$child_section_array_ref == -1)
				{
				splice(@$parent_section_array_ref,$index,1);
				}
			}

		if($path =~ /^(.*)\/([^\/]+)$/)
			{
			$path = $1;
			}
		else
			{
			$path = "";
			}
		}
	}

sub delete_child
	{
	my $ptfRef = shift;
	my $path = shift;
	my $childRef;
	my $parentRef;
	my $index;
	my $sectionArrayRef;

	($parentRef,$childRef,$index) = get_child_by_path($ptfRef,$path,0,1); # extra flag to return parent, too

	$sectionArrayRef = $$parentRef{section} if $parentRef ne "";

	splice(@$sectionArrayRef,$index,1);

	delete_empty_parts_of_path($ptfRef,$path);

	return "";
	}

# --------------------------------
# sub get_data_by_path(ptfRef, path [,default_value])
#
# path is slash-separated of names as above
#
sub get_data_by_path
	{
	my $ptfRef = shift;
	my $path = shift;
	my $default_value = shift;
	my $childRef;
	my $result = "";

	$childRef = get_child_by_path($ptfRef,$path);
	if($childRef)
		{
		$result = $$childRef{data};
		}
	else
		{
		$result = $default_value;
		}

	return revealEscapees($result);
	}

# -----------------------------------
# sub add_child_data(ptfRef, path, data)
#
sub add_child_data
	{
	my $ptfRef = shift;
	my $path = shift;
	my $data = shift;
	my $childRef;

	$childRef = get_child_by_path($ptfRef,$path,1);	# build if absent
	$$childRef{data} = $data;

	return;
	}

# -----------------------------------
# sub add_child(ptfRef, path, childPTFRef)
#
sub add_child
	{
	my $ptfRef = shift;
	my $path = shift;
	my $childPTFRef = shift;

	my $childRef;		# reference down in the path
	my $sectionRef;

	$childRef = get_child_by_path($ptfRef,$path,1);	# build if absent

	$$childRef{section} = [] if(not $$childRef{section});
	$sectionRef = $$childRef{section};
	push(@$sectionRef,$childPTFRef);

	return;
	}

# -----------------------------------
# pp_ptf_to_string(ptfRef[,indentLevel])
#
# (helper routine: sectionToString(ptfRef,indentLevel)
#

sub pp_ptf_to_string
	{
	my $ptfRef = shift;
	my $indentLevel = shift;
	my $gmg = $g_mg_comment_prefix;
	my $comment;

	my $result = "";

	$comment = $$ptfRef{comment};
	$comment = " " . $comment if $comment ne "";

	if($$ptfRef{kind} eq "file"
			and $indentLevel eq "")
		{
      if ($gmg ne "")
         {
		   $result .= "#" . $gmg . "\n";
		   $result .= "#" . $gmg . " file: " . $$ptfRef{name} . "\n";
		   $result .= "#" . $gmg . " date: " . dateTime() . "\n";
		   $result .= "#" . $gmg . " generated by a perl script\n";
		   $result .= "#" . $gmg . " ex:set tabstop=3:\n";
		   $result .= "#" . $gmg . " ex:set shiftwidth=3:\n";
		   $result .= "#" . $gmg . " ex:set expandtab:\n";
		   $result .= "#" . $gmg . "\n";
         }
		$result .= childrenToString($ptfRef,$indentLevel);
		}
	elsif($$ptfRef{section})
		{
		$indentLevel += 3;
		$result .= indentSprint($indentLevel,$$ptfRef{name}," ",$$ptfRef{data},$comment);
		$result .= indentSprint($indentLevel,"{");
		$result .= childrenToString($ptfRef,$indentLevel);
		$result .= indentSprint($indentLevel,"}");
		}
	elsif($$ptfRef{name} ne "")
		{
		$indentLevel += 3;
		$result .= indentSprint($indentLevel,$$ptfRef{name}," = \"",$$ptfRef{data},"\";",$comment);
		}
	elsif($$ptfRef{comment} ne "")
		{
		#
		# oops. remove the leading space
		#
		$comment = substr($comment,1);

		$result .= $$ptfRef{comment};
		my $c;
		}

	return $result;
	}

sub childrenToString
	{
	my $ptfRef = shift;
	my $indentLevel = shift;
	my $result = "";

	my $childCount;
	my $i;
	my $childRef;

	$childCount = get_child_count($ptfRef);
	for($i = 0; $i < $childCount; $i++)
		{
		$childRef = get_child($ptfRef,$i);
		$result .= pp_ptf_to_string($childRef,$indentLevel);
		}
	return $result;
	}

sub indentSprint
	{
	my $indentLevel = shift;
	my $a;
	my $result;
	
	$result = " " x $indentLevel;

	for($a = 0; $a < 15; $a++)
		{
		$result .= shift;
		}
	
	$result .= "\n" if (substr($result,-1) ne "\n");
	return $result;
	}

# -----------------------------------


sub ptf_indent($)
	{
	my $a = shift;
	my $result;
	
	$result = sprintf("[%02d]",$a);
	$result .= "'  " x $a;
	return $result;
	}

# ----------------------------------
# write_ptf_file(ptfRef[,fileName)
#
# Write the ptf to a file, optionally to a new filename
# Return "ok" if success, "" if fail.
#
sub write_ptf_file($;$)
	{
	my $ptfRef = shift;
	my $fileName = shift;
	my $ptfString;
	my $result;

	$fileName = get_name($ptfRef) if !$fileName;

	$ptfString = pp_ptf_to_string($ptfRef);

    # |
    # | Re-escape anything we did hide on import...
    # |

    $ptfString = escapeEscapees($ptfString);

	$result = writeFile($fileName,$ptfString);

	return $result;
	}

# --------------------------
# ptf_PrintRef(a reference)
#
# Print an indented view of the reference
#
# This is only useful as a debugging aid
#

sub ptf_PrintRef($;$$)
	{
	my $theRef = shift;
	my $indent = shift;
	my $visited_list_ref = shift;
	my @visited_list;


	# |
	# | visisted_list_ref is an array of references that
	# | we've visited, depthwise, to get here. We ignore
	# | the current ref if it's already on the list, to
	# | prevent circular reporting
	# |

	@visited_list = @$visited_list_ref if $visited_list_ref;
	my $i;
	for($i = 0; $i <= $#visited_list; $i++)
		{
		if($visited_list[$i] eq $theRef)
			{
			print ptf_indent($indent),"--circular reference--\n";
			return;
			}
		}
	push(@visited_list,$theRef);



	my $refType = ref $theRef;

	if(!$refType)
		{
		my $eL;
		my $done_first_line = 0;

		foreach $eL (split(/\n/,$theRef))
			{
			print ptf_indent($indent)," (more...) " if $done_first_line++;
			print $eL,"\n";
			}
		return;
		}

	if($refType eq "ARRAY")
		{
		my $arraySize = scalar @$theRef;
		my $i;
		my $value;

		print "(ARRAY[$arraySize])\n";
		$indent++;

		for($i = 0; $i < $arraySize; $i++)
			{
			$value = $$theRef[$i];
			if(defined($value))
				{
				print ptf_indent($indent),"[$i]: ";
				ptf_PrintRef($value,$indent,\@visited_list);
				}
			}
		return;
		}

	if($refType eq "HASH" || 1)  # default case, presume its an object/hash
		{
		my $key;
		my $value;

		print "($refType)\n";
		$indent++;

		foreach $key (sort(keys(%$theRef)))
			{
			print ptf_indent($indent),$key," = ";
			$value = $$theRef{$key};
			ptf_PrintRef($value,$indent,\@visited_list);
			}
		return;
		}
	}




# ---------------------------
# aNumber(hex-or-decimal)
#
sub aNumber($)
	{
	my $x = shift;
    my $sign = 1.0;

    # a single leading +'s or -'s mean sign
    if($x =~ /^([-+])(.*)$/)
        {
        my $sign_char = $1;
        $x = $2;

        $sign = ($sign_char eq "+") ? +1.0 : -1.0;
        }

	return 1 if($x =~ /true/i or $x =~/yes/i);

	$x = hex($x) if ($x =~ /^0+x/);

    # | multiply in the sign-value
    # | (this also, handily, forces $x to be a number)

    $x = $x * $sign; 
	return $x;
	}

# --------------------------
# get_number_by_path(ptfRef, path, [default_value])
#
# convenience: extract child, convert to value
#
sub get_number_by_path
	{
	my $ptfRef = shift;
	my $path = shift;
	my $default_value = shift;
	my $x;

	$x = get_data_by_path($ptfRef,$path,$default_value);
	# if(gDebug)
	# 	{
	# 	print_warning("$path is empty");
	# 	}
	$x = aNumber($x);
	return $x;
	}




# ----------------------
# dateTime()
#
# returns a relatively nice date & time string
#
sub dateTime
	{
	my ($sec,$min,$hour,$mday,$mon,$year,$wday,$yday,$isdet) = localtime(time);
	$mon++;
	$year += 1900;

	my $d = sprintf("%04d.%02d.%02d",$year,$mon,$mday);
	my $t = sprintf("%02d:%02d:%02d",$hour,$min,$sec);

	return "$d $t";
	}

# --------------------------
# readFile(fileName)
#
#  returns the complete file contents
#
sub readFile
	{
	my $fileName = shift;
	my $bunch;
	my $result;
	my $did;

	if(open(FILE,$fileName))
		{
		binmode FILE;			# Bite me, Windows! --dvb
		while(read(FILE,$bunch,32000))
			{
			$result .= $bunch;
			}
		close FILE;
		}
	
	return $result;
	}


# -----------------------
# writeFile(fileName,contents)
#
#  creates new file and writes entire
#  file contents. Return "ok" if so,
#  or "" if not.
#
sub writeFile
	{
	my $fileName = shift;
	my $contents = shift;
	my $did;

	#
	# Delete existing file, if any.
	#
	unlink ($fileName) if(-e $fileName);

	$did = open(FILE,">$fileName");
	if($did)
		{
		binmode FILE;			# Bite me, Windows! --dvb
		print FILE $contents;
		close FILE;
		return "ok";
		}
	
	return "";
	}

# ----------------------------
# copyFile(sourceFile,destFile)
#
sub copyFile
	{
	my $sourceFile = shift;
	my $destFile = shift;
	my $fileContents;
	my $result;

	$fileContents = readFile($sourceFile);
	return if !$fileContents;

	$result = writeFile($destFile,$fileContents);

	return $result;
	}

# ----------------------------
# copyDirContents(sourceDir,destDir)
#
# If there's any files in sourceDir,
# then ensure that destDir exists, and
# copy the files over, but don't bother
# with ones which are already there.
#

sub copyDirContents
	{
	my $sourceDir = shift;
	my $destDir = shift;

	my @sourceDir;
	my @destDir;
	my $file;

	opendir (DIR,$sourceDir) or return;
	@sourceDir = readdir(DIR);
	closedir (DIR);

	opendir (DIR,$destDir) or return;
	@destDir = readdir(DIR);
	closedir (DIR);

	foreach $file (@sourceDir)
		{
		#
		# Skip files starting with . or underscore
		#

		if( (!($file =~ /^[\._]/)) && ($file ne "vssver.scc") )
			{
			# (allow overwriting) if( (scalar grep (/^${file}$/,@destDir) ) == 0 )
				{
				mkdir($destDir,511);
				copyFile("${sourceDir}/${file}","${destDir}/${file}");
				}
			}
		}
	
	return "ok";
	}

# -------------------------
# killDirectory
sub killDirectory
	{
	return;
	my $dir = shift;
	my @dirContents;
	my $file;
	my $fullFile;

	return if (! (-e $dir));


	opendir (DIR,$dir) or return;
	@dirContents = readdir (DIR);
	closedir (DIR);

	foreach $file (@dirContents)
		{
		next if($file eq "." or $file eq "..");
		$fullFile = "$dir/$file";

		if( -d $fullFile)
			{
			killDirectory($fullFile);
			}
		else
			{
			unlink($fullFile);
			}
		}
	rmdir($dir);
	}


# -------------------------------------
# ptf_parse_args
#
# Given a list of arguments, return
# a hash where the keys and values
# are taken from those arguments of
# the form "--key=value". The hyphens
# disappear from the key name.
#
# A command line switch of "--key"
# is equivalent to "--key=1".
#
# a special key named _argc contains
# a count of non-dash-dash arguments,
# and they are in the hash as {0}, {1},
# and so on.
#
# (This is not especially unique to the ptf parsing
# functions, but is pretty handy.)

sub ptf_parse_args
	{
	my $arg;
	my $argVal;
	my $argc;
	my %hash;

	$argc = 0;


	while($arg = shift)
		{
		if($arg =~ /^--/)
			{
			if($arg =~ /^--(.*)\=(.*)$/)
				{
				$arg = $1;
				$argVal = $2;
			    $hash{$arg} = $argVal;
				}
			else
				{
                # bug -- we didnt strip off the two leading dashes!
				$argVal = 1;
			    $hash{$arg} = $argVal;

                # strip off the two leading dashes and add.
                # Leave the old one there too, for compat.
                $arg = substr($arg,2);
			    $hash{$arg} = $argVal;
				}

			}
		else
			{
            # add as numeric argument
			$hash{$argc++} = $arg;
			}
		}
	
	$hash{_argc} = $argc;

	return %hash;
	}

# -------------------------------
# ptf_get_switch(hashRef, switchName, defaultValue [, mustBeNumber])
#
# Look at a hash as returned by parseArgs, and
# give the value of the switch, or the defaultValue
# if it was not specified in the command line.

sub ptf_get_switch
	{
	my $hashRef = shift;
	my $switchName = shift;
	my $defaultValue = shift;
	my $mustBeNumber = shift;

	my $switchValue;

	$switchValue = $$hashRef{$switchName};
	$switchValue = $defaultValue if ($switchValue eq "");
	$switchValue *= 1 if ($mustBeNumber);

	return $switchValue;
	}


sub enable_file_header
   {
      my $enable = shift;
      $g_mg_comment_prefix = $enable ? $g_mg_comment_prefix_const : "";
   }

# -------------------------
# pretty printing utility
#
# return a nice date and time string
#

	sub dateTime
		{
		my ($sec,$min,$hour,$mday,$mon,$year,$wday,$yday,$isdet) = localtime(time);
		$year += 1900;
		$mon++;

		my $d = sprintf("%04d.%02d.%02d",$year,$mon,$mday);
		my $t = sprintf("%02d:%02d:%02d",$hour,$min,$sec);

		return "$d $t";
		}

	sub ptf_print_status
		{
		my $dt = dateTime();

		print "# $dt (*) ",@_,"\n";
		}

# --------------------------------------------------------------------
# All routines go above this line.

return 1;	# modules just do this

# End Of File
