#Copyright (C)2001-2002 Altera Corporation
#Any megafunction design, and related net list (encrypted or decrypted),
#support information, device programming or simulation file, and any other
#associated documentation or information provided by Altera or a partner
#under Altera's Megafunction Partnership Program may be used only to
#program PLD devices (but not masked PLD devices) from Altera.  Any other
#use of such megafunction design, net list, support information, device
#programming or simulation file, or any other related documentation or
#information is prohibited for any other purpose, including, but not
#limited to modification, reverse engineering, de-compiling, or use with
#any other silicon devices, unless such use is explicitly licensed under
#a separate agreement with Altera or a megafunction partner.  Title to
#the intellectual property, including patents, copyrights, trademarks,
#trade secrets, or maskworks, embodied in any such megafunction design,
#net list, support information, device programming or simulation file, or
#any other related documentation or information provided by Altera or a
#megafunction partner, remains with Altera, the megafunction partner, or
#their respective licensors.  No other licenses, including any licenses
#needed under any third party's intellectual property, are provided herein.
#Copying or modifying any file, or portion thereof, to which this notice
#is attached violates this copyright.







package europa_utils;
require Exporter;
@ISA = Exporter;
@EXPORT = qw(dwarn dprint ribbit goldfish history e_signal_list log2
             Bits_To_Encode is_blessed copy_of_hash floor ceil max min
             copy_hash_array_or_scalar
             concatenate validate_parameter
             caller_subroutine_list
             strip_enclosing_parentheses
             or_array and_array complement complement_null_ok
             unit_prefix_to_num
             num_to_bin is_power_of_two 
             next_lower_power_of_two next_higher_power_of_two
             round_up_to_next_computer_acceptable_bit_width
             is_computer_acceptable_bit_width
             europa_indent
             System_Win98_Safe
             master_address_width_from_slave_parameters
             Strip_Perl_Comments
             hash_ref_to_parameters_signature
             str2hex str2bin 
             package_setup_fields_and_pointers
             get_mask_of_1_bits
             one_hot_encoding
             spaceless_to_spaced_hash
             to_base_26
             
             get_class_ptf
             find_component_dir
             find_all_component_dirs
             );
use strict;
use print_command;

# Some associated handy modules that aren't europa-specific, but which
# have handy utility functions:


$| = 1;   # Flush STDERR always.


sub indent
{
   my $i = 0;

   while (caller ($i++)){;}

   return (" " x $i);
}

# Exported version.
sub europa_indent {   return &indent(@_); }


sub dwarn
{
   my $msg = join ("$,",@_); #join with special output
                             #separation variable
   my $indent = &indent();

   $msg =~ s/^/$indent/mg;   #every newline gets indented
   warn ($msg);
}

sub dprint
{
   my $msg = join ("$,",@_); #join with special output
                             #separation variable
   my $indent = &indent();

   $msg =~ s/^/$indent/mg;   #every newline gets indented.

   print ($msg);
}

sub fishing_report
{
   my (@msg_array) = (@_);

   # Avoid a warning if $, (global output-field-separator) is not yet defined.
   $, = "" unless defined($,);

   my $n = scalar (@msg_array);
   
   
   my $msg = join ("$,",@msg_array); #join with special output
                                     #separation variable

   $msg =~ s/^\s*(.*?)\s*$/$1/s;
   $msg .= "\n" if $msg =~ /\n$/s;  # Sugar: Prettier multi-line messages.

   my $string_to_figure_out_build = '
   #The builder strips out comments like this /^\s+\#/m
   ';

   # If this is a build, the comment has been stripped
   # ergo, just return $msg
   if ($string_to_figure_out_build !~ /\#/s)
   {
      return $msg;
   }

   # Grabs info about the caller of my caller all the way down.
   # returns a string.
   my @warn_array;
   my $i = 0;
   my ($package, 
       $filename, 
       $line,
       $subr,
       $has_args,
       $wantarray);

   while (($package, 
              $filename, 
              $line,
              $subr,
              $has_args,
              $wantarray) = caller ($i++))
   {
      $subr =~ s/^main\:\://;
      my $warn_string = "";#"\n";
      $warn_string .= "$filename $line CALLED ($subr)";
      push (@warn_array, $warn_string);
   }

   #we don't care about fishing report or ribbit/goldfish but we do
   #care about where it was called;
   
   shift (@warn_array);

   my $ribbit_called = shift (@warn_array);
   $ribbit_called =~ s/\bCALLED\b.*//;

   my $output_string;
   my $indent = "";
   my $indent_increment = " ";
   foreach $line (reverse (@warn_array))
   {
      pop (@warn_array);
      $line =~ s/\n/\n$indent/g;
      #$output_string = $indent.$line;
      $output_string .= $line;
      if (@warn_array)
      {
         $output_string .= "\n$indent$indent_increment";
      }
      else
      {
         $output_string .= 
             " WHERE\n'$msg' OCCURRED";
         $output_string .= " on $ribbit_called\n\n\n";
      }
      $indent .= $indent_increment;
   }

   return ($output_string);
}

sub caller_subroutine_list 
{
  my @result = ();
  my $i = 0;
  my $last_line = 0;
  while (my ($package, 
             $filename, 
             $line,
             $subr,
             $has_args,
             $wantarray) = caller ($i++))
    {
      $subr =~ s/^main\:\://;
      push (@result, "$subr\_line=$last_line");
      $last_line = $line;
    }
  return @result;
}

################################################################
# goldfish  & ribbit & history
#
#  It would be very nice to have the Perl library functions
#  "carp" and "croak", which are defined in the module "carp.pm"
#
# Sadly, including standard Perl libraries in a platform-independent
# way is tricky.  Aaron's solution:  Write our own "carp" and "croak"
# functions, and give them silly names:
#
# Also we put history in this set of functions.  This allows
# e_object->new() to know where an e_object was created which helps
# for debugging and error reporting. e.g. "The e_object created by file
# <my_file> on line (321) has the following problem" is much more
# helpful than "e_foo=(HASH0x12345) has the following problem"


sub goldfish
{
  my (@msg) = (@_);
  warn ("\nWARNING:\n",
        &fishing_report(@msg));
}

sub ribbit
{
  my (@msg) = (@_);
  my $n = scalar (@msg);
  die ("\nERROR:\n",
       &fishing_report(@msg)
       ."\n"
       );
}

sub history
{
  my (@msg) = (@_);
  return &fishing_report(@msg);
}

################################################################
# get_next_arg
#
# Utility you put atop your favorite perl-subroutine to help you
# extract arguments from @_.  The problem with @_ is that users can
# pass you -anything-.  They often pass you the wrong type of thing,
# generally because they forgot what they were supposed to pass you.
# Perl, being perl, will probably "do something" anyhow.  Problem is,
# it will probably be a wrong thing, and the user would have thanked
# you if you had just said:
#
#            "Hey! This isn't a string!".
#
# If you want to check your argments to be sure they're the right
# type, then use this.
# 
################################################################
sub get_next_arg
{
  my $arg_list_ref = shift;
  my $type         = shift;
  my $message      = shift;
  &ribbit ("get_next_arg itself: list-ref required for first argument.")
    unless ref ($arg_list_ref) eq "ARRAY";
  &ribbit ("get_next_arg itself: list-ref required for first argument.")
    unless ref ($type) eq "";
  &ribbit ("get_next_arg itself: two args required (list-ref, type-string).")
    unless scalar (@_) == 0;

  my $arg_value  = shift (@{$arg_list_ref});
  $message   .= ": expected $type";

  if      ($type =~ /string/) {
    &ribbit ($message) unless ref ($arg_value) eq "";
  } elsif ($type =~ /boolean/) {
    &ribbit ($message) unless $arg_value =~ /^[10]$/;
  } elsif ($type =~ /number/) {
    &ribbit ($message) unless $arg_value =~ /^\d+$/;
  } elsif ($type =~ /(.*)ref$/) {
    &ribbit ($message) unless ref ($arg_value) eq uc($1);
  } elsif ($type =~ /^e_/) {
    &ribbit ($message) unless $arg_value->isa($type);
  } else {
    &ribbit ("get_next_arg:  Don't know how to validate type '$type'");
  }
  return $arg_value;
}

################################################################
# &e_signal_list
#
# A nice way to declare multiple signals at one time.
# Takes in a string describing signals; returns a list of signals.
# In a way, it is a very depreciated version of List_Ports_For.
#
# At this point, we expect each signal to have a description of the form:
#   "signal_name     | signal_width     | signal_export ,"
# or 
#   "signal_name     | signal_width ,"
#

sub e_signal_list {
  my @descriptions = (@_);
  my $description = join (",", @descriptions);
  my @signal_list;

  # We don't actually do much here.  Just split on commas into
  # individual port-descriptions, then process each one.
  #
  # But first, strip-out comments as a frill.
  #
  $description = &Strip_Perl_Comments    ($description);
  $description =~ s/\n/ /sg;
  foreach my $signal_description (split(/\s*\,\s*/, $description)) {
    next if $signal_description eq "";  # skip blank entries.
    my @line_list = split(/\s*\|\s*/, $signal_description);
    my $signal_name = $line_list[0];
    my $signal_width = $line_list[1];
    my $signal_export;
       $signal_export = $line_list[2] if (scalar(@line_list) > 2);
    my $signal = e_signal->new ({
        name    => $signal_name,
        width   => $signal_width, });
    $signal->export(1) if ($signal_export ne "");
    push @signal_list, $signal;
  }

  return @signal_list;
}

################################################################
# Strip_Perl_Comments ($expr)
#
# Takes a string, which might be a multi-line Perl expression.
# For some reason, the Perl "eval" function is too lame to ignore
# comments embedded in the eval-expressoin.  Fine.  We'll strip
# the comments so it doesn't have to.
#
################################################################
sub Strip_Perl_Comments
{
    my $expr;
    my @lines;
    my $stripped_expr;
    ($expr) = (@_);

    @lines = split (/\n/, $expr);

    $stripped_expr = "";
    foreach (@lines)
    {
        $stripped_expr .= "\n", next if /^\#/;
        s/^(.*?)[^\\]\#.*$/$1/;
        $stripped_expr .= "$_\n";
     }
    return $stripped_expr;
}

sub log2
{
  my ($number) = (@_);
  &ribbit ("positive-number required for log2 not ($number)") if $number <= 0;
  
  my $log_base_e_of_2 = log 2;
  
  my $log_base_e_of_number = 
    ($number == 0)? 0 :        # Error condition... see below
    log $number;
  my $log_base_2_of_number = $log_base_e_of_number / $log_base_e_of_2;
  &goldfish ("log2 of 0 is undefined!  I'll just give you 0 anyway.\n")
    if ($number == 0);

  return $log_base_2_of_number;
}

################################################################ 
#
#  strip_enclosing_parentheses.
#
#  Examples are worth 10^3 words:
#
#      Input:                Output:
#  ---------------------------------------
#     (((a)))                  a
#     (a+b)                   a+b
#       x                      x
#     (a+b) * (c+d)         (a+b) * (c+d)     (this is the tricky one)
#    ((a+b) * (c+d))        (a+b) * (c+d)
#
################################################################
sub strip_enclosing_parentheses
{
  my ($in_string) = (@_);

  # Look--if it doesn't begin/end with parentheses, return it.
  # try stripping of beginning/end parenthesis, and see if they
  # work.
  #
  my $stripped_string = $in_string;
  return $in_string unless $stripped_string =~ s/^\s*\((.*?)\)\s*$/$1/sg;

  # We now have a string with the first-and-last paren stripped-off
  #
  # If it's still valid, return it:
  #
  if (&has_balanced_parentheses ($stripped_string)) {
    return &strip_enclosing_parentheses($stripped_string);
  } else {
    return $in_string ;
  }
}

sub has_balanced_parentheses
{
  my ($in_string) = (@_);

  # all we care about is parens.  Get rid of everything else:
  $in_string =~ s/[^\(\)]*//sg;

  # If there aren't any parens at all, then I guess they 
  #   match:
  return 1 if length ($in_string) == 0;

  # If the first parenthesis you find is one of these: ')'
  #   then guess what:  They're not balanced.
  #   same if the last one is '('
  return 0 if $in_string =~ /^[^\(]*\)/s;
  return 0 if $in_string =~ /\([^\)]*$/s;

  # so, we have parens.  The first one is "(" and the last one is ")".
  # Count 'em up.  
  my $count = 0;
  my @paren_list = split (//, $in_string);
  foreach my $paren (@paren_list) {
    $count--, next if $paren eq ')';
    return 0 if $count < 0;
    $count++, next if $paren eq '(';
  }

  return $count == 0;
}

################################################################
# copy_of_hash
#
# You pass-in a hash-ref, and we give you back a reference
# to a -copy- of that hash.
#
################################################################
sub copy_of_hash
{
  my ($input_hash) = shift;
  &ribbit ("copy_of_hash requires one argument:  A hash-reference.")
    if (ref($input_hash) ne "HASH" || scalar (@_));

  my %result = %{$input_hash};
  return \%result;
}

######################################################################
# copy: copies a HASH, ARRAY or scalar
sub copy_hash_array_or_scalar
{
   my @return;
   foreach my $thing_to_copy (@_)
   {
      my $ref = ref($thing_to_copy);
      if ($ref eq 'HASH')
      {
         #print $indent."ref is hash\n";
         #main::dumpValue($thing_to_copy);

         my %hash = (&copy_hash_array_or_scalar(%$thing_to_copy));
         my $hash_ptr = \%hash;
         push (@return, $hash_ptr);
      }
      elsif ($ref eq 'ARRAY')
      {
         my @array = &copy_hash_array_or_scalar(@$thing_to_copy);
         push (@return, \@array);
      }
      else
      {
         push (@return, $thing_to_copy);
      }
   }
   if (@_ == 1)
   {
      return $return[0];
   }
   else
   {
      return @return;
   }
}
################################################################
# max
#
# Returns the numeric max of the elements of the input list.
#
################################################################
sub max
{
  my $max = shift;

  for (@_)
  {
     # Handle "" specially-- it counts as the "uninitialized" value.
     next if $_ eq "";
     $max = $_, next if $max eq "";
     $max = $_ if ($_ > $max);
  }

  return $max;
}

################################################################
# min
#
# Returns the numeric min of the elements of the input list.
#
################################################################
sub min
{
  my $min = shift;

  for (@_)
  {
     # Handle "" specially-- it counts as the "uninitialized" value.
     next if $_ eq "";
     $min = $_, next if $min eq "";
     $min = $_ if ($_ < $min);
  }

  return $min;
}

################################################################
# get_file_modified_date
#
# You give a filename.  It tells when it was modified (0 means
# file-not-found). 
#
################################################################
sub get_file_modified_date
{
  my ($fname) = (@_);

  # If the file doesn't exist, then it hasn't existed since the big bang.
  return 0 if !-e $fname;

  # "stat" returns a whole bunch of crap I don't care about,
  # and one little tiny thing that I do care about:       vvvvvv  I care!
  my ($dev,$ino,$mode,$nlink,$uid,$gid,$rdev,$size,$atime,$mtime,
      $ctime, $blksize, $blocks) = stat ($fname);

  return $mtime;
}

BEGIN
{
  # "...when perl is compiling and it finds a builtin, it checks
  # the CORE::GLOBAL:: namespace and inserts a sub call rather
  # than a builtin if the sub exists."
  
  # Override the builtin function "int" with floor, below.
  *CORE::GLOBAL::int = \&floor;
}

my $EPSILON = 2**(-46);

# The following incorrect behavior has been observed with
# perl's builtin "int":
#
#  165 * 1.4 = 231
#   but
#  int(165 * 1.4) = 230.
#
# 165 * 1.4 as a binary number turns out to be 
#
#   11100110.11111111111111111111...
#
# Which is arguably equal to 11100111, but it's understandable
# if the computer can't see that.
#
# Solution: add a tiny number to the argument of int(), to let
# all those 1-bits propagate up to the 1's place position.  The
# ideal "tiny number" would be the smallest value represented
# in whatever floating point format was used.  Since Perl uses
# doubles for its floating point numbers, the right tiny number
# should be about 2**(-1022).  However, experimentally, I've 
# determined that 2**(-47) is too small, but 2**(-46) works.
#
# Note: rather than using "int", use sprintf, as is cryptically
# suggested by perl documentation:
#
# You should not use [int]
#   for rounding: one because it truncates
#   towards 0, and two because machine representations
#   of floating point numbers can sometimes produce
#   counterintuitive results.  For example,
#   int(-6.725/0.025) produces -268 rather than the
#   correct -269; that's because it's really more like
#   -268.99999999999994315658 instead.  Usually, the
#   sprintf(), printf(), or the POSIX::floor and
#   POSIX::ceil functions will serve you better than
#   will int().
#
sub floor
{
  my $num = shift;
  my $sign = $num >= 0 ? 1 : -1;

  my $floored = $sign * sprintf("%d", $sign * $num + $EPSILON);
 
  return $floored;
}

################################################################
# ceil
#
# Integer-value of input, truncated upwards.
#
################################################################
sub ceil
{
  my $num = shift;
  $num = int($num + 1) unless $num == int($num);

  return $num;
}

################################################################
# Bits_To_Encode
#
# Just a mnemonically-named alias for "ceil (log2(x+1))"
#
################################################################
sub Bits_To_Encode 
{
    my $x;
    ($x) = (@_);
    return ceil(log2($x+1));
}

################################################################
# Converts a numerical value to a binary string (a string
# composed of the characters '1' and '0').  The fractional part
# of the number is tragically lost.
#
# If you specify a width, the result is -at least- that wide,
# and maybe wider (zero-padded left).  If you don't specify a width,
# then the result is as wide as it is.
#
################################################################
sub num_to_bin
{
  my ($num, $width) = (@_);
  $num = int ($num);
  my $result = "";
  while ($num || $width > 0) {
    my $next_digit = ($num & 0x01) ? "1" : "0";
    $result = $next_digit . $result;
    $num >>= 1;
    $width -= 1;
  }
  $result = "0" if $result eq "";
  return $result;
}

################################################################
# is_power_of_two
#
# Returns true if it is an integer power of 2, and false if it isn't.
#
################################################################
sub is_power_of_two
{
  my ($val) = (@_);

  # Check arg.
  if (not defined $val or $val eq '' or $val =~ /[^\d+-\.]/)
  {
    ribbit("is_power_of_two expects a number (it was passed '$val')")
  }
  
  # Special cases:
  # 0 is not a power of two - handle it here because log2 explodes
  # on 0.
  return '' if $val == 0;

  my $log = log2($val);
  return $log == int($log);
}

################################################################
# next_lower_power_of_two
#
# The name describes behavior.  10 gives 8, 17 -> 16,   137 -> 128.
# You get the idea.
#
# If number is an even power of two, you get it back unmolested
#  (32 -> 32)
#
################################################################
sub next_lower_power_of_two
{
  my ($val) = (@_);
  return 2 ** (int (log2($val)))
}

################################################################
# next_higher_power_of_two
#
# The name describes behavior.  10 gives 16, 17 -> 32,   137 -> 256.
# You get the idea.
#
# If number is an even power of two, you get it back unmolested
#  (32 -> 32)
#
################################################################
sub next_higher_power_of_two
{
  my ($val) = (@_);
  return $val if (&is_power_of_two($val));
  return 2 * &next_lower_power_of_two($val);
}

################################################################
# round_up_to_next_computer_acceptable_bit_width
#
# Sorry about this name... if you can think of a better one,
# please suggest it.
#
# Given a number, returns a "computer-acceptable" bit width,
# that is, a number from {8, 16, 32, 64, ...}.  The returned
# number is the smallest acceptable value that's no less than
# the input number.
#
################################################################
sub round_up_to_next_computer_acceptable_bit_width
{
  my $val = shift;
  
  $val = 1 if $val <= 0;
  return max(8, 2**(ceil(log2($val))));
}

################################################################
# is_computer_acceptable_bit_width
#
# Returns true if the number is 8, 16, 32, 64, 128, ...
#
################################################################
sub is_computer_acceptable_bit_width
{
  my $val = shift;
  
  return ($val >= 8) && (($val & ($val - 1)) == 0);
}

######################################################################
# is_blessed returns 1 if ref (thing) is not equal to a built-in type
# this allows us to call methods on thing most likely isa
######################################################################
sub is_blessed
{
   my $thing = shift;

   return ref ($thing) !~
              /^(|REF|SCALAR|ARRAY|HASH|CODE|GLOB)$/;

#     my @built_in_refs = (
#                          "",  #not a ref at all.
#                          "REF",
#                          "SCALAR",
#                          "ARRAY",
#                          "HASH",
#                          "CODE",
#                          "GLOB");

#     my $ref_thing = ref ($thing);

#     foreach my $bir (@built_in_refs)
#     {
#        return (0) if ($ref_thing eq $bir);
#     }

#     #we ran the gauntlet succesfully, it must be blessed.
#     return (1);
}

################################################################
# concatenate
#
# You give a list of signal NAMES (strings), and this returns
# a string which represents the Verilog-format bitwise-concatenation
# of them.  For example:
#
#    concatenate ("x", "y", "z");
#
# would return the string:
#
#      "{x,y,z}"
#
# Note that this ONLY works for scalar objects, NOT refs to actual
# e_signal objects.
#
# Also note that this can be used on either the left- or right-hand
# side of an assignment (for example).
#
# Frilly feature: If any of the incoming signal names are null (""),
# Then it's exactly like they didn't exist at all: No syntax error!
#
################################################################
sub concatenate
{
  # Double-check to make sure people aren't passing-in references:
  my @non_null_names = ();
  foreach my $thing (@_) {
    &ribbit ("concatenate: all arguments must be -strings- (signal-names).")
      unless ref ($thing) eq "";

    push (@non_null_names, $thing) unless $thing eq "";
  }

  if (scalar (@non_null_names) == 0)  
  {
     &ribbit ("no names in concatenation");
  }
  elsif (scalar (@non_null_names) == 1)
  {
     return ($non_null_names[0]);
  }
  else
  {
     return "\{" . join (",\n", @non_null_names) . "\}";
  }
}

################################################################
# or_array / and_array
#
# a utility fucntion that takes an array and produces 
# join (" | ", @_) or join (" & ", @_) adds 1s and 0s as default
# values.  If a value in the array eq "", then it doesn't get ANDed or
# ORed at all
#
#################################################################
sub or_array
{
   my @a;
   foreach my $b (@_)
   {
      push (@a, $b) 
          unless $b eq "";
   }
   return 0 unless @a;
   return "(".join (" | ", @a).")";
}

sub and_array
{
   my @a;
   foreach my $b (@_)
   {
      push (@a, $b) 
          unless $b eq "";
   }
   &ribbit ("no signals") unless @a;
   return "(".join (" & ", @a).")";
}

################################################################
# complement
#
# toggles ~ on and off i.e. &complement("a") = "~a"
# complement "~a" = a
#
#################################################################

sub complement
{
   my $thing_to_complement = shift;
   &ribbit ("no_complement") if ($thing_to_complement eq "");
   
   my $need_parentheses = $thing_to_complement;   

   #strip off all parentheses and see if there's anything left.
   while ($need_parentheses =~ s/\([^\(\)]*\)//s)
   {;}

   # if there were an initial set of parentheses around
   # $thing_to_complement the whole thing should be gone.
   # otherwise we need parentheses.

   $thing_to_complement =~ s/^(\s*)(.*)/$2/s or 
       &ribbit ("But that's impossible! This regexp must match!");
   my ($indentation, $guts) = ($1,$2);

   ($need_parentheses =~ s/^(\s*)(\~)?([\w\s]*)([^\w\s])*/$4/s) or
   &ribbit ("But that's impossible! This regexp must match!");

   my $negation = $2;

   if ($negation eq "~")
   {
      ###############
      # we didn't strip off leading ~ for thing to complement because
      # we didn't know if it were in parentheses or not.  Now we know.
      if ($need_parentheses eq "") 
      {
         #simple negation e.g. ~a => a 

         #uncomment the following line to strip outer parentheses.
         #$guts =~ s/^\~\s*\(?(.*?)\)?$/$1/; #e.g. ~(a) => a

         #Now, why would you want to do that?  Nobody ever got hurt
         #from an additional prophylactic parentheses.
         $guts =~ s/^\~/$1/;  #~(a) => (a)
      }
      else #we need parentheses 
      {
         #complicated negation e.g. ~a | b => ~(~a | b)
         $guts = "\~\($guts\)";
      }
      return ($indentation.$guts);
   }
   else
   {
      $guts = "\($guts\)"
          if ($need_parentheses ne "");

      return ("$indentation\~$guts");
   }
}

sub complement_null_ok
{
   my $thing_to_complement = shift;
   $thing_to_complement ne "" or return "";
   return (&complement($thing_to_complement));
}
################################################################
# vp_fail
#
# (associated with the 'validate_parameter' utility, below).
#
# Sometimes, when a parameter is not validated, we want the 
# program to terminate (default behavior).  Other times, we just
# want a warning message.
# 
# This function looks at the incoming arguments to "validate_parameter,"
# and decides how to handle the problem.
#
################################################################
sub vp_fail
{
  my ($vp_args, @err_messages) = (@_);

  my $is_warning = $vp_args->{severity} =~ /^warn/i;

  my $intro   = "ERROR:   Parameter validation failed.";
     $intro   = "WARNING: Parameter validation issue"   if $is_warning;

  my $closing = "\n";
     $closing = "Continuing logic generation...\n"      if $is_warning;


  my @blab_list = ();
  push (@blab_list, $intro);
  push (@blab_list, $vp_args->{message}) if $vp_args->{message} ne "";
  push (@blab_list, @err_messages);
  push (@blab_list, $closing);
  my $msg = join ("\n  ", @blab_list);

  if ($is_warning) {
    &goldfish ($msg);
  } else {
    &ribbit ($msg);
  }
}

################################################################
# validate_parameter
#
# Let's say you got this parameter from someplace.  You want to 
# verify that it's what you expect.  Perhaps the parameter is a value
# in a hash.  Perhaps it's just a value.  
#
# Maybe you want to make sure it exists.  Maybe you want to set it 
# to a default value if it doesn't.  Maybe you want to check to be 
# sure that it's an integer, and that it's within a certain range.
# maybe you want to check to see if it's a boolean.
#
# You call this function with (a ref to a hash of) named arguments,
# and it does the indicated checking.
#
################################################################
sub validate_parameter
{
  my ($arg) = (@_);

  ################
  # First, get the value.  Check that the user hasn't over-specified
  # the value.
  #
  my $value;
  if (exists($arg->{value})) {
    &ribbit ("Can't specify both 'value' and 'hash'") if(exists($arg->{hash}));
    &ribbit ("Can't specify both 'value' and 'name'") if(exists($arg->{name}));
    $value = $arg->{value};
  } else {
    &ribbit ("No value found.",
             "  You must set 'value' or both 'hash' and 'name'")
      if !exists($arg->{hash}) || !exists($arg->{name});
    $value = $arg->{hash}->{$arg->{name}};
  }

  my $name = $arg->{name};

  ################
  # Now set the default, if there is one--and if we need to.
  #
  # Note that we can't set the default unless the user passed-in
  # a "hash" to validate.
  #
  if (exists($arg->{default}) && $value eq "") {
    &ribbit ("Can't set default for parameter '$name' (hash required)")
      unless exists ($arg->{hash});

    # set both our local copy and the user's original data:
    $value                = $arg->{default};
    $arg->{hash}->{$name} = $arg->{default};
  }

  # If $value is not set, then it's bad:
  if (($value eq "") && !$arg->{optional}) {
    &vp_fail ($arg, " Required parameter '$name' is missing.");
  }

  ################
  # check the type.
  #
  #  At present, we do not require you to specify a type,
  #  though this may change.
  #
  # Also, type-checking doesn't yet support blessed objects,
  # but it would be easy enough to do if the need arises.
  #
  if (($arg->{type} =~ /^int/i)) {
    &vp_fail ($arg, " Parameter '$name' must be an integer.")
        unless (ref ($value) eq "");

    # If the user has explicitly said they want an integer, then
    # we automatically convert hex-strings:
    $value = eval ($value) if $value =~ /^0x/;
    &vp_fail ($arg, " Parameter '$name' must be an integer.")
        unless (int ($value) == $value);

  } elsif ($arg->{type} =~ /^string$/i) {
    &vp_fail ($arg, " Parameter '$name' must be a string.")
      unless (ref ($value) eq "");

  } elsif ($arg->{type} =~ /^bool/i) {
    &vp_fail ($arg, " Parameter '$name' must be a boolean (1 or 0).")
      unless (ref ($value) eq "") && (($value == 1) || ($value == 0));
  }

  ################
  # check the range.
  # 
  # If the user has specified a "range", it must be a two-element list.
  # The first element is the lower-bound, and the second element is the 
  # upper bound.  The entire range (bounds inclusive) is the allowed range
  # for this parameter.  We use the "<=" and ">=" operators, whether or not
  # the current type happens to be a number.
  #
  if (exists($arg->{range})) {
    my $range = $arg->{range};
    &vp_fail ($arg, 
              " range for '$name'  must be specified as a two-element list)")
      unless (ref ($range) eq "ARRAY") && (scalar(@{$range}) == 2);

    my ($lower, $upper) = @{$range};
    &vp_fail ($arg, " Parameter '$name' (=$value) is outside of allowed range",
                    " ($lower, $upper)")
      unless ($lower <= $value) && ($upper >= $value);
  }

  ################
  # Check the allowed values.
  #
  # This requires some special-handling for strings,
  # which can be either matched-against or tested-for-equality against
  # the allowed values--and, even when tested-for-equality, they require
  # that we use the "eq" operator.
  #
  if (exists($arg->{allowed})) {
    my $allowed = $arg->{allowed};
    &vp_fail ($arg,
              " allowed values for '$name'  must be specified as a list)")
      unless (ref ($allowed) eq "ARRAY");

    my $is_allowed = 0;
    foreach my $allowed_value (@{$allowed}) 
    {
      if (($arg->{type} =~ /^string$/i)) {
        # If any of the allowed values look like patterns (start with /,
        # and have one more "/" someplace) use them as-such:
        #
        if (($allowed_value =~ /^\s*\/.*\/.*/)) {
          my $match_expr = "\$is_allowed = (\$value =~ $allowed_value)";
          eval ($match_expr);
          &vp_fail ($arg,
                    " Error evaluating match expression [$match_expr]: $@") 
            if $@;
        } else {
          $is_allowed = 1 if $value eq $allowed_value;
        }
      } else {
        $is_allowed = 1 if $value == $allowed_value;
      }
      last if $is_allowed;
    }

    &vp_fail ($arg,
              " Parameter '$name' (= $value)",
              " is not one of the listed allowed values.") unless $is_allowed;
  }

  ################
  # Mutually-exclusive values.
  # 
  # Sometimes, if you have your cake, you can't eat it, too.
  # This checks a given parameter against a list of others.
  # if the indicated value is non-null/non-zero (evaluates true),
  # then none of the items on the mutext-list can be nonzero/non-null
  # or else you get an error.
  #
  if (exists($arg->{exlcudes_all}) && $value) {
    my $exclude_list = $arg->{exlcudes_all};

    foreach my $exclude_name (@{$exclude_list}) {
      my $exclude_value = $arg->{hash}->{$exclude_name};
      next unless $exclude_value;
      &vp_fail ($arg,
              " Parameter '$name' (= $value)\n",
              " is mutually exclusive with parameter",
              " '$exclude_name' (= $exclude_value)\n");
    }
  }

  ################
  # Requirements.
  # 
  # Sometimes, you can't set one parameter without setting another.
  # If the given parameter is nonzero/non-null (evaluates true),
  # then this requires that some other specified parameter also
  # be nonzero/non-null.
  #
  if (exists($arg->{requires}) && $value) {
    my $require_name = $arg->{requires};
    &vp_fail ($arg, 
              " Parameter '$name' (= $value)\n",
              " requires setting paramter '$require_name'") 
      unless $arg->{hash}{$require_name};
  }

  return $value;
}

################################################################
# unit_prefix_to_num
#
# Converts "K" to 1000, "m" to 1/1000, "u" to 1/1,000,000, etc.
# If the incoming string is blank, returns "1.0".
# 
# Also happens to work with prefixes like "milli," "micro," and
# "Kilo".  Single-letter things are case-sensitive ('m' and 'M'
# are very different).  Multi-letter things are case-insensitive
# (mega and Mega are the same thing).
#
# The return-value is, of course, a floating-point number.
#
################################################################
my %unit_prefix_hash = (a     => (1.0 / 1.0E18   ),
                        ato   => (1.0 / 1.0E18   ),
                        f     => (1.0 / 1.0E15   ),
                        femto => (1.0 / 1.0E15   ),
                        p     => (1.0 / 1.0E12   ),
                        pico  => (1.0 / 1.0E12   ),
                        n     => (1.0 / 1.0E9    ),
                        nano  => (1.0 / 1.0E9    ),
                        u     => (1.0 / 1000000.0),
                        micro => (1.0 / 1000000.0),
                        "m"   => (1.0 / 1000.0   ),
                        milli => (1.0 / 1000.0   ),
                        c     => (1.0 / 100.0    ),
                        centi => (1.0 / 100.0    ),
                        d     => (1.0 / 10.0     ),
                        deci  => (1.0 / 10.0     ),
                        K     => (1000.0         ),
                        k     => (1000.0         ),
                        kilo  => (1000.0         ),
                        M     => (1000000.0      ),
                        mega  => (1000000.0      ),
                        G     => (1.0E9          ),
                        giga  => (1.0E9          ),
                        T     => (1.0E12         ),
                        tera  => (1.0E12         ),
                       );

sub unit_prefix_to_num
{
  my ($prefix) = (@_);
  return 1.0 if $prefix eq "";    # no prefix--whole units.

  # Treat all multi-letter prefixes (e.g. Mega) as lower-case:
  $prefix = lc($prefix) if length ($prefix) > 1;

  &ribbit ("unknown unit prefix: '$prefix'") 
    unless exists ($unit_prefix_hash{$prefix});

  return $unit_prefix_hash{$prefix};
}

################################################################
# System_Win98_Safe
#
# Win-98-safe "wrapper" for Perl's built-in 'system' command.
#
# Windows-98 can't handle an executable-name ($ARGV[0]) which
# has forward slashes in it.  WinNT and Win2000 can handle
# either forward- or backward-slashes.  So, if we notice that
# the operating system is Windows (or Cygwin), then we convert
# forward-slashes to backslashes in -only- the program-name
# part of the system-command.  We leave all the arguments 
# alone-- whether '/' or '\' is OK in an arugment is entirely 
# up to the program.
#
################################################################
sub System_Win98_Safe
{
   my (@command_parts) = (@_);
   my $sys_cmd         = join (" ", @command_parts);

   $sys_cmd =~ /^\s*(\S+)\s+(.*)$/ or die
       "System_Win98_Safe: Suspicious system-command: $sys_cmd";

  my $program_path = $1;
  my $arguments    = $2;

  $program_path =~ s|/|\\|g if ($^O =~ /(MSWin|cygwin)/i);

  my $new_sys_cmd = "$program_path $arguments";
  system ($new_sys_cmd);

  my $error_code = ($? >> 8);
  return $error_code;
}

################################################################
# master_address_width_from_slave_parameters
#
# Perhaps a bit specialized, but useful for masters which need
# to compute their address width given a slave's address width,
# data width and address base.
# 
# Input: hash ref of slave's SYSTEM_BUILDER_INFO section, which
# contains
# Address_Width
# Data_Width
# Base_Address
# 
# Output:
#
# convert address width to a byte address, e.g.
#   width 8 and data width 16 -> 0x200
# add that to the address base, e.g. 0x400
# 0x600
# return Bits_To_Encode of the sum.
# 
# Someone might also want to know the actual range of addresses for
# this slave.  Return that in a list, if list context.
# 
################################################################
sub master_address_width_from_slave_parameters
{
  my $project = shift or ribbit("no project!");
  my $master_desc = shift or ribbit("no master desc!");
  my $slave_desc = shift or ribbit("no slave desc!");

  my $slave_hash = $project->SBI($slave_desc);
  &ribbit("$slave_desc: no SBI") if !$slave_hash;
  
  # Require these slave parameters:
  for (qw(Address_Width Data_Width Base_Address Address_Alignment))
  {
    &ribbit ("slave $slave_desc is missing parameter '$_'")
      if !defined($slave_hash->{$_});
  }
  

  my $determining_data_width;
  my $culprit;
  if ($slave_hash->{Address_Alignment} eq "native")
  {
    # Native addresses depend on the data width of the master.
    my $master_hash = $project->SBI($master_desc, "MASTER");
    &ribbit("$master_desc: no SBI") if !$master_hash;
    
    $determining_data_width = $master_hash->{Data_Width};
    $culprit = $master_desc;
  }
  elsif ($slave_hash->{Address_Alignment} eq "dynamic")
  {
    # Dynamic-addressed slave.  Addressing depends on data width, 
    # and on whether or not this slave is an adapter (adapter
    # slaves have byte-aligned address).
    my $is_adapter = 0;
    if ($slave_desc =~ m|^(.+/)(.*)$|)
    {
      my $module_name = $1;
      my $module_sbi = $project->SBI($module_name);
      if ($module_sbi)
      {
        $is_adapter = $project->SBI($module_name)->{Is_Adapter} ||
          $project->SBI($module_name)->{Is_Test_Adapter};
      }
    }

    $determining_data_width = $is_adapter ? 8 : $slave_hash->{Data_Width};
    $culprit = $slave_desc;
  }
  else
  {
    ribbit("I don't understand address_alignment " .
      "'$slave_hash->{Address_Alignment}'\n");
  }
  # Double check the numbers we got, and blame the culprit if there's trouble.
  &ribbit("$culprit: no data width") if !$determining_data_width;
  &goldfish ("$culprit: weird data width '$determining_data_width")
    if log2($determining_data_width) != int(log2($determining_data_width));

  $b = log2($determining_data_width / 8);

  # Get the base address for this slave, which is a byte address.
  my $base_address = $slave_hash->{Base_Address};
  
  # Base_Address is often expressed as a hex number.  Don't assume that this is
  # always true.
  $base_address = hex($base_address) if ($base_address =~ /^0x/);
  
  # Get slave's address width (slave-word-size, not necessarily byte-address):
  my $address_width = $slave_hash->{Address_Width};
  my $last_address = $base_address + 2**($b + $address_width) - 1;

  my $real_addr_width = Bits_To_Encode($last_address);
  
  # If you asked for a scalar, you get the number of bits required to
  # address this slave.  Otherwise, you get the whole shebang.
  if (wantarray)
  {
    return
      ($real_addr_width, $base_address, $last_address);
  }
  
  return $real_addr_width;
}

# hash_ref_to_parameters_signature
#
# Input: a reference to a hash
# Output: a scalar uniquely representing all keys and values of the hash,
#   with recursive traversal of hash and array references.
# 
#   Important! Any key/value pairs with key = 'Parameters_Signature' are
#   ignored.
#
#
# Suggested use of this function: in a generator function, build up the
# hash of all parameters which influence component generation.  Be careful
# to include everything, including subtle items such as address widths,
# perhaps dependent on base addresses of slaves mastered by this component,
# etc!  Before generation, get the 'signature' of that hash (equivalently,
# the signature of the component to-be-built) from this function.  Only
# regenerate the component if the computed parameters signature fails to
# match a saved Parameter_Signature value.  If a successful generation
# occurs, store the signature value in the component module ptf, using
# the ptf value name 'Parameters_Signature'.

sub hash_ref_to_parameters_signature
{
  if (@_ == 1)
  {
    # Top-level call: parameter must be a hash reference.
    my $hr = shift;
    if (ref($hr) ne 'HASH')
    {
      &ribbit("hash_ref_to_parameters_signature() wants a hash reference\n")
    }

    # Make a recursive call. (The 1-parameter invocation of this function
    # is merely a user convenience.)
    return hash_ref_to_parameters_signature('/', $hr);
  }
  elsif (@_ == 2)
  {
    # Recursive call.
    my ($key, $value) = @_;
    my $return_value;

    # It is a known fact that the eventual return value of this function
    # will be stored in a ptf value 'Parameters_Signature' (note carefully
    # the name of this function).  Avoid growth-without-bound on that value
    # by ignoring the value itself, which likely resides in the hash we're
    # currently traversing.
    return if $key eq 'Parameters_Signature';

    # The parameter pair (key, value) can be:
    # 1)
    # key: ptf value name
    # value: ptf value
    #   Recursion leaf.  Concatenate the ptf value name and ptf value into
    #   the return value.
    #
    # 2)
    # key: hash name
    # value: hash reference
    #   Save the hash name in the return value, and deal with the hash
    #   elements recursively, in sorted order by key.
    #
    # 3)
    # key: array name
    # value: array reference
    #   Save the array name in the return value, and deal with the array
    #   elements recursively, in element order.
    #
    if (ref($value) eq 'HASH')
    {
      $return_value .= $key;

      # Recur on a hash.
      for my $subkey (sort keys %{$value})
      {
        $return_value .=
          hash_ref_to_parameters_signature($subkey, $value->{$subkey});
      }
    }
    elsif (ref($value) eq 'ARRAY')
    {
      $return_value .= $key;

      # Recur on an array.
      for my $array_item (@{$value})
      {
        $return_value .= hash_ref_to_parameters_signature('', $array_item);
      }
    }
    elsif (not ref($value))
    {
      # Value is a scalar.  This is a leaf of recursion. Save key and value.
      $return_value .= $key;
      $return_value .= $value;
    }
    else
    {
      &ribbit("Unexpected ref: ", ref($value));
    }

    return $return_value;
  }
  else
  {
    &ribbit("Unexpected number of parameters: ", 0 + @_, "\n");
  }
}

# str2hex ("string"):
# convert an ascii quoted string into the hexadecimal equivalent verilog
# style string of the appropriate length.  A 2 character string requires
# 2*8 bits, so str2hex("Hi") returns 16'h4869.  Display signal =
# 16'h4869 as radix ascii in your wave viewer to display ascii text.
sub str2hex
{
    my $string = shift;
    # it takes two hex chars to represent one ascii char:
    my $length = length ($string) * 2;
    my $TEMPLATE = "H".$length;
    # multiply length by 4 bits to get proper hex size.
    return ($length*4)."'h".unpack($TEMPLATE, $string);
}

# str2bin ("string"):
# convert an ascii quoted string into the bin equivalent vhdl std_logic_vector
# style string of the appropriate length.  A 2 character string requires
# 2*8 bits, so str2bin("Hi") returns 16'b0100100001101001 (16'h4869).
# Display signal as radix ascii in your wave viewer to display ascii text.
sub str2bin
{
    my $string = shift;
    # it takes 8 bits to represent one ascii char:
    my $length = length ($string) * 8;
    my $TEMPLATE = "B".$length;
    return ($length)."'b".unpack($TEMPLATE, $string);
}

my $base_class = 'e_object';
sub assign_subroutine
{
   my ($pkg,$sub_name,$sub) = @_;
   my $glob = $pkg.'::'.$sub_name;

   no strict 'refs';

   ###############
   # if this has a subroutine already defined in the package
   # then put this subroutine as the SUPER::$sub_name in the base
   # class.

   if (defined (&$glob))
   {
      #$glob = $base_class.'::'.$sub_name;
      if (defined (&$glob))
      {
         die ("$pkg, $glob already defined\n");
      }
   }

   #warn "$pkg setting $glob\n";
   *$glob = $sub;
   use strict 'refs';
}

sub package_setup_fields_and_pointers
{
   my ($pkg, $fields, $pointers,$debug) = @_;

   my $super_name = $pkg.'::ISA';
   no strict 'refs';
   my ($super) = @$super_name;
   use strict 'refs';

   my $gp = 'get_pointers';
   my $gf = 'get_fields';
   if (!$super)
   {
      $gp = 'empty_array';
      $gf = 'empty_array';
      $super = "$pkg";
   }
   #print "pkg $pkg, super is $super\n";

   my $get_pointers = sub 
   {
      my $this = shift;
      my %new_args = %$pointers;
      my @array_pointers = keys (%$pointers);
      map {$new_args{$_}++} $super->$gp();
      return (keys (%new_args));
   };

   &assign_subroutine($pkg, 'get_pointers', $get_pointers);
   foreach my $arg (keys (%$pointers))
   {
      my $value = $pointers->{$arg};
      my $sub = sub {
         my $this = shift;
	 &ribbit ("what are you thinking?")
	   unless (ref($this));
         if (@_)
         {
            $this->{$arg} = shift;
         }
         elsif (!defined ($this->{$arg}))
         {
            $this->{$arg} = $value;
         }
         return $this->{$arg};
      };
      &assign_subroutine($pkg,$arg,$sub);
   }

   my $get_fields = sub 
   {
      my $this = shift;
      my %new_args = %$fields;
      my @array_fields = keys (%$fields);
      map {$new_args{$_}++} $super->$gf();
      return (keys (%new_args));
   };
   &assign_subroutine($pkg, 'get_fields', $get_fields);

   foreach my $arg (keys (%$fields))
   {
      my $value = $fields->{$arg};
      my $set = $value;
      my $ref = ref($value);

      my $sub;
      if (&is_blessed($value))
      {
         $sub = sub {
            my $this = shift;

            if (@_)
            {
               my $shift = shift;
               $this->{$arg} = $ref->new($shift);
            }
            elsif (!defined ($this->{$arg}))
            {
               $this->{$arg} = $ref->new($value);
            }
            return $this->{$arg};
         };
      }
      elsif (ref ($value))
      {
         $sub = sub {
            my $this = shift;
            if (@_)
            {
               my $shift = shift;
               $this->{$arg} = &copy_hash_array_or_scalar($shift);
            }
            elsif (!defined ($this->{$arg}))
            {
               $this->{$arg} = &copy_hash_array_or_scalar($value);
            }
            return $this->{$arg};
         };
      }
      else
      {
         $sub = sub {
            my $this = shift;
            if (@_)
            {
               my $shift = shift;
               $this->{$arg} = $shift;
            }
            elsif (!defined ($this->{$arg}))
            {
               $this->{$arg} = $value;
            }
            return $this->{$arg};
         };
      }
      &assign_subroutine($pkg,$arg,$sub);
   }

}

#
# get_mask_of_1_bits
# 
# Given parameter n, construct a mask of n 1-bits, from
# bit position (n - 1) down to 0.
#
# Why make this a routine?  Isn't
#
#   ((1 << $n) - 1)
#
# simple enough to inline?  Well, perl has a nasty tendency to
# overflow: the above expression evalulated at $n = 32 yields 0.
# This subroutine handles that special case.  It's also nicer to
# have the inherent 32-bit-ness of this code in one place, rather
# than throughout the code base.
#
# Judgement call: accepting 0 as an input may or may not be silly.
# I figure that if you pass 0 in (or undef), you want 0 back.
#
sub get_mask_of_1_bits
{
  my $n = 0 + shift;

  ribbit("bad parameter '$n'") if (($n > 32) || ($n < 0));
  return ~0 if ($n == 32);
  return ((1 << $n) - 1);
}

# one_hot_encoding (integer > 0):
# generate verilog type X'b000010000; style strings to create a one-hot
# encoding for FSM's.  'X' will be the integer argument passed in.
# Usage would be like: @states = &one_hot_encoding(3); to generate
# @states = (3'b001, 3'b010, 3'b100);
#
# sub one_hot_encoding # This old version is broken on Solaris:
# {
#    my $number = shift;
#    return 
#        map {$number."'b".sprintf ("\%.".$number."b",(1 << $_))} (0..($number-1));
# }
sub one_hot_encoding
{
    my $number = shift;
    return if ($number < 1);
    # make a list of 'number' 0's
    my @k = (0) x $number;
    # make the last element '1'
    $k[($number-1)] = 1;
    # initialize return value array:
    my @vals = ($number."'b".join('',@k));

    for (2 .. $number) {
        my $zero = shift(@k);
        push (@k, $zero);
        push (@vals, ($number."'b".join('',@k)));
    }

    return @vals;
} # &one_hot_encoding

=item I<make_special_assignments()>
A "spaceless" hash looks like this:
$spaceless = {
  Address_Alignment => "native",
  IRQ_MASTER => {
    cpu/data_master => {
      IRQ_Number => 1,
    }
  }    
  MASTERED_BY => {
    cpu/data_master => {
      priority => 1,
      fictitious_section => {
          with_space => {
          {
            assignment = "1",
          },
        },
      },
    },
    cpu/instruction_master => {
      priority => 1,
    },
  },
  NO_SPACE_SECTION => 
  {
    foo => "bar",
    one => "two",
  },
};

In other words, standard ptf assignments look like regular
key => scalar value pairs, but for ptf subsections whose names are of the form:

(\S+)\s+(\S+)
$key = $1;
$subkey = $2;

are represented in a subhash for each $key value, with a sub-subhash
for each unique $subkey value; each sub-subhash is keyed on $subkey,
and contains all the assignments of ptf subsection "$key<space>$subkey".


Here are the ptf file entries corresponding to the above spaceless hash:

SYSTEM_BUILDER_INFO 
{
  Address_Alignment = "native";
  MASTERED_BY cpu/data_master
  {
     priority = "1";
     fictitious_section with_space
     {
       assignment = "1";
     }
  }
  MASTERED_BY cpu/instruction_master
  {
     priority = "1";
  }
  IRQ_MASTER cpu/data_master
  {
     IRQ_Number = "1";
  }
  NO_SPACE_SECTION 
  {
     foo = "bar";
     one = "two";
  }
}


This routine puts the spaces back in:
$spacy = {
  Address_Alignment => "native",
  "IRQ_MASTER cpu/data_master" => {
    IRQ_Number => 1,
  }    
  "MASTERED_BY cpu/data_master "=> {
    priority => 1,
    "fictitious_section with_space" => {
      assignment = "1";
    },
  },
  "MASTERED_BY cpu/instruction_master "=> {
    priority => 1,
  },
  NO_SPACE_SECTION => 
  {
    foo => "bar",
    one => "two",
  },
};

=cut

sub spaceless_to_spaced_hash($)
{
  my $spaceless_hash = shift;
  
  if (ref($spaceless_hash) ne 'HASH')
  {
    ribbit("expected hash reference, got ", ref($spaceless_hash), "!");
  }
    
  my $spaced_hash = {};
  
  foreach my $key (sort keys %$spaceless_hash)
  {
    my $value = $spaceless_hash->{$key};
    if (!ref ($value))
    {
      $spaced_hash->{$key} = $value;
      next;
    }
    
    # $value isn't a scalar.  Expect a hash reference:
    if (ref($value) ne 'HASH')
    {
      ribbit("expected hash reference; got ", ref($value), "!")
    }

    foreach my $subkey (keys %$value)
    {
      # Values for each subkey are either
      # 1) scalars - this indicates a non-spaced section.  Drop
      #   those scalars into a single sub-hash.
      # 2) hash references.  Put the space back in to create the
      #   key, and recur on the hash reference's contents.
      if (!ref($value->{$subkey}))
      {
        $spaced_hash->{$key} = {} if !exists($spaced_hash->{$key});
        $spaced_hash->{$key}->{$subkey} = $value->{$subkey};
      }
      else
      {
        my $spaced_subkey = "$key $subkey";
        $spaced_hash->{$spaced_subkey} =
          spaceless_to_spaced_hash($value->{$subkey});
      }
    }
  }
  
  return $spaced_hash
}

=item I<to_base_26()>
Convert an input number to base 26 (digits are represented by
the lowercase alphabet [a-z]).

1st parameter: the number to convert
optional 2nd parameter: the number of digits to use.  Defaults 
  to 3, which allows for 26**3 unique tags.

Why this subroutine exists: simulation wave info in the
<MODULE foo>/SIMULATION/DISPLAY section of the ptf is contained
in uniquely-tagged subsections.  Dividers and signals appear in the
wave window in lexically sorted order by tag.  Thus, this routine,
which converts from easily-created sequence numbers to a representation
which admits a lexical sort.

See e_project::set_sim_wave_signals().
=cut

sub to_base_26($;$)
{
  my $x = shift;
  my $digits = shift || 3;
  my @chars = ();
  
  return 'a' x $digits if ($x == 0);
  
  for (0 .. -1 + $digits)
  {
    push @chars, chr(($x % 26) + ord('a'));
    $x = int($x / 26);
  }
  
  my $ret = join("", reverse @chars);
  return $ret;
}

# +---------------------------------------
# | ensure_class_ptfs(globals *g)
# |
# | OK, I relent: this routine reads in
# | every single class.ptf file, but only
# | the first time it is called.
# |
# | They're all stashed up in g{class_ptfs} by
# | component name. I mean module type. Or
# | peripheral class. You know, that thing
# | with many names.
# |
sub ensure_class_ptfs($$)
    {
    my ($g,$verbose) = (@_);

    return if($$g{class_dirs}); # already loaded them. yay.

    require "ptf_parse.pm";

    print_command("Finding all available components") if $verbose;

    $$g{class_ptfs} = {};
    $$g{class_dirs} = {};

    # |
    # | Find the "install.ptf" file which lists
    # | the location and versions of all the
    # | installed available components
    # |

    my $system_directory =$$g{system_directory};

    if(! -d $system_directory)
    {
        ribbit("get_class_ptf and ensure_class_ptfs must be called with g{system_directory}");
    }

    my $f;

    my $dir = $system_directory;
        {
        $f = "$dir/.sopc_builder/install.ptf";
        if(-f $f)
            {
            last;
            }
        $f = "";
        }

    ribbit ("no install.ptf file found") if(! -f $f);

    print_command("Reading $f") if $verbose;

    my $install_ptf = ptf_parse::new_ptf_from_file($f);

    my $install_ptf = ptf_parse::get_child_by_path($install_ptf,"PACKAGE");

    my $component_kind_count = ptf_parse::get_child_count($install_ptf,"COMPONENT");

    # | walk through all the COMPONENT children, and fetch the class.ptf
    # | of only the highest VERSION.

    for(my $i = 0; $i < $component_kind_count; $i++)
        {
        my $component_ptf = ptf_parse::get_child($install_ptf,$i,"COMPONENT");
        my $component_name = ptf_parse::get_data($component_ptf);
        my $version_count = ptf_parse::get_child_count($component_ptf,"VERSION");
#    "found component $component_name with $version_count versions\n";
        my $highest_version = -10;
        my $highest_version_ptf;
        for(my $j = 0; $j < $version_count; $j++)
            {
            my $version_ptf = ptf_parse::get_child($component_ptf,$j,"VERSION");
            my $version = ptf_parse::get_data($version_ptf);
            if($version > $highest_version)
                {
                $highest_version = $version;
                $highest_version_ptf = $version_ptf;
                }
            }

        # |
        # | got a component and found the highest version!
        # | if it exists, load its class.ptf.
        # |

        my $component_directory =
                ptf_parse::get_data_by_path($highest_version_ptf,"local");
        $component_directory =~ s/\\/\//g;  # no bad (backward) slashes
        $$g{class_dirs}{$component_name} = $component_directory;
        }
      print_command("Found $component_kind_count components") if $verbose;
    }

# +---------------------------------------
# | get_class_ptf(globals *g,module_type)
# |
# | Return a ptf_ref to the class.ptf for a particular
# | component class. (This is cached in g{class_ptfs}.
# |

sub get_class_ptf($$;$)
  {
  my ($g,$module_type, $verbose) = (@_);

  ensure_class_ptfs($g, $verbose);

    my $class_ptf = $$g{class_ptfs}{$module_type};

    # |
    # | did we get it? if not, load it (first time)
    # |

    if(!$class_ptf)
        {
        my $component_directory = $$g{class_dirs}{$module_type};
        $class_ptf = ptf_parse::new_ptf_from_file("$component_directory/class.ptf");
        $$g{class_ptfs}{$module_type} = $class_ptf;
        }

  return $class_ptf;
  }

# +--------------------------------------------------
# | find_component_dir(globals *g,module_ref *,module_type)
# |
# | Given a module reference, find out what kind
# | it is, and search the component path list for
# | it. return the path as a string.
# |
# | Actually, use the module_type if it's there, else
# | the module ref
# |

sub find_component_dir
  {
  my ($g,$module_ref,$module_type,$verbose) = @_;
  my $dir;

  ensure_class_ptfs($g,$verbose);

  if(!$module_type)
    {
    $module_type = ptf_parse::get_data_by_path($module_ref,"class");
    }

  return $$g{class_dirs}{$module_type};
  }

# +--------------------------------------------------
# | find_all_component_dirs(globals *g)
# |
# | Handy utility to find the directories of all 
# | components.
# |

sub find_all_component_dirs
  {
  my ($g,$verbose) = @_;
  
  ensure_class_ptfs($g,$verbose);
  my @dirlist = values %{$$g{class_dirs}};
  
  return \@dirlist;
  }


# Every .pm ends with a 1!
1;
