#Copyright (C)2001-2002 Altera Corporation
#Any megafunction design, and related net list (encrypted or decrypted),
#support information, device programming or simulation file, and any other
#associated documentation or information provided by Altera or a partner
#under Altera's Megafunction Partnership Program may be used only to
#program PLD devices (but not masked PLD devices) from Altera.  Any other
#use of such megafunction design, net list, support information, device
#programming or simulation file, or any other related documentation or
#information is prohibited for any other purpose, including, but not
#limited to modification, reverse engineering, de-compiling, or use with
#any other silicon devices, unless such use is explicitly licensed under
#a separate agreement with Altera or a megafunction partner.  Title to
#the intellectual property, including patents, copyrights, trademarks,
#trade secrets, or maskworks, embodied in any such megafunction design,
#net list, support information, device programming or simulation file, or
#any other related documentation or information provided by Altera or a
#megafunction partner, remains with Altera, the megafunction partner, or
#their respective licensors.  No other licenses, including any licenses
#needed under any third party's intellectual property, are provided herein.
#Copying or modifying any file, or portion thereof, to which this notice
#is attached violates this copyright.




################
# europa.pm
#
# An embedded language for describing logic.
#
#
#  All e_objects, below, are a hash with one
#  required key: "type."  There are several known types:
#
# known e_object types:
#    e_signal
#    e_signal_group
#    e_module
#    e_instance
#    e_assignment
#    e_procedure
#
#  All e_objects have one common attribute: type
# 
#    singnals have name, width, and singnal_type attributes.

sub ribbit
{
  my ($msg) = (@_);
  die "Please link with 'ribbit' module.\n$msg\n";
}

sub hash_to_string
{
  my ($h, $space) = (@_);
  my @result = ();
  my $name = $h->{name};
     $name = "unnamed" if $name eq "";
  push (@result, "$space $name, which is:\n");
  foreach $key (keys(%$h)) {
    push (@result, "$space   $key = ($h->{$key})\n");
  }
  return join ("", @result);
}

################################################################
# verilog_object_order
# sorting routine which places module contents in 
# conventional Verilog declaration order:  
#
#  - signal-declarations first (ports, then wires, then registers)
#  - assignments next.
#  - procedural statements next.
#  - last, but not least: instances.
#
################################################################
sub verilog_object_order
{
  # if they're both signals, use finer-level metric to decide:
  if ($a->{type} eq "e_signal" && $b->{type} eq "e_signal") {
    # bonus frill: make inputs come before outputs:
    if ($a->{signal_type} eq "port" && $b->{signal_type} eq "port") {
      return -1 if $a->{direction} eq "input";
      return  1 if $b->{direction} eq "output";
      return  0;
    }
    return -1 if $a->{signal_type} eq "port";
    return  1 if $b->{signal_type} eq "port";

    return -1 if $a->{signal_type} eq "wire";
    return  1 if $b->{signal_type} eq "wire";

    return  0;
  }

  return -1 if $a->{type} eq "e_signal";
  return  1 if $b->{type} eq "e_signal";
}

################################################################
# Validate_Args
#
# Given a document which describes your functions' arguments and a
# hash, this function will process the hash as-follows:
#
#  * It will make sure that any required args (keys) are present.
#  * It will supply default values when necessary.
#  * It will create a binary hash of which args were set by the user.
#  * If appropriate, it will validate the type.
#
# Here are the rules for a document-string:
#
# " -" and "#" are line-based comments.  Everything after one of these
# on a line is ignored.  Notice that " -" only counts as a
# comment-char if it is preceded by a space.  Thus, you could actually
# have an argument named "nios-build" (but not "nios -build").
#
# Blank lines (after comment stripping) are ignored.
#
# The first word (/\w+/) on a line is the name of a "documented"
# argument.  If it is preceded by a "*", then it is a -required-
# argument.  Calling a function without setting a required argument
# produces an error.
#
# After the argument-name, any non-space string preceded by an
# equals-sign (/=\S+/) is the default value for this argument.  The
# default value will apply if the argument is not explicitly set.
#
# Any non-space string preceded by a colon (/:\S+/) is the type for
# this argument.  If a type is given, then the value of the
# arument must be a hash-ref.  The hash must contain a "type" key
# indicating the "type" of the "data-structure" contained in the hash.
# An error will be generated unless the documented argument is a 
# "data-structure" of the indicated "type."
#
################################################################
sub Arg_Error_Message
{
  my ($msg, $subroutine_name, $raw_hash) = (@_);

  return "Invalid arguments for $subroutine_name:\n" . 
         &hash_to_string ($raw_hash, "     ")        . 
         "  $msg\n"                                  ;
}

sub Document_Args
{
  my ($subroutine_name, $doc) = (@_);

  # Local data we keep around:
  my %required       = ();
  my %documented     = ();
  my %default_value  = ();
  my %expected_type  = ();
  my %allowed_values = ();

  # Document strings are, in fact, line-based.
  # It is a bit wasteful to re-process the document
  # string every time this function is called.  How inefficient.
  # One fine future day, we might want to cache this info the first
  # time we see it.
  #
  my @doc_lines = split (/\s*\n\s*/, $doc);
  foreach $doc_line (@doc_lines) {
    # Strip all perl-style comments from the document (and leading
    #    space, while we're stripping things):
    $doc_line =~ s/^\s*([^\#]*)\#.*$/$1/;

    # Strip-off all of the definition (everything after " -"):
    $doc_line =~ s/^(.*?)\s+\-.*$/$1/;

    next unless $doc_line =~ /^(\*?)\s*(\w+)(.*?)$/;
    my $required_star = $1;
    my $arg_name      = $2;
    my $rest_of_line  = $3;

    if ($rest_of_line =~ s/(.*?)\s*\((.*)\)\s*(.*?)/$1$3/) {
      my @value_list = split (/\s*,\s*/, $2);
      $allowed_values{$arg_name} = \@value_list;
    }

    $documented   {$arg_name} =  1;
    $required     {$arg_name} =  1 if $required_star;
    $default_value{$arg_name} = $1 if $rest_of_line =~ /\s*=\s*(\S+)\s*/;
    $expected_type{$arg_name} = $1 if $rest_of_line =~ /\s*:\s*(\S+)\s*/;
  }

  my %sub_doc_hash;
  $sub_doc_hash{documented}     = \%documented;
  $sub_doc_hash{required}       = \%required;
  $sub_doc_hash{default_value}  = \%default_value;
  $sub_doc_hash{expected_type}  = \%expected_type;
  $sub_doc_hash{allowed_values} = \%allowed_values;

  # Keep the predigested information for each subroutine in 
  # a hash keyed by subroutine-name:
  #
  $db_arg_doc{$subroutine_name} = \%sub_doc_hash;

}

sub Validate_Args
{
  my ($subroutine_name, $raw_hash) = (@_);

  my $sub_doc_hash = $db_arg_doc{$subroutine_name};

  # Things we will return:
  my %result         = ();
  my %user_defined   = ();
  my %unrecognized   = ();

  # Now process the actual incoming arguments:
  #
  foreach $raw_arg_name (keys(%$raw_hash)) {
    $user_defined{$raw_arg_name} = 1;

    if ($sub_doc_hash->{documented}{$raw_arg_name} ) {
      $result      {$raw_arg_name} = $raw_hash->{$raw_arg_name};
    } else {
      # For now, we're just going to throw both the recognized- and 
      # unrecognized-arguments together into the same result.  In the 
      # future, we might want to make an option to return the 
      # unrecognized arguments separately.
      #   
      $result      {$raw_arg_name} = $raw_hash->{$raw_arg_name};
      $unrecognized{$raw_arg_name} = $raw_hash->{$raw_arg_name};
    }

    if (my $expecto = $sub_doc_hash->{expected_type}{$raw_arg_name}) {
      my $raw_arg_type = $raw_hash->{$raw_arg_name}{type};
      if ($expecto ne $raw_arg_type) {
        ribbit (&Arg_Error_Message (
                "Wrong arg type for '$raw_arg_name'.\n" . 
                 "  Got type '$raw_arg_type', expected '$expecto'.\n", @_));
      }
    }
  }

  # Now be sure required arguments were there:
  #
  foreach $required_arg (keys(%{$sub_doc_hash->{required}})) {
    next if $user_defined {$required_arg};
    ribbit (&Arg_Error_Message (
               "Required argument '$required_arg' not found.\n", @_));
  }

  # Fill-in defaults:
  foreach $defaulting_arg (keys(%{$sub_doc_hash->{default_value}})) {
    next if $user_defined {$defaulting_arg};
    $result{$defaulting_arg} = $sub_doc_hash->{default_value}{$defaulting_arg};
  }

  # Check against allowed values:
  #
  foreach $limited_arg  (keys (%{$sub_doc_hash->{allowed_values}})) {
    my $value = $result{$limited_arg};
    my $is_allowed = 0;

    my (@allowed_value_list) = 
      (@{$sub_doc_hash->{allowed_values}{$limited_arg}});

    foreach $allowed_value (@allowed_value_list) {
      $is_allowed = 1 if $allowed_value eq $value;
      $is_allowed = 1 if $allowed_value == $value;
    }
    if (!$is_allowed) { ribbit ( &Arg_Error_Message (
       "Error '$value' is not one of the allowed\n" . 
        "  values for '$limited_arg' ("             . 
        join (", ", @allowed_value_list) . ")\n", @_));
    }
  }

  return (\%result, \%user_defined, \%unrecognized);
}

################################################################
# ei_hdl_comment
#
# returns a comment in the language of your choice. Simple
# argument-passing without validation.
#
################################################################
sub ei_hdl_comment
{
  my ($comment, $language) = (@_);
  return "   // $comment\n" if $language =~ /verilog/i;
  return "   -- $comment\n" if $language =~ /vhdl/i;

  ribbit ("e_hdl_comment: unknown language: $language");
}

################################################################
# ei_create_hash_from_attributes
#
# Given a list of "name = value" string-pairs, this routine
# builds the corresponding hash.  The result (hash) is retuned
# by reference.
#
# This routine will also work if any of the incoming elements
# are "compound"-- that is, multiple name=value pairs separated
# by vertical bars ('|').
#
# As an added bonus, any "unnamed" attributes will be "coerced"
# if you provide an ordered list of attrubite-names.  The first
# unnamed atribute will be taken as the value for the first name
# in the coercion-list, the 2nd unnamed attribute will be taken
# as the value of the 2nd name in the coercion-list, etc.
#
&Document_Args ("ei_create_hash_from_attributes", "
 *attribute_list                    - Ref to list of attribute strings.
  unnamed_coercion_order            - Ref to list of coercion names.
 ");
#
################################################################
sub ei_create_hash_from_attributes
{
  my ($arg) = &Validate_Args ("ei_create_hash_from_attributes", @_);

  my %result = ();

  # Create a local copy of this so we can destructively shift it:
  my (@unnamed_attribute_coercion_order) = (@{$arg->{unnamed_coercion_order}});

  my @expanded_attributes = ();
  foreach $attribute (@{$arg->{attribute_list}}) {
    $attribute =~ s/\n/ /smg;
    $attribute =~ s/\s+(.*?)\s+/$1/smg;
    push (@expanded_attributes, split (/\s*\|\s*/, $attribute));
  }

  foreach $pair (@expanded_attributes) {
    my $attrib_name  = "";
    my $attrib_value = "";
    if ($pair !~ /^\s*(\w+)+=(.+?)\s*/) {
      $attrib_name  = shift (@unnamed_attribute_coercion_order);
      $attrib_value = $pair;
    } else {
      $attrib_name  = $1;
      $attrib_value = $2;
    }

    die "ei_create_hash_from_attributes: Bad 'name=value' pair: $pair"
      unless ($attrib_name ne "") && ($attrib_value ne "");
    $result{$attrib_name} = $attrib_value;
  }
  return \%result;
}

################################################################
# ei_merge_hashes
#
# Takes all the key/value pairs from the "source" hash and
# sets corresponding values in the "desitnation" hash.
#
# If there are any conflicts (i.e. if a key is set in both
# hashes) then the existing value in the destination hash
# is not changed:  The destination takes priority.
&Document_Args ("ei_merge_hashes", "
  *source              -Ref to source-hash
  *dest                -Ref to desitnation hash.
 ");
#
################################################################
sub ei_merge_hashes
{
 my ($arg) = &Validate_Args ("ei_merge_hashes", @_);

 foreach $src_key (keys (%{ $arg->{source} })) {
   next if $arg->{dest}{$src_key} ne "";
   $arg->{dest}{$src_key} = $arg->{source}{$src_key};
 }
}

################################################################
# e_signal
#
&Document_Args ("e_signal", "
  *name                                     - Formal HDL name of signal.
   width       = 1                          - Vector (bit) width of signal
   signal_type = wire (port,wire,register)  - What kind of thing?
 ");
#
# Returns (a reference to) an e_signal object.
#
################################################################
sub e_signal
{
  my ($arg)  = &Validate_Args ("e_signal", @_);

  my %result = (type => e_signal);
  &ei_merge_hashes ({ dest => \%result, source=> $arg });

  return \%result;
}

################################################################
# e_port
#
&Document_Args ("e_port", "
  *name                                - Formal port name.
   width      = 1                      - Vector (bit) width.
  *direction  (input,output,inout,I,O) - Direction (w/handy abbreviations!)
 ");
#
# Returns (a reference to) an e_port object.
#
################################################################
sub e_port
{
  my ($arg)= &Validate_Args ("e_port", @_);

  $arg->{direction} = "input"  if $arg->{direction} =~ /^\s*I\s*$/i;
  $arg->{direction} = "output" if $arg->{direction} =~ /^\s*O\s*$/i;

  my %signal_args = (signal_type => "port");
  &ei_merge_hashes ({ dest => \%signal_args, source => $arg });

  return &e_signal (\%signal_args);
}

################################################################
# e_port_from_description
#
# Returns (a reference to) an e_port object which gets created from
# a traditional "list_ports_for"-type description string.  This string
# consists of vertical-bar-delimited ('|') name=value pairs.
#
# If "name" is missing from any of the "name=value" pairs, then that
# is considered an "unnamed" attribute.  This routine will treat the
# first three unnamed attributes as: 
#
#       name, width, and direction   (in that order).
#
# If there are more than three unnamed attributes then you get an error.
#
################################################################
sub e_port_from_description
{
  my (@attribs) = (@_);
  my $port_args = &ei_create_hash_from_attributes ({
                     attribute_list         => \@attribs,
                     unnamed_coercion_order => ["name", "width", "direction"],
                     });

  my $e_port = &e_port ($port_args);
  return $e_port;
}

################################################################
# e_make_port_list
#
# Returns a list of e_port objects.  The e_ports are
# constructed from a traditional "List_Ports_For-" type string.
# 
# This routine works on either a single comma-delimited string, 
# or on a list of comma-delimited strings.
#
################################################################
sub e_make_port_list
{
  my (@description_strings) = (@_);
  my $big_string = join (",", @description_strings);
  my @description_list = split (/\s*,\s*/, $big_string);

  my @result = ();
  foreach $description (@description_list) {
    push (@result, &e_port_from_description ($description))
  }
  return @result;
}

################################################################
# ei_signal_hdl_declare
#
# Returns a string which is a valid declaration for the 
# given signal in the language of your choice.
#
################################################################
sub ei_signal_hdl_declare
{
  my ($sig, $language) = (@_);
  $language = "Verilog" if !$language;
  
  if ($language !~ /verilog/i) { ribbit (
     "Sorry.  Verilog is the only language curently supported.");
  }

  # Build-up result string as a list (for efficiency reasons).
  my @result = ("   ");

  # Type of signal:
  #
  if      ($sig->{signal_type} eq "port") {
    push (@result, $sig->{direction});
  } elsif ($sig->{signal_type} eq "register") {
    push (@result, "reg");
  } else {
    push (@result, "wire");
  }

  # Width-specifier, if any:
  #
  if ($sig->{width} > 1) {
    my $msb = $sig->{width} - 1;
    push (@result, " [$msb : 0]");
  }

  # In Verilog, the name comes last:
  #
  push (@result, " ", $sig->{name}, ";\n");

  return join ("", @result);
}

################################################################
# e_register
#
# Create a new "e_register"-type object.
#
# This is a bit like the old "&Delay" function from Vpp, except that
# the arguments and their names have been limited to more closely 
# resemble a good ol' D-type flip-flop:
#
&Document_Args ("e_register", '
  *q                          -Name of the Q-output, which is also taken
                              -   as the formal name of this register.
   d                          -Signal-name of D-input.  Not required
                              -   if "sync_set" or "sync_reset" inputs presnt.
   clk        =clk            - Name of clock-input signal.
   edge       =pos  (pos,neg) - Which edge of clock.
   enable                     - Name of clock-enable input signal, if any.
   reset                      - Name of asynchronous-reset input signal.
   set                        - Name of asynchronous-set input signal.
   sync_set                   - Name of synchronous-set input, if any.
   sync_reset                 - Name of synchronous-reset input, if any.
   set_value  =1              - Value associated with sync/async set inputs.
   width      =1              - Number of bits in vector.
 ');
#
# This function itself doesn't really do much.  It just creates
# an "e_register" data structure, which consists of little more 
# than the arguments you called it with.  But all this information
# is being saved for nefarious uses later on.
#
# You get (a reference to) the "e_register" object handed back to you.
# You will generally use the returned object as an element in a 
# e_module's "body" list.
#
################################################################
sub e_register
{
  my ($args) = &Validate_Args ("e_register", @_);

  my %result = ( type        => "e_signal",
                 signal_type => "register",
                 name        => $args->{"q"} );

  &ei_merge_hashes ({ dest => \%result, source => $args});
  return \%result;
}

################################################################
# e_assign
#
# Create a new "e_assign"-type object.
#
# An e_assign object contains a continuous-assignment statement.
# This is broken into two parts:  The "target", which is the
# name of a signal, and the "expression" which is, er, an expression.
# The expression is in Verilog syntax, and will almost certainly
# refer to other signal names.
#
# The e_assign statement also includes a width, which defaults to 1.
# The given width is the width of the target.  If your target is
# more than 1 bit wide, you -must- use the "width" argument.
#
&Document_Args ("e_assign", "
  *target             - The signal name being assigned-to
  *expression         - Verilog-syntax continuous-assignment expression.
  width        =1     - Vector (bit) width of target signal.
 ");
#
# As with most e_object creation-routines, this function doesn't
# do much.  It just remembers the arguments you set.  They will all be
# used later.
#
# This function returns (a reference to) an "e_assign" object.
# Typically, you would put this object onto a module's "body" list.
#
################################################################
sub e_assign
{
  my ($arg) = &Validate_Args ("e_assign", @_);

  my %result = ( type => "e_assign" );
  &ei_merge_hashes ({ dest => \%result, source => $args});
  return \%result;
}

################################################################
# ei_assign_hdl_implement
#
# Returns a string which is the HDL implementation of the 
# given e_assig object.
#
################################################################
sub ei_assign_hdl_implement
{
  my ($obj, $language) = (@_);
  my @result = ("   ");
  if ($language =~ /verilog/i) {
    push (@result,
          "assign ", $obj->{target}, " = ", $obj->{expression}, ";\n");
  } else {
    die ("e_assign: '$language' is not a supported language.\n");
  }
  return join ("", @result);
}

################################################################
# e_mux
#
# Creates an "e_mux"-type object.
#
&Document_Args ("e_mux", "
  *out               - Name of output signal
  *width             - Vector (bit) width of output signal.
  *table             - Condition --> result table, comma-delimited.
   type    = and_or  - Can be any of: priority, case, or numeric.
 ");
#
# Returns (a reference to) an e_mux object.  Typically, you would
# insert this object onto some module's "body" list.
#
################################################################
sub e_mux
{
  my ($arg) = &Validate_Args ("e_mux", @_);

  my %result = ( type => "e_mux" );
  &ei_merge_hashes ({ dest => \%result, source => $arg });
  return \%result;
}


################################################################
# e_module
#
# You specify the ports and the body.  The "ports" are supposed to
# be a reference to an "e_port_group" object, and the "body" is 
# just a list of "e_objects" of all descriptions.
#
&Document_Args ("e_module", "
  *name                   -Formal HDL module-name.
  *contents               -List (ref) of other e_object elements.
 ");
#
# This function does, in fact, return (a reference to) an "e_module"
# -type object.  But it also adds the created e_module to a global list of
# defined e_modules.  Most people just ignore the return-value,
# and then rely on other europa-internal smarts to get the module
# datastructure by-name when required.
#
################################################################
sub e_module
{
  my ($arg) = &Validate_Args ("e_module", @_);

  my %result =    ( type => "e_module");
  &ei_merge_hashes ({ dest => \%result, source => $arg });
  $europa_module_database{$result{name}} = \%result;
  return \%result;
}

################################################################
# ei_object_hdl_implement
#
# Returns a string which is the HDL "implementation" of the 
# given object.
#
# This is just a "dispatch" routine  which calls
# the "hdl_implement" function which is appropriate for each type.
#
# Note that "implementation" is distinct 
# from "declaration."  Signals (ports, wires) have no
# "implementation" per-se.  Registers have an implementation
# if they were created with a d-input (otherwise, they
# are "implemented" by some sort of procedural statement).
#
################################################################
sub ei_object_hdl_implement
{
  my ($obj, $language) = (@_);

  if      ($obj->{type} eq "e_signal") {
    return &ei_signal_hdl_implement ($obj, $language)

  } elsif ($obj->{type} eq "e_assign") {
    return &ei_assign_hdl_implement ($obj, $language)

  } elsif ($obj->{type} eq "e_mux") {
    return &ei_mux_hdl_implement ($obj, $language)
  }
}


################################################################
# ei_module_get_objects_of_type
#
# Returns all the "e_objects" of the given type in the 
# given module
# 
################################################################
sub ei_module_get_objects_of_type
{
  my ($mod, $type) = (@_);
  my @result = ();
  foreach $e_object (@{$mod->{contents}}) {
    next unless $e_object->{type} eq $type;
    push (@result, $e_object);
  }
  return @result;
}

################################################################
# ei_module_get_ports
#
# Returns a list of all the ports ("e_signal"-type objects
# with signal_type = "port") in the given module.
#
################################################################
sub ei_module_get_ports
{
  my ($mod) = (@_);
  my @result = ();
  foreach $e_object (&ei_module_get_objects_of_type ($mod, "e_signal")) {
    next unless $e_object->{signal_type} eq "port";
    push (@result, $e_object);
  }
  return @result;
}

################################################################
# ei_module_get_port_names
#
# Returns a list of the names (just the names) of every port-type
# object in the given module.
#
################################################################
sub ei_module_get_port_names
{
  my ($mod) = (@_);
  my @result = ();
  my @ports = &ei_module_get_ports ($mod);
  foreach $e_port (@ports) {
    push (@result, $e_port->{name});
  }
  return @result;
}


################################################################
# e_module_to_hdl_string
#
#
&Document_Args ("e_module_to_hdl_string", "
  module                         - NAME of module to write.
  e_module_ref                   - Ref to e_module object.
  language          = Verilog    - Verilog or VHDL.
 ");
#
# Returns a string which is an HDL implementation of the 
# given module -and all its submodules-.
#
################################################################
sub e_module_to_hdl_string
{
  my ($arg, $user_defined) = &Validate_Args ("e_module_to_hdl_string", @_);

  # Yes, it's true that this routine returns a string.
  # But we don't create the final return-value string until the
  # very end.  Instead, we internally maintain a -list- of substrings
  # which get smashed-together as the very-final step.  We do this
  # for efficiency reasons:  In Perl, growing a string is much less
  # efficient than growing a list.
  #
  my @result = ();

  # The user can give the module by name or by object-reference.
  # accept either.  But not both:
  #
  if ($user_defined->{module}      &&
      $user_defined->{e_module_ref} )  {
    ribbit ("
      e_write_module: module specified by both name and ref.
      Module can be specified either way, but not both.\n");
  }
  # If module given by name, look it up.
  if ($user_defined->{module}) {
    $arg->{e_module_ref} = $europa_module_database{$arg->{module}};
  }

  # We'll be referring to this a lot, so give it a short name:
  #
  my $mod = $arg->{e_module_ref};

  # Recurse down through all sub-instances and write them out:
  #
  # !!! No instaces yet!

  # Expand my own signals.
  #
  #&ei_module_elaborate ($mod);

  # Start with "module" statement and parenthesized list of ports:
  push (@result, "module ", $mod->{name}, "\n(\n");
  push (@result, join (",\n", &ei_module_get_port_names ($mod)));
  push (@result, "\n);\n");

  # Now add signal declarations:
  my @sig_list = &ei_module_get_objects_of_type ($mod, "e_signal");
  foreach $sig (sort verilog_object_order @sig_list) {
    push (@result, &ei_signal_hdl_declare ($sig, $arg->{language}));
  }

  # Next, write-out the implementation of every object:
  foreach $obj (sort verilog_object_order $mod->{contents}) {
    push (@result, &ei_object_hdl_implement ($obj, $arg->{language}));
  }

  push (@result, "endmodule\n");
  return (join ("", @result));
}


# All Perl modules must return a value.  Here's mine:
"John Wilkes Booth";












