#Copyright (C)1991-2003 Altera Corporation
#Any megafunction design, and related net list (encrypted or decrypted),
#support information, device programming or simulation file, and any other
#associated documentation or information provided by Altera or a partner
#under Altera's Megafunction Partnership Program may be used only to
#program PLD devices (but not masked PLD devices) from Altera.  Any other
#use of such megafunction design, net list, support information, device
#programming or simulation file, or any other related documentation or
#information is prohibited for any other purpose, including, but not
#limited to modification, reverse engineering, de-compiling, or use with
#any other silicon devices, unless such use is explicitly licensed under
#a separate agreement with Altera or a megafunction partner.  Title to
#the intellectual property, including patents, copyrights, trademarks,
#trade secrets, or maskworks, embodied in any such megafunction design,
#net list, support information, device programming or simulation file, or
#any other related documentation or information provided by Altera or a
#megafunction partner, remains with Altera, the megafunction partner, or
#their respective licensors.  No other licenses, including any licenses
#needed under any third party's intellectual property, are provided herein.
#Copying or modifying any file, or portion thereof, to which this notice
#is attached violates this copyright.







################################################################
# CLASS: em_instruction_spitter
#
# Herein lies mechanics of getting the next instruction.  
#
use europa_utils;
use e_mnemonic;
use strict;

sub make_instruction_spitter 
{
  my ($Opt, $project) = (@_);

  my @submodules = ( &make_wait_counter  ($Opt, $project),
                     &make_subinstruction($Opt, $project) );
  my $module = e_pipe_module->new 
      ({name        => $Opt->{name}."_instruction_scheduler",
        stage       => "Commit",
        project     => $project,
        pipe_clk_en => "commit",
     });
  my $marker = e_default_module_marker->new ($module);

  my $si_bits = e_mnemonic->subinstruction_bits();
  e_port->adds ([feed_new_instruction  => 1,        "out"],
                [subinstruction        => $si_bits, "out"],
                [is_subinstruction     => 1       , "out"],
                );

  # A bit tidier, because the names of the 
  foreach my $submodule (@submodules)
     {  e_instance->add ({module => $submodule->name()});   }

  # commit is like the local clock enable
  e_assign->add({lhs => e_port->new([commit => 1, "out"]),
                 rhs => "( feed_new_instruction             ) && 
                         (~is_subinstruction                ) &&         
                         ( pipe_run                         )  ", });

  # NOTE:  This register (and others) appeared to violate one of the 
  #        pipeline assumptions:  Its 'enable' was always 1--therefore
  #        it did not appear to heed "pipe_freeze."
  #        I am guessing this would be a potential source of bugs
  #        once we start feeding instructions consecutively.
  # 
  #        I'm going on a "pipe_freeze" Jihad here, so that we can 
  #        stave-off future trouble.  When the pipe freezes, all
  #        sequential activity must, MUST stop.  
  #
  #        This has coding-style consequence.  When sequential things 
  #        are phrased as complex "e_process" if/then/else objects,
  #        it becomes exquisitly difficult to verify that the state
  #        doesn't advance when you don't want it to--and awkward to 
  #        code even when you do.  Instead, an equivalent expression
  #        of combinatorial logic (which can do whatever it wants) 
  #        feeding a single, concentrated sequential nugget (a register
  #        with the proper clock-enable) guarantees correctness in a 
  #        way that's both tidy and easy-to-verify by inspection.
  #
  
  # Say "is_neutrino" when either:
  #        * what is currently presented as the instruction is not valid data
  #        * We're holding-off on "feeding_new_instruction"
  #
  e_register->add ({out         => e_port->new([is_neutrino => 1, "out"]),
                    in          => "p1_is_neutrino",
                    async_value => 1,
                    enable      => "pipe_run",
                 });

  e_assign->add ({lhs => "p1_is_neutrino", 
                  rhs => "instruction_not_ready && (~hold_for_hazard)",
                 });

  e_assign->add ({lhs => "instruction_not_ready", 
                  rhs => "(~feed_new_instruction                 ) | 
                          (d1_instruction_fifo_read_data_bad   &
                            ~is_subinstruction                   ) ",
                });

  ################
  #
  # Note: I think this logic introduced one too many 
  # levels of delay into the skip/cancel logic.  "do_skip" was an 
  # input to a register (p1_is_cancelled) which, in turn, was an input
  # to the "is_cancelled" register.  Thus, when "do_skip" went true, 
  # it took two clocks (well, actually, one raw-clock plus one 
  # "commit" cycle) before "is_cancelled" went true.  This worked in 
  # the simple case where instructions take many clocks, but I don't know if
  # it would work when we start feeding instructions consecutively.
  # 
  # I don't really -want- to change this basic logic right now, because I just
  # want to get things working.  But I do need to add a thing that
  # cancels branch-delay slots on request.  While I'm breaking stuff,
  # I guess there's no additional harm in fixing it at the same time.
  #
  # Cancellation sources--when do we cancel?
  # Well, we cancel on account of two distinct events:
  # 
  #   1) Because the instruction is -skipped-.
  #   2) Because the predecessor -always- cancels whatever follows it.
  #
  e_register->add 
      ({out        => e_port->new([is_cancelled_from_commit_stage => 1,"out"]),
        sync_set   => "skip_acknowledge ||
                       cancel_delay_slot_acknowledge",
        sync_reset => "~p1_is_neutrino",
        priority   => "set",
        enable     => "commit",
       });

  #
  # Case (2) is simpler, so let's consider that first.  There are 
  # two nasty little instructions (TRAP and TRET) which are
  # types of jump-instructions that have no
  # branch-delay slot.  As it happens, the easiest way to implement
  # this behavior is to just cancel whatever comes next. So--if we
  # see a TRAP or TRET, we just set a bit, which is cleared when
  # the next instruction is properly and duly cancelled.  As it
  # happens, we get an input from the committed pipe that tells us 
  # when to do this ("cancel_next_instruction")
  #
  # Let's do that right now, before we forget about it.  Use a request/
  # acknowlege protocol, so no cancellation-requests get dropped, and no
  # extra cancellations are generated.
  #
  if ($Opt->{support_interrupts})
  {
     e_register->add ({out        => "dont_forget_to_cancel_delay_slot",
                       sync_set   => "do_cancel_next_instruction",
                       sync_reset => "(commit && (~d1_instruction_fifo_read_data_bad)) && cancel_delay_slot_acknowledge",
                       priority   => "reset",
                       enable     => "pipe_run",
                    });
     e_assign->add ({lhs => "cancel_delay_slot_acknowledge", 
                     rhs => "do_cancel_next_instruction || 
                             dont_forget_to_cancel_delay_slot"});
  } else {
     # No traps:
     e_assign->add (["cancel_delay_slot_acknowledge", "1'b0"]);
  }

  ################
  # Skips.
  # 
  # Now then.  Reason to cancel #1 is more subtle.  For right now, 
  # we always hold instructions until the outcome of the "skip" is
  # known, and cancel them right here (for that matter, we hold
  # all instructions whether they are after a SKP or not).  But, in
  # the future, we may not hold -all- instructions after a skip,
  # only "scary" ones.  If we release a maybe-skipped instruction into the 
  # pipe (later) for on-the-spot cancellation, we want to ignore the
  # associated "do_skip" pulse--we'll leave that dirty deed in the hands of 
  # the pipeline.
  #
  e_assign->adds
      (["waiting_for_skip_outcome", "1'b1"                               ],
       ["qualified_skip_request",   "waiting_for_skip_outcome && do_skip"],);


  # So. When we are waiting for a skip-outcome, we want to set a 
  # bit whenever "qualified_do_skip" goes true.  This tells us to cancel the 
  # next outgoing instruction.  But when do we clear the bit?
  # We clear it after we cancel the next instruction--unless the cancelled
  # instruction is a PFX, in which case we leave this bit set--so that its
  # successor gets skipped, too.
  # If the skip causes a cancel right away, this bit will not get set.
  #
  e_register->add 
      ({out        => "dont_forget_to_skip",
        sync_set   => "qualified_skip_request",
        sync_reset => " commit            && 
                        ~p1_is_neutrino   &&
                        skip_acknowledge  &&
                        ~do_iPFXx
                        ",
        priority   => "reset",
        enable     => "pipe_run",
     });
  e_assign->add (["skip_acknowledge", "qualified_skip_request || 
                                       dont_forget_to_skip"    ]);

  # this should be moved over to a decode file/module
  e_control_bit->add (qw(do_iPFXx));

  return $module;
} # end of new_contents

sub make_wait_counter
{
  my ($Opt, $project) = (@_);

  my $module = e_pipe_module->new 
      ({name        => $Opt->{name}."_wait_counter",
        stage       => "Commit",
        project     => $project,
        pipe_clk_en => "commit",
     });
  my $marker = e_default_module_marker->new ($module);
  my $depth = 3;

  my @delay_constant;
  for (my $i=0; $i<=$depth; $i++) {
    $delay_constant[$i] = $depth."'b". ("1"x($depth-$i)) . ("0"x$i);
    #print "delay of $i is ".$delay_constant[$i]."\n";
  }
  e_port->add ([feed_new_instruction => 1, "out"]);
  e_signal->add (["instruction_delay",        $depth]),

  e_mux->add ({
     out  => "instruction_delay",
     table   => 
      ["(d1_instruction_fifo_read_data_bad) &  
        (commit                            )"  => $delay_constant[0], 
       "wait_once_after"                       => $delay_constant[1],
       "must_run_to_completion"                => $delay_constant[$depth],
       ],
      default                                  => $delay_constant[0],
     });
  
  my $run_to_completion_list 
      = $Opt->{fast_save_restore}                    ? 
        qq (=do_iSKPx WRCTL    RDCTL     TRAPx TRET) : 
        qq (=do_iSKPx WRCTL SAVE RESTORE TRAPx TRET) ; 

  e_control_bit->adds 
   ([must_run_to_completion => $run_to_completion_list                     ],
    [op_jmpcall             => qq (JMP CALL)                               ],
    qw(do_iSKPx),
   );

  my @wait_once_reasons = ("(op_jmpcall)");

  if ($Opt->{fast_save_restore}) 
  { 

     # If the last instruction was a neutrino, then it didn't write
     # any registers, so we can't run afoul of data-forwarding.
     #
     # Note that, in the conventional application of SAVE and RESTORE
     # in compiled code, the preceding cycle will always be a neutrino (!).
     # For SAVEs, this is because a SAVE is generally the first instruction
     # in a subroutine--which you jumped to.  Jumping to things incurs
     # a branch-delay penalty, which means that you always see a bunch 
     # of neutrinos in advance of arriving at your destination.  Good 
     # news for our friend SAVE.
     #
     # For RESTORE, a different mechanism saves the day.  RESTOREs are
     # conventionally issued in the branch-delay slot of RET (JMP) 
     # op-codes on your way back from a subroutine.  Sounds bad
     # (non-neutrino-like) unless you remember that a neutrino is
     # -always- inserted after all JMP instructions.  How handy.
     #
     # Thus, the following obviously-suboptimal logic is nearly always
     # optimal.  The exceptions arise when you use SAVE and RESTORE
     # for some unconventional purpose--whence they might end up taking
     # two clocks.  Serves you right, you mutant.
     #
     push (@wait_once_reasons, 
           "(op_save_restore && (trap_if_save    || 
                                 trap_if_restore ||
                                 ~is_neutrino      ) )",

           );
     
     # Tie-off these signals sensibly if there are no interrupts.
     #
     if (!$Opt->{support_interrupts}) {
        e_assign->adds (["trap_if_save",    "1'b0"],
                        ["trap_if_restore", "1'b0"], );
     }

     e_control_bit->add  ([op_save_restore  => qq (SAVE RESTORE) ]);
  }

  e_assign->add (["wait_once_after", join (" || ", @wait_once_reasons)]);

  # note: should change reset to async reset.
  e_shift_register->add ({
        name            => "wait_counter_shift_register",
        serial_in       => "1'b1",
        serial_out      => "wait_counter_output",
        shift_length    => $depth,
        direction       => "LSB-first",
        parallel_in     => "instruction_delay",
        load            => "commit",
        enable          => "pipe_run           && 
                            (~|subinstruction) && 
                            ~hold_for_hazard    ",
     });

  # We don't use "do_force_trap" if interrupt-support is disabled:
  #
  my @feed_instruction_terms = 
      ("((wait_counter_output) | (|subinstruction))",
       "( ~hold_for_hazard                        )");
  push(@feed_instruction_terms, 
       "( ~do_force_trap                        )") 
      if ($Opt->{support_interrupts}); 

  e_assign->add 
      ({lhs => "feed_new_instruction", 
        rhs => &and_array (@feed_instruction_terms),
       });

  return $module;
}

sub make_subinstruction
{
  my ($Opt, $project) = (@_);

  my $module = e_pipe_module->new 
      ({name        => $Opt->{name}."_subinstruction_unit",
        stage       => "Commit",
        project     => $project,
        pipe_clk_en => "commit",
     });
  my $marker = e_default_module_marker->new ($module);

  my $si_bits = e_mnemonic->subinstruction_bits();
  return unless $si_bits > 0;

  e_port->add (["d1_instruction_fifo_out" => 16, "in"]);
    
  e_assign->add ({lhs => "subcount_en", 
                  rhs => "(feed_new_instruction & pipe_run)",
               });

  e_register->add 
      ({name      => "subinstruction_counter",
        out       => e_port->new ([subinstruction => $si_bits, "out"]),
        in        => "p1_subinstruction",
        enable    => "subcount_en",
     });

  e_mux->add ({lhs     => [p1_subinstruction => $si_bits],
               table   => ["(is_subinstruction)" => "subinstruction - 1"], 
               default  =>                    "p1_subinstruction_load_value",
            });

  my $load_value_mux = e_mux->add 
      ({lhs      => [p1_subinstruction_load_value => $si_bits],
        default  => "$si_bits\'b0",
      });
  # bad instructions never have subinstructions, no matter what they decode
  # to.  Note this disqulifies this mux from ever being and-or.
  $load_value_mux->add_table (d1_instruction_fifo_read_data_bad => 0);

  # p1_instruction can only be two things: d1_instruction_fifo_out or "TRAP".  
  # we can speed things up, then by explicitly taking the "TRAP" case, and
  # decoding d1_instruction_fifo_out (which is ready a little
  # sooner). 
  # 
  # Note:  I don't actually think this accomplishes anything, since
  #        you make the subinst-load-value mux (above) have an extra
  #        term in order to decode off something that's ready sooner.
  #        Net-0 levels of logic saved.   Still, it doesn't appear to
  #        hurt, so I'll leave it alone for now.
  #
  if ($Opt->{support_interrupts}) 
  {
     my $trap_mnem = e_mnemonic->get_mnemonic_by_name("TRAP")
         or &ribbit ("Holy moley!  running without TRAPs!");
     my $forced_trap_SIs = $trap_mnem->num_subinstructions()-1;
     $load_value_mux->add_table (force_trap_acknowledge   => $forced_trap_SIs);
  }

  # Loop through all the mnemonics, find the ones that have sub-instructions,
  # and add a mux-table entry for them:
  #
  foreach my $mnem (e_mnemonic->get_mnemonic_list())
  {
     # Only consider ones with actual subinstructions.
     next unless $mnem->num_subinstructions() > 1;
     #my $selector = $mnem->make_match_expression("p1_instruction");
     my $selector = $mnem->make_match_expression("d1_instruction_fifo_out");
     $load_value_mux->add_table ($selector => $mnem->num_subinstructions()-1);
  }


  e_assign->add ({lhs => [is_subinstruction => 1, 1],
                  rhs => "(|(subinstruction)) & (~is_neutrino)", 
               });

  return $module;
}

"p'toey!";

