#Copyright (C)1991-2003 Altera Corporation
#Any megafunction design, and related net list (encrypted or decrypted),
#support information, device programming or simulation file, and any other
#associated documentation or information provided by Altera or a partner
#under Altera's Megafunction Partnership Program may be used only to
#program PLD devices (but not masked PLD devices) from Altera.  Any other
#use of such megafunction design, net list, support information, device
#programming or simulation file, or any other related documentation or
#information is prohibited for any other purpose, including, but not
#limited to modification, reverse engineering, de-compiling, or use with
#any other silicon devices, unless such use is explicitly licensed under
#a separate agreement with Altera or a megafunction partner.  Title to
#the intellectual property, including patents, copyrights, trademarks,
#trade secrets, or maskworks, embodied in any such megafunction design,
#net list, support information, device programming or simulation file, or
#any other related documentation or information provided by Altera or a
#megafunction partner, remains with Altera, the megafunction partner, or
#their respective licensors.  No other licenses, including any licenses
#needed under any third party's intellectual property, are provided herein.
#Copying or modifying any file, or portion thereof, to which this notice
#is attached violates this copyright.







################################################################
# CLASS: em_instruction_fetch
#
# Herein lies mechanics of requesting the next instruction.  
#

use europa_utils;
use strict;

sub make_instruction_fetch 
{
   my ($Opt, $project) = (@_);
   
   my @submodules =(&make_Instruction_Address_Request($Opt, $project),
                    &make_target_address_unit        ($Opt, $project),
                    );
   
   my $module = e_module->new ({name => $Opt->{name}."_instruction_fetch"});
   $project->add_module ($module);      
   my $marker = e_default_module_marker->new ($module);

   $module->add_attribute (auto_dissolve => "FALSE");

   foreach my $submod (@submodules) 
     { e_instance->add({module => $submod->name()}); }
   

   #########################
   # create outgoing signals to the instruction memory
   #
   e_assign->add 
       ({lhs => e_port->new ([ic_address => $Opt->{i_Address_Width}, "out"]),
         rhs => "{pc, {1'b0}}", 
      });

   # target_address also needed by instruction_receive.  This exports it.
   e_port->add ([target_address => ($Opt->{i_Address_Width} - 1) , "out"]);

   return $module;
} 

################################################################
# make_target_address_unit
#
# A big production for a trivial task.
#
# The committed pipeline will, occasionally, decide to jump.  Or
# branch.  Or trap.  There are two separate signals that it will
# set which request that the PC be set to a new value:  "do_jump"
# and "do_branch"  ("do_jump" is used for both jumps and traps). 
# When the committed pipeline says "do_jump" or "do_branch," it 
# also says where to go.  It produces two signals: "jump_target_address"
# and "branch_target_address".  
#
# From this simple set of signals, figuring-out when to jump and where
# to jump to shouldn't be very hard.  But there are two gotchas:
# First, this is done in two places:  Once for the actual outgoing
# fetch-address (the ostensible PC, as-it-were), and once again
# for the "next_instruction_address" presented to the pipeline.  
# These two signals are deeply different, so don't think for a minute
# they can be shared.  But they both do the same "if do_jump then 
# jump_target_address else branch_target_address" calculation, thereby
# wasting valuable LEs.
#
# The other (big) gotcha is that one, or the other, or both of the
# running PCs (PC-proper and next_instruction_address) may -not be ready-
# to load a new value when the committed pipeline says "jump."  This
# is all subtle and tricky, but you can easily see that there is one
# such case: when the instruction-master is being told to "ic_wait" 
# by the bus.  We can't load the PC with the new jump/branch_target address!  
# We have to "save" the new address and wait until it's OK to change
# ic_address.  There are several such circumstances that arise where
# we need to "remember" the last jump/branch_target address and use it
# at the next safe opportunity.  By consolidating all this 
# muxing and saving into one tidy module, we can share the common 
# computation and save LEs.  Yay.
#
# To further reduce LE usage (by PC-width LEs, which ain't that much)
# we phrase this as literal LEs.  This overcomes Leo's inscrutable
# reluctance to use both the "regout" and "combout" pins on an LE.
#
#################################################################
sub make_target_address_unit
{
   my ($Opt, $project) = (@_);

   my $module = e_module->new ({name => $Opt->{name}."_target_address"});
   $project->add_module ($module);      
   my $marker = e_default_module_marker->new ($module);

   ################
   # Branch-target adder now lives in here.  Why?  Because
   # this is where the result gets used, and we want to logicLock the 
   # adder and its subsequent logic into the same region.
   #
   my $PC_bits = $Opt->{i_Address_Width} - 1;
   e_assign->add 
       ({lhs    => [branch_target_address => $PC_bits],
         rhs    => "branch_base + signed_branch_offset",
       });

   my $pc_width = $Opt->{i_Address_Width} - 1;
   my $behavioral_tag = $Opt->{use_lcells} ? "simulation" : "normal";

   ##     d d       2   2    
   ##     o o       :   :   ?   
   ##     B A B A   1   1*          * = (with a-input inverted)   
   ##  -----------------------
   ##     d c b a                  # LE inputs
   ## 
   ##     0 0 0 0   1   1   1   0   0
   ##     0 0 0 1   1   1   1   0   0
   ##     0 0 1 0   1   1   1   0   1
   ##     0 0 1 1   1   1   1   1   1
   ##     0 1 0 0   0   1   1   1   0
   ##     0 1 0 1   1   0   0   0   1
   ##     0 1 1 0   0   1   1   1   0
   ##     0 1 1 1   1   0   0   1   1
   ##     1 0 0 0   0   0   0   0   0
   ##     1 0 0 1   0   0   0   0   0 
   ##     1 0 1 0   1   1   1   0   1 
   ##     1 0 1 1   1   1   1   1   1
   ##     1 1 0 0   0   0   0   1   0
   ##     1 1 0 1   0   0   0   0   1
   ##     1 1 1 0   0   0   0   1   0 
   ##     1 1 1 1   0   0   0   1   1
   ##                           
   ##               F   F   F   8   C
   ##               A   5   5   D   A
   ##               C   C   C   8   C
   ##               0   0   0   D   A


   if ($Opt->{use_lcells}) 
   {
      foreach my $i (0..$pc_width-1)
      {
         if ($Opt->{use_apex_lcells}) 
         {
            e_lcell->add 
             ({name          => "target_address_$i",
               tag           => "synthesis",
               parameter_map => {lut_mask       => "ACAC",
                                 operation_mode => "normal",
                                 id             => $i,
                                },
               port_map      => {dataa   => "jump_target_address    [$i]",
                                 datab   => "branch_target_address  [$i]",
                                 datac   => "do_jump",  
                                 datad   => "1'b0",  
                                 regout  => "last_target_address    [$i]",
                                 combout => "current_target_address [$i]",
                                 
                                 ena    => "pipe_run && (do_jump | do_branch)",
                                 clk    => "clk",
                                 aclr   => "~reset_n",
                                 },
               });
         } else {
            # Stratix
            e_lcell->add 
             ({name          => "target_address_$i",
               tag           => "synthesis",
               parameter_map => {lut_mask       => "ACAC",
                                 operation_mode => "normal",
                                 id             => $i,
                                 lpm_type       => "yeager_lcell",
                                },
               port_map      => {dataa   => "jump_target_address    [$i]",
                                 datab   => "branch_target_address  [$i]",
                                 datac   => "do_jump",  
                                 datad   => "1'b0",  
                                 regout  => "last_target_address    [$i]",
                                 combout => "current_target_address [$i]",
                                 
                                 ena    => "pipe_run && (do_jump | do_branch)",
                                 clk    => "clk",
                                 aclr   => "~reset_n",
                                 },
               module        => "yeager_lcell_hidden_from_quartus",
               });
         }
      }
   }
   
   e_assign->add 
       ({lhs => [current_target_address => $pc_width],
         rhs => "do_jump ? jump_target_address : branch_target_address",
         tag => $behavioral_tag,
        });

   e_register->add 
       ({out    => [last_target_address => $pc_width],
         in     => "current_target_address",
         enable => "pipe_run && (do_jump || do_branch)",
         tag    => $behavioral_tag,
       });

   # Note: This mux is not just behavioral--it's present even in the 
   #       literal-LE implementation.
   e_assign->add 
       ({lhs   => e_port->new ([target_address => $pc_width, "out"]), 
         rhs   => "(do_jump || do_branch)   ? 
                    current_target_address  : 
                    last_target_address     ",
       });
   
   return $module;
}

#########################
# Instruction Address Request
# 
# "So whatcha whatcha whatcha want? (Whatcha want?)"
#
# Note that the PC-making code is cancel-independent. It just clocks along,
# requesting addresses no matter what the CPU is doing. 
#
#########################

sub make_Instruction_Address_Request
{
   my ($Opt, $project) = (@_);

   my $module = e_module->new ({name => $Opt->{name}."_address_request"});
   $project->add_module ($module);      
   my $marker = e_default_module_marker->new ($module);

   my $pc_width = $Opt->{i_Address_Width} - 1;
   my $reset_instr_address = $$Opt{"Reset_Address"} >> 1;

   # The PC itself changes only when it's time to increment it, or if it is
   # time to change the direction altogether with a jump, branch, or trap.
   e_register->add ({out         => e_port->new ([pc => $pc_width, "out"]),
                     in          => "next_pc",
                     enable      => "pc_clken",  # see definition below
                     async_value => $reset_instr_address,
                  });
 
   e_assign->add ({lhs => [next_pc_plus_one => $pc_width],
                   rhs => "pc + 1", 
                });
 
   e_assign->add ({
      lhs => [pc_clken => 1],
      rhs => "(ic_read | p1_flush) & ~ic_wait",
   });

   # we want next_pc to be pc + 1, unless we're about to jump or branch.
   e_assign->add 
       ({lhs => [next_pc => $pc_width],
         rhs => "(do_jump | do_branch | 
                 (remember_to_flush & ~waiting_for_delay_slot)) ? 
                    target_address                              :
                    next_pc_plus_one                        ",
       });
 
   e_assign->add (["nonsequential_pc", "(do_branch | do_jump) & pipe_run"]); 
 
   # The instruction cache needs to know the next address early because it
   # is a synchronous memory.
   # ic_address_m1 IS EXACTLY the previous ic_address --
   # The cache assumes this address is real and does NOT compare it to ic_addr
   if ($Opt->{cache_has_icache}) {
    e_assign->add ({
      lhs => e_port->new ([ic_address_m1 => $Opt->{i_Address_Width}, "out"]),
      rhs => "{next_pc, {1'b0}}", 
    });
    e_assign->add ({
      lhs => e_port->new ([ic_address_clken => 1, "out"]),
      rhs => "pc_clken", 
    });
   }

   e_register->add ({out    => e_port->new ([ic_flush => 1, "out"]),
                     in     => "p1_flush",
                     enable => "~ic_wait",
                  });
 
   # p1_flush is the "ready to flush" signal, which will be gated onto the
   # avalon bus next time it's not waiting.  you're only "ready" if the branch
   # delay slot has already been fetched, and is sitting somewhere in the
   # CPU.  If you don't find the branch delay slot right away,  you remember
   # that you were going to flush and wait for the branch delay slot to come
   # in.  
   #
   # NOTE: the logic below does not account for the ic_wait.  p1_flush will
   # not become an ic_flush until the present wait is over, but that is not
   # accounted for here.  Instead, anything that does not expect a
   # persistent, multi-clock p1_flush must be qualified with (~ic_wait). 
   #
   e_assign->add
     ({lhs => e_port->new([p1_flush => 1, "out"]),
       rhs => "(nonsequential_pc  & ~d1_instruction_fifo_read_data_bad )  | 
               (remember_to_flush & ~waiting_for_delay_slot            )  ",
        });
 
   # you must remember to flush when you're told to, but can't service it
   # right now. 
   # 
   # DANGER:  These registers were created without any clock-enable.
   #          I've given them the clock-enable "1'b1" because that's 
   #          what they've had so far, and it seems to work.  BUT:
   #          I'm pretty sure we'll pay for our sins later.  I think
   #          the clock-enable to this register, and the one below,
   #          should be "pipe_run."  REMEMBER:  All registers should
   #          have this stinking clock-enable.
   #
   e_register->add
       ({out        => "remember_to_flush",
         sync_set   => "nonsequential_pc & 
                       (d1_instruction_fifo_read_data_bad | ic_wait)",
         sync_reset => "p1_flush & ~ic_wait",
         priority   => "reset",
         enable     => "1'b1",        # We'll be sorry for this...
       });
 
   # we set waiting_for_delay_slot once... right at the time we're told to
   # branch.  Either it's there or it isn't.  If it's not there, wait until
   # it comes in.
   # the sync_reset has an implied term: "& instruction_fifo_read".  But of
   # course it will be reading every clock because there's a bad instruction
   # in the "on deck circle"... which is why waiting_for_delay_slot is set.
   #
   e_register->add
       ({out        => "waiting_for_delay_slot",
         sync_set   => "nonsequential_pc && d1_instruction_fifo_read_data_bad",
         sync_reset => "~instruction_fifo_read_data_bad",
         priority   => "reset",
         enable     => "pipe_run",        
      });
 
   return $module;
}

1; # every perl module ends with 1.





