
################################################################
# CLASS: e_vfifo
# Validated Synchronous Fifo (single clock), with flowthru output.
# Independent Valid Vector carries an extra bit for each fifo entry to
# determine validity.  This is useful with the cpu's i_flush signal to
# mark data entries as 'flushed'.  Note that the behavior of invalidate
# is different based on 'enable': when the vfifo is enabled, invalidate
# only effects the validation vector; when vfifo is NOT enabled,
# invalidate forces the number of entries to be 0 (if not writing) or 1
# (if writing).  In this way, the vfifo can count how many outstanding
# posted reads are expected to be returned by the system...
# Also Note that Rd + Wr do not happen together during invalidate when
# the vfifo is disabled, and there is no logic to deal with it.
################################################################


=head1 NAME

e_vfifo - description of the module goes here ...

=head1 SYNOPSIS

The e_vfifo class implements ... detailed description of functionality

=head1 METHODS

=over 4

=cut

package e_vfifo;

@ISA = qw(e_module);

use e_module;
use e_port;
use e_parameter;
use europa_utils;

use strict;
my %all_unique_names = ();

my %fields = (
  name_stub        => "",
  data_width       => 1,
  depth            => undef,
  implement_as_esb => 1,
  almost_full_warn => 1, # how many entries before full do I assert almost_full
  almost_empty_warn=> 1, # how many entries before mt do I assert almost_empty
);

my %pointers = (
  unique_names => \%all_unique_names,
);

&package_setup_fields_and_pointers(__PACKAGE__,
                                   \%fields,
                                   \%pointers);

################################################################################

=item I<new()>

Object constructor

=cut

sub new
{
  my $this = shift;
  my $self = $this->SUPER::new(@_);

  # Go off and make the ports, etc.
  $self->make_vfifo();
    
  return $self;
}

### NB: &one_hot_encoding now in europa_utils.pm!

# Here, one hot encoding is used to build a $depth'b0000n0000 vector to
# use to or with 'shift_vector' in case (valid_conrol == 3'd3)!

################################################################################

=item I<make_vfifo()>

method description goes here...
...remember: there must be a newline around each POD tag (e.g. =item, =cut etc)!

=cut

sub make_vfifo
{
  my $this = shift;

  my $name = $this->name_stub() . "_vfifo_module";
  while ($this->{unique_names}->{$name}) {
    $name =~ s/_(\d+)$//;
    my $digit = $1 || 0;
    $digit++;
    
    $name .= "_" . $digit;
  }

  $this->{unique_names}->{$name} = 1;
  $this->name($name);

  # Fifo width is whatever you want:
  my $width = $this->data_width();
  # Fifo Depth must be set -- careful: deeper means more LE's!
  my $depth;
  if (defined ($this->depth())) {
      $depth = $this->depth();
      ribbit ("$this->{name}: Minimum Depth for a validated fifo is 2; ".
	      "you wanted $depth") if ($depth < 2);
  } else {
      ribbit ("$this->{name}: Depth for a validated fifo is not set!")
	  if ($depth < 2);
  }
  my $fifo_address_bits = ceil(log2($depth));
  # Number of entries needs an overflow bit to count to '3'b100'
  my $entries_bits = $fifo_address_bits + 1;
  
  my $almost_empty = $this->almost_empty_warn();
  if ($almost_empty > $depth) {
      goldfish("almost_empty_warn set beyond $depth; forced to $depth");
      $almost_empty = $depth;
  }
  my $almost_full = $this->almost_full_warn();
  if ($almost_full > $depth) {
      goldfish("almost_full_warn set beyond $depth; forced to $depth");
      $almost_full = $depth;
  }

  # Ports.
  $this->add_contents
      (
       e_port->new(["clk"]),
       e_port->new(["reset_n"]),
       e_port->new(["invalidate"]),
       e_port->new(["enable"]),
       e_port->new(["wr"]),
       e_port->new(["rd"]),
       e_port->new(["wr_data",  $width]),
       e_port->new(["rd_data",  $width, "output"]),
       e_port->new(["almost_empty", 1, 'output',]),
       e_port->new(["almost_full", 1, 'output',]),
       e_port->new(["empty", 1, 'output',]),
       e_port->new(["full", 1, 'output',]),
       e_port->new(["valid", 1, 'output',]),
       );
  
  # Internal FIFO control signals.
  $this->add_contents
      (
       e_signal->new({
	   name => "rd_address",
	   width => $fifo_address_bits,
       }),
       e_signal->new({
	   name => "wr_address",
	   width => $fifo_address_bits,
       }),
       e_signal->new({
	   name => "rdwr",
	   width=> 2,
       }),
       e_signal->new({
	   name => "entries",
	   width => $entries_bits,
       }),
       # make a signal with one valid bit for each entry.
       e_signal->new({
	   name => "valid_vector",
	   width => $depth,
       }),
       );
  
  my @one_hot_vectors = &one_hot_encoding($depth);

  my %rd_data_mux_hash = (default => [rd_data => "entry_0"]);
  my %wr_data_mux_hash = (default => []);
  my %seven_mux_hash = (default => ["valid_vector" => $depth."'b0"]);
  my %five__mux_hash = (default => ["valid_vector" => $depth."'b0"]);
  my %three_mux_hash = 
      (default => 
       ["valid_vector" => "shift_vector | ".$depth."'b0"]);
  my %one___mux_hash = (default => ["valid_vector" => $depth."'b0"]);
  
  for (my $i = 0; $i < $depth; $i++) {
      # create only the necessary signals
      $this->add_contents
	  (
	   e_signal->new
	   ({name => "entry_$i",width=> $width})
	   );
      # create the output read mux entries for them
      $rd_data_mux_hash{$i} = [rd_data => "entry_$i"];
      # create the input write mux entries for them
      $wr_data_mux_hash{$i} = ["entry_$i" => "wr_data"];
      # create the valid vector during simultaneous push&pop
      $seven_mux_hash{$i} = ["valid_vector" => $one_hot_vectors[$i] ];
      $five__mux_hash{$i} = ["valid_vector" => $one_hot_vectors[$i] ];
      $three_mux_hash{$i} = 
	  ["valid_vector" => "shift_vector | ".$one_hot_vectors[$i] ];
      $one___mux_hash{$i} = 
          ["valid_vector" => "valid_vector | ".$one_hot_vectors[$i] ];
  }

  $this->add_contents
      (
       e_assign->new({
	   lhs => 'rdwr',
	   rhs => "{rd, wr}",
       }),
       e_assign->new({
	   lhs => 'full',
	   rhs => "(entries == $depth)",
       }),
       e_assign->new({
	   lhs => 'almost_full',
	   rhs => "(entries >= ". ($depth - $almost_full) .")",
       }),
       e_assign->new({
	   lhs => 'empty',
	   rhs => "(entries == 0)",
       }),
       e_assign->new({
	   lhs => 'almost_empty',
	   rhs => "(entries <= ". $almost_empty .")",
       }),
       );

  # my $lang = $this->_project->language();
  # my $i_am_verilog;		# used in $wr_case, far below!
  # if ($lang =~ /verilog/i)
  # {
  #     $i_am_verilog = 1;
  # }
  # else
  # {
  #     $i_am_verilog = 0;
  # }
  
  # mux rd_data output
  $this->add_contents
      (
       e_process->new({
	   clock   => "",
	   contents=> [
		       e_case->new({
			   switch => "rd_address",
			   parallel=> 1,
			   # full => 1,
			   # default_sim => $i_am_verilog,
			   contents=> {%rd_data_mux_hash},
		       }),
		       ],
       }),
       );
  
  my $rw_case = e_case->new ({
      switch => "rdwr",
      parallel=> 1,
      full => 1,
      contents=> {
	  1 => [
		e_if->new({
		    comment => " Write data",
		    condition => "!full",
		    then => [
			     e_assign->new({
				 lhs => "entries",
				 rhs => "entries + 1",
			     }),
			     e_assign->new({
				 lhs => "wr_address",
				 rhs => "(wr_address == ".($depth - 1).") ? ".
				     "0 : (wr_address + 1)",
			     }),
			     ],
		}),
		e_if->new({
		    comment => " Disabled invalidate handling",
		    condition => "!enable & invalidate",
		    then => [
			     e_assign->new({
				 lhs => "entries",
				 rhs => $entries_bits."'h1",
			     }),
			     e_assign->news
			     (
			      ["wr_address" => $fifo_address_bits."'h1"],
			      ["rd_address" => $fifo_address_bits."'h0"],
			      ),
			     ],
		}),
		],
	  2 => [
		e_if->new({
		    comment => " Read data",
		    condition => "(!empty)",
		    then => [
			     e_assign->new({
				 lhs => "entries",
				 rhs => "entries - 1",
			     }),
			     e_assign->new({
				 lhs => "rd_address",
				 rhs => "(rd_address == ".($depth - 1).") ? ".
				     "0 : (rd_address + 1)",
			     }),
			     ],
		}),
		],
	  3 => [
		e_assign->new({
		    lhs => "wr_address",
		    rhs => "(wr_address == ".($depth - 1).") ? ".
			"0 : (wr_address + 1)",
		}),
		e_assign->new({
		    lhs => "rd_address",
		    rhs => "(rd_address == ".($depth - 1).") ? ".
			"0 : (rd_address + 1)",
		}),
		],
	  default => [
                e_if->new({
		    comment => " Disabled invalidate handling",
		    condition => "!enable & invalidate",
		    then => [
			     e_assign->new({
				 lhs => "entries",
				 rhs => $entries_bits."'h0",
			     }),
			     e_assign->news
			     (
			      ["wr_address" => $fifo_address_bits."'h0"],
			      ["rd_address" => $fifo_address_bits."'h0"],
			      ),
			     ],
			 }),
		      ],
      },
  });
  
  my $fsm = e_process->new({
      asynchronous_contents => [
	  e_assign->new({
	      lhs => "wr_address",
	      rhs => "0",
	  }),
	  e_assign->new({
	      lhs => "rd_address",
	      rhs => "0",
	  }),
	  e_assign->new({
	      lhs => "entries",
	      rhs => "0",
	  }),
      ],
      contents => [$rw_case],
  });

  $this->add_contents($fsm);

  my $wr_case = e_case->new({
      switch => "wr_address",
      parallel=> 1,
      # full => 1,
      # default_sim => 1, # $i_am_verilog: patched in e_case to do right thing.
      contents=>{%wr_data_mux_hash},
  });

  $this->add_contents
      (
       e_process->new({
	   contents => [
			e_if->new({
			    comment => "Write data",
			    condition => "wr & !full",
			    then => [$wr_case],
			}),
			],
       }),
       );
  
  # add validation vector
  $this->add_contents
      (
       e_signal->new({name => "valid_vector",  width => $depth}),
       e_signal->new({name => "shift_vector",  width => $depth}),
       e_signal->new({name => "valid_control", width => 3}),
       e_signal->new({name => "emone",
		      width=> $entries_bits,
		      never_export => 1}),
       e_assign->new({lhs  => "shift_vector", 
		      rhs  => "{1'b0, valid_vector[".($depth - 1).":1]}"}),
       e_assign->new({lhs  => "valid_control", 
		      rhs  => "{invalidate, (rd & !empty), (wr & !full)}"}),
       e_assign->new({lhs  => "valid",
		      rhs  => "valid_vector[0] | (rd & wr & empty)"}),
       e_assign->new({lhs  => "emone",
		      rhs  => "entries - 1"}),
       );

  # special cases for simultaneous push&pop of valid_vector.
  my $one_case   = e_case->new ({
      switch   => "entries",
      parallel => 1,
      contents => {%one___mux_hash},
  });

  my $three_case = e_case->new ({
      switch   => "emone",
      parallel => 1,
      contents => {%three_mux_hash},
  });

  my $five_case = e_case->new ({
      switch   => "entries",
      parallel => 1,
      contents => {%five__mux_hash},
  });

  my $seven_case = e_case->new ({
      switch   => "emone",
      parallel => 1,
      contents => {%seven_mux_hash},
  });
  
  my $v_case = e_case->new ({
      switch   => "valid_control",
      parallel => 1,
      contents => {
	  0 => [],
	  1 => [ $one_case, ],
              #["valid_vector[entries]" => "1'b1"],
	  2 => ["valid_vector" => "shift_vector"],
	  3 => [ $three_case,
		# e_assign->new({
		# lhs => "valid_vector",
		# rhs => "shift_vector",
		# }),
		# e_assign->new({
		# lhs => "valid_vector[emone]",
		# rhs => "1'b1",
		# }),
		],
	  5 => [ $five_case,
		# e_assign->new({
                # lhs => "valid_vector",
                # rhs => "{".$depth."{1'b0}}",
                # }),
		# e_assign->new({
                # lhs => "valid_vector[entries]",
                # rhs => "1'b1",
                # }),
		],
	  7 => [ $seven_case,
		# e_assign->new({
                # lhs => "valid_vector",
                # rhs => "{".$depth."{1'b0}}",
                # }),
		# e_assign->new({
                # lhs => "valid_vector[emone]",
                # rhs => "1'b1",
                # }),
		],
	  # note that this default case does WORK! for cases 4 and 6 above!
	  default => ["valid_vector" => "{".$depth."{1'b0}}"],
      },
  });
  
  my $vector = e_process->new({
      asynchronous_contents => 
	  [
	   e_assign->new({lhs => "valid_vector", rhs => "{".$depth."{1'b0}}"}),
	   ],
      contents => 
          [
           e_if->new({
              comment => " special empty case:",
              condition => "rd & wr & empty",
              then => ["valid_vector" => "{".$depth."{1'b0}}"],
              else => [$v_case],
           }),
           ],
  });

  $this->add_contents($vector);

}

# packages must return some non-zero value to be read correctly...
qq {
Under the trees, among the rocks, a thatched hut:
Verses and sacred commentaries live there together.
Ill burn the books I carry in my bag,
But how can I forget the verses written in my gut? 

- Ikkyu (1394-1481) 

End of package e_vfifo.pm
};

=back

=cut

=head1 EXAMPLE

Here is a usage example ...

=head1 AUTHOR

Santa Cruz Technology Center

=head1 BUGS AND LIMITATIONS

list them here ...

=head1 SEE ALSO

The inherited class e_module

=begin html

<A HREF="e_module.html">e_module</A> webpage

=end html

=head1 COPYRIGHT

Copyright (C)2001-2005 Altera Corporation, All rights reserved.

=cut

1;
