



################################################################
# CLASS: e_slave
#
# An e_slave corresponds to a SLAVE {} section in the PTF-file.
# It is a repository for PORT_WIRING and SYSTEM_BUILDER_INFO data
# which gets written-out to the PTF-file when the parent-project
# is "output."
#
# e_slave is meant to be an abstract class, on which you base your
# more-useful slave-object.  "Useful" slave objects override the
# recognized_type() method so that it returns a list of 
# port-types relevant to your bus-standard (e.g. the list of
# recognized Avalon or AHB port-types).
#
# The main operational method is "to_ptf", which saves-back
# information about this slave-port to the relevant MODULE's 
# PTF-section.
#
################################################################

=head1 NAME

e_slave - description of the module goes here ...

=head1 SYNOPSIS

The e_slave class implements ... detailed description of functionality

=head1 METHODS

=over 4

=cut

package e_slave;

use e_thing_that_can_go_in_a_module;
use e_module;
use e_signal;
use e_project;
@ISA = ("e_thing_that_can_go_in_a_module");
use europa_utils;
use strict;
################################################################
# e_slave::new
#
# Nothing special or tricky about this constructor.
################################################################
my %fields = (
              section_name     => "SLAVE",
              sideband_signals => [],
              SBI_section      => {},
              _type_recognized_hash    => {},   # boolean cache-hash.
              _type_already_bound_hash => {},   # discovery-scoreboard
              );

my %pointers = ();

&package_setup_fields_and_pointers
    (__PACKAGE__,
     \%fields, 
     \%pointers,
     );

################################################################################

=item I<new()>

Object constructor

=cut

sub new 
{
   my $this = shift;
   my $self = $this->SUPER::new(@_);
   $self->_setup_type_recognized_hash();

   return $self;
}

################################################################################

=item I<update()>

method description goes here...
...remember: there must be a newline around each POD tag (e.g. =item, =cut etc)!

=cut

sub update
{
   my $this = shift;
   $this->parent(@_);
}

################################################################################

=item I<_add_type_map_pair()>

method description goes here...
...remember: there must be a newline around each POD tag (e.g. =item, =cut etc)!

=cut

sub _add_type_map_pair
{
  my $this = shift;
  my $signal_name = shift;
  my $signal_type = shift;

  my $previous_type = $this->type_map()->{$signal_name};
  if ($previous_type ne "" && $previous_type ne $signal_type) {
    &goldfish ("Suspicious:  Signal $signal_name is already of type ",
               "$previous_type.  Overriding to $signal_type");
  }

  $this->type_map()->{$signal_name} = $signal_type;
  $this->_type_already_bound_hash()->{$signal_type} = 1;
}

################################################################################

=item I<type_map()>

method description goes here...
...remember: there must be a newline around each POD tag (e.g. =item, =cut etc)!

=cut

sub type_map
{
  # Any time anyone explicitly tries to set the type-map,
  # we call the safe access function which keeps track of what 
  # types have already been set.
  my $this = shift;

  if (!defined ($this->{type_map}))
  {
     $this->{type_map} = {};
  }
  if (@_) 
  {
     my ($new_map) = (@_);
     if (scalar(keys (%{$new_map})) == 0)
     {
        return $this->{type_map} = shift;
     }
     
     foreach my $signal_name (keys (%{$new_map})) 
     {
        $this->_add_type_map_pair($signal_name, $new_map->{$signal_name});
     }
  }
  return $this->{type_map};
}

################################################################################

=item I<recognized_types()>

method description goes here...
...remember: there must be a newline around each POD tag (e.g. =item, =cut etc)!

=cut

sub recognized_types
{
  my $this = shift;
  # Override this function to return a list of conventional 
  # names used for this type of slave's ports (e.g. address, readn, etc).
  return (@{$this->sideband_signals()});
}

################################################################################

=item I<_setup_type_recognized_hash()>

method description goes here...
...remember: there must be a newline around each POD tag (e.g. =item, =cut etc)!

=cut

sub _setup_type_recognized_hash
{
  my $this = shift;
  foreach my $type ($this->recognized_types()) {
    # For total gravy, we also recognize signal-types whose
    # names end in "_n".
    $this->_type_recognized_hash()->{$type}        = 1;
    $this->_type_recognized_hash()->{$type . "_n"} = 1;
  }
}

################################################################################

=item I<type_is_recognized()>

method description goes here...
...remember: there must be a newline around each POD tag (e.g. =item, =cut etc)!

=cut

sub type_is_recognized
{
  my $this      = shift;
  my $type_name = shift;

  # Set-up the recognized-type hash if no one's done it yet:
  if (scalar(keys(%{$this->_type_recognized_hash})) == 0) {
    $this->_setup_type_recognized_hash();
  }

  return $this->_type_recognized_hash()->{$type_name};
}

################################################################################

=item I<type_is_already_bound_to_port()>

method description goes here...
...remember: there must be a newline around each POD tag (e.g. =item, =cut etc)!

=cut

sub type_is_already_bound_to_port
{
  my $this      = shift;
  my $type_name = shift;

  return $this->_type_already_bound_hash()->{$type_name};
}

################################################################
# _call_this_when_any_signal_is_updated
#
# During the update-phase, we are responsible for seeing to it that
# all of the ports in the type-map (either explicitly or implicitly)
# have their "type"-field set correctly.  This happens dynamically
# as the ports "pop into existence."  We need to do this dynamically
# because somebody somewhere else might be using the "type"
# information for some other logic-generation purpose.
#
################################################################
################################################################################

=item I<_call_this_when_any_signal_is_updated()>

method description goes here...
...remember: there must be a newline around each POD tag (e.g. =item, =cut etc)!

=cut

sub _call_this_when_any_signal_is_updated
{
  my $this = shift;
  my $updated_signal = shift;

  # First, go through the list of ports.  If we find one that has a
  # "recognized" name not already in the type-map, add it to the
  # type-map:
  $this->_add_recognized_signals_to_map($updated_signal);

  # Now make sure every port mentioned in the type-map has 
  # its "type" field set accordingly:
  $this->_assign_signal_types_from_map();
}

################################################################################

=item I<to_ptf()>

method description goes here...
...remember: there must be a newline around each POD tag (e.g. =item, =cut etc)!

=cut

sub to_ptf
{
  my $this = shift;

  my $daddy   = $this->parent_module();
  my $mod_ptf = shift or &ribbit ("no place to put me\n");

  ref ($mod_ptf) eq "HASH" or &ribbit("expected reference to PTF  (hash).");

  # Now get a list of all the ports going into and out of the 
  foreach my $port_name ($daddy->_get_port_names())
  {
     my $port = $daddy->get_signal_by_name($port_name)
         or &ribbit ("no signal by name $port_name");
     $this->_call_this_when_any_signal_is_updated($port);
  }

  # Make sure explicitly-mapped ports actually exist
  $this->_assign_signal_types_from_map(0);  # don't complain  

  my $own_section_name  = $this->section_name();
     $own_section_name .= " " . $this->name() if $this->name ne "";

  my $own_section = {};
  if (exists ($mod_ptf->{$own_section_name})) {
    $own_section = $mod_ptf->{$own_section_name};
  } else {
    $mod_ptf->{$own_section_name} = $own_section;
  }

  # DELETE ME!  for v1.1 compatibility:
  # Just use MODULE's top-level PORT_WIRING and SBI sections.
  # (no SLAVE sub-section)
  my $version = $daddy->_project()->builder_version();
  my $old_school =  $version < 2.0;
  $own_section = $mod_ptf if $old_school;

  if ($old_school)
  {
    &goldfish ("Writing PTF in old-style format " .
      "(System_Wizard_version: '$version')");
  }

  # Always obliterate PORT_WIRING section.
  foreach my $value (values (%{$own_section->{PORT_WIRING}}))
  {
     $value->{Is_Enabled} = 0;
  }
  $own_section->{SYSTEM_BUILDER_INFO} = {}
    if !exists ($own_section->{SYSTEM_BUILDER_INFO});
    
  return if ($own_section->{SYSTEM_BUILDER_INFO}{Is_Enabled} eq '0');
  foreach my $slave_port ($this->_get_slave_ports()) {
    my $slave_port_type = $this->type_map()->{$slave_port->name()};
    $slave_port->add_to_ptf_section ($own_section->{PORT_WIRING},
                                     $slave_port_type);
  }

  my $SBI_section = $this->SBI_section();
  foreach my $key (keys %$SBI_section)
  {
     $own_section->{SYSTEM_BUILDER_INFO}{$key} = $SBI_section->{$key};
  }
  #die "ptf now looks like",$daddy->_project->ptf_to_string(),"\n";
}

################################################################
# _add_recognized_signals_to_map
#
# Sometimes you want the port "address" to be recognized as the 
# address-type port without having to explicitly say 
# "{ address => 'address' }" when you construct your type-map.
#
# This signal gets called every time a signal in the parent-module
# gets updated.  We look at all the signals as they go by.
#
# This function will add entries to the type_map field for each unclaimed,
# recognized port-name it finds--as long as the type is not already
# taken.
#
################################################################
################################################################################

=item I<_add_recognized_signals_to_map()>

method description goes here...
...remember: there must be a newline around each POD tag (e.g. =item, =cut etc)!

=cut

sub _add_recognized_signals_to_map
{
  my $this = shift;
  my $sig_to_recognize = shift;  # will be e_signal object.

  # Ignore broken/silly signals:
  return if $sig_to_recognize->isa_dummy();
  my $sig_name = $sig_to_recognize->name();
  return if $sig_name eq "";

  # Skip signals that already have types.
  #return if $sig_to_recognize->type() ne "";

  # Skip signals whose names we don't even recognize:
  return unless $this->type_is_recognized($sig_name);

  # Skip signals whose names are the same as types already-mapped:
  next if $this->type_is_already_bound_to_port ($sig_name);

  # We found an unclaimed signal which is named the same as a 
  # recognized port-type which is not otherwise used.  Wow. 
  $this->_add_type_map_pair($sig_name, $sig_name);
}


################################################################
# This function "implements" the type-map.  For every signal named
# in the type-map, this assigns the mapped-type to that signal.  If
# the signal already has a different type, we complain.
#
################################################################
################################################################################

=item I<_assign_signal_types_from_map()>

method description goes here...
...remember: there must be a newline around each POD tag (e.g. =item, =cut etc)!

=cut

sub _assign_signal_types_from_map
{
  my $this = shift;
  my $be_sure_all_ports_are_actually_present = shift;

  my $daddy   = $this->parent_module();

  foreach my $mapped_sig_name (keys(%{$this->type_map()})) {
    my $mapped_type = $this->type_map()->{$mapped_sig_name};
    my $sig = $daddy->get_signal_by_name($mapped_sig_name);

    if ($be_sure_all_ports_are_actually_present) {
      # React with alarm.
      &ribbit ("signal $mapped_sig_name appears in type-map, ",
               "but is not a port on module ", $daddy->name(), "\n")
        if !$sig;
    } else {
      # Don't react with alarm if signals in the map don't exist 
      # in the module...yet.  This function gets called during
      # update()-time when, presumably, signals are popping-into 
      # existence.  Set the magic flag if you want me to react with alarm.  
      #
      next unless $sig;
    }

    #$sig->confirm_type($mapped_type);
  }
}


################################################################
# All the updatin' fun is over.
#
# Now it's time to be sure that every port specified in the 
# type-map is present-and-accounted-for in the module, and has the 
# desired type.
#
################################################################
################################################################################

=item I<_verify_type_map()>

method description goes here...
...remember: there must be a newline around each POD tag (e.g. =item, =cut etc)!

=cut

sub _verify_type_map
{
  my $this = shift;
  # (1) means "verify that all signals are present, complain if not."
  $this->_assign_signal_types_from_map(1);
}

################################################################################

=item I<_get_slave_ports()>

method description goes here...
...remember: there must be a newline around each POD tag (e.g. =item, =cut etc)!

=cut

sub _get_slave_ports
{
  my $this = shift;

  my $daddy   = $this->parent_module();
  my @port_list = ();

  # Go through all the ports explicitly mentioned in the "type_map"
  # field.  Double-check to be sure they actually occur
  # in my parent module.  Set their types exactly as-indicated
  # and put them on the list:
  #
  foreach my $port_name (keys (%{$this->type_map()})) {
    my $sig = $daddy->get_signal_by_name ($port_name) or next;

    #my $got_type      = $sig->type();
    #my $expected_type = $this->type_map()->{$port_name};

    #$got_type eq $expected_type or 
    #&ribbit ("port '$port_name': expected type '$expected_type'",
    #" but got type '$got_type'.");

    push (@port_list, $sig);
  }
  return @port_list;
}

################################################################################

=item I<_get_mapped_types()>

method description goes here...
...remember: there must be a newline around each POD tag (e.g. =item, =cut etc)!

=cut

sub _get_mapped_types
{
  my $this = shift;
  return values (%{$this->type_map()});
}


1;  # ONE!  One vonderful package!  Ah ah ah!

=back

=cut

=head1 EXAMPLE

Here is a usage example ...

=head1 AUTHOR

Santa Cruz Technology Center

=head1 BUGS AND LIMITATIONS

list them here ...

=head1 SEE ALSO

The inherited class e_thing_that_can_go_in_a_module

=begin html

<A HREF="e_thing_that_can_go_in_a_module.html">e_thing_that_can_go_in_a_module</A> webpage

=end html

=head1 COPYRIGHT

Copyright (C)2001-2005 Altera Corporation, All rights reserved.

=cut

1;
