
=head1 NAME

e_ptf_top_module - based upon e_module ...

=head1 SYNOPSIS

The e_ptf_top_module class implements ... detailed description of functionality

=head1 METHODS

=over 4

=cut

package e_ptf_top_module;
use e_module;
@ISA = ("e_module");
use strict;
use europa_utils;

################################################################
# e_ptf_top_module::new
#
################################################################
my %fields   = ();
my %pointers = ();

# &package_setup_fields_and_pointers
#     (__PACKAGE__,
#      \%fields, 
#      \%pointers,
#      );

################################################################################

=item I<_is_redirected_from_slave()>

method description goes here...
...remember: there must be a newline around each POD tag (e.g. =item, =cut etc)!

=cut

sub _is_redirected_from_slave
{
   my $this = shift;
   my $port_name = shift or &ribbit ("no port");

   my $is_redirected_from_slave = 0;

   my @sources = $this->get_signal_sources_by_name
       ($port_name);

   if (@sources == 1)
   {
      my $source = $sources[0];
      if (&is_blessed($source) && $source->isa("e_instance"))
      {
         my $module = $source->module();
         #there's no port redirection
         my @mod_sources  = $module->get_signal_sources_by_name
             ($port_name);

         if ($module->isa("e_ptf_slave_arbitration_module") &&
             (@mod_sources == 1)
             )
         {
            my $mod_source = $mod_sources[0];
            if (&is_blessed($mod_source) && 
                $mod_source->isa("e_assign"))
            {
               my $rhs = $mod_source->rhs();
               if ($rhs->isa_signal_name())
               {
                  my $incoming_sig_name = $rhs->expression();
                  if ($module->is_input($incoming_sig_name))
                  {
                     my @incoming_sources = $this->
                         get_signal_sources_by_name($incoming_sig_name);

                     my $comes_from_ptf_module =
                         $this->signal_comes_from_e_ptf_module
                         ($incoming_sig_name);

                     return (1)
                         if ((@incoming_sources == 0) || $comes_from_ptf_module);
                  }
               }
            }
         }
      }
   }
   return (0);
}

################################################################################

=item I<signal_comes_from_e_ptf_module()>

method description goes here...
...remember: there must be a newline around each POD tag (e.g. =item, =cut etc)!

=cut

sub signal_comes_from_e_ptf_module
{
   my $this = shift;
   my $sig_name = shift;

   my @sources = $this->
       get_signal_sources_by_name($sig_name);

   my $source = $sources[0];
   my $comes_from_ptf_module = 
       (@sources == 1) &&
       $source->isa("e_instance") &&
       $source->module()->isa("e_ptf_module");

   return $comes_from_ptf_module;
}

################################################################################

=item I<signal_goes_to_e_ptf_module()>

method description goes here...
...remember: there must be a newline around each POD tag (e.g. =item, =cut etc)!

=cut

sub signal_goes_to_e_ptf_module
{
   my $this = shift;
   my $sig_name = shift;

   my @dests = $this->
       get_signal_destinations_by_name($sig_name);

   my $dest = $dests[0];
   my $goes_to_ptf_module = 
       (@dests == 1) &&
       $dest->isa("e_instance") &&
       $dest->module()->isa("e_ptf_module");

   return $goes_to_ptf_module;
}

################################################################################

=item I<wire_defaults()>

method description goes here...
...remember: there must be a newline around each POD tag (e.g. =item, =cut etc)!

=cut

sub wire_defaults
{
   my $this = shift;

   my $project = $this->project();
   my $test_bench_module = $project->test_module();

   foreach my $port ($this->_get_port_names())
   {
      my $sig = $this->get_signal_by_name($port);
      next unless ($sig);

      my $type = $sig->type();
      next unless $type;
      next if ($type eq "export");

      if ($type =~ /^pull/)
      {
        $test_bench_module->get_and_set_once_by_name({
          thing => "pull",
          name  => "$type $port",
          signal => $port,
          pull_type => $type,
        });
        next;
      }
      if ($this->is_output($port))
      {
          if ($type =~ /^out_clk/i)
          {
             # we want out_clks to never pop out the top of the module. 
             # Q:what DOES pop out the top? 
             # A:the clock with an out_clk as its source.
             $sig->never_export(1);
             next;
          }
          if (($type && $this->signal_comes_from_e_ptf_module($port))
              || $this->_is_redirected_from_slave($port))
          {
             $sig->never_export(1);
             next; 
          }
       }

      if ($this->is_input($port)
          && $type)
      {
         if ($type =~ /^reset/)
         {
            my $ns_period = $this->_get_period();
            $test_bench_module->get_and_set_once_by_name
                ({
                   thing => "reset_gen",
                   name  => "reset gen $port",
                   reset => $port,
                   ns_period => 10 * $ns_period,
                   reset_active => ($type =~ /_n$/) ? 0 : 1,
                });
            next;
         }
         if ($type =~ /^clk((_\w+)?_(\d+)_*(([mk])hz)?)?/i)
         {
            my $freq_number = $3;
            my $multiplier = $5;
            # If it's an external clock, we make a simulation model for it. 
            # Find the clock domain that this clk port is on.
            # If we can't find the clock source by the port name, go find all
            # of its destinations, and find one ptf module that may define
            # which domain this is.   
            # This will become even more complicated when individual masters
            # and slaves can be on clock domains!
            my $clock_source = $project->get_clock_source($port);
            my $clk_name;
            if ($clock_source) {
              # The port is clock name.
              $clk_name = $port;
            } else {
              # Go find the clock and its source.
              my $parent = $sig->parent_module();
              my @sig_dests = $parent->get_signal_destinations_by_name ($port);
              FIND_PTF_MODULE: foreach my $dest (@sig_dests) {
                next unless $dest->isa("e_instance");
                next unless $dest->module()->isa("e_ptf_module");
                my $mod_name = $dest->module()->name();
                $clk_name = $project->find_clock_domain_by_ptf_module($mod_name);
                last FIND_PTF_MODULE if $clk_name;
              }
              # if we can't even determine a clock domain, fuggetaboutit!
              next unless $clk_name;  
              $clock_source =  $project->get_clock_source ($clk_name);
            }
            # we only need to make a simulation driver for a clock if the clock
            # is fed by an external source. 
            if ($clock_source =~ /^external$/i) {
              my $ns_period = $this->_get_period
                  ($freq_number, $multiplier, $clk_name);
              $test_bench_module->get_and_set_once_by_name
                  ({
                    thing => "clk_gen",
                    name  => "clk gen $ns_period $port",
                    clk   => $port,
                    ns_period => $ns_period,
                  });
              next;
            }
         }

         if ($this->signal_goes_to_e_ptf_module($port))
         {
            my $default = 0;
            $default = 1 if ($type =~ /\_n$/);

            $this->update_item 
                (
                 e_assign->new
                 ({
                    lhs => $port,
                    rhs => $default,
                    comment => "$port of type $type does not connect to ".
                        "anything so wire it to default ($default)",
                     })
                 );
            next;
         }
      }
   }
}


################################################################################

=item I<get_clock_hash>

returns a hash keyed by clock name.  Value is frequency in Hz.

=cut

sub get_clock_hash
{
   my $this = shift;
   # this information is kept by the project.  ask it. 
   return $this->project()->get_clock_hash();
}

################################################################################

=item I<_get_period(number, multiplier, port_name)>

Return the clock frequency for the port_name.

=cut

sub _get_period
{
   my $this = shift;
   my ($number,$multiplier,$port_name) = @_;

   if ($number)
   {
      if ($multiplier =~ /m/i)
      {
         $number *= 1e6;
      }
      elsif ($multiplier =~ /k/i)
      {
         $number *= 1e3;
      }
   }
   else
   {
      if ($port_name)
      {
         $number = $this->project()->get_clock_frequency($port_name);
         if (!$number)
         {
            $number = $this->project()->get_clock_frequency($port_name);
         }


         if(!$number)
         {
            $number = 333333000;
            warn "unable to determine clock frequency for $port_name\n";
            warn "defaulting to $number\n";
         }
      }
      else #reset case here
      {

         # TODO: make an access function for getting all clock domains.
         my @clocks = map {
               $_ =~ /CLOCK\s*(\w+)/;
            } keys (%{$this->get_clock_hash()});
         my @clock_freqs = map {
              $this->project()->get_clock_frequency($_);
            } @clocks;
         $number = &min(@clock_freqs) || 33333000;
      }
   }
   my $ns_period = int (1e9/$number + .5);
   return ($ns_period);
}

################################################################################

=item I<_organize_ports_into_named_groups()>

This code organizes the e_ptf_top's ports according to what
internal e_ptf_module they come from.

This code right here used to be the bulk of the code from
e_project::MakeSymbol.

But it has utility beyond mere symbols!

=cut

sub _organize_ports_into_named_groups
{
   my $this = shift;

   # First, build a hash to keep track of all the system-level ports.
   # the keys of this hash will be the section-names.  There's
   # one section in the symbol for each internal module, and the
   # section-name is the same as the module name.  The values in this
   # hash will be a LIST of e_ptf_port objects.  Whee.
   #
   my %port_section_hash = ();
   my @sys_port_names = ($this->get_input_names(),
                         $this->get_output_names());

   foreach my $sys_port_name (@sys_port_names) 
   {
      my $port_obj = $this->get_signal_by_name($sys_port_name);

      # Find all the internal places this port goes.  Throw out all
      # the ones that don't happen to be "e_ptf_module"s:
      my @all_connections = 
          ($this->get_signal_sources_by_name($sys_port_name),
           $this->get_signal_destinations_by_name($sys_port_name));

      my @ptf_module_connections = ();

      foreach my $connection (@all_connections) 
      {
         if ($connection->isa("e_ptf_instance"))
         {
            push (@ptf_module_connections, $connection);
         }
         elsif ($connection->isa("e_instance"))
         {
            #ports routed to external modules need love too.
            my $mod = $connection->module();

            if ($mod->isa("e_ptf_arbitration_module"))
            {
               my $ms = $mod->_master_or_slave();
               my $routes_to = $ms->parent_module();
               my $sbi = $routes_to->{SYSTEM_BUILDER_INFO};
               if ((!$sbi->{Instantiate_In_System_Module}) ||
                    ($sbi->{Is_Bridge}))
               {
                  push (@ptf_module_connections, $connection);
               }
            }
         }

         if ($this->project()->language() =~ /vhdl/i)
         {
            #thanks to vhdl, all output ports from ptf_instances, 
            #will be redirected via an e_assignment.

            if ($connection->isa("e_assign") &&
                $connection->
                get_signal_sources_by_name($sys_port_name)
                )
            {
               my @rhs_sigs = $connection->rhs()->
                 _get_all_signal_names_in_expression();
               if (@rhs_sigs == 1)
               {
                  my $rhs = $rhs_sigs[0];
                  my @rhs_sources = 
                      $this->
                          get_signal_sources_by_name($rhs);

                  if (@rhs_sources == 1)
                  {
                     my $connection = $rhs_sources[0];
                     if ($connection->isa("e_ptf_instance"))
                     {
                        push (@ptf_module_connections,
                              $connection);
                     }
                     elsif ($connection->isa("e_instance"))
                     {
                        my $mod = $connection->module();

                        if ($mod->isa("e_ptf_arbitration_module"))
                        {
                           my $ms = $mod->_master_or_slave();
                           my $routes_to = $ms->parent_module();
                           my $sbi = $routes_to->{SYSTEM_BUILDER_INFO};
                           if ((!$sbi->{Instantiate_In_System_Module}) ||
                               ($sbi->{Is_Bridge}))
                           {
                              push (@ptf_module_connections, $connection);
                           }
                        }
                     }
                  }
               }
            }
         }
      }

      # Sometimes, signals bubble up to the top level when they shouldn't.
      # Mostly, these signals don't appear anywhere in the ptf file.  We 
      # used to ribbit on these occasions (simply because to make a good
      # symbol, you want to segregate your interface signals by module
      # type) but this is an impediment to progress.  Let's just goldfish,
      # omit the suspicious signal from the symbol, and try to muddle through.
      # No more training wheels!  If something sproings out the top, we'd
      # better fix it elsewhere not here.  Also, vhdl renaming assigns and
      # external modules will have non e_ptf_instance sourced signals.

      #these signals are global signal
      if (($sys_port_name eq "reset_n") || 
          ($port_obj->type() =~ /^clk/) ||
          ($port_obj->type() =~ /^out_clk/) ) 
      {
         push (@ptf_module_connections, 1,2,3,4);
      }

      if (!@ptf_module_connections)
      {
         print STDERR "Warning: " . 
             "suspicious signal '$sys_port_name' at system top ",
             "level.\n";
      }
      # Two kinds of signals:  Those connected to many things 
      # (e.g. "clk") and those connected to just one thing.  All 
      # multi-thing signals go into the same "misc-bucket":
      if (scalar(@ptf_module_connections) == 1)
      {
         my $home_instance = shift (@ptf_module_connections);
         my $section_name = $home_instance->name();
         push (@{$port_section_hash{$section_name}}, $port_obj);
      } else {
         push (@{$port_section_hash{"1) global signals:"}}, $port_obj);
      }
   }
   return \%port_section_hash;
}

################################################################################

=item I<make_reset_synchronizer()>

Construct the reset synchronizer. Asynchronously asserted and synchronously
deasserted behaviour.

=cut

sub make_reset_synchronizer
{
  my $this = shift;
  my ($clock, $reset_for_clock_domain) = @_;
 
  if ($this->get_and_set_once_by_name({
    thing => 'synchronizer',
    comment => "reset is asserted asynchronously and deasserted synchronously",
    name =>  $this->name() . "_reset_${clock}_domain_synch",
    port_map => {
     clk   =>  $clock,
     data_out    => "$reset_for_clock_domain",
     data_in     => "1'b1",
     reset_n  => "reset_n_sources"
    },
  }))
  {
    $this->make_reset_n_sources_mux(0);
  }
}

sub make_reset_n_sources_mux
{
  my $this = shift;
  my $resetrequest = shift;

   #now reset requests
   if ($this->get_and_set_once_by_name
       ({
          thing  => "mux",
          name   => "reset sources mux",
          lhs    => "~reset_n_sources",
          type   => "and_or",
          add_table => ["~reset_n", "~reset_n"],
       }))
   {
      #set the type so that we generate reset generator
      e_signal->new({
         name => 'reset_n',
         type => 'reset_n',
      })->within($this);
   }

   if ($resetrequest ne '')
   {
      $this->get_and_set_thing_by_name
       ({
          thing  => "mux",
          name   => "reset sources mux",
          add_table => [$resetrequest,
                        $resetrequest]
       });
   }
}

=back

=cut

=head1 EXAMPLE

Here is a usage example ...

=head1 AUTHOR

Santa Cruz Technology Center

=head1 BUGS AND LIMITATIONS

list them here ...

=head1 SEE ALSO

The inherited class e_module

=begin html

<A HREF="e_module.html">e_module</A> webpage

=end html

=head1 COPYRIGHT

Copyright (C)2001-2005 Altera Corporation, All rights reserved.

=cut

1;
