#Copyright (C)2001-2002 Altera Corporation
#Any megafunction design, and related net list (encrypted or decrypted),
#support information, device programming or simulation file, and any other
#associated documentation or information provided by Altera or a partner
#under Altera's Megafunction Partnership Program may be used only to
#program PLD devices (but not masked PLD devices) from Altera.  Any other
#use of such megafunction design, net list, support information, device
#programming or simulation file, or any other related documentation or
#information is prohibited for any other purpose, including, but not
#limited to modification, reverse engineering, de-compiling, or use with
#any other silicon devices, unless such use is explicitly licensed under
#a separate agreement with Altera or a megafunction partner.  Title to
#the intellectual property, including patents, copyrights, trademarks,
#trade secrets, or maskworks, embodied in any such megafunction design,
#net list, support information, device programming or simulation file, or
#any other related documentation or information provided by Altera or a
#megafunction partner, remains with Altera, the megafunction partner, or
#their respective licensors.  No other licenses, including any licenses
#needed under any third party's intellectual property, are provided herein.
#Copying or modifying any file, or portion thereof, to which this notice
#is attached violates this copyright.




package e_ptf;
use e_object;
@ISA = ("e_object");
use strict;
use format_conversion_utils;
use europa_utils;

################################################################
# e_ptf::new
#
# Constructor copied straight out of the tutorial.
#
################################################################

my %fields = (
              _array   => [],
              );

my %pointers = (
                spaceless_ptf_hash => {},
               );

&package_setup_fields_and_pointers
    (__PACKAGE__,
     \%fields, 
     \%pointers,
     );

sub ptf_file
{
   my $this = shift;
   if (@_)
   {
      my $infile = shift;
      $this->{ptf_file} = $infile;
      return $infile if $infile eq "";

      $this->ptf_hash($infile);
      $this->make_spaceless_ptf_hash($infile);
   }
   
   return $this->{ptf_file};
}

sub _transform_to_eval_string
{
   my $this = shift;
   my ($ptf_string, $doit_spaceless) = (@_);
   # Ignore (substitute) semicolons and curly-braces inside strings.
   # This is cheating a little bit, because we're using the 
   # semicolon and the equals-sign to anchor our search.  We know
   # that the only double-quotes left are actual non-escaped
   # PTF double-quotes, so there you go.

   # Give me (and my subclasses) an opportunity to do evil voodoo,
   # as-required, on the raw incoming string:
   #
   $ptf_string = $this->_doctor_incoming_ptf_file_string ($ptf_string);

   # Now turn it into something suitable for a Perl eval:
   $ptf_string =~ s/\\\"/__PTF_EVAL_DOUBLE_QUOTE__/sg;
   $ptf_string =~ s/(\=\s*\"[^\"]*)\'([^\"]*\"\s*;)/$1\__PTF_EVAL_SINGLE_QUOTE__$2/sg;
   $ptf_string =~ s/(\=\s*\"[^\"]*);([^\"]*\"\s*;)/$1\__PTF_EVAL_SEMICOLON__$2/sg;
   $ptf_string =~ s/(\=\s*\"[^\"]*)\{([^\"]*\"\s*;)/$1\__PTF_EVAL_OPEN_BRACE__$2/sg;
   $ptf_string =~ s/(\=\s*\"[^\"]*)\}([^\"]*\"\s*;)/$1\__PTF_EVAL_CLOSE_BRACE__$2/sg;

   #now convert it to a hashable equation and eval
   #first 
   #convert foo = "bar"; to 'foo = "bar"',
   $ptf_string =~ s/(\w+\s*\=\s*\".*?\"\s*)\;/\'$1\',/sg;

   if ($doit_spaceless) {
      #convert two space separated strings to 'string1, string2'
      # foo bar {  -->  'foo bar' {
      $ptf_string =~ s/\b([\w\/\.]+)\s+([\w\/\.]+)\s*\{/\'$1\', \'$2\' \{/sg;
   } else {
      #convert two space separated strings to 'string1 string2'
      # foo bar {  -->  'foo bar' {
      $ptf_string =~ s/\b([\w\/\.]+)\s+([\w\/\.]+)\s*\{/\'$1 $2\' \{/sg;
   }

   #convert hash pointers into array pointers
   $ptf_string =~ s/\{/\,\[/g;
   $ptf_string =~ s/\}/\],/g;

   # Re-substitute escaped quotes:
   $ptf_string =~ s/__PTF_EVAL_DOUBLE_QUOTE__/\\\"/sg;
   $ptf_string =~ s/__PTF_EVAL_SINGLE_QUOTE__/\\\'/sg;
   $ptf_string =~ s/__PTF_EVAL_SEMICOLON__/;/sg;
   $ptf_string =~ s/__PTF_EVAL_OPEN_BRACE__/{/sg;
   $ptf_string =~ s/__PTF_EVAL_CLOSE_BRACE__/}/sg;
   return $ptf_string;
}

sub ptf_hash
{
   my $this  = shift;

   # Magic, special overload behavior:
   # If the user passes-in a string, then we 
   # treat it as a filename from which to read hash-data.
   # otherwise (below), we just do the standard access-function behavior.
   if (scalar (@_))
   {
      my $ptf_hash;
      if (ref ($_[0]) eq "")
      {
         my $ptf_file = shift;
         open (FILE, "< $ptf_file")
             or &ribbit ("cannot open $ptf_file");

         my $ptf_string;
         binmode (FILE);
         while (<FILE>)
         {
            s/\#.*$//mg;  #goodbye, Mr. comment.
            $ptf_string .= $_;
         }
         close (FILE);

         $ptf_string = $this->_transform_to_eval_string ($ptf_string);
         
         $ptf_string = "\$ptf = [$ptf_string];";
         my $ptf;
         {
            no strict;
            eval ($ptf_string);
            die "eval failed on this ptf-string:\n      $ptf_string \n($@)"
                if ($@);
         }
         $this->_array($ptf);

         my %root;
         $ptf_hash = \%root;
         $this->{ptf_hash} = $ptf_hash;
         $this->_put_array_in_ptf_hash($ptf,
                                       $ptf_hash);

      }
      else
      {
         $ptf_hash = shift;
      }
      $this->{ptf_hash} = $ptf_hash;
   }
   else 
   {
      # Didn't pass-in a string.  No magic.
      if (!defined ($this->{ptf_hash}))
      {
         $this->{ptf_hash} = {};
      }
   }
   return $this->{ptf_hash};
}

######################################################################
# spaceless_ptf_hash
#
# in order to write back the ptf in a equivalent manner, we need to
# keep track of the spaces ie. "MODULE boot_monitor_rom" needs to be a
# key in the ptf hash.  However it makes much more sense to have
# MODULE be a key that points to all modules, one of which is
# boot_monitor_rom.  Thus _space_free_hash takes the first hash and
# converts it into the second.  You'll have to be a little careful
# with this new hash if you add elements to it.  If you add MODULE foo
# to it, your ptf file will look like MODULE { foo {}}.  Not what you want.
# 
# this method borrows liberally from ptf-hash.  it is identical except
# for one line.  Later, it might be better to merge these or use the
# first one as a template to print the second one.  Then again, it may
# be better to revise the ptf format so that this is the only format.
#
######################################################################

sub make_spaceless_ptf_hash
{
   my $this  = shift;
   my $was_called_statically = ref ($this) eq ""; 
   
   if (scalar (@_)  && (ref ($_[0]) eq ""))  
   {
      my $ptf_file = shift;
      open (FILE, "< $ptf_file")
          or &ribbit ("cannot open $ptf_file");

      my $ptf_string;
      while (<FILE>)
      {
         ###############
         #say goodnight, comments
         s/\#.*$//mg;
         $ptf_string .= $_;
      }
      close (FILE);

      # #now convert it to a hashable equation and eval
      # #first 
      # #convert foo = "bar"; to 'foo = "bar"',
      # $ptf_string =~ s/(\w+\s*\=\s*\".*?\"\s*)\;/\'$1\',/sg;
        
      # #convert two space separated strings to string1,string2
      # $ptf_string =~ s/\b([\w\/]+)\s+([\w\/]+)\s*\{/\'$1\', \'$2\' \{/sg;
        
      # #convert hash pointers into array pointers
      # $ptf_string =~ s/\{/\,\[/g;
      # $ptf_string =~ s/\}/\],/g;

      $ptf_string = 
          $this->_transform_to_eval_string ($ptf_string, "spaceless, please");
      $ptf_string = "\$ptf = [$ptf_string];";

      my $ptf;

      {
         no strict;
         eval ($ptf_string);
         die "eval failed ($@)"
             if ($@);
      }
      my %root;
      my $spaceless_ptf_hash = \%root;
      $this->spaceless_ptf_hash($spaceless_ptf_hash);
      $this->_put_array_in_ptf_hash($ptf,
                                    $spaceless_ptf_hash);

      return ($this->spaceless_ptf_hash());
    } else {
      # Didn't pass-in a string.  No magic.
      return ($this->spaceless_ptf_hash(@_));
    }
}

################################################################
# _doctor_written_ptf_values
#
# We may want to perform some sort of nefarious transformation on 
# all outgoing assignments when we write them into the final PTF-file.
#
# For specific example, we want to substitute all absolute file-paths
# with some sort of relative-path token so that the final PTF-file is 
# "portable" to a different directory.
#
# The base "e_ptf"-class defines the null (identity) transform
# on all outgoing PTF-assignments.  You may wish to define your own 
# bizarre transforms in your
#
################################################################

sub _doctor_written_ptf_values
{
    my $this = shift;
    my $assignment_name = shift;
    my $original_value  = shift;

    &ribbit ("Two arguments required") 
        if $assignment_name eq "" && $original_value eq "";

    return $original_value;
}


################################################################
# _doctor_incoming_ptf_file_string
#
# We may want to perform some sort of nefarious transformation on 
# all incoming PTF-files (like substituting-in the run-time-known
# project-path instead of some placeholder token).
#
# If you have any such job to do, override this function in your
# derived class:
#
################################################################
sub _doctor_incoming_ptf_file_string
{
   my $this = shift;
   my $raw_file_string = shift;
   return $raw_file_string;
}

sub _skip_down_ptf_path
{
    my $this = shift;
    my $ptf_hash = shift or &ribbit ("no ptf_hash");
    my $ptf_path = shift or &ribbit ("no ptf path");

    while (my $stone = shift (@$ptf_path))
    {
       ###############
       # set ptf_hash to null hashref.
       # unless it already exists
	if (!exists $ptf_hash->{$stone})
	{
           $ptf_hash->{$stone} = {};
	}
	$ptf_hash = $ptf_hash->{$stone};
    }
    return ($ptf_hash);
}

sub _put_array_in_ptf_hash
{
   my $this = shift;

   my $ptf_array = shift or &ribbit ("no ptf_array");

   my %hash;
   my $ptf_hash = shift || \%hash;

   my @ptf_path = ();

   foreach my $thing (@$ptf_array)
   {
       if (ref ($thing) eq "")
       {
	   if ($thing =~ /^\s*(\w+)\s*\=\s*\"(.*?)\"$/s)
	   {
	       my $tmp_ptf_hash = $this->_skip_down_ptf_path
		   (
		    $ptf_hash,
		    [@ptf_path]
		    );
	       $tmp_ptf_hash->{$1} = $2;
	   }
	   else
	   {
	       push (@ptf_path, $thing);
	   }
       }
       else
       {
	   if (ref ($thing) eq "ARRAY")
	   {
	       my $tmp_ptf_hash =
                   $this->_skip_down_ptf_path($ptf_hash,[@ptf_path]);
	       $this->_put_array_in_ptf_hash($thing,$tmp_ptf_hash);
	       @ptf_path = ();
	   }
	   else
	   {
	       &ribbit ("pretty confused here, ",ref($thing),"\n");
	   }
       }
   }
   return ($ptf_hash);
}

sub ptf_to_string
{
    my $this = shift;

    my $ptf_hash = shift; 
    my $indent = shift || "";
    my $order    = shift;

    defined ($ptf_hash) or $ptf_hash = $this->ptf_hash();
    defined ($order) or $order = $this->_array();

    my @new_order = @$order;
    my %new_hash = %$ptf_hash;

    my $next_indent = "$indent   ";

    my $string;

    if (1)#ref($new_hash) eq "HASH")
    {
       my $key;
       while ($key = shift (@new_order))#(sort (keys (%$new_hash)))
       {
          next if (ref ($key));
          my $old_key = $key;
          $key =~ s/^\s*(.*?)\s*\=.*/$1/s;
          my $value = $new_hash{$key};

          if (ref($value) eq "HASH")
          {
             my $child_order = shift(@new_order);
             if (ref ($child_order) ne "ARRAY")
	       {
		 my @new_child_order = sort (keys (%$value));
		 #&goldfish ("out of order ($child_order)",join ("\n",@new_child_order),"\n");
		 unshift (@new_order,$child_order);
		
		 $child_order = \@new_child_order;
	       }
             $string .= "$indent$key";
             $string .= "\n$indent\{\n";
             $string .= $this->ptf_to_string
                 ($value,$next_indent,$child_order);
             $string .= "$indent\}\n";
             delete $new_hash{$key};
          }
          elsif (exists $new_hash{$key})
          {
             $string .= "$indent$key";
             my $doctored_value = 
                 $this->_doctor_written_ptf_values ($key, $value);
             $string .= " = \"$doctored_value\"\;\n";
             delete $new_hash{$key};
          }
	  else
	    {
	      delete $new_hash{$key};
	    }
       }

       # now we are done with all known order, but maybe somebody has
       # added something to the new_hash.  If so, print out
       # alphabetically on this level.
       my @remaining = sort (keys (%new_hash));
	if (@remaining)
	{
       	  $string .= $this->ptf_to_string
          (\%new_hash,$indent,\@remaining);
        }
               
    }
    else
    {
       #&ribbit ("This element ($new_hash) is not a hash\n");
    }
    return ($string);
}

sub ptf_to_file
{
   my $this = shift;
   my $ptf_hash = shift || $this->ptf_hash();
   my $file = shift || $this->ptf_file();

   return if (!$file);
   open (FILE, "> $file") or &ribbit 
       ("Could not open file, or write-protected file ($file)($!)\n");

   #don't forget to set binmode dear.
   binmode (FILE);

   print FILE $this->ptf_to_string($ptf_hash);
   close (FILE);
}

#
#  sub _space_free_hash
#  {
#     my this = shift;

#     my $ptf = shift || $this->ptf_hash();
#     my $new_ptf = shift || $this->space_free_hash();

#     my $next_new_ptf;
#     my $next_ptf;
#     foreach my $ptf_key (key %$ptf)
#     {
#        if ($ptf_key =~ s/^(\S+)\s+(\S+)$/)
#        {
#           $new_ptf->{$1}{$2} = $ptf->{$ptf_key};
#        }
#        else
#        {
#           $new_ptf->{$ptf_key} = $ptf->{$ptf_key};
#        }
#        $next_ptf = $ptf->{$ptf_key}
#        if (ref ($next_ptf) eq "HASH")
#        {
#           $this->_space_free_hash($next_ptf,
#                                   $ptf->{$ptf_key},
#        }
#     }
   
#  }

######################################################################
# ptf_type_children
#
# 
sub ptf_type_children
{
    my $this = shift;
    my $type = shift or &ribbit ("no type");
    my $ptf_hash = shift; 

    defined ($ptf_hash) or $ptf_hash = $this->ptf_hash();

    my $hash;
    foreach my $key (keys (%$ptf_hash))
    {
       next unless ($ptf_hash->{$key} =~ /^$type\s+(\w+)/i);
       $hash->{$1} = $ptf_hash->{$key};
    }
    return ($hash);
}

################################################################
# spaceless_system_ptf()
#
# Return a reference to the spaceless system ptf hash
#
################################################################
sub spaceless_system_ptf
{
   my $this = shift;

   my @systems = keys (%{$this->spaceless_ptf_hash()->{SYSTEM}});
   (@systems == 1) or &ribbit ("expected exactly 1 system; got @{[0 + @systems]}\n");

   my $ptf_section = $this->spaceless_ptf_hash()->{SYSTEM}
   {$systems[0]} or &ribbit ("no system");
   return ($ptf_section);
}

################################################################
# Create_Dat_Files
#
# For all modules which have a "CONTENTS" WSA section, 
# 1) Create MIF file(s), byte-laned if necessary, unless they
#   already exist (onchip memory has its own complicated method
#   for creating "chunked" mif files, so let it keep doing that)
# 2) Create DAT files in the sim directory.
#
################################################################
sub Create_Dat_Files
{
  my $this   = shift;
  my $sysdir = shift or &ribbit ("No sysdir");
  my $simdir = shift or &ribbit ("No simdir.");
  &ribbit ("Too many arguments") if @_;

  # Look at all modules.
  # CODE AROUND:  Other way to get names of all MODULE sections.
  my $sys_hash = $this->spaceless_system_ptf();

  foreach my $mod_name (keys(%{$sys_hash->{MODULE}}))
  {
    # CODE AROUND: Clone functionality in-place of "spaceless module".
    my $module_spaceless_hash = $sys_hash->{MODULE}{$mod_name};

    # If this module is not enabled, skip it.
    next if (!$module_spaceless_hash->{SYSTEM_BUILDER_INFO}->{Is_Enabled});

    # No contents? Nothing to do.
    next if (!$module_spaceless_hash->{WIZARD_SCRIPT_ARGUMENTS}->{CONTENTS});
    
    # I'm going to need to know the width of this module, I can feel it.
    # That's needed for srec2mif (if no byte lanes) and mif2dat.

    my @slaves = keys %{$module_spaceless_hash->{SLAVE}};
    next unless (scalar(@slaves) > 0); 

    my $num_lanes;
    my $num_banks;
    my $width;
    my $address_base;
    my $address_width;
    
    for (@slaves)
    {
      my $slave_hash = $module_spaceless_hash->{SLAVE}->{$_};
      next if (!$slave_hash);
      next if ($slave_hash->{SYSTEM_BUILDER_INFO}{Is_Enabled} eq '0');

      $width = $slave_hash->{SYSTEM_BUILDER_INFO}->{Data_Width};
      $address_base =
        eval($slave_hash->{SYSTEM_BUILDER_INFO}->{Base_Address});
      $address_width = $slave_hash->{SYSTEM_BUILDER_INFO}->{Address_Width};

      # Look for overridden num_lanes, num_banks;
      $num_lanes = $slave_hash->{SYSTEM_BUILDER_INFO}->{Simulation_Num_Lanes};
      $num_banks = $slave_hash->{SYSTEM_BUILDER_INFO}->{Simulation_Num_Banks};
      
      my $slave_hash_ports = $slave_hash->{PORT_WIRING}->{PORT};
      next if (!$slave_hash_ports);

      if (!$num_lanes)
      {
        # Look for byteenable signals.  If present, we have to make lanes.
        for (keys %$slave_hash_ports)
        {
          my $last_hash = $slave_hash_ports->{$_};
          if ($slave_hash_ports->{$_}->{type} =~ /byteenable/)
          {
            $num_lanes = $slave_hash_ports->{$_}->{width};
          }
        }
      }
      
      # Mostly we want just one bank.
      if (!$num_banks)
      {
        $num_banks = 1;
      }
    }
    
    my $byte_address_width = $address_width + log2(ceil($width / 8));
    my $mem_size_in_bytes = 2**$byte_address_width;

    # This is bad.
    # What's the right way to find a module's contents file name?
    my $contents_file = $mod_name . "_contents.srec";
    my $full_contents_file_path =
      $sysdir . "/" . $contents_file;
      
    if (!-e $full_contents_file_path)
    {
      # No such file?  This is an odd situation for Nios, but normal
      # for Nios2/Eclipse.  The end result will be that for Nios or 
      # Nios2/old sdk, the source file from which simulation contents
      # files should have been generated is missing.
      next;
    }

    # Now, try to find this module's mif file(s).
    
    # This is bad.
    # The onchip memory generator makes mif files with various names based
    # on the module name.  Rather than make assumptions about the persistence
    # of the rules used to create those file names, I'll just notice modules
    # of class "altera_avalon_onchip_memory"; for those modules, I'll grab all
    # files of the form <name>*.mif and convert them to dat.
    
    my @mif_files = ();
    if ($module_spaceless_hash->{class} eq "altera_avalon_onchip_memory")
    {
      # I have special knowledge about this class of peripheral. MIF files should
      # already exist: if so, add them to my list.  Otherwise, complain.
      my $file_pat = $mod_name . ".*\.mif";
      
      # I'd use glob(), but we don't seem to have File.pm.  Do it the hard
      # way:
      if (!opendir DIR, $sysdir)
      {
        dwarn("Unexpected: can't open directory '$sysdir'");
        next;
      }
      
      my @files = readdir DIR;
      push @mif_files, (grep /^$file_pat$/, @files);
      
      closedir DIR;
    }
    else
    {
      # Always generate new mif files for this module.  This could be a 
      # problem if someone has a CONTENTS section (and therefore an srec file)
      # but had their own custom mif file which matches my pattern, also.
      # A bit of help: don't crush existing mif files, just use them.  But,
      # if we create mif files, delete them after converting to dat.
      
      # How do you make a mif file from an srec?
      # We need to know:
      # Whether this module has byte enable signals - if so, make byte
      # lanes.
      # The data width of the file(s) (8 if byte enables, slave data width otherwise).
      # 
      
      # The user can override our silly ideas about num_lanes and num_banks
      # in the slave/SBI section.
      
      if ($num_lanes == 1 and $num_banks == 1)
      {
        my $mif_file = $mod_name . ".mif";

        # Convert the srec file into a mif.
        fcu_convert({
          "0"      => $full_contents_file_path,
          "1"      => $mif_file,
          oformat  => "mif",
          width    => $width,
        });

        push @mif_files, $mif_file;
      }
      elsif ($num_banks == 1)
      {
        # I've got lanes, but no banks
        for (0 .. $num_lanes - 1)
        {
          my $mif_file = $mod_name . "_lane$_.mif";

          # Convert the srec file into a mif.
          fcu_convert({
            "0"      => $full_contents_file_path,
            "1"      => $mif_file,
            lane     => $_,
            lanes    => $num_lanes,
            width    => 8,
            oformat  => "mif",
          });

          push @mif_files, $mif_file;
        }
      }
      elsif ($num_lanes == 1)
      {
        # Multiple banks, 1 lane.
        my $cur_base = $address_base;
        my $bank_size = $mem_size_in_bytes / $num_banks;
        for my $bank (0 .. $num_banks - 1)
        {
          my $mif_file = $mod_name . "_bank$bank.mif";
        
          # Convert the srec file into a mif.
          fcu_convert({
            "0"      => $full_contents_file_path,
            "1"      => $mif_file,
            width    => $width,
            oformat  => "mif",
            address_low  => $cur_base,
            address_high => $cur_base + $bank_size - 1,
          });
          $cur_base += $bank_size;

          push @mif_files, $mif_file;
        }
      }
      else
      {
        # Multiple banks and multiple lanes.
        dwarn("Unimplemented: multiple banks, multiple lanes.\n");
      }
    }
    
    # Well, I have a list of mif files.  Make them into dat files.
    for my $mif_file (@mif_files)
    {
      my $dat_file;
      ($dat_file = $mif_file) =~ s/\.mif$/.dat/;

      # Guh.  Even though mif files contain a width value, I have to specify
      # it for fcu_convert, lest it use its default value, 16.

      my $dat_width;
      if (open MIF, $mif_file)
      {
        while (<MIF>)
        {
          if (/WIDTH\s*=\s*(\d+);$/)
          {
            $dat_width = $1;
            last;
          }
        }
        close MIF;
      }
      else
      {
        dwarn("Warning: can't open mif file '$mif_file'\n");
      }
      # Well, I guess I need a default.
      $dat_width = 16 unless $dat_width;

      $dat_file = $simdir . "/" . $dat_file;

      fcu_convert({
        "0"      => $mif_file,
        "1"      => $dat_file,
        oformat  => "dat",
        width    => $dat_width,
      });
    }

  }
}

# get_module_slave_hash takes in a / delimited key and returns 
# a hash of enabled {<module_name/slave_name> => value 
# Example usage: get_module_slave_hash("SYSTEM_BUILDER_INFO/Is_Memory_Device");
# Will return every module with a slave interface.
# Those without the key will have an undefined value
sub get_module_slave_hash 
{
   my $this = shift;
   my $slash_delimited_path = shift;

   my @paths;

   if (ref($slash_delimited_path) eq "ARRAY") 
   {
        @paths = @$slash_delimited_path;
   } 
   else 
   {
        @paths = split (/\//s,$slash_delimited_path);
   }

   my $spaceless_system = $this->spaceless_system_ptf();

   my $return_hash;
   my $modules = $spaceless_system->{MODULE};

   #loop over modules
   foreach my $module_name (keys %$modules)
   {
      my $module_ptf = $modules->{$module_name};
      next if ($module_ptf->{SYSTEM_BUILDER_INFO}{Is_Enabled} eq '0');

      #loop over slaves
      my $slaves = $module_ptf->{SLAVE};
      foreach my $slave_name (keys %{$slaves})
      {
         my $slave_ptf = $slaves->{$slave_name};
         next if ($slave_ptf->{SYSTEM_BUILDER_INFO}{Is_Enabled} eq '0');         

         my $return_hash_value = $slave_ptf;
         foreach my $path (@paths)
         {
            $return_hash_value = $return_hash_value->{$path};
         }

         $return_hash->{$module_name.'/'.$slave_name} = $return_hash_value;
      }
   }
   return $return_hash;
}
# get_module_hash takes in a / delimited key and returns 
# a hash of enabled {<module_name/slave_name> => value 
# Example usage: get_module_hash("SYSTEM_BUILDER_INFO/Is_CPU");
# Will return every module 
# Those without the key will have an undefined value

sub get_module_hash 
{
   my $this = shift;
   my $slash_delimited_path = shift;

   my @paths = split (/\//s,$slash_delimited_path);

   my $spaceless_system = $this->spaceless_system_ptf();

   my $return_hash;
   my $modules = $spaceless_system->{MODULE};

   #loop over modules
   foreach my $module_name (keys %$modules)
   {
      my $module_ptf = $modules->{$module_name};
      next if ($module_ptf->{SYSTEM_BUILDER_INFO}{Is_Enabled} eq '0');

      my $return_hash_value = $module_ptf;
      foreach my $path (@paths)
      {
         $return_hash_value = $return_hash_value->{$path};
      }

      $return_hash->{$module_name} = $return_hash_value;
   }
   return $return_hash;
}

#Returns a pointer to the system_ptf in a ptf hash.
sub system_ptf{
    my $this = shift;
    my $hash = $this->ptf_hash();
    my ($system_name) = keys(%{$this->spaceless_ptf_hash()->{SYSTEM}});
    my $system_ptf = $hash->{"SYSTEM $system_name"};
    return $system_ptf;
}


sub get_paths_which_contain_name
{
   my $this = shift;
   my ($name_to_find, $ptf_hash, $array_path) = @_;

   $ptf_hash = $ptf_hash || $this->ptf_hash();
   $array_path = $array_path || [];
   my @return_array;

   foreach my $key (sort keys (%$ptf_hash))
   {
      my $value = $ptf_hash->{$key};
      my @current_path = (@$array_path, $key);
      if (ref ($value) eq 'HASH')
      {
         my @paths = $this->get_paths_which_contain_name
               ($name_to_find, $value, \@current_path);
         
         if (@paths){push (@return_array, @paths)}
      }
      else{
          if ($key eq $name_to_find){
            push (@return_array ,[@current_path]);
          }
      }
   }
   return @return_array;
}

sub set_value_by_path
{
   my $this = shift;
   my $path  = shift;
   my $value = shift;
   my $ptf_hash = shift;

   if (ref ($path) eq 'HASH')
   {
      my $tmp_hash = $path;
      $path        = $tmp_hash->{path};
      $value       = $tmp_hash->{value};
      $ptf_hash    = $tmp_hash->{ptf_hash};
   }

   my $hash = $ptf_hash || $this->ptf_hash();
   
   my $last_stone = pop (@$path);
   foreach my $stone (@$path)
   {
      $hash = $hash->{$stone};
   }

   $hash->{$last_stone} = $value;
}

sub get_value_by_path
{
   my $this = shift;
   my $path  = shift;
   my $value = shift;
   my $ptf_hash = shift;

   if (ref ($path) eq 'HASH')
   {
      my $tmp_hash = $path;
      $path        = $tmp_hash->{path};
      $value       = $tmp_hash->{value};
      $ptf_hash    = $tmp_hash->{ptf_hash};
   }

   my $hash = $ptf_hash || $this->ptf_hash();
   
   foreach my $stone (@$path)
   {
      $hash = $hash->{$stone};
   }

   #at this point, hash might be a value
   return $hash;
}


1; # One, I say.
