







################################################################
# e_pipe_module
# 
# A regular e_module, with some extra features:
#
################################################################

=head1 NAME

e_pipe_module - description of the module goes here ...

=head1 SYNOPSIS

The e_pipe_module class implements ... detailed description of functionality

=head1 METHODS

=over 4

=cut

package e_pipe_module;
use europa_utils;
use e_pipe_register;
use e_module;
@ISA = ("e_module");
use strict;

my %known_stage_numbers = ();
################################################################
# new
# 
################################################################
my %fields = (
              stage         => 0,
              pipe_clk_en   => "pipe_run",
              );
my %pointers = (
                _tmp_project => e_project->dummy(),
                );

&package_setup_fields_and_pointers(__PACKAGE__,
                                   \%fields,
                                   \%pointers);
################################################################################

=item I<new()>

Object constructor

=cut

sub new 
{
   my $that = shift;
   my $self = $that->SUPER::new(@_);

   $self->add_pipeline_ports(); 
   $self->validate();
   return $self;
}


################################################################
# define_stages
#
# Static function which you should call to set-up your list of 
# valid stage-labels (names) before you construct any actual
# e_pipe_module objects.
#
################################################################
################################################################################

=item I<define_stages()>

method description goes here...
...remember: there must be a newline around each POD tag (e.g. =item, =cut etc)!

=cut

sub define_stages
{
   my $this = shift;
   &ribbit ("Please call this function statically") unless ref ($this) eq "";
   my (@stage_defs) = (@_);
   
   foreach my $def (@stage_defs) 
   {
      &ribbit ("expected list-refs (name/number pairs) as arguments")
          unless (ref ($def) eq "ARRAY") && (scalar (@{$def}) == 2);
      my ($label, $number) = (@{$def});
      &ribbit ("Suspicious attempt to re-label stage '$label'")
          if exists ($known_stage_numbers{$label});
      $known_stage_numbers{$label} = $number;
   }
}

################################################################
# get_stage_clk_en_signal
#
#  Each pipe-stage has its own clock-enable signal (default is
#  "pipe_run," but you can override it).  This static function
#  lets anyone ask for the clk_en signal used for a specific stage.
#
################################################################
my %pipe_clk_en_signals = ();

################################################################################

=item I<get_stage_clk_en_signal()>

method description goes here...
...remember: there must be a newline around each POD tag (e.g. =item, =cut etc)!

=cut

sub get_stage_clk_en_signal
{
   my $this = shift;
   my $stage_num = shift;
   &ribbit ("Please call this function statically") unless ref ($this) eq "";
   &ribbit ("Please call with stage-number") unless $stage_num =~/^\d+$/;
   
   my $result = $pipe_clk_en_signals {$stage_num};

   # Default to the default if we need a default:
   $result = $fields{pipe_clk_en} if !$result;
   
   return $result;
}

################################################################
# define_sequential_signals
#
# You can provide a list of signals that get passed-down the 
# pipeline in a coherent manner.   All "e_pipe_module" objects
# have access to these signals.  All "e_pipe_module" objects get
# (as input) their own private version of this signal with 
# the pipe-stage number stuck on the end, like this: 
#
#      instruction_3
#         --or--
#      is_neutrino_1
# 
# Internally, their privately-named ports get renamed thus:
#  
#          assign instruction = instruction_3;
#
# Each "defined" signal may, optionally, specify the stage it 
# was generated in.  All subsequent stages will get their privately-
# named versions as input-ports, and all previous stages will remain
# blissfully ignorant of its existence.  The "Originating" stage is
# unaltered because, presumably, it already knows about the signal.
# 
################################################################
my %sequential_signal_widths       = ();
my %sequential_signal_origins      = ();
my %sequential_signal_reset_values = ();
################################################################################

=item I<define_sequential_signals()>

method description goes here...
...remember: there must be a newline around each POD tag (e.g. =item, =cut etc)!

=cut

sub define_sequential_signals
{
   my $this = shift;
   &ribbit ("Please call this function statically") unless ref ($this) eq "";
   my (@sig_defs) = (@_);
   
   foreach my $sig_def (@sig_defs) 
   {
      &ribbit ("expected list-refs (name/number/origin triads) as arguments")
          unless (ref ($sig_def) eq "ARRAY") && (scalar (@{$sig_def}) <= 4);
      my ($sig_name, $width, $origin, $reset_val) = (@{$sig_def});

      &ribbit ("Suspicious attempt to re-define pipe-signal  '$sig_name'")
          if exists ($sequential_signal_widths{$sig_name});

      $width     = 1   if !$width;            # Default width is 1.
      $reset_val = "0" if $reset_val eq "";   # Reset to zero by default.

      # You must specify the origin by name:
      &ribbit ("unknown stage name: $origin") 
          if $origin && !exists ($known_stage_numbers{$origin});
      my $origin = $known_stage_numbers{$origin} if $origin;

      $sequential_signal_origins     {$sig_name} = $origin;
      $sequential_signal_widths      {$sig_name} = $width;
      $sequential_signal_reset_values{$sig_name} = $reset_val;
   }
}

################################################################
# project
#
# At construction-time, we allow the caller to associate us with 
# an e_project.  This is handy.  But this access-method can get
# called by "set" before we're properly constructed (in particular,
# before we have a name).  Thus, we take the incoming "project" arg
# and set it aside, so we can use it for-real later during the 
# validate()-phase, which is called later in the constructor.
#
################################################################
################################################################################

=item I<project()>

method description goes here...
...remember: there must be a newline around each POD tag (e.g. =item, =cut etc)!

=cut

sub project
{
   my $this = shift;
   return $this->SUPER::project() unless @_;
   my $proj = shift or &ribbit ("project-argument required");
   &ribbit ("argument must be an e_project") 
       unless &is_blessed ($proj) && $proj->isa ("e_project");
   return $this->_tmp_project($proj);
}

################################################################@
# validate
#
################################################################
################################################################################

=item I<validate()>

method description goes here...
...remember: there must be a newline around each POD tag (e.g. =item, =cut etc)!

=cut

sub validate
{
   my $this = shift;
   &ribbit ("too many arguments") if @_;
   my $mod_name = $this->name();
   my $stage_name = $this->stage() 
       or &ribbit ("pipe-module $mod_name not assigned to a stage");

   &ribbit ("pipe-module $mod_name assigned to unknown stage $stage_name")
       unless exists $known_stage_numbers{$stage_name};

   # Validate that we use the same clk_en signal as everyone else in our
   # stage--assuming there -is- someone else in our stage
   #
   my $stage_num = $known_stage_numbers{$stage_name};
   $pipe_clk_en_signals{$stage_num} = $this->pipe_clk_en()
       if (!exists ($pipe_clk_en_signals{$stage_num}) );

   &ribbit ("module ", $this->name(), " has incosistent clk_en signal")
       if $pipe_clk_en_signals{$stage_num} ne $this->pipe_clk_en();
   
   print ".";

   $this->_tmp_project()->add_module($this);
}

################################################################
# get_stage_number
#
# Access-function.
#
################################################################
################################################################################

=item I<get_stage_number()>

method description goes here...
...remember: there must be a newline around each POD tag (e.g. =item, =cut etc)!

=cut

sub get_stage_number
{
   my $this = shift;
   if (ref ($this) eq "")
   {
      # Static version:
      my $stage_name = shift;
      &ribbit ("stage-name argument required") if $stage_name eq "";
      &ribbit ("expected name-string") unless ref ($stage_name) eq "";
      &ribbit ("too many arguments") if @_;

      # If it's -already- a number, return it!
      #
      return $stage_name if $stage_name =~ /^\d+$/;

      return $known_stage_numbers{$stage_name};
   } else {
      # When called on an object:
      &ribbit ("access-only function") if @_;
      return e_pipe_module->get_stage_number($this->stage());
   }      
}   

################################################################
# add_contents
# 
# It is very important that some of our pipeline-aware
# contents know what module they belong to.  I have no idea why
# reg'lar-old e_module::add_contents() doesn't call ->parent() on 
# all the incoming contents.  It used to, but this feature was
# eliminated with a cryptic comment.
# 
# I'm providing this override so that my client objects have some
# idea of where they live.
#
################################################################
################################################################################

=item I<add_contents()>

method description goes here...
...remember: there must be a newline around each POD tag (e.g. =item, =cut etc)!

=cut

sub add_contents 
{
   my $this = shift;
   my $result = $this->SUPER::add_contents (@_);
   foreach my $thing (@_) {
      $thing->parent ($this);
   }
}

################################################################
# add_pipeline_ports
# 
# Every pipeline module gets (at least) a minimum common set of 
# pipeline-control inputs.  Inside the module, these are 
# renamed (by assignment) and known by generic uniform names.
#
################################################################
################################################################################

=item I<add_pipeline_ports()>

method description goes here...
...remember: there must be a newline around each POD tag (e.g. =item, =cut etc)!

=cut

sub add_pipeline_ports
{
   my $this = shift;
   &ribbit ("unexpected arguments") if @_;
   
   my $N = $this->get_stage_number();
   # Inside every pipe-stage, you have some generically-named 
   # pipeline control signals available to you.  But they are, of course,
   # just one of several different delayed versions of the same signal
   # from the outside world.
   #
   my @contents = ();
   foreach my $sig_name (keys(%sequential_signal_widths)) 
   {
      # I don't get signals made in the future (or even the present!)
      next unless $sequential_signal_origins{$sig_name} < $N;

      my $sig_width = $sequential_signal_widths{$sig_name};
      push (@contents, e_assign->new 
            ({lhs => e_signal->new 
                  ({name         => $sig_name, 
                    width        => $sig_width,
                    never_export => 1,           }),
              rhs => e_port->new   (["$sig_name\_$N" => $sig_width, "in"]) 
             })
            );
   }

   push (@contents, 
         e_assign->new
          ({lhs => e_signal->new ({name         => "local_pipe_clk_en",
                                   width        => 1,
                                   never_export => 1,           }),
            rhs => $this->pipe_clk_en(),
          }),
         );
   push (@contents, 
         e_assign->new({lhs => e_signal->new ({name         => "pipe_state_we",
                                               width        => 1,
                                               never_export => 1,           }),
                        rhs => "local_pipe_clk_en && 
                                ~is_neutrino      && 
                                ~is_cancelled      ",
                     }),
         );
   $this->add_contents (@contents);
}

################################################################
# create_delay_logic
#
# This stage knows how to create required delay logic for a signal
# originating in any other stage.  This may include delay-registers, 
# or perhaps just an assignment if the signal originates in the same-
# numbered pipe stage.  We return a list of all such logic.
# 
################################################################
################################################################################

=item I<create_delay_logic()>

method description goes here...
...remember: there must be a newline around each POD tag (e.g. =item, =cut etc)!

=cut

sub create_delay_logic
{
   my $this = shift;
   my ($arg) = (@_) or &ribbit ("no arguments (named-arg hash, please)");

   &validate_parameter ({hash => $arg,   name => "source_signal"});
   &validate_parameter ({hash => $arg,   name => "dest_signal"  });
   &validate_parameter ({hash => $arg,   name => "source_stage" });
   &validate_parameter ({hash => $arg,   name => "width"        });
   &validate_parameter ({hash => $arg,   name => "dest_stage"   ,
                         default => $this->get_stage_number()});

   &ribbit ("source-signal must be a -name-") 
       unless ref ($arg->{source_signal}) eq "";

   # If the caller passed-in a stage name, politely convert it to a number:
   my $dest_stage_number   = 
       e_pipe_module->get_stage_number ($arg->{dest_stage});
   my $source_stage_number = 
       e_pipe_module->get_stage_number ($arg->{source_stage});

   # Simple case: No delay at all.
   if ($source_stage_number == $dest_stage_number) {
      return e_assign->new ([$arg->{dest_signal}, $arg->{source_signal}]);
   }

   my $sig_name = $arg->{source_signal};
   my @result = ();
   foreach my $i ($source_stage_number+1 .. $dest_stage_number)
   {
      my $local_sig = e_signal->new (["$sig_name\_$i", $arg->{width}]);
      if (($i - $source_stage_number <= 1)) {
         push (@result, e_assign->new ([$local_sig, $arg->{source_signal}]));
      } else {
         my $prev_stage = $i-1;
         my $local_clk_en = $pipe_clk_en_signals{$prev_stage};
         &ribbit ("no local clock-enable signal for pipe-stage # $prev_stage")
             unless $local_clk_en;
         push (@result, e_register->new 
               ({out    => $local_sig,
                 in     => "$sig_name\_$prev_stage",
                 enable => $local_clk_en,
                })
               );
      } 
   }
   
   push (@result, 
         e_assign->new ({lhs => $arg->{dest_signal}, 
                         rhs => "$sig_name\_$dest_stage_number" })
         );
   
   return @result;
}

################################################################
# build_sequential_signals
#
# At the beginning of time, someone calls "define_sequential_signals"
# to name the signals that are passed-down the pipe.  Each e_pipe_module
# object does its own wierd voodoo to pick-up the right versions of 
# all these signals.  But someone, somewhere, must make the logic
# that implements the registers that delay these signals as needed.
# That would be here.  
# 
# This function returns a list of e_things_... that you add to whatever
# (presumably top-level) module you're building to implement the pipe-delays.
# Note that the clk-en for each pipe stage is heeded when the registers
# are constructed.
#
################################################################
################################################################################

=item I<build_sequential_signals()>

method description goes here...
...remember: there must be a newline around each POD tag (e.g. =item, =cut etc)!

=cut

sub build_sequential_signals
{
   my $this = shift;
   &ribbit ("Please call statically") unless ref ($this) eq "";

   my @result = ();
   my $N = e_pipe_module->get_max_delay();
   foreach my $pipe_sig (keys(%sequential_signal_widths))
   {
      my $w            = $sequential_signal_widths{$pipe_sig};
      my $origin_stage = $sequential_signal_origins{$pipe_sig};
      my $reset_val    = $sequential_signal_reset_values{$pipe_sig};
      foreach my $i (0..$N)
      {
         next if $i < $origin_stage;
         my $local_sig_name = join ("_", $pipe_sig, $i);
         push (@result, e_signal->new ({name  => $local_sig_name,
                                        width => $w,
                                        never_export => 1     }));

         if ($i == $origin_stage) {
            push (@result, e_assign->new ({lhs => $local_sig_name,
                                           rhs => $pipe_sig,      })
                  );
         } else {
            my $prev_stage = $i-1;
            my $local_clk_en = $pipe_clk_en_signals{$prev_stage};
            push (@result, e_register->new({out    => $local_sig_name,
                                            in     => "$pipe_sig\_$prev_stage",
                                            enable => $local_clk_en,
                                            async_value => $reset_val,
                                          })
                  );
         }
      }
   }
   return @result;
}

################################################################################

=item I<get_max_delay()>

method description goes here...
...remember: there must be a newline around each POD tag (e.g. =item, =cut etc)!

=cut

sub get_max_delay
{
   my $this = shift;
   &ribbit ("Please call statically") unless ref ($this) eq "";
   return &max (values (%known_stage_numbers));
}

################################################################
# provide_to_all_stages
# 
# Take some signal--presumably some input to the pipeline, and generate
# a delayed version for every stage of the pipe.
#
# You can pass-in a list of signals, in which case it gives all of them
# "the treatment."
#
# PRESUMES ALL SIGNALS ARE OF WIDTH 1.
# 
################################################################
################################################################################

=item I<provide_to_all_stages()>

method description goes here...
...remember: there must be a newline around each POD tag (e.g. =item, =cut etc)!

=cut

sub provide_to_all_stages
{
   my $this = shift;
   &ribbit ("Please call statically") unless ref ($this) eq "";
   &ribbit ("Please don't call this function.  It scares the children.");
   my (@in_signal_list) = (@_) or &ribbit ("missing arguments.");
   my $max_delay = e_pipe_module->get_max_delay();

   my @result = ();
   foreach my $in_signal (@in_signal_list) 
   {
      my $sig_name = $in_signal->name();
      my $w        = $in_signal->width();

      push (@result, 
            e_assign->new 
            ({lhs => e_signal->new (["$sig_name\_0", $w]),
              rhs => $sig_name,
             })    );

      foreach my $i (1..$max_delay)
      {
         my $last = $i-1;
         my $reg_in  = e_signal->new ({name         => "$sig_name\_".$last,
                                       width        => $w,
                                       never_export => 1,
                                    });

         my $reg_out = e_signal->new ({name         => "$sig_name\_".$i,
                                       width        => $w,
                                       never_export => 1,
                                    });

         push (@result, e_pipe_register->new ({in  => $reg_in,
                                               out => $reg_out,
                                            }),
               );
      }
   }
   return @result;
}
   
"You know what you doing!";

=back

=cut

=head1 EXAMPLE

Here is a usage example ...

=head1 AUTHOR

Santa Cruz Technology Center

=head1 BUGS AND LIMITATIONS

list them here ...

=head1 SEE ALSO

The inherited class e_module

=begin html

<A HREF="e_module.html">e_module</A> webpage

=end html

=head1 COPYRIGHT

Copyright (C)2001-2005 Altera Corporation, All rights reserved.

=cut

1;
