



################################################################
# CLASS: e_object
#
# Abstract class upon which all europa objects are
# based.  This class uses many tricks out of the Perl
# object-tutorial for creating (for example) generic
# data-access functions, etc.
#
################################################################


=head1 NAME

e_object - description of the module goes here ...

=head1 SYNOPSIS

The e_object class implements ... detailed description of functionality

=head1 METHODS

=over 4

=cut

package e_object;
use europa_utils;

use strict;
use vars qw($AUTOLOAD);  # it's a package global

my $indent;
my $log_history_p = 0;
my %construction_tally = ();

################################################################
# e_object::new
#
################################################################

  my %fields = 
      (
       name               => "",
       _AUTOLOAD_ACCEPT_ALL => 0,
       comment            => "",
       _creation_history  => '',
       isa_dummy => 1,
      );

  my %pointers = ();

&package_setup_fields_and_pointers
    (__PACKAGE__,
     \%fields, 
     \%pointers,
     );


################################################################################

=item I<new()>

Object constructor

=cut

sub new
{
   my $this = shift;
   my $ref = ref($this) || $this;
   my $self = bless ({},$ref);
   #my $self = bless __PACKAGE__->SUPER::new(), $ref;
   $self->set(@_);
   #&goldfish ("returning $self\n");
   return $self;
}

################
# do_log_history
#   A class-global access function to turn on/off "creation history" 
#   tracking.  By default, all e_objects, when constructed, record
#   the entire call-history of their birth in their
#   "_creation_history" member. Handy for things built by the user.
#   More-or-less useless for machine-generated things.  This 
#   function allows control over whether e_objects get or don't-get
#   a _creation_history.
#
################
################################################################################

=item I<do_log_history()>

method description goes here...
...remember: there must be a newline around each POD tag (e.g. =item, =cut etc)!

=cut

sub do_log_history{my $this = shift; ($log_history_p) = (@_);}


################################################################
# e_object::dummy
#
# Make an empty object that has the appropriate type.
#
################################################################
################################################################################

=item I<dummy()>

method description goes here...
...remember: there must be a newline around each POD tag (e.g. =item, =cut etc)!

=cut

sub dummy
{
  my $this  = shift;
  my $class = ref($this) || $this;

  return (bless {}, $class);
}

################################################################
# e_object::news
#
# Creates and sets values for n new objects where n is determined by
# scalar (@_).  Returns an array of new objects.
################################################################

################################################################################

=item I<news()>

method description goes here...
...remember: there must be a newline around each POD tag (e.g. =item, =cut etc)!

=cut

sub news
{
   my $this  = shift;
   my $class = ref($this) || $this;

   my @inputs = @_;

   my @outputs;
   foreach my $in (@inputs)
   {
      push (@outputs, $class->new($in));
   }

   return @outputs;
}

my $blab = 0;
################################################################################

=item I<blab()>

method description goes here...
...remember: there must be a newline around each POD tag (e.g. =item, =cut etc)!

=cut

sub blab 
{
  my $this = shift;
  if (@_) {  $blab = shift};
  return $blab;
}

my %construction_tally = ();

################################################################################

=item I<print_construction_report()>

method description goes here...
...remember: there must be a newline around each POD tag (e.g. =item, =cut etc)!

=cut

sub print_construction_report
{
  print STDERR "Construction history:\n";
  foreach my $class (keys(%construction_tally)) {
    printf STDERR "%9d %s\n", $construction_tally{$class}, $class;
  }
}
################################################################
# e_object::_common_member_setup
#
# The ritualized parts of all e_object constructors.
################################################################

################################################################################

=item I<_common_member_setup()>

method description goes here...
...remember: there must be a newline around each POD tag (e.g. =item, =cut etc)!

=cut

sub _common_member_setup
{
  my $this     = shift;
  my $fields   = shift or &ribbit ("No fields.");
  my $pointers = shift or &ribbit ("No pointers (pass empty hash).");

  ref ($pointers) eq "HASH" or die ("YELL!\n");

  my $class_name = ref($this);
  warn ("creating a $class_name\n") if e_object->blab();
  #$construction_tally{$class_name}++;

  my($element);
  foreach $element (keys %{$fields}) {
      $this->{_permitted}->{$element} = $fields->{$element};
    }

  foreach $element (keys %{$pointers}) {
    $this->{_pointers}->{$element} = $pointers->{$element};
  }
  #$this->set($fields);
  #$this->set($pointers);

  # This next line is bad, because it doesn't create a new
  # -copy- of all the fields for this object (everyone ends-up
  # using the template).
  #@{$this}{keys %{$fields}}   = values %{$fields};
  @{$this}{keys %{$pointers}} = values %{$pointers};

  foreach $element (keys %{$this->{_permitted}}) {
      # Note: This next line is critical, because it 
      # constructs a -new copy- of the template in %fields,
      # instead of literally using that-very-one.
    $this->$element($this->{_permitted}->{$element});
  }

  #setup get_fields and get_pointers for
  no strict 'refs';
  my $glob = $class_name.'::';
  my $p = $glob.'get_pointers';
  my $f = $glob.'get_fields';

  if (!defined(&$p) && !defined(&$f))
  {
     print "$this uses old new method\n";
     my $get_fields = sub 
     {
        my $this = shift;

        my @return_array =  (keys (%{$this->{_permitted}}));
        return (@return_array);
     };

     my $get_pointers = sub 
     {
        my $this = shift;

        my @return_array =  (keys (%{$this->{_pointers}}));
        return (@return_array);
     };

     *$p = $get_pointers;
     *$f = $get_fields;
     use strict 'refs';
  }
  my @pointers  = $this->get_pointers();
  print "pointers are @pointers\n";

  my @fields  = $this->get_fields();
  print "fields are @fields\n";
  return $this;
}

################################################################
# _direct_copy_repair
#
# SPEED OPTIMIZATION.
#
# Constructing things in a rigorous, object-oriented way using
# access functions to set the fields one-by-one is great, but it's
# slow.
#
# Some classes (e.g. "e_expression") get constructed-by-copy zillions
# of times, and it's handy to just slam one object-hash into another
# directly, like this:
#
#        # !WRONG!
#        my %new_obj = %{$e_source_obj};
#        my $self = bless \%new_obj, __PACKAGE__;
#
# Swell.  But now all of %new_obj's member-references refer to the 
# self-same objects that are members of $e_source_obj.  Sometimes this
# is OK, and sometimes it's not.   Examples of when it is OK:
#
#   * When this class doesn't have any members-by-referenece 
#     (i.e. blessed member objects).
#   * When it does have blessed members, but it's OK for more than one
#     instance of this object to refer to the very-same member data.
#
# You need to decide when this is and is not OK on a class-by-class
# basis, and "do something special" for the particular members for
# which this is NOT OK (if any).  You do this by defining (overriding)
# _direct_copy_repair for your class.  If you do this direct-copy
# hooha, then you must call your class's _direct_copy_repair after
# you're done.  Each _direct_copy_repair must call SUPER::_direct_copy_repair.
# Hers' how you'd do the above code right.
# 
#        # !RIGHT!
#        my %new_obj = %{$e_source_obj};
#        my $self = bless \%new_obj, __PACKAGE__;
#        $self->_direct_copy_repair();
#
# If you haven't bothered to say how your class behaves when directly-copied,
# you're a big fat hoser and you lose (see below).
#
################################################################
################################################################################

=item I<_direct_copy_repair()>

method description goes here...
...remember: there must be a newline around each POD tag (e.g. =item, =cut etc)!

=cut

sub _direct_copy_repair
{
  &ribbit ("e_object::_direct_copy_repair called.  You must override.");
}



################################################################
# e_object::identify
#
# identifies the object in human understandable terms.
# e.g. "The e_object created by file <my_file> on line (321)" is much
# more helpful than "e_foo=(HASH0x12345)"
################################################################
################################################################################

=item I<identify()>

method description goes here...
...remember: there must be a newline around each POD tag (e.g. =item, =cut etc)!

=cut

sub identify
{
  my $this  = shift;

  my $id_string = "\n\n---------------------------------------\n";

  if ($this->can("name") && $this->name())
  {
     my $name = $this->name();
     $id_string .= "NAME: ($name)\n";
  }
  $id_string .= $this->_creation_history();
  $id_string .= "\n---------------------------------------\n";
  return $id_string;
}

################################################################
# e_object::copy_this
#
# copies the contents of a ref to a new ref of the same type
################################################################
################################################################################

=item I<copy_this()>

method description goes here...
...remember: there must be a newline around each POD tag (e.g. =item, =cut etc)!

=cut

sub copy_this
{
   my $this  = shift;

   my $thing_to_copy    = shift;
   my $do_copy_pointers = shift;
   my $type = ref($thing_to_copy);

   if ($type eq "SCALAR")
   {
      my $new_thing = $$thing_to_copy;
      return (\$new_thing);
   }
   if ($type eq "ARRAY")
   {
      my @new_thing = @$thing_to_copy;
      return (\@new_thing);
   }
   if ($type eq "HASH")
   {
      my %new_thing = %$thing_to_copy;
      return (\%new_thing);
   }

   # the following commented code gets you in trouble if you copy a
   # reference to a parent since it will put you in an infinite loop
   # return ($thing_to_copy->copy())
   # if ($type);

   #type must just be a scalar or another e_type.
   return ($thing_to_copy);
}

################################################################################

=item I<empty_array()>

method description goes here...
...remember: there must be a newline around each POD tag (e.g. =item, =cut etc)!

=cut

sub empty_array
{
   return ();
}

################################################################
# e_object::copy
#
# makes and returns a copy of an object.
################################################################
################################################################################

=item I<copy()>

method description goes here...
...remember: there must be a newline around each POD tag (e.g. =item, =cut etc)!

=cut

sub copy
{
  my $this  = shift;

  my $class = ref($this) or &ribbit ("copy: requires object reference");
  return $class->new ($this);
}

################################################################
# e_object::convert_to_new_class
#
# Takes a source and destination.  Copies all applicable fields from
# the source to the destination.  returns destination.  If source and
# destination are the same type, it just returns the source.
#
################################################################
################################################################################

=item I<convert_to_new_class()>

method description goes here...
...remember: there must be a newline around each POD tag (e.g. =item, =cut etc)!

=cut

sub convert_to_new_class
{
  my $source  = shift;
  my $source_class = ref($source) || &ribbit ("no source");
  my $destination = shift;
  my $destination_class = ref($destination) || &ribbit ("no destination");

  return ($source)
      if ($source_class eq $destination_class);

  foreach my $key ($destination->get_fields())
  {
     $destination->$key($source->copy_this($source->$key()));
  }

  return $destination;
}

################################################################
# e_object::set
#
# Sets stuff to object through AUTOLOAD.  You call 
# $foo->set ({name => me, width => 32, direction => output});
#
# or if you have class->_order specified as
# ("name","width","direction")
# it makes the equivalent hash for ["me",32,"output"],
#
# Then it calls $foo->name(me); 
#          $foo->width(32); 
#          $foo->direction(output);
#
# It returns nothing
################################################################
################################################################################

=item I<set()>

method description goes here...
...remember: there must be a newline around each POD tag (e.g. =item, =cut etc)!

=cut

sub set
{
  my $this  = shift;
  my $in = shift;
  return if ($in eq '');

  my $p_hash;

  if (ref ($in) eq "ARRAY")
  {
     my @order = @{$this->_order()} or &ribbit 
         ("unable to set based upon array ref, no _order has been ",
          "specified");

     foreach my $input (@$in)
     {
        my $ord = shift (@order);
        $p_hash->{$ord} = $input;
     }
  }
  elsif (ref ($in) eq "HASH") {
    $p_hash = $in;

  }
  elsif (&is_blessed($in) && $in->isa(ref ($this)))
  {
     my @field_list = $this->access_methods_for_auto_constructor();

     foreach my $one_field (@field_list)
     {
        $p_hash->{$one_field} = $in->$one_field();
     }
  }
  else
    {
      &ribbit ("I am sorry, please rephrase what you are setting ",
               "in the form of a hash, array reference, or like object.");
     }

  my $function;
  foreach $function (keys (%$p_hash))
  {
     $this->$function($$p_hash{$function});
  }
  $this->isa_dummy(0);
  return $this;
}

################################################################################

=item I<access_methods_for_auto_constructor()>

method description goes here...
...remember: there must be a newline around each POD tag (e.g. =item, =cut etc)!

=cut

sub access_methods_for_auto_constructor
{
   my $this = shift;
   return ($this->get_fields(), $this->get_pointers());
}

################################################################
# e_object::string_to_vhdl_comment takes a string and converts it to a
# vhdl comment.
# 
################################################################
################################################################################

=item I<string_to_vhdl_comment()>

method description goes here...
...remember: there must be a newline around each POD tag (e.g. =item, =cut etc)!

=cut

sub string_to_vhdl_comment
{
   my $this = shift;

   my $indent = shift;
   my $comment = shift || return;

   my $single_line_comment = '--';
   $comment = $indent.$single_line_comment.$comment;
   $comment =~ s/\n\s*(?=\S)/\n$indent$single_line_comment/sg;

   return("$comment\n");
}

################################################################
# e_object::string_to_verilog_comment takes a string and converts it to a
# verilog comment.
# 
################################################################
################################################################################

=item I<string_to_verilog_comment()>

method description goes here...
...remember: there must be a newline around each POD tag (e.g. =item, =cut etc)!

=cut

sub string_to_verilog_comment
{
   my $this = shift;

   my $indent = shift;
   my $comment = shift || return;

   my $single_line_comment = '//';
   $comment = $indent.$single_line_comment.$comment;
   $comment =~ s/\n\s*(?=\S)/\n$indent$single_line_comment/sg;

   return("$comment\n");
}


################################################################################

=item I<debug_to_string()>

method description goes here...
...remember: there must be a newline around each POD tag (e.g. =item, =cut etc)!

=cut

sub debug_to_string
{
   my $this = shift;

   my $level = shift  || 0;
   my $indent = shift || "";
   my $value = shift or return();

   $level--
       if ($level);

   my $rs;
   if ($level)
   {
      if (&is_blessed($value) && $value->isa("e_object"))
      {
         $rs .= $value->debug
             (
              $level,
              "$indent  "
              );

         return ($rs);
      }
      
      if (ref ($value) eq "ARRAY")
      {
         my $index = 0;
         foreach my $key (@$value)
         {
            $rs .= $indent." $index \-\>  $key\n";
            $rs .= $this->debug_to_string
                (
                 $level,
                 $indent."  ",
                 $key
                 );
            $index++;
         }
         return ($rs);
      }
      
      if (ref($value) eq "HASH")
      {
         foreach my $key (%$value)
         {
            $rs .= "$indent$key \-\> $value->{$key}\n";
            $rs .= $this->debug_to_string
                (
                 $level,
                 $indent."  ",
                 $value->{$key}
                 );
         }
         return ($rs);
      }
   }
   return;
}

################################################################################

=item I<debug()>

method description goes here...
...remember: there must be a newline around each POD tag (e.g. =item, =cut etc)!

=cut

sub debug
{
   my $this = shift;

   my $level = shift;
   my $indent = (shift or "");
   my $return_string;
   if ($level)
   {
      foreach my $field ($this->get_fields(),
                         $this->get_pointers())
      {
         my $value = ($this->{$field});
         $return_string .= "$indent$field \-\> $value\n";
         $return_string .= $this->debug_to_string(
                                                  $level,
                                                  $indent."  ",
                                                  $value
                                                  );
      }
   }
   return ($return_string);
}

######################################################################
#
# Most ptf entries come like this.
#  PORT => {address =>
#  {
#     avalon_role => "address",
#     direction => "input",
#     width => "8",
#  }}
#
# But to use them as objects, it should look like
#  e_port 
#  {
#     name => "address",
#     avalon_role => "address",
#     direction => "input",
#     width => "8",
#  }
#
# this method moves the first key to a name => value pair within the
# value and returns the hash. 
#
######################################################################

################################################################################

=item I<ptf_to_hashes()>

method description goes here...
...remember: there must be a newline around each POD tag (e.g. =item, =cut etc)!

=cut

sub ptf_to_hashes
{
   my $this = shift;
   my $fields = shift or &ribbit 
       ("no ptf fields to convert to object");

   my @return_hashes;
   foreach my $name (sort (keys (%$fields)))
   {
      my $value = $fields->{$name};
      $value->{name} = $name;
      push (@return_hashes, 
            $value);
   }
   return (@return_hashes);
}

################################################################################

=item I<ptf_to_hash()>

method description goes here...
...remember: there must be a newline around each POD tag (e.g. =item, =cut etc)!

=cut

sub ptf_to_hash
{
   my $this = shift;
   my @hashes = $this->ptf_to_hashes(@_);
   return ($hashes[0]);
}

################################################################
# e_object::AUTOLOAD
#
# As per the venerated Perl OO tutorial, this function gets executed
# whenever the user calls an undefined member method.  We use it to
# automatically build access-functions for every data field: Any
# unrecognized call is taken to be an access function for one of the
# class's members.  
#
# AUTOLOAD checks types as well.  If you are setting foo => value,
# value had better be of the same type as $this->{_permitted}{foo} or
# AUTOLOAD will complain.  If your new value is blessed and
# isa->(_permitted type) then AUTOLOAD will let you go along your
# merry way.
#
# If the member doesn't exist, then you get an
# error unless the object has _AUTOLOAD_ACCEPT_ALL field set within
# it.  Then all of the above trickery goes out the window and you'll
# just set key => value just like they did in the old days.
#
# AUTOLOAD is a nifty trick that saves us writing an access function
# for every trivial new member.
#
################################################################
################################################################################

=item I<AUTOLOAD()>

method description goes here...
...remember: there must be a newline around each POD tag (e.g. =item, =cut etc)!

=cut

sub AUTOLOAD 
{
  my $this = shift;
  # Mighty darned suspicious if "$this" is not an object:
  my $reffo = ref($this);
  &goldfish ("You may have called an unknown function ($AUTOLOAD).\n",
             "  Did you 'use' the right modules?") if !$reffo;
  my $type = $reffo || $this;


  my $name = $AUTOLOAD;
  $name =~ s/.*://;   # strip fully-qualified portion

  return if $name eq "DESTROY";

  if (exists $this->{_permitted}->{$name}  ||
      exists $this->{_pointers}->{$name}    )
  {
    # When called with no arguments, it's an access function:
    #
    #    print "JMB -- $name $type\n";
    return ($this->{$name}) unless scalar(@_); 

    ################ 
    # OK.  Now we know that the user has called a function
    #      which is one of our member-things.
    #      Furthermore, we know they have called said function with
    #      arguments, which means they want to -set- the member.
    #      How do you set a member?  It depends.
    #
    #         * Pointers are just assigned (always), after type-checking.
    #         * blessed-object members are constructed from (@_).
    #         * Other, er, things are just assigned, I suppose
    #           (lists, strings, hashes, etc.)
    #
    if (exists $this->{_pointers}->{$name}) {
      my $value = shift;
      my $pointer_thing = $this->{_pointers}->{$name};
      my $allowed_type  = ref ($pointer_thing);
      # Check to be sure pointer is the right type:
      if (&is_blessed($pointer_thing)) {
        # Blessed things must meet the "isa" requirement.
        &ribbit ("Pointer-member $name must be set from a '$allowed_type'-ref")
          unless (&is_blessed($value) && $value->isa($allowed_type));
      } else {
        # Non-blessed things must be of -identical- type:
        &ribbit ("Pointer-member $name must be set from a'$allowed_type'-ref")
          unless ($allowed_type eq ref ($value));
      }

      $this->{$name} = $value;
      return ($this->{$name});  # DONE.
    }

     # If it's not a pointer, it MUST be permitted.
     &ribbit ("no _permitted member '$name' found (THIS CAN'T HAPPEN)")
       unless exists ($this->{_permitted}->{$name});

     my $permitted_thing = $this->{_permitted}->{$name};

     # For blessed-class members, "set" them by passing incoming
     # arguments to their constructor and build a new one:
     #
     if (&is_blessed ($permitted_thing)) {
       my $member_type = ref ($permitted_thing);
       $this->{$name} = $member_type->new (@_);
       return ($this->{$name});  # DONE
     }

     # Well, if we got to here, then:
     # * It -is- _permitted, but
     # * It's not a blessed-class.
     # 
     # So it must be a string or list or something.
     # copy it to this->{name}.

     my $value = shift;
     $this->{$name} = e_object->copy_this($value);
     return ($this->{$name});
  }

  # The user called a member-function which was not recognized from
  # the _pointer or _permitted-hashes.
  # Generally we will complain--unless the magic flag has been set:
  if ($this->_AUTOLOAD_ACCEPT_ALL())
    {
      if (@_)
        {
          my $value = shift;
          $this->{$name} = $value;
        }
      #and we'll still complain if the variable doesn't exist.
      if (!exists $this->{$name})
      {
         #&ribbit("($this) Even though _AUTOLOAD_ACCEPT_ALL is set, nobody has ",
         #"ever set $name before\n");
      }
      return ($this->{$name});
    }
  else
    {
       &is_blessed($this) or &ribbit ("$this is not blessed\n");
       my $known_fields   = "\n " . join ("\n ", sort ($this->get_fields()));
       my $known_pointers = "\n " . join ("\n ", sort ($this->get_pointers()));
       my $keys = "\n " . join("\n ", sort(keys(%$this)));
       my $isa_dummy = $this->isa_dummy();
       
      
      &ribbit ("In object '$this->{name}' of class $type: can't access `$name' field\n",
               "known fields are: $known_fields\n",
               "known pointers are: $known_pointers\n",
               "keys: $keys\n",
               "by the way, this object is ", $isa_dummy ? "" : "not ", "a dummy\n",
              );
    }
}

################################################################################

=item I<handle_array()>

method description goes here...
...remember: there must be a newline around each POD tag (e.g. =item, =cut etc)!

=cut

sub handle_array
{
   my $this = shift;
   my $array = shift;
   if (@_)
   {
      push (@$array, @_);
   }
   return $array;
}

################################################################################

=item I<handle_hash()>

method description goes here...
...remember: there must be a newline around each POD tag (e.g. =item, =cut etc)!

=cut

sub handle_hash
{
   my $this = shift;
   my $hash = shift;
   my @args = @_;
   my $arg_number = @args;

   if (!defined ($hash))
   {
      &ribbit ("hash not defined\n");
   }
   if ($arg_number == 1)
   {
      return $hash->{$args[0]};
   }
   elsif ($arg_number == 0)
   {
      my %new_hash = %$hash;
      return \%new_hash;
   }
   elsif (($arg_number % 2) == 0)
   {
      my $key;
      my $value;

      while (($key,$value,@args) = @args)
      {
         if (ref ($value) eq "ARRAY")
         {
            push (@{$hash->{$key}}, @$value);
         }
         else
         {
            $hash->{$key} = $value;
         }
      }
   }
   else
   {
      &ribbit ("illegal number of arguments to hash",
               "($arg_number)");
   }
   return $hash;
}

################################################################################

=item I<DONE()>

method description goes here...
...remember: there must be a newline around each POD tag (e.g. =item, =cut etc)!

=cut

sub DONE
{
   my $this = shift;
   #&goldfish("$this done\n");
   return 1;
}

################################################################################

=item I<done()>

method description goes here...
...remember: there must be a newline around each POD tag (e.g. =item, =cut etc)!

=cut

sub done
{
   return shift->DONE(@_);
}

1;









=back

=cut

=head1 EXAMPLE

Here is a usage example ...

=head1 AUTHOR

Santa Cruz Technology Center

=head1 BUGS AND LIMITATIONS

list them here ...

=head1 SEE ALSO



=begin html



=end html

=head1 COPYRIGHT

Copyright (C)2001-2005 Altera Corporation, All rights reserved.

=cut

1;
