# For all those of you out there who bag on europa and its "Winchester
# Mystery House Architecture" I present the Smith and Wesson solution.

# e_lpm_base: A new class which helps insert lpm_instances into your
# europa code.  Don't muck around in here.  The only reason why I bring
# it up is so that you can set @ISA = qw (e_lpm_base) in your coolio new
# europa class.

# e_lpm_dcfifo: An example class which uses e_lpm_base.  Check it out.

# e_lpm_base parses the vhdl component declaration (Which I copied
# directly from Quartus LPM Megafunction Documentation) and uses its
# info to figure out width_matches based upon parameter map
# declarations. E.g. In the dcfifo example, e_lpm_base notes that both d
# and q are of width (LPM_WIDTH).  When somebody later on defines a
# signal q of width 24, e_lpm_width automatically propogates the width
# to the "d" pin (unless d has also been defined.  See: standard europa
# signal matching.  e_lpm_base is also smart enough to set the component
# LPM_WIDTH value before outputing as HDL.

# Another cool advantage here is that different tools sometime have
# different ideas about which default settings are actually the default
# settings.  By copying the component directly, europa directly
# specifies ALL component settings.  The tools no longer have any say
# about defaults since europa specifies them directly.


=head1 NAME

e_lpm_base - description of the module goes here ...

=head1 SYNOPSIS

The e_lpm_base class implements ... detailed description of functionality

=head1 METHODS

=over 4

=cut

package e_lpm_base;
use e_blind_instance;
use europa_utils;

@ISA = ('e_blind_instance');
use strict;

my %fields = ();

my %pointers = ();

&package_setup_fields_and_pointers
    (__PACKAGE__,
     \%fields, 
     \%pointers,
     );

################################################################################

=item I<vhdl_declare_component()>

method description goes here...
...remember: there must be a newline around each POD tag (e.g. =item, =cut etc)!

=cut

sub vhdl_declare_component
{
   &ribbit ("overload this with your vhdl declaration");
}

################################################################################

=item I<set_port_map_defaults()>

method description goes here...
...remember: there must be a newline around each POD tag (e.g. =item, =cut etc)!

=cut

sub set_port_map_defaults
{
   return;
}

################################################################################

=item I<set_parameter_map_defaults()>

method description goes here...
...remember: there must be a newline around each POD tag (e.g. =item, =cut etc)!

=cut

sub set_parameter_map_defaults
{
   return;
}

################################################################################

=item I<new()>

Object constructor

=cut

sub new
{
   my $this = shift;
   $this = $this->SUPER::new();
   $this->_parse_component_from_vhdl_component_declaration
       ();

   $this->set_port_map_defaults();
   $this->set_parameter_map_defaults();
   $this->set(@_);
   return $this;
}

###############
# BEGIN PARSING CODE
################################################################################

=item I<_parse_component_from_vhdl_component_declaration()>

method description goes here...
...remember: there must be a newline around each POD tag (e.g. =item, =cut etc)!

=cut

sub _parse_component_from_vhdl_component_declaration
{
   my $this = shift;
   my $string = $this->vhdl_declare_component();

   if ($string =~ /^\s*COMPONENT\s+(\w+)\s+
       GENERIC\s*\(\s*(.*?)\s*\)\s*\;\s*
       PORT\s*\(\s*(.*?)\s*\)\s*\;\s*
       END\s+COMPONENT/six)
   {
      my ($module_name, $parameter_string, $port_string) = 
          ($1,$2,$3);

      $this->module($module_name);

      my @parameters = split (/\s*\;\s*/s, $parameter_string);
      foreach my $parameter (@parameters)
      {
         my ($name, $type, $default) = split 
             (/\s*\:\s*/s, $parameter);

         $default =~ s/^\=\s*(\S*)\s*$/$1/s;
         if ($default ne '')
         {
            $this->parameter_map({$name => $default});
         }
      }

      my @ports = split (/\s*\;\s*/s, $port_string);
      foreach my $port (@ports)
      {
         $this->_process_port($port);
      }
   }
   else
   {
      &ribbit ("couldn't parse $string");
   }
}

################################################################################

=item I<_process_port()>

Called for each port identified in the VHDL component declaration.
Adds this port to the object's port lists.

=cut

sub _process_port
{
   my $this = shift;
   my $port = shift;

   if ($port =~ /^\s*(.*?)\s*\:\s*
       (IN|OUT|INOUT)\s+
       (STD_LOGIC_VECTOR|STD_LOGIC)
       \s*(\(\s*(\w+)\s*(\-\s*1\s*)?DOWNTO\s+0\s*\))?
       \s*(\:\=\s*(\S+))?
       /six
       )
   {
      my (
          $name_string,
          $direction,
          $type, 
          $vector_width, $msb, $minus_one, 
          $default
          ) = ($1,$2,$3,$4,$5,$6,$8);

      $direction = lc ($direction);
      my @names = split (/\s*\,\s*/s,$name_string);
      my @force_these_names_to_std_logic_vector;

      foreach my $name (@names)
      {
         #####
         # Should the name appear in the instance if it hasn't been
         # explicitly typed in by a human?.  Answer: Yes unless it has
         # a default setting for it in the component declaration.
         my $name_appears_in_default_instance = 
             ($direction ne 'in') || ($default eq '');

         push (@force_these_names_to_std_logic_vector, $name)
             if ($type =~ /vector$/i);
         #####
         # Should we match port widths? E.g. a ram has input data and
         # output data.  Obviously, those two values should resolve to
         # the same width unless explicitly specified otherwise.  We
         # keep track of parameterized vector widths by the parameter
         # name used to identify that width so that we can link up the
         # two later in make_linked_signal_conduit_list.  To continue
         # our example above, the component will declare D and Q of
         # width ((DATA_WIDTH - 1) DOWNTO 0).  That shows the two
         # values should be identical.

         my $add_name_to_width_conduit_vector = 
             ($minus_one) && ($msb =~ /\D/);

         my $redirected_port_map_name =
             ($name_appears_in_default_instance)?
             $name : "open";

         my $port_map_name = $direction."_port_map";

         $this->$port_map_name
             ($name => $redirected_port_map_name);

         if ($add_name_to_width_conduit_vector)
         {
            push (@{$this->{_width_conduit_by_parameter_name}{$msb}},
                  $name);

            $this->_expression_port_map()->{$name}->conduit_width(1);
         }

      }

      ###############
      # Treat these as std_logic_vector_signals even if they end up
      # being one bit wide.
      push (@{$this->std_logic_vector_signals()},
            @force_these_names_to_std_logic_vector);
   }
   else
   {
      &ribbit ("couldn't parse $port string $port\n");
   }
}

# END PARSING CODE
###############

###############
# BEGIN CONDUIT WIDTH CODE
################################################################################

=item I<find_linked_signals()>

method description goes here...
...remember: there must be a newline around each POD tag (e.g. =item, =cut etc)!

=cut

sub find_linked_signals
{
   my $this = shift;
   my $signal_name = shift;
   my $remove_conduit_width = shift;

   my @std_logic_vector_signals = @{$this->std_logic_vector_signals()};

   # only std_logic_vector_signals may be linked to each other
   if (grep {$_ eq $signal_name} @std_logic_vector_signals)
   {
      my $parameter_width_hash = $this->{_width_conduit_by_parameter_name};

      foreach my $width_name 
          (keys (%$parameter_width_hash))
      {
         my @signal_list = @{$parameter_width_hash->{$width_name}};
         if (grep {$_ eq $signal_name} @signal_list)
         {
            if ($remove_conduit_width)
            {
               map{$this->_expression_port_map()->{$_}->conduit_width(0)}
               @signal_list;
            }

            return @signal_list;
         }
      }
   }
}

################################################################################

=item I<make_linked_signal_conduit_list()>

method description goes here...
...remember: there must be a newline around each POD tag (e.g. =item, =cut etc)!

=cut

sub make_linked_signal_conduit_list
{
   my $this = shift;
   my $port_name = shift;

   my $port_map = $this->port_map();
   my %reverse_map = reverse (%$port_map);

   my $signal_name = $reverse_map{$port_name};

   if (!$signal_name)
   {
       &ribbit ("No signal for $port_name\n" .%reverse_map);
   }

   my @signal_list = $this->find_linked_signals($signal_name,1);

   #remap signals.  Make sure we avoid duplicates including
   #port name
   my %exclusive_name;
   foreach my $signal (@signal_list)
   {
      my $port_mapped_name = $port_map->{$signal};
      # strip out signal indexes
      $port_mapped_name =~ s/^([A-Za-z_]\w*)(\[\d+\])?$/$1/;
      #No numerical values allowed.
      next unless ($port_mapped_name =~ /^[A-Za-z_]\w*$/);
      $exclusive_name{$port_mapped_name}++;
      delete ($exclusive_name{$port_name});
   }
   my @exclusive = keys (%exclusive_name);

   my $parent_module = $this->parent_module();
   return map 
   {$parent_module->
        make_linked_signal_conduit_list($_);}
   @exclusive;
}

# END CONDUIT WIDTH CODE
###############

###############
# BEGIN to_hdl code

################################################################################

=item I<set_autoparameters()>

method description goes here...
...remember: there must be a newline around each POD tag (e.g. =item, =cut etc)!

=cut

sub set_autoparameters
{
   my $this = shift;
   my $parameter_width_hash =
       $this->{_width_conduit_by_parameter_name};

   my $parent_module = $this->parent_module();
   #base the parameter value upon width of signals
   foreach my $parameter (keys(%$parameter_width_hash))
   {
      my $max_width = 0;
      foreach my $signal_name 
          (@{$parameter_width_hash->{$parameter}})
      {
         my $mapped_signal = $this->port_map($signal_name);
         my $expression = $this->_expression_port_map->{$signal_name};
         my $expression_width = $expression->width();
         if ($expression_width > $max_width)
         {
            $max_width = $expression_width;
         }
      }
      $this->parameter_map({$parameter => $max_width});
   }
}

################################################################################

=item I<to_verilog()>

method description goes here...
...remember: there must be a newline around each POD tag (e.g. =item, =cut etc)!

=cut

sub to_verilog
{
   my $this = shift;
   $this->set_autoparameters();
   return $this->SUPER::to_verilog(@_);
}

################################################################################

=item I<to_vhdl()>

method description goes here...
...remember: there must be a newline around each POD tag (e.g. =item, =cut etc)!

=cut

sub to_vhdl
{
   my $this = shift;
   $this->set_autoparameters();
   return $this->SUPER::to_vhdl(@_);
}

# END to_hdl code
###############

################################################################################

=item I<port_map()>

method description goes here...
...remember: there must be a newline around each POD tag (e.g. =item, =cut etc)!

=cut

sub port_map
{
   my $this = shift;
   
   if (@_)
   {
      if (@_ == 1)
      {
         return $this->SUPER::port_map(@_);
      }
      else
      {
         my $key;
         my $value;

         my @set_these = @_;
         (@set_these % 2 == 0) || &ribbit ("bad number @set_these");
         while (($key, $value, @set_these) = @set_these)
         {
            if ($this->in_port_map($key) ne '') {
               $this->in_port_map($key => $value);
            }
            elsif ($this->out_port_map($key) ne '') {
               $this->out_port_map($key => $value);
            }
            elsif ($this->inout_port_map($key) ne '') {
               $this->inout_port_map($key => $value);
            } else {
               &ribbit ("key ($key) isn't in the in-, out-, or inout-port map, so I don't feel safe adding ($value) to it");
            }
         }
      }
   }
   return $this->SUPER::port_map();
}
__PACKAGE__->DONE();

=back

=cut

=head1 EXAMPLE

Here is a usage example ...

=head1 AUTHOR

Santa Cruz Technology Center

=head1 BUGS AND LIMITATIONS

list them here ...

=head1 SEE ALSO



=begin html



=end html

=head1 COPYRIGHT

Copyright (C)2001-2005 Altera Corporation, All rights reserved.

=cut

1;
