# For all those of you out there who bag on europa and its "Winchester
# Mystery House Architecture" I present the Smith and Wesson solution.

# e_lpm_base: A new class which helps insert lpm_instances into your
# europa code.  Don't muck around in here.  The only reason why I bring
# it up is so that you can set @ISA = qw (e_lpm_base) in your coolio new
# europa class.

# e_lpm_base parses the vhdl component declaration (Which I copied
# directly from Quartus LPM Megafunction Documentation) and uses its
# info to figure out width_matches based upon parameter map
# declarations. E.g. In the dcfifo example, e_lpm_base notes that both d
# and q are of width (LPM_WIDTH).  When somebody later on defines a
# signal q of width 24, e_lpm_width automatically propogates the width
# to the "d" pin (unless d has also been defined.  See: standard europa
# signal matching.  e_lpm_base is also smart enough to set the component
# LPM_WIDTH value before outputing as HDL.

# Another cool advantage here is that different tools sometime have
# different ideas about which default settings are actually the default
# settings.  By copying the component directly, europa directly
# specifies ALL component settings.  The tools no longer have any say
# about defaults since europa specifies them directly.


=head1 NAME

e_lpm_altsyncram - description of the module goes here ...

=head1 SYNOPSIS

The e_lpm_altsyncram class implements ... detailed description of functionality

=head1 METHODS

=over 4

=cut

package e_lpm_altsyncram;
use e_lpm_base;
@ISA = ("e_lpm_base");

use strict;
use europa_utils;

#You must declare your vhdl component declaration for this class to
#work
################################################################################

=item I<vhdl_declare_component()>

method description goes here...
...remember: there must be a newline around each POD tag (e.g. =item, =cut etc)!

=cut

sub vhdl_declare_component
{
return q[
component altsyncram 
   generic
      (operation_mode                     : string  := "single_port";
      width_a                             : integer := 8;
      widthad_a                           : integer := 2;
      numwords_a                          : integer := 4;
      outdata_reg_a                       : string  := "unregistered";    
      address_aclr_a                      : string  := "none";    
      outdata_aclr_a                      : string  := "none";    
      indata_aclr_a                       : string  := "clear0";    
      wrcontrol_aclr_a                    : string  := "clear0";    
      byteena_aclr_a                      : string  := "none";    
      width_byteena_a                     : integer := 1;    
      width_b                             : integer := 8;
      widthad_b                           : integer := 4;
      numwords_b                          : integer := 4;
      rdcontrol_reg_b                     : string  := "clock1";    
      address_reg_b                       : string  := "clock1";    
      indata_reg_b                        : string  := "clock1";    
      wrcontrol_wraddress_reg_b           : string  := "clock1";    
      byteena_reg_b                       : string  := "clock1";    
      outdata_reg_b                       : string  := "unregistered";    
      outdata_aclr_b                      : string  := "none";    
      rdcontrol_aclr_b                    : string  := "none";    
      indata_aclr_b                       : string  := "none";    
      wrcontrol_aclr_b                    : string  := "none";    
      address_aclr_b                      : string  := "none";    
      byteena_aclr_b                      : string  := "none";    
      clock_enable_input_a                : string  := "normal";
      clock_enable_output_a               : string  := "normal";
      clock_enable_input_b                : string  := "normal";
      clock_enable_output_b               : string  := "normal";
      width_byteena_b                     : integer := 1;    
      byte_size                           : integer := 8; 
      read_during_write_mode_mixed_ports  : string  := "dont_care";    
      ram_block_type                      : string  := "auto";    
      init_file                           : string  := "unused";    
      init_file_layout                    : string  := "unused";    
      maximum_depth                       : integer := 0;    
      intended_device_family              : string  := "stratix";
      lpm_hint                            : string  := "unused";
      lpm_type                            : string  := "altsyncram" );

   port (wren_a, wren_b, aclr0,
           aclr1, addressstall_a,
           addressstall_b          : in std_logic                                          := '0';
        rden_b, clock0, clock1,
           clocken0, clocken1      : in std_logic                                          := '1';
        data_a                     : in std_logic_vector(width_a - 1 downto 0)             := (others => '0');
        data_b                     : in std_logic_vector(width_b - 1 downto 0)             := (others => '0');
        address_a                  : in std_logic_vector(widthad_a - 1 downto 0)           := (others => '0');
        address_b                  : in std_logic_vector(widthad_b - 1 downto 0)           := (others => '0');
        byteena_a                  : in std_logic_vector( (width_byteena_a  - 1) downto 0) := (others => 'z');
        byteena_b                  : in std_logic_vector( (width_byteena_b  - 1) downto 0) := (others => 'z');
        q_a                        : out std_logic_vector(width_a - 1 downto 0);
        q_b                        : out std_logic_vector(width_b - 1 downto 0));

end component;
];
}

################################################################################

=item I<set_port_map_defaults()>

method description goes here...
...remember: there must be a newline around each POD tag (e.g. =item, =cut etc)!

=cut

sub set_port_map_defaults
{
   # redefine the default port map here if you wish
   # E.g.set things you don't want popping out to "open"
   my $this = shift;
   $this->port_map({addressstall_a => 0,
                    addressstall_b => 0,
                    clocken0 => 1,
                    clocken1 => 1,
                 });
}

################################################################################

=item I<set_parameter_map_defaults()>

method description goes here...
...remember: there must be a newline around each POD tag (e.g. =item, =cut etc)!

=cut

sub set_parameter_map_defaults
{
   # redefine the default parameter map here if you wish
   my $this = shift;
   $this->set_b_clock(0);
}

################################################################################

=item I<set_b_clock()>

method description goes here...
...remember: there must be a newline around each POD tag (e.g. =item, =cut etc)!

=cut

sub set_b_clock
{
   my $this = shift;
   if (@_)
   {
      my $val = shift;
      $val =~ s/^(\d)/clock$1/;

      foreach my $parameter qw(
                               rdcontrol_reg_b
                               address_reg_b
                               indata_reg_b
                               wrcontrol_wraddress_reg_b
                               byteena_reg_b
                               )
      {
         $this->parameter_map({rdcontrol_reg_b => $val});
      }
   }
}

################################################################################

=item I<set_autoparameters()>

method description goes here...
...remember: there must be a newline around each POD tag (e.g. =item, =cut etc)!

=cut

sub set_autoparameters
{
   my $this = shift;
   my $return = $this->SUPER::set_autoparameters(@_);

   my $parameter_map = $this->parameter_map();
   foreach my $port ('a','b')
   {
      #set number of datawords.
      my $word_size = $parameter_map->{"widthad_$port"};
      $this->parameter_map({"numwords_$port" => 2**$word_size});

      #if byteenable hasn't been specified, set it to all enabled.
      my $byteenable_port = "byteena_$port";
      if ($this->port_map()->{$byteenable_port} eq '')
      {
         my $datawidth = $parameter_map->{"width_$port"};
         my $bytewidth = $datawidth / 8;
         $this->port_map({$byteenable_port => "{$bytewidth\{1\'b1\}}"});

         $this->parameter_map({width_byteena_a => $bytewidth});
      }
   }

   my $device = $this->parent_module()->project()->device_family();
   my $Device = $device;
   $Device =~ s/^(\w)(.*)$/$1.lc($2)/e;
   $this->parameter_map({intended_device_family => $Device});
   return $return;
}

1;
















=back

=cut

=head1 EXAMPLE

Here is a usage example ...

=head1 AUTHOR

Santa Cruz Technology Center

=head1 BUGS AND LIMITATIONS

list them here ...

=head1 SEE ALSO

The inherited class e_lpm_base

=begin html

<A HREF="e_lpm_base.html">e_lpm_base</A> webpage

=end html

=head1 COPYRIGHT

Copyright (C)2001-2005 Altera Corporation, All rights reserved.

=cut

1;
